#include "../CpqCiWrp.h"
#include "../CpqCiHlx.h"
#include "../CpqCiTyp.h"
#include "CpqCiSem.h"

//BEGIN Semaphore implementation
//Sem array
static struct semaphore* global_sem = NULL;
//Sem array size
static int global_sem_size = 0;

#if LINUX_VERSION_CODE >= 0x020300
//locking 
static spinlock_t semaphore_lock = SPIN_LOCK_UNLOCKED;

void cpqci_force_signal(wait_queue_head_t *q, int sig)
{
	struct list_head *tmp, *head;
	struct task_struct *p;

	//DBG("cpqci_force_signal(%p, %d)\n", q, sig);

	if (!q) goto out;

	spin_lock_irq(&semaphore_lock);

	head = &q->task_list;
	tmp = head->next;
	while (tmp != head) {
                wait_queue_t *curr = list_entry(tmp, wait_queue_t, task_list);
		tmp = tmp->next;
		if (curr) { 
			p = curr->task;
			DBG("force_sig (%d. %p)\n", sig, p);
			force_sig(sig, p);
		}
	}
	spin_unlock_irq(&semaphore_lock);
out:
	return;
}

int cpqci_down_interruptible(struct semaphore * sem, int ms)
{
	int retval = 0;
	struct task_struct *tsk = current;
	DECLARE_WAITQUEUE(wait, tsk);
	//DBG("*******sem %p, ms %d\n", sem, ms);
	tsk->state = TASK_INTERRUPTIBLE;
	add_wait_queue_exclusive(&sem->wait, &wait);

	spin_lock_irq(&semaphore_lock);
	sem->sleepers ++;
	for (;;) {
		int sleepers = sem->sleepers;

		if (signal_pending(current)) {
			retval = -EINTR;
			sem->sleepers = 0;
			atomic_add(sleepers, &sem->count);
			break;
		}

		if (!atomic_add_negative(sleepers - 1, &sem->count)) {
			sem->sleepers = 0;
			break;
		}

		sem->sleepers = 1;	/* us - see -1 above */
		spin_unlock_irq(&semaphore_lock);

		if (ms >= 0) {
			int rc;
			//DBG("***schedule_timeout %d\n", ms/MS_PER_JIFFY);
			rc = schedule_timeout(ms/MS_PER_JIFFY);
			//DBG("***schedule_timeout returns with %d\n", rc);
			if (rc == 0) {
				sem->sleepers = 0;
				atomic_add(sleepers, &sem->count);
				goto SCHEDULING_DONE;
			}
		} else {
			//DBG("***schedule\n");
			schedule();
			//DBG("***schedule returns\n");
		}
		//DBG("*** task %p continues....\n", current);
		tsk->state = TASK_INTERRUPTIBLE;
		spin_lock_irq(&semaphore_lock);
	}
	spin_unlock_irq(&semaphore_lock);
SCHEDULING_DONE:
	tsk->state = TASK_RUNNING;
	remove_wait_queue(&sem->wait, &wait);
	wake_up(&sem->wait);
	return retval;
}
#else
#define DOWN_VAR				\
	struct task_struct *tsk = current;	\
	struct wait_queue wait = { tsk, NULL };

#define DOWN_HEAD(task_state)						\
									\
									\
	tsk->state = (task_state);					\
	add_wait_queue(&sem->wait, &wait);				\
									\
	/*								\
	 * Ok, we're set up.  sem->count is known to be less than zero	\
	 * so we must wait.						\
	 *								\
	 * We can let go the lock for purposes of waiting.		\
	 * We re-acquire it after awaking so as to protect		\
	 * all semaphore operations.					\
	 *								\
	 * If "up()" is called before we call waking_non_zero() then	\
	 * we will catch it right away.  If it is called later then	\
	 * we will have to go through a wakeup cycle to catch it.	\
	 *								\
	 * Multiple waiters contend for the semaphore lock to see	\
	 * who gets to gate through and who has to wait some more.	\
	 */								\
	for (;;) {

#define DOWN_TAIL(task_state)			\
		tsk->state = (task_state);	\
	}					\
	tsk->state = TASK_RUNNING;		\
	remove_wait_queue(&sem->wait, &wait);

#define WAIT_QUEUE_HEAD(x) ((struct wait_queue *)((x)-1))

void cpqci_force_signal(struct wait_queue **q, int sig)
{
	struct task_struct *p;
	struct wait_queue *head, *next;

        if (!q) goto out;
	head = WAIT_QUEUE_HEAD(q);

	//read_lock(&waitqueue_lock);
	next = *q;
	if (!next) goto out;

	while (next != head) {
		p = next->task;
		next = next->next;
		force_sig(sig, p);
	}
//out_unlock:
//	read_unlock(&waitqueue_lock);
out:
	return;
}

int cpqci_down_interruptible(struct semaphore * sem, int ms)
{
	DOWN_VAR
	int ret = 0;
	DOWN_HEAD(TASK_INTERRUPTIBLE)

	ret = waking_non_zero_interruptible(sem, tsk);
	if (ret)
	{
		if (ret == 1)
			/* ret != 0 only if we get interrupted -arca */
			ret = 0;
		break;
	}
	ret = schedule_timeout(ms);
	if (ret == 0)
	{
		break;
	}
	DOWN_TAIL(TASK_INTERRUPTIBLE)
	return ret;
}
#endif

int cpqci_global_sem_destroy() 
{
	if (global_sem == NULL) return 0;

	//DBG("***sem_destroy: size %d\n", global_sem_size);

	//for (i=0; i<global_sem_size; i++) {
		//DBG("***force_signal %d\n", i);
		//cpqci_force_signal(&global_sem[i].wait, SIGKILL);
	//}
	kfree(global_sem);
	global_sem = NULL;
	global_sem_size = 0;
	return 0;
}

int cpqci_global_sem_create(int size) 
{
	int i;

	cpqci_global_sem_destroy(); 
	global_sem = kmalloc(sizeof(struct semaphore) * size, GFP_KERNEL);
	if (!global_sem) return -EFAULT;
	global_sem_size = size;
	for (i=0; i<size; i++) sema_init(global_sem + i, 0);
	return 0;
}

asm(
".align 4\n"
".globl cpqci_down_failed_interruptible\n"
"cpqci_down_failed_interruptible:\n\t"
	"pushl %edx\n\t"
	"pushl %ebx\n\t"
	"pushl %ecx\n\t"
	"call cpqci_down_interruptible\n\t"
	"popl %ecx\n\t"
	"popl %ebx\n\t"
	"popl %edx\n\t"
	"ret"
);

int down_timeout(struct semaphore * sem, int ms)
{
	int result;

#if WAITQUEUE_DEBUG
	CHECK_MAGIC(sem->__magic);
#endif

	__asm__ __volatile__(
		"# atomic interruptible down operation\n\t"
		LOCK "decl %1\n\t"     /* --sem->count */
		"js 2f\n\t"
		"xorl %0,%0\n"
		"1:\n"
		".section .text.lock,\"ax\"\n"
		"2:\tcall cpqci_down_failed_interruptible\n\t"
		"jmp 1b\n"
		".previous"
		:"=a" (result), "=m" (sem->count)
		:"c" (sem), "b" (ms)
		:"memory");
	return result;
}

//END SEMAPHORE implementation

int sem_ioctl(struct inode *inod, struct file *f, unsigned int cmd, unsigned long arg)
{
	sem_value_pair data;
	struct semaphore* sem = NULL;
	data.rc = 0;

	if (arg) {
		DBG("copy_from_user %p %p\n", &data, arg);
		if (copy_from_user(&data, (sem_value_pair*)arg, sizeof(sem_value_pair)))  return -EFAULT; 
	}

	sem = (struct semaphore*) data.sem;
	if (sem == NULL && cmd != SEM_CREATE) return -EFAULT;
	if (sem != NULL && sem != global_sem) return -EFAULT;

	DBG("sem_ioctl start: %p, value = %x, rc = %x\n", sem, data.value, data.rc);

	switch (cmd) {
	case SEM_CREATE:
		data.rc = cpqci_global_sem_create(data.value);
		data.sem = global_sem;
		//flush_signals(current);
		//recalc_sigpending(current);
		DBG("SEM_CREATE %p\n", data.sem);
		break;
	case SEM_DESTROY:
		cpqci_global_sem_destroy();
		DBG("SEM_DESTROY %p\n", sem);
		break;
	case SEM_UP:
		up(sem + data.value);
		DBG("SEM_UP %p\n", sem + data.value);
		break;
	case SEM_DOWN:
		DBG("SEM_DOWN start %p\n", sem + data.value);
		data.rc = down_timeout(sem + data.value, -1);
		if (data.rc != 0) DBG("SEM_DOWN %d failure rc = %d\n", data.value, data.rc);
		if (signal_pending(current)) {
			DBG("SEM_DOWN signal pending\n");
			//flush_signals(current);
			//recalc_sigpending(current);
			data.rc = -EINTR;
		}
		DBG("SEM_DOWN stop %p\n", sem + data.value, data.rc);
		break;
	case SEM_DOWN_TIMER:
		DBG("SEM_DOWN_TIMER start %p ms %d\n", sem + data.value, data.rc);
		data.rc = down_timeout(sem + data.value, data.rc);
		if (data.rc != 0) DBG("SEM_DOWN_TIMER %d failure rc = %d\n", data.value, data.rc);
		if (signal_pending(current)) {
			DBG("SEM_DOWN_TIMER signal pending\n");
			//flush_signals(current);
			//recalc_sigpending(current);
			data.rc = -EINTR;
		}
		DBG("SEM_DOWN_TIMER stop %p %d\n", sem + data.value, data.rc);
		break;
	case SEM_GET_VALUE:
		data.rc= atomic_read(&((sem + data.value)->count));
		DBG("SEM_GET_VALUE %p %d\n", sem + data.value, data.rc);
		break;
	case SEM_SET_VALUE:
		sema_init(sem + data.value, data.rc);
		DBG("SEM_SET_VALUE %p %d\n", sem + data.value, data.rc);
		break;
	default:
		return -EFAULT;
	}

	DBG("sem_ioctl end: %p, value = %x, rc = %x\n", sem, data.value, data.rc);

	if (arg) {
		DBG("copy_to_user %p %p\n", arg, &data);
		if (copy_to_user((sem_value_pair*)arg, &data, sizeof(sem_value_pair)))  return -EFAULT; 
	}
	return 0;
}
