/*
 ***************************************************************************
 *        Copyright 2002 Compaq Information Technologies Group, L.P.       *
 ***************************************************************************
 *
 *  Title:    OS Utilities
 *
 *  Module:   cevtw_linux.c
 *
 *  Version:  1.0
 *
 *  Date:     8/14/2002
 *
 ***************************************************************************
 * Description:
 * 
 * The purpose of this file is to isolate OS specific details from the
 * primary code path, thus enabling an 80% common code base.  Examples of
 * functions to include in here are Physical Memory Mapping, Kernel Memory
 * allocation, device discovery, OS specific entry points, etc.
 *
 ***************************************************************************/

#include "oswrap.h"


/*
** We need "sprintf" defined for all OS's except Linux.
*/
#ifdef sprintf
#undef sprintf
#endif

#include "cpqevtw_linux.h"
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <asm-i386/string.h>
#include "cpqevtq.h"
#include "cpqevtc.h"

#ifdef CEVT_DESCRIPTION
MODULE_DESCRIPTION(CEVT_DESCRIPTION);
#endif

typedef int(* PRIB_ENTRY)(VOID *pCridData);

extern PCEVTDATA gpInstanceData;

ULONG cpqwEventHandler( LONG lAction, LONG lMatchCode,
                        LONG lEvtNum, VOID *pEvt );

ULONG cevtw_reg_iml_handler(ULONG ulAction, PIML_LOGGER pImlHndlr);

int (*callCridLog)(void* Log) = NULL;
int (*cevtw_CridLogEntry)(void* Log) = NULL;

EXPORT_SYMBOL_NOVERS(cpqwEventHandler);
EXPORT_SYMBOL_NOVERS(callCridLog);
EXPORT_SYMBOL_NOVERS(cevtw_CridLogEntry);
EXPORT_SYMBOL_NOVERS(cevtw_reg_iml_handler);

DECLARE_WAIT_QUEUE_HEAD(cevt_sleep_q);


/*
** FUNCTION DEFINITIONS
*/

static int cevt_init_one(void);
static int cevt_remove_one(void);
static int cevt_ioctl(struct inode* inode, struct file* fp, unsigned int cmd, unsigned long arg);
static int cevt_open(struct inode* inode, struct file* filep);
static int cevt_close(struct inode* inode, struct file* filep);
static void cevtw_free_instance(CEVTDATA *pInstance);
void cevtw_lock_iml_queue(PCEVTDATA pInstance);
void cevtw_unlock_iml_queue(PCEVTDATA pInstance);
void cevtw_lock_msg_queue(PCEVTDATA pInstance);
void cevtw_unlock_msg_queue(PCEVTDATA pInstance);
static void cevtw_free_instance(PCEVTDATA pInstance);
static int cevtw_RIB_interface(VOID *pCridData);



static int cevt_major = 0;

static struct file_operations cevt_ops = {	
	owner:	THIS_MODULE,
        ioctl:          cevt_ioctl,     /* ioctl */
        open:           cevt_open,      /* open */
        release:        cevt_close,     /* close */
};						


//************************************************************************
//      typdefs
//************************************************************************

// Obtained this #define from the CASM wrapper file for Linux. This is 
// needed in cevtw_microdelay().

#define CASM_USEC_IN_JIFFIES   (1000000 / HZ)

/*********************************************************************

@func  init_module

Routine Description:
This is a Linux specific routine which is called when the module
is loaded by the Linux operating system.


Return Value:


***********************************************************************/
int init_module(void)
{
	cevt_init_one();

	cevt_major = register_chrdev(0, "cevt", &cevt_ops);

        if (cevt_major < 0) {
           printk(KERN_WARNING "cevt:  Not able to get Major Number.  Error (%d)\n",
                  cevt_major);
           return (cevt_major);
        }  /* endif */

        inter_module_register("cevtw_reg_iml_handler", THIS_MODULE, cevtw_reg_iml_handler);

	//  Register an interfase for other modules to use to do logging
	inter_module_register("cpqwEventHandler", THIS_MODULE, cpqwEventHandler);
	//  Register this so the CRID driver can set it up.
        inter_module_register("callCridLog", THIS_MODULE, &callCridLog);

        return (0);

}  /* end init_module */


/*********************************************************************

@func  cleanup_module

Routine Description:
cleanup_module() entry point is called prior to unloading.
This will not be called if any channels are open.


Return Value:


***********************************************************************/
void cleanup_module(void)
{
	cevt_remove_one();
	unregister_chrdev(cevt_major, "cevt");
        inter_module_unregister("cevtw_reg_iml_handler");
        inter_module_unregister("cpqwEventHandler");
        inter_module_unregister("callCridLog");

}   /* end cleanup_module */


/*********************************************************************

@func  cevt_init_one

Routine Description:
This routine will initalize the device and the device driver.


Return Value:


***********************************************************************/
static int cevt_init_one(void)
{
   int            rv;
   struct pci_dev *pPciDev = NULL;
   PCEVTDATA      pInstance = NULL;
   POSDATA        pOsData = NULL;


/* Allocate and initialize a new instance structure. */
   gpInstanceData = (PCEVTDATA)kmalloc(sizeof (CEVTDATA), GFP_KERNEL);

   if (!gpInstanceData) return(-EAGAIN);

   pInstance = gpInstanceData;     /* Set local pointer for convenience */
   memset(pInstance, 0, sizeof(CEVTDATA));

   pInstance->pOsData = (PVOID)kmalloc(sizeof (OSDATA), GFP_KERNEL);
   if (!pInstance->pOsData) return(-EAGAIN);

   pOsData = (POSDATA)pInstance->pOsData;
   memset(pOsData, 0, sizeof(OSDATA));

   spin_lock_init(&pOsData->imlq_lock);
   spin_lock_init(&pOsData->msgq_lock);

   init_waitqueue_head(&pOsData->task_wq);

   cevtw_CridLogEntry = (PRIB_ENTRY)cevtw_RIB_interface;

   if ( (cevtc_common_init(pInstance)) ) {
      cevtw_free_instance(pInstance); 
      return (EOPNOTSUPP);
   }  /* endif */


/* Announce our arrival! */
   printk("cevt:   hp ProLiant Event Logging Driver (rev %s)\n", EFS_VER);

   return (SUCCESS);

}  /* cevt_init_one */


/*********************************************************************

@func  cevt_remove_one

Routine Description:
This routine will tear down the driver when the driver is unloaded.


Return Value:


***********************************************************************/
static int cevt_remove_one(void)
{
#ifdef THIS_NEEDS_TO_BE_DONE
	cevtc_common_shutdown(gpInstanceData);
#endif

   cevtw_CridLogEntry = NULL;
	cevtw_free_instance(gpInstanceData);
	return (0);
}   /* end cevt_remove_one */


/*********************************************************************

@func  cevt_open

Routine Description:
The routine to process "open" requests from application space.


Return Value:


***********************************************************************/
static int cevt_open(struct inode* inode, struct file* filep)
{
/*
** Most drivers don't need to check for privilege, but this one
** controls a critical system resource. Writing to this device
** requires privilege.
*/
   if (!suser()) return (-EPERM);
   if ( (cevtc_common_open(gpInstanceData)) ) return(-EOPNOTSUPP);

#ifndef DEBUG
   MOD_INC_USE_COUNT;
#endif

   return (0);

} /* END cevt_open */



/*********************************************************************

@func  cevt_close

Routine Description:
This routine processes "close" requests from application space


Return Value:


***********************************************************************/
static int cevt_close(struct inode* inode, struct file* filep)
{

   if ( (cevtc_common_close(gpInstanceData)) ) return (-EBADF);
#ifndef DEBUG
   MOD_DEC_USE_COUNT;
#endif

   return (0);

}  /* END cevt_close */




/*********************************************************************

@func  cevt_ioctl

Routine Description:
This routine processes IOCTL requests.


Return Value:


***********************************************************************/
static int cevt_ioctl(struct inode* inode, struct file* fp, unsigned int cmd, unsigned long arg)
{
   ULONG   ulDevNode =  MINOR( inode->i_rdev );
   if ( (cevtc_common_ioctl(cmd, ulDevNode, (void *)arg, 0, gpInstanceData)) )
      return ( -EPERM );
   else
      return ( SUCCESS );

} /* END cevt_ioctl */



/*********************************************************************
** 
** @func  cevtw_free_instance
** 
** Routine Description:
** This routine will clean up all the various things which were started
** or allocated during the initialization phase of this driver.
** 
** Return Value:
** Standard DDI Return codes
** 
***********************************************************************/
static void cevtw_free_instance(PCEVTDATA pInstance)
{
   int32_t     ulMemSize;
   int32_t     i;

/*
** Let's make sure we haven't already done this once already.
*/
   if (pInstance == NULL)
      return;

   if (pInstance->pOsData)
      kfree(pInstance->pOsData);
   kfree(pInstance);
   pInstance = NULL;
   gpInstanceData = NULL;


} /* END cevtw_free_instance */




/***************************************************************************
 ****************       START OF "MUST HAVE" FUNCTION    *******************
 ***************************************************************************/


/*********************************************************************
** 
** @func  cevtw_RIB_interface
** 
** Routine Description:
** This routine is used as a "shim" between the "casm" driver and the
** RIB ("crid") driver. This routine also makes the call to log the
** message to the Console.
** 
** Return Value:
** 
***********************************************************************/
int cevtw_RIB_interface(VOID *pCridData)
{
   int rv;

   gpInstanceData->pRIBLogger = (PRIB_LOGGER)callCridLog;

   if (gpInstanceData->pRIBLogger == NULL) 
      rv = cevtw_find_RIB_entry(gpInstanceData);

   if (gpInstanceData->pRIBLogger)
       gpInstanceData->pRIBLogger(pCridData);

   cevtc_console_log(pCridData);

   return(SUCCESS);

}  /* end cevtw_RIB_interface */


/*********************************************************************
** 
** @func  cevtw_reg_iml_handler
** 
** Routine Description:
** This routine is used to register the "cevt" device driver's IML
** logging routine (handler).  This is the shim used to support the
** legacy storage drivers.
** 
** Return Value:
**       0   : SUCCESS
**       1   : FAILURE
** 
***********************************************************************/
ULONG cevtw_reg_iml_handler(ULONG ulAction, PIML_LOGGER pImlHndlr)
{
   ULONG  retVal = SUCCESS;

   cevtw_lock_iml_queue(gpInstanceData);

   if (gpInstanceData->pImlLogger) {
      retVal = FAILURE;

   } else if (ulAction == CEVT_ADD_HANDLER) {
      gpInstanceData->pImlLogger = pImlHndlr;
      cevtc_common_iml_reg(gpInstanceData);

   } else if (ulAction == CEVT_REMOVE_HANDLER) {
      gpInstanceData->pImlLogger = NULL;
      gpInstanceData->pRIBLogger = NULL;
   /*
   ** We setup up for the HEALTH RELOAD state in the
   ** event the health driver is stopped and started.
   */
      gpInstanceData->ulCfgState |= CEVT_HEALTH_RELOAD;

   } else {
      retVal = FAILURE;
   }  /* endif */

   cevtw_unlock_iml_queue(gpInstanceData);

   return (retVal);

}  /* end cevtw_reg_iml_handler */


/*********************************************************************
** 
** @func  cpqwEventHandler
** 
** Routine Description:
** This routine is unusual in that we have to have it because
** DDI 8 device driver use a "#pragma weak" to locate the 
** health driver for event logging.  This is here purely to
** support legacy device drivers.
** 
** Return Value:
**       0   : SUCCESS
**       1   : FAILURE
** 
***********************************************************************/
ULONG cpqwEventHandler( LONG lAction, LONG lMatchCode,
                              LONG lEvtNum, VOID *pEvt )
{

   return ( (ULONG)cevtc_log_iml_event(lAction, lMatchCode, lEvtNum, pEvt) );

}  /* end cpqwEventHandler */

/*********************************************************************
** 
** @func  cevtw_malloc
** 
** Routine Description:
** This routine will allocate kernel memory for 
** driver use.  This is NOT mapped in memory but
** allocated from the kernel pools.
** 
** Return Value:
** Pointer to allocated memory or NULL if this fails.
** 
***********************************************************************/
PVOID cevtw_malloc(ULONG ulSize)
{
   PVOID  pVMem;
   ULONG  ulOrder;

   if (gpInstanceData->ulFlags & CEVT_DEBUG_ALL)
      printk(KERN_DEBUG "cevtw_malloc:  Asking for %d bytes\n", ulSize);

   ulOrder = get_order(ulSize);
   if (ulOrder > 5) {
      pVMem = (PVOID)__get_free_pages(GFP_ATOMIC, ulOrder);
   } else {
      pVMem = (PVOID)kmalloc(ulSize, GFP_ATOMIC);
   }  /* endif */

   if (gpInstanceData->ulFlags & CEVT_DEBUG_ALL)
      printk(KERN_DEBUG "cevtw_malloc:  %p (size %d)\n",pVMem, ulSize);

   return (pVMem);

}  /* end cevtw_malloc */


/*********************************************************************
** 
** @func  cevtw_free
** 
** Routine Description:
** This routine will free kernel memory previously
** allocated for driver use.  This is NOT mapped in memory but
** allocated from the kernel pools.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
VOID cevtw_free(PVOID pVAddr, ULONG ulSize)
{
   ULONG  ulOrder;

   if (gpInstanceData->ulFlags & CEVT_DEBUG_ALL)
      printk(KERN_DEBUG "cevtw_free:  %p (size %d)\n",pVAddr, ulSize);

   ulOrder = get_order(ulSize);
   if (ulOrder > 5) {
      free_pages((ULONG)pVAddr, ulOrder);
   } else {
      kfree (pVAddr);
   }  /* endif */

}  /* end cevtw_free */

/*********************************************************************
** 
** @func  cevtw_memcpy
** 
** Routine Description:
** This is an OS specific routine which will copy from Source to 
** destination for a certain amount of bytes.  No Alignment is
** assumed so individual OS's may need to double buffer to assure
** alignment if required.
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
ULONG cevtw_memcpy(char *dest, char *source, int length)
{
        memcpy(dest, source,  length);
        return ( (ULONG) dest );

}  /* end cevtw_memcpy */



/*********************************************************************
** 
** @func  cevtw_strcpy
** 
** Routine Description:
** This is an OS specific routine which will copy from Source to 
** destination until a NULL character is received.  No alignment is
** assumed so individual OS's may need to double buffer to assure
** alignment if required.
** 
** Return Value:
** Byte pointer to the Destination
** 
***********************************************************************/
char *cevtw_strcpy (char *s1, char *s2)
{

   return ( (char *) strcpy(s1,s2));

} /* end cevtw_strcpy */



/*********************************************************************
** 
** @func  cevtw_strcat
** 
** Routine Description:
** This is an OS specific routine which will concatenate from Source to 
** destination until a NULL character is received.  No alignment is
** assumed so individual OS's may need to double buffer to assure
** alignment if required.
** 
** Return Value:
** 
** Byte pointer to the destination.
** 
***********************************************************************/
char *cevtw_strcat (char *s1, char *s2)
{

   return ( (char *) strcat(s1,s2));

}  /* end cevtw_strcat */


/*********************************************************************
** 
** @func  cevtw_strlen
** 
** Routine Description:
** This is an OS specific routine which return the length of a null
** terminated string. No alignment is assumed so individual OS's may 
** need to double buffer to assure alignment if required.
** 
** Return Value:
** 
** String Length
** 
***********************************************************************/
ulong cevtw_strlen (char *s1)
{

   return ( (ulong) strlen(s1) );

} /* end cevtw_strlen */



/*********************************************************************
** 
** @func  cevtw_memzero
** 
** Routine Description:
** This OS Specific routine will "zero out" a memory location for the
** length specified.
** No alignment is assumed so individual OS's may need to double buffer 
** to assure alignment if required.
** 
** Return Value:
** 
** VOID pointer to memory addressed passed in.
** 
***********************************************************************/
void *cevtw_memzero(void *s, int n)
{
   memset(s,0,n);
   return (s);
}  /* end cevtw_memzero */


/*********************************************************************
** 
** @func  cevtw_memset
** 
** Routine Description:
** This is an OS Specific routine which will set memory to a particular
** value as passed in. 
** No alignment is assumed so individual OS's may need to double buffer 
** to assure alignment if required.
** 
** Return Value:
** VOID pointer to memory addressed passed in.
**
**
***********************************************************************/
void * cevtw_memset(  void *s, register int c, register int n)
{
   memset(s,c,n);
   return (s);

}  /* end cevtw_memset */

/*********************************************************************
** 
** @func  cevtw_strncmp
** 
** Routine Description:
** This is an OS Specific routine which will compare two strings for
** a specified length.  Follows the standard C Library for "strncmp".
** No alignment is assumed so individual OS's may need to double buffer 
** to assure alignment if required.
** 
** Return Value:
** VOID pointer to memory addressed passed in.
** 
***********************************************************************/
int cevtw_strncmp(register char *s1, register char *s2, register int length)
{
   char *end;

   for (end = s1 + length - 1; s1 < end && *s1 - *s2 == 0; s1++, s2++);
   return((int)(*s1 - *s2));

} /* end cevtw_strncmp */


/*********************************************************************
** 
** @func  cevtw_printf
** 
** Routine Description:
** This is an OS Specific routine which emulate a printf command for a
** kernel driver.  
** The arguments to this driver are the following:
** 
** ulErrorLevel   :  The error level which to log.  
** pString        :  The pointer to the string to pass to "cmn_err".
** ppParms        :  The Address of where the parameters are on the stack.
**                   This is set by the calling routine since the parameters
**                   are on the stack before the following the "pString"
**                   pointer.
** pInstance      :  The "Instance Data" for this driver which is passed to
**                   all routines in this module.
** 
** 
** 
** Return Value:
** Nothing 
** 
***********************************************************************/
VOID cevtw_printf( ULONG ulErrorLevel, PCHAR pString, 
                   PULONG ppParms, PCEVTDATA pInstance )
{
   PULONG p1;
   PULONG p2;
   PULONG p3;
   PULONG p4;
   PULONG p5;
   PULONG p6;
   PULONG p7;
   PULONG p8;
   PULONG p;

   ULONG ulStringLen;
   PCHAR pAlloc;
   PCHAR pCmnMsg;

   /*
   ** Insert the string "cevt: " to the OS_OutErr() messages to 
   ** identify the driver.
   */
   ulStringLen = (cevtw_strlen(pString)) + 21;
   pAlloc = (PCHAR)cevtw_malloc(ulStringLen);
   if (pAlloc)
   {
      pCmnMsg = pAlloc;

   /*
   ** Now make the call to printk with the appropriate error level set
   */

      switch (ulErrorLevel) {

         case CEVT_KERN_INFO:
            cevtw_strcpy(pCmnMsg, KERN_NOTICE);
            cevtw_strcat(pCmnMsg, " NOTICE: cevt: ");
            break;

         case CEVT_KERN_WARN:
            cevtw_strcpy(pCmnMsg, KERN_WARNING);
            cevtw_strcat(pCmnMsg, " WARNING: cevt: ");
            break;

         case CEVT_KERN_DEBUG:
            cevtw_strcpy(pCmnMsg, KERN_DEBUG);
            cevtw_strcat(pCmnMsg, " DEBUG: cevt: ");
            break;

         case CEVT_KERN_CRITICAL:
            cevtw_strcpy(pCmnMsg, KERN_CRIT);
            cevtw_strcat(pCmnMsg, " CRITICAL: cevt: ");
            break;

         case CEVT_KERN_PANIC:
            cevtw_strcpy(pCmnMsg, KERN_EMERG);
            cevtw_strcat(pCmnMsg, " PANIC: cevt: ");
            break;

         default:
            printk(KERN_WARNING "ERROR! cevt:  Invalid Error Level from Common Code:  %d\n",
                                ulErrorLevel );
            cevtw_strcpy(pCmnMsg, KERN_WARNING);
            cevtw_strcat(pCmnMsg, " ERROR: cevt: ");
            break;

      }  /* endswitch */

      cevtw_strcat(pCmnMsg, pString);
      cevtw_strcat(pCmnMsg, "\n");
   } else {
      pCmnMsg = pString;
   }  /* endif */

/*
** Fixup the pointers so they point to upto 8 32bit input parameters.
** The way cmn_err works, it expects the parameters to be located on the
** stack immediately following the pointer.  We will need to copy the
** parameters from the original location on the stack (as pointed to
** by ppParms) to the current stack pointer location when we make the
** call to cmn_err.
*/

   p = p1 = ppParms;
   p2 = ++p;
   p3 = ++p;
   p4 = ++p;
   p5 = ++p;
   p6 = ++p;
   p7 = ++p;
   p8 = ++p;

/*
** Now make the call to printk
*/

   printk(pCmnMsg,*p1,*p2,*p3,*p4,*p5,*p6,*p7,*p8); 

   if (pAlloc)
      cevtw_free(pAlloc, ulStringLen);

}  /* end cevtw_printf */


/*********************************************************************
** 
** @func  cevtw_lock_iml_queue
** 
** Routine Description:
** This routine is OS specific and will take a synchronization lock for
** access to the IML queue.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void cevtw_lock_iml_queue(PCEVTDATA pInstance)
{
    POSDATA pOsData = (POSDATA)pInstance->pOsData;
    spin_lock_irqsave(&pOsData->imlq_lock, pOsData->imlq_pl);
} /* cevtw_lock_iml_queue */


/*********************************************************************
** 
** @func  cevtw_unlock_iml_queue
** 
** Routine Description:
** This routine is OS specific and will free a synchronization lock for
** the IML queue.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void cevtw_unlock_iml_queue(PCEVTDATA pInstance)
{
    POSDATA pOsData = (POSDATA)pInstance->pOsData;

    spin_unlock_irqrestore(&pOsData->imlq_lock, pOsData->imlq_pl);
} /* cevtw_unlock_iml_queue */


/*********************************************************************
** 
** @func  cevtw_lock_msg_queue
** 
** Routine Description:
** This routine is OS specific and will take a synchronization lock for
** the Message queue which gets logged to the console.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void cevtw_lock_msg_queue(PCEVTDATA pInstance)
{
    POSDATA pOsData = (POSDATA)pInstance->pOsData;

    spin_lock_irqsave(&pOsData->msgq_lock, pOsData->msgq_pl);
} /* cevtw_lock_msg_queue */


/*********************************************************************
** 
** @func  cevtw_unlock_msg_queue
** 
** Routine Description:
** This routine is OS specific and will take a synchronization lock for
** the Message queue which gets logged to the console.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void cevtw_unlock_msg_queue(PCEVTDATA pInstance)
{
    POSDATA pOsData = (POSDATA)pInstance->pOsData;

    spin_unlock_irqrestore(&pOsData->msgq_lock, pOsData->msgq_pl);

} /* cevtw_unlock_msg_queue */


/*********************************************************************
** 
** @func  cevtw_wait_for_event
** 
** Routine Description:
** This routine is OS specific and will wait to be notified when something
** is placed on the queue.  When the the routine is released from here,
** one of two things has happened:
** 
** 1.  An Event has been put on the queue.
** 2.  The OS is shutting down.
** 
** If no Event is returned from the queue, the OS is shutting down and
** the application should return.
** 
** NOTE:  The QUEUE lock must be held before calling this routine.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
int cevtw_wait_for_event(PCEVTDATA pInstance)
{
    POSDATA     pOsData = (POSDATA)pInstance->pOsData;
    int		rv = SUCCESS;

    pInstance->ulWaitCondition = 0;
/*
** Make sure we are not shutting down before we go to
** sleep.
*/
    if (!(pInstance->ulCfgState & CEVT_SHUTDOWN)) {
       cevtw_unlock_msg_queue(pInstance);
       if (wait_event_interruptible(pOsData->task_wq, (pInstance->ulWaitCondition))) {
	  /*
	  ** We will set this to -1 as a special case to indicate that 
	  ** we got woken up by a signal. We could use it later.
	  */
          pInstance->ulWaitCondition = -1;
	  rv = FAILURE;
     }


   /*
   ** We need to obtain the lock again since we released it.
   */
       cevtw_lock_msg_queue(pInstance);

    }  /* endif */

    return (rv);
   
} /* cevtw_wait_for_event */


/*********************************************************************
**
** @func  cevtw_event_notify
** 
** Routine Description:
** This routine is OS specific and will notify the application waiting
** on the queue.
** 
** NOTE:  The QUEUE lock does NOT need to be held for this routine.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void cevtw_event_notify(PCEVTDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;
   pInstance->ulWaitCondition = 1;
   wake_up_interruptible(&pOsData->task_wq);

} /* cevtw_event_notify */

/*********************************************************************
**
** @func  cevtw_microdelay
**
** Routine Description:
** This is an OS specific routine which will delay a specific amount of
** time.  The time passed in is in Microseconds.
**
**
** Return Value:
**
** Nothing
**
***********************************************************************/
ULONG cevtw_microdelay( ULONG ulUsec, PCEVTDATA pInstance )
{
   ULONG ulDelay;

   if (pInstance->ulFlags & CEVT_DEBUG_ALL)
      printk(KERN_DEBUG "cevtw_microdelay:  delay:  %d\n", ulUsec);


   if (ulUsec < 1000) {           /* one millisecond       */
      udelay (ulUsec);

   } else if (ulUsec < 15000) {   /* Slightly longer wait  */
      ulDelay = ulUsec / 1000;    /* convert to mSec       */
      mdelay (ulDelay);

   } else {                       /* We just need to sleep */
      ulDelay = ulUsec / CASM_USEC_IN_JIFFIES;
      if (!ulDelay)
         ulDelay = 2;        /* Min sleep time: > 1 JIFFY  */

      interruptible_sleep_on_timeout(&cevt_sleep_q, ulDelay);

   }  /* endif */

   return (ulUsec);

}  /* end cevtw_microdelay */




/*********************************************************************
** 
** @func  cevtw_update_entry_points
** 
** Routine Description:
** This routine will register the entry points for other drivers to
** log events to the hp ProLiant Integrated Management Log (IML).
** Initializes the dynamic event log support advertisement
** mechanism. This is used by other drivers who wish to
** call the event log handler but do not wish to declare
** it as an external symbol.
** 
** Return Value:
**       0   : SUCCESS
**       1   : FAILURE
** 
***********************************************************************/
int cevtw_update_entry_points(PCEVTDATA pInstance)
{

   return (SUCCESS);

}  /* end cevtw_update_entry_points */



/*********************************************************************
** 
** @func  cevtw_find_RIB_entry
** 
** Routine Description:
** This routine will hook the legacy "crid" entry points used to
** log event to the hp ProLiant Remote Insight Board.
** 
** Return Value:
**       0   : SUCCESS
**       1   : FAILURE
** 
***********************************************************************/
int cevtw_find_RIB_entry(PCEVTDATA pInstance)
{

   int           retVal = FAILURE;

   if (callCridLog != NULL) {
      gpInstanceData->pRIBLogger = (PRIB_LOGGER)callCridLog;
      retVal = SUCCESS;

   }  /* endif */

   return (retVal);

}  /* end cevtw_find_RIB_entry */



/*********************************************************************
** 
** @func  cevtw_get_ubuff
** 
** Routine Description:
** This is an OS Specific routine used to copy a buffer from user space
** to kernel space.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int cevtw_get_ubuff( void *pUserBuff,          /* addr to copy from   */
                     void *pKernelBuff,        /* addr to copy to     */
                     ULONG buffer_len,         /* how much to copy    */
                     PCEVTDATA pInstance )     /* OS Specific Data    */
{
   int            rv = SUCCESS;

   rv = (copy_from_user((void*)(pKernelBuff), (void*)(pUserBuff), (buffer_len))?-1:0);

   return (rv);

}  /* end cevtw_get_ubuff */


/*********************************************************************
** 
** @func  cevtw_put_ubuff
** 
** Routine Description:
** This is an OS Specific routine used to copy a buffer from kernel space
** to user space.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int cevtw_put_ubuff( void *pUserBuff,          /* addr to copy from   */
                     void *pKernelBuff,        /* addr to copy to     */
                     ULONG buffer_len,         /* how much to copy    */
                     PCEVTDATA pInstance )     /* OS Specific Data    */
{
   int            rv = SUCCESS;

   rv = (copy_to_user((void*)(pUserBuff), (void*)(pKernelBuff), (buffer_len))?-1:0);

   return (rv);

}  /* end cevtw_put_ubuff */

