/*
 ***************************************************************************
 *        Copyright 2002 Compaq Information Technologies Group, L.P.       *
 ***************************************************************************
 *
 *  Title:    OS Utilities
 *
 *  Module:   casmw_linux.c
 *
 *  Version:  1.0
 *
 *  Date:     08/14/2002
 *
 ***************************************************************************
 * Description:
 * 
 * The purpose of this file is to isolate OS specific details from the
 * primary code path, thus enabling an 80% common code base.  Examples of
 * functions to include in here are Physical Memory Mapping, Kernel Memory
 * allocation, device discovery, OS specific entry points, etc.
 *
 ***************************************************************************/

#include "oswrap.h"

/*
** We need "sprintf" defined for all OS's except Linux.
*/
#ifdef sprintf
#undef sprintf
#endif

#include "casmw_linux.h"
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <asm-i386/string.h>
#include <asm-i386/system.h>

#ifdef CASM_NMI_DEBUG
#include <linux/interrupt.h>
#include <asm-i386/hardirq.h>
#include <asm-i386/smplock.h>
#endif

#include "casmc.h"

#ifdef CASM_DESCRIPTION
MODULE_DESCRIPTION(CASM_DESCRIPTION);
#endif

#ifdef CASM_NMI_DEBUG
extern unsigned char global_irq_holder;
#endif

PCASMDATA gpInstanceData = NULL;
char      gPrintfBuff[256];      /* used to format messages */

extern int (*cevtw_CridLogEntry)(void* Log);
extern ULONG cevtw_reg_iml_handler(ULONG ulAction, void * pImlHndlr);

#define CEVT_ADD_HANDLER       0x01
#define CEVT_REMOVE_HANDLER    0x02

LONG  casmevt_iml_log( int iAction, int iMatchCode,
                       int iEventNumber, PVOID pEvt );

DECLARE_WAIT_QUEUE_HEAD(casm_sleep_q);
DECLARE_MUTEX(casm_proc_sem);

static int casm_init_one(void);
static int casm_remove_one(void);
static int casm_ioctl(struct inode* inode, struct file* fp, unsigned int cmd, unsigned long arg);
static int casm_open(struct inode* inode, struct file* filep);
static int casm_close(struct inode* inode, struct file* filep);
static void casmintr(int irq, void* instance, struct pt_regs* regs);
static int casmw_poweroff_handler(struct notifier_block *nb, unsigned long code, void *p);
static int casmw_nmi_handler(UCHAR reason, struct pt_regs *pregs, PCASMDATA pInstance);
static void casmw_free_instance(CASMDATA *pInstance);
static int casmw_check_irq (struct pci_dev *pPciDev, PCASMDATA pInstance);
static int casm_poweroff_handler(struct notifier_block *nb, unsigned long code, void *p);

void* get_gate(unsigned int n, PCASMDATA pInstance);

/* Missing from casmc.h */
int casmw_wait_for_event(PCASMDATA pInstance);
int casmw_mca_thread_wait(PCASMDATA pInstance, ULONG ulProcId);

/*
 * Driver information structures.
 */

struct proc_dir_entry    *cpqtemp_pd   = NULL;
struct proc_dir_entry    *cpqfan_pd    = NULL;
struct proc_dir_entry    *cpqpwr_pd    = NULL;
struct proc_dir_entry    *cpqiml_pd    = NULL;
struct proc_dir_entry    *cpqpci_pd    = NULL;
struct proc_dir_entry    *casmdbug_pd  = NULL;

 struct notifier_block casm_notifier = {
	casm_poweroff_handler,
	NULL,
	0
};


static int casm_major = 0;

static struct file_operations casm_ops = {	
	owner:		THIS_MODULE,
	ioctl:		casm_ioctl,     /* ioctl */		
	open:		casm_open,	/* open */		
	release: 	casm_close,	/* close */		
};						

/****************************************************************************
**                 ***** Special Linux Interrupt Handlers ******           **
*****************************************************************************
** The following routines are used to hook IRQ13 (for Legacy ASM systems), **
** NMI and MCE Traps.                                                      **
*****************************************************************************/


#define FIRST_EXTERNAL_VECTOR 0x20

/*
 * Obtain __KERNEL_DS from <asm-i386/segment.h>.
 */
static int my_kernel_ds = __KERNEL_DS;

#define CASM_SAVE_ALL \
	"cld\n\t" \
	"pushl %es\n\t" \
	"pushl %ds\n\t" \
	"pushl %eax\n\t" \
	"pushl %ebp\n\t" \
	"pushl %edi\n\t" \
	"pushl %esi\n\t" \
	"pushl %edx\n\t" \
	"pushl %ecx\n\t" \
	"pushl %ebx\n\t" \
	"movl my_kernel_ds,%edx\n\t" \
	"movw %dx,%ds\n\t" \
	"movw %dx,%es\n\t"

#define CASM_RESTORE_ALL	\
	"popl %ebx\n\t"	\
	"popl %ecx\n\t"	\
	"popl %edx\n\t"	\
	"popl %esi\n\t"	\
	"popl %edi\n\t"	\
	"popl %ebp\n\t"	\
	"popl %eax\n\t"	\
	"popl %ds\n\t"	\
	"popl %es\n\t"	\
	"addl $4,%esp\n\t"	

//************************************************************************
//	typdefs
//************************************************************************

#define CASM_USEC_IN_JIFFIES   (1000000 / HZ)

//************************************************************************
//	globals
//************************************************************************

static void* kernel_nmi_address;        /* Orig. NMI Address in IDT     */
static void* kernel_mce_address;        /* Orig. MCE Address in IDT     */
static void* kernel_int13_address;      /* Orig. IRQ13 Addr in IDT      */

static desctableptr_s idt_base;


//************************************************************************
//	funcs
//************************************************************************

asmlinkage void my_nmi();
asmlinkage void my_int13();
asmlinkage void my_mce();


/*********************************************************************

@func  retrieve_idt_base

Routine Description:
This routine will get the base address of the IDT table so we can 
put ourselves into the chain.


Return Value:

   0  : SUCCESS
   1  : FAILURE


***********************************************************************/
static int retrieve_idt_base(PCASMDATA pInstance)
{
   int            i;
   POSDATA        pOsData = (POSDATA)pInstance->pOsData;

   __asm__ __volatile__("sidt idt_base");

   if (idt_base.limit == 0) {
      printk(KERN_CRIT "Did not retrieve the idt base correctly idt_base.base=%xh\n", idt_base.base);
      return (FAILURE);
   } else {
      if (pInstance->ulFlags & CASM_DEBUG_ALL)
         printk(KERN_DEBUG "idt_base.base=%xh idt_base.limit=%xh\n", idt_base.base, idt_base.limit);
   }  /* endif */

   pOsData->idt_table = (gatedesc_s*)(idt_base.base);
   if (pOsData->idt_table == 0) {
      printk(KERN_CRIT "Did not map to the idt table correctly\n");
      return (FAILURE);
   } else {
      if (pInstance->ulFlags & CASM_DEBUG_ALL) {
         printk(KERN_DEBUG "pOsData->idt_table = %xh\n", pOsData->idt_table);
         printk(KERN_DEBUG "&idt_base = %xh\n", &idt_base);
         for (i=0; i<48; i++) {
            printk(KERN_DEBUG "offset_low:%04xh selector:04%xh\n", 
                              pOsData->idt_table[i].offset_low, 
                              pOsData->idt_table[i].selector);
            printk(KERN_DEBUG "pad %02x p_dpl_type: %02x offset_high:%04xh\n", 
                              pOsData->idt_table[i].pad, 
                              pOsData->idt_table[i].p_dpl_type, 
                              pOsData->idt_table[i].offset_high);
            printk(KERN_DEBUG "gate(%d) = (phys)%xh\n", i, (unsigned) get_gate(i, pInstance));

         }  /* endfor */

      }  /* endif */

   }  /* endif */

   memcpy(&pOsData->idt_base, &idt_base, sizeof(desctableptr_s));

   return (SUCCESS);

}  /* end retrieve_idt_base */




/*********************************************************************

@func  set_gate_phys

Routine Description:
This routine will load the address of the passed in routine into
the IDT gate.


Return Value:


***********************************************************************/
void set_gate_phys(unsigned int n, void *addr, PCASMDATA pInstance)
{
   ULONG ulOrigCR0;
   ULONG ulFlags;
   BOOLEAN bReset = 0;                  /* FALSE */
   POSDATA  pOsData = (POSDATA)pInstance->pOsData;

   /*
   ** If the WP bit (CR0, bit16) is set then we need to clear it temporarily
   ** to be able to add the Health driver's handlers (NMI, MCE, IRQ13) to 
   ** the IDT. This is similar to a workaround for the F00F bug.
   */
   ulOrigCR0 = read_cr0();

   if (ulOrigCR0 & 0x00010000) {
      local_irq_save(ulFlags);
      write_cr0(ulOrigCR0 & 0xfffeffff);
      bReset = 1;
   } /* endif */
 
   pOsData->idt_table[n].offset_high = (((unsigned long)addr) >> 16);
   pOsData->idt_table[n].offset_low = (((unsigned long)addr) & 0xffff);

   if (bReset) {
      write_cr0(ulOrigCR0);
      local_irq_restore(ulFlags);
   } /* endif */

}  /* end set_gate_phys */



/*********************************************************************

@func  get_gate

Routine Description:
This routine will return the address of the function for a specific
IDT gate.


Return Value:
Pointer to the function in the gate.

***********************************************************************/
void* get_gate(unsigned int n, PCASMDATA pInstance) 
{
   POSDATA   pOsData = (POSDATA)pInstance->pOsData;

	unsigned long retVal = ( pOsData->idt_table[n].offset_high << 16) + 
                            pOsData->idt_table[n].offset_low;
	return (void*) retVal;
}


/*********************************************************************

@func  do_nmi

Routine Description:
This routine is called from the IDT entry (my_nmi).


Return Value:


***********************************************************************/
asmlinkage void do_nmi(struct pt_regs regs)
{

   unsigned char reason = inb(0x61);
   casmw_nmi_handler(reason, &regs, gpInstanceData);

}  /* end do_nmi */


/*********************************************************************

@func  do_mce

Routine Description:
This routine is called from the IDT entry (my_mce) for processing
of Machine Check Errors.


Return Value:


***********************************************************************/
asmlinkage void do_mce(struct pt_regs regs)
{
   casmw_mca_handler(&regs, gpInstanceData);

}  /* end do_mce */


/*********************************************************************

@func  do_IRQ

Routine Description:
This routine is called form the IDT Entry (my_int13) to process IRQ13
requests on older Linux Kernels.


Return Value:


***********************************************************************/
asmlinkage void do_IRQ(struct pt_regs regs)
{	
   PCASMDATA pInstance = gpInstanceData;
   casmintr(pInstance->ulIRQ, (void*)pInstance, &regs);

}  /* end do_IRQ */



/*********************************************************************

@func  hook_interrupts

Routine Description:
This routine will load our specific handlers into the IDT gates for
IRQ13, NMI and MCE.


Return Value:
Pointer to the function in the gate.

***********************************************************************/
void hook_interrupts(PCASMDATA pInstance)
{
/*
**first, retrieve the idt_base.
*/
   if (retrieve_idt_base(pInstance)) {
      printk(KERN_CRIT "Unable to load IDT table in steal_interrupts!");
      return;
   }  /* endif */

/*
**now, squirrel away original nmi and irq 13 vector
*/
   kernel_nmi_address = get_gate(2, pInstance); //NMI
   kernel_mce_address = get_gate(18, pInstance);

   if (pInstance->ulIRQ == 0xFF)
      kernel_int13_address = get_gate( (FIRST_EXTERNAL_VECTOR + 13),
                                       (pInstance) );

   if (pInstance->ulFlags & CASM_DEBUG_ALL) {
      printk(KERN_DEBUG "kernel_nmi_address:%04xh\n", kernel_nmi_address); 
      printk(KERN_DEBUG "kernel_mce_address:%04xh\n", kernel_mce_address); 
      printk(KERN_DEBUG "kernel_int13_address:%04xh\n", kernel_int13_address); 

   }  /* endif */

   if (pInstance->ulFlags & CASM_DEBUG_ALL) {
      printk(KERN_DEBUG "&my_nmi:%04xh\n",&my_nmi); 
      printk(KERN_DEBUG "&my_mce:%04xh\n",&my_mce); 
      printk(KERN_DEBUG "&my_int13:%04xh\n",&my_int13); 
   }  /* endif */

/*
** Add our vectors.
*/
  set_gate_phys(2, &my_nmi, pInstance);
  set_gate_phys(18, &my_mce, pInstance);

   if (pInstance->ulIRQ == 0xFF) {
      set_gate_phys(FIRST_EXTERNAL_VECTOR + 13, &my_int13, pInstance);
      pInstance->ulIRQ = 13;
   }  /* endif */


}  /* end hook_interrupts */


/*********************************************************************

@func  restore_interrupts

Routine Description:
This routine will reload the IDT gates to the original values


Return Value:

***********************************************************************/
void restore_interrupts(PCASMDATA pInstance) 
{

/*
**reestablish old physical vectors
*/
   if (kernel_nmi_address) {
      set_gate_phys(2, kernel_nmi_address, pInstance);
      kernel_nmi_address = (void *)0;
   }  /* endif */

   if (kernel_mce_address) {
      set_gate_phys(18, kernel_mce_address, pInstance);
      kernel_mce_address = (void *)0;
   }  /* endif */

   if (kernel_int13_address) {
      set_gate_phys(FIRST_EXTERNAL_VECTOR+13, kernel_int13_address, pInstance);
      kernel_int13_address = (void *)0;
   }  /* endif */

}  /* end restore_interrupts */



/*********************************************************************

@func  my_nmi

Routine Description:
This routine is loaded into the IDT and will be called whenever an
NMI is generated to the processor.


Return Value:


***********************************************************************/

__asm__("\n"\
".globl my_nmi\n"\
__ALIGN_STR"\n"\
"my_nmi:\n\t"\
	"pushl $0\n\t"\
	CASM_SAVE_ALL\
	"call "SYMBOL_NAME_STR(do_nmi)"\n\t"\
	CASM_RESTORE_ALL\
	"jmp *"SYMBOL_NAME_STR(kernel_nmi_address)"\n\t"\
);

/* end my_nmi */

/*********************************************************************

@func  my_int13

Routine Description:
This routine is loaded into the IDT and will be called whenever an
IRQ13 is generated to the processor.  This is only used on older
kernels when the request_irq() function fails.


Return Value:


***********************************************************************/
__asm__("\n"\
".globl my_int13\n"\
__ALIGN_STR"\n"\
"my_int13:\n\t"\
	"pushl $13-256\n\t"\
	CASM_SAVE_ALL\
	"call "SYMBOL_NAME_STR(do_IRQ)"\n\t"\
	CASM_RESTORE_ALL\
	"jmp *"SYMBOL_NAME_STR(kernel_int13_address)"\n\t"\
);

/* end my_int13 */

/*********************************************************************

@func  my_mce

Routine Description:
This routine is loaded into the IDT and will be called whenever an
MCE is generated by the processor.


Return Value:


***********************************************************************/

__asm__("\n"\
".globl my_mce\n"\
__ALIGN_STR"\n"\
"my_mce:\n\t"\
	"pushl $0\n\t"\
	CASM_SAVE_ALL\
	"call "SYMBOL_NAME_STR(do_mce)"\n\t"\
	CASM_RESTORE_ALL\
	"iret"
);

/* end my_mce */



/*********************************************************************

@func  casm_check_for_NMI

Routine Description:
The Linux operating system does NOT panic and halt the system when
critical NMI errors occur.  This routine will check the NMI bit at
legacy port 0x61 and if it is set, we will call our NMI handler and
halt the system.


Return Value:


***********************************************************************/
static void casmw_check_for_NMI(PCASMDATA pInstance)
{
   unsigned char    ucByte;
   struct   pt_regs ptregs;  /* dummy pointer to fake out handler */

   ucByte = casmw_read_byte(0x61, pInstance);

/*
** Port 0x61 is a legacy PC port.  The upper bit of the byte is the
** NMI status bit.  If this is set, we will call our standard NMI Handler
** and just halt the system right here.  Since the Health Driver only
** loads during multiuser boot, the user can work around this by booting
** in Single User (initstate 1 or maintenance) mode.
**
** We also check to see if bit 2 is set (Enable Parity Error NMI). This 
** should NOT be set by default.  If set, this may be an indicator that
** Linux took an NMI and turned off NMI processing.  Not generally a
** good idea.
*/
   if (ucByte & 0x84) {
      printk(KERN_CRIT "casm:  A SERR# / PERR# may have occurred prior to this driver loading!\n");
      printk(KERN_CRIT "casm:  Port 0x61 status: 0x%x - Attempting to source NMI\n", ucByte);
      casmw_nmi_handler(ucByte, &ptregs, pInstance); 
   }  /* endif */

}  /* end casmw_check_for_NMI */


/*********************************************************************

@func  init_module

Routine Description:
This is a Linux specific routine which is called when the module
is loaded by the Linux operating system.


Return Value:


***********************************************************************/
int init_module(void)
{
	int rv = 0;

   
	rv = casm_init_one();

	if (rv)
		return (rv);

	casm_major = register_chrdev(0, "cpqasm", &casm_ops);
	if (casm_major < 0) {
		printk(KERN_WARNING "casm:  Not able to get Major Number.  Error (%d)\n", casm_major);
		return (casm_major);
	}  /* endif */

	gpInstanceData->ulCfgState |= CASM_LOADED;

   return (0);

}  /* end init_module */


/*********************************************************************

@func  cleanup_module

Routine Description:
cleanup_module() entry point is called prior to unloading.
This will not be called if any channels are open.


Return Value:


***********************************************************************/
void cleanup_module(void)
{
	unregister_chrdev(casm_major, "cpqasm");
	casm_remove_one();
	gpInstanceData->ulCfgState &= ~CASM_LOADED;

}   /* end cleanup_module */



/*********************************************************************

@func  casm_init_one

Routine Description:
This routine will initalize the device and the device driver.


Return Value:


***********************************************************************/
static int casm_init_one(void)
{
   int            rv;
   int            NotDone = 1;
   struct pci_dev *pPciDev = NULL;
   PCASMDATA      pInstance = NULL;
   POSDATA        pOsData = NULL;


/*
** We need to initialize our PCI Device Structure and get our basic information.
*/

#ifdef DEBUG_INIT
   printk(KERN_DEBUG "START casm_init_one\n");
#endif

   while (NotDone) {

      pPciDev = pci_find_device(0x0E11, 0xA0F0, pPciDev);

      if (!pPciDev)
         NotDone = 0;
      else 
         if ( (pPciDev->subsystem_device == 0xB0F3) ||
              (pPciDev->subsystem_device == 0xF3B0) )
            NotDone = 0;

   }  /* endwhile */

   if (!pPciDev)
      pPciDev = pci_find_device(0x0E11, 0xB203, pPciDev);

   if (!pPciDev)
      pPciDev = pci_find_device(0x0E11, 0x00D7, pPciDev);

   if (!pPciDev) {
      printk(KERN_WARNING "casm:  This server is not supported by this driver!");
      printk(KERN_WARNING "casm:  Can not locate PCI Device 0x0e11a0f0 or PCI Device 0x0e11b203 or PCI Device 0x0e1100d7.");
      return (-ENODEV);
   }  /* endif */

/* Allocate and initialize a new instance structure. */
   gpInstanceData = (PCASMDATA)kmalloc(sizeof (CASMDATA), GFP_KERNEL);

   if (!gpInstanceData) return(-EAGAIN);

   pInstance = gpInstanceData;     /* Set local pointer for convenience */
   memset(pInstance, 0, sizeof(CASMDATA));

   pInstance->pOsData = (PVOID)kmalloc(sizeof (OSDATA), GFP_KERNEL);
   if (!pInstance->pOsData) return(-EAGAIN);
   pOsData = (POSDATA)pInstance->pOsData;
   memset(pOsData, 0, sizeof(OSDATA));

   spin_lock_init(&pOsData->rom_lock);

#ifdef DEBUG_INIT
   pInstance->ulFlags |= CASM_DEBUG_ALL;
   printk(KERN_DEBUG "casm_init_one:  spin_lock_init\n");
#endif

   spin_lock_init(&pOsData->q_lock);

   init_waitqueue_head(&pOsData->task_wq);
   init_waitqueue_head(&pOsData->mca_wq);

/*
** We need to initialize the number of CPU's on this server.
*/
   pInstance->ulCpuCount = smp_num_cpus;
   if (pInstance->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "Number of CPUs  %d\n", pInstance->ulCpuCount);
/*
** We need to fetch our Vendor / Device ID from the resmgr database.
*/
   if ( casmw_get_vend_device_id (pPciDev, pInstance) ) {
      casmw_free_instance(pInstance); 
      return (EOPNOTSUPP);
   } /* endif */

/*
** We need to fetch our I/O address from the resmgr database.
*/
   if ( casmw_get_io_addr (pPciDev, pInstance) ) {
      casmw_free_instance(pInstance); 
      return (EOPNOTSUPP);
   } /* endif */


/*
** We need to fetch our memory address from the PCI DEV structure
** This routine will also perform some clean up work to remove
** our previously allocated memory addresses.
*/
   if ( casmw_get_mem_addr (pPciDev, pInstance) ) {
      casmw_free_instance(pInstance); 
      return (EOPNOTSUPP);
   } /* endif */

   casmw_check_irq(pPciDev, pInstance);

/* Attach our interrupt handler */
   if (request_irq(pInstance->ulIRQ, casmintr, SA_SHIRQ|SA_INTERRUPT, "casm", pInstance) != 0) {
      printk(KERN_DEBUG "request_irq FAILED!\n");
   /*
   ** Some Linux Kernels will not allow us to hook IRQ13 directly.
   ** We will set the IRQ to 0xFF and directly hook the vector in the IDT.
   */
      if (pInstance->ulIRQ == 13) {
         printk(KERN_DEBUG "This is an IRQ13 system.\n");
         pInstance->ulIRQ = 0xFF;
      } else {
         printk(KERN_WARNING "casm:  IRQ (%d) registration FAILED\n",
                             pInstance->ulIRQ);
         casmw_free_instance(pInstance); 
         return (-EOPNOTSUPP);
      }  /* endif */

   }  /* endif */


#ifdef DEBUG_INIT
   printk(KERN_DEBUG "casmw_linux.c::  Calling casmc_common_init\n");
#endif

   if ( (casmc_common_init(pInstance)) ) {
      casmw_free_instance(pInstance); 
      return (EOPNOTSUPP);
   }  /* endif */

/* Register our Power Off Handler */
   register_reboot_notifier(&casm_notifier);

/* Attach our NMI, MCE and possibly IRQ13  handler to that of the system's */
   hook_interrupts(pInstance);

/* Initialize our "proc" file system entries */
   if (casmw_init_proc_fs(pInstance))
      printk(KERN_ALERT "casm:  Cannot initialize PROC file system entries!\n");

/* Announce our arrival! */
   printk("casm:   hp ProLiant Advanced Server Management Driver (rev %s)\n", EFS_VER);

/* Check to see if a pending NMI is present */
   casmw_check_for_NMI(pInstance);

   return (SUCCESS);

}  /* casm_init_one */


/*********************************************************************

@func  casm_remove_one

Routine Description:
This routine will tear down the driver when the driver is unloaded.


Return Value:


***********************************************************************/
static int casm_remove_one(void)
{
	gpInstanceData->ulCfgState |= CASM_SHUTDOWN;                           
   remove_proc_entry("cpqtemp", NULL);
   remove_proc_entry("cpqfan", NULL);
   remove_proc_entry("cpqpwr", NULL);
   remove_proc_entry("cpqpci", NULL);
   remove_proc_entry("casmdbug", NULL);
	unregister_reboot_notifier(&casm_notifier);
	casmc_common_shutdown(gpInstanceData);
	casmw_free_instance(gpInstanceData);
	return (0);
}   /* end casm_remove_one */



/*********************************************************************

@func  casm_open

Routine Description:
The routine to process "open" requests from application space.


Return Value:


***********************************************************************/
static int casm_open(struct inode* inode, struct file* filep) 
{
/*
** Most drivers don't need to check for privilege, but this one
** controls a critical system resource. Writing to this device
** requires privilege.
*/
   if (!suser()) return (-EPERM);
   if ( (casmc_common_open(gpInstanceData)) ) return(-EOPNOTSUPP);  

#ifndef DEBUG
   MOD_INC_USE_COUNT;
#endif
		
   return (0);

} /* END casm_open */



/*********************************************************************

@func  casm_close

Routine Description:
This routine processes "close" requests from application space


Return Value:


***********************************************************************/
static int casm_close(struct inode* inode, struct file* filep) 
{

   if ( (casmc_common_close(gpInstanceData)) ) return (-EBADF);
#ifndef DEBUG
   MOD_DEC_USE_COUNT;
#endif
   
   return (0);

}  /* END casm_close */



/*********************************************************************

@func  casm_ioctl

Routine Description:
This routine processes IOCTL requests.


Return Value:


***********************************************************************/
static int casm_ioctl(struct inode* inode, struct file* fp, unsigned int cmd, unsigned long arg)
{
   ULONG   ulDevNode =  MINOR( inode->i_rdev );
   if ( (casmc_common_ioctl(cmd, ulDevNode, (void *)arg, 0, gpInstanceData)) )
      return ( -EPERM );
   else
      return ( SUCCESS );

} /* END casm_ioctl */



/*********************************************************************

@func  casmintr

Routine Description:
This is the main routine.  This will call the
common code interrupt routine for processing.


Return Value:

    ISTAT_ASSERTED
    ISTAT_NONE


***********************************************************************/
static void casmintr(int irq, void* instance, struct pt_regs* regs)
{
    PCASMDATA p_instance = (PCASMDATA) instance;
    casmc_common_intr(p_instance);

}  /* end casmintr */





/*********************************************************************
** 
** @func  casmw_free_instance
** 
** Routine Description:
** This routine will clean up all the various things which were started
** or allocated during the initialization phase of this driver.
** 
** Return Value:
** Standard DDI Return codes
** 
***********************************************************************/
static void casmw_free_instance(PCASMDATA pInstance)
{
   int32_t     ulMemSize;
   int32_t     i;
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

/*
** Let's make sure we haven't already done this once already.
*/
   if (pInstance == NULL)
      return;

/*
** If we still haven't detached from our interrupt, then
** lets do that first.
*/

if (pInstance->ulIRQ != 0) free_irq(pInstance->ulIRQ, pInstance);
	pInstance->ulIRQ = 0;

/*
** Free up all remaining memory mappings
*/

   if (pOsData) {
      for (i=0; i < CASM_MAX_ADDR; i++) {
         if ( (pOsData->paddrs[i].phys_addr) &&
              (pOsData->paddrs[i].mem_len)  &&
              (pOsData->paddrs[i].virt_addr) ) {

            if (pOsData->paddrs[i].use_count)
               if (pInstance->ulFlags & CASM_DEBUG_ALL)
                  printk(KERN_DEBUG "casm: Physical Address: 0x%x,  use_count:  %d\n",
                          pOsData->paddrs[i].phys_addr,
                          pOsData->paddrs[i].use_count);

            iounmap(pOsData->paddrs[i].virt_addr);
            pOsData->paddrs[i].virt_addr = 0;

         }  /* endif */

      }  /* endfor */


   /* We need to put back the original IDT entries */
      restore_interrupts(pInstance);

   /* Now free up the OsData specific stuff as we are all done */
      kfree(pInstance->pOsData);
      pInstance->pOsData = (void *)0;

   }  /* endif */

   kfree(pInstance);
   pInstance = NULL;


} /* END casmw_free_instance */



/*********************************************************************
** 
** @func  casmw_check_irq
** 
** Routine Description:
** This routine checks to see if we have an IRQ assigned to the ASIC.
** This will also update the resmgr database for future accesses.  The
** IRQ must be correct in the resmgr database inorder to connect to it.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int casmw_check_irq (struct pci_dev *pPciDev, PCASMDATA pInstance)
{
    int         rv = SUCCESS;     /* return values */

    pInstance->ulIRQ = pPciDev->irq;              /* Save the IRQ we read off */

    if ( (pInstance->ulIRQ == 0xFF) || (pInstance->ulIRQ == 0) ) {
       pInstance->ulIRQ = 13;

    }  /* endif */

    return (rv);

}  /* end casmw_check_irq */


/*********************************************************************
** 
** @func  casmw_update_mapped_entries
** 
** Routine Description:
** This routine will add a specific physical address and range to the list
** of addresses in the mapped address array  for this driver.
** 
** Return Value:
** 
** SUCCESS:  A number between 0 and CASM_MAX_ADDR
** FAILURE:  0xFF
** 
***********************************************************************/
BOOLEAN casmw_update_mapped_entries (ULONG ulAddr, ULONG ulLen, PCASMDATA pInstance)
{
   int         iArrayIndex = 0xFF;
   int         i;
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;
   BOOLEAN     bRet = SUCCESS;




/*
** We first look to find an empty spot.  There will never be gaps because
** updates to the memory addresses are always added to the end.
*/
   for (i=0; i < CASM_MAX_ADDR; i++) {
      if (pOsData->paddrs[i].phys_addr == 0) {

         pOsData->paddrs[i].phys_addr   = ulAddr;
         pOsData->paddrs[i].mem_len     = ulLen;
         pOsData->paddrs[i].virt_addr   = 0;
         pOsData->paddrs[i].mem_blk_num = i;

         break;

      }  /* endif */

   }  /* endfor */

   return (bRet);

}  /* casmw_update_mapped_entries */


/*********************************************************************
** 
** @func  casmw_mapped_entry
** 
** Routine Description:
** This routine will check the list of known mapped entries and return
** the array index for the existing address.  If this address has never
** been registered, then 0xFF will be returned.
** 
** Return Value:
** 
** SUCCESS:  A number between 0 and CASM_MAX_ADDR
** FAILURE:  0xFF
** 
***********************************************************************/
int casmw_mapped_entry(ULONG ulAddr, ULONG ulLen, PCASMDATA pInstance)
{
   int      iArrayIndex = 0xFF;
   int      i;
   POSDATA  pOsData = (POSDATA)pInstance->pOsData;

   for (i=0; i < CASM_MAX_ADDR; i++) {
      if ( (pOsData->paddrs[i].phys_addr == ulAddr) &&
           (pOsData->paddrs[i].mem_len   <= ulLen) ) {
         iArrayIndex = i;
         break;
      } else if (pOsData->paddrs[i].phys_addr == 0) {
         /*
         ** We never have gaps so once we find a zero entry, we
         ** are done.
         */
         break;
      }  /* endif */

   }  /* endfor */

   return (iArrayIndex);

}  /* casmw_mapped_entry */


/*********************************************************************
** 
** @func  casmw_get_vend_device_id
** 
** Routine Description:
** This routine is OS specific and will fill in the PCI Vendor / Device
** ID for the hp ProLiant Server Management ASIC.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int casmw_get_vend_device_id (struct pci_dev *pPciDev, PCASMDATA pInstance)
{
    int         rv = SUCCESS;     /* return values */

    pInstance->ulVendDeviceID = ( (pPciDev->vendor<<16) | pPciDev->device );

    return (rv);

}  /* end casmw_vend_dev_id */

/*********************************************************************
** 
** @func  casmw_get_io_addr
** 
** Routine Description:
** This routine is OS specific and will fill in the IO Address for the 
** hp ProLiant Advance Server Management ASIC.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int casmw_get_io_addr (struct pci_dev *pPciDev, PCASMDATA pInstance)
{
    int         rv = SUCCESS;     /* return values */
    u32         ulAddr;


    pci_read_config_dword(pPciDev, PCI_BASE_ADDRESS_0, &ulAddr);
    if (pInstance->ulFlags & CASM_DEBUG_ALL)
       printk(KERN_DEBUG "I/O Address is: 0x%x\n", ulAddr);
    
    pInstance->ulIOAddr = (ulAddr & 0x0000FFF0);
    pInstance->ulIOLength = 256;
    return (rv);

}  /* end casmw_get_io_addr */



/*********************************************************************
** 
** @func  casmw_get_mem_addr
** 
** Routine Description:
** This routine is OS specific and will fill in the physical memory address
** for the hp ProLiant Advanced Server Management ASIC.  
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int casmw_get_mem_addr (struct pci_dev *pPciDev, PCASMDATA pInstance)
{


    u32       ulAddr;

    pci_read_config_dword(pPciDev, PCI_BASE_ADDRESS_1, &ulAddr);

    if (pInstance->ulFlags & CASM_DEBUG_ALL)
       printk(KERN_DEBUG "Memory Mapped  Address is: 0x%x\n", ulAddr);
    
    pInstance->ulMemAddr =  (ulAddr & 0xFFFFFFF0);


/*
** We want the Mapped Memory Address to be first in the list to
** simplify debugging.
*/
   casmw_update_mapped_entries( pInstance->ulMemAddr, 
                                pInstance->ulIOLength, 
                                pInstance );

   return (SUCCESS);

}  /* end casmw_get_mem_addr */


/*********************************************************************
** 
** @func  casmw_read_debug_pd
** 
** Routine Description:
** This routine is Linux specific and will process a read request on
** the /proc/casmdbug PROC file system node.
** 
** Return Value:
** 
**       Length of buffer
** 
***********************************************************************/
int casmw_read_debug_pd( char *pbuf, char **ppstart, off_t offset, 
                         int buf_size, int *peof, void *pdata)
{
   int length = 0;


   length += casmproc_get_data( CASM_GET_DEBUG_PD, pbuf, ppstart,
                                offset, buf_size, peof );


   return (length);

}  /* end casmw_read_debug_pd */

/*********************************************************************
** 
** @func  casmw_read_pci_pd
** 
** Routine Description:
** This routine is Linux specific and will process a read request on
** the /proc/cpqpci PROC file system node.
** 
** Return Value:
** 
**       Length of buffer
** 
***********************************************************************/
int casmw_read_pci_pd( char *pbuf, char **ppstart, off_t offset, 
                       int buf_size, int *peof, void *pdata)
{
   int length = 0;

   //Synchronize access to the point where only one proc entry can be 
   //generated. A little overkill, maybe, but what the heck.
   down_interruptible(&casm_proc_sem);
   if (signal_pending(current)) {
	   return 0;
   }

   length += casmproc_get_data( CASM_GET_PCI_PD, pbuf, ppstart,
                                offset, buf_size, peof );


   up(&casm_proc_sem);
   return (length);

}  /* end casmw_read_pci_pd */

/*********************************************************************
** 
** @func  casmw_read_pwr_pd
** 
** Routine Description:
** This routine is Linux specific and will process a read request on
** the /proc/cpqpwr PROC file system node.
** 
** Return Value:
** 
**       Length of buffer
** 
***********************************************************************/
int casmw_read_pwr_pd( char *pbuf, char **ppstart, off_t offset, 
                       int buf_size, int *peof, void *pdata)
{
   int length = 0;


   length += casmproc_get_data( CASM_GET_PWR_PD, pbuf, ppstart,
                                offset, buf_size, peof );


   return (length);

}  /* end casmw_read_pwr_pd */

/*********************************************************************
** 
** @func  casmw_read_fan_pd
** 
** Routine Description:
** This routine is Linux specific and will process a read request on
** the /proc/cpqfan PROC file system node.
** 
** Return Value:
** 
**       Length of buffer
** 
***********************************************************************/
int casmw_read_fan_pd( char *pbuf, char **ppstart, off_t offset, 
                       int buf_size, int *peof, void *pdata)
{
   int length = 0;


   length += casmproc_get_data( CASM_GET_FAN_PD, pbuf, ppstart,
                                offset, buf_size, peof );


   return (length);

}  /* end casmw_read_fan_pd */

/*********************************************************************
** 
** @func  casmw_read_temp_pd
** 
** Routine Description:
** This routine is Linux specific and will process a read request on
** the /proc/cpqtemp PROC file system node.
** 
** Return Value:
** 
**       Length of buffer
** 
***********************************************************************/
int casmw_read_temp_pd( char *pbuf, char **ppstart, off_t offset, 
                        int buf_size, int *peof, void *pdata)
{
   int length = 0;


   length += casmproc_get_data( CASM_GET_TEMP_PD, pbuf, ppstart,
                                offset, buf_size, peof );


   return (length);

}  /* end casmw_read_temp_pd */


/*********************************************************************
** 
** @func  casmw_init_proc_fs
** 
** Routine Description:
** This routine is Linux specific and will initialize the /proc file
** system entries.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int casmw_init_proc_fs (PCASMDATA pInstance)
{
   int    retval = SUCCESS;

   cpqtemp_pd = create_proc_read_entry( "cpqtemp", 
                                        0, 
                                        NULL,
                                        casmw_read_temp_pd,
                                        NULL);


   if (!cpqtemp_pd) {
      retval = FAILURE;
      printk(KERN_ALERT "casm:  Unable to create /proc/cpqtemp\n");
   }  /* endif */


   cpqfan_pd = create_proc_read_entry( "cpqfan", 
                                        0, 
                                        NULL,
                                        casmw_read_fan_pd,
                                        NULL);


   if (!cpqfan_pd) {
      retval = FAILURE;
      printk(KERN_ALERT "casm:  Unable to create /proc/cpqfan\n");
   }  /* endif */


   cpqpwr_pd = create_proc_read_entry( "cpqpwr", 
                                        0, 
                                        NULL,
                                        casmw_read_pwr_pd,
                                        NULL);


   if (!cpqpwr_pd) {
      retval = FAILURE;
      printk(KERN_ALERT "casm:  Unable to create /proc/cpqpwr\n");
   }  /* endif */


   cpqpci_pd = create_proc_read_entry( "cpqpci", 
                                        0, 
                                        NULL,
                                        casmw_read_pci_pd,
                                        NULL);


   if (!cpqpci_pd) {
      retval = FAILURE;
      printk(KERN_ALERT "casm:  Unable to create /proc/cpqpci\n");
   }  /* endif */



   casmdbug_pd = create_proc_read_entry( "casmdbug", 
                                           0, 
                                           NULL,
                                           casmw_read_debug_pd,
                                           NULL);


   if (!casmdbug_pd) {
      retval = FAILURE;
      printk(KERN_ALERT "casm:  Unable to create /proc/casmdbug\n");
   }  /* endif */


#ifdef NEEDS_TO_BE_DONE

   cpqiml_pd = create_proc_entry("cpqiml", (S_IFREG | S_IRUGO), &proc_root);
   if (cpqiml_pd)
      cpqiml_pd->read_proc = &casmw_read_iml_pd;
   else {
      retval = FAILURE;
      printk(KERN_ALERT "casm:  Unable to create /proc/cpqiml\n");
   }  /* endif */

#endif
   
   return (retval);

}  /* end casmw_init_proc_fs */


/*********************************************************************

@func  casm_poweroff_handler

Routine Description:
This is an OS Specific routine used to call common code power off 
routine when it's safe to pull the plug.  For Linux, we get called
for every reboot so we to check to see if this is a reboot or a
shutdown / poweroff (init 0, init 5).

Also, we only call the poweroff routine for certain items that
the health driver controls.

Return Value:


***********************************************************************/
static int casm_poweroff_handler(struct notifier_block *nb, unsigned long code, void *p)
{
   if ( gpInstanceData->ulCfgState & 
        (CASM_HARD_POWEROFF | CASM_HARD_RESET | CASM_ASR_RESET) ) {
      casmc_common_poweroff(gpInstanceData);
      printk(KERN_EMERG "(casm)  Unable to power off system!\n");

   }  else {
      casmc_asr_timer_killed(gpInstanceData);

   }  /* endif */

   return (0);

}  /* end casm_poweroff_handler */


/*********************************************************************
** 
** @func  casmw_nmi_handler
** 
** Routine Description:
** This is an OS Specific routine used process an NMI.  UNIX operating
** systems, in general, have the capability to recover from non fatal
** NMI's.
** 
** Return Value:
** NMI_UNKNOWN    :  We have no clue to who generated this NMI
** NMI_BENIGN     :  We undertand the NMI and corrected the problem
** NMI_FATAL      :  We are in big trouble.  Stop Everything.
** NMI_REBOOT     :  We need to reboot now.
** 
***********************************************************************/
int casmw_nmi_handler(UCHAR reason, struct pt_regs *pregs, PCASMDATA pInstance)
{
   BOOLEAN   bRet = CASM_NMI_UNKNOWN;
   int       ThisProc;

#ifdef CASM_NMI_DEBUG
   int       i;
   POSDATA   pOsData = (POSDATA)pInstance->pOsData;
   char      nmi_string[128];
#endif

/*
** Get our processor number as we should only run on the BSP.
** I am making an assumption for Linux that this will be processor zero.
** Hopefully, I am right.
*/
   ThisProc = smp_processor_id();
   printk(KERN_CRIT "casm:  NMI Handler has been called on processor %d!\n", ThisProc);

/*
** If this is not Processor 0, then we'll wait a couple of seconds and then
** exit.  This will give Processor 0 enough time to process this NMI.
*/

   if (ThisProc != 0) {
      printk(KERN_CRIT "casm: Spinning for 2 seconds!\n");
      casmw_microdelay(2000000, gpInstanceData);

   /*
   ** Comment out next "if - endif" section if wanting to chain NMI's to a
   ** Linux debug or system dump tool.
   */
      if (!(pInstance->ulCfgState & CASM_ASR_RESET)) {
         printk(KERN_CRIT "casm: Fatal NMI Detected - Processor %d spinning!\n",
                          ThisProc);
         while(1);
      }  /* endif */

   } else {
   /*
   ** Write to 0x70 to stop NMI's from nesting.  We have an extra
   ** check just in case.  If we nest, we need to just return because
   ** we have already processed the NMI.  There is only one (1) NMI
   ** to a customer.
   */
      casmw_write_byte( 0x70, (UCHAR) 0x80, pInstance );
      if (pInstance->ulCfgState & CASM_NMI_IN_PROGRESS) {
         printk( KERN_CRIT "casm:  Nested NMI Occurred on Processor 0!  Reason:  0x%x\n",
                reason );

         return(0);

      }  /* endif */

/*
** The code below may need to be modified as specific kernels are modified.
** This is, however, the way Linux is designed.
*/
#ifdef CASM_NMI_DEBUG
      casmw_memzero(nmi_string, sizeof(nmi_string));
      if (spin_is_locked(&pOsData->rom_lock)) {
         printk( KERN_CRIT "casm:  rom_lock is held\n" );
         casmc_log_iml_msg(pInstance, "casm:  rom_lock is held");
      }

      if (spin_is_locked(&pOsData->q_lock)) {
         printk( KERN_CRIT "casm:  q_lock is held\n" );
         casmc_log_iml_msg(pInstance, "casm:  q_lock is held");
      }

      if (kernel_locked()) {
         printk( KERN_CRIT "casm:  ** Linux Kernel Lock is held **\n" );
         casmc_log_iml_msg(pInstance, "casm:  ** Linux Kernel Lock is held **");
      }

      if (in_interrupt()) {
         printk( KERN_CRIT "casm:  ** Kernel is in interrupt **\n" );
         printk( KERN_CRIT "casm:  ** global_irq_holder:  0x%x **\n", global_irq_holder );
         casmc_log_iml_msg(pInstance, "casm:  ** Kernel is in interrupt **");
         casmw_sprintf( nmi_string, "casm:  ** global_irq_holder:  0x%x **\n", 
                        global_irq_holder );
         casmc_log_iml_msg(pInstance, nmi_string);
         casmw_memzero(nmi_string, sizeof(nmi_string));
      }  /* endif */

      for (i=0; i< smp_num_cpus; i++) {
         printk ( KERN_CRIT "casm:  CPU %d, local_irq_count:  %d, local_bh_count:  %d\n",
                            i, local_irq_count(i), local_bh_count(i) );
         casmw_sprintf( nmi_string, "casm:  CPU %d, local_irq_count:  %d, local_bh_count:  %d\n",
                         i, local_irq_count(i), local_bh_count(i) );
         casmc_log_iml_msg(pInstance, nmi_string);
      }  /* endfor */

#endif

      bRet = casmc_common_nmi_hndlr(pInstance);

      switch (bRet) {
         case CASM_NMI_UNKNOWN:
            bRet = reason;
            panic( "casm:  Unknown  NMI Error Detected!  Reason:  0x%x\n",
                   reason );
            break;

         case CASM_NMI_BENIGN:
            bRet = 0;
            if (!(pInstance->ulCfgState & CASM_ASR_RESET))
               printk(KERN_CRIT "casm:  No NMI detected by ROM.  Continuing execution . . .\n");
            break;

         case CASM_NMI_FATAL:
            panic("casm:  Fatal NMI Error Detected!\n");
            break;

         case CASM_NMI_REBOOT:
            bRet = 0;
            casmw_graceful_shutdown(gpInstanceData);
            break;

         default:
            panic( "casm:  Unknown  NMI Error Detected!  Reason:  0x%x\n",
                   reason );

      }  /* endswitch */

   }  /* endif */

   return ((int) bRet);

}  /* end casmw_nmi_handler */


/*********************************************************************
** 
** @func  casmw_mca_handler
** 
** Routine Description:
** This is an OS Specific routine used process an NMI.  UNIX operating
** systems, in general, have the capability to recover from non fatal
** NMI's.
** 
** Return Value:
** MCA_UNKNOWN    :  We have no clue to who generated this MCA
** MCA_BENIGN     :  We undertand the MCA and corrected the problem
** MCA_FATAL      :  We are in big trouble.  Stop Everything.
** MCA_REBOOT     :  We need to reboot now.
** 
***********************************************************************/
int casmw_mca_handler(struct pt_regs *pRegs, PCASMDATA pInstance)
{
   BOOLEAN   bRet = CASM_NMI_UNKNOWN;

   printk(KERN_CRIT "casm:  MCA Handler has been called!");

   bRet = casmc_common_mca_hndlr(pInstance);

   return ((int) bRet);

}  /* end casmw_mca_handler */

/***************************************************************************
 ****************       START OF "MUST HAVE" FUNCTION    *******************
 ***************************************************************************/

/*********************************************************************
** 
** @func  casmw_map32_bit_phys
** 
** Routine Description:
** This routine is OS specific and will map in a physical address into
** memory space.
** 
** 
** Return Value:
** 
** NULL     : FAILURE
** VirtAddr : Virtual Address mapped in.
***********************************************************************/
void *casmw_map32_bit_phys( ULONG ulPhysicalAddr,
                            ULONG ulLength,
                            PCASMDATA pInstance )
{
    int     rv = 0;     /* return values */
    ULONG   ulMemBlkNum; 
    int     ulMemSize;
    void   *pVirtAddr = NULL;
    POSDATA pOsData = (POSDATA)pInstance->pOsData;

/*
** First check to see if we have previously mapped in this physical
** address range and length.  If not, add it to the array of Mapped
** Entries and then find it's block number (which also happens to be the
** index into the internal array we maintain for speed.
*/
   ulMemBlkNum = casmw_mapped_entry(ulPhysicalAddr, ulLength, pInstance);
   if (ulMemBlkNum  > CASM_MAX_ADDR) {
      rv = casmw_update_mapped_entries(ulPhysicalAddr, ulLength, pInstance);
      ulMemBlkNum = casmw_mapped_entry(ulPhysicalAddr, ulLength, pInstance);
   }  /* endif */


/*
** We now have a valid data range mapped into memory so get a mapping to it.
** NOTE:  We only map addresses in one time.  This saves a lot of overhead.
*/
   if ( (!rv) && (ulMemBlkNum < CASM_MAX_ADDR) ) {

      if (!pOsData->paddrs[ulMemBlkNum].virt_addr) {

         pVirtAddr = (caddr_t)ioremap(ulPhysicalAddr, ulLength);
         pOsData->paddrs[ulMemBlkNum].virt_addr = pVirtAddr;
      }  else {
          pVirtAddr = pOsData->paddrs[ulMemBlkNum].virt_addr;
      }  /* endif */

   }  /* endif */

/*
** We we have an address, then update the use_count field.
*/
   if (pVirtAddr)
      pOsData->paddrs[ulMemBlkNum].use_count++;

   return (pVirtAddr);

}  /* end casmw_map32_bit_phys */


/*********************************************************************
** 
** @func  casmw_umap32_bit_phys
** 
** Routine Description:
** This routine is OS specific and will unmap a physical address from
** the virtual memory space.  We just decrement the use count in this
** routine because we map the same addresses in multiple times from
** the common code initialization routines.  This is faster than
** remapping each time.
** 
** Return Value:
** 
***********************************************************************/
VOID casmw_unmap32_bit_phys ( PVOID pVirtAddr,ULONG ulLength,
                              PCASMDATA pInstance )
{
    int MemSize = 0;
    int i;

    POSDATA pOsData = (POSDATA)pInstance->pOsData;

/*
** First we look for the virtual address in our internal structure.
** We will use the memory size we allocated with to free this up.
*/
   for (i=0; i < CASM_MAX_ADDR; i++) {
      if (pOsData->paddrs[i].virt_addr == pVirtAddr) {
         pOsData->paddrs[i].use_count--;
         break;
      }  /* endif */

   }  /* endfor */



}  /* end casmw_unmap32_bit_phys */



/*********************************************************************
** 
** @func  casmw_malloc
** 
** Routine Description:
** This routine will allocate kernel memory for 
** driver use.  This is NOT mapped in memory but
** allocated from the kernel pools.
** 
** Return Value:
** Pointer to allocated memory or NULL if this fails.
** 
***********************************************************************/
PVOID casmw_malloc(ULONG ulSize)
{
   PVOID  pVMem;
   ULONG  ulOrder;

   if (gpInstanceData->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_malloc:  Asking for %d bytes\n", ulSize);

   ulOrder = get_order(ulSize);
   if (ulOrder > 5) {
      pVMem = (PVOID)__get_free_pages(GFP_ATOMIC, ulOrder);
   } else {
      pVMem = (PVOID)kmalloc(ulSize, GFP_ATOMIC);
   }  /* endif */

   if (gpInstanceData->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_malloc:  %p (size %d)\n",pVMem, ulSize);

   return (pVMem);

}  /* end casmw_malloc */



/*********************************************************************
** 
** @func  casmw_free
** 
** Routine Description:
** This routine will free kernel memory previously
** allocated for driver use.  This is NOT mapped in memory but
** allocated from the kernel pools.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
VOID casmw_free(PVOID pVAddr, ULONG ulSize)
{
   ULONG  ulOrder;

   if (gpInstanceData->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_free:  %p (size %d)\n",pVAddr, ulSize);

   ulOrder = get_order(ulSize);
   if (ulOrder > 5) {
      free_pages((ULONG)pVAddr, ulOrder);
   } else {
      kfree (pVAddr);
   }  /* endif */

}  /* end casmw_free */



/*********************************************************************
** 
** @func  casmw_lock_rom
** 
** Routine Description:
** This routine is OS specific and will take a synchronization lock for
** access to the ROM routines.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_lock_rom(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   spin_lock_irqsave(&pOsData->rom_lock, pOsData->rom_pl);
} /* casmw_lock_rom */


/*********************************************************************
** 
** @func  casmw_unlock_rom
** 
** Routine Description:
** This routine is OS specific and will free a synchronization lock for
** the ROM access point.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_unlock_rom(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   spin_unlock_irqrestore(&pOsData->rom_lock, pOsData->rom_pl);
} /* casmw_unlock_rom */


/*********************************************************************
** 
** @func  casmw_lock_queue
** 
** Routine Description:
** This routine is OS specific and will take a synchronization lock for
** the queue.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_lock_queue(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   spin_lock_irqsave(&pOsData->q_lock, pOsData->q_pl);
} /* casmw_lock_queue */


/*********************************************************************
** 
** @func  casmw_unlock_queue
** 
** Routine Description:
** This routine is OS specific and will free a synchronization lock for
** the queue.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_unlock_queue(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   spin_unlock_irqrestore(&pOsData->q_lock, pOsData->q_pl);
} /* casmw_unlock_queue */


/*********************************************************************
** 
** @func  casmw_lock_iic
** 
** Routine Description:
** This routine is OS specific and will take a synchronization lock 
** to wait on the IIC semaphore.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_lock_iic(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   spin_lock_irqsave(&pOsData->iic_lock, pOsData->iic_pl);
} /* casmw_lock_iic */


/*********************************************************************
** 
** @func  casmw_unlock_iic
** 
** Routine Description:
** This routine is OS specific and will free the synchronization lock for
** the IIC semaphore.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_unlock_iic(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   spin_unlock_irqrestore(&pOsData->iic_lock, pOsData->iic_pl);
} /* casmw_unlock_iic */

/*********************************************************************
**
** @func  casmw_hook_mca_vector
** 
** Routine Description:
** This routine is OS specific and will hook the MCA vector as needed
** 
** We hook the MCA vector during casm_init_one so we just tie into it during
** our initialzation routines.  Other "UNIX" may need this.
**
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_hook_mca_vector(ULONG ulProcessorNumber, PCASMDATA pInstance)
{
   return;

} /* casmw_hook_mca_vector */


/*********************************************************************
**
** @func casmw_run_all_mca_init
** 
** Routine Description:
** This routine is OS specific and is used to allow the kernel to 
** schedule the casmc_common_mca_init routine to run on the specified
** processor.  
**
** This routine is the argument to the smp_call_function call.
** 
** 
** Return Value:
**     0  SUCCESS
**     1  FAILURE
** 
***********************************************************************/
void  casmw_run_all_mca_init( void *pInstance)
{
   ULONG ulMyProc;
   PCASMDATA pDebug = (PCASMDATA)pInstance;

   if (pDebug->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_run_all_mca_init:  Asking for Processor ID\n");

   ulMyProc = smp_processor_id();

   if (pDebug->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_run_all_mca_init:  ulMyProc is %d\n", ulMyProc);
/*
** Now run the init code on this processor.
*/
   casmc_common_mca_init((PCASMDATA)pInstance, ulMyProc);
 

} /* casmw_run_all_mca_init  */

/*********************************************************************
**
** @func casmw_os_mca_init
** 
** Routine Description:
** This routine is OS specific and is used to allow the kernel to 
** schedule the casmc_common_mca_init routine to run on the specified
** processor.  
** 
** UnixWare will use a process bound to a CPU so this routine will
** never be called.  The function, however, needs to be defined so the
** linker does not freak out.
**
** 
** Return Value:
**     0  SUCCESS
**     1  FAILURE
** 
***********************************************************************/
BOOLEAN casmw_os_mca_init( PCASMDATA pInstance, ULONG ulProcessor )
{
   ULONG ulMyProc;

   if (pInstance->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_os_mca_init:  Asking for Processor ID\n");

   ulMyProc = smp_processor_id();

   if (pInstance->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_os_mca_init:  ulMyProc is %d\n", ulMyProc);

/*
** First we initialize this processor . . .
*/
   casmc_common_mca_init(pInstance, ulMyProc);
 
/*
** And then we initialize the rest of them . . .
*/
   if (pInstance->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_os_mca_init:  Calling smp_call_function\n");

   smp_call_function(casmw_run_all_mca_init, (void *)pInstance, 0, 1);

   return (SUCCESS);

} /* casmw_os_mca_init  */


/*********************************************************************
**
** @func casmw_run_all_poll
** 
** Routine Description:
** This routine is OS specific and is used to allow the kernel to 
** schedule the casmc_common_mca_poll routine to run on the specified
** processor.  
**
** This routine is the argument to the smp_call_function call.
** 
** 
** Return Value:
**     0  SUCCESS
**     1  FAILURE
** 
***********************************************************************/
void casmw_run_all_mca_poll(void *pInstance)
{
   ULONG ulMyProc;
   PCASMDATA pDebug = (PCASMDATA)pInstance;

   ulMyProc = smp_processor_id();

   if (pDebug->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_run_all_mca_poll:  ulMyProc is %d\n", ulMyProc);

/*
** Now run the poll  code on this processor.
*/
   casmc_common_mca_poll((PCASMDATA)pInstance, ulMyProc);
 

} /* casmw_os_mca_poll  */

/*********************************************************************
**
** @func casmmca_os_mca_poll
** 
** Routine Description:
**
** This routine is OS specific and is used to allow the kernel to 
** schedule the casmc_common_mca_poll routine to run on the specified
** processor.  
** 
** UnixWare will use a process bound to a CPU so this routine will
** never be called.  The function, however, needs to be defined so the
** linker does not freak out.
**
** 
** Return Value:
**     0  SUCCESS
**     1  FAILURE
** 
** 
** Return Value:
** 
***********************************************************************/
BOOLEAN casmmca_os_mca_poll(PCASMDATA pInstance, ULONG ulProcessor)
{
   ULONG ulMyProc;

   ulMyProc = smp_processor_id();

   if (pInstance->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_os_mca_poll:  ulMyProc is %d\n", ulMyProc);

/*
** First we poll this processor . . .
*/
   casmc_common_mca_poll(pInstance, ulMyProc);
 
/*
** And then we initialize the rest of them . . .
*/
   smp_call_function(casmw_run_all_mca_poll, (void *)pInstance, 0, 1);

   return (SUCCESS);

} /* casmw_os_mca_poll */

/*********************************************************************
** 
** @func  casmw_mca_thread_wait
** 
** Routine Description:
** This routine is OS specific and will wait to be notified when 
** the MCA polling thread is to run.  There will be multiple polling
** threads waiting on the Synchronization Variable (which will be an
** "event queue" in other UNIX / Linux operating systems).  UnixWare
** is capable of "broadcasting" to all threads posted to the 
** synchronization variable so we don't care about the ulProcId.  I 
** suspect FreeBSD, Linux and Solaris may have to know which thread to
** release.
**
** When the the routine is released from here, one of two things has happened:
** 
** 1.  This thread needs to poll it's assigned processor (ulProcId)
** 2.  The OS is shutting down.
** 
** If no Event is returned from the queue, the OS is shutting down and
** the application should return.
**
** NOTE:  The QUEUE lock must be held before calling this routine.
** 
** Return Value:
** 0 = SUCCESS
** 1 = FAILURE
** 
***********************************************************************/
int casmw_mca_thread_wait(PCASMDATA pInstance, ULONG ulProcId)
{
    POSDATA     pOsData = (POSDATA)pInstance->pOsData;
    int		rv = SUCCESS;

    pInstance->ulMCAWaitCondition = 0;

    if (!(pInstance->ulCfgState & CASM_SHUTDOWN)) {
       casmw_unlock_queue(pInstance);
       if (wait_event_interruptible(pOsData->mca_wq, (pInstance->ulMCAWaitCondition))) {
	 /*
	 ** We will set this to -1 as a special case to indicate that 
	 ** we got woken up by a signal. We could use it later.
	 */
          pInstance->ulMCAWaitCondition = -1;
          rv = FAILURE;
       }

   /*
   ** We need to obtain the lock again since SV_WAIT released it.
   */
       casmw_lock_queue(pInstance);

    }  /* endif */

  return (rv);

} /* casmw_mca_thread_wait */


/*********************************************************************
**
** @func  casmw_wakeup_mca_threads
** 
** Routine Description:
** This routine is OS specific and will notify the MCA threads that
** it is time to run.
** 
** UnixWare is capable of "broadcasting" to all threads posted to the 
** synchronization variable so we don't care about the ulProcId.  I 
** suspect FreeBSD, Linux and Solaris may have to know which thread to
** release.
**
** NOTE:  The QUEUE lock does NOT need to be held for this routine.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_wakeup_mca_threads(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   pInstance->ulMCAWaitCondition = 1;

   wake_up_interruptible(&pOsData->mca_wq);

} /* casmw_wakeup_mca_threads */


/*********************************************************************
** 
** @func  casmw_wait_for_event
** 
** Routine Description:
** This routine is OS specific and will wait to be notified when something
** is placed on the queue.  When the the routine is released from here,
** one of two things has happened:
** 
** 1.  An Event has been put on the queue.
** 2.  The OS is shutting down.
** 
** If no Event is returned from the queue, the OS is shutting down and
** the application should return.
** 
** NOTE:  The QUEUE lock must be held before calling this routine.
** 
** Return Value:
** 0 = SUCCESS
** 1 = FAILURE
** 
***********************************************************************/
int casmw_wait_for_event(PCASMDATA pInstance)
{
    POSDATA     pOsData = (POSDATA)pInstance->pOsData;
    int		rv = SUCCESS;
    
    pInstance->ulWaitCondition = 0;
/*
** Make sure we are not shutting down before we go to
** sleep.
*/
    if (!(pInstance->ulCfgState & CASM_SHUTDOWN)) {
       casmw_unlock_queue(pInstance);
       if (wait_event_interruptible(pOsData->task_wq, (pInstance->ulWaitCondition))) {
	  /*
	  ** We will set this to -1 as a special case to indicate that 
	  ** we got woken up by a signal. We could use it later.
	  */
          pInstance->ulWaitCondition = -1;
          rv = FAILURE;
       }

   /*
   ** We need to obtain the lock again since we released it.
   */
       casmw_lock_queue(pInstance);

    }  /* endif */

  return (rv);

} /* casmw_wait_for_event */


/*********************************************************************
**
** @func  casmw_event_notify
** 
** Routine Description:
** This routine is OS specific and will notify the application waiting
** on the queue.
** 
** NOTE:  The QUEUE lock does NOT need to be held for this routine.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_event_notify(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;
   pInstance->ulWaitCondition = 1;
   wake_up_interruptible(&pOsData->task_wq);

} /* casmw_event_notify */


/***************************************************************************
 **************** OS HEALTH COMMON CODE SUPPORT ROUTINES *******************
 ***************************************************************************/

/*********************************************************************
** 
** @func  casmw_in_byte_mem
** 
** Routine Description:
** This is an OS specific routine which will read and return one byte
** of data from a given offset. This is for memory mapped io.
** 
** 
** Return Value:
** 
** UCHAR   :  The data returned from the call
** 
***********************************************************************/
UCHAR casmw_in_byte_mem(PULONG pulAddr, PCASMDATA pInstance)
{
   PUCHAR  pByte;
   pByte = (UCHAR *) pulAddr;
   return ( (UCHAR)*pByte );

}  /* end casmw_in_byte_mem */


/*********************************************************************
** 
** @func  casmw_in_byte_word
** 
** Routine Description:
** This is an OS specific routine which will read and return one word
** of data from a given offset. This is for memory mapped IO.
** 
** 
** Return Value:
** 
** USHORT   :  The data returned from the call
** 
***********************************************************************/
USHORT casmw_in_word_mem(PULONG pulAddr, PCASMDATA pInstance)
{
     USHORT *pWord;

     pWord = (USHORT *)pulAddr;
     return ( (USHORT) *pWord );

}  /* end casmw_in_word_mem */


/*********************************************************************
** 
** @func  casmw_in_dword_mem
** 
** Routine Description:
** This is an OS specific routine which will read and return a double 
** word of data from a given offset. This is for memory mapped IO.
** 
** 
** Return Value:
** 
** ULONG   :  The data returned from the call
** 
***********************************************************************/
ULONG casmw_in_dword_mem(PULONG pulAddr, PCASMDATA pInstance)
{
     ULONG *pDword;

     pDword = pulAddr;
     return ( (ULONG)*pDword );

}  /* end casmw_in_dword_mem */


/*********************************************************************
** 
** @func  casmw_out_byte_mem
** 
** Routine Description:
** This is an OS specific routine which will write the specified byte
** to the given offset. This is for memory mapped IO.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void casmw_out_byte_mem( PULONG pAddr, UCHAR ucData, PCASMDATA pInstance )
{
     PUCHAR pByte;

     pByte = (PUCHAR) pAddr;
     *pByte = ucData; 

}  /* end casmw_out_byte_mem */



/*********************************************************************
** 
** @func  casmw_out_word_mem
** 
** Routine Description:
** This is an OS specific routine which will write the specified word
** to the given offset. This is for Memory Mapped IO.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void casmw_out_word_mem( PULONG pAddr, USHORT usData, PCASMDATA pInstance )
{
   PUSHORT pWord;

   pWord = (PUSHORT) pAddr;
   *pWord = usData; 

}  /* end casmw_out_word_mem */



/*********************************************************************
** 
** @func  casmw_out_dword_mem
** 
** Routine Description:
** This is an OS specific routine which will write the specified byte
** to the given offset. This routine is for Memory Mapped IO.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void casmw_out_dword_mem( PULONG pAddr, ULONG ulData, PCASMDATA pInstance )
{
   PULONG pDword;

   pDword = (PULONG) pAddr;
   *pDword = ulData; 

}  /* end casmw_out_dword_mem */


/*********************************************************************
** 
** @func  casmw_read_byte
** 
** Routine Description:
** This is an OS specific routine which will read and return one byte
** of data from a given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** UCHAR   :  The data returned from the call
** 
***********************************************************************/
UCHAR casmw_read_byte(ULONG ulAddr, PCASMDATA pInstance)
{
   return ( (UCHAR)inb(ulAddr) );

}  /* end casmw_read_byte */


/*********************************************************************
** 
** @func  casmw_read_word
** 
** Routine Description:
** This is an OS specific routine which will read and return one word
** of data from a given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** USHORT   :  The data returned from the call
** 
***********************************************************************/
USHORT casmw_read_word(ULONG ulAddr, PCASMDATA pInstance)
{
   return ( (USHORT)inw(ulAddr) );

}  /* end casmw_read_word */


/*********************************************************************
** 
** @func  casmw_read_dword
** 
** Routine Description:
** This is an OS specific routine which will read and return a double 
** word of data from a given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** ULONG   :  The data returned from the call
** 
***********************************************************************/
ULONG casmw_read_dword(ULONG ulAddr, PCASMDATA pInstance)
{
   return ( (ULONG)inl(ulAddr) );

}  /* end casmw_read_dword */


/*********************************************************************
** 
** @func  casmw_write_byte
** 
** Routine Description:
** This is an OS specific routine which will write the specified byte
** to the given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void casmw_write_byte( ULONG ulAddr, UCHAR ucData, PCASMDATA pInstance )
{
   outb(ucData, ulAddr);

}  /* end casmw_write_byte */



/*********************************************************************
** 
** @func  casmw_write_word
** 
** Routine Description:
** This is an OS specific routine which will write the specified word
** to the given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void casmw_write_word( ULONG ulAddr, USHORT usData, PCASMDATA pInstance )
{
   outw(usData, ulAddr);

}  /* end casmw_write_word */



/*********************************************************************
** 
** @func  casmw_write_dword
** 
** Routine Description:
** This is an OS specific routine which will write the specified byte
** to the given offset. This is for IO (in / out) transfers.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
void casmw_write_dword( ULONG ulAddr, ULONG ulData, PCASMDATA pInstance )
{
   outl(ulData, ulAddr);

}  /* end casmw_write_dword */


/*********************************************************************
** 
** @func  casmw_microdelay
** 
** Routine Description:
** This is an OS specific routine which will delay a specific amount of
** time.  The time passed in is in Microseconds.
** 
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
ULONG casmw_microdelay( ULONG ulUsec, PCASMDATA pInstance )
{
   ULONG ulDelay;

   if (pInstance->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_microdelay:  delay:  %d\n", ulUsec);


   if (ulUsec < 1000) {           /* one millisecond       */
      udelay (ulUsec);

   } else if (ulUsec < 15000) {   /* Slightly longer wait  */
      ulDelay = ulUsec / 1000;    /* convert to mSec       */
      mdelay (ulDelay);
      
   } else {                       /* We just need to sleep */
      ulDelay = ulUsec / CASM_USEC_IN_JIFFIES;
      if (!ulDelay)
         ulDelay = 2;        /* Min sleep time: > 1 JIFFY  */

      interruptible_sleep_on_timeout(&casm_sleep_q, ulDelay);

   }  /* endif */

   return (ulUsec);

}  /* end casmw_microdelay */


/*********************************************************************
** 
** @func  casmw_signal_pending
** 
** Routine Description:
** This is an OS specific routine to determine if a signal is pending.
** 
** Return Value:
** SUCCESS : No signals pending 
** FAILURE : Signal is pending 
** 
***********************************************************************/
BOOLEAN casmw_signal_pending(PCASMDATA pInstance)
{
   BOOLEAN     bRet = SUCCESS;

   if (signal_pending(current)) {
      bRet = FAILURE;
   }

   return (bRet);

}  /* end casmw_signal_pending */


/*********************************************************************
** 
** @func  casmw_timeout
** 
** Routine Description:
** This is an OS specific routine which will delay a specific amount of
** time or if a function is passed in, execute a time delay to run
** that function.  
** 
** NOTE:  If pFunction is equal to NULL, then this routine must ONLY
**        be called in a BLOCKING (i.e. not interrupt) context.  We
**        use "delay" which is not permitted to be called at interrupt
**        context.
** 
** Return Value:
** Nothing
** 
***********************************************************************/
VOID casmw_timeout( PVOID pFunction, PVOID pArg, ULONG ulUsec, 
                    ULONG ulTimeOutID, PCASMDATA pInstance )
{
   struct timer_list   *pTimer;
   ULONG               ulSeconds;
   POSDATA             pOsData = (POSDATA)pInstance->pOsData;

   if (pInstance->ulFlags & CASM_DEBUG_ALL)
      printk(KERN_DEBUG "casmw_timeout:  function %p  delay:  %d\n",
                        pFunction, ulUsec);


   if (!pFunction) {
      casmw_microdelay(ulUsec, pInstance);
   } else {
   /*
   ** We have to convert to seconds from microseconds for Jiffies.
   */
      ulSeconds = ulUsec / 1000000;
      pTimer = &pOsData->timers[ulTimeOutID];
      init_timer(pTimer);
      pOsData->timer_flag[ulTimeOutID] = 1;
      pTimer->function = (void (*)())pFunction;
      pTimer->data = (ULONG)pArg;
      pTimer->expires = jiffies + (ulSeconds * HZ);

      add_timer(pTimer);

   }  /* endif */

}  /* end casmw_timeout */


/*********************************************************************
** 
** @func  casmw_cancel_timeout
** 
** Routine Description:
** This is an OS specific routine will cancel a previously scheduled
** timeout.  This function should be called with a lock held.
** 
** 
** Return Value:
** 
** 
***********************************************************************/
VOID casmw_cancel_timeout( ULONG ulTimeOutID, PCASMDATA pInstance )
{
   struct timer_list   *pTimer;
   POSDATA             pOsData = (POSDATA)pInstance->pOsData;

   if ( pOsData->timer_flag[ulTimeOutID] ) {
      pTimer = &pOsData->timers[ulTimeOutID];
      del_timer_sync(pTimer);
      pOsData->timer_flag[ulTimeOutID] = 0;
   } /* endif */

}  /* end casmw_cancel_timeout */

/*********************************************************************
** 
** @func  casmw_memcpy
** 
** Routine Description:
** This is an OS specific routine which will copy from Source to 
** destination for a certain amount of bytes.  No Alignment is
** assumed so individual OS's may need to double buffer to assure
** alignment if required.
** 
** Return Value:
** 
** Nothing
** 
***********************************************************************/
ULONG casmw_memcpy(char *dest, char *source, int length)
{
        memcpy(dest, source,  length);
        return ( (ULONG) dest );

}  /* end casmw_memcpy */



/*********************************************************************
** 
** @func  casmw_strcpy
** 
** Routine Description:
** This is an OS specific routine which will copy from Source to 
** destination until a NULL character is received.  No alignment is
** assumed so individual OS's may need to double buffer to assure
** alignment if required.
** 
** Return Value:
** Byte pointer to the Destination
** 
***********************************************************************/
char *casmw_strcpy (char *s1, char *s2)
{

   return ( (char *) strcpy(s1,s2));

} /* end casmw_strcpy */



/*********************************************************************
** 
** @func  casmw_strcat
** 
** Routine Description:
** This is an OS specific routine which will concatenate from Source to 
** destination until a NULL character is received.  No alignment is
** assumed so individual OS's may need to double buffer to assure
** alignment if required.
** 
** Return Value:
** 
** Byte pointer to the destination.
** 
***********************************************************************/
char *casmw_strcat (char *s1, char *s2)
{

   return ( (char *) strcat(s1,s2));

}  /* end casmw_strcat */


/*********************************************************************
** 
** @func  casmw_strlen
** 
** Routine Description:
** This is an OS specific routine which return the length of a null
** terminated string. No alignment is assumed so individual OS's may 
** need to double buffer to assure alignment if required.
** 
** Return Value:
** 
** String Length
** 
***********************************************************************/
ulong casmw_strlen (char *s1)
{

   return ( (ulong) strlen(s1) );

} /* end casmw_strlen */



/*********************************************************************
** 
** @func  casmw_memzero
** 
** Routine Description:
** This OS Specific routine will "zero out" a memory location for the
** length specified.
** No alignment is assumed so individual OS's may need to double buffer 
** to assure alignment if required.
** 
** Return Value:
** 
** VOID pointer to memory addressed passed in.
** 
***********************************************************************/
void *casmw_memzero(void *s, int n)
{
   memset(s,0,n);
   return (s);
}  /* end casmw_memzero */


/*********************************************************************
** 
** @func  casmw_memset
** 
** Routine Description:
** This is an OS Specific routine which will set memory to a particular
** value as passed in. 
** 
** Return Value:
** VOID pointer to memory addressed passed in.
**
**
***********************************************************************/
void * casmw_memset(  void *s, register int c, register int n)
{
   memset(s,c,n);
   return (s);

}  /* end casmw_memset */

/*********************************************************************
** 
** @func  casmw_strncmp
** 
** Routine Description:
** This is an OS Specific routine which will compare two strings for
** a specified length.  Follows the standard C Library for "strncmp".
** No alignment is assumed so individual OS's may need to double buffer 
** to assure alignment if required.
** 
** Return Value:
** VOID pointer to memory addressed passed in.
** 
***********************************************************************/
int casmw_strncmp(register char *s1, register char *s2, register int length)
{
   char *end;

   for (end = s1 + length - 1; s1 < end && *s1 - *s2 == 0; s1++, s2++);
   return((int)(*s1 - *s2));

} /* end casmw_strncmp */



/*********************************************************************
** 
** @func  casmw_allocate_iic_semaphore
** 
** Routine Description:
** To allow interrupt driven I2C transactions. 
** 
** Return Value:
** 
** 
** 
***********************************************************************/
PVOID casmw_allocate_iic_semaphore(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   spin_lock_init(&pOsData->iic_lock);
   init_waitqueue_head(&pOsData->iic_wq);

   return ( (PVOID)&pOsData->iic_wq );

}  /* end casmw_allocate_iic_semaphore */

/*********************************************************************
** 
** @func  casmw_wait_iic_semaphore
** 
** Routine Description:
** This routine is OS specific and will wait to be notified.
** 
** NOTE:  The IIC lock must be held before calling this routine.
** 
** Return Value:
** Nothing needed to return
**
***********************************************************************/
VOID casmw_wait_iic_semaphore(PCASMDATA pInstance)
{
    POSDATA     pOsData = (POSDATA)pInstance->pOsData;

/*
** Make sure we are not shutting down before we go to
** sleep.
*/
    if (!(pInstance->ulCfgState & CASM_SHUTDOWN)) {
       casmw_unlock_iic(pInstance);
       interruptible_sleep_on(&pOsData->iic_wq);

   /*
   ** We need to obtain the lock again since we released it.
   */
       casmw_lock_iic(pInstance);

    }  /* endif */

  

}  /* end casmw_wait_iic_semaphore */


/*********************************************************************
** 
** @func  casmw_signal_iic_semaphore
** 
** Routine Description:
** This routine is OS specific and will notify the application waiting
** on an I2C interrupt.
** 
** NOTE:  The IIC lock does NOT need to be held for this routine.
** 
** Return Value:
** Nothing needed to return.
** 
***********************************************************************/
VOID casmw_signal_iic_semaphore(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   wake_up_interruptible(&pOsData->iic_wq);

}  /* end casmw_signal_iic_semaphore */

/*********************************************************************
** 
** @func  casmw_free_iic_semaphore
** 
** Routine Description:
** 
** 
** Return Value:
** 
** 
** 
***********************************************************************/
VOID casmw_free_iic_semaphore(PCASMDATA pInstance)
{
   /*
   ** The lock and semaphore are not pointers that were "alloc'd" so we don't 
   ** need to "free" them? 
   */

}  /* end casmw_free_iic_semaphore */

/*********************************************************************
** 
** @func  casmw_allocate_iic_thread_lock
** 
** Routine Description:
** This lock is to protect us from a MUX that could point at another IIC bus
** while the Health driver's I2C transaction on the initial IIC bus is in 
** progress. This could happen if there's a Diagnostic tool's request coming
** in through an ioctl. With this lock we could lock out the ASIC while 
** performing I2C transactions.
** 
** Return Value:
** 
** 
** 
***********************************************************************/
PVOID casmw_allocate_iic_thread_lock(PCASMDATA pInstance)
{
   POSDATA     pOsData = (POSDATA)pInstance->pOsData;

   pOsData->iicthread_lock = 0;

   return ( (PVOID)&pOsData->iicthread_lock );

} /* casmw_allocate_iic_thread_lock */

/*********************************************************************
** 
** @func  casmw_take_iic_thread_lock
** 
** Routine Description:
** This routine is OS specific and will take a synchronization lock for
** access to the IIC bus.   For Linux, we will use the atomic bit
** functions as we need to "delay" if we can not take the lock.  This
** allows the processor to be rescheduled.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
VOID casmw_take_iic_thread_lock(PCASMDATA pInstance)
{
    POSDATA     pOsData = (POSDATA)pInstance->pOsData;

    while (test_and_set_bit(1, (void *) &pOsData->iicthread_lock) != 0)
      casmw_microdelay(63000, pInstance);

} /* casmw_take_iic_thread_lock */

/*********************************************************************
** 
** @func  casmw_release_iic_thread_lock
** 
** Routine Description:
** This routine is OS specific and will unlock the synchronization lock for
** the IIC bus access.  For Linux, we are using the atomic test_and_clear_bit
** API to clear the lock.  The lock routine will poll on this bit every 63
** microseconds if the bit is held.  Otherwise we have to implement a
** different event strategy which Pat does not feel like messing with
** right now.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
VOID casmw_release_iic_thread_lock(PCASMDATA pInstance)
{
    POSDATA     pOsData = (POSDATA)pInstance->pOsData;

    if (test_and_clear_bit(1, (void *) &pOsData->iicthread_lock) == 0)
      printk(KERN_WARNING "casm:  casmw_release_iic_thread_lock:: iicthread_lock == 0\n");

} /* casmw_release_iic_thread_lock */

/*********************************************************************
** 
** @func  casmw_free_iic_thread_lock
** 
** Routine Description:
** 
** Return Value:
** 
***********************************************************************/
VOID casmw_free_iic_thread_lock(PCASMDATA pInstance)
{
   /*
   ** The IIC Thread lock is not a pointer that was "alloc'd" so we don't 
   ** need to "free" it?
   */

} /* casmw_free_iic_thread_lock */



/*********************************************************************
** 
** @func  casmw_sprintf
** 
** Routine Description:
** This is an OS Specific routine which emulate a sprintf command for a
** kernel driver.   This function takes the same commands as sprintf.
** 
** Return Value:
** length that was written to the buffer.
** 
***********************************************************************/
int casmw_sprintf(char *pbuf, const char *pfmt, ...)
{
   va_list vargs;
   int     length;

   va_start(vargs, pfmt);
   length = vsprintf(pbuf, pfmt, vargs);
   va_end(vargs);
   return (length);

}  /* end casmw_sprintf */

/*********************************************************************
** 
** @func  casmw_printf
** 
** Routine Description:
** This is an OS Specific routine which emulate a printf command for a
** kernel driver.  In the case of UnixWare, this is the cmn_err routine.
** The arguments to this driver are the following:
** 
** ulErrorLevel   :  The error level which to log.  
** pString        :  The pointer to the string to pass to "cmn_err".
** ppParms        :  The Address of where the parameters are on the stack.
**                   This is set by the calling routine since the parameters
**                   are on the stack before the following the "pString"
**                   pointer.
** pInstance      :  The "Instance Data" for this driver which is passed to
**                   all routines in this module.
** 
** 
** 
** Return Value:
** Nothing 
** 
***********************************************************************/
VOID casmw_printf( ULONG ulErrorLevel, PCHAR pString, 
                   PULONG ppParms, PCASMDATA pInstance )
{
   PULONG p1;
   PULONG p2;
   PULONG p3;
   PULONG p4;
   PULONG p5;
   PULONG p6;
   PULONG p7;
   PULONG p8;
   PULONG p;

   PCHAR pCmnMsg;

/*
** Insert the string "casm: " to the OS_OutErr() messages to 
** identify the driver.
*/
   pCmnMsg = &gPrintfBuff[0];

/*
** Now make the call to printk with the appropriate error level set
*/

   switch (ulErrorLevel) {

      case CASM_KERN_INFO:
         casmw_strcpy(pCmnMsg, KERN_NOTICE);
         casmw_strcat(pCmnMsg, "NOTICE: casm: ");
         break;

      case CASM_KERN_WARN:
         casmw_strcpy(pCmnMsg, KERN_ALERT);
         casmw_strcat(pCmnMsg, "WARNING: casm: ");
         break;

      case CASM_KERN_DEBUG:
         casmw_strcpy(pCmnMsg, KERN_DEBUG);
         casmw_strcat(pCmnMsg, "DEBUG: casm: ");
         break;

      case CASM_KERN_CRITICAL:
         casmw_strcpy(pCmnMsg, KERN_ALERT);
         casmw_strcat(pCmnMsg, "CRITICAL: casm: ");
         break;

      case CASM_KERN_PANIC:
         casmw_strcpy(pCmnMsg, KERN_EMERG);
         casmw_strcat(pCmnMsg, "EMERGENCY! casm: ");
         break;

      default:
         printk(KERN_WARNING "ERROR! casm:  Invalid Error Level from Common Code:  %d\n",
                             ulErrorLevel );
         casmw_strcpy(pCmnMsg, KERN_WARNING);
         casmw_strcat(pCmnMsg, "ERROR: casm: ");
         break;

   }  /* endswitch */

   casmw_strcat(pCmnMsg, pString);
   casmw_strcat(pCmnMsg, "\n");

/*
** Fixup the pointers so they point to up to 8 32bit input parameters.
** The way printk works, it expects the parameters to be located on the
** stack immediately following the pointer.  We will need to copy the
** parameters from the original location on the stack (as pointed to
** by ppParms) to the current stack pointer location when we make the
** call to printk.
*/

   p = p1 = ppParms;
   p2 = ++p;
   p3 = ++p;
   p4 = ++p;
   p5 = ++p;
   p6 = ++p;
   p7 = ++p;
   p8 = ++p;

/*
** Now make the call to printk
*/

   printk(pCmnMsg,*p1,*p2,*p3,*p4,*p5,*p6,*p7,*p8); 


}  /* end casmw_printf */


/*********************************************************************
** 
** @func  casmw_graceful_shutdown
** 
** Routine Description:
** This routine is OS specific and will initiate a graceful shutdown of
** the operating system.
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
void casmw_graceful_shutdown(PCASMDATA pInstance)
{
/*
** We do not want to be unloaded if we are supposed to shutdown
** the server.  We need our poweroff routine to execute.  If you
** increment the Module Use Count, Linux will not allow the module
** to be unloaded.  User will get a warning message but such is
** life!
*/
   if ( gpInstanceData->ulCfgState & 
        (CASM_HARD_POWEROFF | CASM_HARD_RESET | CASM_ASR_RESET) ) 
      MOD_INC_USE_COUNT;

   kill_proc(1, SIGINT, 1); 

} /* casmw_graceful_shutdown */


/*********************************************************************
** 
** @func  casmw_log_to_RIB
** 
** Routine Description:
** This routine provides the OS Specific method for logging an event
** to the Remote Insight Board (RIB).
** 
** Return Value:
** Nothing needed to return
** 
***********************************************************************/
BOOLEAN casmw_log_to_RIB(PVOID pCridLog, PCASMDATA pInstance)
{
   BOOLEAN  bRet = SUCCESS;

   if (cevtw_CridLogEntry) 
      bRet = cevtw_CridLogEntry((PVOID)pCridLog);

   return (bRet);

} /* casmw_log_to_RIB */



/*********************************************************************
** 
** @func  casmw_register_IML
** 
** Routine Description:
** This routine provides the OS Specific method for providing an
** interface to other device drivers to log to the hp ProLiant Integrated
** Management Log.
** 
** Return Value:
**      0  :  SUCCESS
**      1  :  FAILURE
***********************************************************************/
BOOLEAN casmw_register_IML(PCASMDATA pInstance)
{
   BOOLEAN  bRet = SUCCESS;

   if (cevtw_reg_iml_handler) {
      bRet = cevtw_reg_iml_handler(CEVT_ADD_HANDLER, (PVOID)casmevt_iml_log);

   }  /* endif */

   return (bRet);

} /* casmw_register_IML */


/*********************************************************************
** 
** @func  casmw_deregister_IML
** 
** Routine Description:
** This routine provides the OS Specific method for releasing the IML
** interface to other device drivers to log to the hp ProLiant Integrated
** Management Log.
** 
** Return Value:
**      0  :  SUCCESS
**      1  :  FAILURE
***********************************************************************/
BOOLEAN casmw_deregister_IML(PCASMDATA pInstance)
{
   BOOLEAN  bRet = SUCCESS;

#ifdef NOT_DONE_YET
   if (cevtw_reg_iml_handler) {
      bRet = cevtw_reg_iml_handler(CEVT_REMOVE_HANDLER, (PVOID)casmevt_iml_log);

   }  /* endif */
#endif

   return (bRet);

} /* casmw_deregister_IML */



/*********************************************************************
** 
** @func  casmw_get_seconds_from_jan11970
** 
** Routine Description:
** This routine provides the OS Specific method for providing 
** the number of seconds since January 1, 1970 (the UNIX EPOCH).
** This time is to be returned as Universal Coordinated Time (UTC).
** 
** Return Value:
**      0  :  SUCCESS
**      1  :  FAILURE
***********************************************************************/
ULONG casmw_get_seconds_from_jan11970(PCASMDATA pInstance)
{
   struct timeval   tv;
   do_gettimeofday(&tv);
   return (tv.tv_sec);

} /* casmw_get_seconds_from_jan11970 */



/*********************************************************************
** 
** @func  casmw_pci_hotplug_nmi
** 
** Routine Description:
** This routine provides the OS Specific method for 
** checking to see if a PCI Slot was opened and generated
** an NMI.
** 
** Return Value:
**      0  :  SUCCESS
**      1  :  FAILURE
***********************************************************************/
BOOLEAN casmw_pci_hotplug_nmi(PCASMDATA pInstance)
{

   BOOLEAN    bRet = SUCCESS;

   return (bRet);

} /* casmw_pci_hotplug_nmi */



/*********************************************************************
** 
** @func  casmw_get_ubuff
** 
** Routine Description:
** This is an OS Specific routine used to copy a buffer from user space
** to kernel space.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int casmw_get_ubuff( void *pUserBuff,          /* addr to copy from   */
                     void *pKernelBuff,        /* addr to copy to     */
                     ULONG buffer_len,         /* how much to copy    */
                     PCASMDATA pInstance )     /* OS Specific Data    */
{
   int            rv = SUCCESS;

   rv = (copy_from_user((void*)(pKernelBuff), (void*)(pUserBuff), (buffer_len))?-1:0);

   return (rv);

}  /* end casmw_get_ubuff */


/*********************************************************************
** 
** @func  casmw_put_ubuff
** 
** Routine Description:
** This is an OS Specific routine used to copy a buffer from kernel space
** to user space.
** 
** Return Value:
** 
** 0   : SUCCESS
** 1   : FAILURE
** 
***********************************************************************/
int casmw_put_ubuff( void *pUserBuff,          /* addr to copy from   */
                     void *pKernelBuff,        /* addr to copy to     */
                     ULONG buffer_len,         /* how much to copy    */
                     PCASMDATA pInstance )     /* OS Specific Data    */
{
   int            rv = SUCCESS;

   rv = (copy_to_user((void*)(pUserBuff), (void*)(pKernelBuff), (buffer_len))?-1:0);

   return (rv);

}  /* end casmw_put_ubuff */

/*********************************************************************
 * casmw_pci_is_device_present
 *
 * return Value:
 *
 * TRUE  : device present
 * FALSE : device not present
 *********************************************************************/
BOOLEAN casmw_pci_is_device_present(ULONG ulBus, ULONG ulDev, ULONG ulFunc)
{
   struct pci_dev *dev= pci_find_slot(ulBus, PCI_DEVFN(ulDev, ulFunc));

   return(dev ? 1 : 0);
}

/*********************************************************************
 *  casmw_pci_get_subordinate_bus 
 *
 *  Returns Value: 
 *  0      :  if not a brigde device or the device is not present
 *  subordinate bus number 
 *********************************************************************/
UCHAR casmw_pci_get_subordinate_bus(ULONG ulBus, ULONG ulDev, ULONG ulFunc)
{
   UCHAR ucSubBusNum = 0;
   struct pci_dev *pdev= pci_find_slot(ulBus, PCI_DEVFN(ulDev, ulFunc));

   // the device is present
   if(pdev)
   {
      // is it a PCI-PCI bridge
      if(pdev->class == 0x060400)  //per pci spec 
      {
         ucSubBusNum = pdev->subordinate->number;
      }
   }
   return ucSubBusNum;
}

/*********************************************************************
**
** @func  casmw_add_memory
**
** Routine Description:
** This is an OS Specific routine used to dynamically add memory
** to the server.
**
** Return Value:
**
** 0   : SUCCESS
** 1   : FAILURE
**
***********************************************************************/
BOOLEAN casmw_add_memory(PVOID pHealthData)
{
   return FAILURE;
}   
