#!/bin/sh
# $Id: $
#
# Component:	TivoliPlus (installation utilities)
#
# $Source: $
# $Revision: $
#
# Description:  Common installation support functions for TivoliPlus modules.	
#		This script should be "sourced" at the top of the TivoliPlus
#		module initialization scripts (for ALI/server and clients).
#
# External Variable References / Initialization:
#
# Author:	Glenn Zazulia
#		Tivoli Systems, Inc.
# Modifications:
#	fpc	10/1/96	made generic
#
# (C) COPYRIGHT Tivoli Systems, Inc. 1996-2000
# Unpublished Work
# All Rights Reserved
# Licensed Material - Property of Tivoli Systems, Inc.
#

######################################################################
#
#	IsValidClass class-name
#
IsValidClass ()
{
if wlookup -r Classes $1 >/dev/null 2>&1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidProfile profile-type profile-name
#
IsValidProfile ()
{

# make sure solaris doesn't stop here.
set +e

PRFNAME="$2"
PRFTYPE="$1"

#Check to see if profile manager exists
PM_CLASS=`wlookup -r Classes $PRFTYPE`
PRFMGR=`idlcall $PM_CLASS lookup_object '"'$PRFNAME'" {0}' 2> /dev/null`
RETCODE=$?

set -e

if [ $RETCODE -eq 0 ]; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidProfileManager profile-manager-name
#
IsValidProfileManager ()
{
if IsValidProfile ProfileManager "$1"; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidSentryProfile profile-name
#
IsValidSentryProfile ()
{
if IsValidProfile SentryProfile "$1"; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidPresentationObject object-name
#
IsValidPresentationObject ()
{
if IsValidProfile Presentation "$1"; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidFilePackageProfile profile-name
#
IsValidFilePackageProfile ()
{
if IsValidProfile FilePackage "$1"; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidTaskLibrary
#
IsValidTaskLibrary ()
{
if wlookup -r TaskLibrary "$1" > /dev/null 2>&1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidTaskCollection task-library-name
#
IsValidTaskCollection ()
{
LIBRARY=`wlookup Library`
COLLECTIONGUI=`idlcall $LIBRARY lookup_object '"imp_TMF_TGC::CollectionGUI" {0}'`
COLLOBJ=`idlcall $COLLECTIONGUI lookup_object '"'$NAME'" {0}'`
if [ $? -eq 0 ]; then

	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidIndicatorCollection indicator-collection-name
#
IsValidIndicatorCollection ()
{
if wlookup -r IndicatorCollection "$1#$IRONAME" >/dev/null 2>&1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	IsValidMonitorCollection monitor-collection-name
#
IsValidMonitorCollection ()
{
if IsValidProfile MonitoringCapabilityCollection "$1" >/dev/null 2>&1; then
	return 0
else
	return 1
fi
}

######################################################################
#
#	lookup_tasklib_id
#
lookup_tasklib_id() {
  [ "$DEBUG" = true ] && set -x
  match_name="$1"
  for i in $TASKLIBS; do
    eval name='"$'${i}'"'
    if [ "$name" = "$match_name" ]; then
      echo "$i"
      return 0
    fi
  done
  return 1
}

######################################################################
#
#	get_precedence_for_level
#
get_precedence_for_level() {
  [ "$DEBUG" = true ] && set -x
  pr=0
  case $1 in
    normal_zero)                       pr=0;;
    normal        | AMS_normal)        pr=1;;
    informational | AMS_informational) pr=50;;
    warning       | AMS_warning)       pr=100;;
    severe        | AMS_severe)        pr=500;;
    critical      | AMS_critical)      pr=1000;;
    fatal         | AMS_fatal)         pr=1500;;
  esac
  echo $pr
}

######################################################################
#
#       get_state_to_copy_for_level
#
get_state_to_copy_for_level() {
  [ "$DEBUG" = true ] && set -x
  state=""
  case $1 in
    normal_zero)                       state="normal#Std_Levels.0";;
    normal        | AMS_normal)        state="normal#Std_Levels.0";;
    informational | AMS_informational) state="normal#Std_Levels.0";;
    warning       | AMS_warning)       state="warning#Std_Levels.100";;
    severe        | AMS_severe)        state="severe#Std_Levels.500";;
    critical      | AMS_critical)      state="critical#Std_Levels.1000";;
    fatal         | AMS_fatal)         state="critical#Std_Levels.1000";;
  esac
  echo $state
}

######################################################################
#
#	add_cond_levels
#
add_cond_levels() {
  [ "$DEBUG" = true ] && set -x
  prfname="$1"
  levels_to_add="$2"
  r_prfname="${prfname}#${IRONAME}"
  r_indcoll="IndicatorCollection#${IRONAME}"

  for lvl in $levels_to_add; do
    precedence=`get_precedence_for_level $lvl`
    if wdellevel $lvl "$r_prfname"; then
      log_info `outmsg LinkInstallCat 118 "%1\\$s level removed for %2\\$s" "$lvl" "$r_prfname"`
    fi
    if waddlevel $lvl $precedence "$r_prfname" > /dev/null; then
      log_success `outmsg LinkInstallCat 119 "%1\\$s level added for %2\\$s, precedence %3\\$s" "$lvl" "$r_prfname" "$precedence"`
      #
      # add a new state to the IndicatorCollection for this level
      #
      state_to_copy=`get_state_to_copy_for_level $lvl`
      wgetbitmap -I "$state_to_copy" -r "$r_indcoll" > ${TEMP}/newicon.xpm
      wgetdialog -I "$state_to_copy" -r "$r_indcoll" > ${TEMP}/newdialog.d
      if eval wputicon -r "$r_indcoll" '"'$lvl'#Std_Levels.'$precedence'"' \
		  ${TEMP}/newdialog.d ${TEMP}/newicon.xpm > /dev/null; then
         log_success `outmsg LinkInstallCat 120 "%1\\$s state added to IndicatorCollection" "$lvl"`
      fi
      rm ${TEMP}/newicon.xpm ${TEMP}/newdialog.d
    fi
  done
}

######################################################################
#
#	normalize_name
#
normalize_name() {
  echo "$1" | sed 's/[^a-zA-Z0-9]/_/g'
}

######################################################################
#
#	roundoff_rate
#
roundoff_rate() {
  [ "$DEBUG" = true ] && set -x
  if [ "$2" = seconds ]; then
    mins=0
    secs=0
    if expr $1 / 60 > /dev/null 2>& 1; then
      mins=`expr $1 / 60`
    fi
    if expr $1 % 60 > /dev/null 2>& 1; then
      secs=`expr $1 % 60`
    fi
    if [ $secs -gt 30 ]; then
      mins=`expr $mins + 1`
    fi
    echo "$mins minutes"
  else
    echo "$1 $2"
  fi  
}  

######################################################################
#
#	is_predefined_dep_task
#
is_predefined_dep_task()
{

    [ "$DEBUG" = true ] && set -x

    case "$1" in
      "File Version")			return 0;;
      "File Excluded")			return 0;;
      "Required Swap Space")		return 0;;
      "Required Memory")		return 0;;
      "Required Disk Space")		return 0;;
      ".INI File Entry Excluded")	return 0;;
      "File Existence")			return 0;;
      ".INI File Entry")		return 0;;
      "OS Version")			return 0;;
      "CPUarchitecture")		return 0;;
      "Registry Entry")			return 0;;
      "Registry Entry Excluded")	return 0;;
    esac
    return 1
}

######################################################################
#
#       check_tme_server
#
check_tme_server()
{
  [ "$DEBUG" = true ] && set -x
  MNOID=`objcall 0.0.0 get_host_location`
  ALI_OID=`wlookup -r distinguished ServerManagedNode`
  if [ "$ALI_OID" = "$MNOID" ]; then
        return 0
  else
        return 1
  fi
}

######################################################################
#
#  wbindmsg utilities
#

outmsg() {

  ## turn off tracing for this routine
  ###if [ "$DEBUG" = true ]; then set +x; fi
  set +xv

  arg1=`echo "$1"`
  arg2=`echo "$2"`
  arg3=`echo "$3"`

  if [ ! "$arg3" ]; then
    eval echo '"$arg1"' '"$arg2"'
    ## turn tracing back on if applicable ...
    if [ "$DEBUG" = true ]; then set -x; fi
    return
  fi

  # got 3 or more args, we must be trying to wbindmsg ...
  file="$arg1"
  num="$arg2"
  string="`echo \"$arg3\" | sed -e 's/*/./g'`"
  shift;shift;shift
  afterstring="$*"

  ## handle up to 4 substitution args
  subarg1=`echo "$1"`; if [ "$subarg1" ]; then shift; fi
  subarg2=`echo "$1"`; if [ "$subarg2" ]; then shift; fi
  subarg3=`echo "$1"`; if [ "$subarg3" ]; then shift; fi
  subarg4=`echo "$1"`; if [ "$subarg4" ]; then shift; fi
  rest="$*"   ## not real graceful, but at least we're not throwing it away ...

  # use USER_LANG if it's set; so our wbindmsg will find the user's locale
  save_lang=$LANG
  if [ "$USER_LANG" ]; then
    LANG=$USER_LANG
  elif [ ! "$LANG" ]; then
    LANG=C
  fi

  if wbindmsg $file $num "$string" "$subarg1" "$subarg2" "$subarg3" "$subarg4" "$rest" 1> /dev/null 2>&1; then
    wbindmsg $file $num "$string" "$subarg1" "$subarg2" "$subarg3" "$subarg4" "$rest"
    echo   ## add a line ending to wbindmsg result
  else
    # couldn't use wbindmsg, but try and spit out something useful
    substitute_strings "$string" "$subarg1" "$subarg2" "$subarg3" "$subarg4" "$rest"
    if [ $? -ne 0 ]; then
      eval echo '"$string"' '"$afterstring"'
    fi
  fi

  # restore LANG
  LANG=$save_lang

  ## turn tracing back on if applicable ...
  if [ "$DEBUG" = true ]; then set -x; fi

}

errmsg() {

  ## turn off tracing for this routine ...
  ###if [ "$DEBUG" = true ]; then set +x; fi
  set +xv

  # preserve arg format for call to outmsg()
  arg1=`echo "$1"`
  arg2=`echo "$2"`
  arg3=`echo "$3"`
  if [ "$arg3" ]; then 
    shift;shift;shift
    subarg1=`echo "$1"`; if [ "$subarg1" ]; then shift; fi
    subarg2=`echo "$1"`; if [ "$subarg2" ]; then shift; fi
    subarg3=`echo "$1"`; if [ "$subarg3" ]; then shift; fi
    subarg4=`echo "$1"`; if [ "$subarg4" ]; then shift; fi
    rest="$*"
  fi

  outmsg "$arg1" "$arg2" "$arg3" "$subarg1" "$subarg2" "$subarg3" "$subarg4" $rest >& 2

  ## turn tracing back on if applicable ...
  if [ "$DEBUG" = true ]; then set -x; fi

}

substitute_strings() {

  ## turn off tracing for this routine
  ###if [ "$DEBUG" = true ]; then set +x; fi
  set +xv

  base="`echo \"$1\" | sed -e 's/*/./g'`"; if [ "$base" ]; then shift; fi
  sub1=`echo "$1"`; if [ "$sub1" ]; then shift; fi
  sub2=`echo "$1"`; if [ "$sub2" ]; then shift; fi
  sub3=`echo "$1"`; if [ "$sub3" ]; then shift; fi
  sub4=`echo "$1"`; if [ "$sub4" ]; then shift; fi
  xtra=`echo "$*"`


  base="`echo \"$base\" | sed -e 's/%1$s/$sub1/g'`"
  base="`echo \"$base\" | sed -e 's/%2$s/$sub2/g'`"
  base="`echo \"$base\" | sed -e 's/%3$s/$sub3/g'`"
  base="`echo \"$base\" | sed -e 's/%4$s/$sub4/g'`"
  eval echo '"$base"' '"$xtra"'

  ## turn tracing back on if applicable ...
  if [ "$DEBUG" = true ]; then set -x; fi

}

#
#       xlate_var -- please pass in a var name, not its value!
#                    we construct variables appending var name 
#                    with _MSGFILE and _MSGID, and use those values
#
xlate_var() {
  varname="$1"
  eval default_val='"$'${varname}'"'
  if [ ! "$default_val" ]; then
    default_val="$1"
  fi
  eval msgid='"$'${varname}'_MSGID"'
  eval msgfile='"$'${varname}'_MSGFILE"'
  outmsg $msgfile $msgid "$default_val"
}

#
############### end of wbindmsg utilities

log_init ()
{
if [ "$WINSTALL" = true ]; then
  LOG=${TEMP}/${SHORT_PRODNAME}_PLUS.log
  export LOG
  rm -f $LOG
fi
}

log_echo ()
{
if [ "$LOG" ]; then
  echo "$*" >>$LOG
else
  echo "$*"
fi
}

log_failure ()
{
log_echo "--- "$*""
}

log_success ()
{
log_echo "+++ "$*""
}

log_info ()
{
log_echo "    "$*""
}

log_start_install ()
{
DATE=`date`
log_echo `outmsg LinkInstallCat 169 "***** Start Install: %1\\$s *****" "$MEDIUM_PRODNAME_XLATED"`
log_echo `outmsg LinkInstallCat 51 "***** %1\\$s *****"` "$DATE"
}

log_start_action ()
{
LAST_ACTION="$*"
log_echo ">>> $*"
}

log_end_action ()
{
if [ $# -eq 0 ]; then
	msg=$LAST_ACTION
else
	msg="$*"
fi
log_echo `outmsg LinkInstallCat 121 "<<< %1\\$s complete!" "$msg"`
}

log_end_install ()
{
DATE=`date`
log_echo `outmsg LinkInstallCat 51 "***** $DATE *****"`
log_echo `outmsg LinkInstallCat 181 "***** End Install: %1\\$s *****" "$MEDIUM_PRODNAME_XLATED"`
}

errecho()
{
echo "$*" >&2
}

######################################################################
#
# check_apps_installed
#
# Checks which applications are installed
# Exports SENTRY_INSTALLED, COURIER_INSTALLED, TEC_INSTALLED, etc.
#	(set to true or false)
#
check_apps_installed()		# check_apps_installed <defsentry> [check_mon]
{

[ "$DEBUG" = true ] && set -x

#
#	See if Sentry is installed
#
#-#-#
log_start_action `outmsg LinkInstallCat 122 "Determine if Sentry is installed"`

if IsValidClass SentryProfile; then
	SENTRY_INSTALLED=true
	#-#-#
	log_success `outmsg LinkInstallCat 123 "Sentry is installed"`

	REGION_LEN=`odadmin | grep Region | awk '{print $3}' | wc -c`

	#Check to see if the required sentry functionality is present
	# only do this once, because there's a bug that causes it to
	# fail if you do it a second time in close proximity to the first
	if [ -z "$SENTRY_MONITORS_OK" ]; then
		if eval wlsmon -s '"'$DEFSEN'"' >/dev/null; then
			SENTRY_MONITORS_OK=true
		else
			SENTRY_MONITORS_OK=false
		fi
	fi
else
	SENTRY_INSTALLED=false
	#-#-#
	log_failure `outmsg LinkInstallCat 124 "Sentry is NOT installed"`
fi
#-#-#
log_end_action

#
#	See if Courier is installed
#
#-#-#
log_start_action `outmsg LinkInstallCat 125 "Determine if Courier is installed"`

if IsValidClass FilePackage; then
	#-#-#
	log_success `outmsg LinkInstallCat 126 "Courier is installed"`
	COURIER_INSTALLED=true
else
	#-#-#
	log_failure `outmsg LinkInstallCat 127 "Courier is NOT installed"`
	COURIER_INSTALLED=false
fi
#-#-#
log_end_action

#
#	See if the TEC is installed
#
#-#-#
log_start_action `outmsg LinkInstallCat 128 "Determine if TEC is installed"`

if IsValidClass EventServer; then
	#-#-#
	log_success `outmsg LinkInstallCat 129 "TEC is installed"`
	TEC_INSTALLED=true
else
	#-#-#
	log_failure `outmsg LinkInstallCat 130 "TEC is NOT installed"`
	TEC_INSTALLED=false
fi
#-#-#
log_end_action

#
#	See if the logfile adapter is installed
#
#-#-#
log_start_action `outmsg LinkInstallCat 131 "Determine if TEC Logfile adapter is installed"`

TECAD_DIR=$BINDIR/TME/TEC/adapters
if [ ! -d $TECAD_DIR ]; then
  TECAD_DIR=$BINDIR/../../lcf/TME/TEC/adapters
fi

if [ x\"$OS\" = x\"Windows_NT\" ] ; then
	TECAD_EXE=$TECAD_DIR/bin/tecad_nt.exe
else
	TECAD_EXE=$TECAD_DIR/bin/tecad_logfile
fi
if [ -f $TECAD_EXE ]; then
	#-#-#
	log_success `outmsg LinkInstallCat 132 "TEC Logfile adapter is installed"`
	TECAD_LOGF_INSTALLED=true
else
	#-#-#
	log_failure `outmsg LinkInstallCat 133 "TEC Logfile adapter is NOT installed"`
	TECAD_LOGF_INSTALLED=false
fi
#-#-#
log_end_action

#
#	See if the Mar stuff is installed
#
#-#-#
log_start_action `outmsg LinkInstallCat 209 "Determine if the Tivoli Manager for Application Performance is installed"`

if IsValidClass MarProfile; then
	#-#-#
	log_success `outmsg LinkInstallCat 208 "Tivoli Manager for Application Performance is installed"`
	TAPM_INSTALLED=true
else
	#-#-#
	log_failure `outmsg LinkInstallCat 206 "Tivoli Manager for Application Performance is NOT installed"`
	TAPM_INSTALLED=false
fi
#-#-#
log_end_action

#
#	See if Inventory is installed
#
#-#-#
log_start_action `outmsg LinkInstallCat 215 "Determine if Tivoli Inventory is installed"`

if IsValidClass InventoryProfile; then
	#-#-#
	log_success `outmsg LinkInstallCat 216 "Tivoli Inventory is installed"`
	INVENTORY_INSTALLED=true
else
	#-#-#
	log_failure `outmsg LinkInstallCat 214 "Tivoli Inventory NOT installed!"`
	INVENTORY_INSTALLED=false
fi
#-#-#
log_end_action
}

######################################################################
#
# add_res_bitmaps
#
add_res_bitmaps()
{

[ "$DEBUG" = true ] && set -x

if [ $# -ne 1 ]; then
	errmsg LinkInstallCat 2 "No resource specified."
	return 1
fi
RESOURCE=$1

wputbitmap -r "$RESOURCE" SLINK < $LINKDIR/SLink_icon.xpm  > /dev/null 2>&1
wputbitmap -r "$RESOURCE" CLINK < $LINKDIR/CLink_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" NCLINK < $LINKDIR/NCLink_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" SULINK < $LINKDIR/SULink_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" JLINK < $LINKDIR/JLink_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" TSPM < $LINKDIR/PM_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" TSPMD < $LINKDIR/PM_icon_dataless.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" TEC < $LINKDIR/TEC_icon.xpm > /dev/null 2>&1
wputbitmap -r "$RESOURCE" LOG < $LINKDIR/LOG_icon.xpm > /dev/null 2>&1
}

######################################################################
#
# install_link_class
#
install_link_class()
{

[ "$DEBUG" = true ] && set -x

# Install the SolutionsLink Class if not already installed
#-#-#
log_start_action `outmsg LinkInstallCat 134 "Installing Link Class"`
if IsValidClass SolutionsLink; then
	log_echo `outmsg LinkInstallCat 3 "Link installed"`
else
	Link.install
fi
#-#-#
log_success `outmsg LinkInstallCat 135 "Link Class installed"`

#-#-#
log_end_action

# This should be done as part of Link install...
add_res_bitmaps Job
add_res_bitmaps TaskLibrary

if IsValidClass EnterpriseClient; then
	wputbitmap -r EnterpriseClient TivoliPlus <$LINKDIR/Sol_icon.xpm  > /dev/null 2>&1
	wputbitmap -r EnterpriseClient JLINK <$LINKDIR/JLink_icon.xpm   > /dev/null 2>&1
fi
}


######################################################################
#
# create_task
#
# Sets up task, job, dialog, etc.
# See below for defaults.
# External dependencies:	R_TASKLIB, JOB_PM, MNNAME, PRNAME
#
create_task()	# -t <task_nm> -m <method> [-d <dialog>] [-r <role>] [-u <user>]
		#[-M <job mode>] [-T <job timeout>] [-p <pm>] [-h <mannode>]
		#[-i <icon state>] [-taskonly]
{
[ "$DEBUG" = true ] && set -x
#
# Variable initializations / defaults:
#
role=admin
user="\$root_user"
mode=parallel
timeout=0
taskonly=false
tasklib=$DEFAULT_TASKLIB

set +e
unset	task method dialog dest notarget # collection
unset	iconstate		# Choices:  SLINK MLINK CLINK NCLINK SULINK JLINK
set -e

#
# Parse arguments:
#
#		-c*) collection=$2;		shift	;;	# collection name

while [ $# -gt 0 ]; do
	case $1 in
		-taskonly) taskonly=true		;;	# Skip remaining setup
		-t*) task="$2";		shift	;;	# Task name
		-m*) method=$2;		shift	;;	# Method name
		-d*) dialog=$2;		shift	;;	# Dialog descriptor
		-r*) role="$2";		shift	;;	# Execution role
		-u*) user=$2;		shift	;;	# Execution user
		-M*) mode=$2;		shift	;;	# Job mode
		-T*) timeout=$2;	shift	;;	# Job timeout
		-i*) iconstate=$2;	shift	;;	# Icon state
		-j*) jobname=$2;	shift	;;	# Job name
		-l*) tasklib=$2;	shift	;; # Task library name
		-h*) dest="$dest -h \"$2#$IRONAME\"";	shift	;; # Job target mannode
		-p*) dest="$dest -p \"$2#$IRONAME\"";	shift	;; # Job target profmgr
		-n*) notarget=1				;;	# No Job target
		*)
			errmsg LinkInstallCat 4 "create_task:  '%1\$s' invalid." "$1"
			return 1
			;;
	esac
	if [ $# -lt 1 ]; then
		errmsg LinkInstallCat 5 "create_task:  syntax error:  %1\$s" "$*"
		return 1
	fi
	shift
done

log_info `outmsg LinkInstallCat 53 "task:        %1\\$s" "$task"`
log_info `outmsg LinkInstallCat 54 "method:      %1\\$s" "$method"`
log_info `outmsg LinkInstallCat 55 "dialog:      %1\\$s" "$dialog"`
log_info `outmsg LinkInstallCat 56 "role:        %1\\$s" "$role"`
log_info `outmsg LinkInstallCat 57 "user:        %1\\$s" "$user"`
log_info `outmsg LinkInstallCat 58 "mode:        %1\\$s" "$timeout"`
log_info `outmsg LinkInstallCat 59 "iconstate:   %1\\$s" "$iconstate"`
log_info `outmsg LinkInstallCat 60 "tasklib:     %1\\$s" "$tasklib"`
log_info `outmsg LinkInstallCat 61 "jobname:     %1\\$s" "$jobname"`
log_info `outmsg LinkInstallCat 62 "dest:        %1\\$s" "$dest"`
log_info `outmsg LinkInstallCat 63 "notarget:    %1\\$s" "$notarget"`

if [ -z "$task" -o -z "$method" ]; then
	errmsg LinkInstallCat 6 "create_task:  must specify task and method names."
	return 1
fi

if is_predefined_dep_task "$task"; then
	log_info `outmsg LinkInstallCat 64 "%1\\$s is a predefined dependency task ... skipping" "$task"`
        return 0
fi

[ -z "$dest$notarget" -a -n "$JOB_PM" ] && dest="-p \"$JOB_PM#$IRONAME\""

if [ "$user" = "$NULL" ]; then
	user=\$root_user
fi

r_tasklib="${tasklib}#${IRONAME}"

#
# Set up task
#
if [ "$USE_TLL" = true ]; then
	task=`normalize_name "$task"`   # strip blanks, since builder would have
fi

#
# force the wlstlib's to run in the C locale
#
USER_LANG=$LANG
LANG=C
export LANG

if wlstlib "$r_tasklib" | grep "^(task)	$task\$" >/dev/null 2>&1; then
	cmd=wsettask
else
	cmd=wcrttask
fi
$cmd -t "$task" -l "$r_tasklib" -r "$role" -u "$user" -i default "$MNNAME" "$method"
$taskonly && return 0

#
# Skip creating a plus icon for the predefined Product_Install task, it wouldn't make sense
#
if [ "$task" = "Product_Install" ]; then
  return 0
fi

#
# Set up job
#

# Make sure job name is unique-ified
task_job="`echo $task | sed s/\" ($SHORT_PRODNAME)$\"//` ($SHORT_PRODNAME)"

if wlstlib "$r_tasklib" | grep "^(job)	$task_job\$" >/dev/null 2>&1; then
	cmd=wsetjob
	#-#-#
	log_info `outmmsg LinkInstallCat 65 "%1\\$s job exists, overriding settings" "$task_job"`
else
	cmd=wcrtjob
	#-#-#
	log_info `outmsg LinkInstallCat 66 "%1\\$s job does NOT exist, creating" "$task_job"`
fi
eval $cmd -j '"$task_job"' -l '"$r_tasklib"' -t '"$task"' -M '"$mode"' -m $timeout -D -o 15 $dest

#
# restore LANG 
#
LANG=$USER_LANG
export LANG

#
# Finish
#
if [ "$dialog" = "$NULL" ]; then
	dialog=
fi

if [ -n "$dialog" -a ! -f "$dialog" ]; then
	log_info `outmsg LinkInstallCat 67 "Could not find dialog %1\\$s ... using default" "$dialog"`
fi
if [ -f "$dialog" ]; then
	#-#-#
	log_info `outmsg LinkInstallCat 69 "Adding dialog to %1\\$s" "$task"`
#	wputdialog -l @TaskLibrary:"$DEFAULT_TASKLIB" "$task" >/dev/null 2>&1 < $dialog
	wputdialog -l @TaskLibrary:"$r_tasklib" "$task" >/dev/null 2>&1 < $dialog
	wputdialog -l @TaskLibrary:"$r_tasklib" "$task_job" >/dev/null 2>&1 < $dialog
fi

wcrtjlink "$MEDIUM_PRODNAME" "$tasklib" "$task" "$jobname" "$task_job" $iconstate
#wcrtjlink "$collection" "$DEFAULT_TASKLIB" "$task" "$task" $iconstate
#-#-#
log_success `outmsg LinkInstallCat 70 "%1\\$s installed" "$task"`
}

######################################################################
#
# create_app_launch
#
create_app_launch()
{

[ "$DEBUG" = true ] && set -x

if [ -n "$LAUNCH_ICONS" ]; then
  if [ -z "$STDBITMAP" ]; then
    STDBITMAP="$PRODDIR/Default.xpm"
  fi
  if [ ! -f "$STDBITMAP" ]; then
    #-#-#
    log_failure `outmsg LinkInstallCat 138 "Cannot create launch icon: STDBITMAP file %1\\$s does not exist" "$STDBITMAP"`
    return 1
  fi
fi

for icon in $LAUNCH_ICONS; do
	eval LAUNCHNAME='"$'$icon'"'
	eval LAUNCHDSL='"$'$icon'_DSL"'
	eval LAUNCHMENU='"$'$icon'_MENUS"'
	eval LAUNCHDEPS='"$'$icon'_DEPS"'

	#-#-#
	log_info `outmsg LinkInstallCat 139 "Checking launch icon dependencies"`

	if check_dependencies $LAUNCHDEPS; then
		#-#-#
		log_start_action `outmsg LinkInstallCat 140 "Create AppLaunch icon %1\\$s" "$LAUNCHNAME"`

		#-#-#
		log_info `outmsg LinkInstallCat 141 "menu dialog: %1\\$s" "$LAUNCHDSL"`

		MENUITEMS=""

		for menu in $LAUNCHMENU; do
			eval MENUNAME='"$'$menu'"'
			eval MENUSCRIPT='"$'$menu'_SCRIPT"'

			#-#-#
			log_info `outmsg LinkInstallCat 142 "menu choice: %1\\$s" "$MENUNAME"`
			#-#-#
			log_info `outmsg LinkInstallCat 143 "menu script: %1\\$s/%2\\$s" "$PRODDIR" "$MENUSCRIPT"`

			MENUITEMS="$MENUITEMS $MENUSCRIPT $PRODDIR/$MENUSCRIPT "
		done

		cfgappicon "$MEDIUM_PRODNAME" "$LAUNCHNAME" $STDBITMAP $LAUNCHDSL \
			$MENUITEMS

		# add any extra dialogs to the presentation object
		for menu in $LAUNCHMENU; do
			eval MENUNAME='"$'$menu'"'
			eval MENUDIALOG='"$'$menu'_DIALOG"'
			if [ -n "$MENUDIALOG" -a -f $PRODDIR/$MENUDIALOG ]; then
				# install the callback dialog
				wputdialog -r "$LAUNCHNAME"_"$MNNAME"_"$MEDIUM_PRODNAME" \
					$MENUDIALOG >/dev/null 2>&1 < $PRODDIR/$MENUDIALOG
				# install it's required bitmap
				wputbitmap -r "$LAUNCHNAME"_"$MNNAME"_"$MEDIUM_PRODNAME" \
					APPICON <$STDBITMAP > /dev/null 2>&1
			fi
		done
	
		#-#-#
		log_success `outmsg LinkInstallCat 70 "%1\\$s installed" "$LAUNCHNAME"`
	fi

	#-#-#
	log_end_action
done
}

######################################################################
#
# create_subscription_lists
#
create_subscription_lists()
{

[ "$DEBUG" = true ] && set -x

for sublist in $SUBLIST_PMS; do
	eval SUBLIST='"$'$sublist'"'
	eval SUBLIST_ICONDSL='"$'$sublist'_ICONDSL"'
	eval SUBLIST_ICONSTATE='"$'$sublist'_ICONSTATE"'
	eval SUBLIST_DATALESS='"$'$sublist'_DATALESS"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 144 "Create Subscription List %1\\$s" "$SUBLIST"`

	if [ -n "$SUBLIST_ICONDSL" -a -f $PRODDIR/$SUBLIST_ICONDSL ]; then

		wputicon -r ProfileManager $SUBLIST_ICONSTATE	\
						$PRODDIR/$SUBLIST_ICONDSL \
						$LINKDIR/PM_icon.xpm

		if [ "$SUBLIST_DATALESS" = "TRUE" ]; then
			wcrtsprfmgr "$MEDIUM_PRODNAME" "$SUBLIST" "dataless"
		else
			wcrtsprfmgr "$MEDIUM_PRODNAME" "$SUBLIST" $SUBLIST_ICONSTATE
		fi
	else
		if [ -n "$SUBLIST_ICONDSL" ]; then
		  log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$SUBLIST_ICONDSL"`
		fi

		if [ "$SUBLIST_DATALESS" = "TRUE" ]; then
			wcrtsprfmgr "$MEDIUM_PRODNAME" "$SUBLIST" "dataless"
		else
			wcrtsprfmgr "$MEDIUM_PRODNAME" "$SUBLIST" 
		fi
	fi


	#-#-#
	log_end_action
done

#
# add subscription list's subscribers
#
for sublist in $SUBLIST_PMS; do
	eval SUBLIST='"$'$sublist'"'
	eval SUBLIST_TARGETS='"$'$sublist'_TARGETS"'

	for target in $SUBLIST_TARGETS; do
		eval subscriber='"$'$target'"'
		if wsub @ProfileManager:"$SUBLIST#$IRONAME" @ProfileManager:"$subscriber#$IRONAME"; then
			:
		fi
	done
done

}

######################################################################
#
# sublist_task_name
#
sublist_task_name()
{
  outmsg LinkInstallCat 7 "Create_Subscriber_List"
}

######################################################################
#
# create_sublist_task
#
create_sublist_task()
{

[ "$DEBUG" = true ] && set -x

	if $LINKDIR/PLUStme_server.sh; then
		#-#-#
		log_start_action `outmsg LinkInstallCat 145 "Create Subscriber List Task"`

		TASK_NAME="`sublist_task_name`"
		TASK_ICON=$LINKDIR/SULink_icon.xpm
		TASK_METHOD=$LINKDIR/PLUScreate_sublist_task.sh
		TASK_DIALOG=$LINKDIR/PLUScreate_sublist_task_sh.d
		TASK_ICONDSL=$LINKDIR/PLUScreate_sublist_task_icon.d
		TASK_ICONSTATE=STATE_PLUScreate_sublist_task_icon.d
		TASK_USER=$NULL

		LINK_CO=`wlookup -r Classes SolutionsLink`
		initpres $LINK_CO -class -put \
			state "$TASK_ICONSTATE" $TASK_ICONDSL \
				xpm $TASK_ICON
	
		create_task \
			-t "$TASK_NAME" \
			-m $TASK_METHOD \
			-d $TASK_DIALOG \
			-h "$MNNAME" \
			-r senior \
			-u $TASK_USER \
			-M parallel \
			-T 60 \
			-j "$TASK_NAME" \
			-i $TASK_ICONSTATE

		#-#-#
		log_end_action
	fi
}

######################################################################
#
# add_host_to_subscription_lists
#
add_host_to_subscription_lists()
{ 

[ "$DEBUG" = true ] && set -x

for sublist in $SUBLIST_PMS; do
	eval R_SUBLIST='"$R_'$sublist'"'
	eval SUBLIST_DEPS='"$'$sublist'_DEPS"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 146 "Add host %1\\$s to Subscription list %2\\$s" "$MNNAME" "$R_SUBLIST"`

	#
	# if there's a subscription list validation script, unsubscribe
	#	the host first, then check the validity and re-subscribe
	#	if appropriate
	#
	if [ -z "$SUBLIST_DEPS" ]; then
		wsub @ProfileManager:"$R_SUBLIST" @ManagedNode:$MNNAME#$IRONAME
	else
		wunsub -l @ProfileManager:"$R_SUBLIST" @ManagedNode:$MNNAME#$IRONAME
		ADD_HOST=true
		for dep in $SUBLIST_DEPS; do
			eval DEPSCRIPT='"$'$dep'"'

			if [ -f "$PRODDIR/$DEPSCRIPT" ]; then
				#-#-#
				log_info `outmsg LinkInstallCat 147 "Executing dependency script %1\\$s" "$DEPSCRIPT"`

				set +e
				$PRODDIR/$DEPSCRIPT
				DEPRETCODE=$?
				set -e

				if [ $DEPRETCODE -eq 0 ]; then
					#-#-#
					log_success `outmsg LinkInstallCat 148 "%1\\$s satisfies dependency script %2\\$s" "$MNNAME" "$DEPSCRIPT"`
				else
					#-#-#
					log_failure `outmsg LinkInstallCat 149 "%1\\$s does not satisfy dependency script %2\\$s" "$MNNAME" "$DEPSCRIPT"`
					ADD_HOST=false
				fi
			fi
		done

		if [ $ADD_HOST = "true" ]; then
		  log_success `outmsg LinkInstallCat 150 "Adding %1\\$s to sublist %2\\$s" "$MNNAME" "$R_SUBLIST"`
		  wsub @ProfileManager:"$R_SUBLIST" @ManagedNode:$MNNAME#$IRONAME
		else 
		  log_failure `outmsg LinkInstallCat 151 "Not adding %1\\$s to sublist %2\\$s" "$MNNAME" "$R_SUBLIST"`
		fi

	fi
	#-#-#
	log_end_action
done
}

######################################################################
#
# check_dependencies
#
check_dependencies()
{

[ "$DEBUG" = true ] && set -x
#
# save the arguments locally first, to prevent hpux9s from losing them
#
DEPSCRIPTS=""
for arg in $*; do
	eval SCRIPTNAME='"$'$arg'"'
	DEPSCRIPTS="$DEPSCRIPTS $SCRIPTNAME"
done

if [ -n "$DEPSCRIPTS" ]; then
	#-#-#
	log_info `outmsg LinkInstallCat 152 "Checking dependencies"`
else
	#-#-#
	log_info `outmsg LinkInstallCat 153 "No dependencies specified"`
	return 0
fi

for script in $DEPSCRIPTS; do
	if [ -f "$PRODDIR/$script" ]; then
		#-#-#
		log_info `outmsg LinkInstallCat 147 "Executing dependency script %1\\$s" "$script"`

		set +e
		$PRODDIR/$script > /dev/null
		DEPRETCODE=$?
		set -e

		if [ $DEPRETCODE -ne 0 ]; then
			#-#-#
			log_failure `outmsg LinkInstallCat 154 "Failed dependency"`
			return 1
		fi
	fi
done

	#-#-#
log_success `outmsg LinkInstallCat 155 "Passed dependency check"`
return 0
}

######################################################################
#
# create_task_collections
#
create_task_collections()
{
#
#	Always create this one.
#
#-#-#
log_start_action `outmsg LinkInstallCat 156 "Create TaskLibrary %1\\$s" "$DFLT_TASKLIB"`
if IsValidTaskLibrary "$DEFAULT_TASKLIB"; then
	#-#-#
	log_success `outmsg LinkInstallCat 71 "%1\\$s exists" "$DFLT_TASKLIB"`
else
	if wcrttlib "$DEFAULT_TASKLIB" "$MEDIUM_PRODNAME#$IRONAME"; then
		#-#-#
		log_success `outmsg LinkInstallCat 72 "%1\\$s created" "$DFLT_TASKLIB"`
	else
		#-#-#
		log_failure `outmsg LinkInstallCat 157 "Failed to create %1\\$s" "$DFLT_TASKLIB"`
	fi
fi

#-#-#
log_end_action

#
#	Create the rest.
#
for tasklib in $TASKLIBS; do
	eval TASKLIB='"$'$tasklib'"'
	#-#-#
#	log_start_action `outmsg LinkInstallCat 158 "Create TaskCollection %1\\$s" "$TASKLIB"`
	log_start_action `outmsg LinkInstallCat 156 "Create TaskLibrary %1\\$s" "$TASKLIB"`
	if IsValidTaskLibrary "$TASKLIB"; then
		#-#-#
		log_success `outmsg LinkInstallCat 71 "%1\\$s exists" "$TASKLIB"`
	else
	  if [ -n "$TASKLIB" ]; then
		if wcrttlib "$TASKLIB" "$MEDIUM_PRODNAME#$IRONAME" 2>/dev/null; then
			#-#-#
			log_success `outmsg LinkInstallCat 72 "%1\\$s created" "$TASKLIB"`
		else
			#-#-#
			log_failure `outmsg LinkInstallCat 157 "Failed to create %1\\$s" "$TASKLIB"`
			return 1
		fi
	  fi
	fi
	#-#-#
	log_end_action
done
}

######################################################################
#
# create_tasks
#
create_tasks ()
{

[ "$DEBUG" = true ] && set -x

for task in $TASK_LIST; do
	eval TASKNAME='"$'$task'"'
	eval TASKJOB='"$'$task'_JOB"'
	eval TASKSCRIPT='"$'$task'_SCRIPT"'
	eval TASKUSER='"$'$task'_USER"'
	eval TASKROLE='"$'$task'_ROLE"'
	eval TASKDIALOG='"$'$task'_DIALOG"'
	eval TASKLIBR='"$'$task'_LIBR"'
	eval TASKDEPENDS='"$'$task'_DEPS"'
	eval TASKICONDSL='"$'$task'_ICONDSL"'
	eval TASKICONSTATE='"$'$task'_ICONSTATE"'
	eval TASKTARGETS='"$'$task'_TARGETS"'

	# _about and _create_sublist tasks should only be created on the TMR server
	if [ "$TASKSCRIPT" = "_about.sh" -o "$TASKSCRIPT" = _create_sublist.sh ]; then
		if $LINKDIR/PLUStme_server.sh; then
			TASKTARGETS=""
		else
			return
		fi
	fi

	#-#-#
	log_start_action `outmsg LinkInstallCat 159 "Create %1\\$s Task %2\\$s" "$TASK_TYPE" "$TASKNAME"`

	# go ahead and expand $TASKICONDSL and $TASKDIALOG
	TASKICONDSL=$PRODDIR/$TASKICONDSL
	if [ "$TASKSCRIPT" = "_create_sublist.sh" ]; then
		TASKDIALOG=$LINKDIR/PLUScreate_sublist_task_sh.d
		TASKICONDSL=$LINKDIR/PLUScreate_sublist_task_icon.d
	elif [ -z "$TASKDIALOG" -o ! -f $PRODDIR/$TASKDIALOG ] ; then
		TASKDIALOG=$NULL
	else
		TASKDIALOG=$PRODDIR/$TASKDIALOG
	fi

	if [ -z "$TASKLIBR" ] ; then
		TASKLIBR=$DEFAULT_TASKLIB
	fi

	if [ -z "$TASKUSER" ] ; then
		TASKUSER=$NULL
	fi

	if [ ! -f $TASKDIALOG ]; then
		log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$TASKDIALOG"`
	fi

	# strip the commas out of the list of Tivoli roles for this task, and 
	# replace them with colons for the w*task call
	if [ -n "$TASKROLE" ]; then
		TASKROLE=" -r \"`echo $TASKROLE | sed -e 's/,/:/g' -e 's/ //g'`\""
	else
		TASKROLE=" -r \"admin\""
	fi

	if [ ! -f $TASKICONDSL ]; then
		log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$TASKICONDSL"`
		TASKICONSTATE=JLINK
	else
		LINK_CO=`wlookup -r Classes SolutionsLink`
		initpres $LINK_CO -class -put \
			state "$TASKICONSTATE" $TASKICONDSL \
				xpm $LINKDIR/$TASK_ICON
	fi
	
	if check_dependencies $TASKDEPENDS; then
		if [ -z "$TASKTARGETS" ]; then
			eval create_task \
				-t \"$TASKNAME\" \
				-m $PRODDIR/$TASKSCRIPT \
				-d $TASKDIALOG \
				-h \"$MNNAME\" \
				-i \"$TASKICONSTATE\" \
				-j \"$TASKJOB\" \
				-l \"$TASKLIBR\" \
				"$TASKROLE" \
				-u $TASKUSER \
				-M parallel \
				-T 0
		else
			TARGETS=""
			for target in $TASKTARGETS; do
				eval TARG='"$'$target'"'
				TARGETS="$TARGETS -p \"$TARG\""
			done

			eval create_task \
				-t \"$TASKNAME\" \
				-m $PRODDIR/$TASKSCRIPT \
				-d $TASKDIALOG \
				$TARGETS \
				-i \"$TASKICONSTATE\" \
				-j \"$TASKJOB\" \
				-l \"$TASKLIBR\" \
				"$TASKROLE" \
				-u $TASKUSER \
				-M parallel \
				-T 0
		fi
	fi

	#-#-#
	log_end_action
done
}

######################################################################
#
# create_administrative_tasks
#
create_administrative_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$TASKS"
TASK_TYPE=`outmsg LinkInstallCat 201 "Administrative"`
TASK_ICON=JLink_icon.xpm

	#-#-#
	log_start_action `outmsg LinkInstallCat 160 "Create Administrative Task %1\\$s" "$TASKNAME"`

if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $TASK_LIST
fi
create_tasks

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_plus_config_tasks
#
create_plus_config_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$CONFIGTASKS $ABOUTTASKS"
TASK_TYPE=`outmsg LinkInstallCat 202 "Plus Configuration"`
TASK_ICON=SULink_icon.xpm

	#-#-#
	log_start_action `outmsg LinkInstallCat 161 "Create Plus Config Task %1\\$s" "$TASKNAME"`

if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $TASK_LIST
fi
create_tasks

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_tec_config_tasks
#
create_tec_config_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$TECTASKS"
TASK_TYPE=`outmsg LinkInstallCat 203 "TEC Configuration"`
TASK_ICON=TEC_icon.xpm

	#-#-#
	log_start_action `outmsg LinkInstallCat 162 "Create TEC Config Task %1\\$s" "$TASKNAME"`

if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $TASK_LIST
fi
create_tasks

	#-#-#
	log_end_action
	
}

######################################################################
#
# create_log_config_tasks
#
create_log_config_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$LOGTASKS"
TASK_TYPE=`outmsg LinkInstallCat 204 "Logfile Adapter Configuration"`
TASK_ICON=LOG_icon.xpm

	#-#-#
	log_start_action `outmsg LinkInstallCat 163 "Create Logfile Adapter Config Task %1\\$s" "$TASKNAME"`

if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $TASK_LIST
fi
create_tasks

	#-#-#
	log_end_action
	
}


######################################################################
#
# create_adapter_config_tasks
#
create_adapter_config_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$ADAPTERTASKS"
TASK_TYPE=`outmsg LinkInstallCat 204 "Custom Adapter Configuration"`
TASK_ICON=LOG_icon.xpm

	#-#-#
	log_start_action `outmsg LinkInstallCat 163 "Create Custom Adapter Config Task %1\\$s" "$TASKNAME"`

if [ "$USE_TLL" = true ]; then
    compile_tll_for_tasks $TASK_LIST
fi
create_tasks

	#-#-#
	log_end_action
}


######################################################################
#
# create_sentry_hidden_tasks
#
create_sentry_hidden_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$SENTRYTASKS"

compile_tll_for_tasks $TASK_LIST
create_hidden_tasks

}

######################################################################
#
# create_tec_hidden_tasks
#	This creates Hidden tasks for Sentry and T/EC.
#
create_tec_hidden_tasks ()
{
[ "$DEBUG" = true ] && set -x
TASK_LIST="$HIDDENTASKS"

compile_tll_for_tasks $TASK_LIST
create_hidden_tasks

}

######################################################################
#
# create_hidden_tasks
#	This creates Hidden tasks for Sentry and T/EC.
#
create_hidden_tasks ()
{

[ "$DEBUG" = true ] && set -x

for task in $TASK_LIST; do
	eval TASKNAME='"$'$task'"'
	eval TASKSCRIPT='"$'$task'_SCRIPT"'
	eval TASKUSER='"$'$task'_USER"'
	eval TASKROLE='"$'$task'_ROLE"'
	eval TASKDEPENDS='"$'$task'_DEPS"'
	eval TASKLIBR='"$'$task'_LIBR"'

	if [ -z "$TASKUSER" ] ; then
		TASKUSER=$NULL
	fi

	if [ -z "$TASKLIBR" ] ; then
		TASKLIBR=$DEFAULT_TASKLIB
	fi

	if [ -n "$TASKROLE" ]; then
		TASKROLE=" -r \"`echo $TASKROLE | sed -e 's/,/:/g' -e 's/ //g'`\""
	else
		TASKROLE=" -r \"admin\""
	fi

	#-#-#
	log_start_action `outmsg LinkInstallCat 164 "Create Hidden Task %1\\$s" "$TASKNAME"`

	if check_dependencies $TASKDEPENDS; then
		eval create_task \
			-t \"$TASKNAME\" \
			-m $PRODDIR/$TASKSCRIPT \
			"$TASKROLE" \
			-u $TASKUSER \
			-l \"$TASKLIBR\" \
			-taskonly
	fi
	#-#-#
	log_end_action
done
}

######################################################################
#
# compile_tll_for_tasks
#
compile_tll_for_tasks()
{
  [ "$DEBUG" = true ] && set -x

  # first compile all tll files associated with the tasks
  processed_libs=""
  for task in $*; do
    task_id="$task"
    eval taskdeps='"$'${task_id}'_DEPS"'
    eval libname='"$'${task_id}'_LIBR"'
    if [ -z "$libname" ] ; then
	libname="$DEFAULT_TASKLIB"
    fi
    r_libname="${libname}#${IRONAME}"
    if lookup_tasklib_id "$libname"; then
      lib_id=`lookup_tasklib_id "$libname"`
      eval tll_file='"$'${lib_id}'_TLL"'
    fi

    #-#-#
    log_start_action `outmsg LinkInstallCat 165 "Create TLL Tasks in TaskLibrary %1\\$s" "$libname"`

    if check_dependencies $taskdeps; then

      # if a tll file is specified and we haven't already, then compile it now!
      if [ "${tll_file}" ]; then
        found=false
        for j in $processed_libs; do
          if [ "$j" = "$lib_id" ]; then
            found=true
          fi
        done
        if $found; then
          :
        else  
          processed_libs="$lib_id $processed_libs"
          if [ -f ${PRODDIR}/${tll_file} ]; then
	    cpp_cmd=`cpp_path`
	    if [ "$cpp_cmd" ]; then
	      cpp_opt="-P"    # tell cpp to suppress next-phase directives 
	    else
	      cpp_cmd=cat
	      cpp_opt=""
	    fi
            tmp_tll=${TEMP}/${tll_file}
            log_info `outmsg LinkInstallCat 166 "       adding all tasks defined in \"%1\\$s\"" "$tll_file"`
            # remove template comments, just in case cat is our preprocessor
            sed s/"TaskLibrary.*"/"TaskLibrary \"${libname}\""/ "${PRODDIR}/${tll_file}" \
		| grep -v '^.\*' > ${tmp_tll}
            wtll -r -p "${MEDIUM_PRODNAME}#${IRONAME}" -P $cpp_cmd "${tmp_tll}" $cpp_opt
          else
            log_info `outmsg LinkInstallCat 167 "       \"%1\\$s/%2\\$s\" not found!" "${PRODDIR}" "$tll_file"`
            return 1
          fi
        fi
      fi

    fi   # if check_dependencies

    #-#-#
    log_end_action

  done

}

######################################################################
#
# create_file_packages
#
create_file_packages()
{
#
#	Create the OmniFilePackage profile, and also create a hidden file
#	package for each one defined.
#

[ "$DEBUG" = true ] && set -x


for fp in $FILEPACKS; do
	eval FILEPACKNAME='"$'$fp'"'
	eval FPNAME='"$'$fp'_SHORTNAME"'
	eval FPDEF='"$'$fp'_FPDEF"'
	eval FPCFG='"$'$fp'_SCRIPT"'
	eval FPDIALOG='"$'$fp'_DIALOG"'
	eval FPCONFIG='"$'$fp'_CONFIG_TASKS"'
	eval FPICONDSL='"$'$fp'_ICONDSL"'
	eval FPICONSTATE='"$'$fp'_ICONSTATE"'
	eval FPTARGETS='"$'$fp'_TARGETS"'
	eval FPDEPS='"$'$fp'_DEPS"'
	eval FPDATALESS='"$'$fp'_DATALESS"'

	TARGETS=""
	if [ -n "$FPTARGETS" ]; then
		for target in $FPTARGETS; do
			eval TARG='"$'$target'"'
			TARGETS="$TARGETS \"@ProfileManager:$TARG\""
		done
	fi

	#-#-#
	log_info `outmsg LinkInstallCat 168 "Checking filepack dependencies"`

	if check_dependencies $FPDEPS; then
		#-#-#
		log_start_action `outmsg LinkInstallCat 80 "Create File Package %1\\$s" "$FILEPACKNAME"`

		#-#-#
		log_info `outmsg LinkInstallCat 170 "Creating state %1\\$s" "$FPICONSTATE"`

		if [ -n "$FPICONDSL" -a ! -f $PRODDIR/$FPICONDSL ]; then
			log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$FPICONDSL"`
		fi
		if [ -n "$FPICONDSL" -a -f $PRODDIR/$FPICONDSL ]; then
			LINK_CO=`wlookup -r Classes SolutionsLink`

			if [ "$FPDATALESS" = "TRUE" ]; then
				initpres $LINK_CO -class -put \
					state "$FPICONSTATE" $PRODDIR/$FPICONDSL \
					xpm $LINKDIR/CLink_dataless_icon.xpm
			else
				initpres $LINK_CO -class -put \
					state "$FPICONSTATE" $PRODDIR/$FPICONDSL \
					xpm $LINKDIR/NCLink_icon.xpm
			fi
		else
			FPICONSTATE=NCLINK
		fi
	
		# Create the file package
		#-#-#
		log_info `outmsg LinkInstallCat 171 "Creating Profile Manager %1\\$s" "$FILEPACKNAME"`

		if IsValidProfileManager "$FILEPACKNAME"; then
			#-#-#
			log_info `outmsg LinkInstallCat 172 "ProfileManager \"%1\\$s\" exists!" "$FILEPACKNAME"`
		else
			#Create the profile manager
			#-#-#
			log_info `outmsg LinkInstallCat 173 "ProfileManager \"%1\\$s\" does NOT exist, creating" "$FILEPACKNAME"`
			wcrtprfmgr @PolicyRegion:"$MEDIUM_PRODNAME#$IRONAME" "$FILEPACKNAME"
		fi
	
		if IsValidFilePackageProfile "$FILEPACKNAME"; then
			#-#-#
			log_info `outmsg LinkInstallCat 174 "FilePackage Profile \"%1\\$s\" exists!" "$FILEPACKNAME"`
		else
			#-#-#
			log_info `outmsg LinkInstallCat 175 "Creating file package profile \"%1\\$s\"" "$FILEPACKNAME"`
			wcrtprf @ProfileManager:"$FILEPACKNAME#$IRONAME" \
				FilePackage "$FILEPACKNAME" 2>&1 > /dev/null
		fi

		#-#-#
		log_success  `outmsg LinkInstallCat 176 "File Package %1\\$s creation successful" "$FILEPACKNAME"`

		#-#-#
		log_end_action


		if [ "$FPCONFIG" != "TRUE" ]; then
			#-#-#
			log_info `outmsg LinkInstallCat 177 "No fp config tasks specified for \"%1\\$s\"!" "$FILEPACKNAME"`
		else
			#-#-#
			log_start_action `outmsg LinkInstallCat 84 "Creating OmniFilePackage \"%1\\$s\"" "$FILEPACKNAME"`

			
			eval wcrtsprf $FPICONSTATE \"$MEDIUM_PRODNAME\" OmniFilePackage \
					\"$FILEPACKNAME\" \"$FILEPACKNAME\" $TARGETS

			#-#-#
			log_success `outmsg LinkInstallCat 85 "OmniFilePackage \"%1\\$s\" creation successful" "$FILEPACKNAME"`
			#-#-#
			log_end_action
		fi
	fi

done

#
#	Import each file package description.  This is done as a separate step
#	so we don't have to determine the order to create filepackages because
#	of nesting.
#
for fp in $FILEPACKS; do
	eval FILEPACKNAME='"$'$fp'"'
	eval FPNAME='"$'$fp'_SHORTNAME"'
	eval FPDEF='"$'$fp'_FPDEF"'
	eval FPCFG='"$'$fp'_SCRIPT"'
	eval FPDIALOG='"$'$fp'_DIALOG"'
	eval FPDEPS='"$'$fp'_DEPS"'

	#-#-#
	log_info `outmsg LinkInstallCat 79 "Checking filepack dependencies"`

	if check_dependencies $FPDEPS; then
		#-#-#
		log_start_action `outmsg LinkInstallCat 86 "Importing definition into File Package %1\\$s" "$FILEPACKNAME"`

		# Import the file package definition
		#-#-#
		log_info `outmsg LinkInstallCat 87 "Importing file package definition from %1\\$s" "$FPDEF"`

		wimprtfp -h @ManagedNode:"$MNNAME#$IRONAME" \
			@FilePackage:"$FILEPACKNAME#$IRONAME" < $PRODDIR/$FPDEF

		wsetfpopts -T gen -S $PRODDIR @FilePackage:"$FILEPACKNAME#$IRONAME"

		#-#-#
		log_success `outmsg LinkInstallCat 88 "File Package %1\\$s definition import successful" "$FILEPACKNAME"`
		#-#-#
		log_end_action
	fi
done

#
#	Create the configuration tasks
#
if [ "$USE_TLL" = true ]; then
	compile_tll_for_tasks $FILEPACKTASKS
fi

for fp in $FILEPACKTASKS; do
	eval FPTASKDEPS='"$'$fp'_DEPS"'

	if check_dependencies $FPTASKDEPS; then
		eval FPNAME='"$'$fp'"'
		eval FPJOB='"$'$fp'_JOB"'
		eval FPSCRIPT='"$'$fp'_SCRIPT"'
		eval FPUSER='"$'$fp'_USER"'
		eval FPROLE='"$'$fp'_ROLE"'
		eval FPDIALOG='"$'$fp'_DIALOG"'
		eval FPTASKICONDSL='"$'$fp'_ICONDSL"'
		eval FPTASKICONSTATE='"$'$fp'_ICONSTATE"'
		eval FPTASKLIBR='"$'$fp'_LIBR"'

		if [ -z "$FPUSER" ] ; then
			FPUSER=$NULL
		fi

		#-#-#
		log_start_action `outmsg LinkInstallCat 89 "Create Configurable File Package %1\\$s" "$FPNAME"`

		if [ -n "$FPDIALOG" -a ! -f $PRODDIR/$FPDIALOG ]; then
			log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$FPDIALOG"`
		fi
		if [ -z "$FPDIALOG" -o ! -f $PRODDIR/$FPDIALOG ] ; then
			FPDIALOG=$NULL
		else
			FPDIALOG=$PRODDIR/$FPDIALOG
		fi

		if [ -n "$FPROLE" ]; then
			FPROLE=" -r \"`echo $FPROLE | sed -e 's/,/:/g' -e 's/ //g'`\""
		else
			FPROLE=" -r \"senior\""
		fi

		if [ -n "$FPTASKICONDSL" -a ! -f $PRODDIR/$FPTASKICONDSL ]; then
			log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$FPTASKICONDSL"`
		fi
		if [ -n "$FPTASKICONDSL" -a -f $PRODDIR/$FPTASKICONDSL ]; then
			LINK_CO=`wlookup -r Classes SolutionsLink`
			initpres $LINK_CO -class -put \
				state "$FPTASKICONSTATE" $PRODDIR/$FPTASKICONDSL \
					xpm $LINKDIR/SULink_icon.xpm
		else
			FPTASKICONSTATE=SULINK
		fi
	
		eval create_task \
			-t \"$FPNAME\" \
			-m $PRODDIR/$FPSCRIPT \
			-d $FPDIALOG \
			-h \"$MNNAME\" \
			"$FPROLE" \
			-l \"$FPTASKLIBR\" \
			-u $FPUSER \
			-M parallel \
			-T 60 \
			-j \"$FPJOB\" \
			-i $FPTASKICONSTATE

		#-#-#
		log_end_action
	fi
done
}

######################################################################
#
# create_sentry_profiles
#
create_sentry_profiles()
{
#
# Create Indicator Collections
#

[ "$DEBUG" = true ] && set -x

for indcoll in $INDCOLLS; do
	eval INDCOLL='"$'$indcoll'"'
	eval R_INDCOLL='"$R_'$indcoll'"'
	eval COLLDEPENDS='"$'$indcoll'_DEPS"'
	eval COLL_ICONDSL='"$'$indcoll'_ICONDSL"'
	eval COLL_ICONSTATE='"$'$indcoll'_ICONSTATE"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 90 "Create Indicator collection %1\\$s" "$INDCOLL"`

        if check_tme_server; then
		if IsValidIndicatorCollection "$INDCOLL"; then
			#make sure it gets added:
			SNTCOLL=`wlookup -r IndicatorCollection "$R_INDCOLL"`
			gcoadd "$COLLOBJ" "$SNTCOLL"

			#-#-#
			log_success `outmsg LinkInstallCat 71 "%1\\$s exists" "$INDCOLL"`
		else
			SNTCOLL=`wcrtsntcoll "$MEDIUM_PRODNAME#$IRONAME" "$INDCOLL"`
			if [ -n "$COLL_ICONDSL" -a ! -f $PRODDIR/$COLL_ICONDSL ]; then
			  log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$COLL_ICONDSL"`
			fi
			if [ -n "$COLL_ICONDSL" -a -f $PRODDIR/$COLL_ICONDSL ]; then
				wputicon -r IndicatorCollection $COLL_ICONSTATE \
					$PRODDIR/$COLL_ICONDSL \
					$INST_DIR/generic_unix/TME/PLUS/LINK/PM_icon.xpm
				idlcall "$SNTCOLL"  _set_state '"'$COLL_ICONSTATE'"'
			fi
			gcoadd "$COLLOBJ" "$SNTCOLL"
			log_success `outmsg LinkInstallCat 72 "%1\\$s created" "$INDCOLL"`
		fi
	fi
	#-#-#
	log_end_action
done
        
#
# Create Sentry Profile Managers
#
for sentrypm in $SENTRY_PMS; do
	eval PMNAME='"$'$sentrypm'"'; PMNAME="`echo $PMNAME`"  # remove imbedded blanks
	eval R_PMNAME='"$R_'$sentrypm'"'; R_PMNAME="`echo $R_PMNAME`"  # remove imbedded blanks
	eval SPM_INDCOLLS='"$'$sentrypm'_INDCOLLS"'
	eval SPM_DEPENDS='"$'$sentrypm'_DEPS"'
	eval ICONDSL='"$'$sentrypm'_ICONDSL"'
	eval ICONSTATE='"$'$sentrypm'_ICONSTATE"'
	eval PM_TARGETS='"$'$sentrypm'_TARGETS"'
	eval PM_DATALESS='"$'$sentrypm'_DATALESS"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 182 "Create Sentry Profile Manager %1\\$s" "$PMNAME"`

        if check_tme_server; then
		# Set the subscrition list
		if [ -z "$PM_TARGETS" ]; then
			TARGETS=@ManagedNode:$MNNAME
		else
			TARGETS=""
			for target in $PM_TARGETS; do
				eval TARG='"$'$target'"'
				TARGETS="$TARGETS \"@ProfileManager:$TARG\""
			done
		fi

		if IsValidSentryProfile "$PMNAME"; then
			log_info `outmsg LinkInstallCat 92 "%1\\$s exists, updating subscription list" "$R_PMNAME"`
			eval wpmsub \"$PMNAME\" $TARGETS
		else
			if [ -n "$ICONDSL" -a ! -f $PRODDIR/$ICONDSL ]; then
				log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$ICONDSL"`
			fi
			if [ -n "$ICONDSL" -a -f $PRODDIR/$ICONDSL ]; then
				LINK_CO=`wlookup -r Classes SolutionsLink`

				if [ "$PM_DATALESS" = "TRUE" ]; then
					initpres $LINK_CO -class -put \
						state "$ICONSTATE" $PRODDIR/$ICONDSL \
						xpm $LINKDIR/SLink_dataless_icon.xpm
				else
					initpres $LINK_CO -class -put \
						state "$ICONSTATE" $PRODDIR/$ICONDSL \
						xpm $LINKDIR/SLink_icon.xpm
				fi
			else
				ICONSTATE=SLINK
			fi

			eval wcrtsprf $ICONSTATE \"$MEDIUM_PRODNAME\" SentryProfile \
				\"$PMNAME\" \
				\"$PMNAME\" \
				$TARGETS
		fi
        
		if IsValidSentryProfile "$PMNAME"; then
			# Profile must run as $root_user
			wsetsntid 0 0 "$R_PMNAME"

			#add precedence levels
			for indcoll in $SPM_INDCOLLS; do
				eval INDCOLL='"$'$indcoll'"'
				eval R_INDCOLL='"$'$indcoll'#$IRONAME"'

				if wsetcoll "$R_INDCOLL" "$R_PMNAME"; then
					log_success `outmsg LinkInstallCat 93 "done"`
				else
					#-#-#
					log_success `outmsg LinkInstallCat 94 "%1\\$s in %2\\$s" "$R_PMNAME" "$INDCOLL"`
				fi
			done

		        add_cond_levels "$PMNAME" "$TIV_COND_LEVELS"

    		fi
	fi

	#-#-#
	log_end_action
done

#
# Create Sentry Monitors Instances
#
for mon in $MONITORS; do
	eval MONITOR='"$'$mon'"'
	eval MONCOLL='"$'$mon'_MONCAPCOL"'; MONCOLL="`echo $MONCOLL`"  # remove imbedded blanks
	eval MONCOLLFILE='"$'$mon'_MONCAPCOLFILE"'
	eval MONPROG='"$'$mon'_PROG"'
	eval MONARGS='"$'$mon'_ARGS"'
	eval MONSAMP='"$'$mon'_SAMP"'
	eval MONCONDS='"$'$mon'_CONDS"'
	eval MONPMVAR='"$'$mon'_PMVAR"'; MONPMVAR="`echo $MONPMVAR`"  # remove imbedded blanks
	eval MONPMDEPS='"$'$MONPMVAR'_DEPS"'
	eval R_MONPM='"$'$mon'_PM#$IRONAME"'; R_MONPM="`echo $R_MONPM`"  # remove imbedded blanks

	# the following rounds seconds to minutes as a workaround for DM 
	# not handling seconds
	if [ "$MONSAMP" ]; then	
	        MONSAMP=`roundoff_rate $MONSAMP`
	fi


        if check_tme_server; then
		log_info `outmsg LinkInstallCat 183 "PM Dependencies passed"`

		if [ -z "$MONCOLL" ]; then
			MONCOLL=Unix_Sentry
		fi

		if [ -z "$MONCOLLFILE" ]; then
			MONCOLLFILE="$MONCOLL"
		fi

		# make collection name match what's in csl file
		MONCOLL=`normalize_name "${MONCOLL}"`

		#
		# If Sentry monitors are installed, then install the col files
		#
		if $SENTRY_MONITORS_OK ; then
			#-#-#
			log_start_action `outmsg LinkInstallCat 184 "Creating Monitoring Collection %1\\$s" "$MONCOLL"`
			if [ -f $PRODDIR/$MONCOLLFILE.col ] ; then
				if mcsl -Ri $PRODDIR/$MONCOLLFILE.col; then
					#-#-#
					if mcsl -Rz "$R_MONPM" > /dev/null 2>&1; then
					  : # update collection on DM3.6 endpts
					fi
					log_success `outmsg LinkInstallCat 70 "%1\\$s installed" "$MONCOLL"`
				else
					#-#-#
					log_failure `outmsg LinkInstallCat 185 "Error %1\\$s installing %2\\$s" "$?" "$MONCOLL"`
					return 1
				fi
			else
				log_failure `outmsg LinkInstallCat 186 "%1\\$s.col does not exist!" "$MONCOLLFILE"`
				# try using the directory name in case it's a 
				# predefined collection
				MONCOLL=`echo $MONPROG | awk -F/ '{print $1}'`
			fi
			#-#-#
			log_end_action
		else
			log_failure `outmsg LinkInstallCat 187 "SENTRY_MONITORS_OK is FALSE!!!!!"`
		fi

		WADDMONARGS=""

		#
		#	create a string with the arguments
		#
		for arg in $MONARGS; do
			eval MONARG='"$'$arg'"'

			# fix for defect 100389; add blank args too, else DM will complain
			### if [ -n "$MONARG" ] ; then

				WADDMONARGS=' '$WADDMONARGS' -a "'$MONARG'"'

			### fi
		done

		#
		#	append the monitoring interval
		#
		if [ -n "$MONSAMP" ] ; then
			WADDMONARGS=' '$WADDMONARGS' -t "'$MONSAMP'"'
		fi

		#
		#	append each severity level and condition clause
		#
		for cond in $MONCONDS; do
			eval CONDLEVEL='"$'$cond'_LEVEL"'
			eval CONDOPER='"$'$cond'_OPER"'
			eval CONDVALUE='"$'$cond'_VALUE"'
			eval CONDCLAUSE='"$'$cond'_TEC_CLAUSE"'
			eval CONDTASK='"$'$cond'_TASK"'
			eval CONDTASKLIB='"$'$cond'_LIBR"'
			eval CONDARGS='"$'$cond'_TASKARGS"'
		        r_libname="${CONDTASKLIB}#${IRONAME}"

			WADDMONTASK=""
			if [ -n "$CONDLEVEL" ] ; then
				WADDMONARGS=' '$WADDMONARGS' -c "'$CONDLEVEL'"'
			fi
			if [ x"$CONDOPER" != x ] ; then
				WADDMONARGS=' '$WADDMONARGS' -R "'$CONDOPER'"'
			fi
			if [ -n "$CONDVALUE" ] ; then
				WADDMONARGS=' '$WADDMONARGS' "'$CONDVALUE'" -i'
			fi
			if [ -n "$CONDCLAUSE" ] ; then
				WADDMONARGS=' '$WADDMONARGS' '$CONDCLAUSE''
			fi
			if [ -n "$CONDTASK" ] ; then
	
				if [ "$USE_TLL" = true ]; then
					# strip blanks, since builder would have
					CONDTASK=`normalize_name "$CONDTASK"`
				fi

				CONDTASKLIB="$r_libname"
				WADDMONTASK=' '$WADDMONTASK' -x "'$CONDTASKLIB'" "'$CONDTASK'"'
				if [ -n "$CONDARGS" ] ; then
					for arg in $CONDARGS; do
						eval CONDARG='"$'$arg'"'
						WADDMONTASK=' '$WADDMONTASK' '"'$CONDARG'"''
					done
				fi
				WADDMONTASK=' '$WADDMONTASK' '"\;"''
				WADDMONARGS=' '$WADDMONARGS' '$WADDMONTASK''
			fi
		done

		#
		#	now try to add the monitor
		#
		#-#-#

                # save the original monitor collection name
		MONCOLL_2=`echo $MONPROG | awk -F/ '{print $1}'`

		# strip off any leading path info
		MONPROG=`echo $MONPROG | awk -F/ '{print $NF}'`

		# get rid of blanks, since builder would have
		MONITOR=`normalize_name "$MONITOR"`

		log_start_action `outmsg LinkInstallCat 188 "Creating monitor %1\\$s" "$MONPROG"`
		if IsValidMonitorCollection $MONCOLL; then
			# Try both MONITOR and MONPROG, predefined's could be 
			#  either!
			if eval waddmon '"$MONCOLL"' '"$MONITOR"' $WADDMONARGS '"$R_MONPM"'; then
				#-#-#
				log_success `outmsg LinkInstallCat 189 "%1\\$s %2\\$s monitor created" "$MONCOLL" "$MONITOR"`
			elif eval waddmon '"$MONCOLL"' '"$MONPROG"' $WADDMONARGS '"$R_MONPM"'; then
				#-#-#
				log_success `outmsg LinkInstallCat 189 "%1\\$s %2\\$s monitor created" "$MONCOLL" "$MONPROG"`
			elif eval waddmon '"$MONCOLL_2"' '"$MONITOR"' $WADDMONARGS '"$R_MONPM"'; then
				#-#-#
				log_success `outmsg LinkInstallCat 189 "%1\\$s %2\\$s monitor created" "$MONCOLL_2" "$MONITOR"`
			elif eval waddmon '"$MONCOLL_2"' '"$MONPROG"' $WADDMONARGS '"$R_MONPM"'; then
				#-#-#
				log_success `outmsg LinkInstallCat 189 "%1\\$s %2\\$s monitor created" "$MONCOLL_2" "$MONPROG"`
			else
				#-#-#
				log_failure `outmsg LinkInstallCat 185 "Error %1\\$s installing %2\\$s" "$?" "$MONPROG"`
				return 1
			fi
		else
			log_failure `outmsg LinkInstallCat 190 "%1\\$s does not exist" "$MONCOLL"`
			return 1
		fi
	
	#
	# dependency check failed
	#
	else
		log_info `outmsg LinkInstallCat 191 "PM Dependencies failed"`
	fi
done
}

######################################################################
#
# create_mar_profiles
#
create_mar_profiles ()
{
[ "$DEBUG" = true ] && set -x

#
# Create Mar Profile Managers
#
for mar in $ARM_PMS; do
	eval PMNAME='"$'$mar'"'
	eval R_PMNAME='"$R_'$mar'"'
	eval ICONDSL='"$'$mar'_ICONDSL"'
	eval ICONSTATE='"$'$mar'_ICONSTATE"'
	eval PM_TARGETS='"$'$mar'_TARGETS"'
	eval PM_DEPENDS='"$'$mar'_DEPS"'
	eval PM_DATALESS='"$'$mar'_DATALESS"'

	#-#-#
	log_start_action `outmsg LinkInstallCat 210 "Creating Mar Profile Manager %1\\$s" "$PMNAME"`

	if check_dependencies $PM_DEPENDS; then
		#-#-#
		log_info `outmsg LinkInstallCat 183 "PM Dependencies passed"`

	  	# Set the subscrition list
		if [ -z "$PM_TARGETS" ]; then
			TARGETS=@ManagedNode:$MNNAME
		else
			TARGETS=""
			for target in $PM_TARGETS; do
				eval TARG='"$'$target'"'
				TARGETS="$TARGETS \"@ProfileManager:$TARG\""
			done
		fi

		if IsValidProfileManager "$PMNAME"; then
			log_info `outmsg LinkInstallCat 92 "%1\\$s exists, updating subscription list" "$R_PMNAME"`
			eval wpmsub \"$PMNAME\" $TARGETS
		else
			if [ -n "$ICONDSL" -a ! -f $PRODDIR/$ICONDSL ]; then
				log_info "Could not find $ICONDSL ... using default"
			fi
			if [ -n "$ICONDSL" -a -f $PRODDIR/$ICONDSL ]; then
				LINK_CO=`wlookup -r Classes SolutionsLink`

				if [ "$PM_DATALESS" = "TRUE" ]; then
					initpres $LINK_CO -class -put \
						state "$ICONSTATE" $PRODDIR/$ICONDSL \
						xpm $LINKDIR/MLink_dataless_icon.xpm
				else
					initpres $LINK_CO -class -put \
						state "$ICONSTATE" $PRODDIR/$ICONDSL \
						xpm $LINKDIR/MLink_icon.xpm
				fi
			else
				if [ "$PM_DATALESS" = "TRUE" ]; then
					ICONSTATE=MLINKLCF
				else
					ICONSTATE=MLINK
				fi
			fi

		fi

		if [ -z "$ICONSTATE" ]; then
			if [ "$PM_DATALESS" = "TRUE" ]; then
				ICONSTATE=MLINKLCF
			else
				ICONSTATE=MLINK
			fi
		fi

		PRFMGR_CO=`wlookup -r Classes ProfileManager`
		# make sure solaris doesn't stop here.
		set +e
		PRFMGR_OID=`idlcall $PRFMGR_CO lookup_object '"'$PMNAME'" {0}'`
		if [ $? -eq 0 ]; then
			wregister -r ProfileManager "$PMNAME" "$PRFMGR_OID"
		fi
		set -e

		eval wcrtsprf "$ICONSTATE" \"$MEDIUM_PRODNAME\" MarProfile \
			\"$PMNAME\" \
			\"$PMNAME\" \
			$TARGETS
		

	else
		#
		# dependency check failed
		#
		log_info `outmsg LinkInstallCat 191 "PM Dependencies failed"`
	fi
done

#
# Create Mar Profiles
#

#
# gather date info for scheduling option
#
THIS_YEAR=`date +"%Y"`
NEXT_YEAR=`expr "$THIS_YEAR" + 1`
THIS_MONTH=`date +"%m"`
TODAY=`date +"%d"`
START_DATE="$THIS_YEAR"-"$THIS_MONTH"-"$TODAY"
STOP_DATE="$NEXT_YEAR"-"$THIS_MONTH"-"$TODAY"

#
# default scheduling rule is all day, everyday, with a collection interval of 10 minutes
# start date is today.  end date is one year from today
# 
DEFAULT_RULE="{rule_MTWTFSS:sunday,monday,tuesday,wednesday,thursday,friday,saturday,sunday:00:00:00:00}"
eval SCHEDULE='600,"$START_DATE","$STOP_DATE","$DEFAULT_RULE"'

BUCKETS=""
ADDED_VUSs=""
registered="false"
suffix=0

for mar in $ARMS; do
	eval APPNAME='"$'$mar'"'
	eval MARPM='"$'$mar'_PM"'
	eval VUSs='"$'$mar'_VUSS"'
	eval TRANSACTION_NAME='"$'$mar'_TRANSACTION_NAME"'
	eval ARM_DEPENDS='"$'$mar'_DEPS"'
	eval ARM_BUCKETS='"$'$mar'_BUCKETS"'

	eval prfmgrname='"$'${MARPM}'"'
	r_prfmgrname="${prfmgrname}#${IRONAME}"

	if check_dependencies $ARM_DEPENDS; then
	    #-#-#
	    log_info `outmsg LinkInstallCat 183 "PM Dependencies passed"`

	    #
	    # d86042 wmarsetcoll don't work so well
	    #
	    #if wmarlscoll -p "$r_prfmgrname" -a "$APPNAME" > /dev/null 2>&1; then
	    #	cmd=wmarsetcoll
	    #else
	    	cmd=wmaraddcoll
	    #fi

	    #
	    # gather response time boundary information, using the first instance encountered.
	    # ordinarily, all response time boundaries for an application will be the same.
	    # curently, wmar* doesn't support each transaction having a unique response time
	    #
	    if [ -z "$BUCKETS" ]; then
        	for bucket in $ARM_BUCKETS; do
                	eval time='"$'$bucket'"'
                	BUCKETS="$BUCKETS,$time"
        	done
	    fi

	    #
	    # use ARM default values if none are provided
	    #
	    if [ -z "$BUCKETS" ]; then
        	    BUCKETS=",100,500,1000,2000,5000,10000"
	    fi

	    #
	    #  wmaraddcoll's transaction filter option currently accepts only "T", so
	    #  transaction names can't be added.  Use a transaction filter that collects
	    #  all transactions for the application.
	    #
	    COLLECTION=",{T,*}$BUCKETS"

	    if [ -z "$VUSs" ]; then
		#
		# register the app without VUSes
		#
		if [ "$registered" = "false" ]; then
			if sh wmarreg.sh -a "$APPNAME" > /dev/null; then
				:
			fi
			registered=true	
		fi

		#-#-#
		log_info `outmsg LinkInstallCat 212 "Creating Mar Profile %1\\$s" "$APPNAME"`	
		# d86042
		set +e
		eval $cmd -e -p '"$r_prfmgrname"' -c '"${APPNAME}${COLLECTION}"' -s '"$SCHEDULE"' 
		set -e
	    else
	    	for vuses in $VUSs; do
			eval vus='"$'$vuses'"'
			eval ARGS='"$'$vuses'_ARGS"'
			eval APPNAME='"$'$mar'"'

			if  [ ! -f "$PRODDIR"/"$vus" ]; then
				#-#-#
				log_info `outmsg LinkInstallCat 212 "Creating Mar Profile %1\\$s" "$APPNAME"`
	
				#
				# if the VUS doesn't exist, register the app without it
				#
				if [ "$registered" = "false" ]; then
					if sh wmarreg.sh -a "$APPNAME" > /dev/null; then
						:
					fi
					registered=true	
				fi
				set +e
				eval $cmd -e -p '"$r_prfmgrname"' -c '"${APPNAME}${COLLECTION}"' -s '"$SCHEDULE"' 
				set -e
			else
				#
				#  register the app with an unique entry for each VUS 
				#
				add=true
				for added in $ADDED_VUSs; do
					if [ "$added" = "$vus" ]; then
						add=false
					fi
				done
				if [ "$add" = true ]; then 
					#-#-#
					log_start_action `outmsg LinkInstallCat 213 "Creating Mar profile for %1\\$s with VUS %2\\$s" "$APPNAME" "$vus"`
					ADDED_VUSs="$ADDED_VUSs $vus"

					#
					# gather VUS args
					#
					VUS_ARGS="00:10"	
					for arg in $ARGS; do
						eval NAME='"$'$arg'_NAME"'
						eval VALUE='"$'$arg'_VALUE"'
						PARM="$NAME"="$VALUE"
						VUS_ARGS="$VUS_ARGS","$PARM"
					done

					suffix=`expr $suffix + 1`
					ALIAS="$APPNAME"_"$suffix"

					#if wmarlscoll -p "$r_prfmgrname" -a "$ALIAS" > /dev/null 2>&1; then
					#	cmd=wmarsetcoll
					#else
						cmd=wmaraddcoll
						if sh wmarreg.sh -a "$ALIAS" -b "$PRODDIR" -v "$vus" > /dev/null; then
							:
						fi
					#fi

					set +e
					eval $cmd -e -p '"$r_prfmgrname"' -c '"${ALIAS}${COLLECTION}"' -s '"$SCHEDULE"' -m '"$VUS_ARGS"'
					set -e
	
					#-#-#
					log_end_action
				fi				

			fi
	    	done # VUS loop
	    fi

	else
	    #
	    # dependency check failed
	    #
	    log_info `outmsg LinkInstallCat 191 "PM Dependencies failed"`
	fi
done # transaction loop
}

######################################################################
#
# create_inventory_profiles
#
create_inventory_profiles ()
{
[ "$DEBUG" = true ] && set -x

#
# Create file signatures
#
for inv in $INVFILES; do
	eval FILENAME='"$'$inv'"'
	eval FILESIZE='"$'$inv'_SIZE"'
	eval FILEVERSION='"$'$inv'_VERSION"'

	wfilesig -a -n "$FILENAME" -s "$FILESIZE" -d "$FILENAME" -v "$FILEVERSION" 
done

#
# Create an inventory profile
#
if [ -n "$INV_PM" ]; then
	PMNAME="$INV_PM"
	R_PMNAME="${PMNAME}#${IRONAME}"
	ICONDSL="$INV_PM_ICONDSL"
	ICONSTATE="$INV_PM_ICONSTATE"

	if [ -z "$INV_PM_TARGETS" ]; then
		TARGETS=@ManagedNode:$MNNAME
	else
		TARGETS=""
		for target in $INV_PM_TARGETS; do
			eval TARG='"$'$target'"'
			TARGETS="$TARGETS \"@ProfileManager:$TARG\""
		done
	fi

	if IsValidProfile InventoryProfile "$PMNAME"; then
		log_info `outmsg LinkInstallCat 92 "%1\\$s exists, updating subscription list" "$R_PMNAME"`
		eval wpmsub \"$PMNAME\" $TARGETS
	else
		if [ -n "$ICONDSL" -a ! -f $PRODDIR/$ICONDSL ]; then
			log_info `outmsg LinkInstallCat 68 "Could not find %1\\$s ... using default" "$ICONDSL"`
		fi
		if [ -n "$ICONDSL" -a -f $PRODDIR/$ICONDSL ]; then
			LINK_CO=`wlookup -r Classes SolutionsLink`

			initpres $LINK_CO -class -put \
				state "$ICONSTATE" $PRODDIR/$ICONDSL \
				xpm $LINKDIR/ILink_icon.xpm
		else
			ICONSTATE=ILINK
		fi

		eval wcrtsprf $ICONSTATE \"$MEDIUM_PRODNAME\" InventoryProfile \
			\"$PMNAME\" \
			\"$PMNAME\" \
			$TARGETS
	
		wsetiprf -m HW_MIF_IGNORE -s HW_SCAN_OFF @InventoryProfile:"${PMNAME}#${IRONAME}" > /dev/null 2>&1
	fi
fi
}

######################################################################
#
# set_install_options
#
set_install_options()
{

[ "$DEBUG" = true ] && set -x

#-#-#
log_start_action `outmsg LinkInstallCat 192 "Setting install options"`

CLO=`wlookup Library`
CO=`idlcall $CLO TMF_SysAdmin::Library::lookup_object \
		'"TMF_imp_TSysAdmin::PolicyDrivenBase" {0}'`

for installopt in $INSTALL_OPTS; do
	eval OPT='"$'$installopt'"'
	eval OPTVAL='"$'$OPT'"'
	eval OPTNAME='"$'$installopt'_NAME"'

	#-#-#
	log_info `outmsg LinkInstallCat 193 "Setting %1\\$s to '%2\\$s'" "$OPT" "$OPTVAL"`

	# create a hidden presentation object to hold the install options
	PRESNAME="$OPTNAME"_"$MNNAME"

	# see if it's already created
	if wlookup -r Presentation "$PRESNAME" > /dev/null 2>& 1; then
		log_info `outmsg LinkInstallCat 194 "Presentation Object '%1\\$s' already exists" "$PRESNAME"`
	else
		#-#-#
		log_info `outmsg LinkInstallCat 195 "Creating Presentation object '%1\\$s'" "$PRESNAME"`

		MNOID=`objcall 0.0.0 get_host_location`
		PRESOBJ=`idlcall $CO \
					TMF_SysAdmin::InstanceManager::create_instance \
					OBJECT_NIL $MNOID \"$PRESNAME\"`
		initpres "$PRESNAME" -pres-name -create
	fi

	# if the attribute doesn't exist, add it, otherwise set it.
	if wgetattr $OPT @Presentation:"$PRESNAME" > /dev/null 2>& 1; then

		#-#-#
		log_info `outmsg LinkInstallCat 196 "Setting attribute %1\\$s to '%2\\$s'" "$OPT" "$OPTVAL"`

		wsetattr "$OPTVAL" $OPT @Presentation:"$PRESNAME"
	else

		#-#-#
		log_info `outmsg LinkInstallCat 197 "Adding attribute %1\\$s (setting to %2\\$s)" "$OPT" "$OPTVAL"`

		waddattr -v "$OPTVAL" $OPT @Presentation:"$PRESNAME"
	fi
done

log_end_action
}

######################################################################
#
# create_plus_collection
#
create_plus_collection()
{

[ "$DEBUG" = true ] && set -x

#-#-#
log_start_action `outmsg LinkInstallCat 198 "Creating top level plus collection"`

#
# make sure the top level TivoliPlus collections exists on $root_user desktop
#
wcrttopcol

#
# Create the /Plus module within the "TivoliPlus" Collection
#
if [ -z "$PLUSBITMAP" ]; then
  PLUSBITMAP="$PRODDIR/Default.xpm"
fi

if [ -f "$PLUSBITMAP" ]; then
  if [ -f "$PRODDIR/$ICONDSL" ]; then
    wcrtsol "$MEDIUM_PRODNAME" TP$SHORT_PRODNAME "$PLUSBITMAP" $PRODDIR/$ICONDSL
  else
    wcrtsol "$MEDIUM_PRODNAME" TP$SHORT_PRODNAME "$PLUSBITMAP"
  fi
else
  #-#-#
  log_failure `outmsg LinkInstallCat 199 "Error creating module collection: PLUSBITMAP file %1\\$s does not exist" "$PLUSBITMAP"`
  return 1
fi


#
# Retrieve the OID of the collection just created
#
COLLECTIONGUI=`idlcall "$LIBRARY" lookup_object '"imp_TMF_TGC::CollectionGUI" {0}'`
COLLOBJ=`idlcall "$COLLECTIONGUI" lookup_object '"'$MEDIUM_PRODNAME'" {0}'`
#-#-#
log_end_action
}

######################################################################
#
#	cpp_path
#
cpp_path()
{
  [ "$DEBUG" = true ] && set -x
  if [ ! "$CPP_PATH" ]; then
    if [ x\"$OS\" = x\"Windows_NT\" ] ; then
      CPP_PATH=$BINDIR/tools/cpp.exe
    elif type gcc > /dev/null 2>& 1; then
      GCCLIB=`gcc -print-libgcc-file-name`
      GCCLIBDIR=`dirname ${GCCLIB}`
      CPP_PATH=${GCCLIBDIR}/cpp
    elif type cpp > /dev/null 2>& 1; then
      # extrapolate from our PATH
      CPP_PATH=`type cpp | awk '{print $NF}'`
    else
      CPP_PATH=""
    fi
  fi
  if [ -x "$CPP_PATH" ]; then
    echo "$CPP_PATH"
  else
    echo ""
  fi
}

######################################################################
#
#       is_TMR_server
#
is_TMR_server() {
  ODNUM=`odadmin | grep Dispatcher | awk '{print $3}'`
  if [ $ODNUM -ne 1 ]; then
    return 1
  fi
  return 0
}


######################################################################
#
#       do_configure_TMA
#
do_configure_TMA() {
  [ "$DEBUG" = true ] && set -x
  if is_TMR_server; then
    . $PRODDIR/PLUSconfig-TMA-utils.sh
    TMA_config_full
    TMA_sync_gateways
  fi
}


######################################################################
#
#       xlate_prodinfo_vars
#
xlate_prodinfo_vars() {
  VERSION_MAJOR_XLATED=`xlate_var VERSION_MAJOR`; export VERSION_MAJOR_XLATED
  VERSION_MINOR_XLATED=`xlate_var VERSION_MINOR`; export VERSION_MINOR_XLATED
  VERSION_REV_XLATED=`xlate_var VERSION_REV`; export VERSION_REV_XLATED
  REAL_PRODNAME_XLATED=`xlate_var REAL_PRODNAME`; export REAL_PRODNAME_XLATED
  SHORT_VERSION_XLATED=`outmsg TMB_plus 64 "%1\\$s.%2\\$s" "$VERSION_MAJOR_XLATED" "$VERSION_MINOR_XLATED"`; export SHORT_VERSION_XLATED
  LONG_VERSION_XLATED=`outmsg TMB_plus 65 "Version %1\\$s, Revision %2\\$s" "$SHORT_VERSION_XLATED" "$VERSION_REV_XLATED"`; export LONG_VERSION_XLATED
  MEDIUM_PRODNAME_XLATED=`outmsg TMB_plus 66 "%1\\$s Plus for Tivoli" "$REAL_PRODNAME_XLATED"`; export MEDIUM_PRODNAME_XLATED
  LONG_PRODNAME_XLATED=`outmsg TMB_plus 67 "%1\\$s, %2\\$s" "$MEDIUM_PRODNAME_XLATED" "$LONG_VERSION_XLATED"`; export LONG_PRODNAME_XLATED
}

######################################################################
#
# common_init
#
common_init()
{

[ "$DEBUG" = true ] && set -x
##set -x

log_init

xlate_prodinfo_vars

DFLT_TASKLIB=`outmsg LinkInstallCat 200 "%1\\$s Utility Task Library" "$MEDIUM_PRODNAME_XLATED"`
DEFAULT_TASKLIB="$MEDIUM_PRODNAME Utility Task Library"
NULL="__NULL__"; export NULL

PATH=$PRODDIR:/bin:/usr/bin:$BINDIR/AEF:$BINDIR/ADE:$BINDIR/tools:$PATH:/usr/ucb:/usr/lib:/usr/ccs/lib
export PATH

#
# Set parallel "R_*" variables with region name appended to avoid collisions
#
for var in $SENTRY_PMS $INDCOLLS $SUBLIST_PMS $TASKLIBS $SENTRY_PM_INDCOLLS $FILEPACKS $ARM_PMS; do
	eval R_$var='"$'$var'#$IRONAME"'
done

AMS_COND_LEVELS="
  AMS_normal
  AMS_informational
  AMS_warning
  AMS_severe
  AMS_critical
  AMS_fatal
"
TIV_COND_LEVELS="
  normal_zero
  normal
  informational
  warning
  severe
  critical
  fatal
"

#
# cd to the appropriate directory
#
cd $PRODDIR || return 1

#
# Install SolutionsLink Class
#
install_link_class || return 1

#
# Check to see which applications are installed
#
check_apps_installed

}
