#!/bin/sh
# $Id: $
#
# Component:	TivoliPlus (installation utilities)
#
# $Source: $
# $Revision: $
#
# Description:  Common installation support functions for TivoliPlus modules.	
#		This script should be "sourced" at the top of the TivoliPlus
#		module initialization scripts (for ALI/server and clients).
#
# External Variable References / Initialization:
#
# Author:	Annelle Harrison
#		Tivoli Systems, Inc.
# Modifications:
#
# (C) COPYRIGHT Tivoli Systems, Inc. 1998-2000
# Unpublished Work
# All Rights Reserved
# Licensed Material - Property of Tivoli Systems, Inc.
#

############  Message Utility Functions
#

outmsg() {

  ## turn off tracing for this routine
  ###if [ "$DEBUG" = true ]; then set +x; fi
  set +xv

  arg1=`echo "$1"`
  arg2=`echo "$2"`
  arg3=`echo "$3"`

  if [ ! "$arg3" ]; then
    eval echo "$arg1" "$arg2"
    ## turn tracing back on if applicable ...
    if [ "$DEBUG" = true ]; then set -x; fi
    return
  fi

  # got 3 or more args, we must be trying to wbindmsg ...
  file="$arg1"
  num="$arg2"
  string="`echo \"$arg3\" | sed -e 's/*/./g'`"
  shift;shift;shift
  afterstring="$*"

  ## handle up to 4 substitution args
  subarg1=`echo "$1"`; if [ "$subarg1" ]; then shift; fi
  subarg2=`echo "$1"`; if [ "$subarg2" ]; then shift; fi
  subarg3=`echo "$1"`; if [ "$subarg3" ]; then shift; fi
  subarg4=`echo "$1"`; if [ "$subarg4" ]; then shift; fi
  rest="$*"   ## not real graceful, but at least we're not throwing it away ...

  # use USER_LANG if it's set; so our wbindmsg will find the user's locale
  save_lang=$LANG
  if [ "$USER_LANG" ]; then
    LANG=$USER_LANG
  elif [ ! "$LANG" ]; then
    LANG=C
  fi

  if wbindmsg $file $num "$string" "$subarg1" "$subarg2" "$subarg3" "$subarg4" "$rest" 1> /dev/null 2>&1; then
    wbindmsg $file $num "$string" "$subarg1" "$subarg2" "$subarg3" "$subarg4" "$rest"
    echo   ## add a line ending to wbindmsg result
  else
    # couldn't use wbindmsg, but try and spit out something useful
    substitute_strings "$string" "$subarg1" "$subarg2" "$subarg3" "$subarg4" "$rest"
    if [ $? -ne 0 ]; then
      eval echo $string $afterstring
    fi
  fi

  # restore LANG
  LANG=$save_lang

  ## turn tracing back on if applicable ...
  if [ "$DEBUG" = true ]; then set -x; fi

}

errmsg() {

  ## turn off tracing for this routine ...
  ###if [ "$DEBUG" = true ]; then set +x; fi
  set +xv

  # preserve arg format for call to outmsg()
  arg1=`echo "$1"`
  arg2=`echo "$2"`
  arg3=`echo "$3"`
  if [ "$arg3" ]; then 
    shift;shift;shift
    subarg1=`echo "$1"`; if [ "$subarg1" ]; then shift; fi
    subarg2=`echo "$1"`; if [ "$subarg2" ]; then shift; fi
    subarg3=`echo "$1"`; if [ "$subarg3" ]; then shift; fi
    subarg4=`echo "$1"`; if [ "$subarg4" ]; then shift; fi
    rest="$*"
  fi

  outmsg "$arg1" "$arg2" "$arg3" "$subarg1" "$subarg2" "$subarg3" "$subarg4" $rest >& 2

  ## turn tracing back on if applicable ...
  if [ "$DEBUG" = true ]; then set -x; fi

}

substitute_strings() {

  ## turn off tracing for this routine
  ###if [ "$DEBUG" = true ]; then set +x; fi
  set +xv

  base="`echo \"$1\" | sed -e 's/*/./g'`"; if [ "$base" ]; then shift; fi
  sub1=`echo "$1"`; if [ "$sub1" ]; then shift; fi
  sub2=`echo "$1"`; if [ "$sub2" ]; then shift; fi
  sub3=`echo "$1"`; if [ "$sub3" ]; then shift; fi
  sub4=`echo "$1"`; if [ "$sub4" ]; then shift; fi
  xtra=`echo "$*"`


  base="`echo \"$base\" | sed -e 's/%1$s/$sub1/g'`"
  base="`echo \"$base\" | sed -e 's/%2$s/$sub2/g'`"
  base="`echo \"$base\" | sed -e 's/%3$s/$sub3/g'`"
  base="`echo \"$base\" | sed -e 's/%4$s/$sub4/g'`"
  eval echo "$base" "$xtra"

  ## turn tracing back on if applicable ...
  if [ "$DEBUG" = true ]; then set -x; fi

}

#
#       xlate_var -- please pass in a var name, not its value!
#                    we construct variables appending var name 
#                    with _MSGFILE and _MSGID, and use those values
#
xlate_var() {
  varname="$1"
  eval default_val='"$'${varname}'"'
  if [ ! "$default_val" ]; then
    default_val="$1"
  fi
  eval msgid='"$'${varname}'_MSGID"'
  eval msgfile='"$'${varname}'_MSGFILE"'
  outmsg $msgfile $msgid "$default_val"
}

#
######## End of Message Utility Functions


######################################################################
#
#       xlate_prodinfo_vars
#
xlate_prodinfo_vars() {
  VERSION_MAJOR_XLATED=`xlate_var VERSION_MAJOR`; export VERSION_MAJOR_XLATED
  VERSION_MINOR_XLATED=`xlate_var VERSION_MINOR`; export VERSION_MINOR_XLATED
  VERSION_REV_XLATED=`xlate_var VERSION_REV`; export VERSION_REV_XLATED
  REAL_PRODNAME_XLATED=`xlate_var REAL_PRODNAME`; export REAL_PRODNAME_XLATED
  SHORT_VERSION_XLATED=`outmsg TMB_plus 64 "%1\\$s.%2\\$s" "$VERSION_MAJOR_XLATED" "$VERSION_MINOR_XLATED"`; export SHORT_VERSION_XLATED
  LONG_VERSION_XLATED=`outmsg TMB_plus 65 "Version %1\\$s, Revision %2\\$s" "$SHORT_VERSION_XLATED" "$VERSION_REV_XLATED"`; export LONG_VERSION_XLATED
  MEDIUM_PRODNAME_XLATED=`outmsg TMB_plus 66 "%1\\$s Plus for Tivoli" "$REAL_PRODNAME_XLATED"`; export MEDIUM_PRODNAME_XLATED
  LONG_PRODNAME_XLATED=`outmsg TMB_plus 67 "%1\\$s, %2\\$s" "$MEDIUM_PRODNAME_XLATED" "$LONG_VERSION_XLATED"`; export LONG_PRODNAME_XLATED
}

usage() {
  THIS_CMD=`echo $0 | tr "\\\\\\\\" "/"`; THIS_CMD=`basename $THIS_CMD`
  errmsg LinkInstallCat 42 "  Usage: %1\$s [ -d] <PLUSproduct-info.sh path>" "$THIS_CMD"
  errmsg LinkInstallCat 43 "           -d  = turn on debug tracing"
  exit 1
}

#
# make sure this script runs in the C locale, but outmsg funcs will look at USER_LANG
#
SAVE_LANG=$LANG
if [ "$LANG" != C ]; then
  USER_LANG=$LANG; export USER_LANG
  LANG=C; export LANG
fi

if [ "$1" = "-d" ]; then
  DEBUG_OPT="-d"
  shift
else
  DEBUG_OPT=""
fi

if [ -z "$1" ]; then
  usage
elif [ ! -f "$1" ]; then
  errmsg LinkInstallCat 44 "Product info file (%1\$s) not found" "$1"
  usage
fi


#
#	Set up environment (similar to old /etc/Tivoli/setup_env.sh)
#
OSERV=`objcall 0.0.0 get_oserv`; export OSERV
INST_DIR=`objcall $OSERV query install_dir | tr "\\\\\\\\" "/"`; export INST_DIR
INTERP=`objcall $OSERV query interp`; export INTERP
BINDIR=$INST_DIR/$INTERP; export BINDIR
PLUSDIR=$INST_DIR/generic_unix/TME/PLUS
LINKDIR=$PLUSDIR/LINK; export LINKDIR
DBDIR=`objcall $OSERV query db_dir | tr "\\\\\\\\" "/"`; export DBDIR
TEMP=/tmp; export TEMP
if [ x"$OS" = x"Windows_NT" ]; then
	TEMP=$DBDIR/tmp
fi
if [ ! -d $TEMP ]; then
	$LINKDIR/mkdir.sh $TEMP
fi
PATH=$LINKDIR:$PATH; export PATH


SCRIPTNAME=`echo $0 | tr "\\\\\\\\" "/"`
SCRIPTNAME=`basename $SCRIPTNAME`; export SCRIPTNAME

. $1

# now xlate some prodinfo variables
xlate_prodinfo_vars

if [ "$LONG_PRODNAME" ]; then
  outmsg LinkInstallCat 45 "Removing Tivoli database information for \"%1\$s\"" "$LONG_PRODNAME_XLATED"
  sleep 5
  PLUSinstruct.sh -ua $DEBUG_OPT $1
else
  errmsg LinkInstallCat 46 "%1\$s is not a valid product info file ..." "$1"
  usage
fi

# restore user's locale
LANG=$SAVE_LANG
