#!/bin/sh
# $Id: $
#
# Component:	TivoliPlus (installation utilities)
#
# $Source: $
# $Revision: $
#
# Description:  Common installation support functions for T/EC.
#
# External Variable References / Initialization:
#
# Author:	Glenn Zazulia
#		Tivoli Systems, Inc.
# Modifications:
#	fpc	10/1/96	made generic
#
# (C) COPYRIGHT Tivoli Systems, Inc. 1996-2000
# Unpublished Work
# All Rights Reserved
# Licensed Material - Property of Tivoli Systems, Inc.
#

######################################################################
#
#	T/EC Functions
#
######################################################################


############  Message Utility Functions
#

outmsg() {

  ## turn off tracing for this routine
  ###if [ "$DEBUG" = true ]; then set +x; fi
  set +xv

  arg1=`echo "$1"`
  arg2=`echo "$2"`
  arg3=`echo "$3"`

  if [ ! "$arg3" ]; then
    eval echo "$arg1" "$arg2"
    ## turn tracing back on if applicable ...
    if [ "$DEBUG" = true ]; then set -x; fi
    return
  fi

  # got 3 or more args, we must be trying to wbindmsg ...
  file="$arg1"
  num="$arg2"
  string="`echo \"$arg3\" | sed -e 's/*/./g'`"
  shift;shift;shift
  afterstring="$*"

  ## handle up to 4 substitution args
  subarg1=`echo "$1"`; if [ "$subarg1" ]; then shift; fi
  subarg2=`echo "$1"`; if [ "$subarg2" ]; then shift; fi
  subarg3=`echo "$1"`; if [ "$subarg3" ]; then shift; fi
  subarg4=`echo "$1"`; if [ "$subarg4" ]; then shift; fi
  rest="$*"   ## not real graceful, but at least we're not throwing it away ...

  # use USER_LANG if it's set; so our wbindmsg will find the user's locale
  save_lang=$LANG
  if [ "$USER_LANG" ]; then
    LANG=$USER_LANG
  elif [ ! "$LANG" ]; then
    LANG=C
  fi

  if wbindmsg $file $num "$string" "$subarg1" "$subarg2" "$subarg3" "$subarg4" "$rest" 1> /dev/null 2>&1; then
    wbindmsg $file $num "$string" "$subarg1" "$subarg2" "$subarg3" "$subarg4" "$rest"
    echo   ## add a line ending to wbindmsg result
  else
    # couldn't use wbindmsg, but try and spit out something useful
    substitute_strings "$string" "$subarg1" "$subarg2" "$subarg3" "$subarg4" "$rest"
    if [ $? -ne 0 ]; then
      eval echo $string $afterstring
    fi
  fi

  # restore LANG
  LANG=$save_lang

  ## turn tracing back on if applicable ...
  if [ "$DEBUG" = true ]; then set -x; fi

}

errmsg() {

  ## turn off tracing for this routine ...
  ###if [ "$DEBUG" = true ]; then set +x; fi
  set +xv

  # preserve arg format for call to outmsg()
  arg1=`echo "$1"`
  arg2=`echo "$2"`
  arg3=`echo "$3"`
  if [ "$arg3" ]; then 
    shift;shift;shift
    subarg1=`echo "$1"`; if [ "$subarg1" ]; then shift; fi
    subarg2=`echo "$1"`; if [ "$subarg2" ]; then shift; fi
    subarg3=`echo "$1"`; if [ "$subarg3" ]; then shift; fi
    subarg4=`echo "$1"`; if [ "$subarg4" ]; then shift; fi
    rest="$*"
  fi

  outmsg "$arg1" "$arg2" "$arg3" "$subarg1" "$subarg2" "$subarg3" "$subarg4" $rest >& 2

  ## turn tracing back on if applicable ...
  if [ "$DEBUG" = true ]; then set -x; fi

}

substitute_strings() {

  ## turn off tracing for this routine
  ###if [ "$DEBUG" = true ]; then set +x; fi
  set +xv

  base="`echo \"$1\" | sed -e 's/*/./g'`"; if [ "$base" ]; then shift; fi
  sub1=`echo "$1"`; if [ "$sub1" ]; then shift; fi
  sub2=`echo "$1"`; if [ "$sub2" ]; then shift; fi
  sub3=`echo "$1"`; if [ "$sub3" ]; then shift; fi
  sub4=`echo "$1"`; if [ "$sub4" ]; then shift; fi
  xtra=`echo "$*"`


  base="`echo \"$base\" | sed -e 's/%1$s/$sub1/g'`"
  base="`echo \"$base\" | sed -e 's/%2$s/$sub2/g'`"
  base="`echo \"$base\" | sed -e 's/%3$s/$sub3/g'`"
  base="`echo \"$base\" | sed -e 's/%4$s/$sub4/g'`"
  eval echo "$base" "$xtra"

  ## turn tracing back on if applicable ...
  if [ "$DEBUG" = true ]; then set -x; fi

}

#
#       xlate_var -- please pass in a var name, not its value!
#                    we construct variables appending var name 
#                    with _MSGFILE and _MSGID, and use those values
#
xlate_var() {
  varname="$1"
  eval default_val='"$'${varname}'"'
  if [ ! "$default_val" ]; then
    default_val="$1"
  fi
  eval msgid='"$'${varname}'_MSGID"'
  eval msgfile='"$'${varname}'_MSGFILE"'
  outmsg $msgfile $msgid "$default_val"
}

#
######## End of Message Utility Functions


get_tme_server_SentryMonitors_path()
{
  # first get the TMR server's host name
  ALI_OID=`wlookup -r distinguished ServerManagedNode`
  TME_HOSTNAME=`idlattr -t -g $ALI_OID label string`

  # now find it's install path
  OSERV=`objcall 0.0.0 get_oserv`
  REGION=`objcall $OSERV query whoami | awk '{print $1}'`
  ALI_OSERV=$REGION.1.2
  INST_DIR=`objcall $ALI_OSERV query install_dir | tr "\\\\\\\\" "/"`

  # return the concatenated result
  eval echo $TME_HOSTNAME:$INST_DIR/generic/SentryMonitors
}

add_to_backout_list()
{
	BackoutList="$* $BackoutList"		# Add to (reverse-)ordered list
}

BackoutList=					# Initialization

backout()
{
	# Warning:  because of some sh's poor handling of scoped args, we
	#	    must restore args after any function call.
	args=$BackoutList
	set -- $args
	while [ $# -gt 0 ]; do
		State=$1  skip=1
		case $State in
			RuleBase)
				errmsg LinkTecCat 1 "Deleting %1\$s rulebase." "$Rulebase"
				wdelrb $Rulebase
				;;
			ClassFile)
				cfile=`echo $2 | awk -F/ '{print $NF}'`
				errmsg LinkTecCat 2 "Deleting new %1\$s class file." "$cfile"
				wdelrbclass $cfile $Rulebase
				skip=2
				;;
			RulesFile)
				rfile=$2  skip=2
				errmsg LinkTecCat 3 "Deleting new %1\$s rules file." "$rfile"
				wdelrbrules $rfile $Rulebase
				;;
			EventSource)
				esrc=$2  skip=2
				errmsg LinkTecCat 4 "Deleting %1\$s event source." "$esrc"
				wdelsrc $esrc
				;;
			EventGroup)
				errmsg LinkTecCat 5 "Deleting %1\$s event group." "$EventGroup"
				wdeleg $EventGroup
				;;
			AssignConsole)
				errmsg LinkTecCat 6 "Unassigning %1\$s from %2\$s." "$EventGroup" "$EventConsole"
				wunassigneg @EnterpriseClient:"$EventConsole" $EventGroup
				;;
			Filters)
				errmsg LinkTecCat 7 "Sorry, unable to delete filters."
				;;
			DeletedPlusRule)
				rulefile=$2  skip=2
				errmsg LinkTecCat 8 "Adding back %1\$s rules file." "$rulefile"
				wimprbrules $rulefile $Rulebase
				;;
		esac
		set -- $args
		shift $skip
		args="$*"
	done
	errmsg LinkTecCat 42 "Error:  Installation Failed."
	exit 1
}

verify_rule_base()
{
	if [ -z "$Rulebase" ]
	then
		errmsg LinkTecCat 43 "Error:  A rulebase must be defined!"
		backout
	fi
	if wlsrb | grep "^${Rulebase}$"				>/dev/null 2>&1
	then
		CreateNewRulebase=false
		# make sure the user didn't also include a clone rulebase
		#if [ ! -z "$CloneRulebase" ]: then
		#	errmsg LinkTecCat 44 "Error:  %1\$s rulebase already exists.  Only define a clone rulebase when creating a new rulebase!" "$Rulebase"
		#	backout
		#fi
	else
		CreateNewRulebase=true
	fi
}

verify_clone_rule_base()
{
	if [ -z "$CloneRulebase" ]
	then
		errmsg LinkTecCat 45 "Error:  Clone rulebase must be defined!"
		backout
	fi
	if wlsrb | grep "^${CloneRulebase}$"			>/dev/null 2>&1
	then
		$DEBUG && outmsg LinkTecCat 10 "Debug:  %1\$s exists." "$CloneRulebase"
	else
		errmsg LinkTecCat 46 "Error:  %1\$s rulebase must already exist!" "$CloneRulebase"
		backout
	fi
}

#verify_clone_dir_path()
#{
#	if [ ! -d "$RulebasePath" ]
#	then
#		if mkdir.sh $RulebasePath
#		then
#			$DEBUG && outmsg LinkTecCat 11 "Debug:  Created %1\$s" "$RulebasePath"
#		else
#			errmsg LinkTecCat 47 "Error:  Cannot create new rulebase path."
#			backout
#		fi
#	fi
#}

verify_event_console()
{
	if [ -n "$EventConsole" ]
	then
		if wlookup -r EnterpriseClient "$EventConsole"	>/dev/null 2>&1
		then
			$DEBUG && outmsg LinkTecCat 12 "Debug:  Event console \"%1\$s\" exists." "$EventConsole"
		else
			errmsg LinkTecCat 48 "Error:  Event console \"%1\$s\" does not exist!" "$EventConsole"
			backout
		fi
	fi
}

verify_sentry_classes()
{
	if wlsrbclass $VerificationRB | grep "^$SentryClassFile$" >/dev/null 2>&1
	then
		$DEBUG && outmsg LinkTecCat 13 "Debug:  Sentry classes are ok."
	else
		$DEBUG && outmsg LinkTecCat 14 "Debug:  Sentry classes have not been installed, adding them."
		for f1 in $SentryClassFiles; do
			new=true
			for f2 in $ClassFiles; do
			  	if [ "$f1" = "$f2" ]; then
					new=false
					break
				fi
			done
			if $new; then
				ClassFiles="$f1 $ClassFiles"
			fi
		done
	fi
}

verify_sentry_rules()
{
	if wlsrbrules $VerificationRB | grep "$SentryRulesFile"	>/dev/null 2>&1
	then
		$DEBUG && outmsg LinkTecCat 15 "Debug:  Sentry rules are ok."
	else
		errmsg LinkTecCat 49 "Error:  Sentry rules have not been installed! Please refer to Tivoli+ ReadMe for instructions."
		backout
	fi
}

err_output=$TEMP/ptc$$

compile_rule_base()
{
	RB=$1
	if [ -d "$DEBUGDIR" -o -f "$DEBUGDIR" ] ;then
		COMPILE_FLAGS="-t"
	fi

	$DEBUG && outmsg LinkTecCat 16 "Debug:  Compiling %1\$s rulebase..." "$RB"
	wcomprules $COMPILE_FLAGS $RB >$err_output 2>&1 &&  rm -f $err_output
}

verify_current_rule_base_compiles()
{
	if compile_rule_base $VerificationRB
	then
		$DEBUG && outmsg LinkTecCat 17 "Debug:  Pre-existing %1\$s rulebase compiles successfully." "$VerificationRB"
	else
		errmsg LinkTecCat 50 "Error:  Pre-existing %1\$s rulebase does not compile! Please correct before continuing installation: " "$VerificationRB"
		[ -f $err_output ] && cat $err_output >&2 && rm -f $err_output
		backout
	fi
}

verify_new_rule_base_compiles()
{
	if compile_rule_base $Rulebase
	then
		$DEBUG && outmsg LinkTecCat 18 "Debug:  New %1\$s rulebase compiles successfully." "$Rulebase"
	else
		errmsg LinkTecCat 51 "Error:  New %1\$s rulebase does not compile: " "$Rulebase"
		[ -f $err_output ] && cat $err_output >&2 && rm -f $err_output
		backout
	fi
}

create_new_rule_base()
{
	if error=`wcrtrb -d $RulebasePath $Rulebase 2>&1`
	then
		$DEBUG && outmsg LinkTecCat 19 "Debug:  Created %1\$s rulebase." "$Rulebase"
		add_to_backout_list RuleBase
	else
		if echo $error | grep "mkdir.*13" >/dev/null 2>&1; then
			error=":  no permission to create \"$RulebasePath\" directory."
		elif [ -n "$error" ]; then
			error=":  $error"
		else
			error=.
		fi
		errmsg LinkTecCat 52 "Error:  Failed to create %1\$s rulebase %2\$s" "$Rulebase" "$error"
		backout
	fi
}

clone_rule_base()
{
	if wcprb -crf $CloneRulebase $Rulebase
	then
		$DEBUG && outmsg LinkTecCat 20 "Debug:  Copied %1\$s rulebase to %2\$s." "$CloneRulebase" "$Rulebase"
	else
		errmsg LinkTecCat 53 "Error:  Failed to copy %1\$s rulebase to %2\$s." "$CloneRulebase" "$Rulebase"
		backout
	fi
}

import_new_classes()
{
	if [ ! "$ClassPathSearchOrder" ]; then
		ClassPathSearchOrder="
			$PRODDIR
			`get_tme_server_SentryMonitors_path`
			$BINDIR/../generic/SentryMonitors
			$LINKDIR
		"
	fi

	# 
	#  Sentry.baroc is a special case.  it's important the DM_Extension_Base 
	# class defined in newer versions of this file always exist, so partners
	# can create custom events w/o regard to the version of DM their end
	# customers are using.  
	#
      	if wlsrbclass $Rulebase $SentryClassFile > /dev/null 2>&1 
	then
        	$DEBUG && outmsg LinkTecCat 22 "Debug:  %1\$s already exists in %2\$s rulebase." "$Rulebase" "$SentryClassFile"
	else
       		wdelrbclass Sentry.baroc $Rulebase > /dev/null 2>&1

		for c_path in $ClassPathSearchOrder; do

			if wimprbclass -a tec.baroc ${c_path}/Sentry.baroc $Rulebase	>/dev/null 2>&1
			then
				if wlsrbclass $Rulebase $SentryClassFile > /dev/null 2>&1
				then
					$DEBUG && outmsg LinkTecCat 24 "Debug:  Imported %1\$s to %2\$s rulebase." "${c_path}/Sentry.baroc" "$Rulebase"
					add_to_backout_list ClassFile ${c_path}/Sentry.baroc
					break
				else
                       			$DEBUG && outmsg LinkTecCat 25 "Debug:  %1\$s does not exist in %2\$s rulebase. trying again." "$SentryClassFile" "$Rulebase"
                       			wdelrbclass Sentry.baroc $Rulebase > /dev/null 2>&1
				fi
			else
				$DEBUG && outmsg LinkTecCat 26 "Debug:  Failed to import %1\$s to %2\$s rulebase." "$cfile_full" "$Rulebase"
			fi
		done
	fi
	
	for cfile in $ClassFiles
	do
		strip_cfile=`echo $cfile | awk -F/ '{print $NF}'`
		
		for c_path in $ClassPathSearchOrder; do
			cfile_full=${c_path}/${strip_cfile}


			#
			#	only import a new class file if we're *sure* it's
			#	not already there
			#
			$DEBUG && outmsg LinkTecCat 21 "Debug:  looking for %1\$s in %2\$s" "$cfile_full" "$Rulebase"
			if wchkclass $cfile_full $Rulebase > /dev/null 2>&1
			then
				$DEBUG && outmsg LinkTecCat 23 "Debug:  trying to import %1\$s into %2\$s" "$cfile_full" "$Rulebase"
				if wimprbclass $cfile_full $Rulebase	>/dev/null 2>&1
				then
					$DEBUG && outmsg LinkTecCat 24 "Debug:  Imported %1\$s to %2\$s rulebase." "$cfile_full" "$Rulebase"
					add_to_backout_list ClassFile $cfile_full
					break
				else
					$DEBUG && outmsg LinkTecCat 26 "Debug:  Failed to import %1\$s to %2\$s rulebase." "$cfile_full" "$Rulebase"
				fi
			fi
       		done
	done
}

delete_plus_rules()
{
	for rfile in $PlusRulesFiles
	do
		RulesFile=`echo $rfile | awk -F/ '{print $NF}'`
		if wdelrbrules $RulesFile $Rulebase		>/dev/null 2>&1
		then
			$DEBUG && outmsg LinkTecCat 27 "Debug:  Deleted %1\$s from %2\$s rulebase." "$RulesFile" "$Rulebase"
			add_to_backout_list DeletedPlusRule $rfile
		fi
	done
}

import_new_rules()
{
	for rfile in $RulesFiles
	do
		RulesFile=`echo $rfile | awk -F/ '{print $NF}'`
		if wlsrbrules $Rulebase $RulesFile		>/dev/null 2>&1
		then
			$DEBUG && outmsg LinkTecCat 28 "Debug:  %1\$s is already configured in %2\$s rulebase." "$RulesFile" "$Rulebase"
		else
			if wimprbrules $rfile $Rulebase
			then
				$DEBUG && outmsg LinkTecCat 24 "Debug:  Imported %1\$s to %2\$s rulebase." "$rfile" "$Rulebase"
				add_to_backout_list RulesFile $RulesFile
			else
				errmsg LinkTecCat 54 "Error:  Failed to import %1\$s to %2\$s rulebase." "$rfile" "$Rulebase"
				backout
			fi
		fi
	done
}

create_new_event_source()
{
	#
	#  insure the event source doesn't already exists before trying to add it:
	#	wcrtsrc errors out if trying to add an already existing source...
	#
	EXISTS="false"
	TEC=`wlookup -r ProductInfo TEC_SERVER 2>/dev/null`
	TEC_REVISION=`idlattr -t -g $TEC revision string 2>/dev/null | sed 's/"//g'`
	if ( expr $TEC_REVISION \>= 3.6 >/dev/null && wlssrc  2>/dev/null | \
                 grep "^${TmpSource}$" >/dev/null 2>&1 ) || \
           ( expr $TEC_REVISION \< 3.6 >/dev/null && wlssrc  2>/dev/null | \
                 grep "^${TmpSource} " >/dev/null 2>&1 )
	then
	   EXISTS="true"
	   $DEBUG && outmsg LinkTecCat 29 "Debug:  %1\$s already exists - not creating." "$TmpSource"
	fi

	if [ "$EXISTS" = "false" ]; then 
	   if wcrtsrc -b $TmpBitmap $TmpSource 
		then
			$DEBUG && outmsg LinkTecCat 30 "Debug:  Created %1\$s event source." "$TmpSource"
			add_to_backout_list EventSource $TmpSource
		else
			errmsg LinkTecCat 55 "Error:  Failed to create %1\$s event source." "$TmpSource"
			backout
	   fi
	fi
}

install_new_event_source()
{
	for SourceID in $Sources
	do
		eval TmpSource=$"$SourceID"
		eval TmpBitmap=$"$SourceID"Bitmap

		#
		#  	TEC3.6 changed the output of wlssrc, so determine the TEC
		# 	version before grep-ing	
		#
		TEC=`wlookup -r ProductInfo TEC_SERVER 2>/dev/null`
		TEC_REVISION=`idlattr -t -g $TEC revision string 2>/dev/null | sed 's/"//g'`
		if ( expr $TEC_REVISION \>= 3.6 >/dev/null && wlssrc  2>/dev/null | \
                        grep "^${TmpSource}$" >/dev/null 2>&1 ) || \
                   ( expr $TEC_REVISION \< 3.6 >/dev/null && wlssrc  2>/dev/null | \
                        grep "^${TmpSource} " >/dev/null 2>&1 )
		then
			$DEBUG && outmsg LinkTecCat 31 "Debug:  %1\$s event source already exists." "$TmpSource"
			if $PreDeleteEventSources; then
				if wdelsrc $TmpSource; then
					$DEBUG && outmsg LinkTecCat 32 "Debug:  Removed %1\$s." "$TmpSource"
				else
					errmsg LinkTecCat 56 "Error:  Failed to delete %1\$s event source." "$TmpSource"
					backout
				fi
				create_new_event_source
			fi
		else
			create_new_event_source
		fi
	done
}

create_new_event_group()
{
	if wcrteg -b TivoliPlus $EventGroup
	then
		$DEBUG && outmsg LinkTecCat 33 "Debug:  Created %1\$s event group." "$EventGroup"
		add_to_backout_list EventGroup
	else
		errmsg LinkTecCat 57 "Error:  Failed to create %1\$s event group." "$EventGroup"
		backout
	fi
}

install_new_event_group()
{
	if wlseg  2>/dev/null | grep "^${EventGroup}$"		>/dev/null 2>&1
	then
		$DEBUG && outmsg LinkTecCat 34 "Debug:  %1\$s event group already exists." "$EventGroup"
		if $PreDeleteEventGroup; then
			if wdeleg $EventGroup; then
				$DEBUG && outmsg LinkTecCat 32 "Debug:  Removed %1\$s." "$EventGroup"
			else
				errmsg LinkTecCat 58 "Error:  Failed to delete %1\$s event group." "$EventGroup"
				backout
			fi
			create_new_event_group
		fi
	else
		create_new_event_group
	fi
}

assign_event_group_to_console()
{
	if [ -n "$EventConsole" ]
	then
		if wlsassigneg @EnterpriseClient:"$EventConsole" |
			grep "^${EventGroup} "			>/dev/null 2>&1
		then
			$DEBUG && outmsg LinkTecCat 35 "Debug:  %1\$s already assigned to event console %2\$s." "$EventGroup" "$EventConsole"
			add_to_backout_list AssignConsole
		else
			if wassigneg @EnterpriseClient:"$EventConsole" \
				$EventGroup $EventGroupRoles	>/dev/null 2>&1
			then
				$DEBUG && outmsg LinkTecCat 36 "Debug:  Assigned %1\$s event group to event console %2\$s." "$EventGroup" "$EventConsole"
			else
				errmsg LinkTecCat 59 "Error:  Failed to assign %1\$s event group to event console %2\$s." "$EventGroup" "$EventConsole"
				backout
			fi
		fi
	fi
}

load_new_rules()
{
	if wloadrb $Rulebase
	then
		$DEBUG && outmsg LinkTecCat 37 "Debug:  Loaded %1\$s rulebase." "$Rulebase"
	else
		errmsg LinkTecCat 60 "Error:  Failed to load %1\$s into EventServer" "$Rulebase"
		backout
	fi
}

restart_event_server()
{
	if wstopesvr
	then
		$DEBUG && outmsg LinkTecCat 38 "Debug:  Stopped Event Server."
		sleep 6
		if wstartesvr; then
			$DEBUG && outmsg LinkTecCat 39 "Debug:  Started Event Server."
		else
			errmsg LinkTecCat 61 "Error:  Failed to start Event Server."
		fi
	else
		errmsg LinkTecCat 62 "Error:  Failed to stop Event Server.  Event Server must be restarted before installation is complete!"
	fi
}

add_filter()
{
	if waddegflt $1 $EventGroup
	then
		$DEBUG && outmsg LinkTecCat 40 "Debug:  Added event filter %1\$s (%2\$s)" "$Filter" "$1"
	else
		errmsg LinkTecCat 63 "Error:  Failed adding event filter %1\$s (%2\$s)" "$Filter" "$1"
		backout
	fi
}

install_new_event_filters()
{
	CurrentFilters=`wlseg -f $EventGroup 2>/dev/null | sed 's/,//g' | \
		awk '/class:/ {
				if ($4 != "*") printf("-c %s",$4);
				if ($6 != "*") printf(" -s %s",$6);
				if ($8 != "*") printf(" -o %s",$8);
				if ($10 != "*") printf(" -r %s",$10);
				if ($12 != "*") printf(" -u %s",$12);
				printf("\n");
				}'`
	for Filter in $Filters
	do
		eval TmpFilter=$"$Filter"
		ALREADY_GOT_IT=0
		OLDIFS=$IFS
		IFS="
"
		export IFS
		for CurrentFilter in $CurrentFilters
		do
			if [ "$TmpFilter" = "$CurrentFilter" ]
			then
				$DEBUG && outmsg LinkTecCat 41 "Debug:  \"%1\$s\" filter already exists." "$TmpFilter"
				ALREADY_GOT_IT=1
				break
			fi
		done
		IFS=$OLDIFS
		export IFS
		if [ $ALREADY_GOT_IT -eq 0 ]; then
			add_filter "$TmpFilter"
		fi
	done

	for sentrypm in $SENTRY_PMS 
	do
        	eval PMNAME='"$'$sentrypm'"'
		ALREADY_GOT_IT=0
		for CurrentFilter in $CurrentFilters
		do
			if [ "$PMNAME" = "$CurrentFilter" ]
			then
				$DEBUG && outmsg LinkTecCat 41 "Debug:  \"%1\$s\" filter already exists." "$TmpFilter"
				ALREADY_GOT_IT=1
				break
			fi
		done
		if [ $ALREADY_GOT_IT -eq 0 ]; then
			if waddegflt -u "$PMNAME" $EventGroup
			then
				$DEBUG && outmsg LinkTecCat 40 "Debug:  Added event filter %1\$s (%2\$s)" "$sentrypm" "$PMNAME"
			else
				errmsg LinkTecCat 63 "Error:  Failed adding event filter %1\$s (%2\$s)" "$sentrypm" "$PMNAME"
				backout
			fi
		fi
	done
 
}

#
# install the Trouble Ticketing script
#
install_trouble_ticket_script ()
{
	if [ -z "$TEC_TT_OPEN_SCRIPT_1" ]; then
		return 0
	fi

	TEC_TOPDIR=$BINDIR/TME/TEC
	TEC_SCRIPTS=$TEC_TOPDIR/scripts
	TEC_MULTI_SCRIPTS=$TEC_SCRIPTS/tt_multi
	TEC_MULTI_TAG=$TEC_TOPDIR/multi-scripts.installed
	TEC_SCRIPT_NAME=tt.$Short_PRODNAME.$SHORT_VERSION.$VERSION.sh

#
#	if there's a file where the directory should be, rename it.
#
	if [ ! -d $TEC_SCRIPTS ] ; then
#
#	if there is a file named $BINDIR/TME/TEC/scripts
#	move it to scripts.old
#
		if [ -f $TEC_SCRIPTS ] ; then
			mv $TEC_SCRIPTS $TEC_SCRIPTS.old
		fi
#
#	Create the directory
#
		mkdir.sh $TEC_SCRIPTS
	fi

#
#	Copy Tivoli Trouble Ticket script to safe place
#
	if [ -f $TEC_MULTI_TAG ]; then
		outmsg LinkTecCat 64 "New trouble ticket script already in place"
	else
		mkdir.sh $TEC_MULTI_SCRIPTS
#
#	Copy any previous trouble ticket script to the new
#	directory so we don't lose anything that was already there.
#
		mv $TEC_TOPDIR/TroubleTicket.sh $TEC_MULTI_SCRIPTS
		cp $LINKDIR/PLUStt_multi.sh $TEC_TOPDIR/TroubleTicket.sh
		touch $TEC_MULTI_TAG
	fi

#
#	Move the old copy
#
	if [ -f $TEC_MULTI_SCRIPTS/$TEC_SCRIPT_NAME ] ; then
		mv $TEC_MULTI_SCRIPTS/$TEC_SCRIPT_NAME \
		$TEC_MULTI_SCRIPTS/$TEC_SCRIPT_NAME.orig
	fi

	cp $TEC_TT_OPEN_SCRIPT_1 $TEC_MULTI_SCRIPTS/$TEC_SCRIPT_NAME

#
#	Do an om_set_id on the native TEC trouble_ticketing method
#
	CLIENT=`wlookup -r Classes EnterpriseClient`
	METH_OBJ=`objcall $CLIENT _get_behavior`
	objcall $METH_OBJ om_set_id run_user_defined_tt_script root ""
}

