/* ***** BEGIN LICENSE BLOCK ***** 
 * Version: RCSL 1.0/RPSL 1.0 
 *  
 * Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved. 
 *      
 * The contents of this file, and the files included with this file, are 
 * subject to the current version of the RealNetworks Public Source License 
 * Version 1.0 (the "RPSL") available at 
 * http://www.helixcommunity.org/content/rpsl unless you have licensed 
 * the file under the RealNetworks Community Source License Version 1.0 
 * (the "RCSL") available at http://www.helixcommunity.org/content/rcsl, 
 * in which case the RCSL will apply. You may also obtain the license terms 
 * directly from RealNetworks.  You may not use this file except in 
 * compliance with the RPSL or, if you have a valid RCSL with RealNetworks 
 * applicable to this file, the RCSL.  Please see the applicable RPSL or 
 * RCSL for the rights, obligations and limitations governing use of the 
 * contents of the file.  
 *  
 * This file is part of the Helix DNA Technology. RealNetworks is the 
 * developer of the Original Code and owns the copyrights in the portions 
 * it created. 
 *  
 * This file, and the files included with this file, is distributed and made 
 * available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER 
 * EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES, 
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS 
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. 
 * 
 * Technology Compatibility Kit Test Suite(s) Location: 
 *    http://www.helixcommunity.org/content/tck 
 * 
 * Contributor(s): 
 *  
 * ***** END LICENSE BLOCK ***** */ 

/****************************************************************************
 * 
 *					   Transform Architecture 
 * 
 *  Defines enums and macros for the audio format field.
 *
 *  These enums transport the datatype of audio samples. There are several
 *  aspects to the data type. The most basic is, of course, endianness (and
 *  yes, there are more endiannesses than little- and big-endian, albeit those
 *  two do cover 99% of the cases).
 *
 *  The second aspect -- byte packing -- is relevant especially for 24-bit data
 *  types. These could be packed space-efficiently into 3 bytes per sample, but
 *  that format is very unwieldy to work with (on a Pentium, instead of one
 *  aligned 32-bit access to memory, you'll need three 8-bit accesses, and two
 *  shifts and additions).
 *  This points out that for fast execution, audio samples should be packed into
 *  the native types that your CPU supports -- usually 16-bit (short) and 32-bit
 *  (int), in (machine-)native endianness.
 *
 *  Most audio transform filters will only support these two datatypes. That is,
 *  they will advertise HXT_AUDIO_FORMAT_NE_16_2 and HXT_AUDIO_FORMAT_NE_32_4 as
 *  supported. At configuration time, they crack the enums using the
 *  HXT_BITS_PER_SAMPLE and/or HXT_BYTES_PER_SAMPLE macros.
 *
 *  On the input side, sample formats could run the whole gamut. A format
 *  converter filter takes care of casting any of the possible input formats
 *  into the canonical format (16-bit or 32-bit, native endianness).
 *
 *  If you are writing an input filter, you don't need to concern yourself with
 *  converting samples to the right endianness or byte packing -- just make sure
 *  you accurately report to the system which datatype is being delivered. For
 *  bits/sample, that's fairly straightforward. Bytes/sample is usually signalled
 *  in the header of the sound file you are reading, or is implied by the specs.
 *  The same goes true for (sound) capture cards.
 *
 *  Endianness is usually, but not always, determined by the endianness of the
 *  file you are reading, *not by the machine endianness*! If you read a
 *  WAV/RIFF file, endianness is little; if you read an AIFF file, it is
 *  big-endian. However, if your code does any endianness conversions (towards
 *  native, hopefully), the endianness to report would be native-endian.
 *
 *  It is a matter of taste how you declare the endianness if the samples are
 *  supplied by an operating-system function. If you get the samples from some
 *  MacOS function, you could decide to report big-endian, since that is the
 *  endianness of about all the Macs out there. On the other hand, are you
 *  *sure* that that MacOS will run on a little-endian machine? I would opt for
 *  using "native" -- that is equivalent on your machine, but will continue to
 *  work if the source is ever compiled for a little-endian machine.
 *
 *
 */

#ifndef _HXTAUDIOFORMAT_H
#define _HXTAUDIOFORMAT_H

#define HXT_ENDIANNESS_LITTLE 0
#define HXT_ENDIANNESS_BIG 1

#if defined _LITTLE_ENDIAN
    #define HXT_ENDIANNESS_NATIVE HXT_ENDIANNESS_LITTLE
#elif defined _BIG_ENDIAN
    #define HXT_ENDIANNESS_NATIVE HXT_ENDIANNESS_BIG
#else
	#if defined _WIN32 || defined _LINUX
		#define HXT_ENDIANNESS_NATIVE HXT_ENDIANNESS_LITTLE
	#elif defined _MAC_UNIX
		#define HXT_ENDIANNESS_NATIVE HXT_ENDIANNESS_BIG
	#else
		#error need to define ENDIANNESS
	#endif    
#endif

/* macro to generate a sample format  */

#define HXT_MAKE_SAMPLE_FORMAT( endianness, bitsPerSample, bytesPerSample ) \
     ( ((bitsPerSample)<<6) | ((bytesPerSample)<<2) | ((endianness)) )

/* macros to crack the sample format
   struct sampleFormat { int bits_per_sample:7 ; int bytes_per_sample : 4 ; int endianness : 2 ; }
 */

#define HXT_BITS_PER_SAMPLE(e)  (((e) >> 6) & 0x7f)
#define HXT_BYTES_PER_SAMPLE(e) (((e) >> 2) & 0x0f)
#define HXT_ENDIANNESS(e)       ( (e)       & 0x03)

enum EHXTAudioSampleFormat
{
	HXT_AUDIO_FORMAT_INVALID = 0,
	HXT_AUDIO_FORMAT_LE_8_1  = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_LITTLE, 8, 1 ),
	HXT_AUDIO_FORMAT_BE_8_1  = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_BIG,	   8, 1 ),
	HXT_AUDIO_FORMAT_NE_8_1  = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_NATIVE, 8, 1 ),
	HXT_AUDIO_FORMAT_LE_16_2 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_LITTLE, 16,2 ),
	HXT_AUDIO_FORMAT_BE_16_2 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_BIG,	   16,2 ),
	HXT_AUDIO_FORMAT_NE_16_2 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_NATIVE, 16,2 ),
	HXT_AUDIO_FORMAT_LE_20_3 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_LITTLE, 20,3 ),
	HXT_AUDIO_FORMAT_BE_20_3 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_BIG,	   20,3 ),
	HXT_AUDIO_FORMAT_NE_20_3 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_NATIVE, 20,3 ),
	HXT_AUDIO_FORMAT_LE_20_4 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_LITTLE, 20,4 ),
	HXT_AUDIO_FORMAT_BE_20_4 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_BIG,	   20,4 ),
	HXT_AUDIO_FORMAT_NE_20_4 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_NATIVE, 20,4 ),
	HXT_AUDIO_FORMAT_LE_24_3 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_LITTLE, 24,3 ),
	HXT_AUDIO_FORMAT_BE_24_3 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_BIG,	   24,3 ),
	HXT_AUDIO_FORMAT_NE_24_3 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_NATIVE, 24,3 ),
	HXT_AUDIO_FORMAT_LE_24_4 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_LITTLE, 24,4 ),
	HXT_AUDIO_FORMAT_BE_24_4 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_BIG,	   24,4 ),
	HXT_AUDIO_FORMAT_NE_24_4 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_NATIVE, 24,4 ),
	HXT_AUDIO_FORMAT_LE_32_4 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_LITTLE, 32,4 ),
	HXT_AUDIO_FORMAT_BE_32_4 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_BIG,	   32,4 ),
	HXT_AUDIO_FORMAT_NE_32_4 = HXT_MAKE_SAMPLE_FORMAT( HXT_ENDIANNESS_NATIVE, 32,4 )
};

enum EHXTAudioChannelFormat
{
	HXT_AUDIO_CHANNEL_FORMAT_INVALID= 0x0,
	HXT_AUDIO_FRONT_LEFT            = 0x1,
	HXT_AUDIO_FRONT_RIGHT           = 0x2,
	HXT_AUDIO_FRONT_CENTER          = 0x4,
	HXT_AUDIO_LOW_FREQUENCY         = 0x8,
	HXT_AUDIO_BACK_LEFT             = 0x10,
	HXT_AUDIO_BACK_RIGHT            = 0x20,
	HXT_AUDIO_FRONT_LEFT_OF_CENTER  = 0x40,
	HXT_AUDIO_FRONT_RIGHT_OF_CENTER = 0x80,
	HXT_AUDIO_BACK_CENTER           = 0x100,
	HXT_AUDIO_SIDE_LEFT             = 0x200,
	HXT_AUDIO_SIDE_RIGHT            = 0x400,
	HXT_AUDIO_TOP_CENTER            = 0x800,
	HXT_AUDIO_TOP_FRONT_LEFT        = 0x1000,
	HXT_AUDIO_TOP_FRONT_CENTER      = 0x2000,
	HXT_AUDIO_TOP_FRONT_RIGHT       = 0x4000,
	HXT_AUDIO_TOP_BACK_LEFT         = 0x8000,
	HXT_AUDIO_TOP_BACK_CENTER       = 0x10000,
	HXT_AUDIO_TOP_BACK_RIGHT        = 0x20000,
	HXT_AUDIO_RESERVED              = 0x80000000,
	
	// convenience defines	
	HXT_AUDIO_MONO = HXT_AUDIO_FRONT_CENTER,
	HXT_AUDIO_STEREO = HXT_AUDIO_FRONT_LEFT | HXT_AUDIO_FRONT_RIGHT, 
	HXT_AUDIO_4_CHANNELS_SURROUND = HXT_AUDIO_FRONT_LEFT | HXT_AUDIO_FRONT_RIGHT | HXT_AUDIO_FRONT_CENTER | HXT_AUDIO_BACK_CENTER,
	HXT_AUDIO_4_CHANNELS_QUADRO = HXT_AUDIO_FRONT_LEFT | HXT_AUDIO_FRONT_RIGHT | HXT_AUDIO_BACK_LEFT | HXT_AUDIO_BACK_RIGHT,
	HXT_AUDIO_5_CHANNELS   = HXT_AUDIO_FRONT_LEFT | HXT_AUDIO_FRONT_RIGHT | HXT_AUDIO_FRONT_CENTER | HXT_AUDIO_BACK_LEFT | HXT_AUDIO_BACK_RIGHT,
	HXT_AUDIO_5_1_CHANNELS = HXT_AUDIO_FRONT_LEFT | HXT_AUDIO_FRONT_RIGHT | HXT_AUDIO_FRONT_CENTER | HXT_AUDIO_BACK_LEFT | HXT_AUDIO_BACK_RIGHT | HXT_AUDIO_LOW_FREQUENCY	
};

static unsigned int HXT_MAKE_CHANNELMASK(int nchannels)
{
	static unsigned int mask[7] = { 0, HXT_AUDIO_MONO, HXT_AUDIO_STEREO, 0, HXT_AUDIO_4_CHANNELS_SURROUND, HXT_AUDIO_5_CHANNELS, HXT_AUDIO_5_1_CHANNELS } ;
	return (nchannels > 0 && nchannels <= 6) ? mask[nchannels] : 0 ;
}

static int HXT_NCHANNELS_IN_MASK(int m)
{
	int n = 0;
	for (; m ; m>>=1)
	{
		n += (m & 1) ;
	}
	return n;
}

#endif // _HXTAUDIOFORMAT_H
