#include <linux/linkage.h>
#include <pthread.h>
#include <linux/timer.h>

#include <stdio.h>


/*
 * This is the scheduler for the user mode universe
 *
 * We run the scheduler out of the main process, by popping
 * in and out of the system.
 */

#define MAX_JOBS 20

pthread_cond_t sched_cond = PTHREAD_COND_INITIALIZER;
pthread_mutex_t sched_mutex;

#define JOB_STATE_INIT		1
#define JOB_STATE_IDLE		2
#define JOB_STATE_RUNNING	3
#define JOB_STATE_RUNOK		4
#define JOB_STATE_SLEEP		5
#define JOB_STATE_EXIT		6
#define JOB_STATE_TIMER		7	/* Pseudo job for timer */

struct user_job {
    pthread_t		job_tid;
    int			(*job_func)(int ac, char **av);
    char		*job_argbase;
    int			job_ac;
    char		**job_av;
    pthread_mutex_t	job_mutex;
    pthread_cond_t	job_cond;
    int			job_state;
    int			job_index;
} *jobs[MAX_JOBS];

void
dump_thread_state()
{
    int i;
    int found = 0;

    printf("Execution harness thread state\n");

    for (i = 0; i < MAX_JOBS; i++) {
	if ( ! jobs[i])
	    continue;

	printf("Thread %d, '%s', state %d\n", i, jobs[i]->job_av[0],
			jobs[i]->job_state);
	found = 1;
    }

    if ( ! found)
	printf("No threads active\n");

}

#undef start_thread
void *start_thread(void *);

int
do_user_cmd(int (*func)(int ac, char **av), char *argbase, int ac, char **av)
{
    struct user_job *job;
    int i;

    for (i = 0; i < MAX_JOBS; i++) {
	if ( ! jobs[i])
	    break;
    }

    if ( i == MAX_JOBS) {
	printf("do_user_cmd: too many active threads for new command\n");
	return 0;
    }

    job = jobs[i] = (struct user_job *) malloc(sizeof(*jobs[i]));

    if (job == NULL) {
	printf("do_user_cmd: too many active threads for new command\n");
	return 0;
    }

    memset(job, 0, sizeof(*job));

    job->job_index = i;

    pthread_mutex_init(&job->job_mutex, NULL);
    pthread_cond_init(&job->job_cond, NULL);

    job->job_state = JOB_STATE_INIT;
    job->job_func = func;
    job->job_argbase = argbase;
    job->job_ac = ac;
    job->job_av = (char **) malloc((ac + 1) * sizeof(*av));

    for (i = 0; i <= ac; i++)
	job->job_av[i] = av[i];

    pthread_mutex_lock(&job->job_mutex);

    pthread_create(&job->job_tid, NULL, start_thread, (void *) job);

    pthread_cond_wait(&job->job_cond, &job->job_mutex);
}

void *
start_thread(void *ap)
{
    struct user_job *job = (struct user_job *) ap;

    pthread_mutex_lock(&job->job_mutex);

    job->job_state = JOB_STATE_RUNOK;
    pthread_cond_signal(&job->job_cond);
    pthread_cond_wait(&job->job_cond, &job->job_mutex);

    job->job_func(job->job_ac, job->job_av);

    jobs[job->job_index] = NULL;
    job->job_state = JOB_STATE_EXIT;

    /* 
     * We are done - wake up the parent, and arrange
     * for the mutex to be released on our dying gasp.
     */
    pthread_cond_signal(&job->job_cond);
    pthread_cleanup_push((void (*)(void *)) pthread_mutex_unlock,
		(void *) &job->job_mutex);

    /*
     * Go gasp
     */
    pthread_exit(0);

    pthread_cleanup_pop(0);
}

struct user_job *
look_me_up()
{
    struct user_job *job = 0;
    int i;

    for (i = 0; i < MAX_JOBS; i++) {
	if ( ! jobs[i])
	    continue;

	job = jobs[i];
	if (pthread_equal(job->job_tid, pthread_self()))
	    break;
    }

    if ( ! job) {
	printf("look_me_up: can't find myself!!!\n");
	abort();
    }

    return job;
}

void *
set_me_to_sleep()
{
    struct user_job *job;

    job = look_me_up();
    job->job_state = JOB_STATE_SLEEP;

    return (void *) job;
}

wake_him_up(void *him)
{
    struct user_job *job;

    job = (struct user_job *) him;

    job->job_state = JOB_STATE_RUNOK;
}

asmlinkage void
schedule(void)
{
    struct user_job *job;

    job = look_me_up();

    pthread_cond_signal(&job->job_cond);
    pthread_cond_wait(&job->job_cond, &job->job_mutex);
}

void
do_scheduler()
{
    struct user_job *job;
    int i;
    static int lastjob = -1;

    job = NULL;

    fflush(stdout);
    for (i = 0; i < MAX_JOBS; i++) {
	if (++lastjob >= MAX_JOBS)
	    lastjob = 0;

	if ( ! (job = jobs[lastjob]))
	    continue;

	if (jobs[lastjob]->job_state == JOB_STATE_RUNOK)
	    break;
    }

    if ( ! job || job->job_state != JOB_STATE_RUNOK) {
	return;
    }

    job->job_state = JOB_STATE_RUNNING;
    pthread_cond_signal(&job->job_cond);
    pthread_cond_wait(&job->job_cond, &job->job_mutex);

    if (job->job_state == JOB_STATE_EXIT) {
	pthread_join(job->job_tid, NULL);
	pthread_cond_destroy(&job->job_cond);
	pthread_mutex_destroy(&job->job_mutex);

	free(job->job_argbase);
	free(job->job_av);
	jobs[job->job_index] = NULL;
	free(job);
    }
    else if (job->job_state == JOB_STATE_RUNNING)
	job->job_state = JOB_STATE_RUNOK;
    fflush(stdout);
}

unsigned long jiffies;
struct timer_list	*tim_list;

pthread_mutex_t		timer_mutex;
pthread_cond_t		timer_cond;

void
add_timer_locked(struct timer_list *timer)
{

#if 0
    /* For now, time advances by hand, so signaling it here
     * could actually cause problems.  Also, note that timer_cond
     * synchronizes between tick() and run_timer(), so the wrong
     * thing might wakeup here
     */
    if (tim_list == NULL)
	pthread_cond_signal(&timer_cond);
#endif
    timer->next = tim_list;
    tim_list = timer;
}


void
add_timer(struct timer_list * timer)
{
    pthread_mutex_lock(&timer_mutex);

    add_timer_locked(timer);

    pthread_mutex_unlock(&timer_mutex);
}

int
del_timer_locked(struct timer_list * timer)
{
    struct timer_list **tim_pp;

    for (tim_pp = &tim_list; *tim_pp; tim_pp = &(*tim_pp)->next) {
	if (*tim_pp == timer) {
	    *tim_pp = timer->next;
	    timer->next = 0;
	    return 1;
	}
    }
    return 0;
}


int
del_timer(struct timer_list * timer)
{
    struct timer_list **tim_pp;

    pthread_mutex_lock(&timer_mutex);
    del_timer_locked(timer);
    pthread_mutex_unlock(&timer_mutex);
    return 0;
}

void
mod_timer(struct timer_list * timer, unsigned long jiff)
{
    pthread_mutex_lock(&timer_mutex);

    timer->expires = jiff;

    if ( ! timer->next)
	add_timer_locked(timer);

    pthread_mutex_unlock(&timer_mutex);
}

static int user_timer_init;

tick_tock()
{
    if ( ! user_timer_init) {
	printf("Timer is not running - can't tick clock\n");
	printf("Use the \"timer\" command to start clock\n");
	return -1;
    }

    /*
     * Drive one clock tick
     */
    pthread_mutex_lock(&timer_mutex);
    pthread_cond_signal(&timer_cond);
    pthread_cond_wait(&timer_cond, &timer_mutex);
    pthread_mutex_unlock(&timer_mutex);

    return 0;
}

run_timer()
{
    struct timer_list *tp, *start_tp, *next;
    void (*fcn)(unsigned long);
    unsigned long data;
    int min_next = -1;

    /*
     * Grab our lock to keep us synchronized with tick_tock()
     */
    pthread_mutex_lock(&timer_mutex);

    while (1) {

	/*
	 * Handshake with tick_tock()
	 */
	fflush(stdout);
	pthread_cond_signal(&timer_cond);
	pthread_cond_wait(&timer_cond, &timer_mutex);

	if (min_next == -1)
	    jiffies++;
	else {
	    jiffies += min_next;
	    min_next = -1;
	}

	start_tp = tp = tim_list;
	if (tp) do {
	    /*
	     * The following detects timers set when ticks
	     * race ahead of the timer arming proceedure.
	     */
	    if (tp->expires == jiffies ||
		    (tp->expires < jiffies && tp->expires > jiffies - 100)) {
		data = tp->data;
		fcn = tp->function;
		next = tp->next;

		del_timer_locked(tp);
		tp = next;

		pthread_mutex_unlock(&timer_mutex);
		fcn(data);
		pthread_mutex_lock(&timer_mutex);
	    }
	    else {
		if (min_next == -1 || min_next > tp->expires - jiffies)
		    min_next = tp->expires - jiffies;
		tp = tp->next;
	    }
	} while (tp && tp != start_tp);
    }
}

start_timer()
{
    struct user_job *job;

    if (user_timer_init) {
	printf("timer already running\n");
	return;
    }
    user_timer_init = 1;

    pthread_mutex_init(&timer_mutex, NULL);
    pthread_cond_init(&timer_cond, NULL);

    job = look_me_up();

    /*
     * Flag so the "scheduler" won't try to run us
     */
    job->job_state = JOB_STATE_TIMER;

    /*
     * Wakeup the scheduler, and release our thread mutex
     * so the scheduler can proceed merrily.
     */
    pthread_cond_signal(&job->job_cond);
    pthread_mutex_unlock(&job->job_mutex);

    run_timer();
    /*
     * never returns
     */
}
// LICENSE:
// This software is subject to the terms of the GNU GENERAL 
// PUBLIC LICENSE Version 2, June 1991
