#include <linux/version.h>
#include <sys/socket.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <netdb.h>
#include <string.h>
#include <fcntl.h>
#include <sys/uio.h>
#include <bw_user.h>
#include <sys/poll.h>

#ifndef min
#define min(x, y) ((x) < (y) ? (x) : (y))
#endif

#define TEST_FAMILY AF_INET
#define TEST_PROTO 6	/* tcp */

extern int errno;
extern char *optarg;
extern unsigned int jiffies;

#define HZ 100
#define INADDR_ANY 0


#define MAX_SOCK_ADDR 128

extern int verbose;

extern void dbg_printf(int flag, const char *fmt, ...);

int
test_create(int ac, char **av)
{
    int fd, c, ret;
    int type = SOCK_STREAM;

    optind = 0;
    if (ac) while ((c = getopt(ac, av, "hd")) != EOF) 
	switch (c) {
	case 'h':
	    printf("create: create a new socket and return the integer "
			"file descriptor\n");
	    printf("usage: create [-h] [-d]\n\t'-h' : print this message\n");
	    printf("\t-d: set datagram mode\n");
	    return;

	case 'd':
	    type = SOCK_DGRAM;
	    break;
	}
    fd = get_sock_fd();

    set_fd_type(fd, type);

    ret = do_create(fd, TEST_PROTO);

    if (ret >= 0) {
	printf("created socket %d\n", fd);
	return fd;
    }
    else {
	errno = -ret;
	perror("create failed");
	free_sock_fd(fd);
	return -1;
    }
}

int         
cvt_int(char *numstr, int *errp)
{
    long val;       
    char *cvt;  

    val = strtol(numstr, &cvt, 0);
    if (cvt == numstr) {
	*errp = 1;
	return 0;
    }

    switch (*cvt) {
    case 'm': case 'M':
	val *= 1024 * 1024;
	break;  
    case 'k': case 'K':
	val *= 1024;
	break;  
    }

    *errp = 0;      
    return val;
}      


int
test_accept(int ac, char **av)
{
    int err, fd, newfd, c, ret;
    int flags;	/* file flags on src socket - O_NONBLOCK, etc. */
    char address[MAX_SOCK_ADDR];
    int len = MAX_SOCK_ADDR;

    optind = 0;
    while ((c = getopt(ac, av, "hf:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("accept: test accept and return the integer id of the"
			"new file descriptor\n");
	    printf("usage: accept [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: accept from fd\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("accept : error converting fd %s\n", optarg);
		return -1;
	    }
	    break;
	}

    if (fd == -1) {
	printf("accept: must set file descriptor\n");
	return -1;
    }

    newfd = get_sock_fd();

    if (newfd < 0) {
	printf("accept: cannot create new socket\n");
	return -1;
    }

    set_fd_type(newfd, get_fd_type(fd));

    if ((ret = do_dup(fd, newfd)) < 0) {
	errno =  -ret;
	perror("accept: dup failed");
	goto bad_out;
    }

    if ((ret = do_accept(newfd, fd, flags)) < 0) {
	errno =  -ret;
	perror("accept: accept failed");
	goto bad_out;
    }
    /*
     * this is here because it is in the kernel.
     * not really necessary for return value
     */
    if ((ret = do_getname(newfd, (struct sockaddr *) address, &len, 1)) < 0) {
	errno =  -ret;
	perror("accept: getname failed");
	goto bad_out;
    }
    return 0;

bad_out:
    free_sock_fd(newfd);
    return -1;
}

int
test_connect(int ac, char **av)
{
    int flags = 0, c, fd = -1, ret, err;
    char address[MAX_SOCK_ADDR];

    optind = 0;
    while ((c = getopt(ac, av, "hf:n")) != EOF) 
	switch (c) {
	case 'h':
	    printf("connect: test connect\n");
	    printf("usage: connect [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: connect fd\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("connect : error converting fd %s\n", optarg);
		return -1;
	    }
	case 'n':
	    flags = O_NONBLOCK;
	    break;
	}

    if (fd == -1) {
	printf("connect: must set file descriptor\n");
	return -1;
    }

    ret = do_connect(fd, (struct sockaddr *)address, MAX_SOCK_ADDR, flags);

    if (ret < 0) {
	errno = -ret;
	perror("connect failed\n");
	return -1;
    }
    return 0;
}

int
test_socketpair(int ac, char **av)
{
    int fd0, fd1, c, ret;

    optind = 0;
    while ((c = getopt(ac, av, "h")) != EOF) 
	switch (c) {
	case 'h':
	    printf("socketpair: test socketpair");
	    printf("usage: socketpair [-h]\n\t-h: print this message\n");
	    return;
	}

    fd0 = test_create(0, NULL);
    fd1 = test_create(0, NULL);

    if (fd0 < 0 || fd1 < 0) {
	printf("socketpair: cannot create sockets\n");
	goto bad_out;
    }

    ret = do_socketpair(fd0, fd1);

    if (ret < 0) {
	errno = -ret;
	perror("socketpair failed");
	goto bad_out;
    }
    return 0;

bad_out:
    if (fd0 >= 0)
	free_sock_fd(fd0);

    if (fd1 >= 0)
	free_sock_fd(fd1);

    return -1;
}

int
test_getname(int ac, char **av)
{

    int fd = -1, c, len, ret, err;
    struct sockaddr address;
    int peer;

    optind = 0;
    while ((c = getopt(ac, av, "hf:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("getname: test getname and print the address\n");
	    printf("usage: getname [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: getname from fd\n");
	    return;
	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("getname : error converting fd %s\n", optarg);
		return -1;
	    }
	}

    if (fd == -1) {
	printf("getname: must set file descriptor\n");
	return -1;
    }

    ret = do_getname(fd, &address, &len, peer);
    if (ret < 0) {
	errno = -ret;
	perror("getname failed\n");
	return -1;
    }
    printf("getname %d returned address 0x%x\n", fd,
		* (int *) &address.sa_data[0]);
    return 0;
}

int
test_select(int ac, char **av)
{
#if 1 || LINUX_VERSION_CODE >= 0x20200
    printf("select is not a primitive on this version of linux\n");
    return -1;
#else

    int fd = -1, ret, c;

    optind = 0;
    while ((c = getopt(ac, av, "hf:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("select: test select and return the integer mask\n");
	    printf("usage: select [-h]\n\t-h: print this message\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("select : error converting fd %s\n", optarg);
		return -1;
	    }
	    if (fd >= MAX_TESTSOCK || ! socklist[fd]) {
		printf("select : bad socket %d\n", fd);
		return -1;
	    }
	}

    if (fd == -1) {
	printf("select: must set file descriptor\n");
	return -1;
    }

#endif
}

int
test_poll(int ac, char **av)
{
#if 0 || LINUX_VERSION_CODE < 0x20200
    printf("poll is not a primitive on this version of linux\n");
    return -1;
#else
    int fd = -1, ret, c, err;

    optind = 0;
    while ((c = getopt(ac, av, "hf:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("poll: test poll and return the integer poll result\n");
	    printf("usage: poll [-h]\n\t-h: print this message\n");
	    printf("\t-f fd: poll from fd\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("poll : error converting fd %s\n", optarg);
		return -1;
	    }
	}

    if (fd == -1) {
	printf("poll: must set file descriptor\n");
	return -1;
    }

    ret = do_poll(fd);

    if (ret < 0) {
	errno = -ret;
	perror("poll failed");
	return -1;
    }
    printf("poll returned 0x%x\n", ret);
    if (ret)
	printf("\t");
#define poll_output(ret, bit)	if (ret & bit) printf("%s ", #bit)
    poll_output(ret, POLLIN);
	/* poll_output(ret, POLLRDNORM); */
	/* poll_output(ret, POLLRDBAND); */
    poll_output(ret, POLLOUT);
	/* poll_output(ret, POLLWRNORM); */
    poll_output(ret, POLLHUP);
    poll_output(ret, POLLPRI);
	/* poll_output(ret, POLLRDNORM); */
	/* poll_output(ret, POLLWRBAND); */

    if (ret)
	printf("\n");

    return 1;

#endif
}

int
test_listen(int ac, char **av)
{
    int fd = -1, ret, c, err;

    optind = 0;
    while ((c = getopt(ac, av, "hf:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("listen: test listen and return the integer id of the"
			"new file descriptor\n");
	    printf("usage: listen [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: listen from fd\n");
	    return;
	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("listen : error converting fd %s\n", optarg);
		return -1;
	    }
	}

    if (fd == -1) {
	printf("listen: must set file descriptor\n");
	return -1;
    }

    ret = do_listen(fd, 5);

    if (ret < 0) {
	errno = -ret;
	perror("listen failed");
	return -1;
    }
    return 0;
}

int
test_setsockopt(int ac, char **av)
{
    int fd = -1, ret, c, opts = 0, err;
    int level = 42, val;

    optind = 0;
    while ((c = getopt(ac, av, "hf:o:l:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("setsockopt: test setsockopt\n");
	    printf("usage: setsockopt [-h] -f fd [-o option] [-l level]\n");
	    printf("\t-h: print this message\n");
	    printf("\t-f fd: setsockopt on fd\n");
	    printf("\t-o options: option to set\n");
	    printf("\t-v value: value to set for option\n");
	    printf("\t-l level: set level for option\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("getsockopt : error converting fd %s\n", optarg);
		return -1;
	    }
	    break;

	case 'o':
	    opts = cvt_int(optarg, &err);
	    if (err) {
		printf("setsockopt : error converting options %s\n", optarg);
		return -1;
	    }
	    break;

	case 'v':
	    val = cvt_int(optarg, &err);
	    if (err) {
		printf("setsockopt : error converting option value %s\n",
				optarg);
		return -1;
	    }
	    break;

	case 'l':
	    level = cvt_int(optarg, &err);
	    if (err) {
		printf("setsockopt : error converting level %s\n", optarg);
		return -1;
	    }
	}

    if (fd == -1) {
	printf("setsockopt: must set file descriptor\n");
	return -1;
    }

    ret = do_setsockopt(fd, level, opts, (char *) &val, sizeof(val));

    if (ret < 0) {
	errno = -ret;
	perror("setsockopt failed");
	return -1;
    }
    return 0;
}

int
test_getsockopt(int ac, char **av)
{
    int fd = -1, ret, c, opts = 0, err;
    int level = 42, val, len = sizeof(val);

    optind = 0;
    while ((c = getopt(ac, av, "hf:o:l:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("getsockopt: test getsockopt\n");
	    printf("usage: getsockopt [-h] -f fd [-o option] [-l level]\n");
	    printf("\t-h: print this message\n");
	    printf("\t-f fd: getsockopt on fd\n");
	    printf("\t-o options: options to get\n");
	    printf("\t-l level: get level for option\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("getsockopt : error converting fd %s\n", optarg);
		return -1;
	    }
	    break;

	case 'o':
	    opts = cvt_int(optarg, &err);
	    if (err) {
		printf("getsockopt : error converting option %s\n", optarg);
		return -1;
	    }
	    break;

	case 'l':
	    level = cvt_int(optarg, &err);
	    if (err) {
		printf("getsockopt : error converting level %s\n", optarg);
		return -1;
	    }
	}

    if (fd == -1) {
	printf("getsockopt: must set file descriptor\n");
	return -1;
    }

    ret = do_getsockopt(fd, level, opts, (char *) &val, &len);

    if (ret < 0) {
	errno = -ret;
	perror("getsockopt failed");
	return -1;
    }
    printf("getsockopt returned 0x%x\n", val);
    return 0;
}

int
test_fcntl(int ac, char **av)
{
    int fd = -1, ret, c, err;
    int cmd, val;

    optind = 0;
    while ((c = getopt(ac, av, "hf:c:v:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("fcntl: test fcntl and return the integer id of the"
			"new file descriptor\n");
	    printf("usage: fcntl [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: fcntl from fd\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("fcntl : error converting fd %s\n", optarg);
		return -1;
	    }
	    break;

	case 'c':
	    cmd = cvt_int(optarg, &err);
	    if (err) {
		printf("fcntl : error converting command %s\n", optarg);
		return -1;
	    }
	    break;

	case 'v':
	    val = cvt_int(optarg, &err);
	    if (err) {
		printf("fcntl : error converting argument %s\n", optarg);
		return -1;
	    }
	    break;
	}

    if (fd == -1) {
	printf("fcntl: must set file descriptor\n");
	return -1;
    }

    ret = do_fcntl(fd, cmd, val);

    if (ret < 0) {
	errno = -ret;
	perror("fcntl failed");
	return -1;
    }

    printf("fcntl returned value 0x%x\n", ret);
    return 0;
}

int
test_sendmsg(int ac, char **av)
{
    int fd = -1, ret, c, err;
    struct msghdr       msg_sys;
    struct iovec        iovec;
    char                buf[1000];
    int size = sizeof(buf);
    char *buffp = buf;
    int start_time, stop_time;


    optind = 0;
    while ((c = getopt(ac, av, "hf:s:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("sendmsg: send a message\n");
	    printf("usage: sendmsg [-h] -f fd [-s size]\n");
	    printf("\t-h: print this message\n");
	    printf("\t-f fd: send on fd\n");
	    printf("\t-s size: set message size (default %d)\n", size);
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("sendmsg : error converting fd %s\n", optarg);
		return -1;
	    }
	    break;

	case 's':
	    size = cvt_int(optarg, &err);
	    if (err) {
		printf("sendmsg : error converting size %s\n", optarg);
		return -1;
	    }
	    break;
	}

    if (size > sizeof(buf)) {
	if ((buffp = malloc(size)) == NULL) {
	    printf("sendmsg: cannot allocate %d bytes for test\n", size);
	    return -1;
	}
    }

    if (fd == -1) {
	printf("sendmsg: must set file descriptor\n");
	return -1;
    }

    iovec.iov_base = (void *) buffp;
    iovec.iov_len = size;

    memset(&msg_sys, 0, sizeof(msg_sys));
    msg_sys.msg_iov = &iovec;
    msg_sys.msg_iovlen = 1;


    start_time = jiffies;
    ret = do_sendmsg(fd, &msg_sys, size, NULL);
 
    if ((stop_time = jiffies) == start_time)
	stop_time = start_time + 1;

    if (ret < 0) {
	errno = -ret;
	perror("sendmsg failed");
	return -1;
    }

    printf("sendmsg on fd %d sent %d bytes @ %d bytes per second\n",
	    fd, ret, size * HZ / (stop_time - start_time));

    return 0;
}

int
test_recvmsg(int ac, char **av)
{
    int fd = -1, ret, c, err;
    int size;
    int start_time, stop_time;
    struct msghdr       msg_sys;
    struct iovec        iovec;
    char                buf[1000];
    char *buffp = buf;


    optind = 0;
    while ((c = getopt(ac, av, "hf:s:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("recvmsg: test recvmsg and return the integer id of the"
			"new file descriptor\n");
	    printf("usage: recvmsg [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: recvmsg from fd\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("recvmsg : error converting fd %s\n", optarg);
		return -1;
	    }
	    break;

	case 's':
	    size = cvt_int(optarg, &err);
	    if (err) {
		printf("recvmsg : error converting size %s\n", optarg);
		return -1;
	    }
	}

    if (size > sizeof(buf)) {
	if ((buffp = malloc(size)) == NULL) {
	    printf("recvmsg: cannot allocate %d bytes for test\n", size);
	    return -1;
	}
    }

    if (fd == -1) {
	printf("recv: must set file descriptor\n");
	return -1;
    }

    iovec.iov_base = (void *) buffp;
    iovec.iov_len = size;

    memset(&msg_sys, 0, sizeof(msg_sys));
    msg_sys.msg_iov = &iovec;
    msg_sys.msg_iovlen = 1;


    start_time = jiffies;
    ret = do_recvmsg(fd, &msg_sys, size, 0, NULL);

    if ((stop_time = jiffies) == start_time)
	stop_time = start_time + 1;

    if (ret < 0) {
	errno = -ret;
	perror("recvmsg failed");
	return -1;
    }

    printf("recvmsg sent %d bytes @ %d bytes per second\n",
	    ret, size * HZ / (stop_time - start_time));

    return 0;
}

int
test_release(int ac, char **av)
{
    int fd = -1, ret, c, err;

    optind = 0;
    while ((c = getopt(ac, av, "hf:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("release: test release and return the integer id of the"
			"new file descriptor\n");
	    printf("usage: release [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: release from fd\n");
	    return;
	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("release : error converting fd %s\n", optarg);
		return -1;
	    }
	}

    if (fd == -1) {
	printf("release: must set file descriptor\n");
	return -1;
    }

    ret = do_release(fd);

    free_sock_fd(fd);

    if (ret < 0) {
	errno = -ret;
	perror("release failed");
	return -1;
    }

    printf("release succeded\n");
    return 0;
}

int
test_shutdown(int ac, char **av)
{
    int fd = -1, ret, c, err;
    int how = 0;

    optind = 0;
    while ((c = getopt(ac, av, "hf:o:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("shutdown: test shutdown and return the integer id of the"
			"new file descriptor\n");
	    printf("usage: shutdown [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: shutdown from fd\n");
	    printf("\t-o how: set how : 0 == rx, 1 == tx, 2 == both\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("shutdown : error converting fd %s\n", optarg);
		return -1;
	    }
	    break;

	case 'o':
	    how = cvt_int(optarg, &err);
	    if (err) {
		printf("shutdown : error converting how %s\n", optarg);
		return -1;
	    }
	}

    if (fd == -1) {
	printf("shutdown: must set file descriptor\n");
	return -1;
    }

    ret = do_shutdown(fd, how);

    free_sock_fd(fd);

    if (ret < 0) {
	errno = -ret;
	perror("shutdown failed");
	return -1;
    }

    printf("shutdown succeded\n");
    return 0;

}

int
test_setbw(int ac, char **av)
{
    int fd, ret, c, err;
    struct hostent *hent = 0;
    struct servent *serv;
    struct protoent *protop;
    struct sockaddr saddr;
    char *protname;
    char *cp;
    int mask = 32;
    short proto = TEST_PROTO;
    short port = INADDR_ANY;
    int type = BW_RDWRIO;
    int bps;

    optind = 0;
    while ((c = getopt(ac, av, "hi:p:t:b:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("setbw: set an IP based bandwidth limit\n");
	    printf("usage: setbw [-h] -i addr_spec -p proto_spec -t rw -b bps\n");
	    printf("\t-h: print this message\n");
	    printf("-i addr_spec: limit host or address, optional trailing '/mask'\n");
	    printf("-p proto_spec: 'protocol/port' specifier, port optional\n");
	    printf("-t rw: read/write limit, 'r', 'w', or 'rw'\n");
	    printf("-b bps: byte limit per second\n");
	    return;

	case 'i':
	    /* 
	     * ipaddr/mask
	     */
	    if (cp = strchr(optarg, '/')) {
		*cp++ = 0;
		mask = cvt_int(cp, &err);

		if (err) {
		    printf("setbw : error converting mask %s\n", optarg);
		    return -1;
		}
	    }

	    hent = gethostbyname(optarg);
	    if ( ! hent) {
		printf("setbw: error looking up host %s\n", optarg);
		return -1;
	    }

	    memcpy(&saddr.sa_data, hent->h_addr, hent->h_length);
	    saddr.sa_family = hent->h_addrtype;
	    break;

	case 'p':
	    /*
	     * protocol/port
	     */
	    if (cp = strchr(optarg, '/'))
		*cp++ = 0;

	    if (protop = getprotobyname(optarg)) {
		proto = protop->p_proto;
		protname = optarg;
	    }
	    else {
		proto = cvt_int(optarg, &err);

		if ( ! err && (protop = getprotobynumber(proto))) {
		    protname = strdup(protop->p_name);
		}
		else if (proto == TEST_PROTO || ! strcmp(optarg, "test")) {
		    proto = TEST_PROTO;
		    protname = "test";
		}
		else {
		    printf("setbw: error converting protocol %s\n", optarg);
		    return -1;
		}
	    }

	    if (cp) {
		if (serv = getservbyname(cp, protname)) {
		    port = htons(serv->s_port);
		}
		else {
		    port = cvt_int(cp, &err);

		    if (err) {
			printf("setbw : error converting port %s\n", optarg);
			return -1;
		    }
		}
	    }
	    break;
	case 't':
	    /*
	     * type : read/write/both
	     */
	    if ( ! strcmp(optarg, "r"))
		type = BW_RDIO;
	    else if ( ! strcmp(optarg, "w"))
		type = BW_WRIO;
	    else if ( ! strcmp(optarg, "rw") || ! strcmp(optarg, "wr"))
		type = BW_RDWRIO;
	    else {
		printf("setbw: unknown read/write type %s\n", optarg);
		return -1;
	    }
	    break;
	case 'b':
	    bps = cvt_int(optarg, &err);

	    if (err) {
		printf("setbw : error converting bps %s\n", optarg);
		return -1;
	    }
	    break;
	}

    if ( ! hent) {
	printf("setip: must set ip address\n");
	return -1;
    }

    if ((fd = test_create(0, NULL)) < 0) {
	printf("setbw cannot open control fd\n");
	return -1;
    }

    ret = bw_set_ip_limit(fd, &saddr, hent->h_length, mask, proto,
		port, type, bps);

    free_sock_fd(fd);

    if (ret < 0) {
	errno = -ret;
	perror("setbw failed");
	return -1;
    }
    printf("setbw succeded\n");
    return 0;
}

int
test_proc(int ac, char **av)
{
    int ret, c;
    char *file;

    optind = 0;
    while ((c = getopt(ac, av, "hf:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("proc: test proc and return the integer id of the"
			"new file descriptor\n");
	    printf("usage: proc [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: proc from fd\n");
	    return;
	case 'f':
	    file = optarg;
	}

    /*
     * open the file and cat it
     */
}

extern void do_scheduler();

int
test_run(int ac, char **av)
{
    int fd, ret, c, err;
    int time;

    optind = 0;
    while ((c = getopt(ac, av, "ht:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("run: test run and return the integer id of the"
			"new file descriptor\n");
	    printf("usage: run [-h] -f fd\n\t-h: print this message\n");
	    printf("\t-f fd: run from fd\n");
	    return;
	case 't':
	    time = cvt_int(optarg, &err);
	    if (err) {
		printf("run : error converting fd %s\n", optarg);
		return -1;
	    }
	}

    do_scheduler();
}

int
test_timer(int ac, char **av)
{
    int fd, ret, c, err;
    extern void start_timer();

    optind = 0;
    while ((c = getopt(ac, av, "h")) != EOF) 
	switch (c) {
	case 'h':
	    printf("timer: start system timer - use \"tick\" to advance\n");
	    printf("usage: timer [-h]\t-h: print this message\n");
	    return;
	}

    start_timer();
}

int
test_tick(int ac, char **av)
{
    int fd, ret, c, err;
    int time = 1;
    int run_flg = 0;
    extern int tick_tock();
    int end_time;

    optind = 0;
    while ((c = getopt(ac, av, "ht:r")) != EOF) 
	switch (c) {
	case 'h':
	    printf("tick: run system clock - use \"timer\" to start\n");
	    printf("usage: tick [-h] [-t time] [-r]\t-h: print this message\n");
	    printf("\t-t time : run for time ticks (default is 1 tick)\n");
	    printf("\t-r : run scheduler between clock ticks\n");
	    return;

	case 't':
	    time = cvt_int(optarg, &err);
	    if (err) {
		printf("tick : error converting time %s\n", optarg);
		return -1;
	    }
	    break;

	case 'r':
	    run_flg = 1;
	    break;
	}

    end_time = jiffies + time;
    while (jiffies < end_time && tick_tock() >= 0)
	if (run_flg)
	    do_scheduler();

    printf("time is now %d\n", jiffies);
}

int error_freq = 0;

int
test_seterror(int ac, char **av)
{
    int fd, ret, c;
    int tmp = -1, err;

    optind = 0;
    while ((c = getopt(ac, av, "he:")) != EOF) 
	switch (c) {
	case 'h':
usage:
	    printf("seterror: set a percentage for errors\n");
	    printf("usage: seterror [-h] -e error\n\t-h: print this message\n");
	    printf("\t-e error_percent: set percent of operations to fail\n");
	    return;

	case 'e':
	    tmp = cvt_int(optarg, &err);
	    if (err) {
		printf("tick : error converting time %s\n", optarg);
		return -1;
	    }
	    break;
	}

    if (tmp < 0 || tmp > 100) {
	goto usage;
    }
    error_freq = tmp;
}

test_bind(int ac, char **av)
{
    int fd = -1, ret, c, err;
    struct sockaddr saddr;
    struct hostent *hent = 0;

    optind = 0;
    while ((c = getopt(ac, av, "hf:a:")) != EOF) 
	switch (c) {
	case 'h':
	    printf("bind: set an error number for all operations\n");
	    printf("usage: seterror [-h] -f fd -a host_addr\n");
	    printf("\t-h: print this message\n");
	    printf("\t-f fd: set file descriptor\n");
	    printf("\t-a host_addr: set address or name for bind\n");
	    return;

	case 'f':
	    fd = cvt_int(optarg, &err);
	    if (err) {
		printf("bind : error converting fd %s\n", optarg);
		return -1;
	    }
	    break;


	case 'a':
	    hent = gethostbyname(optarg);
	    if ( ! hent) {
		printf("bind: error looking up host %s\n", optarg);
		return -1;
	    }

	    memcpy(&saddr.sa_data, hent->h_addr, hent->h_length);
	    saddr.sa_family = hent->h_addrtype;
	    break;
	}

    if (fd == -1) {
	printf("bind: must set file descriptor\n");
	return -1;
    }

    if ( ! hent) {
	printf("bind: must set bind address\n");
	return -1;
    }

    ret = do_bind(fd, (struct sockaddr *) &saddr, hent->h_length);

    if (ret < 0) {
	errno = -1;
	perror("bind failed");
	return -1;
    }

    printf("bind succeded\n");

    return 0;
}

int
test_quit(int ac, char **av)
{
    /*
     * Can't get here - input loop processes "quit" directly
     */
    return 0;
}

int
test_ps(int ac, char **av)
{
    extern void dump_thread_state();
    dump_thread_state();
}

int test_help(int ac, char **av);

#define USER_LOCAL 1
#define USER_THREAD 2
struct user_cmd_look {
    char *cmd;
    int (*func)(int, char **);
    int flags;
} user_cmds[] = {
    { "create",		test_create,		USER_THREAD },
    { "accept",		test_accept,		USER_THREAD },
    { "connect",	test_connect,		USER_THREAD },
    { "socketpair",	test_socketpair,	USER_THREAD },
    { "getname",	test_getname,		USER_THREAD },
    { "select",		test_select,		USER_THREAD },
    { "poll",		test_poll,		USER_THREAD },
    { "listen",		test_listen,		USER_THREAD },
    { "setsockopt",	test_setsockopt,	USER_THREAD },
    { "getsockopt",	test_getsockopt,	USER_THREAD },
    { "fcntl",		test_fcntl,		USER_THREAD },
    { "sendmsg",	test_sendmsg,		USER_THREAD },
    { "recvmsg",	test_recvmsg,		USER_THREAD },
    { "release",	test_release,		USER_THREAD },
    { "shutdown",	test_shutdown,		USER_THREAD },
    { "setbw",		test_setbw,		USER_THREAD },
    { "/proc",		test_proc,		USER_THREAD },
    { "bind",		test_bind,		USER_THREAD },
    { "help",		test_help,		USER_LOCAL },
    { "run",		test_run,		USER_LOCAL },
    { "seterror",	test_seterror,		USER_LOCAL },
    { "quit",		test_quit,		USER_LOCAL },
    { "ps",		test_ps,		USER_LOCAL },
    { "timer",		test_timer,		USER_THREAD },
    { "tick",		test_tick,		USER_LOCAL },
};

int
test_help(int ac, char **av)
{
    int i;
    /*
     * scan through the list and say something useful
     */
    printf("Interactive socket test interface\n");
    printf("Supported commands:\n");
    for (i = 0; i < sizeof(user_cmds) / sizeof(user_cmds[0]); i++) {
	printf("\t%s\n", user_cmds[i].cmd);
    }
    printf("Each command accepts a '-h' flag for help\n");
}

extern int do_user_cmd(int (*)(int, char **), char *base, int ac, char **av);

user_interact()
{
    char *buf = NULL, *cp;
    char *av[10];
    int len;
    int i, ac;
    int found;

    printf("\n\nInteractive user test, v0.001\n");
    printf("All safeties are off!!\n");

    printf("bwio> ");
    fflush(stdout);

    while (1) {

#define INP_BUF_SZ 128
	if ( ! buf)
	    buf = (char *) malloc(INP_BUF_SZ);

	if (fgets(buf, INP_BUF_SZ, stdin) == NULL)
	    break;

	ac = 0;
	found = 0;
	cp = buf;

	while (isspace(*cp))
	    cp++;

	len = strlen(cp);

	if ( ! len)
	    continue;

	do {
	    while (isspace(*cp))
		cp++;

	    av[ac++] = cp;

	    while ( ! isspace(*cp) && cp < buf + len)
		cp++;

	    if (cp < buf + len)
		*cp++ = 0;

	} while (ac < sizeof(av) - 1 && cp < buf + len);

	if (ac > 0 && ! strcmp(av[0], "quit")) 
	    break;

	av[ac] = 0;

	for (i = 0; i < sizeof(user_cmds) / sizeof(user_cmds[0]); i++) {
	    if ( ! strcmp(user_cmds[i].cmd, buf)) {
		found = 1;
		break;
	    }
	}
	if ( ! found) {
	    printf("Unknown cmd %s\n", buf);
	}
	else if (user_cmds[i].flags == USER_THREAD) {
	    do_user_cmd(user_cmds[i].func, buf, ac, av);
	    buf = 0;
	}
	else user_cmds[i].func(ac, av);

	printf("bwio> ");
	fflush(stdout);
    }
    printf("\ndone\n");
}
// LICENSE:
// This software is subject to the terms of the GNU GENERAL 
// PUBLIC LICENSE Version 2, June 1991
