#ifdef MODULE
#if LINUX_VERSION_CODE >= 0x20400
#include <linux/module.h>
#endif
#include <linux/modversions.h>
#endif

/*
 * bw_proc.c : /proc interface to bandwidth management.
 *
 * Copyright (C) 1999-2001, Sun Microsystems, Inc.
 * All rights reserved.
 *
 * This is pretty straight forward.  We produce directories
 * and files that say neat things.  Sort of.
 */
#include <bw_mgmt.h>
#include <linux/proc_fs.h>

#if LINUX_VERSION_CODE < 0x20400
struct proc_dir_entry bw_dir_proc_fs = {
    0,					/* low_ino */
    sizeof("bw_mgmt") - 1,
    "bw_mgmt",
    S_IFDIR | S_IRUGO | S_IXUGO,	/* mode */
    2, 0, 0,				/* nlink, uid, gid */
    0,					/* size */
    NULL,				/* inode_operations */
    NULL,				/* get_info() */
    NULL, 				/* fill_inode() */
    NULL,				/* next proc_dir_entry */
    &proc_root,				/* parent proc_dir_entry */
    NULL,				/* subdir proc_dir_entry */
    NULL,				/* void *data */
    NULL,				/* read_proc() */
    NULL,				/* write_proc() */
    NULL,				/* readlink_proc() */
    0,					/* count */
    0,					/* delete flag */
};
#else
struct proc_dir_entry *bw_dir_proc_fs;
#endif

read_proc_t bw_read_tot_cnt;
write_proc_t bw_write_tot_cnt;

#if LINUX_VERSION_CODE < 0x20400
struct proc_dir_entry bw_tot_proc_fs = {
    0,					/* low_ino */
    sizeof("tot_cnt") - 1,
    "tot_cnt",
    S_IFREG | S_IRUGO,			/* mode */
    1, 0, 0,				/* nlink, uid, gid */
    0,					/* size */
    NULL,				/* inode_operations */
    NULL,				/* get_info() */
    NULL, 				/* fill_inode() */
    NULL,				/* next proc_dir_entry */
    &bw_dir_proc_fs,			/* parent proc_dir_entry */
    NULL,				/* subdir proc_dir_entry */
    NULL,				/* void *data */
    bw_read_tot_cnt,			/* read_proc() */
    bw_write_tot_cnt,			/* write_proc() */
    NULL,				/* readlink_proc() */
    0,					/* count */
    0,					/* delete flag */
};
#else
struct proc_dir_entry *bw_tot_proc_fs;
#endif 

read_proc_t bw_read_qinfo;
write_proc_t bw_write_qinfo;

#if LINUX_VERSION_CODE < 0x20400
struct proc_dir_entry bw_qinfo_proc_fs = {
    0,					/* low_ino */
    sizeof("qinfo") - 1,
    "qinfo",
    S_IFREG | S_IRUGO,			/* mode */
    1, 0, 0,				/* nlink, uid, gid */
    0,					/* size */
    NULL,				/* inode_operations */
    NULL,				/* get_info() */
    NULL, 				/* fill_inode() */
    NULL,				/* next proc_dir_entry */
    &bw_dir_proc_fs,			/* parent proc_dir_entry */
    NULL,				/* subdir proc_dir_entry */
    NULL,				/* void *data */
    bw_read_qinfo,			/* read_proc() */
    bw_write_qinfo,			/* write_proc() */
    NULL,				/* readlink_proc() */
    0,					/* count */
    0,					/* delete flag */
};
#else
struct proc_dir_entry *bw_qinfo_proc_fs;
#endif

extern struct bw_time_cnt bw_total_mincnt[2][BW_NUMMINUTES];

int bw_proc_count(struct bw_time_cnt *, char *title_str, char *page);

/*
 * page is the base of a MIN(count, PROC_BUF_SIZE) region to
 * do the I/O to.  The higher layer shuffles this into memory.
 *
 * start is a hack that lets us set our initial offset in the
 * output to other than *page.
 *
 * off is the file offset.
 *
 * count is count.
 *
 * Set *eof == 1 when we hit the end of file
 *
 * data is our proc_dir_entry cookie 
 */
int
bw_read_tot_cnt(char *page, char **start, off_t off,
	    int count, int *eof, void *data)
{
    extern bwl_totcnt_t bw_total_count[2];
    int ret = 0;
    NASTY_LOCK_DECL;

    NASTY_LOCK( NASTY_LOCK_FLAGS );
    BW_ENTER_FUNC();

    ret = bw_proc_count(bw_total_mincnt[BW_WRIO], "transmit", page);
    ret += sprintf(page + ret, "total transmit bytes %Lu\n",
		bw_total_count[BW_WRIO]);

#ifdef BW_TRACK_READ
    ret += bw_proc_count(bw_total_mincnt[BW_RDIO], "receive", page + ret);
    ret += sprintf(page + ret, "total receive bytes %Lu\n",
		bw_total_count[BW_RDIO]);
#endif

    if (off != 0) {
	if (ret < off) {
	    *eof = 1;
	    BW_EXIT_FUNC();
	    NASTY_UNLOCK( NASTY_LOCK_FLAGS );
	    return 0;
	}
	ret -= off;
	*start = page + off;
    }

    if (count < ret) {
	ret = count;
    }
    else *eof = 1;

    BW_EXIT_FUNC();
    NASTY_UNLOCK( NASTY_LOCK_FLAGS );
    return ret;
}

#define BYTES_TO_UNITS_MULT	8
#define BYTES_TO_UNITS_DIV	1000
#define UNITS "kbits"

#define BW_SCALE_PRECISION 1000

/*
 * Tack macro to get the right order of operations
 * in the presence of integer overflows.
 */
#define MULDIV(x, mult, div)		\
    (((x) > 0x7ffffff / (mult))		\
	? (((x) / (div)) * (mult))	\
	: (((x) * (mult)) / (div)))

/*
 * The bandwidth counter averager.  This is the common backend
 * to the global stats and the per limit stats.
 */
int
bw_proc_count(struct bw_time_cnt *bwcnt, char *title_str, char *page)
{
    unsigned long one_min = 0, five_min = 0, fifteen_min = 0;
    struct bw_time_cnt tmp_cnt[BW_NUMMINUTES];
    

    bwl_time_t cur_jiff;
    bwl_time_t cur_min;
    bwl_time_t tmp_time;
    int first_frac;
    int count;
    int delta;
    int i;
    int ret = 0;

    BW_ENTER_FUNC();

    /*
     * Snapshot the counters - avoid weird edges when time
     * increments while we are running.
     */
    memcpy(tmp_cnt, bwcnt, sizeof(tmp_cnt));
    cur_jiff = jiffies;

    cur_min = cur_jiff / BW_MIN_RATE;

    /*
     * first_frac is how far into the first time bucket we need
     * to track data.  Since we don't have actual data at this
     * granularity, we scale linearly.
     *
     * Note we don't need to scale the current time bucket because
     * it is "automatically" scaled.  If the minute is half over,
     * we have half a minute of data.
     */
    delta = cur_jiff - (cur_min * BW_MIN_RATE);
    first_frac = (BW_MIN_RATE - delta) * BW_SCALE_PRECISION / BW_MIN_RATE;

    for (i = 0; i < BW_NUMMINUTES; i++) {

	tmp_time = tmp_cnt[i].bwcnt_time;
	count = tmp_cnt[i].bwcnt_cnt;

	/*
	 * We use an extra minute for all the averages because
	 * otherwise we might only get a jiffy worth of data.
	 * Eg. if the clock just ticked into a new minute, our
	 * one minute average would be an extrapolation based
	 * on one jiffy of data.  So we lie and average over
	 * one to two minutes.  Same for 5 and 15 minutes.
	 *
	 * The scaling stuff guarantees a smooth curve.  We
	 * pretend the universe is linear.
	 */
	if ((unsigned) (cur_min - tmp_time) == 0) {
	    one_min += count;
	    five_min += count;
	    fifteen_min += count;
	}
	else {
	    if ((unsigned) (cur_min - tmp_time) == 1)
		one_min += MULDIV(count, first_frac, BW_SCALE_PRECISION);

	    if ((unsigned) (cur_min - tmp_time) < 5)
		five_min += tmp_cnt[i].bwcnt_cnt;
	    else if ((unsigned) (cur_min - tmp_time) == 5)
		five_min += MULDIV(count, first_frac, BW_SCALE_PRECISION);

	    if ((unsigned) (cur_min - tmp_time) < 15)
		fifteen_min += tmp_cnt[i].bwcnt_cnt;
	    if ((unsigned) (cur_min - tmp_time) == 15)
		fifteen_min += MULDIV(count, first_frac, BW_SCALE_PRECISION);
	}
    }

    /*
     * Next, we need to calculate the time for the bandwidth
     * counts above.
     *
     * We scale the counts up by 100 so we can get 2 decimal
     * precision without floats. We divide by 5 minutes for
     * the 5 minute average (etc.) because the scaling stuff
     * above guarantees that we have a reasonable (i.e. linear)
     * estimate for an even number of minutes.
     *
     * Furthermore, we don't know whether the byte count is
     * way high or low, so we check and do the calculation in
     * the correct order to maintain precision.
     */


#ifdef BW_DEBUG
    ret += sprintf(page + ret, "%7s raw bytes %10ld %10ld %10ld\n",
	    title_str,
	    one_min, five_min, fifteen_min);
    ret += sprintf(page + ret, "delta is %d\n", delta);
#endif

#define CVT_TIME(x, mult)					\
    if (x > 0x7ffffff / (mult * BW_CLOCKRATE * 100)) {		\
	x /= mult * BW_MIN_RATE;				\
	x *= BW_CLOCKRATE * 100;				\
	x *= BYTES_TO_UNITS_MULT;				\
	x /= BYTES_TO_UNITS_DIV;				\
    }								\
    else {							\
	x *= BW_CLOCKRATE * 100;				\
	x /= mult * BW_MIN_RATE;				\
	x *= BYTES_TO_UNITS_MULT;				\
	x /= BYTES_TO_UNITS_DIV;				\
    }

    CVT_TIME(one_min, 1);
    CVT_TIME(five_min, 5);
    CVT_TIME(fifteen_min, 15);

    ret = sprintf(page + ret,
	"%s 1/5/15 minute average %s/s %10ld.%02ld %10ld.%02ld %10ld.%02ld\n",
	    title_str, UNITS,
	    one_min / 100, one_min % 100,
	    five_min / 100, five_min % 100,
	    fifteen_min / 100, fifteen_min % 100);

    BW_EXIT_FUNC();
    return ret;
}


int
bw_write_tot_cnt(struct file *file, const char *buffer,
	    unsigned long count, void *data)
{
    BW_ENTER_FUNC();
    BW_EXIT_FUNC();
    return 0;
}

#if LINUX_VERSION_CODE < 0x20400
struct proc_dir_entry bw_ip_proc_fs = {
    0,					/* low_ino */
    sizeof("ip") - 1,
    "ip",
    S_IFDIR | S_IRUGO | S_IXUGO,	/* mode */
    2, 0, 0,				/* nlink, uid, gid */
    0,					/* size */
    NULL,				/* inode_operations */
    NULL,				/* get_info() */
    NULL, 				/* fill_inode() */
    NULL,				/* next proc_dir_entry */
    &bw_dir_proc_fs,			/* parent proc_dir_entry */
    NULL,				/* subdir proc_dir_entry */
    NULL,				/* void *data */
    NULL,				/* read_proc() */
    NULL,				/* write_proc() */
    NULL,				/* readlink_proc() */
    0,					/* count */
    0,					/* delete flag */
};

struct proc_dir_entry bw_limit_template = {
    0,					/* low_ino */
    0,					/* namelen */
    NULL,				/* name */
    S_IFREG | S_IRUGO,			/* mode */
    1, 0, 0,				/* nlink, uid, gid */
    0,					/* size */
    NULL,				/* inode_operations */
    NULL,				/* get_info() */
    NULL, 				/* fill_inode() */
    NULL,				/* next proc_dir_entry */
    NULL,				/* parent proc_dir_entry */
    NULL,				/* subdir proc_dir_entry */
    NULL,				/* void *data */
    NULL,				/* read_proc() */
    NULL,				/* write_proc() */
    NULL,				/* readlink_proc() */
    0,					/* count */
    0,					/* delete flag */
};
#else
struct proc_dir_entry *bw_ip_proc_fs;
#endif

atomic_t bw_num_ip_procs;

void bw_proc_newip(struct bw_limit_hash *bw_limit);

/* as far as I can tell... this funcation is no longer used */
#if 0
void
bw_proc_newlimit(struct bw_limit_hash *bw_limit)
{
    BW_ENTER_FUNC();
    switch (bw_limit->bwh_lim_id.bwid_type) {
    case BW_IP:
	bw_proc_newip(bw_limit);
	BW_EXIT_FUNC();
	return;

    default:
	BW_EXIT_FUNC();
	return;
    }
    BW_EXIT_FUNC();
}
#endif
int bw_proc_readip(char *page, char **start, off_t off,
	    int count, int *eof, void *data);
int bw_proc_writeip(struct file *file, const char *buffer,
	    unsigned long count, void *data);

void
bw_proc_newip(struct bw_limit_hash *bw_limit)
{
    struct proc_dir_entry *filp;
    int addr_len;
    int i, cnt, off, tbyte, tprec;
    u_int32_t addr;
    char *name;
    
    BW_ENTER_FUNC();

    addr = htonl(bw_limit->bwh_lim_id.bwid_addr);
    addr_len = sizeof(addr);

#if LINUX_VERSION_CODE < 0x20400
    filp = (struct proc_dir_entry *) kmalloc(sizeof(*filp), BW_GFP );
    if ( ! filp)
    {
	BW_EXIT_FUNC();
	return;
    }
    memcpy(filp, &bw_limit_template, sizeof(*filp));
#endif

    for (i = 0, cnt = 0; i < addr_len; i++) {
	tbyte = (addr >> (i * 8)) & 0xff;

	if (tbyte >= 100)
	    cnt += 3;
	else if (tbyte >= 10) 
	    cnt += 2;
	else cnt++;
    }

    cnt += addr_len - 1;	/* add 1 for '.'s */

    /*
     * We add '1' to keep sprintf happy.
     * The /proc interface doesn't care.
     */
    name = (char *) kmalloc(cnt + 1, BW_GFP );

#if LINUX_VERSION_CODE < 0x20400
    filp->name = name;
    filp->namelen = cnt;
#endif

    if ( ! name) {
#if LINUX_VERSION_CODE < 0x20400
	kfree(filp);
#endif
	BW_EXIT_FUNC();
	return;
    }

    for (i = 0, off = 0; i < addr_len; ) {

	tbyte = (addr >> (i * 8)) & 0xff;

	if (tbyte >= 100)
	    tprec = 3;
	else if (tbyte >= 10)
	    tprec = 2;
	else tprec = 1;

	off += sprintf(name + off, "%*d", tprec, tbyte);

	if (++i != addr_len) {
	    off += sprintf(name + off, ".");
	}
    }
#if LINUX_VERSION_CODE < 0x20400
    filp->parent = &bw_ip_proc_fs;
    filp->read_proc = bw_proc_readip;
    filp->write_proc = bw_proc_writeip;
    filp->data = bw_limit;
    proc_register(&bw_ip_proc_fs, filp);
#else
    filp = create_proc_entry( name,
			      S_IFREG | S_IRUGO, bw_ip_proc_fs );

    filp->read_proc = bw_proc_readip;
    filp->write_proc = bw_proc_writeip;
    filp->data = bw_limit;
#endif

    atomic_inc( &bw_num_ip_procs );
    bw_limit->bwh_proc_ent = filp;
    BW_EXIT_FUNC();
}

void
bw_proc_delip(struct bw_limit_hash *bw_limit)
{
    struct proc_dir_entry *filp = bw_limit->bwh_proc_ent;

    BW_ENTER_FUNC();

    if ( ! filp)
    {
	BW_EXIT_FUNC();
	return;
    }

#if LINUX_VERSION_CODE < 0x20400
    proc_unregister(&bw_ip_proc_fs, filp->low_ino);

    kfree(filp->name);
    kfree(filp);
#else
    remove_proc_entry( filp->name, bw_ip_proc_fs );
#endif
    BW_EXIT_FUNC();
}

extern struct bw_limit_hash *bw_iphash(struct bw_limit_id *, enum bw_iotype rw);

int
bw_proc_readip(char *page, char **start, off_t off,
	    int count, int *eof, void *data)
{
    struct bw_limit_hash *bw_limit, *wlim;
    u_int32_t addr;
    u_int32_t rate;
#ifdef BW_TRACK_READ
    struct bw_limit_hash *rlim;
#endif
    int ret = 0;
    NASTY_LOCK_DECL;

    NASTY_LOCK( NASTY_LOCK_FLAGS );
    BW_ENTER_FUNC();

    bw_limit = (struct bw_limit_hash *) data;

    addr = bw_limit->bwh_lim_id.bwid_addr;
    ret += sprintf(page + ret, "%d.%d.%d.%d\n",
	(addr >> 24) & 255, (addr >> 16) & 255, (addr >> 8) & 255, addr & 255);

    wlim = bw_iphash(&bw_limit->bwh_lim_id, BW_WRIO);

    rate = wlim->bwh_bps * BYTES_TO_UNITS_MULT / BYTES_TO_UNITS_DIV;

    ret += sprintf(page + ret, "Write limit %d %s/second\n", rate, UNITS);

    ret += bw_proc_count(wlim->bwhcnt_minutes, "write", page + ret);
    ret += sprintf(page + ret, "total bytes written %Lu\n",
		wlim->bwh_lim_tot_bytes);

#ifdef BW_TRACK_READ
    rlim = bw_iphash(&bw_limit->bwh_lim_id, BW_RDIO);

    rate = rlim->bwh_bps * BYTES_TO_UNITS_MULT / BYTES_TO_UNITS_DIV;

    ret += sprintf(page + ret, "Read limit %d %s/second\n", rate, UNITS);

    ret += bw_proc_count(rlim->bwhcnt_minutes, "read", page + ret);
    ret += sprintf(page + ret, "total bytes read %Lu\n",
		rlim->bwh_lim_tot_bytes);
#endif


    if (off != 0) {
	if (ret < off) {
	    *eof = 1;
	    BW_EXIT_FUNC();
	    NASTY_UNLOCK( NASTY_LOCK_FLAGS );
	    return 0;
	}
	ret -= off;
	*start = page + off;
    }

    if (count < ret) {
	ret = count;
    }
    else *eof = 1;

    BW_EXIT_FUNC();
    NASTY_UNLOCK( NASTY_LOCK_FLAGS );
    return ret;
}

int
bw_proc_writeip(struct file *file, const char *buffer,
	    unsigned long count, void *data)
{
    BW_ENTER_FUNC();
    BW_EXIT_FUNC();
    return 0;
}

void
bw_proc_init(void)
{
    BW_ENTER_FUNC();
    atomic_set( &bw_num_ip_procs, 0 );

#if LINUX_VERSION_CODE < 0x20400
    proc_register(&proc_root, &bw_dir_proc_fs);
    proc_register(&bw_dir_proc_fs, &bw_tot_proc_fs);
    proc_register(&bw_dir_proc_fs, &bw_ip_proc_fs);
    proc_register(&bw_dir_proc_fs, &bw_qinfo_proc_fs);
#else
    bw_dir_proc_fs = create_proc_entry( "bw_mgmt", 
					S_IFDIR | S_IRUGO | S_IXUGO, &proc_root );
    bw_dir_proc_fs->nlink = 2;
    
    bw_tot_proc_fs = create_proc_entry( "tot_cnt",
					S_IFREG | S_IRUGO, bw_dir_proc_fs );
    bw_tot_proc_fs->read_proc = bw_read_tot_cnt;
    bw_tot_proc_fs->write_proc = bw_write_tot_cnt;

    bw_qinfo_proc_fs = create_proc_entry( "qinfo",
					S_IFREG | S_IRUGO, bw_dir_proc_fs );
    bw_qinfo_proc_fs->read_proc = bw_read_qinfo;
    bw_qinfo_proc_fs->write_proc = bw_write_qinfo;

    bw_ip_proc_fs = create_proc_entry( "ip",
				       S_IFDIR | /*S_IFREG |*/ S_IRUGO | S_IXUGO,
				       bw_dir_proc_fs );
    bw_ip_proc_fs->nlink=2;
#endif
    BW_EXIT_FUNC();
}

void
bw_proc_disable(void)
{
    BW_ENTER_FUNC();
#if LINUX_VERSION_CODE < 0x20400
    proc_unregister(&bw_dir_proc_fs, bw_ip_proc_fs.low_ino);
    proc_unregister(&bw_dir_proc_fs, bw_qinfo_proc_fs.low_ino);
    proc_unregister(&bw_dir_proc_fs, bw_tot_proc_fs.low_ino);
    proc_unregister(&proc_root, bw_dir_proc_fs.low_ino);
#else
    remove_proc_entry( "ip", bw_dir_proc_fs );
    remove_proc_entry( "qinfo", bw_dir_proc_fs );
    remove_proc_entry( "tot_cnt", bw_dir_proc_fs );
    remove_proc_entry( "bw_mgmt", &proc_root );
#endif
    BW_EXIT_FUNC();
}

extern spinlock_t bw_iosched_lock, bw_ioactive_lock;

int
bw_read_qinfo(char *page, char **start, off_t off,
	    int count, int *eof, void *data)
{
    extern struct bw_queue bw_sched_list;
    extern struct bw_queue bw_iolist[];
    extern atomic_t bw_numconnections;
    struct bw_ioreq *bwreq;
    int i;
    int ret = 0;
    int totcnt;
    int actcnt[BW_NUMIO_CNT];
    int actslot;
    BW_LOCK_FLAGS_DECL
    NASTY_LOCK_DECL;
    
    BW_ENTER_FUNC();
    NASTY_LOCK( NASTY_LOCK_FLAGS );
    BW_LOG_LOCK(&bw_iosched_lock, BW_LOCK_FLAGS);

    totcnt = 0;
    for (bwreq = bw_sched_list.bwq_head; bwreq; bwreq = bwreq->bw_iosched) {
	totcnt++;
    }

    BW_LOG_UNLOCK(&bw_iosched_lock, BW_LOCK_FLAGS);

    ret += sprintf(page + ret, "I/O Sched Q : %d entries\n", totcnt);

    memset(actcnt, 0, sizeof(actcnt));
    totcnt = 0;

    BW_LOG_LOCK(&bw_ioactive_lock, BW_LOCK_FLAGS);

    for (i = 0; i < BW_NUMIO_CNT; i++) {
	actslot = BW_IOSLOT_TOSCHED(i);

	for (bwreq = bw_iolist[i].bwq_head; bwreq;
			bwreq = bwreq->bw_ioactive[actslot]) {
	    actcnt[i]++;
	    totcnt++;
	}       
    }    

    BW_LOG_UNLOCK(&bw_ioactive_lock, BW_LOCK_FLAGS);

    if (totcnt) {
	ret += sprintf(page + ret, "%d total I/O requests active\n", totcnt);
	for (i = 0; i < BW_NUMIO_CNT; i++) {
	    ret += sprintf(page + ret, "I/O Active Q %d : %d entries\n",
		i, actcnt[i]);
	}
    }
    else {
	ret += sprintf(page + ret, "No I/O requests active\n");
    }
    ret += sprintf(page + ret, "Total connections %d\n", atomic_read( &bw_numconnections) );

    if (off != 0) {
	if (ret < off) {
	    *eof = 1;
	    BW_EXIT_FUNC();
	    NASTY_UNLOCK( NASTY_LOCK_FLAGS );
	    return 0;
	}
	ret -= off;
	*start = page + off;
    }

    if (count < ret) {
	ret = count;
    }
    else *eof = 1;

    BW_EXIT_FUNC();
    NASTY_UNLOCK( NASTY_LOCK_FLAGS );
    return ret;
}

int bw_write_qinfo(struct file *file, const char *buffer,
	    unsigned long count, void *data)
{
    BW_ENTER_FUNC();
    BW_EXIT_FUNC();
    return 0;
}
// LICENSE:
// This software is subject to the terms of the GNU GENERAL 
// PUBLIC LICENSE Version 2, June 1991
