#ifdef MODULE
#if LINUX_VERSION_CODE >= 0x20400
#include <linux/module.h>
#endif
#include <linux/modversions.h>
#endif

/*
 * bw_iov.c : bandwidth management I/O structure alloc/update routines.
 *
 * Copyright (C) 1999-2001, Sun Microsystems, Inc.
 * All rights reserved.
 *
 * These routines initialize the io request structure, and update
 * io vectors to track the partial I/O that the bandwidth management
 * layer schedules.
 */

#include <bw_mgmt.h>
#if LINUX_VERSION_CODE >= 0x20400
#include <linux/vmalloc.h>
#endif

#define DPRINTF if (0) printk

/*
 * Initialize the bwio structure to enable us to manage the iov
 */
void
bw_init_ioreq(struct bw_ioreq *bwreq, struct bw_info *bwi, int rw,
		struct msghdr *m, int len)
{
    BW_ENTER_FUNC();
    if (rw >= 2) {
	panic("bw_init_ioreq: bad rw flag %d\n", rw);
    }

    memset(bwreq, 0, sizeof(*bwreq));
#ifdef DEBUG
    bwreq->bw_magic = BW_REQ_MAGIC;
#endif
    atomic_set( &bwreq->bw_prealloc, 0 );
    atomic_set( &bwreq->bw_totallen, 0 );
    bwreq->bw_bwi = bwi;
    bwreq->bw_rw = rw;
    bwreq->bw_msg = m;
    bwreq->bw_iov_curbound = NULL;
    atomic_set( &bwreq->bw_totallen, len );
    bwreq->bw_total_ioc = len;
#if LINUX_VERSION_CODE >= 0x020400
	/* because of linux2.4s head type we need to init this structure */
    bwreq->bw_sleep_qpp = &bwreq->bw_sleep;
    init_waitqueue_head(bwreq->bw_sleep_qpp);
#endif
    if (len < atomic_read( &bwi->bw_io[rw].bw_prealloc) ){
	atomic_set( &bwreq->bw_prealloc, len );
	atomic_sub( len, &bwi->bw_io[rw].bw_prealloc );
    }
    else {
	atomic_set( &bwreq->bw_prealloc,
		    atomic_read(&bwi->bw_io[rw].bw_prealloc) );
	atomic_set( &bwi->bw_io[rw].bw_prealloc, 0 );
    }

    bwreq->bw_error_ret = 0;

    spin_lock_init(&bwreq->bw_busylock); 

    /*
     * msghdr is NULL for poll operations
     */
    if (m)
	bwreq->bw_iov_end = m->msg_iov + m->msg_iovlen;
    BW_EXIT_FUNC();
}

/*
 * To delete an io_request, we just return any excess allocation
 * to the pre-allocated pool for this socket.  When the socket
 * is closed, we will return any remaining excess to the limit.
 */
void
bw_delete_ioreq(struct bw_ioreq *bwreq, struct bw_info *bwi)
{
    BW_ENTER_FUNC();
    spin_lock(&bwreq->bw_busylock); 
    atomic_add( atomic_read( &bwreq->bw_prealloc ),
		&bwi->bw_io[bwreq->bw_rw].bw_prealloc );
    BW_EXIT_FUNC();
}

/*
 * Given the bw_ioreq arg, update its iov to only include the
 * first nbytes bytes.  Store the partial state.
 *
 * safe for the same reason as bw_iov_pullup
 */
void
bw_iov_front(struct bw_ioreq *bwreq, int nbytes)
{
    int saved_len;
    int curlen = 0;
    struct iovec *iov;

    BW_ENTER_FUNC();
    BW_CHECK_REQ(bwreq);

    for (iov = bwreq->bw_msg->msg_iov; iov < bwreq->bw_iov_end; iov++) {
	ASSERT(iov);
	curlen += iov->iov_len;
	saved_len = curlen - nbytes;
	if (saved_len >= 0) {
	    bwreq->bw_saved_iov.iov_len = saved_len;
	    iov->iov_len -= saved_len;
	    bwreq->bw_saved_iov.iov_base = iov->iov_base + iov->iov_len;
	    bwreq->bw_iov_curbound = iov;
	    bwreq->bw_msg->msg_iovlen = 1 + iov - bwreq->bw_msg->msg_iov;
	    DPRINTF("bw_iov_front: sock 0x%x, base 0x%x, len %d\n",
		    (int) bwreq->bw_bwi->bw_sock,
		    (int) bwreq->bw_msg->msg_iov->iov_base,
		    bwreq->bw_msg->msg_iov->iov_len);
	    BW_EXIT_FUNC();
	    return;
	}
    }
    /*
     * Fell off the end - the iov is shorter than nbytes.
     * update universe so that the sky won't fall
     */
    bwreq->bw_iov_curbound = iov;
    BW_CHECK_REQ(bwreq);
    BW_EXIT_FUNC();
}

/*
 * Given an bw_ioreq previously updated by bw_front_iov(), set the iov
 * to include the previously saved partial state.  Fix the iov to start
 * at the new beginning.
 *
 * this is only ever called by bw_do_sendmsg and bw_do_some_active.
 * bw_do_some_active will only opperate on it after bw_do_sendmsg
 * has finnished calling... i.e bwreq->bw_iov_curbound is safe for
 * a +=
 */
void
bw_iov_pullup(struct bw_ioreq *bwreq, int nplanned, int nio)
{
    struct iovec *iov;
    int len;

    BW_ENTER_FUNC();

    BW_CHECK_REQ(bwreq);
    if (bwreq->bw_iov_curbound >= bwreq->bw_iov_end)
    {
	BW_EXIT_FUNC();
	return;
    }

    if (nplanned == nio) {
	*bwreq->bw_iov_curbound = bwreq->bw_saved_iov;
	bwreq->bw_msg->msg_iov = bwreq->bw_iov_curbound;
	bwreq->bw_msg->msg_iovlen =
		    1 + bwreq->bw_iov_end - bwreq->bw_iov_curbound;
    }
    else {
	/*
	 * A mess - have to back up the pointers by the shortfall
	 */
	DPRINTF("bw_iov_pullup - short io - hard case\n");
	bwreq->bw_iov_curbound->iov_len += bwreq->bw_saved_iov.iov_len;
	iov = bwreq->bw_msg->msg_iov;
	len = 0;
	while (iov <= bwreq->bw_iov_curbound) {
	    if (len + iov->iov_len > nio)
		break;
	    len += iov->iov_len;
	    iov++;
	}
	/*
	 * Got the iov that marks the done/not_done boundary
	 */
	iov->iov_base += nio - len;
	iov->iov_len += nio - len;
	bwreq->bw_msg->msg_iovlen =
		    1 + bwreq->bw_iov_end - bwreq->bw_msg->msg_iov;
	
	atomic_add( nplanned-nio, &bwreq->bw_totallen );
    }
    BW_CHECK_REQ(bwreq);
    DPRINTF("bw_iov_pullup: sock 0x%x, base 0x%x, len %d\n",
	    (int) bwreq->bw_bwi->bw_sock,
	    (int) bwreq->bw_msg->msg_iov->iov_base,
	    bwreq->bw_msg->msg_iov->iov_len);
    BW_EXIT_FUNC();
}

/*
 * Allocate a kernel buffer (available at interrupt time),
 * and copyin the iov for later use.
 */
int
bw_iov_copyin(struct bw_ioreq *bwreq, unsigned long *flags)
{
    struct iovec *iov;
    int len = 0;
    char *buff;
#if LINUX_VERSION_CODE < 0x20400
    extern void *vmalloc(int);
#endif
    BW_ENTER_FUNC();
    BW_CHECK_REQ(bwreq);

    for (iov = bwreq->bw_msg->msg_iov; iov < bwreq->bw_iov_end; iov++) {
	len += iov->iov_len;
    }

    /*
     * Use valloc to get contiguous virtual memory.  If it
     * fails, try halving the length until it gets silly.
     *
     * Application using very large buffers will need to
     * deal with short I/O return values.
     */
    NASTY_UNLOCK( *flags );
    while ((buff = (char *) vmalloc(len)) == NULL) {
	len = len / 2;

	if ( len < 1024)
	{
	    NASTY_LOCK( *flags );
	    BW_EXIT_FUNC();
	    return 0;
	}
    }
    NASTY_LOCK( *flags );
    iov = bwreq->bw_msg->msg_iov;

    memcpy_fromiovec(buff, iov, len);

    bwreq->bw_allocbase = iov->iov_base = buff;
    iov->iov_len = len;

    bwreq->bw_msg->msg_iovlen = 1;

    bwreq->bw_msg->msg_iov = iov;
    bwreq->bw_iov_curbound = iov;
    bwreq->bw_iov_end = iov + 1;
    atomic_set( &bwreq->bw_totallen, len );
    bwreq->bw_total_ioc = len;

    DPRINTF("bw_iov_copyin: sock 0x%x, base 0x%x, len %d\n",
	    (int) bwreq->bw_bwi->bw_sock,
	    (int) bwreq->bw_msg->msg_iov->iov_base,
	    bwreq->bw_msg->msg_iov->iov_len);
    BW_EXIT_FUNC();
    return 1;
}

/*
 * After an I/O is complete, we need to free the in kernel buffer.
 */
void
bw_iov_dealloc(struct bw_ioreq *bwreq, unsigned long *flags)
{
    extern void vfree(void *);
    BW_ENTER_FUNC();
    NASTY_UNLOCK(*flags);
    vfree(bwreq->bw_allocbase);
    NASTY_LOCK(*flags);
    BW_EXIT_FUNC();
}
// LICENSE:
// This software is subject to the terms of the GNU GENERAL 
// PUBLIC LICENSE Version 2, June 1991
