/*
 * $Id: bw_trace.h,v 1.2 2001/05/30 15:00:19 erik Exp $
 * 
 * Copyright (C) 2001, Sun Microsystems, Inc.
 * All rights reserved
 *
 * Trace logging stuffs
 */

#ifndef __bw_trace_h__
#define __bw_trace_h__

#ifndef BW_TRACE_LEVEL
#define BW_TRACE_LEVEL 1
#endif


#if defined( DEBUG ) && BW_TRACE_LEVEL>0

#define BW_TRACE_LOG_LEN 256



struct bw_trace_point_log_t
{
    char name[32];
};

struct bw_lock_log_t
{
    char name[32];
};

#define BW_TRACE_TYPE_NULL  0
#define BW_TRACE_TYPE_POINT 1
#define BW_TRACE_TYPE_LOCK  2
#define BW_TRACE_TYPE_ENTRY 3
#define BW_TRACE_TYPE_EXIT  4

struct bw_trace_log_t
{
    unsigned int stamp;
    unsigned int processor;
    unsigned int pid;
    
    unsigned int type;
    
    int line;
    char file[32];

    union
    {
	struct bw_trace_point_log_t point;
	struct bw_lock_log_t lock;
    } data;
};

extern struct bw_trace_log_t bw_trace_log[BW_TRACE_LOG_LEN];
extern int bw_trace_log_pos;
extern unsigned int bw_trace_log_stamp;


static inline volatile void
bw_log_trace_point( char *name, char * file, int line, 
		    unsigned int type, int level )
{
    if( level > BW_TRACE_LEVEL )
	return;

    bw_trace_log[bw_trace_log_pos].type =  type;

    strncpy( bw_trace_log[bw_trace_log_pos].file, file, 31 );
    bw_trace_log[bw_trace_log_pos].file[31] = '\0';

    strncpy( bw_trace_log[bw_trace_log_pos].data.point.name, name, 31 );
    bw_trace_log[bw_trace_log_pos].data.point.name[31] = '\0';

    bw_trace_log[bw_trace_log_pos].processor =  current->processor;
    bw_trace_log[bw_trace_log_pos].pid =  current->pid;
    
    bw_trace_log[bw_trace_log_pos].line = line;
    bw_trace_log[bw_trace_log_pos].stamp = bw_trace_log_stamp++;
    bw_trace_log_pos++;
    bw_trace_log_pos %= BW_TRACE_LOG_LEN;
}


static inline volatile void
bw_log_lock(char *lock_name, char * file, int line, char prefix )
{
    bw_trace_log[bw_trace_log_pos].type =  BW_TRACE_TYPE_LOCK;

    bw_trace_log[bw_trace_log_pos].data.lock.name[0]=prefix;

    strncpy( bw_trace_log[bw_trace_log_pos].file, file, 31 );
    bw_trace_log[bw_trace_log_pos].file[31] = '\0';
    strncpy( bw_trace_log[bw_trace_log_pos].data.lock.name+1, lock_name, 30 );
    bw_trace_log[bw_trace_log_pos].data.lock.name[31] = '\0';

    bw_trace_log[bw_trace_log_pos].processor =  current->processor;
    bw_trace_log[bw_trace_log_pos].pid =  current->pid;
    
    bw_trace_log[bw_trace_log_pos].line = line;
    bw_trace_log[bw_trace_log_pos].stamp = bw_trace_log_stamp++;
    bw_trace_log_pos++;
    bw_trace_log_pos %= BW_TRACE_LOG_LEN;
}

static volatile void
bw_dump_trace_log( void )
{
    int i;
    for( i=0 ; i<BW_TRACE_LOG_LEN; i++ )
    {
	if( bw_trace_log[ i ].line )
	{
	    printk( "[%04x:%d:%d]%s:%d", 
		    bw_trace_log[ i ].stamp & 0xffff,
		    bw_trace_log[ i ].processor,
		    bw_trace_log[ i ].pid,
		    bw_trace_log[ i ].file,
		    bw_trace_log[ i ].line );

	    switch( bw_trace_log[ i ].type )
	    {
		case  BW_TRACE_TYPE_POINT:
		    printk( " <%s>\n", bw_trace_log[ i ].data.point.name );
		    break;

		case BW_TRACE_TYPE_LOCK:
		    printk( " <%c>%s\n", 
			    bw_trace_log[ i ].data.lock.name[0],
			    bw_trace_log[ i ].data.lock.name+1);
		    break;

		case BW_TRACE_TYPE_ENTRY:
		    printk( " > %s\n", bw_trace_log[ i ].data.point.name );
		    break;

		case BW_TRACE_TYPE_EXIT:
		    printk( " < %s\n", bw_trace_log[ i ].data.point.name );
		    break;
		    
	    }
	}
    }
    
}

static inline volatile void
bw_spin_on_lock( spinlock_t *lock, char * lock_name,
		 char * file, int line)
{
    int i;
    unsigned int stamp = bw_trace_log_stamp;

	/* spin for a bit to see if we are deadlocked-ish */
    for( i=0 ; i<0xffff; i++ )
    {
	if( spin_trylock( lock ) )
	    return;
/*	if( ! ((i+1)%50) )
	printk(".");*/
    }

    printk( "[%04x:%d:%d] lock %s spinning at: %s:%d\n", 
	    stamp & 0xffff, 
	    current->processor, current->pid,
	    lock_name, file, line);
    bw_dump_trace_log();
    while( !spin_trylock( lock ) );
}

#define BW_INSERT_TRACE(name, level)  \
     bw_log_trace_point(name, __FILE__, __LINE__, BW_TRACE_TYPE_POINT, level )
#define BW_DUMP_TRACE_LOG() bw_dump_trace_log()

#define BW_ENTER_FUNC()  \
     bw_log_trace_point(__FUNCTION__, __FILE__, __LINE__, BW_TRACE_TYPE_ENTRY, 2 )

#define BW_EXIT_FUNC()  \
     bw_log_trace_point(__FUNCTION__, __FILE__, __LINE__, BW_TRACE_TYPE_EXIT, 2 )

#define BW_LOCK_DEBUG_PRE(lock, flags, file, line) local_irq_save(flags); \
  if( ! spin_trylock( lock ) )\
     bw_spin_on_lock( lock, #lock, file, line) ;
#define BW_LOCK_DEBUG(lock, flags, file, line) 
#define BW_LOCK_DEBUG_POST(lock, flags, file, line) ; bw_log_lock( #lock, file, line,'L')

#define BW_UNLOCK_DEBUG_PRE(lock, flags, file, line) bw_log_lock(#lock, file, line,'U');
#define BW_UNLOCK_DEBUG_POST(lock, flags,  file, line)

#else /* DEBUG */
#define BW_INSERT_TRACE(name, level)
#define BW_DUMP_TRACE_LOG()

#define BW_ENTER_FUNC()
#define BW_EXIT_FUNC()

#define BW_LOCK_DEBUG_PRE(lock, flags, file, line)
#define BW_LOCK_DEBUG(lock, flags, file, line) spin_lock_irqsave( lock, flags )
#define BW_LOCK_DEBUG_POST(lock, flags, file, line)
#define BW_UNLOCK_DEBUG_PRE(lock, flags, file, line)
#define BW_UNLOCK_DEBUG_POST(lock, flags, file, line)



#endif /* DEBUG */

#define BW_LOG_LOCK_FULL0( lock, flags, file, line  ) do {\
                                   BW_LOCK_DEBUG_PRE(lock,flags, file,line)\
                                   BW_LOCK_DEBUG(lock, flags, file, line)\
                                   BW_LOCK_DEBUG_POST(lock,flags, file,line); } while(0)

#define BW_LOG_LOCK0( lock, flags ) BW_LOG_LOCK_FULL0( lock, flags, __FILE__, __LINE__ )

#define BW_LOG_UNLOCK_FULL0( lock, flags, file, line) do{\
                                   BW_UNLOCK_DEBUG_PRE(lock,flags, file,line)\
                                   spin_unlock_irqrestore( lock, flags )\
                                   BW_UNLOCK_DEBUG_POST(lock, flags, file, line);} while(0)

#define BW_LOG_UNLOCK0( lock, flags ) BW_LOG_UNLOCK_FULL0( lock, flags, __FILE__, __LINE__ )
                    

#ifdef BW_SKIP_LOCKS

#define BW_LOCK_FLAGS_DECL 
#define BW_LOCK_FLAGS      
#define BW_LOCK_FLAGS_REF  NULL


#define BW_LOG_LOCK_FULL( lock, flags, file, line  )
#define BW_LOG_LOCK( lock, flags ) 
#define BW_LOG_UNLOCK_FULL( lock, flags, file, line)
#define BW_LOG_UNLOCK( lock, flags ) 
#else

#define BW_LOCK_FLAGS_DECL unsigned long flags
#define BW_LOCK_FLAGS      flags
#define BW_LOCK_FLAGS_REF  &flags


#define BW_LOG_LOCK_FULL( lock, flags, file, line  ) \
      BW_LOG_LOCK_FULL0( lock, flags, file, line  )

#define BW_LOG_LOCK( lock, flags ) BW_LOG_LOCK0( lock, flags ) 
#define BW_LOG_UNLOCK_FULL( lock, flags, file, line)\
      BW_LOG_UNLOCK_FULL0( lock, flags, file, line)

#define BW_LOG_UNLOCK( lock, flags ) BW_LOG_UNLOCK0( lock, flags ) 
#endif

#endif /* __bw_trace_h__ */
// LICENSE:
// This software is subject to the terms of the GNU GENERAL 
// PUBLIC LICENSE Version 2, June 1991
