/*
 * bw_setip - User mode command to setthe bandwidth limit.
 *
 * Copyright (C) 1999, Cobalt Networks, Inc.
 * All rights reserved.
 *
 * Uses the ioctl interface defined in bw_user.h to set the 
 * IP bandwidth limit.  A limit can be attached to
 *	-i address/mask -p protocol/port -t r/w/rw
 * The address, protocol, and port are all looked up by the
 * apropriate getXbyY database routine, so the following
 * works if you don't feel like looking up numbers.
 *	-i some_host -p tcp/http -t r
 */

#include <sys/types.h>
#include <sys/socket.h>

#include <bw_user.h>

#include <stdio.h>
#include <errno.h>
#include <netinet/in.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <string.h>
#include <netdb.h>
#include <stdlib.h>

int bw_set_ip_limit(int fd, struct sockaddr *addr, int alen, int mask,
                short protocol, short port, enum bw_iotype rw, int bps);

int bw_delete_ip_limit(int fd, struct sockaddr *addr, int alen, int mask,
                short protocol, short port, enum bw_iotype rw);

int cvt_int(char *numstr, int *errp);

char *progname;

int
main(int ac, char **av)
{
    int fd, ret, c, err;
    struct hostent *hent = 0;
    struct servent *serv;
    struct protoent *protop;
    struct sockaddr saddr;
    char *protname;
    char *cp;
    int mask = 32;
    short proto = -1;
    short lim_proto = 0;
    short port = INADDR_ANY;
    int type = BW_WRIO;
    int bps=0, got_bps = 0;
    int delete = 0;

    if ((progname = strrchr(*av, '/')))
	progname++;
    else progname = *av;

    optind = 0;
    while ((c = getopt(ac, av, "hi:p:t:b:d")) != EOF) 
	switch (c) {
	case 'h':
	    fprintf(stderr, "%s: set an IP based bandwidth limit\n", progname);
	    fprintf(stderr, "usage: %s [-h] -i addr_spec -p proto_spec -t rw [-b bps] [-d]\n",
			progname);
	    fprintf(stderr, "\t-h: print this message\n");
	    fprintf(stderr, "\t-i addr_spec: limit host or address, optional trailing '/mask'\n");
	    fprintf(stderr, "\t-p proto_spec: 'protocol/port' specifier, port optional\n");
	    fprintf(stderr, "\t-t rw: read/write limit, 'r', 'w', or 'rw'\n");
	    fprintf(stderr, "\t-b bps: byte limit per second\n");
	    fprintf(stderr, "\t-d: delete an existing limit\n");
	    return 1;

	case 'd':
	    delete = 1;
	    break;

	case 'i':
	    /* 
	     * ipaddr/mask
	     */
	    if ((cp = strchr(optarg, '/'))) {
		*cp++ = 0;
		mask = cvt_int(cp, &err);

		if (err) {
		    fprintf(stderr, "%s: error converting mask %s\n", progname, optarg);
		    return 1;
		}
	    }

	    hent = gethostbyname(optarg);
	    if ( ! hent) {
		fprintf(stderr, "%s: error looking up host %s\n", progname, optarg);
		return 1;
	    }

	    memcpy(&saddr.sa_data, hent->h_addr, hent->h_length);
	    saddr.sa_family = hent->h_addrtype;
	    break;

	case 'p':
	    /*
	     * protocol/port
	     */
	    if ((cp = strchr(optarg, '/')))
		*cp++ = 0;

	    if ((protop = getprotobyname(optarg))) {
		lim_proto = protop->p_proto;
		protname = optarg;
	    }
	    else {
		lim_proto = cvt_int(optarg, &err);

		if ( ! err && (protop = getprotobynumber(proto))) {
		    protname = strdup(protop->p_name);
		}
		else if (lim_proto == 0) {
		    /*
		     * If the protocol is wildcarded, the service better
		     *     1) have the same port on all protocols,
		     *     2) be specified by number
		     *     3) not be specified.
		     * We use the string "tcp" in the hopes that one
		     * of the above is true.
		     */
		    protname = "tcp";
		}
		else {
		    fprintf(stderr, "%s: error converting protocol %s\n",
				progname, optarg);
		    return 1;
		}
	    }

	    if (cp) {
		if ((serv = getservbyname(cp, protname))) {
		    port = serv->s_port;
		}
		else {
		    port = cvt_int(cp, &err);

		    if (err) {
			fprintf(stderr, "%s: error converting port %s\n",
					progname, optarg);
			return 1;
		    }
		}
	    }
	    break;
	case 't':
	    /*
	     * type : read/write/both
	     */
	    if ( ! strcmp(optarg, "r"))
		type = BW_RDIO;
	    else if ( ! strcmp(optarg, "w"))
		type = BW_WRIO;
	    else if ( ! strcmp(optarg, "rw") || ! strcmp(optarg, "wr"))
		type = BW_RDWRIO;
	    else {
		fprintf(stderr, "%s: unknown read/write type %s\n", progname, optarg);
		return 1;
	    }
	    break;
	case 'b':
	    bps = cvt_int(optarg, &err);
	    got_bps = 1;

	    if (err) {
		fprintf(stderr, "%s: error converting bps %s\n", progname, optarg);
		return 1;
	    }
	    break;
	}

    if ( ! hent) {
	fprintf(stderr, "bind: must set ip address\n");
	return 1;
    }

    if (delete && got_bps) {
	fprintf(stderr, "%s: can not delete and set in the same operation\n",
		progname);
	return 1;
    }

    if ((protop = getprotobyname("tcp"))) {
	proto = protop->p_proto;
    }
    else {
	fprintf(stderr, "%s: error looking up tcp\n", progname);
	return 1;
    }

    if ((fd = socket(AF_INET, SOCK_STREAM, proto)) < 0) {
	perror("socket");
	fprintf(stderr, "%s cannot open control fd\n", progname);
	return 1;
    }


    lim_proto = htons(lim_proto);
    port = htons(port);

    if (delete) {
	ret = bw_delete_ip_limit(fd, &saddr, hent->h_length, mask, lim_proto,
			port, type);
	if (ret < 0 && errno != ENOENT) {
	    perror("delete_ip_limit");
	    fprintf(stderr, "%s: cannot delete IP limit\n", progname);
	    return 1;
	}
    }

    if (got_bps) {
	ret = bw_set_ip_limit(fd, &saddr, hent->h_length, mask, lim_proto,
		port, type, bps);
	if (ret < 0) {
	    perror("set_ip_limit");
	    fprintf(stderr, "%s: cannot set IP limit\n", progname);
	    return 1;
	}
    }

    close(fd);

    return 0;
}


int
bw_set_ip_limit(int fd, struct sockaddr *addr, int alen, int mask,
                short protocol, short port, enum bw_iotype rw, int bps)
{   
    struct bwlim_params bwp;

    bwp.bwp_bwid.bwid_type = BW_IP;
    bwp.bwp_bwid.bwid_rw = rw;
    bwp.bwp_bwid.bwid_addr = * (u_int32_t *) addr->sa_data;
    bwp.bwp_bwid.bwid_protocol = protocol;
    bwp.bwp_bwid.bwid_mask = mask;
    bwp.bwp_bwid.bwid_port = port;

    bwp.bwp_flags = BWPARAM_SETBPS;
    bwp.bwp_bps = bps;

    return ioctl(fd, BW_IOC_SETPARAM, (unsigned long int) &bwp);
}

int
bw_delete_ip_limit(int fd, struct sockaddr *addr, int alen, int mask,
                short protocol, short port, enum bw_iotype rw)
{   
    struct bwlim_params bwp;

    bwp.bwp_bwid.bwid_type = BW_IP;
    bwp.bwp_bwid.bwid_rw = rw;
    bwp.bwp_bwid.bwid_addr = * (u_int32_t *) addr->sa_data;
    bwp.bwp_bwid.bwid_protocol = protocol;
    bwp.bwp_bwid.bwid_mask = mask;
    bwp.bwp_bwid.bwid_port = port;

    bwp.bwp_flags = BWPARAM_DELETE;

    return ioctl(fd, BW_IOC_DELETE, (unsigned long int) &bwp);
}

int         
cvt_int(char *numstr, int *errp)
{
    long val;       
    char *cvt;  

    val = strtol(numstr, &cvt, 0);
    if (cvt == numstr) {
	*errp = 1;
	return 0;
    }

    switch (*cvt) {
    case 'm': case 'M':
	val *= 1024 * 1024;
	break;  
    case 'k': case 'K':
	val *= 1024;
	break;  
    }

    *errp = 0;      
    return val;
}      


// LICENSE:
// This software is subject to the terms of the GNU GENERAL 
// PUBLIC LICENSE Version 2, June 1991
