# Bluelinq::Sausalito.pm
# Copyright 2002, Sun Microsystems, Inc
# $Id: Sausalito.pm,v 1.3 2003/02/04 22:07:14 ssmith Exp $

package Bluelinq::Sausalito;
use base qw/Bluelinq/;
use Bluelinq;
use Bluelinq::Sausalito::Meta;
use strict;
use vars qw/$AUTOLOAD $OPTSTR/;
use Carp;
use File::Basename;
use Cwd;

$OPTSTR = {
	'='	=> 'eq',
	'!'	=> 'ne',
	'!='	=> 'ne',
	'>'	=> 'gt',
	'>='	=> 'ge',
	'<'	=> 'lt',
	'<='	=> 'le',
};

my %Data = (
		filename	=>	'',
		name		=>	undef,	# file basename
		locales		=>	['en'],
		templocation	=>	undef,	# unpacked location
		md5sum		=>	undef,
		meta		=>	undef,	# Bluelinq::Sausalito::Meta
		log		=>	[],	# anon array of strings
);

sub new {
	my $self = shift;
	my $class = ref($self) || $self;
	my $this = { %Data, @_ };
	bless $this, $class;
	#$this->init();
	return $this;
}

sub AUTOLOAD {
	my $self = shift;
	# only handle instance methods, not class methods.
	croak ("autoload $AUTOLOAD: $self not an object") unless ref($self);
	my $name = $AUTOLOAD;
	return if $name =~ /::DESTROY$/;
	$name =~ s/.*://;       # Field names not fully qualified, strip it.
	unless (exists $self->{$name}) {
		croak ("Can't access '$name' field in $self $AUTOLOAD");
	}
	if (@_) { return $self->{$name} = shift }
	else    { return $self->{$name} }
}

sub init {
	my $self = shift;
	croak ("$self not an object") unless ref($self);
	$self->blqerror("File doesn't exist") unless (-e $self->filename());
	# set file md5sum
	$self->md5sum(Bluelinq::md5sum_file($self->filename()));
	$self->name(basename($self->filename()));	# set file name
}

# DESTROY()
# removes the tempdir
sub DESTROY {
	my($self);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	return unless $self->templocation();
	# remove templocation
	if (-d $self->templocation()) {
		$self->blqinfo("Removing tempdir: ", $self->templocation());
		system($Bluelinq::RM_BIN, "-rf", $self->templocation());
		$self->blqerror("Couldn't remove temp dir: $!") if $?;
	}
}

# unpack()
#  - unpacks (gpg/ungzip/bzip2/tars) $self->filename into
#    $self->templocation()
#  - calls _do_unpack
# takes: none
# returns: none
sub unpack {
	my $self = shift;
	croak ("$self not an object") unless ref($self);
	# get a tempdir
	$self->blqinfo("unpack starting: " . $self->filename());
	# make tempdir
	if (!($self->templocation())) {
		$self->templocation($self->get_temp_dir());
		# make tempdir
		mkdir $self->templocation(), 0744
			or $self->blqerror("Couldn't make directory");
	}
	$self->blqinfo("$Bluelinq::GUNZIP_BIN -c " . $self->filename() .
					" | $Bluelinq::TAR_BIN -xf -");
	$self->blqerror("Sausalito package unpack failed!")
		unless &_do_unpack($self->filename(),$self->templocation());
	$self->blqinfo("unpack finished: " . $self->filename());
}

# _do_unpack()
# does bulk of the unpack
# takes: file to unpack, destination of contents
# returns: location of unpacked pkg on success, 0 on failure
sub _do_unpack {
	my($file,$dest,$basename,$fm,$cwd,$uncompress,$signed);
	($file,$dest) = (@_)[0,1];
	Bluelinq::blqlog(0,\"No such file: $file") unless (-e $file);
	Bluelinq::blqlog(0,\"No such directory: $dest") unless (-d $dest);
	$signed = 0;

	UNPACK: {
	last UNPACK if (-e "$dest/.UNPACKED");
	$basename = basename($file);
	$cwd = getcwd();
	chdir $dest or Bluelinq::blqlog(0,\"Couldn't chdir $dest: $!");
	chomp($fm = `$Bluelinq::FILE_BIN $file`);	# file magic.
	# gpg signed? note: we don't verify anything here
	# note: destructor will clean up for us
	if ( ($fm =~ /PGP\s+armored\s+data\s+signed\s+message/)
			or ($fm =~ /data$/) ) {
		`$Bluelinq::GPG_BIN --quiet --batch --no-tty --decrypt --output $basename $file > /dev/null 2>&1`;
		Bluelinq::blqlog(0,\"Failed to gpg decrypt: $!") if $?;
		# source file is now in tempdir ($dest)
		$file = $basename; ++$signed;
	}
	chomp($fm = `$Bluelinq::FILE_BIN $file`);	# file magic.
	if ($fm =~ /(\w+)\s+compressed/) {		# compressed
		if ($1 =~ /bzip2/) {
			$uncompress = $Bluelinq::BZIP2_BIN;
		} else {
			$uncompress = $Bluelinq::GUNZIP_BIN;
		}
		system("$uncompress -q -c $file | $Bluelinq::TAR_BIN -xf -");
		Bluelinq::blqlog(0,\"Failed to uncompress file: $!") if $?;
	} elsif ($fm =~ /tar/) {			# tar file
		system("$Bluelinq::TAR_BIN -xf $file");
		Bluelinq::blqlog(0,\"Failed to untar file: $!") if $?;
	} else {					# who knows, bail
		Bluelinq::blqlog(0,\"Unrecognized file type:\n$fm");
	}
	if ($signed) {		# cleanup temp file
		unlink($file)
			or Bluelinq::blqlog(0,\"Failed to rm tempfile: $!");
	}
	&_set_unpacked_flag($dest);
	chdir $cwd	or Bluelinq::blqlog(0,\"Couldn't chdir $cwd");
	};	# end UNPACK block
	if (!(_check_valid_structure($dest))) {
		Bluelinq::blqlog(2,\"Not a sausalito package!");
		return 0;
	}
	return $dest;
}

# _check_valid_structure()
# - does crude validity checking of an _unpacked_ package structure
# takes: dirname where a package is supposedly unpacked
# returns: 1 if it appears valid, or 0 otherwise
sub _check_valid_structure {
	my($loc);
	$loc = shift
		or Bluelinq::blqlog(0,\"_check_valid_structure: No args given");
	Bluelinq::blqlog(0,\"_check_valid_structure: Not a dir!")
		unless (-d $loc);
	if ((-e "$loc/packing_list") && (-d "$loc/pkginfo/locale")) { return 1 }
	return 0;
}

# reads $self->templocation/packing_list
# extracts information into $self->meta
# calls 
sub get_meta_info {
	my($self);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	# parse the packing list
	$self->read_packing_list();
}

# dumps meta infor in plist format to STDOUT
sub dumpmeta {
	my($self,$fh);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$self->meta()->writemeta(\*STDOUT);
}

sub dumpRPMS {
	my($self);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	map { print STDOUT "$_\n" } @{$self->meta->RPMList()};
}

sub dumpSRPMS {
	my($self);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	map { print STDOUT "$_\n" } @{$self->meta->SRPMList()};
}

# parse packing list, setting values in self->meta()
# maintain compatibility with bluelinq utilities and...
# cvsraq:base-swupdate.mod/glue/sbin/SWUpdate.pm:read_pkgheader()
# cvsraq:bigdaddy/swmgmt/lib/perl5/Swmgmt.pm::read_pkgheader()
sub read_packing_list {
	my($self,$plistfile,$plist);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$plistfile = $self->templocation() . '/' . 'packing_list';
	if (!(-e $plistfile)) { $self->blqerror("Packing list doesn't exist!") }
	if (!(-r $plistfile)) { $self->blqerror("Cannot read packing list!") }
	open(PLIST,"<$plistfile")	
			or $self->blqerror("Cannot open packing list: $!");
	$_ = readline \*PLIST;
	seek PLIST, 0, 0;		# reset file pointer
	# match <?xml version="1.0" or <?xml version='1.0'
	# valid xml docs must have <?xml version="1.0?> on first line
	if (m/$<\?xml\s+version=[\'\"]1.0[\'\"]/i) {
		$self->blqerror("I don't do xml packing lists yet.");
	} else {
		use Bluelinq::Sausalito::Meta::plist;
		$plist = Bluelinq::Sausalito::Meta::plist->new();
		$plist->readmeta(\*PLIST);
		$self->meta($plist);		# store away our plist object
		$self->Size(-s $self->filename());
	}
	close(PLIST) or $self->blqerror("Cannot close packing list: $!");
	# query for SRPMS, add to self->meta->SRPMList
	my($rpm,$srpm,$dir,%shash);
	$dir = $self->templocation();
	foreach $rpm (@{$self->meta->RPMList()}) {
		if (-e "$dir/RPMS/$rpm") {
			$srpm = `/bin/rpm -q --queryformat '%{SOURCERPM}' -p $dir/RPMS/$rpm`;
			if (! ($srpm =~ m/^\(none\)/) ) {
				$shash{$srpm} = 1;
			}
		}
	}
	push @{$self->meta()->SRPMList()}, (keys %shash);
}

# returns: original_val => {	'vendor'	=> 'Cobalt',
#		'name'		=> 'OS',
#		'version'	=> '6.4',
#		'op'		=> '{eq,ne,lt,le,gt,ge}'}
# takes: string representing a token based, ses also global $OPTSTR
# from glazed:base-swupdate.mod/doc/package_list_spec:
#  vendor:package		vendor-package must exist
#  vendor:package !		vendor-package must not exist
#  vendor:package <=> version	vendor-package lt,eq,gt version
#  vendor:package != version	vendor-package ne version
sub _normalize_token_item {
	my($str,$ref,$ven,$name,$op,$ver);
	$str = shift or Bluelinq::blqlog(0,\"You must pass a token item!");
	$ref = {
		'vendor'	=> 'unknown',
		'name'		=> 'unknown',
		'version'	=> undef,
		'op'		=> undef,
		'orig'		=> $str,
	};
	if($str =~ /^\s*(\S+):(\S+)\s*(?:([!>=<]*)\s*(\S+))*/i) {
		( $ref->{'vendor'},$ref->{'name'},
		  $ref->{'op'},$ref->{'version'} ) = ($1,$2,$3,$4);
		# normalize operator to its string equivalent
		if (defined($ref->{'op'})) {
			$ref->{'op'} = $OPTSTR->{$ref->{'op'}}
		}
	} else {
		Bluelinq::blqlog(1,\"Unrecognized token item: $str");
	}
	return $ref;
}

# writes the sausalito-style package style directory of meta-information
# takes: a directory to create metadir in
# warning: will overwrite any previous existing metadir
sub write_metadir {
	my($self,$destdir,$metadir,$temploc,$cwd);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$destdir = shift or $self->blqerror("You must specify a destination!");
	$self->blqerror("No such directory: $destdir") unless (-d $destdir);
	$temploc = $self->templocation();
	# style: [VENDOR]-[NAME]-[VERSION]
	$metadir =	$destdir . '/' .
			$self->meta()->{'vendor'} . '-' .
			$self->meta()->{'name'} . '-' .
			$self->meta()->{'version'};
	if (-d $metadir) {
		$self->blqinfo("$metadir already existed in $destdir!");
		system($Bluelinq::RM_BIN, "-rf", $metadir);
		$self->blqerror("Couldn't remove temp dir: $!") if $?;
	}
	# make destination directory
	mkdir($metadir,0770) or $self->blqerror("Failed to make metadir: $!");
	$self->blqerror("package's locale directory doesn't exist!")
				unless (-d "$temploc/pkginfo/locale");
	$cwd = getcwd();
	chdir "$temploc/pkginfo"
	  or $self->blqerror("Couldn't chdir $temploc/pkginfo: $!");
	$self->blqinfo("$Bluelinq::TAR_BIN -cf - locale | (cd $metadir && $Bluelinq::TAR_BIN -xf -)");
	# use tar to copy
	system("$Bluelinq::TAR_BIN -cf - locale | (cd $metadir && $Bluelinq::TAR_BIN -xf -)");
	$self->blqerror("Failed to copy locale dir: $!") if $?;
	chdir $cwd or $self->blqerror("Couldn't chdir $cwd: $!");
}

# location()
# location field accessor
sub Location {
	my($self,$loc);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	if (@_)	{ return $self->meta()->{'location'} = shift }
	else	{ return $self->meta()->{'location'} }
}

# vendor()
# vendor field accessor
sub Vendor {
	my($self,$loc);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	return;		# do not set sausalito vendors, they are in pkg
#	if (@_)	{ return $self->meta()->{'vendor'} = shift }
#	else	{ return $self->meta()->{'vendor'} }
}

# size()
# size field accessor
sub Size {
	my($self,$loc);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	if (@_)	{ return $self->meta()->{'size'} = shift }
	else	{ return $self->meta()->{'size'} }
}

# touches $dir/.UNPACKED to alert package unpacking routines not re-unpack it
sub _set_unpacked_flag {
	my($dir) = shift
		or Bluelinq::blqlog(0,"Specifiy a directory to set flag");
	open(FLAG,"> $dir/.UNPACKED")
		or Bluelinq::blqlog(0,"Could not open flag file: $!");
	print FLAG scalar localtime;
	close(FLAG) or Bluelinq::blqlog(0,"Could not close flag file: $!");
}

1;
# Copyright (c) 2003 Sun Microsystems, Inc. All  Rights Reserved.
# 
# Redistribution and use in source and binary forms, with or without 
# modification, are permitted provided that the following conditions are met:
# 
# -Redistribution of source code must retain the above copyright notice, 
# this list of conditions and the following disclaimer.
# 
# -Redistribution in binary form must reproduce the above copyright notice, 
# this list of conditions and the following disclaimer in the documentation  
# and/or other materials provided with the distribution.
# 
# Neither the name of Sun Microsystems, Inc. or the names of contributors may 
# be used to endorse or promote products derived from this software without 
# specific prior written permission.
# 
# This software is provided "AS IS," without a warranty of any kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
# 
# You acknowledge that  this software is not designed or intended for use in the design, construction, operation or maintenance of any nuclear facility.
