# Bluelinq::Sauce::Meta::plist.pm
# Copyright 2002, Sun Microsystems, Inc
# $Id: plist.pm,v 1.7 2003/02/04 22:07:14 ssmith Exp $

package Bluelinq::Sauce::Meta::plist;
use base qw/Bluelinq::Sauce::Meta/;
use strict;
use Carp;

# keep in sync with SUPER::Data
my %Data = (
		package		=>	undef,
		version		=>	undef,
		listfile	=>	'',
		reboot		=>	'no',
		uninstall	=>	'no',
		url		=>	'',
		rpms		=>	[],
		srpms		=>	[],
		scripts		=>	[],
		products	=>	[],
		dependencyList	=>	[],
		obsoletesList	=>	[],
		vendor		=>	'unknown',
		filesize	=>	undef,
		location	=>	'unknown',
		log		=>	[],		# anon array of strings
);

sub new {
	my $self = shift;
	my $class = ref($self) || $self;
	my $this = { %Data, @_ };
	bless $this, $class;
	return $this;
}

sub DESTROY { }

# takes an open filehandle
# returns a ::plist object
sub readmeta {
	my($self,$fh,$inpackage,$key,$hash,$k,$v);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$fh = shift or $self->blqerror("parse: You must pass a filehandle!");
	while(<$fh>) {
		next if m/^\s*#/;		# skip comments
		next unless /\S/;		# skip blank lines
		if (/^Package:\s*(\S+)/i) {
			$self->{"package"} = $1; next;
		}
		if (/^List File:\s*(\S+)/i) {
			$self->{"listfile"} = $1; next;
		}
		if (/^Version:\s*(\S+)/i) {
			$self->{"version"} = $1; next;
		}
		if (/^URL:\s*(\S+)/i) {
			$self->{"url"} = $1; next;
		}
		if (/REBOOT:\s*(\S+)/i) {
			$self->{"reboot"} = $1; next;
		}
		if (/UNINSTALL:\s*(\S+)/i) {
			$self->{"uninstall"} = $1; next;
		}
		# push all rpms onto array
		if (/^RPM:\s*(\S+)/i) {
			push(@{$self->{"rpms"}},$1); next;
		}
		# push all scripts onto array
		if (/^SCRIPT:\s*(\S+)/i) {
			push(@{$self->{"scripts"}},$1); next;
		}
		# warn about unknown lines now
		if (/^(\S+)$/) {
			$self->blqinfo("Unknown packing_list content: $1");
		}
	}
	# this sucks, special case for casp_interbase_patch_raq3.pkg
	if ($self->package() eq 'Chili!Soft') {
		$self->{"package"} = "ChiliSoft-ASP";
	}
	# guess/discover the products this is for
	$self->set_products();
	return 1;
}

# writes plist to a filehandle
# takes: an open filehandle
sub writemeta {
	my($self,$fh,$val);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$fh = shift or $self->blqerror("You must pass a filehandle");
	print $fh '[Package -- Version=1.0]', "\n";
	$self->_write_plist_line($fh,"Name",
				$self->package() . '-' . $self->version());
	$self->_write_plist_line($fh,"Location", $self->location());
	$self->_write_plist_line($fh,"Size",$self->filesize());
	# print all products
	foreach $val (@{$self->products()}) {
		$self->_write_plist_line($fh,"Product",$val);
	}
	$self->_write_plist_line($fh,"Version",$self->version());
	$self->_write_plist_line($fh,"Vendor",$self->vendor());
	if ($self->reboot() =~ /yes/i) {
		$self->_write_plist_line($fh,"Options","reboot");
	}
	$self->_write_dependency_array($fh,"Depend",$self->dependencyList());
	$self->_write_dependency_array($fh,"Depend",$self->obsoletesList());
	print $fh '[/Package]', "\n";
}

sub _write_dependency_array {
	my($self,$fh,$label,$list,$ref,$line);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	($fh,$label,$ref) = (@_)[0,1,2];
	foreach $ref (@{$ref}) {
		$line = "$ref->{'vendor'}:$ref->{'name'}";
		$line .= ($ref->{"op"}) ? " $ref->{'op'}" : "";
		$line .= ($ref->{"version"}) ? " $ref->{'version'}" : "";
		$self->_write_plist_line($fh,$label,$line);
	}
}

# takes: references to a filehandle, element label, and array of elements,
# returns: none
sub _write_plist_array {
	my($self,$fh,$label,$ref,$i);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$fh	= shift or $self->blqerror("Must pass a filehandle!");
	$label	= shift or $self->blqerror("Must pass a label!");
	$ref	= shift or $self->blqerror("Must pass a ref to print!");
	foreach $i (@{$ref}) { $self->_write_plist_line($fh,${$label},$i) }
}

sub _write_plist_line {
	my($self,$fh,$key,$value);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$fh = shift or $self->blqerror("Must pass a filehandle!");
	(($key,$value) = (@_)[0,1]);
	return unless $value;
	$key .= ':';
	# 24 chars for key, fill rest of line with value
	format plist_line =
@<<<<<<<<<<<<<<<<<<<<<<<@*
$key,                   $value
.
	$~ = "plist_line";
	write();
}

# sets self->products();
# this is a cheesy hueristic based on filename 
# uses common definitions with control station
sub set_products {
	my($self,%namemap,$nametag);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$nametag = $self->{"package"};	# using meta package
	# snarf out [TOKEN]-[TOKEN]
	# turn "RaQ4-en-Security-1.0.1.pkg" to "raq4-en" in next two stmts
	#       /^(\w+)-(\w+)-(\S*)$/
	$nametag =~ s/^(\w+)\-(\w+)\-(\S*)$/$1-$2/;
	$nametag = lc($nametag);			# lowercase!
	$self->blqinfo("package token is $nametag");
	%namemap = (
		# x86
		'raq4-all' => ['3001R','3100R','3001R-ja','3100R-ja','3001R-IC'],
		'raq4-en' => ['3001R','3100R'],
		'raq4-ja' => ['3001R-ja','3100R-ja'],
		'raq3-all' => ['3000R','3000R-ja'],
		'raq3-en' => ['3000R'],
		'raq3-ja' => ['3000R-ja'],
		# x86 misc
		'chilisoft-asp' => ['3000R','3000R-ja'],
		# mips
		'raq2-all' => ['2800R','2800RJ','2800RS'],
		'raq2-en' => ['2800R'],
		'raq2-ja' => ['2800RJ'],
		'qube2-all' =>
			['2800WG','2800WGJ','2800WGF','2800WGG','2800WGZHT'],
		'qube2-en' => ['2800WG'],
		'qube2-ja' => ['2800WGJ'],
		'qube2-de' => ['2800WGG'],
		'qube2-fr' => ['2800WGF'],
		'qube2-zh' => ['2800WGZHT'],
		'cacheraq4-all' => ['3100CR'],
		'cacheraq2-all' => ['2800CR'],
		'cacheraq1' => ['2700CR'],
		'cacheraq' => ['2700CR'],
		'cacheqube' => ['2700C'],
		'manageraq-en' => ['3000R-mr'],
		'raq3-mr' => ['3000R-mr'],
		'all-kernel_mips' => [	'2800R','2800RJ','2800RS', '2800WG',
					'2800WGJ','2800WGF','2800WGG',
					'2800WGZHT','3100CR','2800CR',
					'2700CR','2700C'],
		# mips misc
		'cacheraq2-update' => ['2800CR'],
		'qube2f-update' => ['2800WGF'],
		'qube2-update' =>
			['2800WG','2800WGF','2800WGG','2800WGZHT'],
		'qube2j-update' => ['2800WGJ'],
		'qube2-jazh' => ['2800WGJ','2800WGZHT'],
		'raq2-update' => ['2800R'],
		'raq2j-update' => ['2800RJ'],
		'all-kernel_i386' => ['3000R','3000R-ja'],
	);
	if (exists $namemap{$nametag}) {
		$self->products(\@{$namemap{$nametag}});
		foreach (@{$self->products()}) {
			$self->blqinfo("\tapplies to: $_");
		}
	} else {
		$self->blqerror("$nametag didn't match a known product string");
	}
}

1; # end of Bluelinq::Sauce::Meta:;plist
# Copyright (c) 2003 Sun Microsystems, Inc. All  Rights Reserved.
# 
# Redistribution and use in source and binary forms, with or without 
# modification, are permitted provided that the following conditions are met:
# 
# -Redistribution of source code must retain the above copyright notice, 
# this list of conditions and the following disclaimer.
# 
# -Redistribution in binary form must reproduce the above copyright notice, 
# this list of conditions and the following disclaimer in the documentation  
# and/or other materials provided with the distribution.
# 
# Neither the name of Sun Microsystems, Inc. or the names of contributors may 
# be used to endorse or promote products derived from this software without 
# specific prior written permission.
# 
# This software is provided "AS IS," without a warranty of any kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
# 
# You acknowledge that  this software is not designed or intended for use in the design, construction, operation or maintenance of any nuclear facility.
