# Bluelinq::Sauce.pm
# Copyright 2002, Sun Microsystems, Inc
# $Id: Sauce.pm,v 1.4 2003/02/04 22:07:14 ssmith Exp $

package Bluelinq::Sauce;
use base qw/Bluelinq/;
use Bluelinq;
use strict;
use vars qw/$AUTOLOAD/;
use Carp; 
use Cwd;
use File::Basename;

my %Data = (
		filename	=>	'',
		name		=>	undef,	# basename
		filedate	=>	undef,	# unix timestamp
		locales		=>	['en'],
		templocation	=>	undef,	# unpacked location
		md5sum		=>	undef,
		plist		=>	undef,	# Bluelinq::Sauce::plist
		xml		=>	undef,	# Bluelinq::Sauce::xml
		log		=>	[],	# anon array of strings
);

sub new {
	my $self = shift;
	my $class = ref($self) || $self;
	my $this = { %Data, @_ };
	bless $this, $class;
	#$this->init();
	return $this;
}

sub init {
	my $self = shift;
	croak ("$self not an object") unless ref($self);
	$self->filedate((stat($self->filename()))[9]);	# set file mtime
}

sub AUTOLOAD {
	my $self = shift;
	# only handle instance methods, not class methods.
	croak ("autoload $AUTOLOAD: $self not an object") unless ref($self);
	my $name = $AUTOLOAD;
	return if $name =~ /::DESTROY$/;
	$name =~ s/.*://;       # Field names not fully qualified, strip it.
	unless (exists $self->{$name}) {
		croak ("Sauce: Can't access '$name' field in $self");
	}
	if (@_) { return $self->{$name} = shift }
	else    { return $self->{$name} }
}

# clean up
sub DESTROY {
	my($self,$tempdir);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$tempdir = $self->templocation();
	return unless $tempdir;
	# remove templocation
	if (-d $tempdir) {
		$self->blqinfo("Removing tempdir: ", $tempdir);
		system($Bluelinq::RM_BIN, "-rf", $tempdir);
		$self->blqerror("Couldn't remove temp dir: $!") if $?;
	}
}

# writes plist to STDOUT
sub dumpmeta {
	my($self);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$self->plist()->writemeta(\*STDOUT);
}

sub dumpRPMS {
	my($self);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	map { print STDOUT "$_\n" } @{$self->plist->rpms()};
}

sub dumpSRPMS {
	my($self);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	map { print STDOUT "$_\n" } @{$self->plist->srpms()};
}

# unpack()
#  unpacks (ungzip/tars) $self->filename into $self->templocation()
# takes and returns: none
sub unpack {
	my $self = shift;
	croak ("$self not an object") unless ref($self);
	# get a tempdir
	$self->blqinfo("unpack starting: " . $self->filename());
	if (!($self->templocation())) {
		$self->templocation($self->get_temp_dir());
		# make tempdir
		mkdir $self->templocation(), 0744
			or $self->blqerror("Couldn't make directory");
	}
	$self->blqinfo("$Bluelinq::GUNZIP_BIN -c " . $self->filename() .
					" | $Bluelinq::TAR_BIN -xf -");
	$self->blqerror("Sauce package unpack failed!")
		unless &_do_unpack($self->filename(),$self->templocation());
	$self->blqinfo("unpack finished: " . $self->filename());
}

# _do_unpack()
# does bulk of the unpack
# takes: file to unpack, destination of contents
# returns: location of unpacked pkg on success, 0 on failure
# gotchas: touches $dest/.UNPACKED after it has unpacked a package
sub _do_unpack {
	my($file,$dest,$fm,$cwd);
	($file,$dest) = (@_)[0,1];
	Bluelinq::blqlog(0,\"No such file: $file") unless (-e $file);
	Bluelinq::blqlog(0,\"No such directory: $dest") unless (-d $dest);
	# check file magic, sauce pkgs only know about gzip
	chomp($fm = `$Bluelinq::FILE_BIN $file`);
	if ( !($fm =~ /gzip/) ) { return 0 }
	UNPACK: {
	last UNPACK if (-e "$dest/.UNPACKED");
	$cwd = getcwd();
	chdir $dest or Bluelinq::blqlog(0,\"Couldn't chdir $dest: $!");
	# portably pipe gunzip to tar using filehandles
	#  solaris tar does not support gunzip or chdir'ing for us
	open(PKGFILE,"$Bluelinq::GUNZIP_BIN --stdout $file |")
			or Bluelinq::blqlog(0,\"Can't fork gunzip: $!");
	open(TARCMD,"| $Bluelinq::TAR_BIN -xf -")
			or Bluelinq::blqlog(0,\"Can't fork tar: $!");
	while(<PKGFILE>) {
		print TARCMD $_;
	}
	close(PKGFILE)	or Bluelinq::blqlog(0,\"Can't close gunzip pipe");
	close(TARCMD)	or Bluelinq::blqlog(0,\"Can't close tar pipe");
	_set_unpacked_flag($dest);
	chdir $cwd	or Bluelinq::blqlog(0,\"Couldn't chdir $cwd");
	};	# end UNPACK block
	# does it look valid, or...
	# is it casp_interbase_patch_raq3.pkg, whose structure is broken
	if ( (basename($file) ne 'casp_interbase_patch_raq3.pkg') &&
			!(_check_valid_structure($dest))) {
		Bluelinq::blqlog(2,\"Not a sauce package!");
		return 0;
	}
	return $dest;
}

# _check_valid_structure()
# - does crude validity checking of an _unpacked_ package structure
# takes: dirname where a package is supposedly unpacked
# returns: 1 if it appears valid, or 0 otherwise
sub _check_valid_structure {
	my($loc);
	$loc = shift
		or Bluelinq::blqlog(0,\"_check_valid_structure: No args given");
	Bluelinq::blqlog(0,\"_check_valid_structure: Not a dir!")
		unless (-d $loc);
	if ( (-e "$loc/packing_list") && (-e "$loc/upgrade_me") ) { return 1 }
	return 0;
}

# reads $self->templocation/packing_list
# extracts information into $self->plist()
# calls 
sub get_meta_info {
	my($self);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	# parse the packing list
	$self->read_packing_list();
}


# parse packing list, setting values in self->[plist,xml]
# maintain compatibility with bluelinq utilities and...
# glazed:upgrade/usr/local/sbin/cobalt_upgrade
# cvsraq:bigdaddy/swmgmt/lib/perl5/Swmgmt.pm::get_sauce_meta()
sub read_packing_list {
	my($self,$packlist,$plist);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$packlist = $self->templocation() . '/' . 'packing_list';
	if (!(-e $packlist)) { $self->blqerror("Packing list doesn't exist!") }
	if (!(-r $packlist)) { $self->blqerror("Cannot read packing list!") }
	open(PLIST,"<$packlist")	
			or $self->blqerror("Cannot open packing list: $!");
	# parse whole packing_list into $self->[plist]()
	$_ = readline \*PLIST;		# read first line
	seek PLIST, 0, 0;		# reset file pointer
	# match <?xml version="1.0" or <?xml version='1.0'
	# valid xml docs must have <?xml version="1.0?> on first line
	if (m/$<\?xml\s+version=[\'\"]1.0[\'\"]/i) {
		$self->blqerror("I don't do xml packing lists yet.");
	} else {
		use Bluelinq::Sauce::Meta::plist;
		$plist = Bluelinq::Sauce::Meta::plist->new();
		$plist->readmeta(\*PLIST);
		$plist->filesize(-s $self->filename());	# set file size
		$self->plist($plist);		# store away our plist object
	}
	close(PLIST) or $self->blqerror("Cannot close packing list: $!");
	my($rpm,$srpm,$dir,%shash);
	$dir = $self->templocation();
	foreach $rpm (@{$self->plist->rpms()}) {
		if (-e "$dir/RPMS/$rpm") {
			$srpm = `/bin/rpm -q --queryformat '%{SOURCERPM}' -p $dir/RPMS/$rpm`;
			if (! ($srpm =~ m/^\(none\)/) ) {
				$shash{$srpm} = 1;
			}
		}
	}
	push @{$self->plist()->srpms()}, (keys %shash);
}

# location()
# location field accessor
sub Location {
	my($self,$loc);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	if (@_)	{ return $self->plist()->{'location'} = shift }
	else	{ return $self->plist()->{'location'} }
}

# vendor()
# vendor field accessor
sub Vendor {
	my($self,$loc);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	if (@_)	{ return $self->plist()->{'vendor'} = shift }
	else	{ return $self->plist()->{'vendor'} }
}

sub addDepend {
	my($self,$vendor,$name);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$self->plist()->_add_depend($_[0],$_[1]);
}

sub addObsoletes {
	my($self,$vendor,$name);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$self->plist()->_add_obsoletes($_[0],$_[1]);
}

# size()
# size field accessor
sub Filesize {
	my($self,$loc);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	if (@_)	{ return $self->plist()->{'filesize'} = shift }
	else	{ return $self->plist()->{'filesize'} }
}

# touches $dir/.UNPACKED to alert package unpacking routines not re-unpack it
sub _set_unpacked_flag {
	my($dir) = shift
		or Bluelinq::blqlog(0,"Specifiy a directory to set flag");
	open(FLAG,"> $dir/.UNPACKED")
		or Bluelinq::blqlog(0,"Could not open flag file: $!");
	print FLAG scalar localtime;
	close(FLAG) or Bluelinq::blqlog(0,"Could not close flag file: $!");
}

1;
# Copyright (c) 2003 Sun Microsystems, Inc. All  Rights Reserved.
# 
# Redistribution and use in source and binary forms, with or without 
# modification, are permitted provided that the following conditions are met:
# 
# -Redistribution of source code must retain the above copyright notice, 
# this list of conditions and the following disclaimer.
# 
# -Redistribution in binary form must reproduce the above copyright notice, 
# this list of conditions and the following disclaimer in the documentation  
# and/or other materials provided with the distribution.
# 
# Neither the name of Sun Microsystems, Inc. or the names of contributors may 
# be used to endorse or promote products derived from this software without 
# specific prior written permission.
# 
# This software is provided "AS IS," without a warranty of any kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
# 
# You acknowledge that  this software is not designed or intended for use in the design, construction, operation or maintenance of any nuclear facility.
