# Bluelinq.pm
# Copyright 2002, Sun Microsystems, Inc
# $Id: Bluelinq.pm,v 1.5 2003/02/25 22:30:53 ssmith Exp $

package Bluelinq;
use Bluelinq::Sauce;
use Bluelinq::Sausalito;
use strict;
use Carp;
use POSIX;
use File::Basename;

# global configuration
use vars qw/	$VERSION $VERBOSE $TEMP_DIR
		$TAR_BIN  $GZIP_BIN $GUNZIP_BIN $MD5SUM_BIN $FILE_BIN
		$GPG_BIN $BZIP2_BIN $RM_BIN/;
### GLOBALS ###
$VERSION	= 1.0;
$VERBOSE	= 0;
$TEMP_DIR	= "/tmp/";
$RM_BIN		= "/bin/rm";
$TAR_BIN	= "/bin/tar";
$GZIP_BIN	= "/bin/gzip";
$GUNZIP_BIN	= "/bin/gunzip";
$GPG_BIN	= "/usr/bin/gpg";
$FILE_BIN	= "/usr/bin/file";
$BZIP2_BIN	= "/usr/bin/bzip2";
$MD5SUM_BIN	= "/usr/bin/md5sum";

# bluelinq logging facility, currently to STDERR
# takes a log level and a scalar ref to a log message
# level 0: error log level, errors always printed, see blqerror(), dies
# level 1: warning log level, printed if <= $VERBOSE, see blqwarn()
# level 2: info. log lovel, printed if <= $VERBOSE, see blqinfo()
sub blqlog {
	my ($level,$message);
	$level = shift;
	isdigit($level) || croak "ERROR blqlog(): Bad log level specified";
	$message = shift || \"-- No log message specified --\n";
	if ($level == 0) {
		print STDERR ("ERROR: $$message\n");
		exit(1);
	} elsif ($level == 1 ) {
		print STDERR (" WARNING: $$message\n") if ($VERBOSE >= $level);
	} elsif ($level == 2 ) {
		print STDERR ("  INFO: $$message\n") if ($VERBOSE >= $level);
	} else {
		isdigit($level) &&
			croak ("ERROR blqlog(): invalid log level: $level");
	}
}

sub get_temp_dir {
	my ($tmpname);
	do {
		$tmpname = tmpnam();
	} while (stat $tmpname);
	return $tmpname;
}

# get/set verbosity level as class or instance method
sub verbose {
	my($self,$level);
	$self = shift;
	if(ref($self)) {				# instance method
		$level = shift if @_;			# arg?
	} else {					# class method
		$level = $self if isdigit($self);	# arg?
	}
	if ($level && ($level == 0 || $level == 1 || $level == 2)) {
		$VERBOSE = $level;
	}
	return $VERBOSE;
}

# bluelinq error handling
# calls blqlog(0,\$message), pushes error onto $self->log(), and dies
# takes a string
sub blqerror {
	my($self,$message);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$message = shift || "-- unspecified error --";
	push(@{$self->log()},\$message);
	blqlog(0,\$message);
}

# bluelinq warning message
# calls blqlog(1,\$message), pushes warning onto $self->log()
# takes a string
sub blqwarn {
	my($self,$message);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$message = shift || "-- unspecified warning --";
	blqlog(1,\$message);
	push(@{$self->log()},\$message);
}

# bluelinq informational message
# calls blqlog(2,\$message), pushes message onto $self->log()
# takes a string
sub blqinfo {
	my($self,$message);
	$self = shift;
	croak ("$self not an object") unless ref($self);
	$message = shift || "-- unspecified info message --";
	Bluelinq::blqlog(2,\$message);
	push(@{$self->log()},\$message);
}

# returns md5sum of a file
# takes: filename
# returns: md5sum
sub md5sum_file {
	my($md5sum,$filename,$binresult);
	$filename = shift or blqlog(0,\"md5sum_file: You must specify a file");
	$md5sum = undef;
	blqlog(0,\"Can't read file: $filename") unless (-r $filename);
	# call external binary for portability with solaris
	chomp($binresult = `$Bluelinq::MD5SUM_BIN $filename`);
	if ($binresult =~ m/^(\w+)(\s+)/) {
		$md5sum = $1;
		blqlog(2,\"Setting md5sum to $1");
	} else {
		blqlog(0,\"md5sum of $filename unsucessful");
	}
	return $md5sum;
}

# factory()
# - attempts to figure out what type of Bluelinq file you have and return
# an object based upon it. Use this method if you don't know what type
# of file you have (Sausalito,Sauce). Useful in blqfile2meta
# takes: filename and args like constructors
# returns: an initialized Bluelinq::[Sauce,Sausalito] object
# expense: will only unpack a file once, avoids multiple gunzip ops
sub factory {
	my($filename,@suffixes,$tempdir,$obj);
	$filename = shift
		or Bluelinq::blqlog(0,\"factory: You must specify a file!");
	@suffixes = qw[.iso.gz .pkg.gpg .src.rpm .iso .pkg .rpm];
	$obj = undef;
	$tempdir = Bluelinq::get_temp_dir();
	mkdir $tempdir, 0744 or blqlog(0,\"Couldn't make directory");
	push(@_,"templocation",$tempdir);
	croak("Error: $filename does not exist!") unless (-e $filename);
	GUESS: {
	  my ($name,$dir,$suffix) = fileparse($filename,@suffixes);
	  last GUESS unless $suffix; $_ = $suffix;
	  /^\.pkg.gpg$/	&& do {	
	    if (Bluelinq::Sausalito::_do_unpack($filename,$tempdir)) {
		$obj = Bluelinq::Sausalito->new(@_);
	    }
	    last GUESS;
	  };
	  /^\.pkg$/ && do {
	    if (Bluelinq::Sauce::_do_unpack($filename,$tempdir)) {
		$obj = Bluelinq::Sauce->new(@_); last GUESS;
	    }
	    if (Bluelinq::Sausalito::_do_unpack($filename,$tempdir)) {
		$obj = Bluelinq::Sausalito->new(@_); last GUESS;
	    }
	    # this is such a joke
	    if (basename($filename) eq 'casp_interbase_patch_raq3.pkg') {
		;
	    }
	  };
	}; # end GUESS block
	Bluelinq::blqlog(0,\"factory(): Could not unpack file!") unless $obj;
	return $obj;
}

1;
# Copyright (c) 2003 Sun Microsystems, Inc. All  Rights Reserved.
# 
# Redistribution and use in source and binary forms, with or without 
# modification, are permitted provided that the following conditions are met:
# 
# -Redistribution of source code must retain the above copyright notice, 
# this list of conditions and the following disclaimer.
# 
# -Redistribution in binary form must reproduce the above copyright notice, 
# this list of conditions and the following disclaimer in the documentation  
# and/or other materials provided with the distribution.
# 
# Neither the name of Sun Microsystems, Inc. or the names of contributors may 
# be used to endorse or promote products derived from this software without 
# specific prior written permission.
# 
# This software is provided "AS IS," without a warranty of any kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
# 
# You acknowledge that  this software is not designed or intended for use in the design, construction, operation or maintenance of any nuclear facility.
