/*****************************************************************************
;++
Name            :   VWIN.C
Title 			:   Simple MIC System II Video Window
C Author        :   Angus Gray
Created         :   13/07/92
Description     :   A very simple Windows application which demonstrates how
					to put video in a window using MIC System II.
					This code is linked to MICDRV.LIB, if you want your
					application to run even if MIC System II is not present
					the do not link with MICDRV.LIB and use LoadLibrary,
					GetProcAddress and FreeLibrary to communicate with MIC
					System II.
					
Program Type    :   Windows 3.x
Modifications   :   None
Notes           :   Requires MIC System II version 2.0 or above.
;--
*****************************************************************************/

#include "Windows.h"
#include "MIC2.H"
#include "VWIN.H"

/****************************************************************************/
/*      Local defines.                                                      */
/****************************************************************************/              
#define MYLOGPALSIZE	16

/****************************************************************************/
/*      Local Variables.													*/
/****************************************************************************/              
static HANDLE	hInst;
static DWORD	MICId;
static char		szMICCmdBuf[256];
static char		szMICRspBuf[256];
static WORD		xVideoWin;
static WORD		yVideoWin;
static WORD		cxVideoWin;
static WORD		cyVideoWin;
static WORD		cxScreen;
static WORD		cyScreen;

/****************************************************************************/
/*      Local Procedure Definitions.                                        */
/****************************************************************************/
BOOL AssignDVA(VOID);
VOID DeassignDVA(VOID);
VOID InitVideoWindow(HWND, WORD);
VOID SizeVideoWindow(WORD, WORD);
VOID MoveVideoWindow(WORD, WORD);
VOID DeInitVideoWindow(VOID);

/*****************************************************************************
;+
Function Name   :   WinMain
Inputs          :  	
Outputs         :  	
Side Effects    :  	
Description     :  	Main message loop for the application.
					Only one copy of this application can run at any one time
;-
*****************************************************************************/
int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG		msg;
	WORD	Ret;
	HWND	hwndPrev;

    if (hPrevInstance)			 /* Other instances of app running? */
	{
		/* Find the current instance of the app and bring to top */
		if (hwndPrev = FindWindow(APPLICATION_CLASS, APPLICATION_TITLE))
		{
			if (IsIconic(hwndPrev))
				ShowWindow(hwndPrev, SW_SHOWNORMAL);
			else
				BringWindowToTop(hwndPrev);
		}
		/* exit noe the current instance is visible */
		return (FALSE);
	}

	if (!InitApplication(hInstance))
	{
		return (FALSE);
	}

	/* check for MIC open success */
	Ret = MICOpen(&MICId);
	if (Ret != MIC2_OK)
	{
		/* Error can't open MIC */
		MessageBox(GetFocus(),
			(LPSTR) "Failed To Open MIC System II",
			(LPSTR) APPLICATION_TITLE,
			MB_OK | MB_APPLMODAL | MB_ICONSTOP);
		DeInitApplication(hInstance);
		return (FALSE);
	}

	if (!AssignDVA())
	{
		/* Error can't assign to the DVA-4000 */
		MessageBox(GetFocus(),
			(LPSTR) "Failed To Assign to the DVA-4000",
			(LPSTR) APPLICATION_TITLE,
			MB_OK | MB_APPLMODAL | MB_ICONSTOP);
		DeInitApplication(hInstance);
		return (FALSE);
	}

	if (!InitInstance(hInstance, nCmdShow))
	{
		/* Report Error to user */
		MessageBox(GetFocus(),
			(LPSTR) "Failed To Create Application Windows",
			(LPSTR) APPLICATION_TITLE,
			MB_OK | MB_APPLMODAL | MB_ICONSTOP);

		/* CLose MIC */
		MICClose(&MICId);
		DeInitApplication(hInstance);
		return (FALSE);
	}

    /* Acquire and dispatch messages until a WM_QUIT message is received. */
    while (GetMessage(&msg, NULL, NULL, NULL))
	{
		TranslateMessage(&msg);	   /* Translates virtual key codes	     */
		DispatchMessage(&msg);	   /* Dispatches message to window	     */
    }

	DeInitInstance();
	DeassignDVA();
	MICClose(&MICId);
	DeInitApplication(hInstance);

    return (msg.wParam);
}

/*****************************************************************************
;+
Function Name   :   InitApplication
Inputs          :  	
Outputs         :  	
Side Effects    :  	
Description     :  	Register all the necessary window classes for the app.
;-
*****************************************************************************/
BOOL InitApplication(HANDLE hInstance)
{
	WNDCLASS  wc;

	hInst = hInstance;
	
	wc.style = CS_VREDRAW | CS_HREDRAW;
	wc.lpfnWndProc = VideoWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInstance;
	wc.hIcon = NULL;
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = NULL;
	wc.lpszMenuName =  NULL;
	wc.lpszClassName = APPLICATION_CLASS;

	if (!RegisterClass(&wc))
		return (FALSE);

	return (TRUE);
}

/*****************************************************************************
;+
Function Name   :   DeInitApplication
Inputs          :  	
Outputs         :  	
Side Effects    :  	
Description     :  	
;-
*****************************************************************************/
BOOL DeInitApplication(HANDLE hInstance)
{
	return (TRUE);
}

/*****************************************************************************
;+
Function Name   :   InitInstance
Inputs          :  	
Outputs         :  	
Side Effects    :  	
Description     :  	Create the application window.
;-
*****************************************************************************/
BOOL InitInstance(HANDLE hInstance, int nCmdShow)
{
	HWND		hwnd;
	RECT		Rect;

	Rect.left = 0;
	Rect.top = 0;
	Rect.right = 320;
	Rect.bottom = 240;
	AdjustWindowRect(&Rect, WS_OVERLAPPEDWINDOW, FALSE);
	hwnd = CreateWindow(
		APPLICATION_CLASS,
		APPLICATION_TITLE,
		WS_OVERLAPPEDWINDOW,
		CW_USEDEFAULT,
		CW_USEDEFAULT,
		Rect.right - Rect.left,
		Rect.bottom - Rect.top,
		NULL,
		NULL,
		hInst,
		NULL);
	ShowWindow(hwnd, SW_SHOW);
	UpdateWindow(hwnd);

	return (TRUE);
}

/*****************************************************************************
;+
Function Name   :   DeInitInstance
Inputs          :  	
Outputs         :  	
Side Effects    :  	
Description     :  	
;-
*****************************************************************************/
VOID DeInitInstance(VOID)
{
}

/*****************************************************************************
;+
Function Name   :   VideoWndProc
Inputs          :  	
Outputs         :  	
Side Effects    :  	
Description     :  	Main message handler for the Video Window.
					This is a very simple implementation of video in a window
					that is designed to show the basic concept, and not to
					show all the features of MIC System II and the DVA-4000
;-
*****************************************************************************/
LONG FAR PASCAL VideoWndProc(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
{
	RECT			Rect;
	LONG			lRet;
	HDC				hDC;
	HMENU			hMenuSystem;
	int				iLoop;
	FARPROC			fpfn;
	NPLOGPALETTE	pLogPal;
	static	HANDLE	hPal;
	static	HBRUSH	hBrushTrans;
	static	int		TransColourIndex;

	lRet = 0L;
	switch (message)
	{
		case WM_CREATE:
			/*
				Need to setup a logical palette so that the Video Window
				remains transparent even if the physical palette changes.
			*/
			TransColourIndex = 13;
			hDC = GetDC(hwnd);
			/*
				Allocate enough memory for a logical palette with
				MYLOGPALSIZE entries and set the size and version fields
				of the logical palette structure.
			*/
			pLogPal = (NPLOGPALETTE) LocalAlloc (LMEM_FIXED,
						(sizeof (LOGPALETTE) +
						(sizeof (PALETTEENTRY) * (MYLOGPALSIZE))));

			pLogPal->palVersion	= 0x300;
			pLogPal->palNumEntries = MYLOGPALSIZE;

			/* fill in intensities for all palette entry colors */
			for (iLoop = 0; iLoop < MYLOGPALSIZE; iLoop++)
			{
				*((WORD *) (&pLogPal->palPalEntry[iLoop].peRed)) = iLoop;
				pLogPal->palPalEntry[iLoop].peBlue  = 0;
				/*
					Setup the flags so that the palette entries refer
					to the hardware palette entries
				*/
				pLogPal->palPalEntry[iLoop].peFlags = PC_EXPLICIT;
			}

			/* Create the palette */
			hPal = CreatePalette ((LPLOGPALETTE) pLogPal);

			/* Select the palette, and realize it */
			SelectPalette(hDC, hPal, 1);
			RealizePalette(hDC);

			/*
				Create the brush that is to be transparent, the actual
				colour of this brush is not important, so create it
				using a palette index.
			*/
			hBrushTrans = CreateSolidBrush(PALETTEINDEX(TransColourIndex));
			SelectObject(hDC, hBrushTrans);
			/* Free the memory allocated for the logical palette */
			LocalFree((HANDLE) pLogPal);
			ReleaseDC(hwnd, hDC);

			/* add the About... menu to the system menu */
			hMenuSystem = GetSystemMenu(hwnd, FALSE);
			AppendMenu(hMenuSystem, MF_SEPARATOR, 0, NULL);
			AppendMenu(hMenuSystem, MF_STRING, IDM_ABOUT,
				"&About Video Window...");

			/* Initialise the Video Window */
			InitVideoWindow(hwnd, TransColourIndex);
			break;

		case WM_ERASEBKGND:
			/* Fill the client window with the transparent brush */
			GetClientRect(hwnd, &Rect);
			SelectObject((HDC) wParam, hBrushTrans);
			FillRect((HDC) wParam, &Rect, hBrushTrans);
			break;

		case WM_SIZE:
			/* size the Video Window */
			SizeVideoWindow(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_MOVE:
			/* Move the Video Window */
			MoveVideoWindow(LOWORD(lParam), HIWORD(lParam));
			break;

		case WM_DESTROY:
			/* delete the palette created at WM_CREATE time */
			DeleteObject(hPal);
			/* Clean up MIC before exiting */
			DeInitVideoWindow();
			PostQuitMessage(0);
			break;

		case WM_SYSCOMMAND:
			if (wParam == IDM_ABOUT)
			{
				/* Display "About" dialog box.  */
				fpfn = MakeProcInstance((FARPROC) AboutDlgProc, hInst);
				DialogBox(hInst, "ABOUT", hwnd, (FARPROC) fpfn);
				FreeProcInstance(fpfn);
			}
			else
				lRet = DefWindowProc(hwnd, message, wParam, lParam);
			break;

		default:			  /* Passes it on if unproccessed	*/
			lRet = DefWindowProc(hwnd, message, wParam, lParam);
	}
	return (lRet);
}

/*****************************************************************************
;+
Function Name   :   AssignDVA
Inputs          :	
Outputs         :  	
Side Effects    :  	
Description     :  	Assign the DVA to our application, if any of the assigns
					fail deassign any already assigned.
					The error returned from MIC should really be interpreted
					to determine the true cause of the failed assignment.
;-
*****************************************************************************/
BOOL AssignDVA(VOID)
{
	WORD	wMICErr;
	BOOL	bRet;

	bRet = FALSE;
	wMICErr = MICWrRdStr(MICId, (LPSTR) "FStore $VWin Assign DVA", szMICRspBuf, 256);
	if (wMICErr == MIC2_OK)
	{
		wMICErr = MICWrRdStr(MICId, (LPSTR) "Graphics $VWin Assign DVA", szMICRspBuf, 256);
		if (wMICErr == MIC2_OK)
		{
			wMICErr = MICWrRdStr(MICId, (LPSTR) "Video $VWin Assign DVA", szMICRspBuf, 256);
			if (wMICErr == MIC2_OK)
			{
				bRet = TRUE;
			}
			else
			{
				/* assign failed */
				MICWriteStr(MICId, (LPSTR) "Graphics $VWin Deassign");
				MICWriteStr(MICId, (LPSTR) "FStore $VWin Deassign");
			}
		}
		else
		{
			/* assign failed */
			MICWriteStr(MICId, (LPSTR) "Graphics $VWin Deassign");
		}
	}
	else
	{
		/* assign failed */
	}
	return (bRet);
}

/*****************************************************************************
;+
Function Name   :   DeassignDVA
Inputs          :	
Outputs         :  	
Side Effects    :  	
Description     :  	Deassign the DVA from our application
;-
*****************************************************************************/
VOID DeassignDVA(VOID)
{
	/* should really do some error checking here */
	MICWriteStr(MICId, (LPSTR) "Video $VWin Deassign");
	MICWriteStr(MICId, (LPSTR) "Graphics $VWin Deassign");
	MICWriteStr(MICId, (LPSTR) "FStore $VWin Deassign");
}

/*****************************************************************************
;+
Function Name   :   InitVideoWindow
Inputs          :	
Outputs         :  	
Side Effects    :  	
Description     :  	Turn Video on, set the transparent colour.
;-
*****************************************************************************/
VOID InitVideoWindow(HWND hwnd, WORD TransColor)
{
	HDC		hDC;

	/* get the screen resolution of the device */
	hDC = GetDC(hwnd);
	cxScreen = GetDeviceCaps(hDC, HORZRES);
	cyScreen = GetDeviceCaps(hDC, VERTRES);
	ReleaseDC(hwnd, hDC);

	/*
		Set FStore into mode normal, this gives us the default coordinate
		system the same as the display.
	*/
	MICWriteStr(MICId, (LPSTR) "FStore $VWin Mode Normal");
	/* set the transparent colour */
	wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "Graphics $VWin Transparent  %d", TransColor);
	MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
	/* turn Video on */
	MICWriteStr(MICId, (LPSTR) "Video $VWin On");
}

/*****************************************************************************
;+
Function Name   :   SizeVideoWindow
Inputs          :	
Outputs         :  	
Side Effects    :  	This code compensates for a bug on the DVA-4000 board
					which causes video coruption when the image is displayed
					off the bottom right of the screen. This code is only valid
					if inputs are enables.
Description     :  	Set the size of the Video Window (In screen coordinates)
;-
*****************************************************************************/
VOID SizeVideoWindow(WORD cx, WORD cy)
{
	/* save the new position */
	cxVideoWin = cx;
	cyVideoWin = cy;

	/*
		Hold FStore so all FStore commands are sent to the DVA at
		the same time to minimise screen disturbance.
	*/
	MICWriteStr(MICId, (LPSTR) "FStore $VWin Hold");
	/* set the display and window sizes to the same values */
	wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "FStore $VWin DSize %d %d %d", 3, cx, cy);
	MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
	wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "FStore $VWin WSize %d %d %d", 3, cx, cy);
	MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
	/* check if need to remove m2 bug compensation */
	if (xVideoWin + cxVideoWin <= cxScreen)
	{
		/* restore iregion */
		wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "FStore $VWin Iregion 3 %d %d %d %d",
			0, 0, cxScreen - 1, cyScreen - 1);
		MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
	}
	/* send all the commands down to the DVA */
	MICWriteStr(MICId, (LPSTR) "FStore $VWin Release");
}

/*****************************************************************************
;+
Function Name   :   MoveVideoWindow
Inputs          :	
Outputs         :  	
Side Effects    :  	This code compensates for a bug on the DVA-4000 board
					which causes video coruption when the image is displayed
					off the bottom right of the screen. This code is only valid
					if inputs are enables.
Description     :  	Moves the Video Window (In screen coordinates)
;-
*****************************************************************************/
VOID MoveVideoWindow(WORD x, WORD y)
{
	xVideoWin = x;
	yVideoWin = y;

	/*
		Hold FStore so all FStore commands are sent to the DVA at
		the same time to minimise screen disturbance.
	*/
	MICWriteStr(MICId, (LPSTR) "FStore $VWin Hold");
	wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "FStore $VWin WPos 3 %d %d", x, y);
	MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
	wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "FStore $VWin DPos 3 %d %d", 0, 0);
	MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
	if ((xVideoWin + cxVideoWin) > cxScreen)
	{
		/* need to stop the M2 bug */
		wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "FStore $VWin DSize 3 %d %d",
			(cxScreen - xVideoWin),
			cyVideoWin);
		MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
		wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "FStore $VWin Iregion 3 %d %d %d %d",
			0, 0,
			MulDiv((cxScreen - xVideoWin), cxScreen, cxVideoWin) - 1,
			cyScreen - 1);
		MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
	}
	else
	{
		wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "FStore $VWin DSize 3 %d %d",
			cxVideoWin, cyVideoWin);
		MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
		wsprintf((LPSTR) szMICCmdBuf, (LPSTR) "FStore $VWin Iregion 3 %d %d %d %d",
			0, 0, cxScreen - 1, cyScreen - 1);
		MICWriteStr(MICId, (LPSTR) szMICCmdBuf);
	}
	/* send all the commands down to the DVA */
	MICWriteStr(MICId, (LPSTR) "FStore $VWin Release");
}

/*****************************************************************************
;+
Function Name   :   DeInitVideoWindow
Inputs          :	
Outputs         :  	
Side Effects    :  	
Description     :  	
;-
*****************************************************************************/
VOID DeInitVideoWindow()
{
	/* Remove the transparent colour */
	MICWriteStr(MICId, (LPSTR) "Graphics $VWin Transparent None");
	/* Turn video off */
	MICWriteStr(MICId, (LPSTR) "Video $VWin Off");
}

/*****************************************************************************
;+
Function Name   :   AboutDlgProc
Inputs          :	
Outputs         :  	
Side Effects    :  	
Description     :  	Dialog box proc for the About... box.
;-
*****************************************************************************/
BOOL FAR PASCAL AboutDlgProc(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
{
	BOOL	bRet;

	bRet = FALSE;
	switch (message)
	{
		case WM_INITDIALOG:
			bRet = TRUE;
			break;

		case WM_COMMAND:
			if (wParam == IDOK)
				EndDialog(hwnd, TRUE);
			break;
	}
	return (bRet);
}

/****************************************************************************/
/*	End Of File (VWIN.C)													*/
/****************************************************************************/
