/*****************************************************************************

	Module	    : DVAUTILS.C
	Created      : 13th July 1988
	Author	    : Mike Brassington
	Amended		 : 20th May 1991 (calls to DLL instead of to MIC direct)

	Description : This contains the hardware interface & utility
				  routines for the DVA Control Program.

    Copyright 1988,1989 by VideoLogic Limited. All rights reserved.

	No part of this software, either material or conceptual may be
	copied or distributed, transmitted, transcribed, stored in a
	retrieval system or translated into any human or computer language
	in any form by any means, electronic, mechanical, manual or other-
	wise, or disclosed to third parties without the express written
	permission of VideoLogic Limited, Unit 8, Homepark industrial estate,
	Kings Langley, Herts. WD4 8LZ. UK.

	************************************************************************
	*                                                                      *
	*                  FOR DEMONSTRATION PURPOSES ONLY                     *
	*  This source code is supplied as a demonstration of the techniques   *
	*  involved in the writing of Windows applications that intend to      *
	*  make use of the Videologic DVA-4000 Digital Video Adapter.          *
	*  Videologic Ltd. accept no liability for loss of business or damage  *
	*  arising from the use of this software either in whole or in part.   *
	*                                                                      *
	************************************************************************

	MODIFICATIONS:

*****************************************************************************/

#define LINT_ARGS TRUE				/* enable type checking in libraries */
#include <windows.h>				/* required for all Windows applications */
#include <dos.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "DVA.h"                    /* specific to this program */
#include "micdll.h"

/* Debug flag */
#define  FDEBUG    FALSE

/* Declare external references */
extern BOOL SquashActive;
extern BOOL	FreezeActive;			/* Video freeze enabled or not */
extern BOOL	PropActive;				/* Keep video in proportion? */
extern BOOL	LockActive;				/* Lock on to an area of video */
extern LONG WinStyle;				/* current window style */

/* Declare Variables */
BOOL SizIcon = FALSE;				/* default to not previously iconic */
BOOL VidIcon = FALSE;				/* default to not currently iconic */
BOOL VidRestore = TRUE;				/* default to video mode restored */
int	VertRes, HorzRes;				/* Y and X resolution of screen */
int	WinX, WinY;						/* X - origin and Y - origin of window */
int	WinWidth, WinHeight;			/* Window width and height */
int LockX, LockY;						/* Window pos when picture lock enabled */
DWORD WinColour;						/* Colour for window background */
HBRUSH hWindBrush;					/* Brush object handle */
char micbuff[100];					/* Buffer for MIC commands */
char pString[100];
char pBuffer[100];
char transbuff[10];					/* Buffer for Trans colours */
int FSto1 = 1;							/* default to fieldstore 1 in use */
int FSto2 = 2;							/* other fieldstore */ 
int FMode = 0;							/* current Service system mode */
int F2DPosX,F2DPosY;
int F2WPosX,F2WPosY;
int F2DSizX,F2DSizY;
int F2WSizX,F2WSizY;
int F2IRegX1,F2IRegY1,F2IRegX2,F2IRegY2;
char F2RelAbs[9];

#if FDEBUG
int hFile;
char  GenText[50];
OFSTRUCT OfStruct;
int A,B;
#endif

/* Start of Code */

/****************************************************************************

    FUNCTION:	DVAGetFMode(void)

    PURPOSE:	Returns the current framestore mode.

    COMMENTS:	Returns integer values:

    			0	Normal Mode
    			3	FrzMan Mode 
				6	Dual Mode
				7	Parallel Mode

****************************************************************************/
int DVAGetFMode ( void )
{
	int Mode;
	QFMode(&Mode);
	return(Mode);
}


/****************************************************************************

    FUNCTION:	DVAIsLoadable(char *)

    PURPOSE:	Checks that this instance of DVA is loadable.

    COMMENTS:	First instance of DVA uses fieldstore in Normal mode.
    			Second instance uses Parallel or Dual mode.
    			A third instance is disallowed.
    			Returns integer error values:

    			0	No Error
    			1	MIC-DD not memory resident
    			2	Error opening MIC
				3	Service Functions not memory resident
    			4	This is a third instance

****************************************************************************/
int DVAIsLoadable ( void )
{
   if (isMICloaded() == 0)
		return(2);						/* MIC not loaded!?! */

	SendMIC("!EOFF");				/* turn off error report display */
	SendMIC("!TERM=0D");			/* and set terminator */
	QFMode(&FMode);					/* get current framestore mode */

	if (FMode == 5 || FMode == 6 || FMode == 7)		/* if it's Parallel or Dual then */
		return(4); 						/* report third instance */

	if (FMode)							/* if it's not Normal then */ 
		return(3);						/* MIC-SERV is probably not loaded */
	return(0);							/* all pretty damn groovy */
}


/****************************************************************************

    FUNCTION: 	SetScreenAttr(HANDLE)

    PURPOSE:  	Get size of this display screen & set up brush &c.

    COMMENTS:	It is entirely possible for Windows to report a screen
    			resolution of some multiple of its basic operating screen
    			mode resolution, eg. in a multi graphics adapter environment.
    			Current hardware contraints limit the video output to approx.
    			one screen and attempts to position video in areas outside of
    			this fail. If the internal MIC coordinate scheme is congruent
    			with the origin of the base screen, video will merely not be
    			displayed when positioned off-screen. If the scheme maps onto
    			an extended screen resolution, the hardware will be driven
    			incorrectly and predictable but unsightly results will occur.

****************************************************************************/
void SetScreenAttr ( HANDLE PrevInst )
{
#define PALETTESIZE 1
    HDC hDC;								/* Display context handle */

/* Select Background colour & brush */
   WinColour = RGB(255,0,255);			/* Background = light magenta */
	hWindBrush = CreateSolidBrush(WinColour);

/* Get Display resolution */

    hDC = GetDC((HWND)NULL);				/* Get display context */
    VertRes = GetDeviceCaps(hDC, VERTRES);	/* Height of display (lines) */
    HorzRes = GetDeviceCaps(hDC, HORZRES);	/* Width of dispay (pixels) */

    ReleaseDC((HWND)NULL, hDC );			/* Release display context */
}


/****************************************************************************

    FUNCTION: DVAInit(HANDLE)

    PURPOSE:  Initialise the DVA board for display upon window creation

    COMMENTS:

****************************************************************************/
int DVAInit ( HANDLE PrevInst )
{
    int InitErr = 0;

    SendMIC("FST HOLD 0");

/* Initialise Client window size and position. */
 
    WinX = HorzRes/4;		    		/* X position */
    WinY = VertRes/4;		    		/* Y position */
    WinWidth = HorzRes/2;	    		/* window width */
    WinHeight = VertRes/2;	    		/* window height */


/* If First invocation of DVA, initialise MIC System to player 1 stilled 
and ready, video on, logical colours 5 13 (magenta) transparent, framestore
mode normal. If second, then set Parallel mode. */


	if (PrevInst)
	{
		InitErr = 0;
		FSto2 = 1;
		FSto1 = 2;
    	SendMIC("FST MODE 0 PARALLEL");		
		FMode = 7;
	}
	else
	{
		SendMIC("INI #1");
		QueryMIC(&InitErr, "!STATUS");

		if (InitErr) 
			InitErr = 1; 
		SendMIC("STILL #1 100; TRA 5,13;VID ON; FST MODE 0 NORMAL");		
		FMode = 0;
		FSto2 = 2;
		FSto1 = 1;
	}

	DVASource(FALSE);

/* set up internal MIC coordinate scheme. The MIC System defaults to 640x480,
this is fine for Windows but we do it explicitly. The coordinate scheme
remains in effect until another COORDS command or a system reload */

	SendMIC("FST COORDS 0 %d %d",HorzRes,VertRes);

/* set input region to maximum dimensions of internal coordinate system */

   SendMIC("FST IREGION 0 %d FULL",FSto1);	

/* set dimensions of area in which to display video */

	SendMIC("FST DSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);

/* set top left position of display area and make this position relative 
to the input region viewport. (The default case is RELATIVE, don't really 
need to specify keyword explicitly) */

	SendMIC("FST DPOS 0 %d 0 0 RELATIVE",FSto1);

/* set dimensions of viewport. The viewport acts like a sliding window on the
contents of the input region, video data is available for screen display only
in the viewport area. */

	SendMIC("FST WSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);

/* set top left position of viewport */

	SendMIC("FST WPOS 0 %d %d %d",FSto1,WinX,WinY);
	SendMIC("FST RELEASE 0");
	return(InitErr);
}


/****************************************************************************

    FUNCTION:	DVASize(HWND, WORD, LONG)

    PURPOSE:	Keeps a track of the window client area size

    COMMENTS:	If going iconic, the code switches the current 
				video mode to full screen video compressed to the icon 
				size, restoring the previous mode on exit from iconic status.

****************************************************************************/
void DVASize ( HWND hWnd, WORD wParam, LONG lParam )
{
	SendMIC("FST HOLD 0");

/* !! WINDOWS BUG WORKAROUND !! */
/* When application maximized and titlebar, menubar, and frame removed via
   double clicking, Windows sends incorrect values for the new window size
   and position */

	if (wParam == SIZEFULLSCREEN && (WinStyle & WS_TILEDWINDOW) != WS_TILEDWINDOW)
	{
		WinWidth = HorzRes;						/* New window width	*/
    	WinHeight = VertRes;						/* New window height */
    	WinX = WinY = 0;							/* Set viewport to origin */
		SendMIC("FST WPOS 0 %d %d %d",FSto1,WinX,WinY);	/* and repos */
	}
	else
	{
		WinWidth = (int)LOWORD(lParam);		/* New window width	*/
		WinHeight = (int)HIWORD(lParam);		/* New window height */
	}
	VidIcon = (wParam != SIZEICONIC) ? FALSE : TRUE;
	if (!VidIcon)									/* if not iconic, then... */
	{
		if (SizIcon)								/* if previously iconic */
		{
			if (PropActive)						/* restore pre-iconic vid mode */
			{
				DVAProp(TRUE);
				if (LockActive && wParam != SIZEFULLSCREEN)
				{							
					SendMIC("FST IREGION 0 %d %d %d %d %d",FSto1,LockX,
								LockY,LockX+WinWidth-1,LockY+WinHeight-1);
					SendMIC("FST DPOS 0 %d 0 0 RELATIVE",FSto1);
				}
			}
			else
			{
				SendMIC("FST DSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);
				SendMIC("FST WSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);
			}
			SizIcon = FALSE;				/* and flag not previously iconic */
		}
		else									/* Not previously iconic so... */
		{
			if (PropActive)				/* if keeping 1:1 ratio, then... */
			{
												/* Change viewport size */
		 		SendMIC("FST WSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);
				if (LockActive)			/* and if locked to video area */
				{

/* Change the input region to hold only the video data that pertains to the
window dimensions and position. If we are maximising, switch temporarily to 
non locked mode - the on-screen effect is better */

					if (wParam == SIZEFULLSCREEN)	
					{
						SendMIC("FST IREGION 0 %d FULL",FSto1);
						SendMIC("FST DPOS 0 %d 0 0 ABSOLUTE",FSto1);
					}
					else
					{
						SendMIC("FST IREGION 0 %d %d %d %d %d",LockX,
							FSto1,LockY,LockX+WinWidth-1,LockY+WinHeight-1);
						SendMIC("FST DPOS 0 %d 0 0 RELATIVE",FSto1);
					}
				}	
			}
			else								/* not keeping 1:1 ratio, so... */
			{
				/* Change the size of area in which video will be displayed */ 
			
				SendMIC("FST DSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);
				SendMIC("FST WSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);
			}
		}
	}
	else										/* going iconic */
	{ 
		if (LockActive)
		{
			DVALock(FALSE);				/* execute mode changes */
			DVAProp(FALSE);
		}			
		else
		{
			if (PropActive)
				DVAProp(FALSE);			
			else
			{
				/* Change the size of area in which video will be displayed */ 

				SendMIC("FST DSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);
				SendMIC("FST WSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);
			}
		}
		SizIcon = TRUE;					/* set previously iconic flag */
	}
	SendMIC("FST RELEASE 0");
}


/****************************************************************************

    FUNCTION:	DVAMove(HWND, WORD, LONG)

    PURPOSE:	Keeps a track of client window position.

    COMMENTS:	The code moves the viewport to the position of the client 
    			window.

****************************************************************************/
void DVAMove ( HWND hWnd, WORD wParam, LONG lParam )
{
	WinX = (int)LOWORD(lParam); 	   		/* New X origin in screen Co-ords */
   WinY = (int)HIWORD(lParam); 	   		/* New Y origin	in screen Co-ords */

	SendMIC("FST WPOS 0 %d %d %d",FSto1,WinX,WinY);
}


/****************************************************************************

    FUNCTION:	DVAFreeze(BOOL)

    PURPOSE:	Toggles between frozen or full motion video.

    COMMENTS:	This routine takes different actions depending on the 
    			fieldstore mode. If not in Normal mode, it enables or disables 
    			video input to the framestore associated with the window 
    			referenced in the thaw or freeze request. Output from the 
    			framestore is not affected, and the onscreen video image 
    			reflects the live or static video data within the framestore.

				The DVA-4000's ability to compress and position video data on
				screen is a function of the scaling and sampling of the live,
				incoming data, it does not manipulate the contents of the 
				fieldstore RAM. Thus the frozen (video input disabled) image
				is not manipulable in the sense of the sizing and positioning
				possible with live data.

				If in Normal mode, the routine toggles Freeze Manipulation
				mode. In Normal mode, whilst two framestores are used for
				video display, only one is absolutely necessary. FrzMan
				mode internally freezes input to fstore 1 and uses it as a 
				pseudo live source for fstore 2, circulating the frozen data 
				around the merge logic in emulation of incoming live data.
				The code here swaps the FSto1 and 2 variables because in FrzMan
				mode fstore 2 is used for video output.

				On a thaw request normal mode is reentered after which video
				input is reenabled. In normal mode the framestore parameter
				supplied to the input enable command is redundant. Setting
				normal mode does not reenable video input (implicitly disabled
				by the switch to FrzMan) hence we do it explicitly.
****************************************************************************/
void DVAFreeze ( BOOL FzAct )
{
	if (FzAct)								/* if request to freeze and */
	{
		if (DVAGetFMode() != 0)			/* not in normal mode then... */
		{
			/* Disable the input of new video data to framestore. */

			SendMIC("FST INPUT 0 %d DISABLE",FSto1);
		} else { 							/* in normal mode */
/* Switch to freeze manipulation mode.  */

			SendMIC("FST MODE 0 FRZMAN");
			FSto1 ^= 3;						/* swap fstore variables, so that */
			FSto2 ^= 3;						/* future commands ref. fstore 2 */
		}
	}
	else										/* if request to thaw and */
	{
		if (DVAGetFMode() != 3)			/* not in frzman mode then... */
		{
/* Enable the input of video data to framestore. */

			SendMIC("FST INPUT 0 %d ENABLE",FSto1);
		}
		else									/* in frzman mode */
		{
/* switch to back to normal mode using positioning data from fstore 2*/

			SendMIC("FST MODE 0 NORMAL=FS2");
			SendMIC("FST INPUT 0 %d ENABLE",FSto1);
			FSto1 ^= 3;						/* reset fstore variables */
			FSto2 ^= 3;
		}
	}
}


/****************************************************************************

    FUNCTION:	DVASquash(BOOL)

    PURPOSE:	Toggles between compressed and non compressed locked video.

    COMMENTS:	
 
****************************************************************************/
void DVASquash ( BOOL SqAct )
{
	SquashActive = SqAct;
	if (SqAct)
	{
		if (LockActive) DVALock(FALSE);
		DVAProp(FALSE);
	}
	else
	{
		if (!LockActive) DVALock(TRUE);
		DVAProp(TRUE);
	}
}


/****************************************************************************

    FUNCTION:	DVAProp(BOOL)

    PURPOSE:	Toggles between 1 : 1 proportion and compressed modes.

    COMMENTS:	In proportionality mode the video on screen is always
    			uncompressed. At any point in time we view an area of
    			video through a sliding window. If proportionality mode
    			is selected, video display size and position are set to 
    			the screen size and origin, while the viewport size and 
    			position is set to those of the client window. Re-enabling
    			compression mode is the reverse of this process.

****************************************************************************/
void DVAProp ( BOOL PrAct )
{
	if (PrAct)								/* if proportionality request */
	{

/* Set the area in which video is displayed to the dimensions of the screen. */

		SendMIC("FST DSIZE 0 %d %d %d",FSto1,HorzRes,VertRes);

/* Set the position at which to display video to screen origin. Make this
position independent of the viewport by specifying keyword ABSOLUTE */ 

		SendMIC("FST DPOS 0 %d 0 0 ABSOLUTE",FSto1);

	}
	else										/* if compression request */
	{

/* Set area in which video is displayed to the current dimensions of the
client window */ 

		SendMIC("FST DSIZE 0 %d %d %d",FSto1,WinWidth,WinHeight);

/* Set video display position to the origin relative to viewport */

	   SendMIC("FST DPOS 0 %d 0 0 RELATIVE",FSto1);

	}
}


/****************************************************************************

    FUNCTION:	DVALock(BOOL)

    PURPOSE:	Toggles between locking and unlocking video to client window.

    COMMENTS:	Control transfered here only if we are in 1:1 proportion. 
    			Viewport is already set to client window size and position, 
    			for lock we need merely to restrict access to video data to 
    			that currently in the viewport area, the code thus changes the
    			input region parameters to those of the client window. For 
    			unlock, we need to re-enable access to full coordinate scheme 
    			video data, thus the code resets the input region parameters 
    			to full screen. 

****************************************************************************/
void DVALock ( BOOL LoAct )
{
	if (LoAct)								/* if lock requested */
	{

/* Set top left and bottom right coordinates for the input region to client
window parameters */

		SendMIC("FST IREGION 0 %d %d %d %d %d",FSto1,WinX,
						WinY,WinX+WinWidth-1,WinY+WinHeight-1);

/* set display position to origin of viewport */

		SendMIC("FST DPOS 0 %d 0 0 RELATIVE",FSto1);
		LockX = WinX;
		LockY = WinY;
	}
	else										/* if unlock requested */
	{

/* Set input region back to full size */
	   SendMIC("FST IREGION 0 %d FULL",FSto1);

/* And display position back to origin of coordinate system */
		SendMIC("FST DPOS 0 %d 0 0 ABSOLUTE",FSto1);
	}
}


/****************************************************************************

    FUNCTION:	DVAAudio(BOOL,BYTE)

    PURPOSE:	Switches audio source between DVA input channels 1 and 2

    COMMENTS:	The code merely sets the audio input channel to that 
    			specified by the input parameters.  

****************************************************************************/
void DVAAudio ( BOOL AuAct, BYTE DevNum )
{
	if (AuAct)								/* enable audio */
		SendMIC("AUD INPUT%d",DevNum);
	else									/* disable audio */
		SendMIC("AUD INPUT%d",DevNum ^ 3);
}


/****************************************************************************

    FUNCTION:	DVAPrior(void)

    PURPOSE:	Prioritises display from specific FieldStore

    COMMENTS:	Called on acquisition of the input focus

****************************************************************************/
void DVAPrior ( void )
{
	SendMIC("FST PRIOR 0 %d",FSto1);
}


/****************************************************************************

    FUNCTION:	DVAtrans(void)

    PURPOSE:	Sets the video transparent colour

    COMMENTS:	

****************************************************************************/
void DVAtrans ( void )
{
	SendMIC("TRA 5,13;VID ON");
}


/****************************************************************************

    FUNCTION:	DVASource(BOOL)

    PURPOSE:	Toggle video input source 

    COMMENTS:	Arrive here if user clicks RH mouse button 

****************************************************************************/
void DVASource( BOOL Source)
{
	if (Source)
		SendMIC("AUD INPUT2; VID INPUT2");
	else
		SendMIC("AUD INPUT1; VID INPUT1"); 
}



/****************************************************************************

    FUNCTION:	DVAClose(HANDLE)

    PURPOSE:	Shuts down board.

    COMMENTS:

****************************************************************************/
void DVAClose ( HANDLE PrevInst )
{
	QFMode(&FMode);								/* get FieldStore mode */
	if (!FMode || FMode == 3)					/* if norm or frzman mode... */
	{
		SendMIC("TRA NONE;VID OFF");		/* tidy up */
		SendMIC("!EON");						/* enable MIC error messages */
		DVASource(FALSE);
	}
	else												/* if parallel mode... */
	{
		if (PrevInst)							/* if 2nd invocation set norm */
			SendMIC("FST MODE 0 NORMAL=FS1");	/* mode using FS1 params */
		else											/* if 1st, set normal mode */
			SendMIC("FST MODE 0 NORMAL=FS2");	/* using FS2 params */
		FMode = 0;
	}
    if ( !PrevInst )
    {
	     SendMIC("STILL; vid off; aud off");

    }
}


/****************************************************************************

    FUNCTION:	QFMode(int *Mode)

    PURPOSE:	Queries the Fstore Mode

    COMMENTS:

****************************************************************************/
void QFMode ( int *Mode )
{
	QueryMIC(Mode, "FST QMODE 0");	/* request Fieldstore mode */
}


int SendMIC(const char *fmt, ... )
{
	va_list arg_ptr;
   va_start( arg_ptr, fmt );	    	/* arg_ptr points to format string */

	vsprintf(pString, fmt, arg_ptr);
	return(MICWriteStr((LPSTR)pString));
	}

int QueryMIC (int *Result, const char *fmt, ... )
{
	int status;
	va_list arg_ptr;
   va_start( arg_ptr, fmt );	    	/* arg_ptr points to format string */

	vsprintf(pString, fmt, arg_ptr);
	if ((status = MICWrRdStr((LPSTR)pString, pBuffer, 6)) == 0)
	{
		sscanf(pBuffer,"%d",Result);
	}
	return (status);
}




/* end of dvau.c */
