/******************************************************************************
Name			:   TERMIIC.C
Title			:   C Action routines for I2C device access
C Author		:   Marc Stevens/John Metcalfe
Created			:   30/03/92

Copyright		:   1992 by VideoLogic Limited. All rights reserved.
				    No part of this software, either material or conceptual
				    may be copied or distributed, transmitted, transcribed,
 					stored in a retrieval system or translated into any
				    human or computer language in any form by any means,
				    electronic, mechanical, manual or other-wise, or
				    disclosed to third parties without the express written
				    permission of VideoLogic Limited, Unit 8, HomePark
				    Industrial Estate, King's Langley, Hertfordshire,
				    WD4 8LZ, U.K.

Description		:   This file contains the action routines for IIC device
					related command processing.


Modifications   :   Adapted for use by Terminator Driver by Nick Jones

Notes   :

Contents :

	WriteIICDevice()
	ReadIICDevice()
	SetIICClock()
	SetIICData()
	IICSetup()
	IICStart()
	IICStop()
	IICSendByte()
	IICGetByte()
	IICWaitForSDA()
	IICWriteBytes()
	IICReadByte()
	ClockDelay()
	ReadControlPort()
	ReadIICPort()
	WriteIICPort()

*****************************************************************************/
#include "masdefs.h"
#include "ext.h"
#include "drvio.h"
#include "termreg.h"

#include "TERMIIC.H"

static BOOL	IICClockCopy = TRUE;
static BOOL	IICDataCopy = TRUE;
static BYTE IICDRegCopy;
static BYTE	Pulse1;
static BYTE	gbTempReg;


//#define OK 0

/*****************************************************************************
	Externally defined function prototypes (for extraction to other .h file)
*****************************************************************************/
void 	ReadControlPort(PBYTE CtrlBits);
void 	ReadIICPort(PBYTE IICBits);
void 	WriteIICPort(BYTE IICBits);

/*****************************************************************************
	Local function prototypes
*****************************************************************************/
void	SetIICClock(BOOL);
void	SetIICData(BOOL);
void	IICSetup(void);
void	IICStart(void);
LRESULT	IICStop(void);
LRESULT	IICSendByte(BYTE IICByte);
void	IICGetByte(PBYTE IICBytePtr);
LRESULT	IICWaitForSDA(void);
void	IICWriteBytes(IIC_WRITEBUFF_PTR WriteBufferPtr);
void	IICReadByte(IIC_READBUFF_PTR ReadBufferPtr);

/*****************************************************************************
        Externally Callable Procedures.
*****************************************************************************/

/***************************************************************************
;++
Name:			WriteIICDevice.
Inputs:			IIC_WRITEBUFF_PTR pointer to IIC device and data structure.
Outputs:		Status.
Side Effects:	None.
Description:	Calls the IIC comms primitives to check IIC bus setup
				and then write the data to the device.
;--
****************************************************************************/
LRESULT WriteIICDevice(IIC_WRITEBUFF_PTR WBPtr)
{
	IICSetup();				/* Check the IIC bus is setup */
	IICWriteBytes(WBPtr);	/* Write The Bytes */
	return(WBPtr->Status);	/* Return Status */
}

/***************************************************************************
;++
Name:			ReadIICDevice.
Inputs:			IIC_READBUFF_PTR pointer to IIC device/data structure.
Outputs:		Status.
Side Effects:	None.
Description:	Calls the IIC comms primitives to read device status.
;--
****************************************************************************/
LRESULT ReadIICDevice(IIC_READBUFF_PTR RBPtr)
{
	IICSetup();				/* Check the IIC bus is setup */
	IICReadByte(RBPtr);		/* read DMSD status */
	return(RBPtr->Status);	/* return status of attempted read */
}


/*****************************************************************************
        Local Procedures.
*****************************************************************************/

/*****************************************************************************
	General notes on the following local procedures

	1.	The I2C bus floats high. The Clock and data bits are mapped
        into adjoining bits of the same R/W Terminator register.

    2.  The code that follows is time critical and has been designed
        for polled IO of the Terminator TIMER bit to implement the
		required delay. The IIC bus has a max clock rate of 100KHz ie
		the clock period must be <= 10us.

*****************************************************************************/


/*****************************************************************************
;+
Function Name   :   SetIICClock
Inputs          :   TRUE or FALSE
Outputs         :   None
Side Effects    :   None
Description     :   Sets SCL to required level.
					If Clock, then set IIC word to either Clock and Data
					or just Clock depending on IICDataCopy state.
					If !Clock, IIC Word gets set to either Data or nowt,
					again depending on IICDataCopy.
					In either case set IICClockCopy.
;-
*****************************************************************************/
void SetIICClock(BOOL Clock)
{

	ReadIICPort(&IICDRegCopy);
	if (Clock)
	{
		IICDRegCopy |= IIC_CLOCK_BIT;
		IICClockCopy = TRUE;
	}
	else
	{
		IICDRegCopy &= ~IIC_CLOCK_BIT;
		IICClockCopy = FALSE;
	}
	if (IICDataCopy)
		IICDRegCopy |= IIC_DATA_BIT;
	else
		IICDRegCopy &= ~IIC_DATA_BIT;
	WriteIICPort(IICDRegCopy);				/* set bits in I2C port */
}


/*****************************************************************************
;+
Function Name   :   SetIICData
Inputs          :   TRUE or FALSE
Outputs         :   None
Side Effects    :   None
Description     :   Sets SDA to required level.
					If Data, then set IIC word to either Clock and Data
					or just Data depending on IICClockCopy state.
					If !Data, IIC Word gets set to either Clock or nowt,
					again depending on IICClockCopy.
					In either case set IICDataCopy.
;-
*****************************************************************************/
void SetIICData(BOOL Data)
{

	ReadIICPort(&IICDRegCopy);
	if (Data)
	{
		IICDRegCopy |= IIC_DATA_BIT;
		IICDataCopy = TRUE;
	}
	else
	{
		IICDRegCopy &= ~IIC_DATA_BIT;
		IICDataCopy = FALSE;
	}
	if (IICClockCopy)
		IICDRegCopy |= IIC_CLOCK_BIT;
	else
		IICDRegCopy &= ~IIC_CLOCK_BIT;
	WriteIICPort(IICDRegCopy);			/* set bits in I2C port */
}

/*****************************************************************************
;+
Function Name   :   IICSetup
Inputs          :   None
Outputs         :   None
Side Effects    :   Clock & Data are high on exit.
Description     :   Setup IIC bus in preparation for a write or read access.
;-
*****************************************************************************/
void IICSetup(void)
{
	SetIICClock(FALSE);				/* Clock low    */
	ClockDelay(5);

	SetIICData(FALSE);				/* and data low */
	ClockDelay(1);

	SetIICClock(TRUE);				/* release clock */
	ClockDelay(5);

	SetIICData(TRUE);				/* & data       */
	ClockDelay(5);
                                                                                	/* i.e. Stop    */
}

/*****************************************************************************
;+
Function Name   :   IICStart
Inputs          :   None
                    Data is high.
Outputs         :   None
Side Effects    :   Clock is high & Data is low on exit.
Description     :   Send a start condition onto IIC bus.
;-
*****************************************************************************/
void IICStart(void)
{
	ClockDelay(5);

	SetIICClock(TRUE);			/* Check clock is high */
	ClockDelay(5);

	SetIICData(FALSE);			/* Data low = Start! */
	ClockDelay(5);
}



/*****************************************************************************
;+
Function Name   :   IICStop
Inputs          :   None
                    Clock low & Data high.
Outputs         :   OK if success
                    IOERR_DRV_I2C_BUS_ERROR if IIC bus error occurred
Side Effects    :   Clock is high & Data is high on exit.
Description     :   Send a Stop condition onto IIC bus.
;-
*****************************************************************************/
LRESULT IICStop(void)
{
	ClockDelay(5);

	SetIICData(FALSE);
	ClockDelay(5);

	SetIICClock(TRUE);
	ClockDelay(5);

	SetIICData(TRUE);					/* Data High = Stop! */
	return(IICWaitForSDA());			/* Wait for IIC SDA Release  */
}

/*****************************************************************************
;+
Function Name   :   IICSendByte
Inputs          :   Byte to send on IIC
                    Clock & Data Indeterminate.
Outputs         :   OK if success
                    IOERR_DRV_I2C_BUS_ERROR if IIC bus error occurred
Side Effects    :   Clock is low & Data is high on exit.
Description     :   Send a byte over IIC bus.
                    The byte is sent MSB first.
;-
*****************************************************************************/
LRESULT IICSendByte(BYTE IICByte)
{
	LONG   	Status;
	int    	BitCount;
	BYTE	Ack;

	ClockDelay(5);

	for (BitCount = 7; BitCount >= 0; BitCount--)
	{
    	SetIICClock(FALSE);

		ClockDelay(2);
		SetIICData((BOOL)((IICByte >> BitCount) & 0x01));

		ClockDelay(3);

		SetIICClock(TRUE);	/* MSB first            */
		ClockDelay(5);
	}

	SetIICClock(FALSE);
	ClockDelay(3);

	SetIICData(TRUE);					/* Release Data */
	ClockDelay(6);

	SetIICClock(TRUE);
	ClockDelay(2);

	ReadIICPort(&IICDRegCopy);
	Ack = IICDRegCopy & IIC_DATA_BIT;

	ClockDelay(3);

	SetIICClock(FALSE);

	if (Ack == (BYTE)0)   				/* Get Acknowledgement from device */
       	Status = RET_OK;
	else
       	Status = IOERR_DRV_I2C_BUS_ERROR;

	return Status;
}

/*****************************************************************************
;+
Function Name   :   IICGetByte
Inputs          :   Pointer to byte.
                    Clock & Data Indeterminate.
Outputs         :   IIC Data byte
Side Effects    :   Clock is low & Data is high on exit.
Description     :   Get a byte from IIC bus.
                    The byte is received MSB first.
;-
*****************************************************************************/
void IICGetByte(PBYTE IICBytePtr)
{
	int    	BitCount;
	BYTE   	RetData;
	BYTE   	Temp;

	RetData = (BYTE)0;

	ClockDelay(5);

	SetIICData(TRUE);					/* Release Data line */
	ClockDelay(2);

	for (BitCount = 7; BitCount >= 0; BitCount--)
	{
    	SetIICClock(FALSE);
		ClockDelay(5);

    	SetIICClock(TRUE);
		ClockDelay(2);

		ReadIICPort(&IICDRegCopy);
		Temp = (BYTE)((IICDRegCopy & IIC_DATA_BIT) >> IIC_DATA_POS);
    	RetData |= Temp << BitCount;	/* Get the bits, MSB first */

		ClockDelay(3);
	}

	SetIICClock(FALSE);
	ClockDelay(5);

	SetIICClock(TRUE);					/* No ACK tells slave to stop */
	ClockDelay(2);

	*IICBytePtr = RetData;				/* Store return byte */
}

/*****************************************************************************
;+
Function Name   :   IICWaitForSDA
Inputs          :   None
Outputs         :   OK if SDA goes high.
                    IOERR_DRV_I2C_NO_ACK if no ack.
Side Effects    :
Description     :   Wait for slave device to send release Data over IIC bus.
                    A software timeout is used for this currently.
;-
*****************************************************************************/
LRESULT IICWaitForSDA(void)
{
	LONG	Status;
	int Delay;

	Delay = (int)2000;

	do
	{
		ReadIICPort(&IICDRegCopy);
		ClockDelay(100);
	}
	while (--Delay > (int)0 && !(IICDRegCopy & IIC_DATA_BIT));	/* loop approx 200mS */

	ReadIICPort(&IICDRegCopy);

	if ((IICDRegCopy & IIC_DATA_BIT) != 0)
    	Status = RET_OK;
	else
    	Status = IOERR_DRV_I2C_NO_ACK;

	return Status;
}

/*****************************************************************************
;+
Function Name   :   IICWriteBytes

Inputs          :   Pointer to Command Structure

Outputs         :   OK if success.
                    IOERR_DRV_I2C_BUS_ERROR if IIC bus error occurred
                    Above returned in IIC_WRITEBUFF.Status.

Side Effects    :   None

Description     :   Outputs a variable number of data bytes to a given
                    device on the IIC bus.

;-
*****************************************************************************/
void IICWriteBytes(IIC_WRITEBUFF_PTR WriteBufferPtr)
{
	LONG    Status;
	int    BytesSent = 0;

	IICStart();							/* Start an access on IIC Bus */

	/*
    	Send Device Address
	*/
	Status = IICSendByte(WriteBufferPtr->SlaveAddress);

	if (!Status)
	{
    	Status = IICWaitForSDA();
    }

	/*
        Send Sub Address
	*/
	if (!Status)
	{
		Status = IICSendByte(WriteBufferPtr->SubAddress);

        if (!Status)
        {
            Status = IICWaitForSDA();
        }
	}

	/*
       	Send Data
	*/
	while	(
				(BytesSent < (int)WriteBufferPtr->WBufferLength)
				&& (Status == RET_OK)
			)
	{
		Status = IICSendByte(WriteBufferPtr->WriteSpace[BytesSent]);

        if (!Status)
        {
        	Status = IICWaitForSDA();
        }

        if (!Status)
        {
        	BytesSent++;
        }
	}

	IICStop();

	WriteBufferPtr->Status = Status;
	WriteBufferPtr->WBufferLength = BytesSent;

}


/*****************************************************************************
;+
Function Name   :   IICReadByte

Inputs          :   Pointer to Command Structure

Outputs         :   OK if success.
                    IOERR_DRV_I2C_BUS_ERROR if IIC bus error occurred
                    Above returned in IIC_READBUFF.Status.

Side Effects    :   None

Description     :   Outputs a variable number of data bytes to a given
                    device on the IIC bus.

;-
*****************************************************************************/
void IICReadByte(IIC_READBUFF_PTR ReadBufferPtr)
{
	LONG   Status;
	BYTE   SAddress;

	IICStart();							/* Start an access on IIC Bus */

	SAddress = (ReadBufferPtr->SlaveAddress | IIC_READ);	/* set read bit */

	Status = IICSendByte(SAddress);

	if (Status == RET_OK)		/* get data */
		IICGetByte((PBYTE) &(ReadBufferPtr->ReturnData));

	IICStop();

	ReadBufferPtr->Status = Status;
}

/*****************************************************************************
        Non Generic Procedures.
*****************************************************************************/

/*****************************************************************************
;+
Function Name   :	ClockDelay
Inputs          :	Number of 1uS periods to delay
Outputs         :	None
Side Effects    :	None
Description     :	Delays for a set number of uS
					The accuracy of this is quite suspect but sufficient!
					Code uses Terminator CONTROL port b7 (TIMER) which
					toggles every 1.185 us.
;-
*****************************************************************************/
void ClockDelay(int Delay)
{
	BYTE	Pulse2;
	WORD	wTimeOut = 1000;

	while (--Delay > 0)
	{
		Pulse2 = Pulse1;
		while (Pulse1 == Pulse2 && wTimeOut)
		{
			ReadControlPort(&Pulse1);
			Pulse1 &= 0x80;
			wTimeOut--;
		}
	}
}

/*****************************************************************************
;+
Function Name   :	ReadControlPort
Inputs          :	Pointer to BYTE
Outputs         :	None
Side Effects    :	None
Description     :	Performs the low-level read access on the Terminator
					CONTROL register.
;-
*****************************************************************************/
void ReadControlPort(PBYTE CtrlBits)
{
	WORD TempBase;

	TempBase = gwBaseReg;
#if OLDREGS
	_asm
	{
		push	dx	 				; save dx
		mov		dx,gwBaseReg		; get configured base addr
		add		dx,ARSENAL_CONTROL	; add CONTROL offset
		in		al,dx				; get CONTROL signal states
		mov		gbTempReg,al		; and return
		pop		dx					; recover dx
	}
#else
	_asm
	{
		push	dx	 				; save dx
		mov		dx,TempBase		; get configured base addr
		add		dx,ARSENAL_CONTROL	; add CONTROL offset
		in		ax,dx				; get CONTROL signal states
		mov		gbTempReg,al		; and return
		pop		dx					; recover dx
	}
#endif
	*CtrlBits = gbTempReg;
}

/*****************************************************************************
;+
Function Name   :	ReadI2CPort
Inputs          :	Pointer to BYTE
Outputs         :	None
Side Effects    :	None
Description     :	Performs the low-level read access on the Terminator I2C
					register.
;-
*****************************************************************************/
void ReadIICPort(PBYTE IICBits)
{
	WORD TempBase;

	TempBase = gwBaseReg;
#if OLDREGS
	_asm
	{
		push	dx	 				; save dx
		mov		dx,gwBaseReg		; get configured base addr
		add		dx,ARSENAL_I2C		; add I2C offset
		in		al,dx				; get I2C signal states
		mov		gbTempReg,al		; and return
		pop		dx					; recover dx
	}
#else
	_asm
	{
		push	dx	 				; save dx
		mov		dx,TempBase		; get configured base addr
		add		dx,ARSENAL_I2C		; add I2C offset
		in		ax,dx				; get I2C signal states
		mov		gbTempReg,al		; and return
		pop		dx					; recover dx
	}
#endif
	*IICBits = gbTempReg;
}

/*****************************************************************************
;+
Function Name   :	WriteI2CPort
Inputs          :	BYTE
Outputs         :	None
Side Effects    :	None
Description     :	Performs the low-level write access on the Terminator I2C
					register.
;-
*****************************************************************************/
void WriteIICPort(BYTE IICBits)
{
	WORD TempBase;

	TempBase = gwBaseReg;
#if OLDREGS
	_asm
	{
		push	dx	 				; save dx
		mov		dx,gwBaseReg 		; get configured base addr
		add		dx,ARSENAL_I2C		; add I2C offset
		mov		al,IICBits			; I2C data and clock to al
		out		dx,al				; write I2C signal states
		pop		dx					; recover dx
	}
#else
	_asm
	{
		push	dx	 				; save dx
		mov		dx,TempBase 		; get configured base addr
		add		dx,ARSENAL_I2C		; add I2C offset
		mov		al,IICBits			; I2C data and clock to al
		out		dx,ax				; write I2C signal states
		pop		dx					; recover dx
	}
#endif
}

/***************************************************************************/
/*  End of TERMIIC.C.													   */
/***************************************************************************/



