/***************************************************************************
;++

Name		:	M2.C
Title		:	M2 control module for Terminator.
Author		:	Nick Jones
Created		:	21/12/92

	Copyright 1992 by VideoLogic Limited. All rights reserved.
	No part of this software, either material or conceptual may be
	copied or distributed, transmitted, transcribed, stored in a
	retrieval system or translated into any human or computer language
	in any form by any means, electronic, mechanical, manual or other-
	wise, or disclosed to third parties without the express written
	permission of VideoLogic Limited, Unit 8, Homepark Industrial
	Estate, King's Langley, Hertfordshire, WD4 8LZ, U.K.

Description	:	Contains code to control M2 chip for Terminator card.

Contents	:

	M2SoftReset()
	M2VSize()
	M2IOffset()
	M2IRegion()
	IClip()
	M2OSize()
	M2StartCapturing()
	M2StopCapturing()
	SourceRectToIregion()
	SetScaleFilterRegs()
	SetIRegionRegs()
	SetWRegionRegs()
	SetM2SPGRegs()
	VidStandard()
	VidType()
	DetectVideo()
	DetectColour()
	ResetM2Group0Regs()
	ResetM2Group1Regs()
	ResetM2Group2Regs()
	ResetM2Group4Regs()
	ResetM2Group5Regs()
	ResetM2Group6and7Regs()

;--
****************************************************************************/

#include "masdefs.h"
#include "ext.h"
#include "drvio.h"
#include "m2.h"
#include "termreg.h"
#include "termiic.h"


/*
	To make 32 bit integer arithmetic more precise, we multiply
	everything by a fixed ammount, PRECISION, and divide it out
	at the end when we do the rounding.
	(Note that PRECISION Squared must fit into a LONG)
*/
#define PRECISION			10000L

/*
	FILTER THRESHOLD = 0.35

	This is the vertical scaling below which filtering is turned
	on, above which it is off. (Interpolation is on all the time.)

	The M2 manual says this figure should be 0.5 but in practice
	0.35 is a better figure.
	(It made MediaSpace movies look much better)

	This is subjective, however, and it	depends on the video material.

	A setting of 0.35 SHARPENS the video between the following sizes :
	133x100 to 188x141 (PAL), and 112x84 to 160x120 (NTSC).

	Note, therefore, that this INCLUDES the 160x120 default size,
	and will increase the resolution of detail but also the resolution
	of certain artefacts, namely 'diagonal jaggeds'.

*/

#define FILTER_THRESHOLD	3500L		/* 0.35 x PRECISION */

/* Source rect logical coords */
#define XSOURCE		640
#define YSOURCE		480


enum
{
	NOCHANGE,
	CHANGE_TYPE,
	CHANGE_STANDARD
};



LRESULT M2IRegion (LPSIZE lpISize);


/* Protos */
static void SourceRectToIregion(void);
static void SetScaleFilterRegs(void);
static void SetIRegionRegs(void);
static void SetWRegionRegs(void);
static void SetM2SPGRegs(void);
static void ResetM2Group0Regs(void);
static void ResetM2Group1Regs(void);
static void ResetM2Group2Regs(void);
static void ResetM2Group4Regs(void);
static void ResetM2Group5Regs(void);
static void ResetM2Group6and7Regs(void);
static LRESULT DetectColour (BYTE bChangeFirst, LPBYTE lpbColourDetected, WORD wDelay);



/* Static globals */

static SIZE		gVSize;			/* Size of M2 input video from 9051		*/
static RECT 	gIRegion;   	/* Source rect in physical coords		*/
static RECT 	gSourceRect;   	/* Source rect in logical coords		*/
static POINT	gIOffset;   	/* Input offset, in sites				*/
static SIZE		gISize;     	/* M2 Input size (VSize minus clipping)	*/
static SIZE		gOSize;     	/* M2 Output size 						*/
static LONG		glXScale;		/* { Overall M2 Scaling (* PRECISION):} */
static LONG		glYScale;       /* { scale = Output/Input * PRECISION } */
static BYTE 	gHFiltBand[5];  /* RegRHSCALE bands						*/
static BYTE 	gHFiltValue[5]; /* RegRHFILT values						*/
static BYTE		gbVidStandard;
static BYTE		gbVidType;


/*****************************************************************************
;+
Function Name   :   M2SoftReset
Inputs          :
Outputs         :
Side Effects    :
Discription     :   Resets M2 registers and static variables
;-
*****************************************************************************/
LRESULT M2SoftReset (void)
{
	LONG lRet;

	/* Reset globals */
	gVSize.cx = (int)676; /* Active pixels per line from 9051 */
	gVSize.cy = (int)283; /* 283=PAL 240=NTSC */
	gbVidStandard = (BYTE)DRV_STANDARD_PAL;
	gbVidType = (BYTE)DRV_TYPE_COMPOSITE;

	gSourceRect.left = (int)0;
	gSourceRect.top = (int)0;
	gSourceRect.right = (int)(XSOURCE-1);
	gSourceRect.bottom = (int)(YSOURCE-1);

	SourceRectToIregion();

	gIOffset.x = (int)5;	/* PAL */
	gIOffset.y = (int)23;	/* PAL */

	gISize.cx = gVSize.cx;
	gISize.cy = gVSize.cy;

	gOSize.cx = (int)160; /* Default output size */
	gOSize.cy = (int)120;

	glXScale = PRECISION; /* Scaling 1.0 for now */
	glYScale = PRECISION;

	/* RegRHSCALE bands */
	gHFiltBand[0] = (BYTE) 90;
	gHFiltBand[1] = (BYTE) 131;
	gHFiltBand[2] = (BYTE) 172;
	gHFiltBand[3] = (BYTE) 214;
	gHFiltBand[4] = (BYTE) 255;

	/* RegRHFILT values */
	gHFiltValue[0] = (BYTE) 0x5D;
	gHFiltValue[1] = (BYTE) 0x6B;
	gHFiltValue[2] = (BYTE) 0x6A;
	gHFiltValue[3] = (BYTE) 0x79;
	gHFiltValue[4] = (BYTE) 0x78;

    /* Reset M2 registers by functional group */
	ResetM2Group0Regs();
	ResetM2Group1Regs();
    ResetM2Group2Regs();
    ResetM2Group4Regs();
    ResetM2Group5Regs();
    ResetM2Group6and7Regs();

    /* Now send data to M2 */
    lRet = SendChanges();

    return lRet;

}/*M2SoftReset*/


/*****************************************************************************
;+
Function Name   :   M2VSize
Inputs          :
Outputs         :
Side Effects    :
Discription     :   Sets the size of the incoming video
;-
*****************************************************************************/
LRESULT M2VSize (LPSIZE lpVSize)
{
	LONG lRet;
	SIZE ISize;

	gVSize.cx = lpVSize->cx;
	gVSize.cy = lpVSize->cy;

	/* Map Vsize onto Iregion for now */
	ISize = gVSize;
	lRet = M2IRegion ((LPSIZE) &ISize);

    return lRet;

}/*M2VSize*/


/*****************************************************************************
;+
Function Name   :   M2IOffset
Inputs          :
Outputs         :
Side Effects    :
Discription     :   Sets the input offset of the incoming video
;-
*****************************************************************************/
LRESULT M2IOffset (LPPOINT lpIOffset)
{
	LONG lRet;

	if (!lpIOffset)
		return IOERR_DRV_BAD_PARAMS;

	gIOffset.x = lpIOffset->x;
	gIOffset.y = lpIOffset->y;

    SetIRegionRegs();
	lRet = SendChanges();

    return lRet;

}/*M2IOffset*/


/*****************************************************************************
;+
Function Name   :   M2IRegion
Inputs          :
Outputs         :
Side Effects    :
Discription     :   Sets the M2 input region
;-
*****************************************************************************/
LRESULT M2IRegion (LPSIZE lpISize)
{
    LONG lRet;

	gISize.cx = lpISize->cx;
	gISize.cy = lpISize->cy;

    /* Rectangle calculations */
    SetScaleFilterRegs();
    SetIRegionRegs();
	lRet = SendChanges();

    return lRet;

}/* M2IRegion */


/*****************************************************************************
;+
Function Name   :   M2IClip
Inputs          :   DWORD dwClipDeltas = top left bottom right (chars)
Outputs         :
Side Effects    :
Discription     :   Sets the input clipping
;-
*****************************************************************************/
LRESULT M2IClip (DWORD dwClipDeltas)
{
	BYTE bX1,bY1,bX2,bY2;
	DWORD dwX1,dwY1,dwX2,dwY2;
	int iX1, iY1, iX2, iY2;

    dwX1 = dwClipDeltas & 0x000000FFL;
    dwY1 = (dwClipDeltas & 0x0000FF00L)>>8;
    dwX2 = (dwClipDeltas & 0x00FF0000L)>>16;
    dwY2 = (dwClipDeltas & 0xFF000000L)>>24;

    bX1 = (BYTE)dwX1;
    bY1 = (BYTE)dwY1;
    bX2 = (BYTE)dwX2;
    bY2 = (BYTE)dwY2;

    /* sign extend */
    iX1 = (int)bX1;
    iY1 = (int)bY1;
    iX2 = (int)bX2;
    iY2 = (int)bY2;

	gIOffset.x += iX1;
	gIOffset.y += iY1;
	gISize.cx += bX2;
	gISize.cy += bY2;

	return RET_OK;

}/*M2IClip*/


/*****************************************************************************
;+
Function Name   :   M2OSize
Inputs          :
Outputs         :
Side Effects    :
Discription     :   Sets the M2 output size
;-
*****************************************************************************/
LRESULT M2OSize (LPSIZE lpOSize)
{
    LONG lRet;

	gOSize.cx = lpOSize->cx;
	gOSize.cy = lpOSize->cy;

    /* Rectangle calculations */
    SetScaleFilterRegs();
    SetIRegionRegs();
    SetWRegionRegs();

    /* SetSPG registers */
    SetM2SPGRegs();

	lRet = SendChanges();

    return lRet;

}/* M2OSize */


/*****************************************************************************
;+
Function Name   :   M2StartCapturing
Inputs          :
Outputs         :
Side Effects    :
Discription     :   Enables live video capturing to fieldstores
;-
*****************************************************************************/
LRESULT M2StartCapturing (void)
{
	LONG lRet;
    /*
    	CAPUSE 0x03

		Odd fields into FS0
		Nothing into FS1
    	Don't trigger on latest.

		INTENB 0x90
		Enable global ints
		M2 interrupts on FS0 not capturing
    */
    ForceReg ((WORD)MAP_CAPUSE, (WORD)0x03);
    ForceReg ((WORD)MAP_INTENB, (WORD)0x90);
    lRet = SendChanges();

    return lRet;

}/*M2StartCapturing*/


/*****************************************************************************
;+
Function Name   :   M2StopCapturing
Inputs          :
Outputs         :
Side Effects    :
Discription     :   Disables live video capturing to fieldstores.
					(Video freezes)
;-
*****************************************************************************/
LRESULT M2StopCapturing (void)
{
	LONG lRet;

    ForceReg ((WORD)MAP_CAPUSE, (WORD)0x00);
    ForceReg ((WORD)MAP_INTENB, (WORD)0x00);
    lRet = SendChanges();

    return lRet;

}/*M2StopCapturing*/


/*****************************************************************************
;+
Function Name   :   SourceRectToIregion
Inputs          :
Outputs         :
Side Effects    :
Discription     :   Converts Source rectangle from logical to physical coords
					(Maps Source rectangle onto input region)
;-
*****************************************************************************/
static void SourceRectToIregion(void)
{
	gIRegion.left	= (int)((gVSize.cx * (LONG)gSourceRect.left)   / (LONG)XSOURCE);
	gIRegion.top	= (int)((gVSize.cy * (LONG)gSourceRect.top)    / (LONG)YSOURCE);
	gIRegion.right	= (int)((gVSize.cx * (LONG)gSourceRect.right)  / (LONG)XSOURCE);
	gIRegion.bottom = (int)((gVSize.cy * (LONG)gSourceRect.bottom) / (LONG)YSOURCE);

}/* SourceRectToIregion */


/*****************************************************************************
;+
Function Name	:	SetScaleFilterRegs
Inputs			:	Uses globals glXScale and glYScale
Side Effects    :   Register Map is updated.
Discription     :   Calculates the register values and updates the map

	To make 32 bit integer arithmetic more precise, we multiply
	everything by a fixed ammount, PRECISION, and divide it out
	at the end when we do the rounding.
	(Note that PRECISION Squared must fit into a LONG)

;-
*****************************************************************************/
static void SetScaleFilterRegs(void)
{
	BYTE bRegRVScale,bRegRHScale;/* Input vertical/horizontal scale registers */
	BYTE bRegRVFilt,bRegRHFilt; /* Input vertical/horizontal filter registers */
	BYTE bRegDVScale;			/* Output vertical scale register */
	LONG lPreciseValue;
	int iFilter;

    /* Prevent divide-by-zero crash */
    if (!gISize.cx)
    	return;

    if (!gISize.cy)
    	return;

	glXScale = (LONG) ((PRECISION * gOSize.cx)/gISize.cx);
	glYScale = (LONG) ((PRECISION * gOSize.cy)/gISize.cy);

	/* VERTICAL SCALING */

	if (glYScale > (1L * PRECISION)) /* if scale > 1.0000 */
	{
		/* bRegDVScale : enable output line doubling */
		lPreciseValue  = (PRECISION * ((512L * PRECISION) / glYScale))
								- (256L * PRECISION) + (PRECISION/2L);

		/* Remove precision factor and limit to 255 */
		if (lPreciseValue > (255*PRECISION))
			bRegDVScale = (BYTE)255; /* Limit to 255 */
		else
			bRegDVScale = (BYTE)(lPreciseValue / PRECISION);

		/* Disable input line dropping */
		bRegRVScale = (BYTE)255;
	}
	else
    {
    	/* Enable input line dropping */
        lPreciseValue = (glYScale * 256L) + (PRECISION/2L);

		/* Remove precision factor and limit to 255 */
		if (lPreciseValue > (255L * PRECISION))
			bRegRVScale = (BYTE)255;
		else
			bRegRVScale = (BYTE)(lPreciseValue / PRECISION);

        /* Disable output line doubling */
		bRegDVScale = (BYTE)255;
	}


	if (bRegDVScale != (BYTE) 255)
		bRegRVFilt = (BYTE) 0x04;	/* Interpolation OFF filter OFF */
	else
		if (glXScale <= FILTER_THRESHOLD)  /* Scaling <= 0.35 */
			bRegRVFilt = (BYTE) 0x07;	/* Interpolation ON filter ON */
		else
			bRegRVFilt = (BYTE) 0x06;	/* Interpolation ON filter OFF */

	/* HORIZONTAL SCALING */

	if (glXScale >= 1L * PRECISION)
		bRegRHScale = (BYTE)255;
	else
	{
        lPreciseValue = (glXScale * 256L) /* + (PRECISION/2L) */;

		/* Remove precision factor and limit to 255 */
		if (lPreciseValue > (255L * PRECISION))
			bRegRHScale = (BYTE)255;
		else
			bRegRHScale = (BYTE)((lPreciseValue / PRECISION)-1);
	}

	/* Get horizontal filter band and value from look-up table */
	iFilter = (int)0;

	while ( gHFiltBand[iFilter] < bRegRHScale )
		iFilter++;

	bRegRHFilt = gHFiltValue[iFilter];

    /* Scaling fix */
    switch (gOSize.cx)
    {
    	case 160 :
    	case 224 :
    	case 232 :
    	case 240 :
    	case 248 :
    	case 256 :
    	case 264 :
    	case 272 :
    		bRegRHScale++;
    		break;

    	case 544 :
    	case 552 :
    		bRegRHScale--;
    		break;
    }

	/* Now set the registers */
	SetReg (MAP_RVSCALE,	(WORD)bRegRVScale);
	SetReg (MAP_RHSCALE,	(WORD)bRegRHScale);
	SetReg (MAP_RVFILT,		(WORD)bRegRVFilt);
	SetReg (MAP_RHFILT,		(WORD)bRegRHFilt);
	SetReg (MAP_DVSCFS2,	(WORD)bRegDVScale);
	SetReg (MAP_DVSCFS1,	(WORD)bRegDVScale);

}/* SetScaleFilterRegs */


/*****************************************************************************
;+
Function Name   :    	SetIRegionRegs
Inputs          :   	Global gIRegion
Outputs         :  		None
Side Effects    :   	Register Map is updated.
Description     :   	Sets the following registers:
						RHSTART RHEND RVSTART RVEND
;-
*****************************************************************************/
static void SetIRegionRegs(void)
{
	BYTE bRegRHSTART;
	BYTE bRegRHEND;
	WORD wRegRVSTART;
	WORD wRegRVEND;

	/* Convert to sites, add offset, round down */
	bRegRHSTART = (BYTE) ((gIRegion.left/4) + gIOffset.x);

	/* Convert to sites, add offset, round up */
	bRegRHEND = (BYTE) ( ((gIRegion.right + 4) /4) + gIOffset.x);

	wRegRVSTART = (WORD) (gIRegion.top + 1 + gIOffset.y);
	wRegRVEND = (WORD) (gIRegion.bottom + 1 + gIOffset.y);

	SetReg ((WORD)MAP_RHSTART, (WORD)bRegRHSTART);
	SetReg ((WORD)MAP_RHEND, (WORD)bRegRHEND);
	SetReg ((WORD)MAP_RVSTART, wRegRVSTART);
	SetReg ((WORD)MAP_RVEND, wRegRVEND);

}/*SetIRegionRegs*/


/*****************************************************************************
;+
Function Name   :   SetWRegionRegs
Inputs          :
Outputs         :
Side Effects    :   .
Discription     :
;-
*****************************************************************************/
static void SetWRegionRegs(void)
{
	SetReg ((WORD) MAP_RTLXFS1, (WORD)0);
	SetReg ((WORD) MAP_RBRXFS1, gOSize.cx);
	SetReg ((WORD) MAP_RTLYFS1, (WORD)0);
	SetReg ((WORD) MAP_RBRYFS1, gOSize.cy);

	SetReg ((WORD) MAP_RTLXFS2, (WORD)0);
	SetReg ((WORD) MAP_RBRXFS2, gOSize.cx);
	SetReg ((WORD) MAP_RTLYFS2, (WORD)0);
	SetReg ((WORD) MAP_RBRYFS2, gOSize.cy);

}/*SetWRegionRegs*/




/*****************************************************************************
;+
Function Name   :	SetM2SPGRegs
Inputs          :
Outputs         :	Register Map updated
Side Effects    :
Description     :   Sets M2's internal SPG.

		Calculations are based on M2 SPG code written for MediaSpace.
;-
*****************************************************************************/
static void SetM2SPGRegs(void)
{
    WORD PixStart;
    WORD PixPLine;
    WORD PicEnd;
    WORD HSEnd;
    WORD LinPFld;
    WORD LinPPic;
    WORD VSStart;
    WORD VSEnd;

    WORD RegPixStart;
    WORD RegPixPLine;
    WORD RegPicEnd;
    WORD RegHSEnd;
    WORD RegLinPFld;
    WORD RegLinPPic;
    WORD RegVSStart;
    WORD RegVSEnd;

	/* Assume  gOSize.cx = 160  gOSize.cy = 120 */
	/* Horizontal timing */                 /*** Decimal ***/
	PixStart = (WORD) 8;                    	/*   8 */
    PicEnd = (WORD) (PixStart + gOSize.cx);		/* 168 */
    HSEnd = (WORD) 4;                       	/*   4 */
    PixPLine = PicEnd;                      	/* 168 */

    /* Vertical timing */
    LinPPic = (WORD) (gOSize.cy + 4);			/* 124 */
    LinPFld = (WORD) (LinPPic + 4);			    /* 128 */
    VSStart = (WORD) 4;                     	/*   4 */
    VSEnd = (WORD) 8;                       	/*   8 */

	/* M2 registers */
	RegPixStart = (WORD)(PixStart-5);			/*   3 */
	RegPixPLine = (WORD)(PixPLine-2);			/* 166 */
	RegPicEnd = (WORD)(PicEnd-2);				/* 166 */
	RegHSEnd = (WORD)(HSEnd-2);					/*   2 */
	RegLinPPic = (WORD)(LinPPic+2);				/* 126 */
	RegLinPFld = (WORD)(LinPFld-2);				/* 126 */
	RegVSStart = (WORD)(VSStart-2);				/*   2 */
	RegVSEnd = (WORD)((VSEnd-2)&0x0F);			/*   6 */
    
    RegLinPPic+=4;
    RegLinPFld+=4;
	

	/* Set registers */                      /***   Hex   ***/
	SetReg ((WORD)MAP_PIXPLINE, RegPixPLine);   /* 0x00A6 */
	SetReg ((WORD)MAP_PICSTART, RegPixStart);   /* 0x0003 */
	SetReg ((WORD)MAP_PICEND, RegPicEnd);       /* 0x00A6 */
	SetReg ((WORD)MAP_HSEND, RegHSEnd);         /* 0x02   */
	SetReg ((WORD)MAP_LINPFLD, RegLinPFld);     /* 0x007E */
	SetReg ((WORD)MAP_LINPPIC, RegLinPPic);     /* 0x007E */
	SetReg ((WORD)MAP_VSSTART, RegVSStart);     /* 0x0002 */
	SetReg ((WORD)MAP_VSEND, RegVSEnd);         /* 0x06   */

    /* Set DPIXNO to MAP_PIXPLINE setting */
	SetReg ((WORD)MAP_DPIXNO,  RegPixPLine);    /* 0x00A6 */

}/*SetM2SPGRegs*/


/*****************************************************************************
;+
Function Name   :	VidStandard
Inputs          :   Video Standard
Outputs         :
Side Effects    :
Description     :   Sets the video standard
;-
*****************************************************************************/
LRESULT VidStandard (WORD wStandardID)
{
	LONG lRet;
	SIZE VSize;

	switch(wStandardID)
	{
		case DRV_STANDARD_PAL:
			VSize.cy = VID_LINES_9051_PAL;
			gIOffset.x = (int)5;
			gIOffset.y = (int)23;
			break;

		case DRV_STANDARD_NTSC:
			VSize.cy = VID_LINES_9051_NTSC;
			gIOffset.x = (int)6;
			gIOffset.y = (int)16;
			break;

		default:
			return IOERR_DRV_BAD_PARAMS;
	}

    VSize.cx = (int)676; /* Active pixels per line from 9051 */
    gbVidStandard = (BYTE) wStandardID;

	lRet = M2VSize(&VSize);

	if (lRet)
		return lRet;

	lRet = SetDMSD (wStandardID, (WORD)gbVidType);

	/* TELL VXD */


	return lRet;

}/*VidStandard*/


/*****************************************************************************
;+
Function Name   :	VidType
Inputs          :   Video Type
Outputs         :
Side Effects    :
Description     :   Sets the video type
;-
*****************************************************************************/
LRESULT VidType (WORD wTypeID)
{
	LONG lRet;

	gbVidType = (BYTE)wTypeID;

	lRet = SetDMSD ((WORD)gbVidStandard, wTypeID);

	return lRet;

}/*VidType*/


/*****************************************************************************
;+
Function Name   :	DetectVideo
Inputs          :   lpwStandard should point to current video standard (WORD)
                    lpwType should point to current video type (WORD)
                    If either is Zero, the values are taken from system.ini
Outputs         :
Side Effects    :
Description     :   Auto detects the video standard/type,
					and sets everything up correctly.
					Assumption :
    				If we've got colour, the standard and type must be OK.

;-
*****************************************************************************/
LRESULT DetectVideo(LPWORD lpwStandard, LPWORD lpwType)
{
	LONG lRet;
	SIZE VSize;
	WORD wDMSDStatus;
	BYTE bColourDetected;
	BYTE bHNotLocked;
	WORD wSaveType;
	WORD wDelay;	/* DMSD Settling time */

    /* Read status register to see what is going on */
	lRet = RegIn ((WORD) DEVICE_MAP, (WORD)MAP_DMSD_STATUS, (LPWORD)&wDMSDStatus);

    if (lRet)
    	return(lRet);

    bColourDetected = (BYTE)(wDMSDStatus & 0x04);

    /* Fast exit if all is OK */
    if (bColourDetected)
    	return RET_OK;

    if (lpwStandard)
    	gbVidStandard = (BYTE) *lpwStandard;
    else
    	gbVidStandard = (BYTE)DRV_STANDARD_PAL;
    if (lpwType)
    	gbVidType = (BYTE) *lpwType;
    else
		gbVidType = (BYTE)DRV_TYPE_COMPOSITE;

    /* Save start type */
    wSaveType = gbVidType;

    bHNotLocked = (BYTE)(wDMSDStatus & 0x40);
    wDelay = (WORD)40; /* 40mS settling time for DMSD */

   	/* If not locked, toggle standard and try again */
    if (bHNotLocked)
     	lRet = DetectColour ((BYTE)CHANGE_STANDARD, (LPBYTE)&bColourDetected, wDelay);

	if (lRet)
		return lRet;

	/* If no colour, toggle type and try again */
    if (!bColourDetected)
    {
    	wDelay+=40;
    	lRet = DetectColour ((BYTE)CHANGE_TYPE, (LPBYTE)&bColourDetected, wDelay);
    }

	if (lRet)
		return lRet;

	/* Toggle standard and try again */
    if (!bColourDetected)
	    lRet = DetectColour ((BYTE)CHANGE_STANDARD, (LPBYTE)&bColourDetected, wDelay);

	if (lRet)
		return lRet;

	/* Toggle type and try again */
    if (!bColourDetected)
    {
    	wDelay+=40;
    	lRet = DetectColour ((BYTE)CHANGE_TYPE, (LPBYTE)&bColourDetected, wDelay);
    }

	if (lRet)
		return lRet;

	/* Toggle standard and try again */
    if (!bColourDetected)
	    lRet = DetectColour ((BYTE)CHANGE_STANDARD, (LPBYTE)&bColourDetected, wDelay);

	if (lRet)
		return lRet;

    /*
    	We have now cycled round everything,
    	with no results, so change down a gear
    	and try again more slowly
    */

	/* Toggle type and try again */
    if (!bColourDetected)
    {
    	wDelay+=200;
    	lRet = DetectColour ((BYTE)CHANGE_TYPE, (LPBYTE)&bColourDetected, wDelay);
    }

	if (lRet)
		return lRet;

	/* Toggle standard and try again */
    if (!bColourDetected)
    {
    	wDelay+=80;
	    lRet = DetectColour ((BYTE)CHANGE_STANDARD, (LPBYTE)&bColourDetected, wDelay);
    }

	if (lRet)
		return lRet;

	/* Toggle type and try again */
    if (!bColourDetected)
    {
    	wDelay+=80;
     	lRet = DetectColour ((BYTE)CHANGE_TYPE, (LPBYTE)&bColourDetected, wDelay);
    }

    if (!bColourDetected)
    {
    	/* Still if no color detected, set to start condition */
	    gbVidStandard = DRV_STANDARD_PAL; /* set to PAL to prevent hang */
	    gbVidType = (BYTE)wSaveType;

		lRet = SetDMSD ((WORD)gbVidStandard, (WORD)gbVidType);
    }

	if (lRet)
		return lRet;

    /* Now set the M2 */
	switch(gbVidStandard)
	{
		case DRV_STANDARD_PAL:
			VSize.cy = VID_LINES_9051_PAL;
			gIOffset.x = (int)5;
			gIOffset.y = (int)23;
			break;

		case DRV_STANDARD_NTSC:
			VSize.cy = VID_LINES_9051_NTSC;
			gIOffset.x = (int)6;
			gIOffset.y = (int)16;
			break;

		default:
			return IOERR_DRV_BAD_PARAMS;
	}

    VSize.cx = (int)676; /* Active pixels per line from 9051 */

    /* Set M2 video input size */
	lRet = M2VSize(&VSize);

	if (lRet)
		return lRet;

	/* Tell VXD of the frame-rate  (dwMicroSecPerFrameTick) */
	/* But StartStream will automatically send the ccb ? */

	if (lRet)
		return lRet;

	/* Update results if they were requested */
	if (lpwStandard)
    	*lpwStandard = (WORD) gbVidStandard;

	if (lpwType)
    	*lpwType = (WORD) gbVidType;

	if (!bColourDetected)
		return IOERR_DRV_NOVIDEO; /* Can't detect any video */
    else
		return lRet;

}/*DetectVideo*/


/*****************************************************************************
;+
Function Name   :	DetectColour
Inputs          :
Outputs         :
Side Effects    :
Description     :
;-
*****************************************************************************/
static LRESULT DetectColour (BYTE bChangeFirst, LPBYTE lpbColourDetected, WORD wDelay)
{
	WORD wDMSDStatus;
	LONG lRet;

    switch (bChangeFirst)
    {
    	case CHANGE_STANDARD:
			if (gbVidStandard == DRV_STANDARD_PAL)
				gbVidStandard = (BYTE)DRV_STANDARD_NTSC;
			else
				gbVidStandard = (BYTE)DRV_STANDARD_PAL;
    		break;

    	case CHANGE_TYPE:
			if (gbVidType == DRV_TYPE_COMPOSITE)
				gbVidType = (BYTE)DRV_TYPE_SVIDEO;
			else
				gbVidType = (BYTE)DRV_TYPE_COMPOSITE;
    		break;
    }

	lRet = SetDMSD ((WORD)gbVidStandard, (WORD)gbVidType);

    if (lRet)
    	return(lRet);

    /* Delay */
	DelayTime(wDelay);

    /* Read status register to see what happened */
	lRet = RegIn ((WORD) DEVICE_MAP, (WORD)MAP_DMSD_STATUS, (LPWORD)&wDMSDStatus);

    if (lRet)
    	return(lRet);

    /* Update result if it was requested */
    if (lpbColourDetected)
	    *lpbColourDetected = (BYTE)(wDMSDStatus & 0x04);

    return lRet;

}/*DetectColour*/


/*****************************************************************************
;+
Function Name   :	ResetM2Group0Regs
Inputs          :
Outputs         :
Side Effects    :
Description     :   Resets the M2 registers 0x00 to 0x0F
;-
*****************************************************************************/
static void ResetM2Group0Regs(void)
{
	/* MAP_IDNUM is read-only */

	/* Disable interrupts */
	ForceReg (MAP_INTENB, (WORD)0x00);

	/* Cancel ready interrupts */
	ForceReg (MAP_INTFLG, (WORD)0xFF);

	/* MAP_INTRDY is read-only */

	/* MAP_STATUS is read-only */

}/*ResetM2Group0Regs*/


/*****************************************************************************
;+
Function Name   :	ResetM2Group1Regs
Inputs          :
Outputs         :
Side Effects    :
Description     :   Resets the M2 registers 0x10 to 0x1F
;-
*****************************************************************************/
static void ResetM2Group1Regs(void)
{
	/*
		VIDSEL : 0x90 (1001 0000b)

		Base video bus o/p :
			source : don't care (base bus o/p is disabled anyway)
			drive : disable base video output
			clock : output disabled

		Feature video bus o/p :
			source : merged video data
			drive : enable feature video data output
			clock : output disabled

		Video source :
			base video
	*/
	ForceReg (MAP_VIDSEL, (WORD)0x90);

	/* FSDATA Only used for download/upload */
	/* FSRESET Only used for download/upload */

	/* Disable processor access to fieldstores */
	ForceReg (MAP_FSMODE, (WORD)0x00);

}/*ResetM2Group1Regs*/



/*****************************************************************************
;+
Function Name   :	ResetM2Group2Regs
Inputs          :
Outputs         :
Side Effects    :
Description     :   Resets the M2 registers 0x20 to 0x2F
;-
*****************************************************************************/
static void ResetM2Group2Regs(void)
{

	ForceReg (MAP_DVPRIO,	(WORD)0x04);

	/*
		MCTL	0x11 (0001 0001b)

		bit0[1] FS0 enabled

			[0] FS selection : (dont care because dynamic only)
			[0]

			[0] FS1 disabled

		bit4[1] FS0 merge enable:
				 1 for o/p to feature bus (default)
				 	(via VIDSEL) if VScale <= 1.0
				 	because does not have 4 extra lines on end,
				 	so capturing is faster.

				 0 for o/p to display bus
				 	if VScale > 1.0 because only display bus
				 	can line double, but extra 4 blank lines on end.

			[0] FS1 merge disabled
			[0] Priority FS0
		bit7[0] FS Static mode
	*/

	ForceReg (MAP_MCTL,	 (WORD)0x01);

/*
    #if USE_DISPLAY_BUS
		ForceReg (MAP_MCTL,	 (WORD)0x01);
	#else
		ForceReg (MAP_MCTL,	 (WORD)0x11);
	#endif
*/

	ForceReg (MAP_DISPCTL, (WORD)0x01);

	/*
		MAP_DPIXNO is set by
		SetM2SPGRegs() in group 4 regs.
	*/
	/*ForceReg (MAP_DPIXNO,  (WORD)0xA6);*/

	ForceReg (MAP_BCKGNDY, (WORD)0x00);
	ForceReg (MAP_BCKGNDU, (WORD)0x00);
	ForceReg (MAP_BCKGNDV, (WORD)0x00);

	/* Disable external merge */
	ForceReg ((WORD)MAP_XMCTRL, 0xC0);

}/*ResetM2Group2Regs*/





/*****************************************************************************
;+
Function Name   :	ResetM2Group4Regs
Inputs          :
Outputs         :
Side Effects    :
Description     :   Resets the M2 registers 0x40 to 0x4F
;-
*****************************************************************************/
static void ResetM2Group4Regs(void)
{
	/*
		SetM2SPGRegs() sets following group 4 regs :
			PIXPLINE
			PICSTART
			PICEND
			HSEND
			LINPFLD
			LINPPIC
			VSSTART
			VSEND
		(SetM2SPGRegs() also sets DPIXNO)
	*/
	SetM2SPGRegs();

	/* Force the update */
	CHANGE(MAP_PIXPLINE);
	CHANGE(MAP_PICSTART);
	CHANGE(MAP_PICEND);
	CHANGE(MAP_HSEND);
	CHANGE(MAP_LINPFLD);
	CHANGE(MAP_LINPPIC);
	CHANGE(MAP_VSSTART);
	CHANGE(MAP_VSEND);

	ForceReg ((WORD)MAP_SPGMODE,	(WORD)0x03);

}/*ResetM2Group4Regs*/





/*****************************************************************************
;+
Function Name   :	ResetM2Group5Regs
Inputs          :
Outputs         :
Side Effects    :
Description     :   Resets the M2 registers 0x50 to 0x5F
;-
*****************************************************************************/
static void ResetM2Group5Regs(void)
{
	/*
		SetIRegionRegs() sets the following group 5 registers:
			RHSTART
			RHEND
			RVSTART
			RVEND
	*/
	SetIRegionRegs();

	/* Force the update */
	CHANGE(MAP_RHSTART);
	CHANGE(MAP_RHEND);
	CHANGE(MAP_RVSTART);
	CHANGE(MAP_RVEND);



	/*
		SetScaleFilterRegs() sets the following registers:
			RVSCALE
			RHSCALE
			RVFILT
			RHFILT
			DVSCFS2
			DVSCFS1
	*/
	SetScaleFilterRegs();

	/* Force the update */
	CHANGE(MAP_RVSCALE);
	CHANGE(MAP_RHSCALE);
	CHANGE(MAP_RVFILT);
	CHANGE(MAP_RHFILT);
	CHANGE(MAP_DVSCFS2);
	CHANGE(MAP_DVSCFS1);

    /*
    	FLDSYN	0x01 (strict alternate)

    	Strict alternate can cope with any field
    	types at the input, but may not always select
    	the same type of field.

    	Phased alternate may not work with CDI
    	because field types from CDI don't alternate,
    	but the advantage is that it always selects the
    	same field type.DVA uses phased alternate.

    	Action : if we can prove that Phased alternate
    	works with CDI then we should use it.
    */
    ForceReg ((WORD)MAP_FLDSYN, (WORD)0x01);


    /*
    	CAPUSE 0x03

		Odd fields into FS0
		Nothing into FS1
    	Don't trigger on latest.
    */
    ForceReg ((WORD)MAP_CAPUSE, (WORD)0x03);

    ForceReg ((WORD)MAP_RVSODD, (WORD)0x2D);
    ForceReg ((WORD)MAP_RVSEVEN, (WORD)0x11);



}/*ResetM2Group5Regs*/




/*****************************************************************************
;+
Function Name   :	ResetM2Group6and7Regs
Inputs          :
Outputs         :
Side Effects    :
Description     :   Resets the M2 registers 0x60 to 0x7F
;-
*****************************************************************************/
static void ResetM2Group6and7Regs(void)
{
	/*
		SetWRegionRegs() sets the following registers:

			RTLXFS1
			RBRXFS1
			RTLYFS1
			RBRYFS1

			RTLXFS2
			RBRXFS2
			RTLYFS2
			RBRYFS2

	*/
    SetWRegionRegs();

	/* Force the update */
	CHANGE(MAP_RTLXFS1);
	CHANGE(MAP_RBRXFS1);
	CHANGE(MAP_RTLYFS1);
	CHANGE(MAP_RBRYFS1);
	CHANGE(MAP_RTLXFS2);
	CHANGE(MAP_RBRXFS2);
	CHANGE(MAP_RTLYFS2);
	CHANGE(MAP_RBRYFS2);

	ForceReg ((WORD)MAP_FSXFS1, (WORD)0);

	ForceReg ((WORD)MAP_FSYFS1, (WORD)4);
/*
    #if USE_DISPLAY_BUS
		ForceReg ((WORD)MAP_FSYFS1, (WORD)4);
	#else
		ForceReg ((WORD)MAP_FSYFS1, (WORD)0);
	#endif
*/
	ForceReg ((WORD)MAP_FSXFS2, (WORD)0);
	ForceReg ((WORD)MAP_FSYFS2, (WORD)0);


    /*
    	Note : DVSCFS2 and DVSCFS1 regs were set in
    	SetScaleFilterRegs() called from ResetM2Group5Regs().
    */

	ForceReg ((WORD)MAP_DVCTLFS1, (WORD)0x03);
	ForceReg ((WORD)MAP_DVCTLFS2, (WORD)0x03);


}/*ResetM2Group6and7Regs*/


/***** End of M2.C *****/



