/***************************************************************************
;++

Name		:	DRVIO.C
Title		:	Capture Driver's Low level IO layer
Author		:	Nick Jones
Created		:	21/12/92

	Copyright 1992 by VideoLogic Limited. All rights reserved.
	No part of this software, either material or conceptual may be
	copied or distributed, transmitted, transcribed, stored in a
	retrieval system or translated into any human or computer language
	in any form by any means, electronic, mechanical, manual or other-
	wise, or disclosed to third parties without the express written
	permission of VideoLogic Limited, Unit 8, Homepark Industrial
	Estate, King's Langley, Hertfordshire, WD4 8LZ, U.K.

Description	:	Terminator I/O and low level commands.

Contents :

	DriverIO()			<-- Entry Point
	DelayTime()
	InitStatics()
	ResetArsenal()
	ResetDMSD()
	SetDMSD()
	StartAutoSense()
	EndAutoSense()
	AutoDetectTimerFunc()
	RegOut()
	RegIn()
	PortWrite()
	PortRead()
	M2Write()
	M2Read()
	I2CWrite()
	WriteAllI2CRegs()
	I2CPrimeMap()
	I2CRead()
	VXDWrite()
	VXDRead()
	SetReg()
	ForceReg()
	FindReg()
	GetRegMap()
	SendChanges()
	VXDInit()
	VXDSetup()
	VXDSetupStream()
	SetupCCB()
	VXDCapture()
	VXDStreamStart()
	VXDStreamStop()
	VXDKill()
	VXDAddBuffer()
	GetErrorString()
	ProcessError()
	PrimeCapture()



;--
****************************************************************************/
#include "masdefs.h"
#include "ext.h"
#include "drvio.h"
#include "termreg.h"
#include "m2.h"
#include "termiic.h"
#include "caputl.h"
#include "conio.h"

/* static globals */

static BOOL				bInitNeeded = TRUE;
static IIC_WRITEBUFF	I2CWriteBuff;
static IIC_READBUFF		I2CReadBuff;
static WORD				gwBoardID;


/* InitStatics Macros */
#define REG(R,A,S) \
    RegMap[R].bAddress = (BYTE) (A); \
    RegMap[R].bStatus = (BYTE) (S); \
    RegMap[R].wData = (WORD)0;

/* internal protos */
static void I2CPrimeMap(WORD wMapAddr);




/*****************************************************************************
;+
Function Name	:	DelayTime
Inputs			:
Outputs			:
Side Effects    :
Discription     :   Delays the required number of milliseconds
;-
*****************************************************************************/
LRESULT DelayTime(WORD wMilliSec)
{
	DWORD dwCurrentTime;
	DWORD dwStartTime;
	DWORD dwDelta;

	dwStartTime = GetSystemTime();

	do
	{
		dwCurrentTime = GetSystemTime();
		dwDelta = dwCurrentTime - dwStartTime;
	} while(dwDelta < wMilliSec);

	return RET_OK;

}/*DelayTime*/


/*****************************************************************************
;+
Function Name	:	InitStatics
Inputs			:	None
Outputs			:	Updates register map
Side Effects    :
Discription     :   Initialises statics (Only once)
;-
*****************************************************************************/
void InitStatics(void)
{
	if (!bInitNeeded)
		return;

    bInitNeeded = FALSE;
    gwBoardID = (WORD)0;

	/*
		Use REG(R,A,S) macro to initialise register map :
	*/

	REG (MAP_ARSENAL_CAPTURE,  ARSENAL_CAPTURE,	DEVICE_ARSENAL|ATTRIB_WORD|ATTRIB_READONLY)

	REG (MAP_ARSENAL_M2WR_ID16,
							   ARSENAL_M2WR_ID16,
							   					DEVICE_ARSENAL|ATTRIB_WORD)

	REG (MAP_ARSENAL_M2READ,   ARSENAL_M2READ,	DEVICE_ARSENAL|ATTRIB_WORD)
	REG (MAP_ARSENAL_I2C,	   ARSENAL_I2C,		DEVICE_ARSENAL|ATTRIB_WORD)
	REG (MAP_ARSENAL_SETUP,	   ARSENAL_SETUP,	DEVICE_ARSENAL|ATTRIB_WORD)
	REG (MAP_ARSENAL_CONTROL,  ARSENAL_CONTROL,	DEVICE_ARSENAL|ATTRIB_WORD)
	REG (MAP_ARSENAL_ICR,	   ARSENAL_ICR,		DEVICE_ARSENAL|ATTRIB_WORD)
	REG (MAP_ARSENAL_MEM_BASE, ARSENAL_MEM_BASE,DEVICE_ARSENAL|ATTRIB_WORD)
	REG (MAP_SPARE,			   0xFF,			DEVICE_ARSENAL|ATTRIB_WORD)

	REG (MAP_IDNUM,		M2_IDNUM,		DEVICE_M2|ATTRIB_READONLY)
	REG (MAP_INTENB,	M2_INTENB,		DEVICE_M2)
	REG (MAP_INTFLG,	M2_INTFLG,		DEVICE_M2)
	REG (MAP_INTRDY,	M2_INTRDY,		DEVICE_M2|ATTRIB_READONLY)
	REG (MAP_STATUS,	M2_STATUS,		DEVICE_M2|ATTRIB_READONLY)
	REG (MAP_TSTMODE,	M2_TSTMODE,		DEVICE_M2|ATTRIB_WRITEONLY)

	REG (MAP_VIDSEL,	M2_VIDSEL,		DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_FSDATA,	M2_FSDATA,		DEVICE_M2)
	REG (MAP_FSRESET,	M2_FSRESET,		DEVICE_M2)
	REG (MAP_FSMODE,	M2_FSMODE,		DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_DVPRIO,	M2_DVPRIO,		DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_MCTL,		M2_MCTL,        DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_DISPCTL,	M2_DISPCTL,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_DPIXNO,	M2_DPIXNOL,     DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_BCKGNDY,	M2_BCKGNDY,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_BCKGNDU,	M2_BCKGNDU,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_BCKGNDV,	M2_BCKGNDV,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_XMCTRL,	M2_XMCTRL,		DEVICE_M2)
	REG (MAP_PIXPLINE,	M2_PIXPLINEL,   DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_PICSTART,	M2_PICSTARTL,   DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_PICEND,	M2_PICENDL,     DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_HSEND,		M2_HSEND,       DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_LINPFLD,	M2_LINPFLDL,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_LINPPIC,	M2_LINPPICL,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_VSSTART,	M2_VSSTARTL,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_VSEND,		M2_VSEND,       DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_SPGMODE,	M2_SPGMODE,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RHSTART,	M2_RHSTART,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RHEND,		M2_RHEND,       DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RVSTART,	M2_RVSTARTL,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_RVEND,		M2_RVENDL,      DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_RHFILT,	M2_RHFILT,      DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RVFILT,	M2_RVFILT,      DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RHSCALE,	M2_RHSCALE,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RVSCALE,	M2_RVSCALE,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_FLDSYN,	M2_FLDSYN,      DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_CAPUSE,	M2_CAPUSE,      DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RVSODD,	M2_RVSODD,      DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RVSEVEN,	M2_RVSEVEN,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RTLXFS1,	M2_RTLXLFS1,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_RTLYFS1,	M2_RTLYLFS1,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_RBRXFS1,	M2_RBRXLFS1,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_RBRYFS1,	M2_RBRYLFS1,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_FSXFS1,	M2_FSXLFS1,     DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_FSYFS1,	M2_FSYLFS1,     DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_DVSCFS1,	M2_DVSCFS1,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_DVCTLFS1,	M2_DVCTLFS1,    DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_RTLXFS2,	M2_RTLXLFS2,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_RTLYFS2,	M2_RTLYLFS2,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_RBRXFS2,	M2_RBRXLFS2,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_RBRYFS2,	M2_RBRYLFS2,    DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_FSXFS2,	M2_FSXLFS2,     DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_FSYFS2,	M2_FSYLFS2,     DEVICE_M2|ATTRIB_WORD|ATTRIB_WRITEONLY)
	REG (MAP_DVSCFS2,	M2_DVSCFS2,     DEVICE_M2|ATTRIB_WRITEONLY)
	REG (MAP_DVCTLFS2,	M2_DVCTLFS2,    DEVICE_M2|ATTRIB_WRITEONLY)

	REG (MAP_DMSD_STATUS,	SUBADDR_STATUS,		DEVICE_I2C|ATTRIB_READONLY)
	REG (MAP_DMSD_IDEL,		SUBADDR_IDEL,		DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_HSYSTART,	SUBADDR_HSYSTART,	DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_HSYSTOP,	SUBADDR_HSYSTOP,	DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_HCSTART,	SUBADDR_HCSTART,	DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_HCSTOP,	SUBADDR_HCSTOP,		DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_HSSTART,	SUBADDR_HSSTART,	DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_HPEAK,	SUBADDR_HPEAK,		DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_HUE,		SUBADDR_HUE,		DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_CONTROL1,	SUBADDR_CONTROL1,	DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_CONTROL2,	SUBADDR_CONTROL2,	DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_CONTROL3,	SUBADDR_CONTROL3,	DEVICE_I2C|ATTRIB_WRITEONLY)
	REG (MAP_DMSD_SDELAY,	SUBADDR_SDELAY,		DEVICE_I2C|ATTRIB_WRITEONLY)


}/*InitStatics*/



/*****************************************************************************
;+
Function Name	:	ResetArsenal
Inputs			:   None
Outputs			:	Status
Side Effects    :
Discription     :   Resets Arsenal chip (and hence entire board)
;-
*****************************************************************************/
LRESULT ResetArsenal(void)
{
	WORD wData;
	LONG lRet;

    /*
    	Add card search routine eventually !!!
    */
	lRet = RegIn((WORD) 0, (WORD) MAP_ARSENAL_M2WR_ID16, (LPWORD) &wData);

	if (lRet)
		return lRet;

	gwBoardID = (WORD)(wData & 0xFFF0);
/*	gbHwRevision = (BYTE) (wData & 0x000F);*/

	if (gwBoardID == 0xAAA0)
	{

		/* Reset board */
	    lRet = PortWrite ((WORD)MAP_ARSENAL_CONTROL, (WORD) 0);

		if (lRet)
			return lRet;

		/* 100mS */
		DelayTime(100);

	    lRet = PortWrite ((WORD)MAP_ARSENAL_CONTROL, (WORD) 1);

		if (lRet)
			return lRet;

		/* Wait 100mS */
		DelayTime(100);

	}
	else
		lRet =  IOERR_DRV_NO_HARDWARE;

	return lRet;

}/*ResetArsenal*/

/*****************************************************************************
;+
Function Name   :   ResetDMSD
Inputs          :   None
Outputs         :   Status
Side Effects    :
Discription     :   Loads DMSD with default values for PAL Composite.
;-
*****************************************************************************/
LRESULT ResetDMSD (void)
{
	LONG lRet;

	SetReg (MAP_DMSD_IDEL,      (WORD)0x5E);
	SetReg (MAP_DMSD_HSYSTART,  (WORD)0x37);
	SetReg (MAP_DMSD_HSYSTOP,   (WORD)0x07);
	SetReg (MAP_DMSD_HCSTART,   (WORD)0xF6);
	SetReg (MAP_DMSD_HCSTOP,    (WORD)0xC7);
	SetReg (MAP_DMSD_HSSTART,   (WORD)0xFF);
	SetReg (MAP_DMSD_HPEAK,     (WORD)0x02);
	SetReg (MAP_DMSD_HUE,       (WORD)0x00);
	SetReg (MAP_DMSD_CONTROL1,  (WORD)0x38);
	SetReg (MAP_DMSD_CONTROL2,  (WORD)0xE0);
	SetReg (MAP_DMSD_CONTROL3,  (WORD)0x58);
	SetReg (MAP_DMSD_SDELAY,    (WORD)0x00);

	lRet = WriteAllI2CRegs();

	return lRet;

}/*ResetDMSD*/


/*****************************************************************************
;+
Function Name   :   SetDMSD
Inputs          :   None
Outputs         :   Status
Side Effects    :
Discription     :   Programs DMSD for vid standard/type
;-
*****************************************************************************/
LRESULT SetDMSD (WORD wStandardID, WORD wType)
{
	LONG lRet;

	if (wStandardID == DRV_STANDARD_PAL && wType == DRV_TYPE_COMPOSITE)
	{
		SetReg (MAP_DMSD_HPEAK, (WORD)0x02);
	}
	else
	{
		if (wStandardID == DRV_STANDARD_PAL && wType == DRV_TYPE_SVIDEO)
		{
			SetReg (MAP_DMSD_HPEAK, (WORD)0x02);
	    }
		else
		{
			if (wStandardID == DRV_STANDARD_NTSC && wType == DRV_TYPE_COMPOSITE)
			{
				SetReg (MAP_DMSD_HPEAK, (WORD)0x62);
			}
		    else
		    {
		    	if (wStandardID == DRV_STANDARD_NTSC && wType == DRV_TYPE_SVIDEO)
		    	{
					SetReg (MAP_DMSD_HPEAK, (WORD)0x77);
				}
				else
				{
					return IOERR_DRV_BAD_PARAMS;
				}
			}
		}
    }

	if (wStandardID == DRV_STANDARD_PAL)
	{
		SetReg (MAP_DMSD_CONTROL1, (WORD)0x38);
		SetReg (MAP_DMSD_HUE, (WORD)0x00);
	}
    else
    {
    	if (wStandardID == DRV_STANDARD_NTSC)
    	{
			SetReg (MAP_DMSD_CONTROL1, (WORD)0x77);
			SetReg (MAP_DMSD_HUE, (WORD)0x06);
		}
		else
		{
			return IOERR_DRV_BAD_PARAMS;
		}
	}

	if (wType == DRV_TYPE_SVIDEO)
	{
		SetReg (MAP_DMSD_CONTROL2, (WORD)0xE1);
		SetReg (MAP_DMSD_CONTROL3, (WORD)0x38);
	}
	else if (wType == DRV_TYPE_COMPOSITE)
	{
		SetReg (MAP_DMSD_CONTROL2, (WORD)0xE0);
		SetReg (MAP_DMSD_CONTROL3, (WORD)0x58);
	}
	else
		return IOERR_DRV_BAD_PARAMS;

	lRet = SendChanges();

	return lRet;

}/*SetDMSD*/


/*****************************************************************************
;+
Function Name	:	RegOut

Inputs			:	Device ID, reg address and data.
					wDevice == 0 implies Logical Map address.
					wDevice != 0 implies Physical address.

Outputs			:	Status
Side Effects    :
Discription     :   Main Register output routine, supports logical
					or physical addressing.

	Physical addressing is slower, because the logical
	map must be kept up to date and the physical
	to logical mapping involves a map search.

;-
*****************************************************************************/
LRESULT RegOut(WORD wDevice, WORD wAddress, WORD wData)
{
	LONG lRet = RET_OK;

	if (wDevice) /* If physical addressing mode */
	{
		/* Convert address from physical to logical */
		lRet = FindReg (wAddress, (WORD)wDevice, (LPWORD)&wAddress);

	}
	else /* Else logical addressing mode */
	{
		/* Get real device ID from map */
	    wDevice = (WORD)(RegMap[wAddress].bStatus & DEVICE_MASK);
	}

    if (lRet)
    	return lRet;

	switch(wDevice)
	{
		case DEVICE_ARSENAL :
			lRet = PortWrite (wAddress, wData);
			break;

		case DEVICE_M2 :
			lRet = M2Write (wAddress, wData);
			break;

		case DEVICE_I2C :
			lRet = I2CWrite (wAddress, wData);
			break;
	}

	return lRet;

}/*RegOut*/



/*****************************************************************************
;+
Function Name	:	RegIn
Inputs			:	Device ID, reg address and reg data buffer.
					wDevice == 0 implies Logical Map address.
					wDevice != 0 implies Physical address.
Outputs			:	Status
Side Effects    :
Discription     :   Main Register input routine, supports logical
					or physical addressing.

	Physical addressing is slower, because the logical
	map must be kept up to date and the physical
	to logical mapping involves a map search.

;-
*****************************************************************************/

LRESULT RegIn(WORD wDevice, WORD wAddress, LPWORD lpwData)
{
	LONG lRet = RET_OK;

	if (wDevice)/* If physical addressing mode */
	{
		/* convert address from physical to logical */
		lRet = FindReg (wAddress, (WORD)wDevice, (LPWORD)&wAddress);

	}
	else /* Else logical addressing mode */
	{
		/* get real device ID from map */
	    wDevice = (WORD)(RegMap[wAddress].bStatus & DEVICE_MASK);
	}

    if (lRet)
    	return lRet;

	switch(wDevice)
	{
		case DEVICE_ARSENAL :
			lRet = PortRead (wAddress, lpwData);
			break;

		case DEVICE_M2 :
			lRet = M2Read (wAddress, lpwData);
			break;

		case DEVICE_I2C :
			lRet = I2CRead (wAddress, lpwData);
			break;
	}

	return lRet;

}/*RegIn*/



/*****************************************************************************
;+
Function Name	:	PortWrite
Inputs			:	Port ID, data.
Outputs			:	Status
Side Effects    :
Discription     :   Writes data to card I/O Port
;-
*****************************************************************************/
LRESULT PortWrite (WORD wMapAddress, WORD wData)
{
	REG_MAP *pReg;
	WORD wPortAddress;
	LONG lRet = RET_OK;

	pReg = (REG_MAP*)&RegMap[wMapAddress];
	pReg->bStatus &= NSTATUS_CHANGED;
	pReg->bStatus |= STATUS_VALID;
    pReg->wData = wData;

	wPortAddress = (WORD) (gwBaseReg + pReg->bAddress);

	if (pReg->bStatus & ATTRIB_WORD)
		outpw ((unsigned)wPortAddress,(unsigned)wData);
	else
		outp ((unsigned)wPortAddress,(unsigned)wData);


    return lRet;

}/*PortWrite*/




/*****************************************************************************
;+
Function Name	:	PortRead
Inputs			:	Port ID, data buffer.
Outputs			:	Status
Side Effects    :
Discription     :   Reads data from card I/O Port
;-
*****************************************************************************/
LRESULT PortRead (WORD wMapAddress, LPWORD wpDataPtr)
{
	REG_MAP *pReg;
	WORD wPortAddress;
	WORD wPortData;

	pReg = (REG_MAP*)&RegMap[wMapAddress];

	if (pReg->bStatus & ATTRIB_WRITEONLY)
	{
		if (!(pReg->bStatus & STATUS_VALID))
		{
			return (LONG)IOERR_DRV_REG_WRITEONLY;
		}

		*wpDataPtr = pReg->wData;
		return (LONG)IOWARN_REG_WRITEONLY;
    }

	wPortAddress = (WORD) (gwBaseReg + pReg->bAddress);

	pReg->bStatus &= NSTATUS_CHANGED;
	pReg->bStatus |= STATUS_VALID;

	if (pReg->bStatus & ATTRIB_WORD)
		wPortData = inpw ((unsigned)wPortAddress);
	else
		wPortData = inp ((unsigned)wPortAddress);

	*wpDataPtr = wPortData;
    pReg->wData = wPortData;


    return (LONG)RET_OK;

}/*PortRead*/


/*****************************************************************************
;+
Function Name	:	M2Write
Inputs			:   Reg ID, and data.
Outputs			:	Status
Side Effects    :
Discription     :   Writes to a byte or word M2 register.
;-
*****************************************************************************/
LRESULT M2Write (WORD wMapAddress, WORD wData)
{
	REG_MAP *pM2Reg;
	WORD wPortAddress;
	WORD wPortData;
	LONG lRet = RET_OK;

	pM2Reg = (REG_MAP*)&RegMap[wMapAddress];
	pM2Reg->bStatus &= NSTATUS_CHANGED;
	pM2Reg->bStatus |= STATUS_VALID;
    pM2Reg->wData = wData;

	#if OLDREGS
		wPortAddress = (WORD) (gwBaseReg + MAP_ARSENAL_M2WRITE);
	#else
		wPortAddress = (WORD) (gwBaseReg + MAP_ARSENAL_M2WR_ID16);
	#endif

    /* Send lo data byte */
    wPortData = (WORD) ((WORD)(pM2Reg->bAddress <<8)
    					| (WORD)((BYTE)wData));

	outpw ((unsigned)wPortAddress,(unsigned)wPortData);

    /* Send hi data byte if it is a word register */
	if (pM2Reg->bStatus & ATTRIB_WORD)
	{
	    wPortData = (WORD) ((WORD)((pM2Reg->bAddress+1) <<8)
	    					| (WORD)(wData >> 8));
		outpw ((unsigned)wPortAddress,(unsigned)wPortData);
    }


    return lRet;

}/*M2Write*/




/*****************************************************************************
;+
Function Name	:	M2Read
Inputs			:   Reg ID, and data buffer.
Outputs			:	Status
Side Effects    :
Discription     :   Reads a byte or word M2 register.
;-
*****************************************************************************/
LRESULT M2Read (WORD wMapAddress, LPWORD wpDataPtr)
{
	REG_MAP *pM2Reg;
	WORD wPortAddress;
	WORD wPortData;
	BYTE bM2DataLo;
	BYTE bM2DataHi = 0;
	WORD wM2RegData;

	pM2Reg = (REG_MAP*)&RegMap[wMapAddress];

	if (pM2Reg->bStatus & ATTRIB_WRITEONLY)
	{
		if (!(pM2Reg->bStatus & STATUS_VALID))
		{
			return (LONG)IOERR_DRV_REG_WRITEONLY;
		}

		*wpDataPtr = pM2Reg->wData;
		return (LONG)IOWARN_REG_WRITEONLY;
    }

	wPortAddress = (WORD) (gwBaseReg + ARSENAL_M2READ);

	/* Output M2 address */
    wPortData = (WORD) (pM2Reg->bAddress <<8);
	outpw ((unsigned)wPortAddress,(unsigned)wPortData);

    /* Input M2 data */
	wPortData = inpw ((unsigned)wPortAddress);
	bM2DataLo = (BYTE)wPortData;

    /* If the register is 16 bits, read the hi byte */
    if (pM2Reg->bStatus & ATTRIB_WORD)
    {
		/* Output M2 address */
	    wPortData = (WORD) ((pM2Reg->bAddress + 1) <<8);
		outpw ((unsigned)wPortAddress,(unsigned)wPortData);

	    /* Input M2 data */
		wPortData = inpw ((unsigned)wPortAddress);
		bM2DataHi = (BYTE)wPortData;
    }

    wM2RegData = (WORD)(bM2DataHi<<8 | bM2DataLo);

    /* Keep map up to date */
    pM2Reg->wData = wM2RegData;
	pM2Reg->bStatus &= NSTATUS_CHANGED;
	pM2Reg->bStatus |= STATUS_VALID;

    /* Output the result */
	*wpDataPtr = wM2RegData;

    return (LONG)RET_OK;

}/*M2Read*/





/*****************************************************************************
;+
Function Name	:	I2CWrite
Inputs			:   Reg ID, and data.
Outputs			:	Status
Side Effects    :
Discription     :   Writes to dmsd register over i2c bus.
;-
*****************************************************************************/
LRESULT I2CWrite (WORD wMapAddress, WORD wData)
{
	REG_MAP *pI2CReg;
	LONG lRet;

	pI2CReg = (REG_MAP*)&RegMap[wMapAddress];
	pI2CReg->bStatus &= NSTATUS_CHANGED;
	pI2CReg->bStatus |= STATUS_VALID;
    pI2CReg->wData = wData;


    I2CWriteBuff.Status = 0L;
    I2CWriteBuff.SlaveAddress = (BYTE)0x8A; /* As advised by Piers */
    I2CWriteBuff.SubAddress = pI2CReg->bAddress;
    I2CWriteBuff.WBufferLength = (WORD)1;
    I2CWriteBuff.WriteSpace[0] = (BYTE)wData;

	lRet = (LONG) WriteIICDevice((IIC_WRITEBUFF_PTR) &I2CWriteBuff);

    if (!lRet)
    	lRet = (DWORD)I2CWriteBuff.Status;

    if (lRet == IOERR_DRV_I2C_NO_ACK)
    {
    	/* No ack, so try again */

    	/* Wait 40mS */
		DelayTime(40);

    	lRet = (LONG) WriteIICDevice((IIC_WRITEBUFF_PTR) &I2CWriteBuff);

	    if (!lRet)
	    	lRet = (DWORD)I2CWriteBuff.Status;
    }


	return lRet;

}/*I2CWrite*/


/*****************************************************************************
;+
Function Name	:	WriteAllI2CRegs
Inputs			:   Reg ID, and data.
Outputs			:	Status
Side Effects    :
Discription     :   Writes all dmsd registers over i2c bus.
;-
*****************************************************************************/
LRESULT WriteAllI2CRegs (void)
{
	LONG lRet;

	/* Keep map up to date */
	I2CPrimeMap(MAP_DMSD_IDEL);
	I2CPrimeMap(MAP_DMSD_HSYSTART);
	I2CPrimeMap(MAP_DMSD_HSYSTOP);
	I2CPrimeMap(MAP_DMSD_HCSTART);
	I2CPrimeMap(MAP_DMSD_HCSTOP);
	I2CPrimeMap(MAP_DMSD_HSSTART);
	I2CPrimeMap(MAP_DMSD_HPEAK);
	I2CPrimeMap(MAP_DMSD_HUE);
	I2CPrimeMap(MAP_DMSD_CONTROL1);
	I2CPrimeMap(MAP_DMSD_CONTROL2);
	I2CPrimeMap(MAP_DMSD_CONTROL3);
	I2CPrimeMap(MAP_DMSD_SDELAY);

    /* Load PB */
    I2CWriteBuff.Status = 0L;
    I2CWriteBuff.SlaveAddress = (BYTE)0x8A; /* As advised by Piers */
    I2CWriteBuff.SubAddress = (BYTE)SUBADDR_IDEL;
    I2CWriteBuff.WBufferLength = (WORD)12;
    I2CWriteBuff.WriteSpace[0] = (BYTE) RegMap[MAP_DMSD_IDEL].wData;
    I2CWriteBuff.WriteSpace[1] = (BYTE) RegMap[MAP_DMSD_HSYSTART].wData;
    I2CWriteBuff.WriteSpace[2] = (BYTE) RegMap[MAP_DMSD_HSYSTOP].wData;
    I2CWriteBuff.WriteSpace[3] = (BYTE) RegMap[MAP_DMSD_HCSTART].wData;
    I2CWriteBuff.WriteSpace[4] = (BYTE) RegMap[MAP_DMSD_HCSTOP].wData;
    I2CWriteBuff.WriteSpace[5] = (BYTE) RegMap[MAP_DMSD_HSSTART].wData;
    I2CWriteBuff.WriteSpace[6] = (BYTE) RegMap[MAP_DMSD_HPEAK].wData;
    I2CWriteBuff.WriteSpace[7] = (BYTE) RegMap[MAP_DMSD_HUE].wData;
    I2CWriteBuff.WriteSpace[8] = (BYTE) RegMap[MAP_DMSD_CONTROL1].wData;
    I2CWriteBuff.WriteSpace[9] = (BYTE) RegMap[MAP_DMSD_CONTROL2].wData;
    I2CWriteBuff.WriteSpace[10] = (BYTE) RegMap[MAP_DMSD_CONTROL3].wData;
    I2CWriteBuff.WriteSpace[11] = (BYTE) RegMap[MAP_DMSD_SDELAY].wData;

	lRet = (LONG) WriteIICDevice((IIC_WRITEBUFF_PTR) &I2CWriteBuff);

    if (!lRet)
    	lRet = (DWORD)I2CWriteBuff.Status;

    if (lRet == IOERR_DRV_I2C_NO_ACK)
    {
    	/* No ack, so try again */

    	/* Wait 40mS */
		DelayTime(40);

		lRet = (LONG) WriteIICDevice((IIC_WRITEBUFF_PTR) &I2CWriteBuff);

	    if (!lRet)
	    	lRet = (DWORD)I2CWriteBuff.Status;
    }



	return lRet;

}/*WriteAllI2CRegs*/

static void I2CPrimeMap(WORD wMapAddr)
{
	REG_MAP *pI2CReg;

	/* Keep map up to date */
	pI2CReg = (REG_MAP*)&RegMap[wMapAddr];
	pI2CReg->bStatus &= NSTATUS_CHANGED;
	pI2CReg->bStatus |= STATUS_VALID;

}/*I2CMapPrime*/

/*****************************************************************************
;+
Function Name	:	I2CRead
Inputs			:   Reg ID, and data buffer.
Outputs			:	Status
Side Effects    :
Discription     :   Reads a dmsd register over i2c bus.
;-
*****************************************************************************/
LRESULT I2CRead (WORD wMapAddress, LPWORD wpDataPtr)
{
	REG_MAP *pI2CReg;
	BYTE bPortData;
	LONG lRet;

	pI2CReg = (REG_MAP*)&RegMap[wMapAddress];

	if (pI2CReg->bStatus & ATTRIB_WRITEONLY)
	{
		if (!(pI2CReg->bStatus & STATUS_VALID))
		{
			return (LONG)IOERR_DRV_REG_WRITEONLY;
        }
		*wpDataPtr = pI2CReg->wData;
		return (LONG)IOWARN_REG_WRITEONLY;
    }

    I2CReadBuff.Status = 0L;
    I2CReadBuff.SlaveAddress = (BYTE)0x8A; /* As advised by Piers */

	lRet = (LONG) ReadIICDevice ((IIC_READBUFF_PTR) &I2CReadBuff);

    if (!lRet)
    	lRet = (DWORD)I2CWriteBuff.Status;

    if (lRet == IOERR_DRV_I2C_NO_ACK)
    {
    	/* No ack, so try again */

    	/* Wait 40mS */
		DelayTime(40);

		lRet = (LONG) ReadIICDevice ((IIC_READBUFF_PTR) &I2CReadBuff);

	    if (!lRet)
	    	lRet = (DWORD)I2CWriteBuff.Status;
    }

	if (lRet)
	{
		return lRet;
	}

	bPortData = I2CReadBuff.ReturnData;

    /* Keep map up to date */
    pI2CReg->wData = (WORD)bPortData;
	pI2CReg->bStatus &= NSTATUS_CHANGED;
	pI2CReg->bStatus |= STATUS_VALID;

    /* Output the result */
	*wpDataPtr = (WORD)bPortData;


    return (LONG)RET_OK;

}/*I2CRead*/







/*****************************************************************************
;+
Function Name	:	SetReg
Inputs			:   Reg ID, and data.
Outputs			:	Status
Side Effects    :
Discription     :   Sets register map, skips if no change.
					Flags register as changed if it has changed.
					Does not do any i/o.
;-
*****************************************************************************/
LRESULT SetReg (WORD RegID, WORD Value)
{
	BYTE *pbStatus;
    WORD *pwData;

    pwData = &RegMap[RegID].wData;
    pbStatus = &RegMap[RegID].bStatus;

	if (*pbStatus & ATTRIB_READONLY)
	{
		return IOERR_DRV_REG_READONLY;
	}

    /* If new value is same as valid old, don't bother */
    if (*pwData == Value && (*pbStatus & STATUS_VALID))
    	return RET_OK;

    /* Set register value in the map */
    *pbStatus |= (BYTE) STATUS_CHANGED | STATUS_VALID;
    *pwData = Value;
    return RET_OK;

}/*SetReg*/


/*****************************************************************************
;+
Function Name	:	ForceReg
Inputs			:   Reg ID, and data.
Outputs			:	Status
Side Effects    :
Discription     :   Sets register map, does not skip if no change.
					Flags register as changed even if it has not changed.
					Does not do any i/o.
;-
*****************************************************************************/
LRESULT ForceReg (WORD RegID, WORD Value)
{
	LONG lRet;

    CHANGE(RegID);
    lRet = SetReg(RegID, Value);
    return lRet;

}/*ForceReg*/



/*****************************************************************************
;+
Function Name	:	FindReg
Inputs			:   Reg physical address, device ID, and result buffer.
Outputs			:	Status
Side Effects    :
Discription     :   Returns logical address (register ID).
;-
*****************************************************************************/
LRESULT FindReg (WORD wRegAddress, WORD wDevice, LPWORD pwMapAddr)
{
	REG_MAP *Reg;
	WORD wStart;
	WORD wEnd;
	WORD x;

	switch(wDevice)
	{
		case DEVICE_ARSENAL:
			wStart = (WORD)MAP_ARSENAL_CAPTURE;
			wEnd = (WORD)MAP_ARSENAL_MEM_BASE;
			break;

		case DEVICE_M2:
			wStart = (WORD)MAP_IDNUM;
			wEnd = (WORD)MAP_DVCTLFS2;
			break;

		case DEVICE_I2C:
			wStart = (WORD)MAP_DMSD_STATUS;
			wEnd = (WORD)MAP_DMSD_SDELAY;
			break;

		default:
			return (LONG)IOERR_DRV_REG_UNKNOWN;
	}

	Reg = (REG_MAP*)&RegMap[0];

	for (x = wStart; x <= wEnd; x++)
	{
		/* Try to match register address */
		if (Reg[x].bAddress == (BYTE)wRegAddress)
		{
			/* Found it */
			*pwMapAddr = x;
			return RET_OK;
		}
		/* Trap word register hi-byte addressing error */
		if	( Reg[x-1].bAddress == (BYTE)(wRegAddress-1)
		        	&& (Reg[x-1].bStatus & (BYTE) ATTRIB_WORD) )
		{
			return IOERR_DRV_REG_HIBYTE;
		}
    }

	return (LONG)IOERR_DRV_REG_UNKNOWN;

}/*FindReg*/



/*****************************************************************************
;+
Function Name	:	GetRegMap
Inputs			:   dwRegDevice :	lo word - Address
									hi word - Device ID

					Device ID == 0 implies Logical Map address.
					Device ID != 0 implies Physical address.

Outputs			:	Status
Side Effects    :
Discription     :   Gets the register's map address, supports logical
					or physical addressing.
;-
*****************************************************************************/
LRESULT GetRegMap (DWORD dwRegDevice, LPDWORD lpdwRegMap)
{
	WORD wRegAddr;
	WORD wDeviceId;
	WORD wMapAddr;
	LPDWORD lpdwMap;
	LONG lRet = RET_OK;

	wRegAddr = (WORD)dwRegDevice;
	wDeviceId = (WORD)(dwRegDevice>>16);

	/* Get map address */
	if (wDeviceId)
		/* Physical addressing */
		lRet = FindReg (wRegAddr, wDeviceId, (LPWORD)&wMapAddr);
    else
		/* Logical addressing */
    	wMapAddr = wRegAddr;

	if (lRet)
		return lRet;

	lpdwMap = (LPDWORD) &RegMap[wMapAddr];

	/* Output map struct address */
	*lpdwRegMap = (DWORD)lpdwMap;

    return (LONG)RET_OK;

}/*GetRegMap*/



/*****************************************************************************
;+
Function Name	:	SendChanges
Inputs			:   None
Outputs			:	Status
Side Effects    :
Discription     :   Scans register map for changes, and updates the hardware.
;-
*****************************************************************************/
LRESULT SendChanges(void)
{
	WORD wMapAddress;
	LONG lRet = RET_OK;

	for (	wMapAddress = (WORD)MAP_IDNUM;
			wMapAddress <= (WORD)MAP_DMSD_SDELAY;
			wMapAddress++
		)
	{
	    if (RegMap[wMapAddress].bStatus & (BYTE)STATUS_CHANGED)
	    {
			lRet = RegOut ((WORD) DEVICE_MAP, wMapAddress, RegMap[wMapAddress].wData);

			if (lRet)
				break;
		}
	}
	return lRet;

}/*SendChanges*/



/*****************************************************************************
;+
Function Name	:	PrimeCapture
Inputs			:	None
Outputs			:	None
Description     :   Sets ARSENAL_SETUP Register
;-
*****************************************************************************/
LRESULT PrimeCapture(void)
{
	WORD wSetupReg;
	LONG lRet;

	/*
		Note : we use a read-modify-write approach
		to allow external control and setup of this register,
		via tolbox for instance.
	*/

	/* Read setup register using logical addressing */
	lRet = RegIn ((WORD)DEVICE_MAP, (WORD)MAP_ARSENAL_SETUP, (LPWORD)&wSetupReg);

	if (lRet)
		return lRet;


	/* Clear old mode bits (bits 0 and 1) */
    wSetupReg &= (WORD)(~0x0003);

	/* Set new mode bits */
	switch(CaptureStatus.CaptureFormat)
	{
		case CAPTURE_TYPE_FORMAT1:
			wSetupReg |= ARSENAL_PACKED_YUV;
			break;

		case CAPTURE_TYPE_FORMAT2:
			wSetupReg |= ARSENAL_UNPACKED_YUV;
			break;

		case CAPTURE_TYPE_FORMAT3:
			wSetupReg |= ARSENAL_SUPERPACKED_YUV;
			break;

		case CAPTURE_TYPE_FORMAT4:
			wSetupReg |= ARSENAL_LUMA_ONLY;
			break;
	}

    /* Clear DMA */
    wSetupReg &= (WORD)(~0x0030);

	/* Clear old interrupt address bits (bits 6 to 9) */
    wSetupReg &= (WORD)(~0x03C0);

	/* Set new interrupt address bits */
	wSetupReg |= (WORD)(CaptureStatus.IRQLevel)<<6;

	/* Select display bus */
	wSetupReg &= (WORD)(~0x0008);

	/* Write setup register using logical addressing */
	lRet = RegOut ((WORD)DEVICE_MAP, (WORD)MAP_ARSENAL_SETUP, wSetupReg);


	return lRet;

}/*PrimeCap*/



/* end of drvio.c */
