/***************************************************************************
;++

Name			:	CAP.C
Title			:	Capture Library Entry Point Source
Author			:	Marc Stevens
Created			:	31/MAY/93

	Copyright 1992 by VideoLogic Limited. All rights reserved.
	No part of this software, either material or conceptual may be
	copied or distributed, transmitted, transcribed, stored in a
	retrieval system or translated into any human or computer language
	in any form by any means, electronic, mechanical, manual or other-
	wise, or disclosed to third parties without the express written
	permission of VideoLogic Limited, Unit 8, Homepark Industrial
	Estate, King's Langley, Hertfordshire, WD4 8LZ, U.K.

Description		:	Contains the externally callable function declarations
					for the Capture Library.

Program Type    :   C Source File. 

Modifications   :

Notes           :

;--
****************************************************************************/
#define CAPLIB_MAIN
#include "masdefs.h"
#include "lresult.h"
#include "ext.h"
#include "cap.h"
#include "drvio.h"
#include "m2.h"
#include "conio.h"
#include "dos.h"
#include "math.h"

/* Stack and pointer checking off, use intrinsic port io functions */
#pragma check_stack( off )
#pragma check_pointer( off )
#pragma intrinsic( _enable, _disable, outp, outpw, inp, inpw )

typedef struct _YUVQUAD
{
	WORD Quad1;
	WORD Quad2;
	WORD Quad3;
	WORD Quad4;
} YUVQUAD, _far * FPYUVQUAD;

typedef struct _STREAMINFO
{
	char _huge *hpbStreamBuf;
	long lMicroSecPerFrame;	   	/* PAL/NTSC frame rate (us) */
	long lMicroSecPerCapture;		/* allowed time for one field capture */
	long lTimeToCapture;		/* time remaining until capture */
	long lWordsPerFrame;
	int	CaptureStatus;
	int StreamStatus;
	int Odd;
} STREAMINFO, _far * FPSTREAMINFO;

#define CAP_ERR	    			(TYPE_ERROR | MODULE_CAP)
#define IOERR_DRV_DATA_OVERRUN	(CAP_ERR | CAT_HWNOTCAPABLE		| 1)
#define IOERR_DRV_NO_BUFFER		(CAP_ERR | CAT_BADRESULT		| 2)
#define IOERR_DRV_STREAMING		(CAP_ERR | CAT_BADRESULT		| 3)

#define BRI_DEFAULT		((BYTE)100)
#define SAT_DEFAULT		((BYTE)100)
#define CON_DEFAULT		((BYTE)100)
#define GAM_DEFAULT		((BYTE)100)

#define RGB565_SELECT	((BYTE)1)
#define RGB555_SELECT	((BYTE)0)

#define STREAM_IDLE		((int)0)
#define STREAM_IN_PROG	((int)1)

#define CAPTURE_IDLE	((int)0)
#define CAPTURE_PENDING	((int)1)
#define CAPTURE_IN_PROG	((int)2)


#define	MPIC_CTRL_REG	((WORD)0x20)
#define	MPIC_MASK_REG	((WORD)0x21)
#define	SPIC_CTRL_REG	((WORD)0xa0)
#define	SPIC_MASK_REG	((WORD)0xa1)

#define NS_EOI			((WORD)0x20)

extern BYTE CorrectionFor9051;
static WORD RGB565Tab[32768];

static STREAMINFO StreamInfo;
static STREAMSTATS StreamStats;
static INT MPICMaskReg;
static INT SPICMaskReg;

static DOUBLE dBri = 0.0;
static DOUBLE dSat = 1.0;
static DOUBLE dCon = 1.0;
static DOUBLE dGam = 1.0;

VOID _near _pascal BuildRGB16LUT(VOID _far *fpwRGB16LUT, BYTE Bri, BYTE Sat, BYTE Con, BYTE Flags);
VOID _near _pascal ConvertSiteToYUV15Quad(WORD wYUVLo,WORD wYUVHi, FPYUVQUAD lpwYUVQuad);
VOID _near _pascal ConvertF3SiteToYUV15Quad(WORD wYUVLo,WORD wYUVHi, FPYUVQUAD lpwYUVQuad);
VOID _near _pascal ConvertF0SiteToYUV15Quad(DWORD dYUVLo,BYTE bYUVHi, FPYUVQUAD lpwYUVQuad);
VOID _near _pascal ConvertF1SiteToYUV15Quad(DWORD dYUVLo,WORD wYUVHi, FPYUVQUAD lpwYUVQuad);

VOID _interrupt _far StreamISR( VOID );
VOID (_interrupt _far *OldISR)( VOID );
WORD SetPIC(WORD irq,INT *mmask, INT *smask);
WORD ResetPIC(INT mmask,INT smask);
VOID HookIRQ(WORD irq);
VOID UnHookIRQ(WORD irq);
WORD ValidateStreamBuffer(VOID);
VOID InstigateCapture(VOID);
VOID CaptureISR(VOID);
VOID FSyncISR(VOID);
VOID (_interrupt far *GetVect(BYTE inum))();
VOID SetVect(BYTE inum, VOID (_interrupt far *Handler)()); 
LONG Setup9051Tables (VOID);

long CapReset( void )
{
	long lRetData;

	if (StreamInfo.StreamStatus == STREAM_IN_PROG)
		CapStreamStop();	
	if (!(lRetData = ResetArsenal()))
	{
		CaptureStatus.CaptureFormat = (int)0;
		if (!(lRetData = M2SoftReset()))
		{
			CaptureStatus.DestRectWidth = (int)160;
			CaptureStatus.DestRectHeight = (int)120;
			if (!(lRetData = ResetDMSD()))
			{
				CaptureStatus.InputType = DRV_TYPE_COMPOSITE;
				CaptureStatus.InputStandard = DRV_STANDARD_PAL;
				CaptureStatus.Brightness = (int)BRI_DEFAULT;
				CaptureStatus.Saturation = (int)SAT_DEFAULT;
				CaptureStatus.Contrast = (int)CON_DEFAULT;
				CaptureStatus.Gamma = (int)GAM_DEFAULT;

				/* start fieldstores aquiring data */
				lRetData = M2StartCapturing();	

			}
		}
	}
	return lRetData;
}

long CapGetStatus( CAP_STATUS *CapStatus )
{
	*CapStatus = CaptureStatus;

	return (LONG)OK;
}

long CapConfigure( int IOBase, int IRQLevel )
{
	CaptureStatus.IOBase = IOBase;
	CaptureStatus.IRQLevel = IRQLevel;
	gwBaseReg = (WORD)IOBase;
	StreamInfo.StreamStatus = STREAM_IDLE;
	InitStatics();
	BuildRGB16LUT((void _far *)RGB565Tab,BRI_DEFAULT,	/* build RGB 565 LUT */
							SAT_DEFAULT,
							CON_DEFAULT,
							RGB565_SELECT);	/* prev. B=0,S=16,C=128 */
	return (LONG)OK;
}

long CapSetInput( int InputType, int InputStandard )
{
	long lRetData;

	if (!(lRetData = VidType((WORD)InputType )))
	{
		CaptureStatus.InputType = InputType;
		if (!(lRetData = VidStandard((WORD)InputStandard )))
		{
			CaptureStatus.InputStandard = InputStandard;
			StreamInfo.lMicroSecPerFrame = (CaptureStatus.InputStandard) 
													? PALMICROSECPERTICK
													: NTSCMICROSECPERTICK;
		}
	}
	return lRetData;
}

long CapSetFormat ( int CaptureFormat )
{

	CaptureStatus.CaptureFormat = CaptureFormat;
	return (LONG)OK;
}

long CapSetDestRect ( int Width, int Height)
{
	long lRetData;
	SIZE OSize;

	OSize.cx = Width;
	OSize.cy = Height;

	if (!(lRetData = M2OSize(&OSize)))
	{
		CaptureStatus.DestRectWidth = Width;
		CaptureStatus.DestRectHeight = Height;
	}
	return lRetData;
}

long CapGrabOneFrame ( char _huge *BufferPointer, long MaxBufferSize )
{
	WORD wICRReg;
	WORD wCONTROLReg;
	PHWORD phwYUV;
	long lYUVWordsInImage;
	long lSuppliedBufferSize;
	int iWidth,iHeight;
	long lRet;

	if (StreamInfo.StreamStatus == STREAM_IN_PROG)
	return IOERR_DRV_STREAMING;

	lRet = PrimeCapture();
	phwYUV = (PWORD)BufferPointer;
	lSuppliedBufferSize = MaxBufferSize / (LONG)2;

	iWidth = CaptureStatus.DestRectWidth;
	iHeight = CaptureStatus.DestRectHeight;
	switch(CaptureStatus.CaptureFormat)
	{
		case CAPTURE_TYPE_FORMAT1:
			lYUVWordsInImage = (LONG)iWidth * (LONG)iHeight * (LONG)5 / (LONG)8;
			break;

		case CAPTURE_TYPE_FORMAT2:
			lYUVWordsInImage = (LONG)iWidth * (LONG)iHeight * (LONG)6 / (LONG)8;
			break;

		case CAPTURE_TYPE_FORMAT3:
		case CAPTURE_TYPE_FORMAT4:
			lYUVWordsInImage = (LONG)iWidth * (LONG)iHeight / (LONG)2;
			break;
	}

	/* Read interrupt control register using logical addressing */
	lRet = RegIn ((WORD)DEVICE_MAP, (WORD)MAP_ARSENAL_ICR, (PWORD)&wICRReg);

	if (lRet)
		return lRet;

	/* Read control register using logical addressing */
	lRet = RegIn ((WORD)DEVICE_MAP, (WORD)MAP_ARSENAL_CONTROL, (PWORD)&wCONTROLReg);

	if (lRet)
		return lRet;

	/* enable 'capture ready signal' in reg copy */
	wICRReg = RDY_IRQ_ENB;

	/* set 'go capture signal' in reg copy */
	wCONTROLReg |= GO_CAP;

	/* Write ICR register using logical addressing */
	lRet = RegOut ((WORD)DEVICE_MAP, (WORD)MAP_ARSENAL_ICR, wICRReg);

	if (lRet)
		return lRet;

	/* Write CONTROL register using logical addressing */
	lRet = RegOut ((WORD)DEVICE_MAP, (WORD)MAP_ARSENAL_CONTROL, wCONTROLReg);

	if (lRet)
		return lRet;
	
	/* wait for Arsenal to say m2 capture complete */	
	do
		wICRReg = inpw(gwBaseReg + ARSENAL_ICR);
	while (!(wICRReg & RDY_IRQ_SET));

	/* stop fieldstores aquiring data */
	M2StopCapturing();	

	/* then go return captured data */		   
	while (lYUVWordsInImage-- && lSuppliedBufferSize--)
		*phwYUV++ = inpw(gwBaseReg + ARSENAL_CAPTURE);

	wICRReg = (WORD)0;
	wCONTROLReg &= ~GO_CAP;

	/* Write ICR register using logical addressing */
	lRet = RegOut ((WORD)DEVICE_MAP, (WORD)MAP_ARSENAL_ICR, wICRReg);

	if (lRet)
		return lRet;

	/* Write CONTROL register using logical addressing */
	lRet = RegOut ((WORD)DEVICE_MAP, (WORD)MAP_ARSENAL_CONTROL, wCONTROLReg);

	/* start fieldstores aquiring data */
	M2StartCapturing();	

	return lRet;
}

long CapConvertYUVFrameToRGB(char _huge *hpbYUVBuf, char _huge *hpbRGBBuf, int iWidth,
					int iHeight, int iFormat, long lYUVMax)
{
	register BYTE _huge *hpbOut;
	register WORD wRGB565;
	BYTE _huge *hpbIn;
	WORD wInLo;
	WORD wInHi;
	DWORD dwIn;
	BYTE bIn;
	WORD wLine;
	WORD wBytesPerLine;
	WORD wByteCnt;
	YUVQUAD YUVQuad;
	FPYUVQUAD lpYUVQuad;

	hpbOut = (BYTE _huge *) hpbRGBBuf;
	hpbIn = (BYTE _huge *) hpbYUVBuf;
	lpYUVQuad = &YUVQuad;

	if (iFormat == 2 || iFormat == 3)
	{
		/* In YUV Format 2 there is 1 RGB pixel per 1 byte of input YUV data */
		wByteCnt = wBytesPerLine = iWidth;

		wLine = iHeight;

		/* move input pointer to beginning of last line in frame */
		/* DIBs encode data such that the top left DIB byte represents */
		/* the bottom left image byte */
		hpbIn += (DWORD)wBytesPerLine * (DWORD)(iHeight - (WORD)1);

		while (wLine--)
		{
			while (wByteCnt && lYUVMax > 0)
			{
				wInLo = (WORD)*(WORD _huge *)hpbIn;
				hpbIn += 2;
				wInHi = (WORD)*(WORD _huge *)hpbIn;
				hpbIn += 2;
				if (iFormat == 2)
					ConvertSiteToYUV15Quad(wInLo,wInHi,lpYUVQuad);
				else
					ConvertF3SiteToYUV15Quad(wInLo,wInHi,lpYUVQuad);
				wRGB565 = RGB565Tab[YUVQuad.Quad1];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y1 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wRGB565 = RGB565Tab[YUVQuad.Quad2];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y2 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wRGB565 = RGB565Tab[YUVQuad.Quad3];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y3 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wRGB565 = RGB565Tab[YUVQuad.Quad4];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y4 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wByteCnt -= 4;
				lYUVMax -= (long)4;
			}
			wByteCnt = wBytesPerLine;
			hpbIn -= (wBytesPerLine << 1);					/* move up a line */
		}
	}
	else if (iFormat == 0)
	{
		/* In YUV Format 0 there is 5 bytes of YUV data for 4 bytes RGB */
		wByteCnt = wBytesPerLine = (iWidth * 5) >> 2;

		wLine = iHeight;

		/* move input pointer to beginning of last line in frame */
		/* DIBs encode data such that the top left DIB byte represents */
		/* the bottom left image byte */
		hpbIn += (DWORD)wBytesPerLine * (DWORD)(iHeight - (WORD)1);

		while (wLine--)
		{
			while (wByteCnt && lYUVMax > 0)
			{
				dwIn = (DWORD)*(DWORD _huge *)hpbIn;
				hpbIn += 4;
				bIn = *hpbIn++;
				ConvertF0SiteToYUV15Quad(dwIn,bIn,lpYUVQuad);
				wRGB565 = RGB565Tab[YUVQuad.Quad1];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y1 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wRGB565 = RGB565Tab[YUVQuad.Quad2];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y2 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wRGB565 = RGB565Tab[YUVQuad.Quad3];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y3 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wRGB565 = RGB565Tab[YUVQuad.Quad4];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y4 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wByteCnt -= 5;
				lYUVMax -= 5;
			}
			wByteCnt = wBytesPerLine;
			hpbIn -= (wBytesPerLine << 1);
		}
	}
	else if (iFormat == 1)
	{
		/* In YUV Format 1 there is 6 bytes of YUV data for 4 bytes RGB */
		wByteCnt = wBytesPerLine = (iWidth * 6) >> 2;

		wLine = iHeight;

		/* move input pointer to beginning of last line in frame */
		/* DIBs encode data such that the top left DIB byte represents */
		/* the bottom left image byte */
		hpbIn += (DWORD)wBytesPerLine * (DWORD)(iHeight - (WORD)1);

		while (wLine--)
		{
			while (wByteCnt && lYUVMax > 0)
			{
				dwIn = (DWORD)*(DWORD _huge *)hpbIn;
				hpbIn += 4;
				wInLo = (WORD)*(WORD _huge *)hpbIn;
				hpbIn += 2;
				ConvertF1SiteToYUV15Quad(dwIn,wInLo,lpYUVQuad);
				wRGB565 = RGB565Tab[YUVQuad.Quad1];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y1 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wRGB565 = RGB565Tab[YUVQuad.Quad2];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y2 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wRGB565 = RGB565Tab[YUVQuad.Quad3];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y3 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wRGB565 = RGB565Tab[YUVQuad.Quad4];
				*hpbOut++ = (BYTE)(wRGB565 << 3);		 	/* store B from Y4 */
				*hpbOut++ = (BYTE)((wRGB565 >> 3) & ~3);	/* store G */
				*hpbOut++ = (BYTE)((wRGB565 >> 8) & ~7);	/* store R */
				wByteCnt -= 6;
				lYUVMax -= 6;
			}
			wByteCnt = wBytesPerLine;
			hpbIn -= (wBytesPerLine << 1);
		}
	}
	return (LONG)OK;
}

long CapSetStreamParams(char _huge *hpbStreamBuf, int iFramesPerSec )
{
	StreamInfo.hpbStreamBuf = hpbStreamBuf;

	if (CaptureStatus.InputStandard == DRV_STANDARD_PAL)
	{
		StreamInfo.lMicroSecPerFrame = PALMICROSECPERTICK;
		StreamInfo.lMicroSecPerCapture = ((LONG)PALMICROSECPERTICK
										  * (LONG)PALFRAMESPERSEC)
										  / (LONG)iFramesPerSec;		
	}
	else
	{
		StreamInfo.lMicroSecPerFrame = NTSCMICROSECPERTICK;
		StreamInfo.lMicroSecPerCapture = ((LONG)NTSCMICROSECPERTICK
										  * (LONG)NTSCFRAMESPERSEC)
										  / (LONG)iFramesPerSec;		
	}

	StreamInfo.lTimeToCapture = StreamInfo.lMicroSecPerCapture;
	StreamInfo.CaptureStatus = CAPTURE_PENDING;
	StreamInfo.StreamStatus = STREAM_IDLE;
	
	return (LONG)OK;
}

long CapStreamStart( void )
{
	WORD wTempCONTROL;
	WORD wTempICR;
	int iWidth;
	int	iHeight;

	iWidth = CaptureStatus.DestRectWidth;
	iHeight = CaptureStatus.DestRectHeight;
	switch(CaptureStatus.CaptureFormat)
	{
		case CAPTURE_TYPE_FORMAT1:
			StreamInfo.lWordsPerFrame = (LONG)iWidth * (LONG)iHeight * 
											(LONG)5 / (LONG)8;
			break;

		case CAPTURE_TYPE_FORMAT2:
			StreamInfo.lWordsPerFrame = (LONG)iWidth * (LONG)iHeight * 
											(LONG)6 / (LONG)8;
			break;

		case CAPTURE_TYPE_FORMAT3:
		case CAPTURE_TYPE_FORMAT4:
			StreamInfo.lWordsPerFrame = (LONG)iWidth * (LONG)iHeight 
											/ (LONG)2;
			break;
	}
	StreamStats.lError = (LONG)OK;
	StreamStats.lFramesSkipped = (long)0;
	StreamStats.lFramesDropped = (long)0;
	StreamStats.lFramesCaptured = (long)0;
	StreamStats.lFrameCount = (long)0;
	StreamInfo.CaptureStatus = CAPTURE_PENDING;
	StreamInfo.StreamStatus = STREAM_IN_PROG;
	StreamInfo.Odd = (int)0;

	PrimeCapture();

 	HookIRQ(CaptureStatus.IRQLevel);

	SetPIC(CaptureStatus.IRQLevel,&MPICMaskReg,&SPICMaskReg);

	/* now do the things that PrimeCapture() doesn't */
	wTempCONTROL = inpw(gwBaseReg + ARSENAL_CONTROL);
	wTempICR = inpw(gwBaseReg + ARSENAL_ICR);

	/* generate field sync ints, not frame */
	wTempCONTROL &= ~FRAME;

	/* enable global Arsenal interrupts */
	wTempCONTROL |= IRQ_ENB;

	/* enable Field Sync and Capture ready interrupts */
	wTempICR |= FS_IRQ_ENB | RDY_IRQ_ENB;

	/* get rid of any previous irq indicators */
	wTempICR &= ~(FS_IRQ_SET | RDY_IRQ_SET);

	/* write temp registers to Arsenal */
	outpw(gwBaseReg + ARSENAL_ICR,wTempICR);

	/* and start Stream operations with... */
	outpw(gwBaseReg + ARSENAL_CONTROL,wTempCONTROL);

	return (LONG)OK;
}

long CapStreamStop( void )
{
	WORD wTempCONTROL;
	WORD wTempICR;

	while (StreamInfo.CaptureStatus == CAPTURE_IN_PROG);

	/* get CONTROL and ICR registers */
	wTempCONTROL = inpw(gwBaseReg + ARSENAL_CONTROL);
	wTempICR = inpw(gwBaseReg + ARSENAL_ICR);

	/* disable global Arsenal interrupts */
	wTempCONTROL &= ~IRQ_ENB;

	/* and do it for real */
	outpw(gwBaseReg + ARSENAL_CONTROL,wTempCONTROL);

	/* 
		disable Field Sync and Capture ready interrupts and associated
		interrupt indicator flags.
	*/
	wTempICR &= ~(FS_IRQ_ENB | RDY_IRQ_ENB | FS_IRQ_SET | RDY_IRQ_SET);

	/* and do it for real */
	outpw(gwBaseReg + ARSENAL_ICR,wTempICR);

	/* enable M2 global interrupts and 'FieldStore's not capturing' int */
	outpw(gwBaseReg + ARSENAL_M2WR_ID16,M2_INTENB | (WORD)0x90);

	ResetPIC(MPICMaskReg,SPICMaskReg);

 	UnHookIRQ(CaptureStatus.IRQLevel);

	StreamInfo.StreamStatus = STREAM_IDLE;

	return (LONG)OK;
}

long CapGetStreamStatistics( STREAMSTATS *StreamStatistics )
{
	*StreamStatistics = StreamStats;
 	return (LONG)OK;
}


long CapSetBrightness ( int iBrightness )
{
	CaptureStatus.Brightness = iBrightness;
	dBri = (DOUBLE)((DOUBLE)iBrightness - 100.0);
	Setup9051Tables();
	return (LONG)OK;
}

long CapSetSaturation ( int iSaturation )
{
	CaptureStatus.Saturation = iSaturation;
	dSat = (DOUBLE)((DOUBLE)iSaturation / 100.0);
	Setup9051Tables();
	return (LONG)OK;
}

long CapSetContrast ( int iContrast )
{
	CaptureStatus.Contrast = iContrast;
	dCon = (DOUBLE)((DOUBLE)iContrast / 100.0);
	Setup9051Tables();
	return (LONG)OK;
}

long CapSetGamma ( int iGamma )
{
	CaptureStatus.Gamma = iGamma;
	dGam = (DOUBLE)((DOUBLE)iGamma / 100.0);
	Setup9051Tables();
	return (LONG)OK;
}

WORD SetPIC(WORD irq,INT *mmask, INT *smask)
{
	INT OMmask,OSmask,NMmask,NSmask;

	if (irq > (WORD)7)
	{
		NSmask = (INT)1 << (irq - (WORD)8);
		NMmask = (INT)4;
	}
	else
	{
		NMmask = (INT)1 << irq;
		NSmask = (irq != 2) ? (INT)0 : (INT)2;
	}

	_disable();
	OSmask = inp(SPIC_MASK_REG);
	OMmask = inp(MPIC_MASK_REG);
	outp(SPIC_MASK_REG,OSmask & ~NSmask);
	outp(MPIC_MASK_REG,OMmask & ~NMmask);
	_enable();

	*mmask = OMmask;
	*smask = OSmask;
	
	return(OK);
}	

WORD ResetPIC(INT mmask,INT smask)
{
	_disable();
	outp(MPIC_MASK_REG,mmask);
	outp(SPIC_MASK_REG,smask);
	_enable();
	return(OK);
}

VOID _interrupt _far StreamISR( VOID )
{
	static WORD wTempICR;
	static WORD wTempCONTROL;
	static WORD wTempSETUP;

	wTempICR = inpw(gwBaseReg + ARSENAL_ICR);
	if (wTempICR & FS_IRQ_SET)
	{
		/* clear M2 interrupt flags */
		outpw(gwBaseReg + ARSENAL_M2WR_ID16,M2_INTFLG | (WORD)0x00);

		/* force a low onto the IRQ line to enable more IRQs */
		wTempCONTROL = inpw(gwBaseReg + ARSENAL_CONTROL);
		outpw(gwBaseReg + ARSENAL_CONTROL,wTempCONTROL | FORCE_LOW);

		/* clear field sync IRQ indicator */
		wTempICR = inpw(gwBaseReg + ARSENAL_ICR);
		outpw(gwBaseReg + ARSENAL_ICR,wTempICR & ~FS_IRQ_SET);

		if (!(++StreamInfo.Odd & 1))	/* skip odd syncs */
			FSyncISR();

		wTempCONTROL = inpw(gwBaseReg + ARSENAL_CONTROL);
		outpw(gwBaseReg + ARSENAL_CONTROL,wTempCONTROL & ~FORCE_LOW);

		wTempSETUP = inpw(gwBaseReg + ARSENAL_SETUP);

		_disable();
		if ((wTempSETUP >> 6) > (WORD)7)
			outp(SPIC_CTRL_REG,NS_EOI);
		outp(MPIC_CTRL_REG,NS_EOI);
		_enable();

	}
	else 
	{
		if (wTempICR & RDY_IRQ_SET)
		{
			/* clear M2 interrupt flags */
			outpw(gwBaseReg + ARSENAL_M2WR_ID16,M2_INTFLG | (WORD)0x00);

			/* force a low onto the IRQ line to enable more IRQs */
			wTempCONTROL = inpw(gwBaseReg + ARSENAL_CONTROL);
			outpw(gwBaseReg + ARSENAL_CONTROL,wTempCONTROL | FORCE_LOW);

			/* clear capture ready IRQ indicator */
			wTempICR = inpw(gwBaseReg + ARSENAL_ICR);
			outpw(gwBaseReg + ARSENAL_ICR,wTempICR & ~RDY_IRQ_SET);

			CaptureISR();

			wTempCONTROL = inpw(gwBaseReg + ARSENAL_CONTROL);
			outpw(gwBaseReg + ARSENAL_CONTROL,wTempCONTROL & ~FORCE_LOW);

			wTempSETUP = inpw(gwBaseReg + ARSENAL_SETUP);

			_disable();
			if ((wTempSETUP >> 6) > (WORD)7)
				outp(SPIC_CTRL_REG,NS_EOI);
			outp(MPIC_CTRL_REG,NS_EOI);
			_enable();
		}
		else
		{
			_asm
			{
				mov		sp,bp
				pop		es
				pop		ds
				popa	
				jmp 	DWORD PTR [OldISR]
			}
		}
	}
}


VOID FSyncISR(VOID)
{
	StreamInfo.lTimeToCapture -= StreamInfo.lMicroSecPerFrame;
	if (StreamInfo.CaptureStatus == CAPTURE_PENDING ||
						StreamInfo.lTimeToCapture <= 0)
	{
		StreamInfo.lTimeToCapture = StreamInfo.lMicroSecPerCapture;
		if (StreamInfo.CaptureStatus != CAPTURE_IN_PROG)
 		{
			if (!ValidateStreamBuffer())
			{
				InstigateCapture();
				StreamStats.lFramesCaptured++;
			}
			else
			{
				StreamStats.lError = IOERR_DRV_NO_BUFFER;
				StreamStats.lFramesDropped++;
			}
		}
		else
		{
			StreamStats.lError = IOERR_DRV_DATA_OVERRUN;
			StreamStats.lFramesDropped++;
		}
	}
	else
	{
		StreamStats.lFramesSkipped++;
	}
	StreamStats.lFrameCount++;
}

VOID CaptureISR(VOID)
{
	static WORD _huge *hpwBuf;
	static LONG lWordCount;
	static WORD wTempCONTROL;

	StreamInfo.CaptureStatus = CAPTURE_IN_PROG;

	_enable();								/* enable interrupts */

	/* clear all M2 syncs */
	outpw(gwBaseReg + ARSENAL_M2WR_ID16,M2_INTENB | (WORD)0);

	/* clear all M2 int flags */
	outpw(gwBaseReg + ARSENAL_M2WR_ID16,M2_INTFLG | (WORD)0);

	/* move to actual start of data buffer */
	hpwBuf = (WORD _huge *)(StreamInfo.hpbStreamBuf + 
								sizeof(char _huge *) + sizeof(BYTE));

	/* indicate to library client that buffer is in use by capture process */
	*(StreamInfo.hpbStreamBuf) = BUFFER_FILLING;

	/* set words to read count */
	lWordCount = StreamInfo.lWordsPerFrame;

	while (lWordCount--)
		*hpwBuf++ = inpw(gwBaseReg + ARSENAL_CAPTURE);

	/* reset GO_CAPTURE flag in Arsenal control reg */
	wTempCONTROL = inpw(gwBaseReg + ARSENAL_CONTROL);
	outpw(gwBaseReg + ARSENAL_CONTROL,wTempCONTROL & ~GO_CAP);

	/* indicate to library client that buffer is no longer in use */
	*(StreamInfo.hpbStreamBuf) = BUFFER_FULL;

	/* move stream buffer pointer to next available buffer */
	StreamInfo.hpbStreamBuf = (char _huge *)*(char _huge * _huge *)(StreamInfo.hpbStreamBuf + sizeof(BYTE));

	StreamInfo.CaptureStatus = CAPTURE_IDLE;
}

VOID InstigateCapture(VOID)
{
 	static WORD wTempReg;

	/* enable M2 global interrupts and 'FieldStore's not capturing' int */
	outpw(gwBaseReg + ARSENAL_M2WR_ID16,M2_INTENB | (WORD)0x90);

	/* get CONTROL register and indicate desire for capture */
	wTempReg = inpw(gwBaseReg + ARSENAL_CONTROL);
	outpw(gwBaseReg + ARSENAL_CONTROL, wTempReg | GO_CAP);
}

WORD ValidateStreamBuffer(VOID)
{
	if (StreamInfo.hpbStreamBuf && 
						*(StreamInfo.hpbStreamBuf) == (BYTE)0)
		return (WORD)OK;
	else
		return (WORD)1;
}

VOID HookIRQ(WORD irq)
{ 
	WORD inum;

	inum = (irq < 8) ? irq + (WORD)8 : irq + (WORD)0x68;
	OldISR = GetVect((BYTE)inum); 
	SetVect((BYTE)inum, StreamISR);
}

VOID UnHookIRQ(WORD irq)
{ 
	WORD inum;

	inum = (irq < 8) ? irq + (WORD)8 : irq + (WORD)0x68;
	SetVect((BYTE)inum, OldISR);
}

VOID (_interrupt far *GetVect(BYTE inum))()
{
	VOID (_interrupt _far *TempISR)( VOID );

	_asm
	{
		push	es
		push	bx

		mov		ah,35h
		mov		al,inum
		int		21h

		mov		WORD PTR TempISR,bx
		mov		ax,es
		mov		WORD PTR TempISR+2,ax
		pop		bx
		pop		es
	}
	return TempISR;
}

VOID SetVect(BYTE inum, VOID (_interrupt far *Handler)())
{
	_asm
	{
		push	ds
		push	dx

		mov		ah,25h
		mov		al,inum
		lds		dx,Handler
		int		21h

		pop		dx
		pop		ds
	}
}

LONG Setup9051Tables (VOID)
{
    int i;
    double dY;
    double dUV;
    double dScale;
    BYTE bY;
    BYTE bUV;
    BYTE bTable;
    LONG lRet;
    DWORD dwTableAddr;
    LPBYTE lpbTable;

    /* Get 9051 Table address */

    lpbTable = (LPBYTE)&CorrectionFor9051;

    /****  Y  *****/
	/* Generate _CorrectionFor9051 Table */

	dScale = 255.0 / ( 115.0 - 7.0 );
	for (i = 0; i <= 127; i++)
	{
		/* Coded for debugging */

		dY = (double)(i - 7);
		dY = (dY + dBri) * dScale;

		if (dY > 255.0)
		{
			dY = 255.0;
		}

		if (dY < 0.0)
		{
			dY = 0.0;
		}

		dY = dY / 255.0;

		/* Gamma */
		dY = pow (dY, dGam);

		dY = (dY * 255 * dCon) + 0.5;

		if (dY > 255.0)
		{
			dY = 255.0;
		}

		if (dY < 0.0)
		{
			dY = 0.0;
		}

		bY = (BYTE) dY;
#if 0
        bTable = *lpbTable;

        /* DEBUG */
        if (bY != bTable)
        {
        	lRet = 0L; /* DEBUG */
        }
#endif
        *lpbTable++ = bY;
	}

    /****  UV  ***** (Note : Should really have seperate U and V tables) */
	/* Generate _UVCorrectionFor9051 Table */

	for (i = 0; i <= 127; i++)
	{
		dUV = (double)i;

		/* Coded for debugging */

		if (dUV < 64.0)
		{
			/* UV positive */
			dUV = dUV * dSat;

	        if (dUV > 50.0)
	        {
	        	dUV = 50.0;
	        }

	        if (dUV < -51.0)
	        {
	        	dUV = -51.0;
	        }

	        dUV = dUV * 127.0 / 50.0;
		}
		else
		{
			/* UV negative */
			dUV = (double) -(128.0 - dUV);

			dUV = dUV * dSat;

	        if (dUV > 50.0)
	        {
	        	dUV = 50.0;
	        }

	        if (dUV < -51.0)
	        {
	        	dUV = -51.0;
	        }

	        dUV = dUV * 128.0 / 51.0;
		}

        if (dUV < 0.0)
        {
        	dUV = 256.0 + dUV;
        }

		dUV = dUV + 0.5;

		bUV = (BYTE) dUV;
#if 0
        bTable = *lpbTable;

        /* DEBUG */
        if (bUV != bTable)
        {
        	lRet = 0L; /* DEBUG */
        }
#endif
        *lpbTable++ = bUV;
	}

	BuildRGB16LUT((void _far *)RGB565Tab,BRI_DEFAULT,	/* build RGB 565 LUT */
							SAT_DEFAULT,
							CON_DEFAULT,
							RGB565_SELECT);	

	return (LONG)OK;

}/*Setup9051Tables*/

/***************************************************************************/
/* End of File */
/***************************************************************************/
