#include <conio.h>
#include <malloc.h>
#include "penny.h"



/* board information is placed in this structure */
PENNYINFO _PennyInfo;           /* Penny pcb info structure */

/* and this is a pointer to the first structure in the linked list */
PENNYPTR  _PennyPtr = &_PennyInfo;


/* Find a Penny PCB in IO space, read its capabilities into the structure */

PENNYPTR FindPenny()
{
	UWORD ioaddr, mapreg;
	WORD penniesfound = FALSE;

	/* initialise the first penny structure */
	InitPennyStruct();
	_PennyPtr->PrevPenny = NULL;

	/* start looking for penny pcb's at 280 Hex */
	ioaddr = PENNY_BASE_IOADDR;

	while( ioaddr <= PENNY_LAST_IOADDR )
		{
		/* input the value from the MAP chip POS register */
		mapreg = ReadPenny( ioaddr, PENNY_MAP_POSREG1 );

		if( mapreg == PENNY_ID_1 )        /* probably found one */
			{
			/* increment to next POS register address and read it */
            mapreg = ReadPenny( ioaddr, PENNY_MAP_POSREG2 );

			if( mapreg == PENNY_ID_2 )    /* definitely found one */
				{
				/* if finding second or more penny */
				if( penniesfound++ )
					{
					/* allocate a new structure for the new penny */
					_PennyPtr->NextPenny = (PENNYPTR)malloc( sizeof( PENNYINFO ) );
					if( _PennyPtr->NextPenny == NULL )
						{
						_PennyPtr = &_PennyInfo;
						return (PENNYPTR)NFG;
						}

					/* now make links forward and backward to the structures*/
					_PennyPtr->NextPenny->PrevPenny = _PennyPtr;
					_PennyPtr = _PennyPtr->NextPenny;

					/* and initialise the new structure */
                    InitPennyStruct();
                    }

				/* save the io address */
				_PennyPtr->PennyAddr = ioaddr;

				/* to check for a microchannel machine, */
				/* increment to last POS register address and read it */
				mapreg = ReadPenny( ioaddr, PENNY_MAP_POSREG3 );
				if( mapreg & MICROCHANNEL_BIT )
					_PennyPtr->StreamChip.MicroChannel = TRUE;

				/* read the map HIRR register for location of HI */
                mapreg = ReadPenny( ioaddr, PENNY_MAP_HIRR );
				_PennyPtr->HIAddr = (mapreg & 0xf000) >> 4;
				mapreg = ReadPenny( ioaddr, (PENNY_MAP_HIRR + 0x10) );
				_PennyPtr->HIAddr |= ((mapreg & 0x0f00) << 4);

				/* read the mapchip config register */
				mapreg = ReadPenny( ioaddr, PENNY_MAP_CFGREGLO ) & LO_MASK;
				mapreg |= ReadPenny( ioaddr, PENNY_MAP_CFGREGHI ) & HI_MASK;

				/* dump the whole word into the mapchip bitfield union */
				_PennyPtr->MapChip.MapChipWord  = mapreg;

				/* Write TRUE if WORD mode (in memory) is enabled */
				if( (_PennyPtr->MapChip.MapChipWord & WORDMODE_BITS)== WORDMODE_BITS )
				     _PennyPtr->StreamChip.WordMode = TRUE;

				/* put raw input into the streambreg0 bitfield union */
				_PennyPtr->StreamBReg0.StreamBReg0Word =
							     ReadPenny( ioaddr, PENNY_STREAMB_REG0 );

				/* ok, start identifying the penny pcb and its capabilities */
				if( GetPcbIdent( ioaddr ) >= OK )
					{
					if( GetPennyLevel( ioaddr ) != OK )
						{
						/* reset to the first penny structure just in case */
						_PennyPtr = &_PennyInfo;
						return (PENNYPTR)NFG;
						}

					/* write the boards name into the structure */
					SetPennyName();
					}
				}
			}

		/* increment to next io address */
        ioaddr += PENNY_IO_JMP;
		}

	/* reset the struct pointer to first penny found */
	_PennyPtr = &_PennyInfo;

	/* return how many penny boards found */
	return _PennyPtr;
}


/* Get the Penny PCB Identification */

WORD GetPcbIdent( ioaddr )
UWORD ioaddr;      /* the current penny pcb io address */
{
	UWORD streambreg;
	BOOL roms;

    /* output the ROM address and then the info register address */
	/* input the value from the MAP chip config register */
	streambreg = ReadPenny( ioaddr, PENNY_ROM_ID1 );

	if( streambreg == PENNY_ID_1 )        /* probably found a ROM based pcb */
		{
		/* increment to next id word address and read it */
        streambreg = ReadPenny( ioaddr, PENNY_ROM_ID2 );

		if( streambreg == NNIOS_ID )    /* definitely found one */
			roms = TRUE;
		}
	else
		roms = FALSE; /* not a rom based penny pcb */

	/* input the value from the STREAMB chip config register */
	streambreg = ReadPenny( ioaddr, PENNY_STREAMB_REG1 );

	/* put this raw into the streambreg1 bitfield union */
	_PennyPtr->StreamBReg1.StreamBReg1Word = streambreg;

	/* is there a vga cable present, this is active low! */
	if( !(streambreg & PENNY_VGA_CABLE) )
		_PennyPtr->StreamChip.VgaCable = TRUE;

	/* test if Sync on Grn dip switch is set */
	if( !_PennyPtr->StreamChip.MicroChannel )
		if( streambreg & PENNY_SOG )
			_PennyPtr->StreamChip.SogEnable = TRUE;

	/* get monitor type from streamb. these bits are set by the monitor */
	_PennyPtr->StreamChip.MonitorType = streambreg & PENNY_MON_MASK;

	/* calculate the board type using galens formula */
	_PennyPtr->BoardType = (WORD)((~((streambreg & PENNY_BID_MASK) >> 10) & 5)
						 + (!((streambreg & PENNY_MON3_MASK) >> 3) * 8) +
						 + (roms*2)) + 40 ;

	/* return true if its a valid boardtype */
	if( (_PennyPtr->BoardType >= PENNY_1024) &&
		(_PennyPtr->BoardType <= PENNY_1600_ROM) )
		return TRUE;
	else
		return FALSE;   /* else, no penny board type was really found */
}


/* identify the penny mem configuration and set globals */

WORD GetPennyLevel( ioaddr )
UWORD ioaddr;      /* the current penny pcb io address */
{
	UWORD streambreg, level;

    /* output the ROM address and then the info register address */
	/* input the value from the STREAMB chip config register */
	streambreg = ReadPenny( ioaddr, PENNY_STREAMB_REG2 );

	/* make sure streamb reg has valid data, do pennymemtest() if not */
    level = (streambreg & PENNY_LEVELMASK) >> 12;

	/* if the validity bit is not set, we must set the memory strap */
	if( !(streambreg & PENNY_VALID_DATA) )
		{
        PennyMemTest( ioaddr );
        WritePennyLevel( ioaddr );
        streambreg = ReadPenny( ioaddr, PENNY_STREAMB_REG2 );
        level = (streambreg & PENNY_LEVELMASK) >> 12;

		/* if still not valid, there must be a problem with the hardware */
		if( !(streambreg & PENNY_VALID_DATA) )
        	return( NOT_OK );

		WritePenny(ioaddr, PENNY_STREAMB_REG2, (streambreg ^= PENNY_VALID_DATA) );
		}

	/* Write the penny's memory level from streamb reg */
	_PennyPtr->BoardLevel = level;

	/* put this raw into the streambreg1 bitfield union */
	_PennyPtr->StreamBReg2.StreamBReg2Word = streambreg;

	/* test if Boot from ROM is selected */
	if( streambreg & PENNY_ROMBOOT )
		_PennyPtr->StreamChip.RomBoot = TRUE;

	/* is there extra memory on board? */
	if( streambreg & PENNY_EXTRAMEM )
		_PennyPtr->StreamChip.ExtraMemory = TRUE;

	/* which emulation type is selected */
	_PennyPtr->StreamChip.EmulationType = (BYTE)((streambreg & PENNY_EMULTYPE) >> 7);

	/* there is the chance that emulation has not been initialised */
	/* in this case i manualy test for VGA cable and report as indicated */
	if( !_PennyPtr->StreamChip.EmulationType && _PennyPtr->StreamChip.VgaCable )
		_PennyPtr->StreamChip.EmulationType = PENNY_VGA_LOOP;

	/* fill in the actual memory size on the penny structure */
	if( (streambreg & PENNY_IRAMMASK) == PENNY_NO_IRAM )
		{
		_PennyPtr->IramSize = 0;
		}
    else if( (streambreg & PENNY_IRAMMASK) == PENNY_1_2MB_IRAM )
		{
		_PennyPtr->IramSize = 512;
		}
	else if( (streambreg & PENNY_IRAMMASK) == PENNY_1MB_IRAM )
		{
		_PennyPtr->IramSize = 1024;
		}
	else if( (streambreg & PENNY_IRAMMASK) == PENNY_2MB_IRAM )
		{
		_PennyPtr->IramSize = 2048;
		}
    else if( (streambreg & PENNY_IRAMMASK) == PENNY_4MB_IRAM )
		{
		_PennyPtr->IramSize = 4096;
		}

	if( (streambreg & PENNY_VRAMMASK) == PENNY_1_2MB_VRAM )
		{
		_PennyPtr->VramSize = 512;
		}
	else if( (streambreg & PENNY_VRAMMASK) == PENNY_1MB_VRAM )
		{
		_PennyPtr->VramSize = 1024;
		}
    else if( (streambreg & PENNY_VRAMMASK) == PENNY_2MB_VRAM )
		{
		_PennyPtr->VramSize = 2048;
		}

	/* if there is minimum memory, iram is shadowed in vram */
	if( !_PennyPtr->IramSize && (_PennyPtr->VramSize >= 512) )
	  	{
		_PennyPtr->VramSize -= MIN_SHADOW_IRAM;
		_PennyPtr->IramSize += MIN_SHADOW_IRAM;
		}

    return OK;
}


/* test amount of memory present and write streamb reg 2 word */

VOID PennyMemTest( ioaddr )
UWORD ioaddr;      /* the current penny pcb io address */
{
	UWORD flag = 0x0040;

	flag |= ReadPenny(ioaddr, PENNY_STREAMB_REG2) & 0x0010;
	WritePenny(ioaddr, PENNY_STREAMB_REG2, flag);
	WritePenny(ioaddr, 0x07FFFFF0, 0xABCD);

	if (ReadPenny(ioaddr, 0x07FFFFF0) == 0xABCD)
		{
		/* There are DRAMs, check how much */
		WritePenny(ioaddr, 0x077FFFF0, 0xAAAA);
		WritePenny(ioaddr, 0x07BFFFF0, 0xBBBB);
		if (ReadPenny(ioaddr, 0x077FFFF0) == 0xAAAA)
			{
			/* Dram chips are 4Mbit */
			WritePenny(ioaddr, 0x067FFFF0, 0x9999);
			if (ReadPenny(ioaddr, 0x067FFFF0) == 0x9999 &&
				ReadPenny(ioaddr, 0x077FFFF0) == 0xAAAA)
				{
				flag |= 0x0010;
				}
			}
		else
			{
			/* Dram chips are 1Mbit */
			flag |= 0x0004;
			WritePenny(ioaddr, PENNY_STREAMB_REG2, flag);
			WritePenny(ioaddr, 0x07FFFFF0, 0xDDDD);
			WritePenny(ioaddr, 0x07BFFFF0, 0xEEEE);
			if (ReadPenny(ioaddr, 0x07BFFFF0) == 0xEEEE &&
				ReadPenny(ioaddr, 0x07FFFFF0) == 0xDDDD)
				{
				/* 1MB of DRAM */
				flag |= 0x0010;
				}
			}
		}
	else
		{
		flag |= 0x0002;   /* No DRAMs */
		}

	/* VRAM detection */
	WritePenny(ioaddr, 0x00000000L, 0x5555);
	WritePenny(ioaddr, 0x00000010L, 0x6666);
	if (ReadPenny(ioaddr, 0x00000010L) == 0x6666 &&
		ReadPenny(ioaddr, 0x00000000L) == 0x5555)
		{
		WritePenny(ioaddr, 0x00800000L, 0x7777);
		if (ReadPenny(ioaddr, 0x00800000L) == 0x7777)
		flag |= 0x20;
		}
	else
		flag |= 0x01;

	WritePenny(ioaddr, PENNY_STREAMB_REG2, flag);
}


/* Write the level information into the register for future use */

VOID WritePennyLevel( ioaddr )
UWORD ioaddr;             /* the current penny pcb io address */
{
	UWORD streambreg, dip;

	/* output the ROM address and then the info register address */
	/* input the value from the STREAMB chip config register */
    dip        = ReadPenny( ioaddr, PENNY_STREAMB_REG1 );
	streambreg = ReadPenny( ioaddr, PENNY_STREAMB_REG2 );

	/*  Calculate board level... */
_asm {
	test        dip,1000h
	jz          lev_BID3

	test        dip,8h
	jz          lev_MON3

	test        streambreg,2h
	jnz         level1

	test        streambreg,1h
	jnz         level2

	test        dip,400h
	jnz         level3
	or          streambreg, 04000h  ; #9GX 1280

	test        streambreg,20h
	jz          lev_check_iram
	or          streambreg,05000h
	jmp         lev_check_iram

level1:
	or          streambreg, 01000h
	test        streambreg,1
	jz          lev_plus

	test        dip,400h
	jz          lev_plus
	jmp         leveldone

level2:
	or          streambreg,02000h
	test        dip,400h
	jz          lev_plus
	jmp         lev_check_iram

level3:
	or          streambreg,03000h
	jmp         lev_check_vram

lev_BID3:       ; check level of Twig or 1600
lev_MON3:       ; Reserved board type

lev_check_vram:
	test        streambreg,20h
	jnz         lev_plus
lev_check_iram:
	test        streambreg,10h
	jnz         lev_plus

	test        streambreg,4h
	jnz         leveldone
lev_plus:
	or          streambreg,08000h

leveldone:
    }

	WritePenny( ioaddr, PENNY_STREAMB_REG2, streambreg );
}


/* Read a WORD from the penny pcb at a specified location */

WORD ReadPenny( ioaddr, tiaddr )
UWORD ioaddr;
DWORD tiaddr;
{
	outpw( ioaddr    , (UWORD) (tiaddr & 0x0000FFFF) );
	outpw( ioaddr + 2, (UWORD)((tiaddr & 0xFFFF0000) >> 16) );
	return( inpw( ioaddr + 4 ) );
}


/* Write a WORD to the penny pcb at a specified location */

VOID WritePenny( ioaddr, tiaddr, wordsent )
UWORD ioaddr;
DWORD tiaddr;
UWORD wordsent;
{
	outpw( ioaddr    , (UWORD) (tiaddr & 0x0000FFFF) );
	outpw( ioaddr + 2, (UWORD)((tiaddr & 0xFFFF0000) >> 16) );
	outpw( ioaddr + 4, wordsent );
}


/* put the board name into the penny structure */

VOID SetPennyName()
{
	if( _PennyPtr->BoardType == 40 )
		strcpy( _PennyPtr->BoardName, "#9GX - 1024" );
	else if( _PennyPtr->BoardType == 41 )
		strcpy( _PennyPtr->BoardName, "#9GX - 1280" );
	else if( _PennyPtr->BoardType == 42 )
		strcpy( _PennyPtr->BoardName, "#9GX - 1024 w/Roms" );
	else if( _PennyPtr->BoardType == 43 )
		strcpy( _PennyPtr->BoardName, "#9GX - 1280 w/Roms" );
	else if( _PennyPtr->BoardType == 44 )
		strcpy( _PennyPtr->BoardName, "#9GX - Video" );
	else if( _PennyPtr->BoardType == 45 )
		strcpy( _PennyPtr->BoardName, "#9GX - 1600" );
	else if( _PennyPtr->BoardType == 46 )
		strcpy( _PennyPtr->BoardName, "#9GX - Video w/Roms" );
	else if( _PennyPtr->BoardType == 47 )
		strcpy( _PennyPtr->BoardName, "#9GX - 1600 w/Roms" );
}



VOID InitPennyStruct()
{
	_PennyPtr->PennyAddr     = 0;
	_PennyPtr->HIAddr        = 0;
	_PennyPtr->BoardLevel    = 0;
	_PennyPtr->BoardType     = 0;
	_PennyPtr->IramSize      = 0;
	_PennyPtr->VramSize      = 0;
	_PennyPtr->NextPenny     = NULL;
	_PennyPtr->BoardName[0]  = 0;
	_PennyPtr->StreamChip.EmulationType = FALSE;
	_PennyPtr->StreamChip.MonitorType   = 0;
	_PennyPtr->StreamChip.VgaCable      = FALSE;
	_PennyPtr->StreamChip.SogEnable     = FALSE;
	_PennyPtr->StreamChip.MicroChannel  = FALSE;
	_PennyPtr->StreamChip.Interlaced    = FALSE;
	_PennyPtr->StreamChip.RomBoot       = FALSE;
	_PennyPtr->StreamChip.ExtraMemory   = FALSE;
	_PennyPtr->StreamChip.WordMode      = FALSE;
}


#if 0
; Calculate board level...

	btst    0Ch,dip
	jrz lev_BID3

	btst    3,dip
	jrz lev_MON3

	btst    1,memstr
	jrnz    level1

	btst    0,memstr
	jrnz    level2

	btst    0Ah,dip
	jrnz    level3
	ori 04000h,memstr   ; #9GX 1280

	btst    5,memstr
	jrz lev_check_iram
	ori 05000h,memstr
	jruc    lev_check_iram

level1:
	ori 01000h,memstr
	btst    0,memstr
	jrz lev_plus

	btst    0Ah,dip
	jrz lev_plus
	jruc    leveldone

level2:
	ori 02000h,memstr
	btst    0Ah,dip
	jrz lev_plus
	jruc    lev_check_iram

level3:
	ori 03000h,memstr
	jruc    lev_check_vram

lev_BID3:   ; check level of Twig or 1600
lev_MON3:   ; Reserved board type

lev_check_vram:
	btst    5,memstr
	jrnz    lev_plus
lev_check_iram:
	btst    4,memstr
	jrnz    lev_plus

	btst    2,memstr
	jrnz    leveldone
lev_plus:
	ori 08000h,memstr

leveldone:
	move    memstr,@MEMSTRAP
	#endif
