/**********************************************************************
 Program : 4029if
 Pupose  : IBM 4029 plain text filter for lpd
 Version : 0.2 Beta
 Date    : 11/04/92
 Author  : Matt King (Lexmark International, Inc.)
-----------------------------------------------------------------------
 Copyright (c) 1992 Lexmark International, Inc.
 Program Property of Lexmark International, Inc.
 All Rights Reserved.
**********************************************************************/

/**********************************************************************
 Include files
**********************************************************************/
#include <stdio.h>
#include <string.h>

/**********************************************************************
 Constant and Macro definitions
**********************************************************************/
#define VERSION "0.2 Beta"

#define SIC_PPDS_DAT "\033[K\003\000\005\061\001"
#define SIC_PPDS_LEN 8
#define SIC_PS_DAT   "\033[K\003\000\005\061\010"
#define SIC_PS_LEN   8

/* PPDS Setup string */
/* Sets 66 lines/page, 12-pitch Courier, 1/2-inch left margin, default tab-stops */
#define SETUP_DAT "\033[\\\002\000\260\004\033A\300\0332\033:\033X\006\000\033D\016\026\036&.6>FNV^\000"
#define SETUP_LEN 32

#define EOT '\004'

#define cbn(s) ((s)?(s):"[Null]")

/**********************************************************************
 Global variables
**********************************************************************/
int bin_mode=0, width=-1, length=-1, indent=-1;
int query_version=0;
char *filt_name=NULL, *login=NULL, *host=NULL, *acct_file=NULL;

#ifdef DEBUG
  int filt_count=0;
#endif

/**********************************************************************
 Function prototypes
**********************************************************************/
int parse_args();
int ppds();
int ps();
int filter();
void send_cmd();

/**********************************************************************
 Function: main
**********************************************************************/
main(argc, argv)
  int argc;
  char *argv[];
{
  int c1, c2;

  parse_args(argc, argv);

#ifdef DEBUG
  fprintf(stderr, "%s STARTED\n", filt_name);
#endif

  if(query_version)
    {
      fprintf(stderr, "%s -- IBM 4029 plain text filter for lpd\n", filt_name);
      fprintf(stderr, "Copyright 1992 Lexmark International, Inc.\n");
      fprintf(stderr, "Sun-4, SunOS 4.1 Version %s.\n", VERSION);
      exit(1);
    }

  if( (c1=getchar())!=EOF )
    {
      if( (c2=getchar())!=EOF )
	{
	  if( (c1=='%' && c2=='!') || c1==EOT) /* Has Postscript header or EOT */
	    ps(c1,c2);
	  else
	    ppds(c1,c2); /* Assume ASCII text */
	}
      else
	ppds(c1,c2); /* Only one character assume ASCII text */
    }
  else
    return 0; /* Nothing there */
      
#ifdef DEBUG
      fprintf(stderr, "%s ENDING\n", filt_name);
#endif
  fflush(stderr);
  fflush(stdout);
  exit(0);
}

/**********************************************************************
 Function: ppds
**********************************************************************/
int ppds(c1, c2)
  int c1, c2;
{
  int c;
#ifdef DEBUG
  fputs("  PPDS\n",stderr);
#endif
  send_cmd(SIC_PPDS_DAT, SIC_PPDS_LEN); /* Enter PPDS mode */

if(!bin_mode)
  send_cmd(SETUP_DAT, SETUP_LEN); /* Setup PPDS mode */

  filter(c1); /* First character */

  if(c2!=EOF) /* Second character */
    filter(c2);
  else
    return 0;

  while(1)    /* Rest of characters */
    {
      if((c=getchar())!=EOF)
	filter(c);
      else
	break;
      c1=c;
    }

#ifdef DEBUG
  fprintf(stderr, "  Filtered %d bytes.\n", filt_count);
#endif

  if(c1!='\f') /* Make sure page gets kicked */
    putchar('\f');
  return 0;
}

/**********************************************************************
 Function: filter
**********************************************************************/
int filter(c)
  int c;
{
#ifdef DEBUG
  filt_count++;
#endif

  switch(c)
    {
    case '\n':
      if(!bin_mode)
	putchar('\r');
      /* Fall through */
    default:
      putchar(c);
#ifdef DEBUG2
      fputc(c,stderr);
#endif
    }
  return c;
}

/**********************************************************************
 Function: ps
**********************************************************************/
int ps(c1, c2)
  int c1, c2;
{
  int c;

#ifdef DEBUG
  fputs("  PS\n",stderr);
#endif
  send_cmd(SIC_PS_DAT, SIC_PS_LEN); /* Enter PS mode */

  putchar(c1);
  putchar(c2);

  while(1)    /* Rest of characters */
    {
      if((c=getchar())!=EOF)
	putchar(c);
      else
	break;
      c1=c;
    }

  if(c1!=EOT) /* Make sure EOT is sent */
    putchar(EOT);
  return 0;
}

/**********************************************************************
 Function: parse_args
**********************************************************************/
int parse_args(argc, argv)
  int argc;
  char *argv[];
{
  int i;
  char *word;

#ifdef DEBUG2
  fprintf(stderr,"Command line: ");
  for(i=0;i<argc;i++)
    {
      fprintf(stderr, "%s ",argv[i]);
    }

  fputc('\n',stderr);
#endif

  for(i=1;i<argc;i++)
    {
      word=argv[i];
      
      if(word[0]=='-')
	{
	  switch(word[1])
	    {
	    case 'v':
	      query_version=1;
	      break;
	    case 'c':
	      bin_mode=1;
	      break;
	      
	    case 'w':
	      if(word[2])
		width=atoi(&word[2]);
	      else
		width=atoi(argv[++i]);
	      break;
	      
	    case 'l':
	      if(word[2])
		length=atoi(&word[2]);
	      else
		length=atoi(argv[++i]);
	      break;
	      
	    case 'i':
	      if(word[2])
		indent=atoi(&word[2]);
	      else
		indent=atoi(argv[++i]);
	      break;
	      
	    case 'n':
	      if(word[2])
		login=&word[2];
	      else
		login=argv[++i];
	      break;
	      
	    case 'h':
	      if(word[2])
		host=&word[2];
	      else
		host=argv[++i];
	      break;
	    }
	}
      else
	acct_file=word;
    }

  if(filt_name=strrchr(argv[0],'/'))
    filt_name++;
  else
    filt_name=argv[0];

#ifdef DEBUG2
  fprintf(stderr,"Invoked as: [%s]\n", filt_name);
  fprintf(stderr,"  Width:%d Length:%d Indent:%d\n",
	  width, length, indent);
  fprintf(stderr,"  Login: %s Host: %s\n", cbn(login), cbn(host));
  fprintf(stderr,"  Accounting: %s\n", cbn(acct_file));
#endif

  return 0;
}

/**********************************************************************
 Function: send_cmd
**********************************************************************/
void send_cmd(cmd, len)
  char *cmd;
  int len;
{
  int i;

  for(i=0;i<len;i++)
    putchar(cmd[i]);
}
