/*
 * Copyright (c) 1996 Distributed Processing Technology Corporation
 * All rights reserved.
 *
 * Redistribution and use in source form, with or without modification, are
 * permitted provided that redistributions of source code must retain the
 * above copyright notice, this list of conditions and the following disclaimer.
 *
 * This software is provided `as is' by Distributed Processing Technology and
 * any express or implied warranties, including, but not limited to, the
 * implied warranties of merchantability and fitness for a particular purpose,
 * are disclaimed. In no event shall Distributed Processing Technology be
 * liable for any direct, indirect, incidental, special, exemplary or
 * consequential damages (including, but not limited to, procurement of
 * substitute goods or services; loss of use, data, or profits; or business
 * interruptions) however caused and on any theory of liability, whether in
 * contract, strict liability, or tort (including negligence or otherwise)
 * arising in any way out of the use of this driver software, even if advised
 * of the possibility of such damage.
 *
 * DPT SCSI host adapter driver
 *
 * TODO:
 *      Compute scatter/gather maps when buffers are queued, not when used
 *      Compute CCB's when buffers are queued
 *      Add support for EATA pass through.
 *      Add support for Software RAID-0 over multiple Adapters.
 *
 *      dpt V1.0C 1996/11/08 salyzyn@dpt.com
 *              Timeing out on the card is bad, removed timeouts.
 *
 *      dpt V1.0B 1996/09/01 salyzyn@dpt.com
 *              residual handling is wrong for block allocation of memory.
 *
 *      dpt V1.0A 1996/08/01 salyzyn@dpt.com
 *              Updated the BUS Reset code still furthur.
 *
 *      dpt V1.08 1996/06/10 salyzyn@dpt.com
 *              Addressed comments from Phil. Cleaned up some code.
 *
 *      bsdi 1996/06/03 pjd@bsdi.com
 *              file name changes only
 *
 *      bsdi 1996/05/31 pjd@bsdi.com
 *              changed probe and attach routines to use DRQ from config
 *              instead of all DRQs. Added comments.
 *
 *      dpt V1.06 1996/03/06 salyzyn@dpt.com
 *              A Test Unit Ready is not a valid way of probing devices. Some
 *                      devices will return their TAGs links with LUN 0 if a
 *                      non-existent LUN is tested. One must do an Inquiry
 *                      first to check if the LUN is valid.
 *              Added debug code to check for over-freed CCBs.
 *              Pass through links added.
 *              Removed automatic Interpret bit setting if ID == HBA.
 *      dpt V1.05 1996/02/27 salyzyn@dpt.com
 *              Some minor Cleanup. Removal of Redundant Code addition of
 *                      disclaimers.
 *              Found a bug in Multichannel and Wide support and fixed it.
 *              Allowed the OS to jam the requests to the drive (optional
 *                      behavior because no noticeable difference in
 *                      performance [OVERLAPCMDS]).
 *      dpt V1.04 1996/02/16 salyzyn@dpt.com
 *              Changed over to V2.1 of the BSDi OS, and made changes to the
 *                      driver to allow it to co-exist with this OS version in
 *                      a cleaner manner.
 *              Limited the number of CCBs that can be sent to a single HBA.
 *      dpt V1.03 1996/02/13 salyzyn@dpt.com
 *              Fixed bug in forced EISA handling that failed to eliminate
 *                      ISA addresses that the card mapped to.
 *              Added support for 2012 and 2011 by allowing CP_READ_CFG_PIO
 *                      command as backup. Let this work also for ISA cards
 *                      when allocation excedes 16MB ISA limit. Older firmware
 *                      that fails to identify DMA and IRQ channels in the
 *                      EATA config remain unsupported.
 *              A bug in EISA primary support which caused primary address to
 *                      always end up pointing to slot 15 was eliminated.
 *              Timeout within dpticmd for certain SCSI commands added to
 *                      prevent ungraceful lockup situations.
 *              Not necessarily a bug, but cleaned up defaulting of LUN and
 *                      Target maximums.
 *      dpt V1.02 1996/02/06 salyzyn@dpt.com
 *              Added the DPT signature
 *              Improved Scatter Gather by removing ab_subr Dependancy
 *              Added multichannel and >7 Scsi Target IDs
 *              Added some depth to the Go_CCBs so that multiple requests
 *                      to the same target device can be supported.
 *      dpt V1.01 1996/01/30 salyzyn@dpt.com
 *              Added EISA support.
 *              Improved the Scatter Gather by coalescing blocks.
 *              Added PCI support, but untested.
 *              Dynamically allocated the Sense storage for Auto Request
 *                      sense functionality.
 *      dpt V1.00 1995/12/13 salyzyn@dpt.com
 */

#define DPT_VERSION     1
#define DPT_REVISION    '0'
#define DPT_SUBREVISION 'C'
#define DPT_MONTH       11
#define DPT_DAY         8
#define DPT_YEAR        16      /* 1996 */

/*
 *      DPT_DEBUG_FLAGS
 *              - any non-zero value enables the ha_debug flag and is
 *                equivalent to a DPT_DEBUG_BASE being set.
 */
# define DPT_DEBUG_BASE    0x0001       /* Minor Overhead for validity checks */
# define DPT_DEBUG_CHECK   0x0002       /* Additional more detailed checks */
# define DPT_DEBUG_CALL    0x0004       /* Print subroutine entry */
# define DPT_DEBUG_VERBOSE 0x0004       /* Detailed packet entry */
# define DPT_DEBUG_SET     0x8000       /* Set ha_debug default */
#define DPT_DEBUG_FLAGS 0

#include        <i386/isa/dptsig.h>

static dpt_sig_S dpt_sig = {
        'd', 'P', 't', 'S', 'i', 'G', SIG_VERSION, PROC_INTEL,
        PROC_386 | PROC_486 | PROC_PENTIUM | PROC_SEXIUM, FT_HBADRVR, 0,
        OEM_DPT, OS_BSDI_UNIX, CAP_ABOVE16MB, DEV_ALL, ADF_ALL_MASTER,
        0, 0, DPT_VERSION, DPT_REVISION, DPT_SUBREVISION,
        DPT_MONTH, DPT_DAY, DPT_YEAR,
        "DPT BSDi 2.1 Unix SCSI HBA Driver"
        /*        ^^^   dptattach alters these to match OS */
};
extern char     osrelease[];

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/proc.h>
#include <sys/conf.h>

#include <vm/vm.h>

#include <dev/scsi/scsi.h>
#include <dev/scsi/scsivar.h>
#include <dev/scsi/disk.h>
#include <dev/scsi/tape.h>

#include <machine/cpu.h>

#include <i386/isa/isa.h>
#include <i386/isa/isavar.h>
#include <i386/eisa/eisa.h>
#include <i386/pci/pci.h>
#include <i386/isa/dma.h>
#include <i386/isa/icu.h>

extern int maxmem;      /* Gives us the number of pages of available memory */

#define STATIC static

/* Per-request Flag values      */
#define FL_NOSLEEP      0x01    /* Not a user... don't sleep            */
#define FL_NOMASK       0x02    /* dont allow interrupts.. booting      */
#define FL_QUIET        0x04    /* Don't print error messages           */


/* Configuration Definitions    */

#define NCCBS           64      /* Max CCBs, Also Max Queue Size        */
#define SG_SIZE         64      /* Scatter Gather list Size             */
/* Maximum TARGET Supported     */
#define MAX_TARGET_ID   (int)(sizeof(((struct hba_softc *)0L)->hba_targets) \
                       / sizeof(((struct hba_softc *)0L)->hba_targets[0]))
/* Maximum LUN Supported        */
#define MAX_LUN         (int)(sizeof(((struct targ *)0L)->t_units) \
                       / sizeof(((struct targ *)0L)->t_units[0]))
/* Maximum Channel # Supported  */
#define MAX_CHANNEL     (int)(sizeof(((DptConfig_t *)0L)->HBA) \
                       / sizeof(((DptConfig_t *)0L)->HBA[0]))
/* Allow the driver to queue up multiple commands at a time */
#define OVERLAPCMDS

#if (defined(OVERLAPCMDS) && (NCCBS > 1))
#       define  IF              while
#else
#       define  IF              if
#endif

/************************** board definitions *******************************/

/*
 * I/O Port Interface
 */

typedef struct {
        char    dpt_data;
        char    dpt_error;
        char    dpt_addr[4];
#define immed_mod dpt_addr[3]
        char    immed_func;
        union   {
                char    dpt_command;
                char    dpt_status;
        }       dpt_reg;
        char    dpt_aux_status;
} DptPort_t;

#define HA_DATA         ((int)(&(((DptPort_t *)0L)->dpt_data)))
#define HA_ERROR        ((int)(&(((DptPort_t *)0L)->dpt_error)))
#define HA_DMA_BASE     ((int)(((DptPort_t *)0L)->dpt_addr))
#define HA_COMMAND      ((int)(&(((DptPort_t *)0L)->dpt_reg.dpt_command)))
/* EATA PIO commands */
#define CP_READ_CFG_PIO 0xF0    /* Read Configuration Data, PIO   */
#define CP_PIO_CMD              0xF2    /* Execute Command, PIO           */
#define CP_TRUNCATE_CMD 0xF4    /* Truncate Transfer Command, PIO */
#define CP_EATA_RESET           0xF9    /* Reset Controller And SCSI Bus  */
/* EATA DMA commands */
#define CP_READ_CFG_DMA 0xFD    /* Read Configuration Data, DMA   */
#define CP_DMA_CMD              0xFF    /* Execute Command, DMA           */
/* Other commands */
#define CP_IMMEDIATE            0xFA    /* EATA Immediate command         */

#define HA_STATUS       ((int)(&(((DptPort_t *)0L)->dpt_reg.dpt_status)))
#define HA_ST_ERROR             0x01
#define HA_ST_INDEX             0x02
#define HA_ST_CORRCTD           0x04
#define HA_ST_DRQ               0x08
#define HA_ST_SEEK_COMPLETE     0x10
#define HA_ST_WRT_FLT           0x20
#define HA_ST_READY             0x40
#define HA_ST_BUSY              0x80
#define HA_ST_DATA_RDY (HA_ST_SEEK_COMPLETE | HA_ST_READY | HA_ST_DRQ)

#define HA_AUX_STATUS   ((int)(&(((DptPort_t *)0L)->dpt_aux_status)))
#define HA_AUX_BUSY             0x01    /* Aux Register Busy bit.         */
#define HA_AUX_INTR             0x02    /* Aux Register Interrupt Pending */

#define DPT_NPORT       sizeof(DptPort_t)

/*************************************************************************
**                      EATA Command Packet Definitions                 **
**************************************************************************/

#define EATA_SIG (((long)'E')|(((long)'A')<<8L)|(((long)'T')<<16L)|(((long)'A')<<24L))

/*******************************************************************************
** EATA Command Packet Control Bits Structure Definition - This Controlls The **
**      Data Direction, CDB Interpret, And Other Misc Controll Bits.          **
*******************************************************************************/

typedef vm_offset_t     vm_addr_t;      /* Byte reversed addresses      */
typedef vm_size_t       vm_len_t;       /* Byte reversed counts         */

typedef struct {
        union { struct {
                u_char  reserved[5];
                u_char  x_FWNest:1;     /* Firmware Nested Bit For RAID */
                u_char  reserved1:7;
                u_char  x_PhsUnit:1;    /* Physical Bit For RAID        */
                u_char  x_IAT:1;        /* Inhibit Address Translation  */
                u_char  x_Disable_Cache:1; /* Inhibit caching           */
                }       x;
                struct {
                u_char  y_SCSI_Reset:1; /* Issue A SCSI Bus Reset       */
                u_char  y_HBA_Init:1;   /* ReInitialize The Controller  */
                u_char  y_Auto_Req_Sen:1;/* Do Auto Request Sense       */
                u_char  y_Scatter_Gather:1;/* Data Pointer To S.G. List*/
                u_char  y_Quick:1;      /* Perform Quick PIOin, DMA in/out*/
                u_char  y_Interpret:1;  /* CDB Interpreted By Controller*/
                u_char  y_DataOut:1;    /* Data Out Phase With Command  */
                u_char  y_DataIn:1;     /* Data In Phase With Command   */
                u_char  y_Req_Len;      /* AutoRequestSense Data length.*/
                u_char  reserved2[3];
                u_char  y_FWNest;       /* Firmware Nested Byte for RAID*/
                u_char  y_PhsUnit;      /* Physical Byte for RAID       */
                u_char  y_id:5;         /* SCSI Target ID               */
                u_char  y_Channel:3;    /* HBA Channel Number           */
                }       y;
        }       cp_u;
#define cp_SCSI_Reset           cp_u.y.y_SCSI_Reset
#define cp_HBA_Init             cp_u.y.y_HBA_Init
#define cp_Auto_Req_Sen         cp_u.y.y_Auto_Req_Sen
#define cp_Scatter_Gather       cp_u.y.y_Scatter_Gather
#define cp_Quick                cp_u.y.y_Quick
#define cp_Interpret            cp_u.y.y_Interpret
#define cp_DataOut              cp_u.y.y_DataOut
#define cp_DataIn               cp_u.y.y_DataIn
#define cp_Req_Len              cp_u.y.y_Req_Len
#define cp_FWNest               cp_u.y.y_FWNest
#define cp_PhsUnit              cp_u.y.y_PhsUnit
#define cp_IAT                  cp_u.x.x_IAT
#define cp_Disable_Cache        cp_u.x.x_Disable_Cache
#define cp_id                   cp_u.y.y_id
#define cp_Channel              cp_u.y.y_Channel
        u_char  cp_msg_lun:3;           /* Message Lun          */
        u_char  reserved3:2;
        u_char  cp_msg_luntar:1;        /* Target Routine       */
        u_char  cp_msg_disco_rico:1;    /* Disconnect Reconnect */
        u_char  cp_msg_identify:1;      /* Indentify            */
        u_char  cp_msg[3];      /* Identify and Disconnect Message.     */
        union { struct {
                u_char  x_scsi_cmd;     /* Partial SCSI CDB def */
                u_char  x_extent:1;
                u_char  x_bytchk:1;
                u_char  x_reladr:1;
                u_char  x_cmplst:1;
                u_char  x_fmtdata:1;
                u_char  x_lun:3;
                u_char  x_page;
                u_char  reserved4;
                u_char  x_len;
                u_char  x_link:1;
                u_char  x_flag:1;
                u_char  reserved5:4;
                u_char  x_vendor:2;
                }       x;
                u_char  z[12];          /* Total SCSI CDB               */
        }       cp_w;
#define cp_cdb  cp_w.z
#define cp_scsi_cmd     cp_w.x.x_scsi_cmd
#define cp_extent       cp_w.x.x_extent
#define cp_lun          cp_w.x.x_lun
#define cp_page         cp_w.x.x_page
#define cp_len          cp_w.x.x_len
        vm_len_t  cp_datalen;   /* Byte Swapped Data Length In Bytes    */
        caddr_t   cp_vp;        /* Command Packet Virtual Pointer.      */
        vm_addr_t cp_dataDMA;   /* Byte Swapped Data Physical Address   */
        vm_addr_t cp_statDMA;   /* Byte Swapped Status Packet Physical  */
        vm_addr_t cp_reqDMA;    /* Byte Swapped AutoRequestSense Phys   */
} EataCp_t;

/*************************************************************************
** Scatter Gather List Element Structure.                               **
**************************************************************************/

typedef struct {        /* Remember these are byte reversed */
        vm_addr_t       data_addr;      /* Physical Data Address        */
        vm_len_t        data_len;       /* Count In Bytes               */
} SgElement_t;

typedef struct DptBounce {      /* DMA Bounce support */
        struct DptBounce *      bo_next;/* Next bounce page in list     */
        caddr_t         bo_dst; /* virtual address of destination       */
        vm_offset_t     bo_p;   /* physical address of bounce page      */
        caddr_t         bo_v;   /* Virtual address of bounce page       */
        vm_size_t       bo_len; /* amount of data in the page           */
} DptBounce_t;

/******************************************************************************
** Controller Command Block - CCB - This Structure Contains The EATA Command **
**   Packet, OsRequest_t Pointer And Other Command Control Information. The  **
**   EATA Command Packet Should Remain At The Begining Of The Structure For  **
**   Ease Of Processing The Command To The Controller.                       **
*******************************************************************************/

typedef struct scsi_sense       Scsi_Sense_t;

typedef struct DptCcb {
/************** First part for convenience should be an EataCp **********/
        EataCp_t        EataCp;

/***************************** Response flags ***************************/

        vm_size_t       ccb_Residue;    /* Residual non-transfered bytes */
        u_char          ccb_Host_Status;/* Host Command Complete Status */
        u_char          ccb_Target_Status;/* SCSI Command Complete Status */

        u_char          ccb_isa:1;      /* CCB ok for use in ISA system */
        vm_addr_t       ccb_addr;       /* CCB Physical Address.        */
        struct DptCcb * ccb_next;       /* Pointer To Next active CCB   */
        DptBounce_t *   ccb_Bounce;
        Scsi_Sense_t    ccb_sense;      /* Auto Request Sense data      */
                                        /* The Physical Address of the  */
                                        /* sense data is in EataCp      */
        void *          ccb_Device;     /* Device CCB is attached to    */

/********************** Additional flags for BSDi ***********************/

        scintr_fn       ccb_intr;       /* base interrupt handler       */
        struct device * ccb_intrdev;    /* link back to associated unit */
        u_long          ccb_stamp;      /* timestamp (for watchdog)     */

/********************** Scatter Gather Portion **************************/

        vm_addr_t       ccb_SG_addr;    /* Byte Swapped Physical S.G. List */
        /* Must be Last as it could be dynamically allocated */
        SgElement_t     ccb_SG[SG_SIZE];/* Scatter Gather List          */
} DptCcb_t;

/******************************************************************************
** Status Packet Data Structure - This Structure Is Returned By Controller   **
**   Upon Command Completion.   It Contains Status, Message Info And Pointer **
**   To The Initiating Command CCB (virtual).                                **
*******************************************************************************/

typedef struct {
        u_char          sp_HBA_stat:7;  /* Host Adapter Status          */
        u_char          sp_EOC:1;       /* End of Command (Not Safe)    */
        u_char          sp_SCSI_stat;   /* SCSI Bus Status              */
        u_char          reserved[2];    /* Reserved                     */
        vm_len_t        sp_inv_residue; /* Bytes Not Transfered         */
        DptCcb_t *      sp_vp;          /* Command Packet Virtual Pointer. */
        u_char          sp_ID_Message;
        u_char          sp_Que_Message;
        u_char          sp_Tag_Message;
        u_char          sp_Messages[9];
} DptStat_t;

/*
 *      sp_EOC is not `safe', so I will check sp_Messages[0] instead!
 */
#define DptStat_BUSY(x) ((x)->sp_Messages[0])
#define DptStat_Reset_BUSY(x)   ((x)->sp_Messages[0] = 0xA5, (x)->sp_EOC = 0, (x)->sp_vp = (DptCcb_t *)NULL)

/*************************************************************************
**              HBA Status Packet Host Status Definitions               **
**************************************************************************/

#define HA_NO_ERROR             0x00    /* No Error on command          */
#define HA_ERROR_SEL_TO         0x01    /* Device Selection Time Out    */
#define HA_ERROR_CMD_TO         0x02    /* Device Command Time Out      */
#define HA_ERROR_RESET          0x03    /* SCSI Bus was RESET !         */
#define HA_INIT_POWERUP         0x04    /* Initial Controller Power Up  */
#define HA_UNX_BUSPHASE         0x05    /* Unexpected BUS Phase         */
#define HA_UNX_BUS_FREE         0x06    /* Unexpected BUS Free          */
#define HA_BUS_PARITY           0x07    /* SCSI Bus Parity Error        */
#define HA_SCSI_HUNG            0x08    /* SCSI Bus Hung                */
#define HA_UNX_MSGRJCT          0x09    /* Unexpected Message Reject    */
#define HA_RESET_STUCK          0x0A    /* SCSI Bus Reset Stuck         */
#define HA_RSENSE_FAIL          0x0B    /* Auto-Request Sense Failed    */
#define HA_PARITY               0x0C    /* HBA Memory Parity error      */
#define HA_ABORT_NA             0x0D    /* CP aborted - NOT on Bus      */
#define HA_ABORTED              0x0E    /* CP aborted - WAS on Bus      */
#define HA_RESET_NA             0x0F    /* CP was reset - NOT on Bus    */
#define HA_RESET                0x10    /* CP was reset - WAS on Bus    */
#define HA_ECC                  0x11    /* HBA Memory ECC Error         */
#define HA_PCI_PARITY           0x12    /* PCI Parity Error             */
#define HA_PCI_MASTER           0x13    /* PCI Master Abort             */
#define HA_PCI_TARGET           0x14    /* PCI Target Abort             */
#define HA_PCI_SIGNAL_TARGET    0x15    /* PCI Signalled Target Abort   */
#define HA_ABORT                0x20    /* Software Abort (too many retries) */

/*******************************************************************************
** ReadConfig Data Structure - This Structure Contains The EATA Configuration **
*******************************************************************************/

typedef struct {
        u_char  DevType;
        u_char  PageCode;
        u_char  Reserved0;
        u_char  ConfigLength;           /* Length In Bytes After This Field   */
        u_char  EATAsignature[4];       /* EATA Signature Bytes               */
        u_char  EATAversion;            /* EATA Version Number                */
        struct {
                u_char x_OverLapCmds:1; /* Overlapped Cmds Supported          */
                u_char x_TargetMode:1;  /* Target Mode Supported              */
                u_char x_TrunNotNec:1;  /* Truncate Command Not Supported     */
                u_char x_MoreSupported:1;/* More Command Supported            */
                u_char x_DMAsupported:1; /* DMA Mode Supported                */
                u_char x_DMAChannelValid:1;/* DMA Channel Field Is Valid.     */
                u_char x_ATAdevice:1;   /* This Is An ATA Device              */
                u_char x_HBAvalid:1;    /* HBA field Is Valid                 */
        }       flag0;
#define OverLapCmds     flag0.x_OverLapCmds
#define TargetMode      flag0.x_TargetMode
#define TrunNotNec      flag0.x_TrunNotNec
#define MoreSupported   flag0.x_MoreSupported
#define DMAsupported    flag0.x_DMAsupported
#define DMAChannelValid flag0.x_DMAChannelValid
#define ATAdevice       flag0.x_ATAdevice
#define HBAvalid        flag0.x_HBAvalid
        u_char  PadLength[2];           /* Pad Bytes For PIO Commands         */
        u_char  HBA[4];                 /* Host Adapter SCSI ID               */
        u_char  CPlength[4];            /* Command Packet Length              */
        u_char  SPlength[4];            /* Status Packet Length               */
        u_char  QueueSize[2];           /* Controller Queue Depth             */
        u_char  Reserved1[2];
        u_char  SG_Size[2];             /* Maximum Scatter Gather List Size   */
        u_char  IRQ_Number:4;           /* IRQ Number                         */
        u_char  IRQ_Trigger:1;          /* IRQ Trigger: 0 = Edge, 1 = Level   */
        u_char  Secondary:1;            /* Controller Not At Address 0x170    */
        u_char  DMA_Channel:2;          /* DMA Channel Index For ISA          */

        u_char  IRQ;                    /* IRQ address                        */
        struct {
                u_char x_Disable:1;     /* ISA I/O Address Disabled           */
                u_char x_ForceAddr:1;   /* PCI Forced To An EISA/ISA Addr     */
                u_char x_SG64K:1;       /* 64K of SG space                    */
                u_char x_SGUnaligned:1; /* Supports unaligned SG, otherwise 4 */
                u_char x_Reserved2:4;   /* Reserved Field                     */
        }       flag1;
#define Disable flag1.x_Disable
#define ForceAddr       flag1.x_ForceAddr
#define SG64K   flag1.x_SG64K
#define SgUnaligned     flag1.x_SgUnaligned
        u_char  MaxScsiID:5;            /* Maximun SCSI Target ID Supported   */
        u_char  MaxChannel:3;           /* Maximun Channel Number Supported   */
        u_char  MaxLUN;                 /* Maximum LUN Supported              */
        struct {
                u_char x_Reserved3:3;   /* Reserved Field                     */
                u_char x_AutoTermination:1;/* Support auto term (low byte)    */
                u_char x_PCIM1:1;       /* PCI M1 chipset                     */
                u_char x_RaidIDQuestionable:1;/* Raid ID may be questionable  */
                u_char x_PCIbus:1;      /* PCI Adapter Flag                   */
                u_char x_EISAbus:1;     /* EISA Adapter Flag                  */
        }       flag2;
#define AutoTermination flag2.x_AutoTermination
#define PCIM1   flag2.x_PCIM1
#define RaidIDQuestionable      flag2.x_RaidIDQuestionable
#define PCIbus  flag2.x_PCIbus
#define EISAbus flag2.x_EISAbus

        u_char  RaidNum;                /* Raid Host Adapter Number           */
} DptConfig_t;

/**************************************************************************
** DPT Host Adapter structure - One Structure For Each Host Adapter That **
**  Is Configured Into The System.  The Structure Supplies Configuration **
**  Information, Status Info, Queue Info And An Active CCB List Pointer. **
***************************************************************************/

typedef struct DptHa {
        struct hba_softc        ha_hba; /* must be first              */
        struct isadev           ha_isa;
        struct intrhand         ha_ih;
        u_short                 ha_IO_Pending;
        u_short                 ha_Total_IO_Pending;    /* Parent Only        */
        u_short                 ha_Wedgetime:15;
        u_short                 ha_Watchdog:1;
        u_short                 ha_Wedgecounter;
        u_short                 ha_Watchtime;

        u_short                 ha_Base;        /* base port for each board   */
        DptCcb_t *              ha_CCB;         /* CCBs in use                */
        DptCcb_t *              ha_Go_CCB;      /* send from start to go      */
        u_char                  ha_Channel:2;   /* The Channel *we* represent */
        u_char                  ha_ID:5;        /* The ID offset              */
        u_char                  ha_debug:1;     /* Debug Flag                 */
        u_char                  ha_inquired[MAX_TARGET_ID];     /* INQUIRY 1st*/
        u_char                  ha_unavailable[MAX_TARGET_ID];  /* Bad LUN    */

        DptConfig_t             ha_Config;
        Scsi_Sense_t **         ha_sense;
#define is_pci(sc)      ((sc)->ha_Config.PCIbus)
#define is_eisa(sc)     ((sc)->ha_Config.EISAbus)
#define is_isa(sc)      (!is_pci(sc) && !is_eisa(sc))
        /* Status Packet For This Adapter (not for the BUS)  */
        DptStat_t *             ha_Status_Packet;       /* Common for HBA */
        vm_addr_t               ha_Status_Packet_addr;  /* Byte swapped */
        struct DptHa *          ha_next;        /* Virtual HBA list     */
        struct DptHa *          ha_parent;      /* Virtual HBA parent   */
        struct DptHa *          ha_next_parent;/* Next parent           */
} DptHa_t;

DptHa_t *       DptHa;

/*
 *      Describe the Inquiry Data returned on Page 0 from the Adapter. The
 *      Page C1 Inquiry Data is described in the DptConfig_t structure above.
 */
typedef struct {
        u_char  deviceType;
        u_char  rm_dtq;
        u_char  otherData[6];
        u_char  vendor[8];      /* 'DPT     '   */
        u_char  modelNum[7];    /* 'PM????A'    */
        u_char  modelSuf[9];    /* '?? -RDLW '  */
        u_char  firmware[3];    /* v'07G'       */
        u_char  revision;       /*      .'0'    */
} DptInq_t;

/*
 *      Prototypes of the routines we have in this object.
 */
void            dptattach __P((struct device *parent, struct device *dev,
                    void *args));
STATIC int      dptccb __P((DptHa_t *sc, DptCcb_t *ccb, int flags));
STATIC void     dptqueue __P((DptHa_t *sc, DptCcb_t *ccb));
int             dptdump __P((struct hba_softc *hba, int target,
                    struct scsi_cdb *cdb, caddr_t buf, int len));
int             dptgo __P((struct device *self, int target, scintr_fn intr,
                    struct device *dev, struct buf *bp, int pad));
void            dpthbareset __P((struct hba_softc *hba, int resetunits));
int             dptioctl __P((dev_t dev, int cmd, caddr_t data, int flag,
                    struct proc *p));
int             dptopen __P((dev_t dev, int flags, int ifmt, struct proc *p));
int             dpticmd __P((struct hba_softc *hba, int target,
                    struct scsi_cdb *cdb, caddr_t buf, int len, int rw));
int             dptintr __P((DptHa_t *sc));
int             dptprobe __P((struct device *parent, struct cfdata *cf,
                    void *aux));
void            dptstart __P((struct device *self, struct sq *sq,
                    struct buf *bp, scdgo_fn dgo, struct device *dev));
void            dptrel __P((struct device *self));
void            dptwatch __P((DptHa_t *sc));

/* Bounce and CCB management routines */
STATIC void     dpt_cleanup __P((DptHa_t *sc, DptCcb_t *ccb));

STATIC void     dpt_init_sccb __P((DptHa_t *sc, DptCcb_t *ccb, int target,
                    scintr_fn intr, struct device *dev, struct buf *bp));

/* CCB allocation routines */
STATIC void     dpt_init_ccb __P((DptHa_t *sc));
STATIC DptCcb_t *dpt_get_ccb __P((DptHa_t *sc,int flags, struct scsi_cdb *cdb));
STATIC void     _dpt_free_ccb __P((int flags, DptCcb_t *ccb));
STATIC inline void dpt_free_ccb __P((DptHa_t *sc, int flags, DptCcb_t *ccb));

/* Bounce buffer allocation routines */
STATIC void     dpt_init_bounce __P((DptHa_t *sc));
STATIC DptBounce_t *dpt_get_bounce __P((DptHa_t *sc, int flags));
STATIC void     _dpt_free_bounce __P((int flags, DptBounce_t *bounce));
STATIC inline void dpt_free_bounce __P((DptHa_t *sc, int flags,
                                DptBounce_t *bounce));

/* Utility routines */
static inline void              dpt_prstring __P((u_char *s, int n));
static inline vm_offset_t       KVTOPHYS __P((caddr_t v));
static inline vm_addr_t         VMTOSWAP __P((vm_offset_t v));
static inline vm_offset_t       SWAPTOVM __P((vm_addr_t v));
static inline vm_len_t          LMTOSWAP __P((vm_size_t l));
static inline vm_size_t         SWAPTOLM __P((vm_len_t l));
static inline int               dpt_test_unit_ready __P((DptHa_t *sc,
                                        int target, int channel));
static inline DptInq_t *        dpt_hba_inquiry __P((DptHa_t *sc));
static inline vm_offset_t       dpt_bounce __P((DptHa_t *sc, DptCcb_t *ccb,
                                        caddr_t v, vm_size_t len, int rw));
static inline void              dpt_cmd __P((int io_addr, vm_offset_t up));
static inline int               dpt_wait __P((int io_addr,int bits,int state));
static inline void              dpt_add_to_do __P((DptHa_t *sc, int ID,
                                        int Channel));
static inline void              dpt_remove_to_do __P((void));
static inline void              dpt_error __P((DptHa_t *sc, DptCcb_t *ccb,
                                        char * msg, int errno, char **table,
                                        int size));
/*
 *      Conversion macros.
 *      The GNU compiler knows about inline, a far better way of dealing with
 *      macros as we can ensure strong type checking.
 *
 * pjd - Unfortunately inline functions tend to screw up the compiler's
 * register allocation mechanism. Its probably better to use macros than
 * inline functions.
 *
 * mgs - I made sure we didn't use any registers.
 */
#define ISA_ADDR        ((vm_offset_t)ISA_MAXADDR)

static inline vm_offset_t
KVTOPHYS(caddr_t v)
{
        return ((vm_offset_t)pmap_extract (kernel_pmap, (vm_addr_t)v));
}

static inline vm_addr_t
VMTOSWAP(vm_offset_t v)
{
        return ((vm_addr_t)ntohl (v));
}

static inline vm_offset_t
SWAPTOVM(vm_addr_t v)
{
        return ((vm_offset_t)htonl (v));
}

static inline vm_len_t
LMTOSWAP(vm_size_t l)
{
        return ((vm_len_t)ntohl (l));
}

static inline vm_size_t
SWAPTOLM(vm_len_t l)
{
        return ((vm_size_t)htonl (l));
}

/*
 *      Provided to give a consistent interface should we decide to do more
 *      cleanup later.
 */
static inline void
dpt_free_ccb(DptHa_t *sc, int flags, DptCcb_t *ccb)
{
        _dpt_free_ccb (flags, ccb);
}

static inline void
dpt_free_bounce(DptHa_t *sc, int flags, DptBounce_t *bounce)
{
        _dpt_free_bounce (flags, bounce);
}

/*
 *      List of all known EATA or DPT EISA ids. This is handled better by
 *      doing a generalized test like the PCI probe, but the BSDi OS chose
 *      an explicit list for their implementation. In any case, it forces us
 *      to document all of the EISA products we work with.
 */
static char *dpt_ids[] = {
        "DPT2402",      /* DPT PM2012A/9X               */
        "DPTA401",      /* DPT PM2012B/9X               */
        "DPTA402",      /* DPT PM2012B2/9X              */
        "DPTA410",      /* DPT PM2x22A/9X               */
        "DPTA411",      /* DPT Spectre                  */
        "DPTA412",      /* DPT PM2021A/9X               */
        "DPTA420",      /* DPT Smart Cache IV (PM2042)  */
        "DPTA501",      /* DPT PM2012B1/9X              */
        "DPTA502",      /* DPT PM2012Bx/9X              */
        "DPTA701",      /* DPT PM2011B1/9X              */
        "DPTBC01",      /* DPT PM3011/7X ESDI           */
        "NEC8200",      /* NEC EATA SCSI                */
        "ATT2408",      /* ATT EATA SCSI                */
        NULL
};

/*
 *      Here is the auto-probe structure used to nest our tests appropriately
 *      during the startup phase of the operating system.
 */
struct cfdriver dptcd =
        { 0, "dpt", dptprobe, dptattach, DV_DULL, sizeof(DptHa_t), dpt_ids, 0 };
/*
 *      Here is the driver entry points supported by the SCSI handler.
 */
static struct hbadriver dpthbadriver =
        { dpticmd, dptdump, dptstart, dptgo, dptrel, dpthbareset };

/*
 * devsw for dpt hba driver
 *
 * only ioctl is used. the sd driver provides all other access.
 */
struct devsw dptsw = {
        &dptcd,
        dptopen, nullclose, noread, nowrite, dptioctl, noselect, nommap,
        nostrat, nodump, nopsize, 0,
        nostop
};

/*
 *      dpt_cmd (io_port, up);
 *      Perform a CP_DMA_CMD. This routine might best be optimized into
 *      assembler code, but I chose to support portability instead.
 */
static inline void
dpt_cmd (io_addr, up)
        int             io_addr;
        vm_offset_t     up;
{       /*
         *      We could be called by an Interrupt Service, so lets not
         * sleep or wait unduely long (50ms). We must escape quickly from
         * any tests here so do not do any spinloops or sleep delays.
         *
         *      A Timeout would be very bad, so lets not ...
#define dpt_cmd_TIMEOUT 50000L
         */
#       if (defined(dpt_cmd_TIMEOUT))
                if (inb(io_addr + HA_AUX_STATUS) & HA_AUX_BUSY) {
                        struct timeval  stamp = time;
#                       if (dpt_cmd_TIMEOUT > 1000000L)
                                stamp.tv_sec += dpt_cmd_TIMEOUT / 1000000L;
#                       endif
                        if ((stamp.tv_usec += (dpt_cmd_TIMEOUT%1000000L))
                            >= 1000000L) {
                                ++stamp.tv_sec;
                                stamp.tv_usec -= 1000000L;
                        }
                        while ((inb(io_addr + HA_AUX_STATUS) & HA_AUX_BUSY)
                         && ((time.tv_sec == stamp.tv_sec)
                           ? (time.tv_usec < stamp.tv_usec)
                           : (time.tv_sec < stamp.tv_sec)));
                }
#       else
                while (inb(io_addr + HA_AUX_STATUS) & HA_AUX_BUSY);
#       endif
#if (BYTE_ORDER == LITTLE_ENDIAN)
                outb (io_addr + HA_DMA_BASE + 0, ((u_char)(u_long)up));
                outb (io_addr + HA_DMA_BASE + 1, ((u_short)(u_long)up) >> 8);
                outb (io_addr + HA_DMA_BASE + 2, ((u_long)up) >> 16);
                outb (io_addr + HA_DMA_BASE + 3, ((u_long)up) >> 24);
#elif (BYTE_ORDER == BIG_ENDIAN)
                outb (io_addr + HA_DMA_BASE + 0, ((u_long)up) >> 24);
                outb (io_addr + HA_DMA_BASE + 1, ((u_long)up) >> 16);
                outb (io_addr + HA_DMA_BASE + 2, ((u_short)(u_long)up) >> 8);
                outb (io_addr + HA_DMA_BASE + 3, ((u_char)(u_long)up));
#elif (BYTE_ORDER == PDP_ENDIAN)
                outb (io_addr + HA_DMA_BASE + 0, ((u_long)up) >> 16);
                outb (io_addr + HA_DMA_BASE + 1, ((u_long)up) >> 24);
                outb (io_addr + HA_DMA_BASE + 2, ((u_char)(u_long)up));
                outb (io_addr + HA_DMA_BASE + 3, ((u_short)(u_long)up) >> 8);
#else
                UNKNOWN BYTE_ORDER BYTE ORDER
#endif
        outb (io_addr + HA_COMMAND, CP_DMA_CMD);
}

/*
 *      Wait for the DPT card to accept the next command when manually polling.
 *      This routine need not be efficient, as it is only to be used at
 *      probe time.
 */
static inline int
dpt_wait(io_addr, bits, state)
        u_short io_addr;
        u_char  bits;
        u_char  state;
{       u_short i;
        u_char  c;

        for (i = 400;; ) {      /* wait 20ms for not busy */
                c = inb (io_addr + HA_STATUS) & bits;
                if (c == state)
                        return (0);
                DELAY (50);     /* 50 us */
                if (--i <= 0)
                        return (-1);
        }
}

#if 1 || (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS & DPT_DEBUG_VERBOSE))
static void
Print_EataCp (ccb)
        DptCcb_t *      ccb;
{
        printf ("EataCp:%s%s%s%s%s%s%s%s%s%s ReqLen=%d\n (%d,%d,%d)%s%s%s %02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n Len=%d vp=%x dma=%x stat=%x req=%x\n Bounce=%x Device=%x\n",
                ccb->EataCp.cp_DataIn ? " In" : "",
                ccb->EataCp.cp_DataOut ? " Out" : "",
                ccb->EataCp.cp_Interpret ? " Interpret" : "",
                ccb->EataCp.cp_Quick ? " Quick" : "",
                ccb->EataCp.cp_Scatter_Gather ? " ScatterGather" : "",
                ccb->EataCp.cp_Auto_Req_Sen ? " AutoRequestSense" : "",
                ccb->EataCp.cp_HBA_Init ? " Init" : "",
                ccb->EataCp.cp_SCSI_Reset ? " Reset" : "",
                ccb->EataCp.cp_FWNest ? " FWNest" : "",
                ccb->EataCp.cp_PhsUnit ? " PhysUnit" : "",
                ccb->EataCp.cp_Req_Len,
                ccb->EataCp.cp_Channel, ccb->EataCp.cp_id,
                ccb->EataCp.cp_msg_lun,
                ccb->EataCp.cp_msg_luntar ? " Target" : "",
                ccb->EataCp.cp_msg_disco_rico ? " Disco/Reco" : "",
                ccb->EataCp.cp_msg_identify ? " Identify" : "",
                ccb->EataCp.cp_cdb[0], ccb->EataCp.cp_cdb[1],
                ccb->EataCp.cp_cdb[2], ccb->EataCp.cp_cdb[3],
                ccb->EataCp.cp_cdb[4], ccb->EataCp.cp_cdb[5],
                ccb->EataCp.cp_cdb[6], ccb->EataCp.cp_cdb[7],
                ccb->EataCp.cp_cdb[8], ccb->EataCp.cp_cdb[9],
                ccb->EataCp.cp_cdb[10], ccb->EataCp.cp_cdb[11],
                SWAPTOLM(ccb->EataCp.cp_datalen), ccb->EataCp.cp_vp,
                SWAPTOVM(ccb->EataCp.cp_dataDMA),
                SWAPTOVM(ccb->EataCp.cp_statDMA),
                SWAPTOVM(ccb->EataCp.cp_reqDMA),
                ccb->ccb_Bounce, ccb->ccb_Device);
        DELAY(100000);
}
#endif
/*
 *      Read EATA Config and test for supported HBA - returns pointer to
 *      Configuration structure if supported
 */
STATIC DptConfig_t *
dpt_EATA_ReadConfig (io_addr, drq)
        u_short         io_addr;
        u_short         drq;
{
        int             i, j;
        u_char          stat;
        u_short *       ip;
        DptCcb_t *      ccb;
        DptStat_t *     status;
        DptConfig_t *   config;
        vm_offset_t     addr;

        while ((((stat = inb(io_addr + HA_STATUS))
         != (HA_ST_READY|HA_ST_SEEK_COMPLETE))
          && (stat != (HA_ST_READY|HA_ST_SEEK_COMPLETE|HA_ST_ERROR))
          /* This results from the `wd' probe at our addresses */
          && (stat != (HA_ST_READY|HA_ST_SEEK_COMPLETE|HA_ST_ERROR|HA_ST_DRQ)))
         || (dpt_wait(io_addr, HA_ST_BUSY, 0))) {
                /*
                 *      RAID Drives still Spinning up? (This should only occur
                 *      if the DPT controller is in a NON PC (PCI?) platform).
                 */
                if ((inb(io_addr + HA_ERROR) != 'D')
                 || (inb(io_addr + HA_ERROR + 1) != 'P')
                 || (inb(io_addr + HA_ERROR + 2) != 'T'))
                        return ((DptConfig_t *)NULL);
        }
        if ((config = (DptConfig_t *)malloc (sizeof(DptConfig_t), M_TEMP,
            M_WAITOK)) == (DptConfig_t *)NULL)
                return ((DptConfig_t *)NULL);
        /*
         *      If any of our buffers allocate above the 16M boundary for isa
         * looking cards, then resort to a Read EATA Config command instead.
         */
        if (io_addr >= 0x400) do {
                addr = KVTOPHYS ((caddr_t)config);
                if ((ccb = (DptCcb_t *)malloc (
                    sizeof(DptCcb_t), M_TEMP, M_WAITOK)) == (DptCcb_t *)NULL)
                        break;
                if ((status = (DptStat_t *)malloc (sizeof(DptStat_t), M_TEMP,
                    M_WAITOK)) == (DptStat_t *)NULL) {
                        free ((caddr_t)ccb, M_TEMP);
                        break;
                }
                /*
                 * Do a Read Vital Data Page SCSI command instead of using the
                 * CP_READ_CFG_PIO/CP_READ_CFG_DMA comand!
                 */
                (void)bzero ((char *)ccb, sizeof(DptCcb_t));
                ccb->EataCp.cp_dataDMA = VMTOSWAP (addr);
                addr = KVTOPHYS ((caddr_t)status);
                ccb->EataCp.cp_statDMA = VMTOSWAP (addr);
                ccb->EataCp.cp_reqDMA = VMTOSWAP (KVTOPHYS ((caddr_t)
                    &(ccb->ccb_sense)));
                ccb->EataCp.cp_Interpret = 1;
                ccb->EataCp.cp_DataIn = 1;
                ccb->EataCp.cp_Auto_Req_Sen = 1;/* Clear sense data on error */
                ccb->EataCp.cp_id = 7;/* Doesn't matter since interpret mode */
                ccb->EataCp.cp_scsi_cmd = CMD_INQUIRY;
                ccb->EataCp.cp_extent = 1;
                ccb->EataCp.cp_page = 0xC1;
                ccb->EataCp.cp_len = sizeof(DptConfig_t);
                ccb->EataCp.cp_datalen=LMTOSWAP((vm_size_t)sizeof(DptConfig_t));
                ccb->EataCp.cp_vp = (caddr_t)ccb;
                ccb->EataCp.cp_msg_identify = 1;

                (void)bzero ((char *)config, sizeof(DptConfig_t));
                (void)bzero ((char *)status, sizeof(DptStat_t));
                DptStat_Reset_BUSY (status);
#               if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS & DPT_DEBUG_VERBOSE))
                        Print_EataCp (ccb);
#               endif
                dpt_cmd (io_addr, KVTOPHYS ((caddr_t)ccb));
                /*
                 * Wait for the board to report a finished instruction.
                 */
                i = 20000;      /* One second to respond */
                while (DptStat_BUSY (status)) {
                        DELAY(50);
                        if (--i <= 0)
                                break;
                }

                i = (!(status->sp_HBA_stat)) && (!(status->sp_SCSI_stat))
                 && (!(status->sp_inv_residue));

                /* Clear interrupt by reading status register */

                if (((inb(io_addr+HA_STATUS) & HA_ST_ERROR) == 0) && (i)
                 && (*((u_long *)(config->EATAsignature)) == EATA_SIG)
                 && config->HBAvalid && config->DMAsupported) {
                        free ((caddr_t)ccb, M_TEMP);
                        free ((caddr_t)status, M_TEMP);

                        /* 4G.Z did this to me */
                        if ((i = config->ConfigLength
                            + (int)(&(((DptConfig_t *)0L)->ConfigLength))
                            + sizeof(config->ConfigLength))
                            < (sizeof(DptConfig_t))) {
                                (void)bzero ((char *)config + i,
                                    sizeof(DptConfig_t) - i);
                                /* Defaults for older Firmware */
                                        /* 2001? */
/* ip looks uninitialized at this point! - pjd */
                                ip = (u_short *)((char *)config
                                    + config->ConfigLength);
/* Yes, ip was uninitialized - mgs */
                                if (ip <= (u_short *)config->HBA[MAX_CHANNEL-1])
                                        config->HBA[MAX_CHANNEL - 1] = 7;
                                        /* 2012 */
                                if ((ip <= (u_short *)(&(config->flag2)))
                                 && ((io_addr & 0xFFF) == 0xC88))
                                        config->EISAbus = 1;
                        }
                        return (config);
                }
                outb (io_addr + HA_COMMAND, CP_EATA_RESET);
                DELAY (750000L);        /* Wait 1/2 second and try following */
                free ((caddr_t)ccb, M_TEMP);
                free ((caddr_t)status, M_TEMP);
        } while (0);    /* break marker */
        /*
         * I *could* do a Read Config INQUIRY using a polled instruction, but
         * have been asked to not call these instructions if at all possible.
         * So, we do a CP_READ_CFG_PIO for the older (2011 + 2012) boards,
         * or for ISA boards when we mistakenly allocated buffers above the
         * 16MB mark.
         */
        if (dpt_wait(io_addr, HA_ST_BUSY, 0)) {
                free ((caddr_t)config, M_TEMP);
                return ((DptConfig_t *)NULL);
        }
        outb (io_addr + HA_COMMAND, CP_READ_CFG_PIO);
        (void)bzero ((char *)(ip = (u_short *)config), sizeof(DptConfig_t));
        i = ((int)(&(((DptConfig_t *)0L)->ConfigLength))
            + sizeof(config->ConfigLength)) / sizeof(u_short);
/* ? - mgs */
        if (dpt_wait(io_addr, 0xFF, HA_ST_DATA_RDY)) {
                free ((caddr_t)config, M_TEMP);
                return ((DptConfig_t *)NULL);
        }
        do {
                *(ip++) = inw (io_addr + HA_DATA);
        } while (--i > 0);

        if ((i = config->ConfigLength) > (sizeof(DptConfig_t)
            - (int)(&(((DptConfig_t *)0L)->ConfigLength))
            - sizeof(config->ConfigLength)))
                i = sizeof(DptConfig_t)
                  - (int)(&(((DptConfig_t *)0L)->ConfigLength))
                  - sizeof(config->ConfigLength);
        j = i + (int)(&(((DptConfig_t *)0L)->ConfigLength))
            + sizeof(config->ConfigLength);
        i /= sizeof(u_short);
        do {
                *(ip++) = inw (io_addr + HA_DATA);
        } while (--i > 0);
        /*
         *      Flush until we have read 512 bytes.
         */
        i += (512 - j) / sizeof(u_short);
        do {
                (void)inw (io_addr + HA_DATA);
        } while (--i > 0);
        /* Defaults for older Firmware */
        if (ip <= (u_short *)config->HBA[MAX_CHANNEL - 1])
                config->HBA[MAX_CHANNEL - 1] = 7;       /* 2001? */
        if ((ip <= (u_short *)(&(config->flag2)))
         && ((io_addr & 0xFFF) == 0xC88))
                config->EISAbus = 1;                    /* 2012 */
        if ((config->OverLapCmds == 0)
         && (config->ConfigLength == 0x1C)
         && (config->QueueSize[1] == (1 & 0xFF))
         && (config->QueueSize[0] == (1 >> 8))
         && (config->CPlength[3] == 0x20)) {    /* 2011 & 2012 3G.0 */
                /*
                 *      There are some versions of Firmware that report
                 *      no overlap comands when you do a PIO retrieval of
                 *      the parameters. Lets Lie and say they can handle it.
                 */
                /* config->OverLapCmds = 1; */
                config->QueueSize[1] = NCCBS & 0xFF;
                config->QueueSize[0] = NCCBS >> 8;
                config->SG_Size[1] = (SG_SIZE & 0xFF);
                config->SG_Size[0] = (SG_SIZE >> 8);
                config->CPlength[3] = 0x2C;
        }
        if (((inb(io_addr + HA_STATUS) & HA_ST_ERROR) == 0)
         && (*((u_long *)(config->EATAsignature)) == EATA_SIG)
         && config->HBAvalid && config->DMAsupported)
                return (config);

        free ((caddr_t)config, M_TEMP);
        return ((DptConfig_t *)NULL);
}

/*
 *      Return true if the Adapter is a DPT SCSI HBA.
 */
STATIC int
dpt_pci_match(pci)
        pci_devaddr_t * pci;
{
        return ((pci_inw (pci, PCI_VENDOR_ID) == 0x1044)
         && (pci_inw (pci, PCI_DEVICE_ID) == 0xA400)
         && (pci_inb (pci, PCI_CLASSCODE_L) == 0x00)
         && (pci_inw (pci, PCI_CLASSCODE_H) == 0x0100));
}

/*
 *      The following sections of code help virtualize the 16 Targets or up to
 *      three channels of SCSI buses per an adapter. It is a dynamically
 *      allocated list of virtual adapters to create on subsequent non-forced
 *      device probes. First in First out placement and removal.
 */
typedef struct {
        u_char          todo_Channel:2;/* Actual channel (0 based)         */
        u_char          todo_ID:5;      /* 4 bits are used, HBA reserves 5 */
        DptHa_t *       todo_parent;    /* *real* Host Bus Adapter         */
} Dpt_ToDo_t;

STATIC Dpt_ToDo_t *     dpt_todo;
STATIC int              dpt_todo_len;

static inline void
dpt_add_to_do (sc, ID, Channel)
        DptHa_t *       sc;
        int             ID, Channel;
{       Dpt_ToDo_t *    p;

        if (!(p=malloc(sizeof(Dpt_ToDo_t)*(++dpt_todo_len), M_TEMP, M_WAITOK)))
                return;
        if (dpt_todo) {
                if (dpt_todo_len > 1)
                        (void)bcopy ((caddr_t)dpt_todo, (caddr_t)p,
                            sizeof(Dpt_ToDo_t) * (dpt_todo_len - 1));
                free (dpt_todo, M_TEMP);
        }
        dpt_todo = p;
        p += dpt_todo_len - 1;
        p->todo_parent = sc;
        p->todo_Channel = Channel;
        p->todo_ID = ID;
}

static inline void
dpt_remove_to_do ()
{
        if (!dpt_todo)
                return;
        if (--dpt_todo_len <= 0) {
                dpt_todo_len = 0;
                free (dpt_todo, M_TEMP);
                dpt_todo = (Dpt_ToDo_t *)NULL;
                return;
        }
        (void)bcopy ((caddr_t)(dpt_todo + 1), (caddr_t)dpt_todo,
            sizeof(Dpt_ToDo_t) * dpt_todo_len);
}

/*
 * Probe for DPT controller.  If we find it we will use it. We may return
 * virtual adapters and allow non-forced probing in Software RAID order.
 */
int
dptprobe(parent, cf, aux)
        struct device *         parent;
        struct cfdata *         cf;
        void *                  aux;
{
        struct isa_attach_args *ia = (struct isa_attach_args *) aux;
        DptConfig_t *           config;
        u_short *               ip;
        int                     i;
        pci_devaddr_t *         pci;
        u_short                 board_drq;

        extern u_short          eisa_slot_map;
        extern int              dpt_todo_len;
        extern Dpt_ToDo_t *     dpt_todo;

        static DptConfig_t      Config;
        static u_short          dpt_eisa_slot_map;
        static int              primary_done;
        static u_short          isa_regs[] = {
                0x1F0, 0x170, 0x330, 0x230
        };

        if (!ia)
                return (0);
        if (ia->ia_iobase == 0) {
        /*
         * Base I/O address not specified in config file, search all
         * possible controller locations.
         */

                /*
                 * Check todo list for virtual controllers
                 * return any found
                 */
                if ((dpt_todo_len > 0) && dpt_todo) {
                        ia->ia_aux = (void *)NULL;
                        ia->ia_iosize = 0;
                        ia->ia_irq = 0;
                        ia->ia_drq = DRQNONE;
                        ia->ia_maddr = (caddr_t)NULL;
                        ia->ia_msize = 0;
                        return (1);
                }

                /*
                 *      Lets probe all possible controllers. First do Primary,
                 * PCI, EISA and then ISA cards. This is done in this order to
                 * allow us to support software RAID (later).
                 */
                dpt_eisa_slot_map |= eisa_slot_map;
                config = (DptConfig_t *)NULL;

                /*
                 * Check for "Primary" controller which must be EISA by
                 * definition.
                 */
                if (primary_done == 0) {
                        ++primary_done;
                        i = 1;
                        do {ia->ia_iobase = EISA_PROD_ID_BASE(i) + 8;
                            /*
                             *  Remove any EISA cards that have the
                             *  incorrect PAL.
                             */
                            if (eisa_match(cf, ia) == 0) {
                                dpt_eisa_slot_map |= 1 << i;
                                continue;
                            }
                            /*
                             * If we've already found one EISA card and its
                             * a "Primary", don't read the config from the
                             * rest.
                             * Do all EISA now for PAL test ???
                             */
                            if (config)
                                continue;

                            if (config = dpt_EATA_ReadConfig (ia->ia_iobase,
                            ia->ia_drq)) {
                                /* Save Primary Address */
                                primary_done = ia->ia_iobase;
                                /*
                                 * if EISA and not disabled, clear address
                                 * of secondary so we don't find it later.
                                 */
                                if (config->EISAbus && !(config->Disable))
                                    isa_regs[config->Secondary] = 0;

                                /*
                                 * Its NOT EISA or we found a secondary,
                                 * skip it.
                                 */
                                if (!config->EISAbus || config->Secondary) {
                                    free (config, M_TEMP);
                                    config = NULL;
                                }
                            }
                        } while (++i < EISA_NUM_PHYS_SLOT);

                        /*
                         * If we didn't already find a "Primary", check
                         * its well known port address, the first one
                         * possible. If we find it, clear the first
                         * possible address slot so we don't find it
                         * again later.
                         */
                        if (isa_regs[0] && !config
                         && (config = dpt_EATA_ReadConfig (isa_regs[0],
                         ia->ia_drq))) {
                                /* Save Primary Address */
                                primary_done = isa_regs[0];
                                isa_regs[0] = 0;
                        }
                        /* 1 or Primary Address */
                        ia->ia_iobase = primary_done;
                }
                if (!config) {
                    switch(ia->ia_bustype) {
                    case BUS_PCI: /* PCI */
                        while (pci = pci_scan(dpt_pci_match)) {
                        ia->ia_iobase = pci_inw (pci, PCI_PORT_BA) & 0xFFFFFFFE;
                        /*
                         *      The following may be unnecessary since
                         * we check for PCI Forced Address in the
                         * configuration. I do the following to reduce
                         * calls to read config that may bear false
                         * information, or superfluous calls to read
                         * the configuration when obviously forced.
                         */
                        /*
                         * Check to make sure it is not a forced
                         * address EISA that could be probed later.
                         */
                        switch (inb (i = ia->ia_iobase)) {
                            case 0x12:  /* DPT */
                                if (inb (++i) == 0x14)
                                        continue;
                                break;
                            case 0x38:  /* NEC */
                                if ((inb (++i) == 0xA3) && (inb (++i) == 0x82))
                                        continue;
                                break;
                            case 0x06:
                                if ((inb (++i) == 0x94) && (inb (++i) == 0x24))
                                        continue;
                                break;
                            }
                        /*
                         * Check to make sure it is not a forced
                         * address ISA already in use, or overlaps
                         * another device.
                         */
                        if (isa_portcheck (ia->ia_iobase+=0x10, DPT_NPORT) == 0)
                                continue;
                        /*
                         * Check to make sure it is not a forced
                         * address ISA that could be probed later.
                         */
                        for (ip = isa_regs; ip < &isa_regs[sizeof(isa_regs)
                            / sizeof(isa_regs[0])]; ++ip)
                                if (*ip == ia->ia_iobase)
                                        break;

                        if (ip < &isa_regs[sizeof(isa_regs)
                            / sizeof(isa_regs[0])])
                                continue;
                        /*
                         *      Now, check if it is a *valid* DPT
                         * controller. It is too late to add this back
                         * to the PCI slots :-(
                         */
                        if (config = dpt_EATA_ReadConfig (ia->ia_iobase,
                        ia->ia_drq)) {
                                /*
                                 *      Well, lets now check if the
                                 * card tells me it is forced to an
                                 * ISA/EISA address.
                                 */
                                if (config->PCIbus && config->ForceAddr) {
                                        free (config, M_TEMP);
                                        config = NULL;
                                        continue;
                                }
                                break;
                        }
                    }
                    break;

                    case BUS_EISA: /* EISA */
                    for (i = 1; i < EISA_NUM_PHYS_SLOT;
                      dpt_eisa_slot_map |= 1 << (i++)) {
                        if ((dpt_eisa_slot_map & (1 << i))
                         || (dpt_eisa_slot_map |= 1 << i,
                          ((config = dpt_EATA_ReadConfig (ia->ia_iobase
                          = EISA_PROD_ID_BASE(i) + 8, ia->ia_drq)) == NULL)))
                                continue;
                        /* Primary has already been dealt with */
                        if (config->EISAbus && !(config->Disable) &&
                         config->Secondary)
/* looks like this index should be an i, instead of a 1 - pjd */
/* Clear the 0170 address from being available, 1 is appropriate - mgs */
                                isa_regs[1] = 0;
                        break;
                    }
                    break;

                    default: /* ISA */
                    for (ip = isa_regs;
                    ip < &isa_regs[sizeof(isa_regs)/sizeof(isa_regs[0])];
                    ++ip) {
                        /*
                         * Check to make sure it is not overlapping another.
                         */
                        if (((i = *ip) == 0) ||
                         (*ip = 0, (isa_portcheck (i, DPT_NPORT) == 0)) ||
                         (!(config=dpt_EATA_ReadConfig (ia->ia_iobase = i,
                         ia->ia_drq))))
                                continue;
                        break;
                    }
                    break;
                }
                if (!config)
                        return (0);
            }
        } else {
            /*
             * a non-zero io_base pretty much means ISA bus. - pjd
             */

            /*
             * It is decided that any forced addressing should not
             * restrict our configuration to handle any possible future
             * problems. This includes *even* re-using an address that may
             * have already been eliminated by an auto probe ...
             */
            if ((isa_portcheck (ia->ia_iobase, DPT_NPORT) == 0) ||
             (!(config = dpt_EATA_ReadConfig (ia->ia_iobase, ia->ia_drq))))
                return (0);
             /*
              * Make sure that a subsequent auto probe will not
              * re-use an EISA address.
              */
             if ((ia->ia_iobase & 0xFFF) == 0xC88)
                 dpt_eisa_slot_map |= 1 << ((ia->ia_iobase >> 12) & 0xF);
             /*
              * Make sure that a subsequent auto probe will not
              * re-use the ISA addresses.
              */
              if (config->EISAbus && !(config->Disable))
                  isa_regs[config->Secondary] = 0;
              for (ip = isa_regs;
                ip < &isa_regs[sizeof(isa_regs)/sizeof(isa_regs[0])];
                ++ip) {
                  if (*ip == ia->ia_iobase)
                      *ip = 0;
               }
        }
#if (defined(BETA_DPT))
        {       static u_char   not_first_time_through;

                if (!not_first_time_through) {
                        ++not_first_time_through;
                        printf ("Copyright 1996 Distributed Processing Technology. Evaluation copy not for sale!\n");
                }
        }
#endif
        /*
         *      Note, it is not safe to use ia_aux to point to dynamically
         *      allocated memory as there are no guarantees it will be used
         *      by the attach function.
         */

        Config = *config;

        free (config, M_TEMP);
        /*
         *      Fill in the isa_attach arguments
         */
        ia->ia_aux = &Config;
        if (((i = Config.IRQ_Number) || (i = Config.IRQ)) && (i == 2))
                i = 9;

        /* verify correctness of irq here */
        if (i) {
            if (ia->ia_irq == IRQUNK)
                ia->ia_irq = irq_indextomask(i);
            else if (ia->ia_irq != irq_indextomask(i)) {
                printf("dpt: probe failed - hardcoded IRQ doesn't match config data (irq%d)\n", i);
                return(0);
            }
        }
        /*
         *      Setting the size to something other than DPT_NPORT will make no
         * difference since the size is only translated to an isa address range.
         */
        ia->ia_iosize = DPT_NPORT;
        if (Config.EISAbus) {
                eisa_slotalloc ((ia->ia_iobase >> 12) & 0xF);
                /*
                 *      The EISA card is also visible at this ISA address.
                 */
                if (!(config->Disable))
                        isa_portalloc (config->Secondary ? 0x170 : 0x1F0,
                            DPT_NPORT);
        }
        /*
         *      PCI does this as part of it's probe, but lets just do this here
         * to cover all cases.
         */
        if (Config.IRQ_Trigger) {
                outb (IO_ELCR1, inb (IO_ELCR1) | ia->ia_irq);
                outb (IO_ELCR2, inb (IO_ELCR2) | ia->ia_irq >> 8);
        }
        if (Config.DMAChannelValid) {
            /* verify correctness of drq here */
            board_drq = "\0\7\6\5"[Config.DMA_Channel];
            if ((ia->ia_drq == DRQNONE) || (ia->ia_drq == (u_short)-1)) {
                printf("dpt: warning - no hardcoded DRQ, yet config data specifies one (drq%d)\n", board_drq);
            } else if (ia->ia_drq != board_drq) {
                printf("dpt: probe failed - hardcoded DRQ (drq%d) doesn't match config data (drq%d)\n", ia->ia_drq, board_drq);
                return(0);
            }
        }
        ia->ia_maddr = (caddr_t)NULL;
        ia->ia_msize = 0;

        return (1);
}

/*
 * Make sure that we don't get stuck. This way of watching the controller
 * is superior to a timeout generated for each transaction only because it
 * reduces the need for timeout slots in the kernel.
 */
#define DPT_WEDGETIME   (30*hz)

void
dptwatch (sc)
        DptHa_t *       sc;
{       DptCcb_t *      ccb;
        DptCcb_t *      go;
        int             i, s, fasttoolong, slowtoolong;
        struct sq *     sq;

        if (!sc)
                return;
        sc->ha_Watchdog = 0;
        if (sc->ha_IO_Pending == 0)
                return;

        /* did we miss an interrupt? */
        s = splbio();   /* avoid races */
        (void)dptintr (sc);
        splx (s);
        if (sc->ha_IO_Pending == 0)
                return;

        if (!sc->ha_Watchdog) {
                sc->ha_Watchdog = 1;
                timeout((void (*)(void *))dptwatch, sc, sc->ha_Watchtime);
        }
        if (--(sc->ha_Wedgecounter) != 0)
                return;
        sc->ha_Wedgecounter = sc->ha_Wedgetime;

        /*
         * Scan ccb list, look for stale entries.
         * If a read or write entry hasn't been serviced in a reasonable
         * amount of time, complain (should abort the transaction?).
         */
        fasttoolong = slowtoolong = 0;
        for (ccb = sc->ha_CCB, go = sc->ha_Go_CCB; ccb; ) {
                if (ccb->ccb_Device && (ccb->ccb_stamp < time.tv_sec)) {
                        s = splbio();
                        if (ccb->ccb_Device && (ccb->ccb_stamp < time.tv_sec))
                        switch (ccb->EataCp.cp_scsi_cmd) {
                        case CMD_READ10:
                        case CMD_READ:
                        case CMD_WRITE10:
                        case CMD_WRITE:
                                fasttoolong = 1;
                                printf ("%s: operation exceeds %d seconds\n",
                                    sc->ha_hba.hba_dev.dv_xname,
                                    time.tv_sec-ccb->ccb_stamp+DPT_WEDGETIME);
                                break;
                        default:
                                slowtoolong = 1;
                        }
                        splx(s);
                }
/* service ccb chain first, then Go chain  - pjd */
                if (!(ccb = ccb->ccb_next)) {
                        ccb = go;
                        go = (DptCcb_t *)NULL;
                }
        }
        if (!fasttoolong || slowtoolong)
                return;

        /*
         * to get here, we must have found a READ or WRITE command which
         * took too long - pjd
         */
        printf ("%s: attempting bus reset", sc->ha_hba.hba_dev.dv_xname);
        if (sc->ha_IO_Pending <= sc->ha_parent->ha_Total_IO_Pending)
                sc->ha_parent->ha_Total_IO_Pending -= sc->ha_IO_Pending;
        else
                sc->ha_parent->ha_Total_IO_Pending = 0;
        sc->ha_IO_Pending = 0;
        wakeup ((caddr_t)&(sc->ha_parent->ha_Total_IO_Pending));
        dpthbareset (&(sc->ha_hba), 1);
        DELAY (3000000);
        printf ("\n");

        sc->ha_Wedgecounter = sc->ha_Wedgetime;
}

/*
 *      Check if a non-HBA device is available at the specified target and
 * channel. This test is used to determine if a deeper scan (to virtualize
 * channels and luns > MAX_LUN, or to even do a scan for all LUNs) is necessary
 * and thus is hard coded for LUN 0 only. I can't use the dpticmd since we
 * need to override the channel. Record valid LUNs here (calling an Inquiry
 * command) so as not to make the mistake of sending commands to invalid LUNS.
 */
static inline int
dpt_test_unit_ready (sc, target, channel)
        DptHa_t *       sc;
        int             target, channel;
{       DptCcb_t *      ccb;
        struct scsi_cdb cdb;

        /* Ohh nooo, not the HBA! */
        if (target == sc->ha_Config.HBA[MAX_CHANNEL - 1 - (channel & 0x3)])
                return (0);

        /* If LUN 0 is not there, why even waste time on the others */
        (void)bzero (&cdb, sizeof(cdb));
        cdb.cdb_bytes[0] = CMD_TEST_UNIT_READY;
        ccb = dpt_get_ccb (sc, 0, &cdb);
        ccb->EataCp.cp_id = target;
        ccb->EataCp.cp_Channel = channel & 0x3;
        ccb->EataCp.cp_DataIn = 0;
        ccb->EataCp.cp_DataOut = 0;
        ccb->EataCp.cp_Interpret = 0;   /* Don't talk to the HBA please */
        ccb->EataCp.cp_Scatter_Gather = 0;
        ccb->EataCp.cp_Auto_Req_Sen = 1;
        ccb->EataCp.cp_msg_disco_rico = 0;
        ccb->EataCp.cp_datalen = 0;
        ccb->EataCp.cp_dataDMA = (vm_addr_t)0;
        ccb->EataCp.cp_statDMA = sc->ha_Status_Packet_addr;
        ccb->ccb_intr = (scintr_fn)NULL;
        ccb->ccb_intrdev = (struct device *)NULL;
        ccb->ccb_sense.sn_vcc = 0;      /* To check if sense data missing */
        switch (dptccb (sc, ccb, 0)) {
            case STS_CHECKCOND: /* 2011 and 2012 return real responses  */
#if (defined(DPT_DEBUG_CHECK))  /* PM3224W w/SX4000 fixup */
                        /*
                         * If a SX4000/ (narrow multichannel board) is added to
                         * a PM3224W (wide base) then we get a Check Condition
                         * response from the Targets above 8 on the narrow
                         * channels. This is for firmware after 3.C3 to 7.G0
                         * and the fixup is strictly for beauty.
                         */
                        if (SENSE_ISXSENSE(&(ccb->ccb_sense))
                         && XSENSE_ISSTD (&(ccb->ccb_sense))
                         && (XSENSE_KEY (&(ccb->ccb_sense)) == SKEY_ILLEGAL))
                                return (0);
                        /* General HBA failure to retrieve Sense */
                        if (ccb->ccb_sense.sn_vcc == 0)
                                return ((channel & 0x80) ? 0 :
                                        dpt_test_unit_ready (sc, target,
                                            channel | 0x80));
#endif
            case STS_GOOD:      /* DPT Adapters Cache Test Unit Ready   */
                return (1);

            default:   /* default added - pjd */
                return(0);
            }
        return (0);
}

/*
 *      Do an inquiry on the HBA. We did this *manually* during probe time
 *      to pick up page C1, but since we have the connections on now, lets
 *      use the tools here to pick up the page 0 values.
 */
static inline DptInq_t *
dpt_hba_inquiry (sc)
        DptHa_t *       sc;
{
        struct scsi_cdb cdb;
        DptInq_t *      inquire;

        if ((inquire = (DptInq_t *)malloc (sizeof(DptInq_t), M_TEMP, M_WAITOK))
            == (DptInq_t *)NULL)
                return ((DptInq_t *)NULL);
        (void)bzero ((caddr_t)inquire, sizeof(DptInq_t));
        (void)bzero ((caddr_t)&cdb, sizeof(cdb));
        CDB6(&cdb)->cdb_cmd = CMD_INQUIRY;
        CDB6(&cdb)->cdb_len = sizeof(DptInq_t);
        if ((dpticmd (&(sc->ha_hba), -1, &cdb, (caddr_t)inquire,
            sizeof(DptInq_t), B_READ) & STS_MASK) != STS_GOOD) {
                /*
                 *      Pre 3.G0 firmware may not respond, we could intuit the
                 *      response ourselves, but why waste our time!
                 */
                free ((caddr_t)inquire, M_TEMP);
                return ((DptInq_t *)NULL);
        }
        return (inquire);
}

static inline void
dpt_prstring (s, n)
        u_char *        s;
        int             n;
{       while ((--n >= 0) && (*s) && (*s != ' ') && (*s != '-'))
                printf ("%c", *(s++));
}

/*
 *      Attach the devices, and virtual devices to the driver list.
 */
void
dptattach (parent, self, aux)
        struct device *         parent;
        struct device *         self;
        void *                  aux;
{
        struct cfdata *         cf;
        int                     i, target, j, max;
        char *                  cp;
        char *                  name;
        vm_offset_t             addr;
#define sc              ((DptHa_t *)self)
        DptInq_t *              iq;
        static DptHa_t *        last_parent;

        if (!aux)
                return;

        if (!parent || !(cf = parent->dv_cfdata)) {
                if (!(((struct isa_attach_args *)aux)->ia_aux))
                        dpt_remove_to_do ();
                return;
        }
        if (!DptHa)
                DptHa = sc;
        /*
         * Most of the work is in common code, but we need to do some
         * DPT-specific setup in between, so the job is split.
         */
        sc->ha_ih.ih_fun = dptintr;
        if ((sc->ha_Watchtime = hz / 10) == 0)
                sc->ha_Watchtime = 1;
        sc->ha_Wedgetime = DPT_WEDGETIME / sc->ha_Watchtime;

        /*
         *      Lets see if we are virtualizing other channels and targets.
         */
        if (!(((struct isa_attach_args *)aux)->ia_aux)) {
                if ((!dpt_todo) || (dpt_todo_len <= 0))
                        return;
                /* Link to the parent software structure */
                sc->ha_parent = dpt_todo->todo_parent;
                sc->ha_next = dpt_todo->todo_parent->ha_next;
                dpt_todo->todo_parent->ha_next = sc;
                /* Fill in the important instance variables */
                sc->ha_Channel = dpt_todo->todo_Channel;
                sc->ha_ID = dpt_todo->todo_ID;
                /* Use common data */
                sc->ha_Base = dpt_todo->todo_parent->ha_Base;
                sc->ha_Config = dpt_todo->todo_parent->ha_Config;
                sc->ha_Status_Packet = dpt_todo->todo_parent->ha_Status_Packet;
                sc->ha_Status_Packet_addr
                    = dpt_todo->todo_parent->ha_Status_Packet_addr;
                dpt_remove_to_do ();
        } else {/*
                 *      This is the real McCoy!
                 */
                sc->ha_parent = sc;
                sc->ha_Base = ((struct isa_attach_args *)aux)->ia_iobase;
                sc->ha_Config = *((DptConfig_t *)(
                    ((struct isa_attach_args *)aux)->ia_aux));
                if ((i = sc->ha_Config.SPlength[3]
                  + (sc->ha_Config.SPlength[2] << 8)
                  + (sc->ha_Config.SPlength[1] << 16)
                  + (sc->ha_Config.SPlength[0] << 24)) < sizeof(DptStat_t))
                        i = sizeof(DptStat_t);
                if (addr = (vm_offset_t)malloc (i, M_DEVBUF, M_WAITOK)) {
                        (void)bzero ((char *)addr, sizeof(DptStat_t));
                        sc->ha_Status_Packet = (DptStat_t *)addr;

                        if (((addr = KVTOPHYS ((caddr_t)addr)) > ISA_ADDR)
                         && is_isa (sc)) {
                                printf (
                                    "%s: status struct cannot malloc ISA!\n",
                                    sc->ha_hba.hba_dev.dv_xname);
                                return;
                        }
                        sc->ha_Status_Packet_addr = VMTOSWAP (addr);
                } else {
                        printf ("%s: status struct cannot malloc!\n",
                           sc->ha_hba.hba_dev.dv_xname);
                        return;
                }
                /*
                 *      Construct the Bounce Buffers.
                 */
                if (sc->ha_Config.SG64K) {      /* Ignore value in SG_Size */
                        sc->ha_Config.SG_Size[1] = ((65536/sizeof(SgElement_t))
                            & 0xFF);
                        sc->ha_Config.SG_Size[0] = ((65536/sizeof(SgElement_t))
                            >> 8);
                }
                if (((i = sc->ha_Config.SG_Size[1]
                  + (sc->ha_Config.SG_Size[0] << 8)) > SG_SIZE)
                 || (i <= 0)) {
                        sc->ha_Config.SG_Size[1] = (SG_SIZE & 0xFF);
                        sc->ha_Config.SG_Size[0] = (SG_SIZE >> 8);
                }
                if (is_isa (sc) && ((vm_offset_t)(maxmem * NBPG) > ISA_ADDR))
                        dpt_init_bounce (sc);
                /*
                 * link the ccb's into a free-list
                 */
                if (((i = sc->ha_Config.QueueSize[1]
                    + (sc->ha_Config.QueueSize[0] << 8)) > SG_SIZE)
                 || (i <= 0)) {
                        sc->ha_Config.QueueSize[1] = (NCCBS & 0xFF);
                        sc->ha_Config.QueueSize[0] = (NCCBS >> 8);
                }
                dpt_init_ccb (sc);
                /*
                 *      Link parents
                 */
                if (last_parent)
                        last_parent->ha_next_parent = sc;
                last_parent = sc;
        }
#       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS & DPT_DEBUG_SET))
                sc->ha_debug = DPT_DEBUG_FLAGS & ~DPT_DEBUG_SET;
#       endif

        /* We only allocate storage for each MAX_TARGET_IDs */
        if ((i = sc->ha_Config.MaxScsiID) <= 0)
                sc->ha_Config.MaxScsiID = i = 7;        /* default */
        if ((i -= sc->ha_ID) >= MAX_TARGET_ID)
                i = MAX_TARGET_ID - 1;

        /*
         *      Allocate the Auto Request Sense Cache Buffers, and ensure a
         * per target and LUN record.
         */
        if ((sc->ha_sense = malloc ((++i) * sizeof (sc->ha_sense[0]),
            M_DEVBUF, M_WAITOK)) == (Scsi_Sense_t **)NULL)
                printf ("%s: Failed to allocated auto-request sense buffer\n",
                   sc->ha_hba.hba_dev.dv_xname);
        else {  if ((j = sc->ha_Config.MaxLUN) == 0)
                        sc->ha_Config.MaxLUN = j = MAX_LUN - 1; /* default */
                if (j >= MAX_LUN)
                        j = MAX_LUN - 1;
                j = (j + 1) * sizeof (sc->ha_sense[0][0]);
                while (--i >= 0)
                        sc->ha_sense[i] = malloc (j, M_DEVBUF, M_WAITOK);
        }

        /*
         *      Link into isa and set interrupt handler (our caller has
         *      already set ih_fun). negotiate bus master DMA.
         */
        isa_establish (&(sc->ha_isa), &(sc->ha_hba.hba_dev));
        sc->ha_ih.ih_arg = sc;
        if ((((struct isa_attach_args *)aux)->ia_aux)) {
                intr_establish (((struct isa_attach_args *)aux)->ia_irq,
                    &sc->ha_ih, DV_DISK);
                if ((i = ((struct isa_attach_args *)aux)->ia_drq) != DRQNONE)
                        at_dma_cascade (i);
        }
        /*
         * Check installed targets?
         */
        sc->ha_hba.hba_driver = &dpthbadriver;
        /*
         *      Only do a bus/HBA reset on the first time through.
         */
        if ((((struct isa_attach_args *)aux)->ia_aux))
                dpthbareset (&(sc->ha_hba), 0);
        printf (":");

        if (iq = dpt_hba_inquiry (sc)) {
                if (iq->vendor[0] && (iq->vendor[0] != ' ')) {
                        printf (" ");
                        dpt_prstring (iq->vendor, sizeof(iq->vendor));
                }
                if (iq->modelNum[0] && (iq->modelNum[0] != ' ')) {
                        printf (" ");
                        dpt_prstring (iq->modelNum, sizeof(iq->modelNum));
                        /* Differential? */
                        if (iq->modelSuf[5] && (iq->modelSuf[5] != ' '))
                                printf ("%c", iq->modelSuf[5]);
                        /* Wide Board? */
                        if (iq->modelSuf[7] && (iq->modelSuf[7] != ' '))
                                printf ("%c", iq->modelSuf[7]);
                        dpt_prstring (iq->modelSuf, sizeof(iq->modelSuf));
                        printf (" SCSI HBA");
                }
                if (iq->firmware[0] && (iq->firmware[0] != ' ')) {
                        printf (" v");
                        if (iq->firmware[0] == '0')
                                dpt_prstring (iq->firmware + 1,
                                    sizeof(iq->firmware) - 1);
                        else    dpt_prstring (iq->firmware,
                                    sizeof(iq->firmware));
                        printf (".%c", iq->revision);
                }
                free ((caddr_t)iq, M_TEMP);
        }
        /*
         *      Probe the bus and identify the targets.
         */
        /*
         *      Add in additional probe responses for more channels
         * and targets. Do this only if there are targets present.
         */
        if (((struct isa_attach_args *)aux)->ia_aux) {
                max = 0;        /* No used channels, no SCSI-3 targets */
#               define TOO_MANY_TARGETS 0x8000
                for (i = 0; i <= sc->ha_Config.MaxChannel; ++i) {
                        /*
                         *      Handle BUS 0 Target MAX_TARGET_ID and up.
                         */
                        target = 0;
                        if (i)  target = -MAX_TARGET_ID;
                        /*
                         *      Scan for all targets (LUN 0) on the virtual
                         *      adapters and determine if it is worthwhile to
                         *      virtualize it.
                         */
                        while ((target += MAX_TARGET_ID) < sc->ha_Config.MaxScsiID) {
                                j = 0;
                                while ((dpt_test_unit_ready(sc,target+j,i) == 0)
                                 && (++j < MAX_TARGET_ID));
                                if (j < MAX_TARGET_ID) {
                                        if (target >= MAX_TARGET_ID)
                                                max |= TOO_MANY_TARGETS;
                                        if (i > (max & ~TOO_MANY_TARGETS))
                                                max = (max&TOO_MANY_TARGETS)|i;
                                        dpt_add_to_do (sc, target, i);
                                }
                        }
                }
                /*
                 *      Patch these characteristics based on the hardware
                 *      found on the additional IDs to cleanup ID and BUS
                 *      reporting below if they are not in fact used by the
                 *      virtual Adapters.
                 */
                if ((max & TOO_MANY_TARGETS) == 0)
                        sc->ha_Config.MaxScsiID = MAX_TARGET_ID - 1;
                sc->ha_Config.MaxChannel = max & ~TOO_MANY_TARGETS;
        }

        if (sc->ha_Config.MaxScsiID >= MAX_TARGET_ID)
                printf (" id %d-%d", sc->ha_ID, sc->ha_ID + MAX_TARGET_ID - 1);
        if (sc->ha_Config.MaxChannel > 0)
                printf (" bus %d", sc->ha_Channel);
        printf ("\n");
        for (target = 0; target < MAX_TARGET_ID; ++target) {
                if (dpt_test_unit_ready (sc, target + sc->ha_ID,
                    sc->ha_Channel) == 0) {
                        free (sc->ha_sense[target], M_DEVBUF);
                        sc->ha_sense[target] = (Scsi_Sense_t *)NULL;
                        continue;
                }
                /*
                 * Register the target and it's LUNs
                 * I would like to re-allocate the sense buffer to the
                 * maximum LUN detected to optimize the sense buffer storage
                 * still furthur (typical savings of 794 bytes), but know of
                 * no convenient way to do this, that is transportable to
                 * possible future OS designs.
                 */
                SCSI_FOUNDTARGET (&(sc->ha_hba), target);
        }
        /*
         *      What about dynamically attached units?
         */

        /*
         *      Fixup the OS revision as saved in the dptsig for the
         *      engine (dptioctl.h) to pick up.
         */
        dpt_sig.dsDescription[9] = osrelease[0];
        dpt_sig.dsDescription[10] = osrelease[1];
        dpt_sig.dsDescription[11] = osrelease[2];
#undef sc
}

/*
 *      Reset the HBA, targets and BUS.
 */
void
dpthbareset(hba, resetunits)
        struct hba_softc *      hba;
        int                     resetunits;
{
        outb (((DptHa_t *)hba)->ha_Base + HA_COMMAND, CP_EATA_RESET);
        /* Old firmware has a headache if you talk to it too soon! */
        DELAY (750000);
        if (resetunits)
                scsi_reset_units(hba);
}

/*
 * Allocate an outbound bounce page, copying data if necessary.
 */
static inline vm_offset_t
dpt_bounce(sc, ccb, v, len, rw)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
        caddr_t         v;
        vm_size_t       len;
        int             rw;
{       DptBounce_t *   bo;
        vm_size_t       o;

        bo = dpt_get_bounce (sc, FL_NOMASK);
        bo->bo_next = ccb->ccb_Bounce;
        ccb->ccb_Bounce = bo;
        /*
         *      This bit of magic has no known purpose. I am simply copying
         *      what was done in other HBA drivers assuming that it had a
         *      performance advantage for the bcopy routine.
         */
/* check this! - pjd */
        o = (int)v & PGOFSET;
        if (rw == B_READ) {
                bo->bo_dst = (caddr_t)v;
                bo->bo_len = len;
        } else  (void)bcopy((void *)v, (void *)(bo->bo_v + o), len);
        return (bo->bo_p + o);
}

/*
 * Finish up bouncing in on a read. Must be called at splbio. This routine
 * removes the CCB from any active or pending lists for an HBA as well.
 */
STATIC void
dpt_cleanup(sc, ccb)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
{       vm_offset_t     dst;
        DptBounce_t *   bo;
        DptCcb_t *      ccb_p;
        DptCcb_t *      ccb_l;

        while (bo = ccb->ccb_Bounce) {
                if (dst = (vm_offset_t)bo->bo_dst) {
                        (void)bcopy((void *)(bo->bo_v + (dst & PGOFSET)),
                            (void *)dst, bo->bo_len);
                        bo->bo_dst = (caddr_t)NULL;
                }
                ccb->ccb_Bounce = bo->bo_next;
                dpt_free_bounce (sc, FL_NOMASK, bo);
        }
/* seems like common queue and dequeue routines would be useful here - pjd */
        /* Remove from list of active ccbs for this HBA (for dptwatch) */
        for (ccb_l = (DptCcb_t *)NULL, ccb_p = sc->ha_CCB;
            ccb_p;
            ccb_l = ccb_p, ccb_p = ccb_p->ccb_next) {
                if (ccb_p == ccb) {
                        if (ccb_l == (DptCcb_t *)NULL)
                                sc->ha_CCB = ccb_p->ccb_next;
                        else    ccb_l->ccb_next = ccb_p->ccb_next;
                        break;
                }
        }
        for (ccb_l = (DptCcb_t *)NULL, ccb_p = sc->ha_Go_CCB;
            ccb_p;
            ccb_l = ccb_p, ccb_p = ccb_p->ccb_next) {
                if (ccb_p == ccb) {
                        if (ccb_l)
                                ccb_l->ccb_next = ccb_p->ccb_next;
                        else    sc->ha_Go_CCB = ccb_p->ccb_next;
                        break;
                }
        }
        dpt_free_ccb (sc, FL_NOMASK, ccb);
}

/*
 * Initialize (most of the changeable section) a SCCB.
 * We assume that the CDB has already been set up, so all we do here is
 * generate the Scatter Gather list.
 */
STATIC void
dpt_init_sccb(sc, ccb, target, intr, dev, bp)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
        int             target;
        scintr_fn       intr;
        struct device * dev;
        struct buf *    bp;
{       int             i, n, o, isa, rw, sgpages, pages;
        SgElement_t *   sg;
        u_int           aphys;
        vm_offset_t     offset, phys;
        caddr_t         v;
        vm_size_t       size, len;
#if (defined(DPT_DEBUG_FLAGS))
                static char     SG_size[] = "SG size";
#endif

        ccb->EataCp.cp_Scatter_Gather = 0;
        ccb->EataCp.cp_DataIn = 0;
        ccb->EataCp.cp_DataOut = 0;
        /*
         *      If we are talking to target -1, lets actually talk to the HBA
         *      rather than to the bus.
         */
        i = target + sc->ha_ID;
        if (target == -1) {
                i = sc->ha_Config.HBA[MAX_CHANNEL - 1 - sc->ha_Channel];
                ccb->EataCp.cp_Interpret = 1;
        }
        ccb->EataCp.cp_id = i;
        ccb->EataCp.cp_Channel = sc->ha_Channel;
        ccb->EataCp.cp_msg_disco_rico = 1;
        ccb->EataCp.cp_msg_lun = ccb->EataCp.cp_lun;

        /*
         * Given a buffer describing a transfer, set up a scatter/gather map
         * in a ccb to map that SCSI transfer. Too bad their generic routine
         * has the sg32 structure backwards with respect to the DPT variant,
         * we had to handle the ISA bounce in any case ... Initialize the CCB
         * opcode for the correct transfer type.
         */

        rw = bp->b_flags & B_READ;
        isa = is_isa (sc);
        /*
         * Given a transfer described by a `struct buf', figure out what kind
         * of transfer is needed (direct or scatter/gather).  If scatter/gather
         * is required, set up the given s/g map and return the number of
         * entries; otherwise store the physical address for the transfer's
         * page(s) and return 0.
         *
         * If the transfer uses a chain (bp->b_chain != NULL), we assume that
         * no <addr,len> pair ever crosses a page boundary.  In any case, at
         * most two pages can be partial (one at the start and one at the end).
         *
         * (Lots of ifdef'd panics here, from sheer raging paranoia.)
         */
        sg = &ccb->ccb_SG[0];

        /*
         * ###: We used to xfer 0 bytes at addr 0 for len < 0, but I
         * think this is an error.  If not, our caller will have to
         * check len < 0 and change that to len = 0 anyway.
         */
        len = bp->b_iocount;
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                if ((sc->ha_debug) && (len < 0))
                        panic("count < 0");
#endif
        if (len <= 0)
                aphys = 0;              /* physadr = 0 */
        else if ((sgpages = i386_btop (i386_round_page (len + (o = (int)(v
            = bp->b_un.b_addr) & PGOFSET)))) == 1) {
                /* Transfer lies entirely within a page. */
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                        if ((sc->ha_debug) && (bp->b_chain))
                                panic("SG chain 1page");
#endif
                aphys = KVTOPHYS(v);
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                        if ((sc->ha_debug)
                         && ((aphys + len - 1) != KVTOPHYS(v + len - 1)))
                                panic("SG split page");
#endif
                /*
                 * Point the CCB directly at the transfer, or at a
                 * bounce buffer. The i386_btop program will ensure that
                 * we never reach here if we have an operation that
                 * excedes the page size (the fundamental size of the
                 * bounce buffer pool).
                 */
                if (isa && (aphys >= ISA_ADDR))
                        aphys = dpt_bounce (sc, ccb, v, len, rw);
                aphys = VMTOSWAP (aphys);
        } else {/*
                 * Transfer too big (at least 2 pages) -- we'll need a
                 * scatter/gather map.  (Actually, if the underlying pages
                 * are contiguous, we could be OK, but this gets complicated,
                 * especially when using bounce buffers in, so just use the
                 * map initially and then see if the resulting buffer is
                 * contiguous).
                 *
                 * Transfer may also be chained, in which case we must step
                 * to the next buffer each time bp->b_bcount runs out (note
                 * that len is the head buffer's bp->b_iocount).
                 */
                if ((pages = sgpages) > (sc->ha_Config.SG_Size[1]
                    + (sc->ha_Config.SG_Size[0] << 8)))
                        panic("SG pages");

                /* First page may start at some offset, and hence be short. */
                n = bp->b_bcount;
                size = NBPG - o;
                phys = KVTOPHYS(v);
                if (isa && (phys >= ISA_ADDR))
                        phys = dpt_bounce (sc, ccb, v, (vm_size_t)size, rw);
                sg->data_addr = VMTOSWAP (offset = phys);
                (sg++)->data_len = LMTOSWAP (size);
                v += size;
                n -= size;

                /* Pages 1 through (pages-1), if pages > 2, are full size. */
                for (i = 2; i < sgpages; i++) {
                        if (n <= 0) {
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                        if ((sc->ha_debug)
                                         && (n < 0 || (bp->b_chain == NULL)))
                                                panic("ab_sgmap mid chain");
#endif
                                n = (bp = bp->b_chain)->b_bcount;
                                v = bp->b_un.b_addr;
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                        if (sc->ha_debug && ((int)v & PGOFSET))
                                                panic("SG mid addr");
#endif
                        }
                        phys = KVTOPHYS(v);
                        if (isa && (phys >= ISA_ADDR))
                                phys = dpt_bounce (sc, ccb, v, (vm_size_t)NBPG,
                                    rw);
                        /*
                         *      Coalesce physically adjacent buffers.
                         */
                        if ((offset + size) == phys) {
                                size += NBPG;
                                --sg;
                                --pages;
                        } else {
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                     if (sc->ha_debug) {
                                        if (sg > &ccb->ccb_SG[SG_SIZE - 1])
                                                panic (SG_size);
                                        if (sg > &ccb->ccb_SG[
                                             sc->ha_Config.SG_Size[1]
                                          + (sc->ha_Config.SG_Size[0]<<8) - 1])
                                                printf (SG_size);
                                    }
#endif
                                sg->data_addr = VMTOSWAP (offset = phys);
                                size = NBPG;
                        }
                        (sg++)->data_len = LMTOSWAP (size);
                        v += NBPG;
                        n -= NBPG;
                }

                /* Last page is n remaining bytes. */
                if (n <= 0) {
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                if ((sc->ha_debug)
                                 && (n < 0 || bp->b_chain == NULL))
                                        panic("SG last chain");
#endif
                        n = (bp = bp->b_chain)->b_bcount;
                        v = bp->b_un.b_addr;
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                                if ((sc->ha_debug) && ((int)v & PGOFSET))
                                        panic("SG last addr");
#endif
                }
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                        if ((sc->ha_debug) && (n > NBPG))
                                panic("SG lastpg %d", n);
#endif
                /*
                 *      Coalesce physically adjacent buffers.
                 */
                phys = KVTOPHYS(v);
                if (isa && (phys >= ISA_ADDR))
                        phys = dpt_bounce (sc, ccb, v, (vm_size_t)n, rw);
                if ((offset + size) == phys) {
                        n += size;
                        --sg;
                        --pages;
                } else {
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                            if (sc->ha_debug) {
                                if (sg > &ccb->ccb_SG[SG_SIZE - 1])
                                        panic (SG_size);
                                if (sg > &ccb->ccb_SG[
                                     sc->ha_Config.SG_Size[1]
                                  + (sc->ha_Config.SG_Size[0]<<8) - 1])
                                        printf (SG_size);
                            }
#endif
                        sg->data_addr = VMTOSWAP (phys);
                }
                sg->data_len = LMTOSWAP (n);
                /*
                 * Point the CCB at the scatter/gather map if there
                 * remains more than one scatter gather entry.
                 */
                if (pages <= 1) {
                        aphys = ccb->ccb_SG[0].data_addr;
                        /*
                         *      I wish I could use
                         *              len = bp->b_iocount;
                         *      but we have long lost the bp counter.
                         */
                        len = SWAPTOLM (ccb->ccb_SG[0].data_len);
                } else {aphys = ccb->ccb_SG_addr;
                        len = (vm_size_t)(pages * sizeof (ccb->ccb_SG[0]));
                        ccb->EataCp.cp_Scatter_Gather = 1;
                }
        }
        if (len) {
                if (rw) ccb->EataCp.cp_DataIn = 1;
                else    ccb->EataCp.cp_DataOut = 1;
        }
        ccb->EataCp.cp_dataDMA = aphys;
        ccb->EataCp.cp_datalen = LMTOSWAP (len);
        ccb->EataCp.cp_statDMA = sc->ha_Status_Packet_addr;

        ccb->ccb_intr = intr;
        ccb->ccb_intrdev = dev;
}

static inline void
dpt_error (sc, ccb, msg, errno, table, size)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
        char *          msg;
        int             errno;
        char **         table;
        int             size;
{       char *          cp;
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
            static struct {
                u_char  command;
                char *  name;
            } scsi_commands[] = {
                { 0x00, "Test Unit Ready" },
                { 0x01, "Rezero Unit" },
                { 0x03, "Request Sense" },
                { 0x04, "Format Unit" },
                { 0x07, "Reassign Blocks" },
                { 0x08, "Read" },
                { 0x0A, "Write" },
                { 0x0B, "Seek" },
                { 0x12, "Inquiry" },
                { 0x15, "Mode Select" },
                { 0x16, "Reserve" },
                { 0x17, "Release" },
                { 0x18, "Copy" },
                { 0x1A, "Mode Sense" },
                { 0x1B, "Start/Stop/Eject/Load Unit" },
                { 0x1C, "Receive Diagnostic Results" },
                { 0x1D, "Send Diagnostic" },
                { 0x1E, "Prevent/Allow Media Removal" },
                { 0x25, "Read Capacity" },
                { 0x26, "Last Session" },
                { 0x28, "Extended Read" },
                { 0x2A, "Extended Write" },
                { 0x2B, "Extended Seek" },
                { 0x2E, "Write and Verify" },
                { 0x2F, "Verify" },
                { 0x30, "Search Data High" },
                { 0x31, "Search Data Equal" },
                { 0x32, "Search Data Low" },
                { 0x33, "Set Limits" },
                { 0x34, "Pre-Fetch" },
                { 0x35, "Synchronize Cache" },
                { 0x36, "Lock/Unlock Cache" },
                { 0x37, "Read Defect Data" },
                { 0x39, "Compare" },
                { 0x3A, "Copy and Verify" },
                { 0x3B, "Write Buffer" },
                { 0x3C, "Read Buffer" },
                { 0x3E, "Read Long" },
                { 0x3F, "Write Long" },
                { 0x40, "Change Definition" },
                { 0x41, "Write Same" },
                { 0x42, "Read Sub-Channel" },
                { 0x43, "Read Table of Contents" },
                { 0x44, "Read Header" },
                { 0x45, "Play Audio" },
                { 0x47, "Play Audio MSF" },
                { 0x48, "Play Audio TRK" },
                { 0x49, "Play Audio REL" },
                { 0x4B, "Pause/Resume" },
                { 0x4C, "Log Select" },
                { 0x4D, "Log Sense" },
                { 0x55, "Extended Mode Select" },
                { 0x5A, "Extended Mode Sense" },
                { 0xA5, "Play Audio 12" },
                { 0xA8, "Read 12" },
                { 0xA9, "Play Audio REL 12" },
                { 0xC0, "Set Address Format" },
                { 0xC1, "Read Table of Contents (SONY)" },
                { 0xC2, "Read Sub-Q" },
                { 0xC3, "Set Stop Time" },
                { 0xC4, "Playback Status" },
                { 0xC5, "Pause/Resume VU" },
                { 0xC6, "Play Audio TRK (SONY)" },
                { 0xC7, "Play Audio MSF (SONY)" },
                { 0xC8, "Play Audio VU" },
                { 0xC9, "Playback Control" },
                { 0xCA, "Pause/Resume (PIONEER)" },
                { 0xCB, "Audio Stop" },
                { 0xCC, "Audio Status" },
                { 0xD4, "Read D/A (NEC)" },
                { 0xD8, "Read D/A" },
                { 0xD9, "Audio Play" },
                { 0xDA, "Set Speed" },
                { 0xDC, "Eject/Load" },
                { 0xDD, "Set Speed (CHINON)" },
                { 0xDE, "Read Table of Contents (NEC)" },
                { 0xE0, "Drive Status" },
                { 0xE3, "Write CDP" },
                { 0xE4, "Read CDP" }
            };
            int i;
#endif

        printf ("%s", sc->ha_hba.hba_dev.dv_xname);
        if (sc->ha_Config.MaxChannel)
                printf (" bus%d", sc->ha_Channel);
        if (!(ccb->EataCp.cp_Interpret))
                printf (" target%d lun%d", ccb->EataCp.cp_id + sc->ha_ID,
                    ccb->EataCp.cp_lun);
        printf (": ");
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
          if (sc->ha_debug) {
            for (i=0; i < (sizeof(scsi_commands)/sizeof(scsi_commands[0])); ++i)
                if (ccb->EataCp.cp_scsi_cmd == scsi_commands[i].command)
                    break;
            if (i < (sizeof(scsi_commands)/sizeof(scsi_commands[0])))
                printf ("%s", scsi_commands[i].name);
            else
                printf ("%x", ccb->EataCp.cp_scsi_cmd);
          } else
#endif
                printf ("%x", ccb->EataCp.cp_scsi_cmd);
        printf (" %s ", msg);
        if ((errno < size) && (cp = table[errno]))
            printf ("%s\n", cp);
        else
            printf ("error 0x%x\n", errno);
}

/*
 * Fire off a CCB.
 *      dptqueue - Queue one into the adapter.
 *      dptccb - Wait until completion, do not rely on interrupts.
 */
STATIC void
dptqueue(sc, ccb)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
{       int             i;

        /* Don't do it if we have no cacheing of the SCSI sense information */
        ccb->EataCp.cp_Auto_Req_Sen = (sc->ha_sense
         && ((i = ccb->EataCp.cp_id - sc->ha_ID) >= 0) && (i < MAX_TARGET_ID)
         && (sc->ha_sense[i]));

        /*
         *      Limit the number of CCBs are sent to this HBA. Better to sleep,
         *      than to hardware loop like a nut! By limiting the number of
         *      CCBs to an individual HBA here, we manage to perform all the
         *      processing of the CCB ready to drop the next one into the
         *      controller. We could limit the CCBs we are allowed to take,
         *      but that may have a performance hit.
         */
        ++(sc->ha_IO_Pending);
        ++(sc->ha_parent->ha_Total_IO_Pending);
        while (sc->ha_parent->ha_Total_IO_Pending > (sc->ha_Config.QueueSize[1]
            + (sc->ha_Config.QueueSize[0] << 8)))
                tsleep ((caddr_t)&(sc->ha_parent->ha_Total_IO_Pending), PRIBIO,
                    "dptqueue", 0);

#       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS & DPT_DEBUG_VERBOSE))
                if (sc->ha_debug & DPT_DEBUG_VERBOSE)
                        Print_EataCp (ccb);
#       endif
        dpt_cmd (sc->ha_Base, ccb->ccb_addr);

        if (!sc->ha_Watchdog) {
                sc->ha_Watchdog = 1;
                sc->ha_Wedgecounter = sc->ha_Wedgetime;
                timeout((void (*)(void *))dptwatch, sc, sc->ha_Watchtime);
        }
}

STATIC int
dptccb(sc, ccb, flags)
        DptHa_t *       sc;
        DptCcb_t *      ccb;
        int             flags;
{       int             i, status, s;
        DptCcb_t *      ccb_p;
        DptCcb_t *      ccb_l;
        static char *   HBA_stat[] = {
                "No Error on command",
                "Device Selection Time Out",
                "Device Command Time Out",
                "SCSI Bus was RESET !",
                "Initial Controller Power Up",
                "Unexpected BUS Phase",
                "Unexpected BUS Free",
                "SCSI Bus Parity Error",
                "SCSI Bus Hung",
                "Unexpected Message Reject",
                "SCSI Bus Reset Stuck",
                "Auto-Request Sense Failed",
                "HBA Memory Parity error",
                "CP aborted - NOT on Bus",
                "CP aborted - WAS on Bus",
                "CP was reset - NOT on Bus",
                "CP was reset - WAS on Bus",
                "HBA Memory ECC Error",
                "PCI Parity Error",
                "PCI Master Abort",
                "PCI Target Abort",
                "PCI Signalled Target Abort",
                NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                "Software Abort"
        };
#if (defined(DPT_DEBUG_FLAGS))
            static char *       tastat[] = {
                "Success, command done",
                NULL,
                "Check condition",
                NULL,
                "Condition met",
                NULL, NULL, NULL,
                "Busy",
                NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                "succeeded, doing linked command",
                NULL, NULL, NULL,
                "Condition met, doing linked command",
                NULL, NULL, NULL,
                "Reservation conflict",
                NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                "Command terminated",
                NULL, NULL, NULL, NULL, NULL,
                "Queue full",                           /* SCSI 2 */
                NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                "Auto Contingent Allegiance condition"  /* SCSI 3 */
            };
#endif

        /* Don't do it if we have no cacheing of the SCSI sense information */
        ccb->EataCp.cp_Auto_Req_Sen = (sc->ha_sense
         && ((i = ccb->EataCp.cp_id - sc->ha_ID) >= 0) && (i < MAX_TARGET_ID)
         && (sc->ha_sense[i]));

        if (!(flags & FL_NOMASK))
                s = splbio ();
        /*
         *      Limit the number of CCBs are sent to this HBA. Better to sleep,
         *      than to hardware loop like a nut! By limiting the number of
         *      CCBs to an individual HBA here, we manage to perform all the
         *      processing of the CCB ready to drop the next one into the
         *      controller. We could limit the CCBs we are allowed to take,
         *      but that may have a performance hit.
         */
        ++(sc->ha_IO_Pending);
        ++(sc->ha_parent->ha_Total_IO_Pending);
        while (sc->ha_parent->ha_Total_IO_Pending > (sc->ha_Config.QueueSize[1]
            + (sc->ha_Config.QueueSize[0] << 8)))
                tsleep ((caddr_t)&(sc->ha_parent->ha_Total_IO_Pending), PRIBIO,
                    "dptccb", 0);
#       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS & DPT_DEBUG_VERBOSE))
                if (sc->ha_debug & DPT_DEBUG_VERBOSE)
                        Print_EataCp (ccb);
#       endif
        dpt_cmd (sc->ha_Base, ccb->ccb_addr);

        /*
         *      Move from Pending to Active list
         */
        for (ccb_l = (DptCcb_t *)NULL, ccb_p = sc->ha_Go_CCB;
            ccb_p;
            ccb_l = ccb_p, ccb_p = ccb_p->ccb_next) {
                if (ccb_p == ccb) {
                        if (ccb_l)
                                ccb_l->ccb_next = ccb_p->ccb_next;
                        else    sc->ha_Go_CCB = ccb_p->ccb_next;
                        break;
                }
        }
        ccb->ccb_next = sc->ha_CCB;
        sc->ha_CCB = ccb;

        /*
         * Wait for the board to report a finished instruction. We should
         * time out on this but we could be doing a format command (NBL) ...
         */
        i = 40000;      /* Wait 2 seconds on selected commands */
        for (;;) {
                (void)dptintr (sc);
                for (ccb_p = sc->ha_CCB; ccb_p; ccb_p = ccb_p->ccb_next)
                        if (ccb_p == ccb)
                                break;
                if (ccb_p == (DptCcb_t *)NULL)
                        break;
                switch (ccb->EataCp.cp_scsi_cmd) {
                case CMD_TEST_UNIT_READY:
                case CMD_INQUIRY:
                case CMD_REQUEST_SENSE:
                        if (--i <= 0) {
                                /*
                                 *      We could have simply lost the interrupt
                                 *      (SCSI Bus Reset on old hardware)
                                 */
                                if (sc->ha_Status_Packet->sp_vp == ccb) {
                                        ccb->ccb_Host_Status =
                                          sc->ha_Status_Packet->sp_HBA_stat;
                                        ccb->ccb_Target_Status =
                                          sc->ha_Status_Packet->sp_SCSI_stat;
                                        ccb->ccb_Residue = SWAPTOLM (
                                          sc->ha_Status_Packet->sp_inv_residue);
                                } else {
                                        ccb->ccb_Host_Status = HA_ABORT;
                                        dpthbareset (&(sc->ha_hba), 0);
                                }
                                dpt_cleanup (sc, ccb);
                                break;
                        }
                        DELAY (50);     /* Gives meaning to the timer */
                default:continue;
                }
                break;
        }

        /*
         *      Quickly before this ccb is taken and used for any
         *      subsequent commands. We are at splbio right now, so it is not
         *      a problem.
         */
        if ((status = ccb->ccb_Host_Status) == HA_NO_ERROR) {
            status = ccb->ccb_Target_Status;
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
                if ((sc->ha_debug) && (status != STS_GOOD)
                 && (!(flags & FL_QUIET)))
                    dpt_error (sc, ccb, "Target", status, tastat,
                        sizeof(tastat) / sizeof(tastat[0]));
#endif
        }
        else if ((ccb->EataCp.cp_scsi_cmd == CMD_TEST_UNIT_READY)
         || (flags & FL_QUIET))
                /* probing, don't make noise */
                status = -1;
        else {  dpt_error (sc, ccb, "Host", status, HBA_stat,
                    sizeof(HBA_stat) / sizeof(HBA_stat[0]));
                status = -1;
        }
        if (!(flags & FL_NOMASK))
                splx(s);
        return (status);
}

/* immediate command: should not depend on receiving interrupts */
int
dpticmd(hba, target, cdb, buf, len, rw)
        struct hba_softc *      hba;
        int                     target;
        struct scsi_cdb *       cdb;
        caddr_t                 buf;
        int                     len, rw;
{       DptCcb_t *              ccb;
        struct buf              bufhdr;
#define sc              ((DptHa_t *)hba)
        Scsi_Sense_t *          sp;

        /*
         *      Do we do any spoofing of requests?
         */
        if (cdb) {
                switch (cdb->cdb_bytes[0]) {
                /*
                 * Even `ICMD' should have Request Sense Cacheing!
                 */
                case CMD_REQUEST_SENSE:
                        if ((target >= 0) && (target < MAX_TARGET_ID)
                         && (sc->ha_sense) && (sp = sc->ha_sense[target])
                         && (len <= sizeof(Scsi_Sense_t))) {
                                (void)bcopy (&(sp[(CDB6(cdb)->cdb_lun_lbah>>5)
                                    & 0x7]), buf, len);
                                return (STS_GOOD);
                        }
                        break;
                /*
                 *      During probe phase only. The BSDi kernel breaks from
                 *      the Defacto standard of always doing an INQUIRY
                 *      command first to a device. This is not serious
                 *      except that the Micropolis drives will do a disconnect
                 *      on a TEST_UNIT_READY to an invalid LUN and reconnect
                 *      on LUN 0 with their response indicating that the
                 *      device is unsupported. They do, however, handle an
                 *      inquiry correctly as per the Defacto Standard.
                 */
                case CMD_TEST_UNIT_READY:
                        if ((target >= 0) && (target < MAX_TARGET_ID)) {
                                int             lun_lbah, mask;

                                sc->ha_inquired[target] |= 1;
                                if ((sc->ha_inquired[target] & (mask
                                    = (1 << ((lun_lbah
                                    = CDB6(cdb)->cdb_lun_lbah & (0x7 << 5))
                                      >> 5)))) == 0) {
                                        struct scsi_cdb         cdb_tmp;
                                        struct scsi_inquiry     si;

                                        (void)bzero (&cdb_tmp, sizeof(cdb_tmp));
                                        CDB6(&cdb_tmp)->cdb_cmd = CMD_INQUIRY;
                                        CDB6(&cdb_tmp)->cdb_len = sizeof(si);
                                        CDB6(&cdb_tmp)->cdb_lun_lbah = lun_lbah;
                                        if (((dpticmd (hba, target, &cdb_tmp,
                                            (caddr_t)&si, sizeof(si),
                                            B_READ) & STS_MASK) != STS_GOOD)
                                         || ((si.si_type & TYPE_TYPE_MASK)
                                            == TYPE_NP))
                                                sc->ha_unavailable[target]
                                                    |= mask;
                                        sc->ha_inquired[target] |= mask;
                                }
                                if (sc->ha_unavailable[target] & mask)
                                        return (-1);
                                /*
                                 *      Fall through and do a Test Unit Ready.
                                 */
                        }
                        break;
                }
        }
        ccb = dpt_get_ccb (sc, 0, cdb);
        bufhdr.b_un.b_addr = buf;
        bufhdr.b_iocount = bufhdr.b_bcount = len;
        bufhdr.b_flags = rw;
        bufhdr.b_chain = NULL;
        dpt_init_sccb (sc, ccb, target, (scintr_fn)NULL, (struct device *)NULL,
            &bufhdr);
        return (dptccb(sc, ccb, 0));
#undef sc
}

/* crash dump: like dpticmd(B_WRITE), but possibly from physical */
int
dptdump(hba, target, cdb, buf, len)
        struct hba_softc *      hba;
        int                     target;
        struct scsi_cdb *       cdb;
        caddr_t                 buf;
        int                     len;
{       DptCcb_t *              ccb;
        u_long                  from, to, lastfrom;
        DptBounce_t *           bo;
#define sc              ((DptHa_t *)hba)

        ccb = dpt_get_ccb (sc, FL_NOMASK, cdb);
        /*
         * Several assumptions:
         * +    The upper-level dump code always calls us on aligned
         *      2^n chunks which never cross 64 KB physical memory boundaries
         * +    We're running in virtual mode with interrupts blocked
         */

        ccb->EataCp.cp_id = target + sc->ha_ID;
        ccb->EataCp.cp_Channel = sc->ha_Channel;
        if (is_isa (sc) && ((vm_offset_t)buf >= ISA_ADDR)) {
                /* assumes len <= SG_SIZE * NBPG and contiguous bounce pages */
                from = btoc((u_long)(vm_offset_t)buf);
                bo = dpt_get_bounce (sc, FL_NOMASK);
                to = btoc(bo->bo_p);
                lastfrom = btoc((u_long)&((char *)(vm_offset_t)buf)[
                    (vm_size_t)len]);
                while (from < lastfrom)
                        physcopyseg (from++, to++);
                buf = (caddr_t)bo->bo_p;
                bo->bo_next = ccb->ccb_Bounce;
                ccb->ccb_Bounce = bo;
        }
        ccb->EataCp.cp_dataDMA = VMTOSWAP ((vm_offset_t)buf);
        ccb->EataCp.cp_statDMA = sc->ha_Status_Packet_addr;
        ccb->EataCp.cp_datalen = LMTOSWAP ((vm_size_t)len);

        ccb->ccb_intr = (scintr_fn)NULL;
        ccb->ccb_intrdev = (struct device *)NULL;

        return (dptccb (sc, ccb, FL_NOMASK));
#undef sc
}

/*
 * Start a transfer. Allocate DMA & bus
 *
 * Since the Adapter handles transactions in parallel (with disconnect /
 * reconnect), we don't have to queue requests for the host adapter.
 */
void
dptstart(self, sq, bp, dgo, dev)
        struct device * self;
        struct sq *     sq;
        struct buf *    bp;
        scdgo_fn        dgo;
        struct device * dev;
{
        if (bp) {
                /* asynch transaction */
                (*dgo)(dev, (struct scsi_cdb *)(dpt_get_ccb ((DptHa_t *)self,
                    FL_NOMASK, (struct scsi_cdb *)NULL)->EataCp.cp_cdb));
                return;
        }
        /* let dpticmd() allocate its own ccb */
        (*dgo)(dev, (struct scsi_cdb *)NULL);
}

/*
 * Get the host adapter going on a command. start DMA xfer on bus.
 *
 * XXX should we allocate the DMA channel here, rather than dedicate one?
 * XXX must be called at splbio() since interrupts can call it
 * XXX but it'd be much better for the work to get done at low ipl!
 */
int
dptgo(self, target, intr, dev, bp, pad)
        struct device * self;
        int             target, pad;
        scintr_fn       intr;
        struct device * dev;
        struct buf *    bp;
{       Scsi_Sense_t *  sp;
#define sc      ((DptHa_t *)self)
        DptCcb_t *      ccb = sc->ha_Go_CCB;

        /*
         *      Move from Pending (ha_Go_CCB) to Active (ha_CCB) list
         */
        sc->ha_Go_CCB = ccb->ccb_next;
        ccb->ccb_next = sc->ha_CCB;
        sc->ha_CCB = ccb;
        /*
         *      Perform any cached Request Sense activities.
         */
        if ((ccb->EataCp.cp_scsi_cmd == CMD_REQUEST_SENSE)
         && (target >= 0) && (target < MAX_TARGET_ID)
         && (sc->ha_sense) && (sp = sc->ha_sense[target])
         && (bp->b_bcount <= sizeof(Scsi_Sense_t))) {
                (void)bcopy (&(sp[ccb->EataCp.cp_lun]), bp->b_un.b_addr,
                    bp->b_bcount);
                if ((sc->ha_hba.hba_intr = intr)
                 && (sc->ha_hba.hba_intrdev = dev)) {
                        struct sq *     sq;

                        /*
                         * For non-immediate commands,
                         * pass status back to higher levels and
                         * start the next transfer.
                         */
                        (*intr)(dev, STS_GOOD, 0);
/* Look out for macro "IF" below! - pjd */
                        IF ((sq = sc->ha_hba.hba_head) != (struct sq *)NULL) {
                                sc->ha_hba.hba_head = sq->sq_forw;
                                (*sq->sq_dgo)(sq->sq_dev, (struct scsi_cdb *)(
                                    dpt_get_ccb (sc, FL_NOMASK,
                                      (struct scsi_cdb *)NULL)->EataCp.cp_cdb));
                        }
                }
                dpt_cleanup (sc, ccb);
                return (STS_GOOD);
        }
        /*
         *      Initialize and send the command
         */
        dpt_init_sccb (sc, ccb, target, intr, dev, bp);
        dptqueue (sc, ccb);
        return (STS_GOOD);
#undef sc
}

/*
 * Handle processing of current CCB as pointed to by the Status.
 */
STATIC int
dptintr (sc)
        DptHa_t *       sc;
{       u_char          Host_Status, Target_Status;
        vm_len_t        Residue;
        DptCcb_t *      ccb;
        DptCcb_t *      ccb_p;
        int             target, io_addr;
        Scsi_Sense_t *  sp;

        /*
         * First *check* for the interrupt, Then if it's not telling
         * about a completed operation just proceed to the next
         * adapter. We don't check the real status register because
         * this would cause us to lose the Status Packet!
         */
        if (!(inb ((io_addr = sc->ha_Base) + HA_AUX_STATUS) & HA_AUX_INTR))
                return (0);
        /*
         * If it is not busy then process the completed operation as
         * pointed to by the status packet.
         */
        ccb = sc->ha_Status_Packet->sp_vp;
#if (!defined(DPT_DEBUG_FLAGS) || !(DPT_DEBUG_FLAGS))
                if (!ccb || !(ccb->ccb_Device)) {
                        (void)inb (io_addr + HA_STATUS);
                        return (0);
                }
#else
#define DPT_DEBUG_FIXUP /* Be more lenient */
#endif
        Host_Status = sc->ha_Status_Packet->sp_HBA_stat;
        Target_Status = sc->ha_Status_Packet->sp_SCSI_stat;
        Residue = sc->ha_Status_Packet->sp_inv_residue;
        /*
         *      Just to be safe, I found the 2012 not filling in the
         *      Status Packet in some revisions of it's firmware.
         */
        sc->ha_Status_Packet->sp_vp = (DptCcb_t *)NULL;

        /*
         *      Now, clear the interrupt so the next Status could be
         * transfered to the structure now that we have cleared out the
         * information we need.
         */
        (void)inb (io_addr + HA_STATUS);

        /*
         *      Do a validity check of the CCB and the virtual HA
         */
#if (defined(DPT_DEBUG_FIXUP))  /* Mutually exclusive of above */
                /*
                 *      We are messed up, lets fix things up so that we
                 *      don't panic the Kernel and allow us to debug the
                 *      problem. Some (unreleased) versions of firmware and
                 *      drive combinations limp along with this.
                 */
                if (!ccb || !(ccb->ccb_Device)) {
                        DptCcb_t *      p;
                        DptHa_t *       q;

                        if (!sc->ha_debug)
                                return (0);
                        p = NULL;
                        for (q = sc->ha_parent; q; q = q->ha_next) {
                                for (ccb_p = q->ha_CCB; ccb_p;
                                    ccb_p = ccb_p->ccb_next) {
                                        if (ccb_p->ccb_Device) {
                                                if (p)  p = (DptCcb_t *)-1;
                                                else    p = ccb_p;
                                        }
                                }
                                for (ccb_p = q->ha_Go_CCB; ccb_p;
                                    ccb_p = ccb_p->ccb_next) {
                                        if (ccb_p->ccb_Device) {
                                                if (p)  p = (DptCcb_t *)-1;
                                                else    p = ccb_p;
                                        }
                                }
                        }
                        if (p && (p != (DptCcb_t *)-1))
                                ccb = p;
                }
                if (ccb && (ccb->ccb_Device))
#endif
#       if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS))
            if (ccb->ccb_Device)
#       endif
                sc = ((DptHa_t *)(ccb->ccb_Device));
        if (sc->ha_IO_Pending > 0)
                sc->ha_IO_Pending--;
        if ((sc->ha_parent->ha_Total_IO_Pending > 0)
         && ((--(sc->ha_parent->ha_Total_IO_Pending))
            <= (sc->ha_Config.QueueSize[1]
             + (sc->ha_Config.QueueSize[0] << 8))))
                wakeup ((caddr_t)&(sc->ha_parent->ha_Total_IO_Pending));
#if (defined(DPT_DEBUG_FIXUP))  /* Accompanies above */
                if (!ccb || !(ccb->ccb_Device))
                        return (0);
#endif

        if (((ccb->ccb_Target_Status = Target_Status) == STS_CHECKCOND)
         && (ccb->EataCp.cp_Auto_Req_Sen)
         && ((target = (int)ccb->EataCp.cp_id - (int)sc->ha_ID) >= 0)
         && (target < MAX_TARGET_ID) && (sc->ha_sense)
         && (sp = sc->ha_sense[target]))
                sp[ccb->EataCp.cp_lun] = ccb->ccb_sense;
        ccb->ccb_Host_Status = Host_Status;
        ccb->ccb_Residue = SWAPTOLM (Residue);
        if ((sc->ha_hba.hba_intr = ccb->ccb_intr)
         && (sc->ha_hba.hba_intrdev = ccb->ccb_intrdev)) {
                struct sq *     sq;

                /*
                 * For non-immediate commands,
                 * pass status back to higher levels and
                 * start the next transfer.
                 */
                (*sc->ha_hba.hba_intr)(sc->ha_hba.hba_intrdev,
                   ((Host_Status != HA_NO_ERROR) && (Target_Status == STS_GOOD))
                     ? HBAINTR_BUSRESET : Target_Status, Residue);
                IF (((sq = sc->ha_hba.hba_head) != (struct sq *)NULL)
                 && (ccb_p = dpt_get_ccb (sc, FL_NOMASK|FL_NOSLEEP,
                     (struct scsi_cdb *)NULL))) {
                        sc->ha_hba.hba_head = sq->sq_forw;
                        (*sq->sq_dgo) (sq->sq_dev,
                            (struct scsi_cdb *)(ccb_p->EataCp.cp_cdb));
                }
        }
        dpt_cleanup (sc, ccb);

        return (1);
}

/* release bus early */
void
dptrel(self)
        struct device * self;
{       DptCcb_t *      ccb;
        struct sq *     sq;
#define sc      ((DptHa_t *)self)

        if (ccb = sc->ha_Go_CCB) {
                DptBounce_t *   bo;

                ccb->ccb_intr = (scintr_fn)NULL;
                ccb->ccb_intrdev = (struct device *)NULL;
                /* Don't do anything but free resources */
                for (bo = ccb->ccb_Bounce; bo; bo = bo->bo_next)
                        bo->bo_dst = (caddr_t)NULL;
        }
        if (sc->ha_IO_Pending > 0)
                sc->ha_IO_Pending--;
        if ((sc->ha_parent->ha_Total_IO_Pending > 0)
         && ((--(sc->ha_parent->ha_Total_IO_Pending))
            <= (sc->ha_Config.QueueSize[1]
             + (sc->ha_Config.QueueSize[0] << 8))))
                wakeup ((caddr_t)&(sc->ha_parent->ha_Total_IO_Pending));
        IF ((sq = sc->ha_hba.hba_head) != (struct sq *)NULL)
                (*sq->sq_dgo)(sq->sq_dev,
                    (struct scsi_cdb *)(dpt_get_ccb (sc, FL_NOMASK,
                        (struct scsi_cdb *)NULL)->EataCp.cp_cdb));
        if (ccb)
                dpt_cleanup (sc, ccb);
#undef sc
}

/*
 *      The following is all the buffer pool allocation and management
 *      routines. This includes bounce buffers and Software CCBs.
 */
/*
 *      Called to allocate additional CCB resources
 *      By allocating resources in page size chunks, we can save memory and
 *      physical to virtual mapping resources.
 */
STATIC DptCcb_t *       dpt_ccb_free;           /* EISA/PCI based CCBs     */
STATIC DptCcb_t *       dpt_ccb_free_isa;       /* ISA/EISA/PCI based CCBs */

STATIC void
dpt_init_ccb (sc)
        DptHa_t *       sc;
{       DptCcb_t *      ccb;
        int             i, n, residual;
#define         MAX_CCB_PG      (NBPG / sizeof(DptCcb_t))
        static u_char   dpt_ccb_isa_error;
        static u_short  dpt_ccb_num_isa;        /* Actual number of ISA */
        static u_short  dpt_ccb_total_num_isa;  /* Required number of ISA */

        /*
         *      Always allocate one extra CCB so that we can start the
         *      initialization of a CCB while the HBA is busy processing the
         *      Maximal CCBs it is capable of dealing with.
         */
        n = sc->ha_Config.QueueSize[1] + (sc->ha_Config.QueueSize[0] << 8) + 1;
        for (residual = i = 0; i < n; i++) {
                if (residual > 1) {
                        ++ccb;
                        --residual;
                } else {if ((residual = n - i) > MAX_CCB_PG)
                                residual = MAX_CCB_PG;
                        if (!(ccb = malloc (residual * sizeof(DptCcb_t),
                            M_DEVBUF, M_WAITOK))) {
                                printf ("%s: ccb struct cannot malloc!\n",
                                    sc->ha_hba.hba_dev.dv_xname);
                                break;
                        }
                        (void)bzero ((char *)ccb, residual * sizeof(DptCcb_t));
                }
                if (is_isa (sc))
                        ++dpt_ccb_total_num_isa;
                ccb->EataCp.cp_vp = (caddr_t)ccb;
                ccb->EataCp.cp_Req_Len = sizeof(Scsi_Sense_t);
                ccb->EataCp.cp_Auto_Req_Sen = 1;
                ccb->EataCp.cp_msg_identify = 1;
                ccb->EataCp.cp_reqDMA = VMTOSWAP(KVTOPHYS(
                    (caddr_t)&(ccb->ccb_sense)));
                ccb->ccb_SG_addr = VMTOSWAP(KVTOPHYS((caddr_t)(ccb->ccb_SG)));
                ccb->ccb_addr = KVTOPHYS((caddr_t)ccb);
                if (ccb->ccb_addr < ISA_ADDR) {
                        ccb->ccb_isa = 1;
                        if ((++dpt_ccb_num_isa >= dpt_ccb_total_num_isa)
                         && (dpt_ccb_isa_error)) {
                                printf (
                                  "%s: ccb struct has managed to malloc ISA!\n",
                                  sc->ha_hba.hba_dev.dv_xname);
                                dpt_ccb_isa_error = 0;
                        }
                } else {if ((dpt_ccb_num_isa < dpt_ccb_total_num_isa)
                         && (dpt_ccb_isa_error == 0)) {
                                printf ("%s: ccb struct cannot malloc ISA!\n",
                                  sc->ha_hba.hba_dev.dv_xname);
                                ++dpt_ccb_isa_error;
                        }
                }
                dpt_free_ccb (sc, 0, ccb);
        }
}

/*
 *      Get a free ccb, and do some initializations.
 */
STATIC DptCcb_t *
dpt_get_ccb (sc, flags, cdb)
        DptHa_t *               sc;
        int                     flags;
        struct scsi_cdb *       cdb;
{       unsigned                s;
        DptCcb_t *              ccb;
        DptCcb_t *              ccb_p;
        int                     i;

        if (!(flags & FL_NOMASK))
                s = splbio();
        /*
         *      If we can and have to, sleep waiting for one to come free.
         */
        i = is_isa (sc);
        while ((i || (!(ccb = dpt_ccb_free)))
         && (!(ccb = dpt_ccb_free_isa)) && (!(flags & FL_NOSLEEP)))
                tsleep (i
                    ? (caddr_t)&dpt_ccb_free_isa
                    : (caddr_t)&dpt_ccb_free, PRIBIO, "dpt_get_ccb", 0);
        if (ccb == (DptCcb_t *)NULL) {
                if (!(flags & FL_NOMASK))
                        splx(s);
                return (ccb);
        }
        if (ccb->ccb_isa)
                dpt_ccb_free_isa = ccb->ccb_next;
        else    dpt_ccb_free = ccb->ccb_next;
        ccb->ccb_Device = (void *)sc;
        if (!(flags & FL_NOMASK))
                splx(s);
        ccb->ccb_stamp = time.tv_sec + DPT_WEDGETIME;
        if (cdb) {
                if (((i = SCSICMDLEN (cdb->cdb_bytes[0]))
                    > sizeof (ccb->EataCp.cp_cdb))
                 || (i == 0))
                        i = sizeof(ccb->EataCp.cp_cdb);
                (void)bcopy(cdb, ccb->EataCp.cp_cdb, i);
                ccb->EataCp.cp_msg_lun = ccb->EataCp.cp_lun;
        }
        ccb->EataCp.cp_Auto_Req_Sen = 0;
        ccb->EataCp.cp_Interpret = 0;
        ccb->EataCp.cp_msg_disco_rico = 1;
        /*
         *      Add to HBA resource list of pending CCBs (in order of arrival)
         */
        if (ccb_p = sc->ha_Go_CCB) {
                while (ccb_p->ccb_next)
                        ccb_p = ccb_p->ccb_next;
                ccb_p->ccb_next = ccb;
        } else  sc->ha_Go_CCB = ccb;
        ccb->ccb_next = (DptCcb_t *)NULL;
        return (ccb);
}

STATIC void
_dpt_free_ccb (flags, ccb)
        int             flags;
        DptCcb_t *      ccb;
{       unsigned        s;

        if (!(flags & FL_NOMASK))
                s = splbio ();
#if (defined(DPT_DEBUG_FLAGS) && (DPT_DEBUG_FLAGS & DPT_DEBUG_CHECK))
                /*
                 *      We are freeing up a CCB twice! If we are, then lets
                 * check if it was in fact freed up on the list. If not, lets
                 * just assume that the ccb_Device entry has been damaged and
                 * perform some recovery to allow the OS to continue.
                 */
                if (!(ccb->ccb_Device)) {
                        DptCcb_t *      ccb_p;

                        printf ("dpt: CCB %x over-freed\n");
                        for (ccb_p = (ccb->ccb_isa)
                            ? dpt_ccb_free_isa : dpt_ccb_free;
                            ccb_p; ccb_p = ccb_p->ccb_next) {
                                if (ccb_p == ccb) {
                                        if (!(flags & FL_NOMASK))
                                                splx(s);
                                        return;
                                }
                        }
                }
#endif
        ccb->ccb_Device = (void *)NULL;
        if (ccb->ccb_isa) {
                ccb->ccb_next = dpt_ccb_free_isa;
                dpt_ccb_free_isa = ccb;
        } else {
                ccb->ccb_next = dpt_ccb_free;
                dpt_ccb_free = ccb;
        }
        /*
         *      If there were none, wake anybody waiting for one to come free,
         * starting with queued entries.
         */
        if (ccb->ccb_next == (DptCcb_t *)NULL) {
                if (ccb->ccb_isa)
                        wakeup((caddr_t)&dpt_ccb_free_isa);
                /* Wake up regardless if ISA or not */
                wakeup((caddr_t)&dpt_ccb_free);
        }
        if (!(flags & FL_NOMASK))
                splx(s);
}

/*
 *      Called to allocate additional Bounce resources
 *      By allocating resources in NBPG chunks, we can save some memory and
 *      physical to virtual memory mapping overhead.
 */
STATIC DptBounce_t *    dpt_bounce_free;        /* ISA based bounce buffers */

STATIC void
dpt_init_bounce (sc)
        DptHa_t *       sc;
{       DptBounce_t *   bounce;
        int             i, n, residual, pass;
#define         MAX_BOUNCE_PG   (NBPG / sizeof(DptBounce_t))

        /* For ISA cards, we could starve for resources in extreme cases */
        n = sc->ha_Config.SG_Size[1] + (sc->ha_Config.SG_Size[0] << 8);
        for (residual = i = 0; i < n; i++) {
                if (residual > 1) {
                        --residual;
                        ++bounce;
                        ++pass;
                } else {if ((residual = n - i) > MAX_BOUNCE_PG)
                                residual = MAX_BOUNCE_PG;
                        if (!(bounce = malloc (residual * sizeof(DptBounce_t),
                            M_DEVBUF, M_WAITOK))) {
                                printf ("%s: bounce struct cannot malloc!\n",
                                    sc->ha_hba.hba_dev.dv_xname);
                                break;
                        }
                        bzero ((caddr_t)bounce, residual * sizeof(DptBounce_t));
                        pass = 0;
                }
                if (!(bounce->bo_v = malloc (NBPG, M_DEVBUF, M_WAITOK))) {
                        if (pass == 0)
                                free (bounce, M_DEVBUF);
                        printf ("%s: bounce buffer cannot malloc!\n",
                            sc->ha_hba.hba_dev.dv_xname);
                        break;
                }
                if ((bounce->bo_p = KVTOPHYS(bounce->bo_v)) > ISA_ADDR) {
                        free (bounce->bo_v, M_DEVBUF);
                        if (pass == 0)
                                free (bounce, M_DEVBUF);
                        printf ("%s: bounce buffer cannot malloc ISA!\n",
                            sc->ha_hba.hba_dev.dv_xname);
                        break;
                }
                dpt_free_bounce (sc, 0, bounce);
        }
#undef MAX_BOUNCE_PG
}

/*
 *      Get a free bounce, and do some initializations.
 */
STATIC DptBounce_t *
dpt_get_bounce (sc, flags)
        DptHa_t *       sc;
        int             flags;
{       unsigned        s;
        DptBounce_t *   bounce;

        if (!(flags & FL_NOMASK))
                s = splbio();
        /*
         *      If we can and have to, sleep waiting for one to come free.
         */
        while ((!(bounce = dpt_bounce_free)) && (!(flags & FL_NOSLEEP)))
                tsleep((caddr_t)&dpt_bounce_free, PRIBIO, "dptbounce", 0);
        if (bounce)
                dpt_bounce_free = bounce->bo_next;
        if (!(flags & FL_NOMASK))
                splx(s);
        return (bounce);
}

/*
 *      Name: dpt_free_bounce
 *      Description: Free up the bounce buffer, but to make the chances of this
 *      buffer being received sequentially, we place them back in physical
 *      address order.
 */
void
_dpt_free_bounce (flags, bounce)
        int             flags;
        DptBounce_t *   bounce;
{       unsigned        s;
        DptBounce_t *   this;
        DptBounce_t *   last = (DptBounce_t *)NULL;

        if (!(flags & FL_NOMASK))
                s = splbio ();
        /*
         *      If there were none, wake anybody waiting for one to come free,
         * starting with queued entries, or search to sort the entry into the
         * list of bounce buffers.
         */
        if ((this = dpt_bounce_free) == (DptBounce_t *)NULL)
                wakeup((caddr_t)&dpt_bounce_free);
        else for (;this && (bounce->bo_p < this->bo_p);
                last = this, this = this->bo_next);
        bounce->bo_next = this;
        if (last)
                last->bo_next = bounce;
        else    dpt_bounce_free = bounce;
        /*
         *      If there were none, wake anybody waiting for one to come free,
         * starting with queued entries.
         */
        if (bounce->bo_next == (DptBounce_t *)NULL)
                wakeup((caddr_t)&dpt_bounce_free);
        if (!(flags & FL_NOMASK))
                splx(s);
}

#include        <i386/isa/dptioctl.h>
