/* yacc grammar for Turtle Command Language, a simple
 functional language for telling Sam the turtle what 
 to do.     cbd 12/11/86  */
%start script

/* define token values for the primitives */
%token TCL_L     257
%token TCL_R     258
%token TCL_F     259
%token TCL_B     260

%token TCL_UP    261
%token TCL_DOWN  262
%token TCL_HOME  263
%token TCL_HIDE  264
%token TCL_SHOW  265

/* statement keywords */
%token TCL_DO     266
%token TCL_REPEAT 267
%token TCL_ASSIGN 268
%token TCL_TO     280
%token TCL_END    281      
%token TCL_LOCALS 275

/* the various token types returned by the lexical analyzer:
        TCL_FUNC = name of function (eg. poly, spiral, etc)
        TCL_ARG  = a function argument or local (eg, :size, :times, etc)
        TCL_KEYWORD = a keyword in a function call (eg. size: )
	TCL_NUMBER = a numeric literal (currently just integers)
         */

%token TCL_FUNC         270
%token TCL_ARG          271
%token TCL_KEYWORD      273
%token TCL_NUMBER	274

%%
/* first, define the primitive operations in the language - the 
 various commands to move Sam the turtle */

prim    : TCL_L  value           { doPrimVal }
        | TCL_R  value           { doPrimVal }
        | TCL_F  value           { doPrimVal }
        | TCL_B  value           { doPrimVal }
        | TCL_UP                 { doPrim }
        | TCL_DOWN               { doPrim }
        | TCL_HOME               { doPrim }
        | TCL_HIDE               { doPrim }
        | TCL_SHOW               { doPrim }
        ;

/* calls to user-defined functions.  These are of the form:
        func [ value  keyWord: value keyWord: value ... ]
        where everything between the braces is optional   */

fcall   : TCL_FUNC               { doPrim }
        | TCL_FUNC value keyArgs { doFcall }  
        | TCL_FUNC value         { doPrimVal }
        ;

keyArgs : TCL_KEYWORD value      { doKeyArg }
        | keyArgs TCL_KEYWORD value  { doKeyList }
        ; 

/* only locals can be assigned to */
assgn   : TCL_ARG TCL_ASSIGN value  { doAssgn }
        ;

/* a simple do-loop  */
do      : TCL_DO value stmtList TCL_REPEAT  { doDo }
        ;

/* infix arithmetic */
infix   : value '+' value            { doInfix }
        | value '-' value            { doInfix }
        | value '*' value            { doInfix }
        | value '/' value            { doInfix }
        ;

/* statements */
stmt    : do                         { pass }
        | value                      { pass }
        ;

stmtList : /* nothing */	     { doNoArgs }
        | stmt                       { doStmt }
        | stmtList ';'               { pass }          
        | stmtList ';' stmt          { doStmtList }
        ;

/* define the constructs in TCL that produce a value */
value   : prim                  { pass }
        | fcall                 { pass }
        | TCL_ARG               { pass }
        | TCL_NUMBER            { pass }
        | infix                 { pass }
        | assgn                 { pass }
	| '(' value ')'		{ doParen }
        ; 

/* formal arguments for a function definition */
keyParms : TCL_KEYWORD TCL_ARG      { doKeyArg }
        | keyParms TCL_KEYWORD TCL_ARG  { doKeyList }
        ;
 
argList : /*  nothing */        { doNoArgs }
        | TCL_ARG               { doArg }
        | TCL_ARG keyParms      { doArgList }

locs    : TCL_ARG 		{ doStmt }
	| locs TCL_ARG 		{ doLocList }
	;

locDef  : /* nothing */		{ doNoArgs }
	| TCL_LOCALS locs ';'	{ doLocDef }
	;

/* function definition: 
        TO poly :sides width: :w
	LOCALS :x :y;
                %1 := 360/:sides;
                do :sides
                        f :sides;
                        r %1;
                repeat
        END       */

def     : TCL_TO TCL_FUNC argList locDef stmtList TCL_END   { doDef }
        ;

/* finally, we will present the parser with either a list of
 statements or a function definition. */
script  : def                   { doScriptDef }
        | stmtList              { doScriptSlist }
        ;
                                                                 