/* A Dictionary object is a keyed collection whose elements
 are Association objects.  A Dictionary can have any kind of
 object as a key, because its keys are looked up on the basis
 of equality rather than equivalence.

 For example, you could have a Dictionary object called Countries
 where each key is a nation and each value is the capital of
 the nation.  Then, you could say Countries["France"], which
 would refer to the value "Paris".  In this example, the key
 is "France", and the value is "Paris".  The key and value are
 kept together in an Association object. */ !!

/* inherit(KeyedCollection, #Dictionary, nil, 2, 1)  */!!

now(class(Dictionary)) !!

now(Dictionary) !!

/* Enumerate over the elements in the
  Dictionary.  */
Def  do(self, aBlock)
{       ^do( new(Interval, 0, limit(self),
  1),
  { using(x | assoc)
    if (assoc := at(self:Object, x))
    then  eval(aBlock, assoc.value)
    else  nil
    endif;
  });
}!!


/* Evaluate the one-argument block over
  the classes in the receiver Dictionary. */
 Def  classesDo(self, aBlock)
{ do(self,
  {using(elem)
    if class(class(elem)) == Meta
    then eval(aBlock, elem);
    else nil
    endif;
  });
}!!


/* Return the set of classes in the
  receiver Dictionary. */
 Def classes(self | clSet)
{  clSet := new(Set, 64);
  classesDo(self,
  {using(cl)  add(clSet, cl)
  });
  ^clSet;
}!!


/* Return a Dictionary whose keys
  are the classes in the receiver
  Dictionary, and whose elements
  are SortedCollections of the key's
  immediate descendants.  */
 Def buildClassLists(self | aDict)
{       aDict := new(Dictionary,256);
  classesDo(self,
  { using(aCL) aDict[aCL] :=
    new(SortedCollection,16);
  });
  classesDo(self,
  { using(aCL)
    if aCL <> Object
    then  add(aDict[aCL.ancestor], aCL);
    endif;
  });
  ^aDict
}!!

/* Evaluate the one argument block over
  the keys of the Dictionary.  */
 Def keysDo(self, aBlock)
{       ^keysDo( self:KeyedCollection,
  { using(assoc)  eval(aBlock, assoc.key);
  });
}!!

/* Remove the element with the specified
  key from the Dictionary.  If there is no
  element corresponding to aKey, then
  an element not found error is generated.
  The remove method returns the removed
  key.  */
Def remove(self, aKey | idx)
{
  if not( at(self:Object, idx :=
    find(self, aKey)))
  then  error(self, stackTop(0),
    #elemNotFndError);
  endif;
  putElem(self, nil, nil, idx);
  fixUp(self, idx);
  tally := tally - 1;
  ^aKey;
}!!

/* Re-hash all the elements of the receiver.
  This needs to be done after deleting an entry
  because other hash values might need to
  occupy the empty slot.  This is a generic
  fixup that works for Dict and
  MethodDictionary. */
 Def fixUp(self, idx | nextIdx, newIdx, elem, val, lim)
{ nextIdx := idx;
  lim := limit(self) - 1;
  loop nextIdx := (
    if nextIdx == lim
    then  0
    else  nextIdx + 1
    endif)
  while   (elem := at(self:Object,
    nextIdx))
  begin newIdx := find(self,
    getKey(self, elem));
    if not(at(self:Object, newIdx))
    then val := getVal(self, nextIdx);
      putElem(self, nil, nil, nextIdx);
      putElem(self, elem, val, newIdx);
    endif;
  endLoop;
}!!


/* Return the key part of an element.  (Private method) */
 Def getKey(self, elem)
{ ^elem.key
}!!

/* Return the value for a particular
  physical index of the Dictionary.  (Private method) */
 Def getVal(self, idx)
{  ^nil
}!!

/*  Store a new key/value pair
  at the specified physical index.  (Private method) */
 Def putElem(self, assoc, val, idx)
{   put(self:Object, assoc, idx);
}!!


/* Replace a current element or creates a
  new one.  The put method for this
  class is identical to the add method
  except for the order of its arguments. */
 Def put(self, anElement, aKey | assoc)
{ if (assoc := assocAt(self:Dictionary,
    aKey))
  then  assoc.value := anElement;
  else  init(assoc := new(Association),
    aKey, anElement);
    addAssoc(self:Dictionary, assoc);
  endif;
  ^anElement;
}!!


/* Evaluate the block over each of the
  receiver's Associations. */
 Def assocsDo(self, aBlock)
{ ^keysDo(self:KeyedCollection, aBlock);
}!!


/* Add a key and an element to a
  Dictionary.  For example, if you have
  a Dictionary called Sam, then the
  following message associates the element
  "Hello" with the key "Greeting":
  put(Sam, "Greeting", "Hello").  */
Def add(self, aKey, anElement )
{       ^put(self, anElement, aKey);
}!!

/* Copy elements into larger collection
  and swap with the old collection.  */
 Def grow(self | newColl)
{ newColl := new(class(self), limit(self)
  * 2);
  assocsDo(self:Dictionary,
  {using(assoc)   addAssoc(newColl,
    assoc);
  });
  swap(self, newColl);
}!!


/* Add an Association to the
  Dictionary.  */
Prim addAssoc(self, anAssoc):self!!


/* Return the Association object which
  has the specified key.  */
Prim assocAt(self, aKey):Association!!


/* Return the element associated with the
  specified key.  */
Prim at(self, aKey):DictionaryElement!!


/* Return the key residing at the
  specified physical index in the
  Dictionary.  */
Prim keyAt(self, index):aKey!!

/* Return the physical index of the 
  specified key.  */
Prim find(self, aKey):Int!!
