// Whirlwind simulator, based on Guy Fedorkow's simulator

/**
 * Top-level class for the simulator.
 * Holds other objects.
 * @constructor
 */
Sim = function() {
  this.cb = new ConstWWbitClass();
  this.sw = new WWSwitchClass();
  this.coreMem = null;
  this.cpu = new CPUClass(this);

  this.symTab = {};
  this.jumpto_addr = null;
  this.ww_file = null;
  this.ww_tapeid = null;

  // use these vars to control how much Helpful Stuff emerges from the this.sim
  this.tracePC = false;  // print a line for each instruction
  this.traceALU = false;   // print a line for add, multiply, negate, etc
  this.traceBranch = true;  // print a line for each branch
  this.traceQuiet = false;
  this.traceDisplayScope = false;


  /**
   * input for the simulation comes from a "core" file giving the contents of memory
   * Sample core-file input format, from tape-decode or wwasm
   * The image file contains symbols as well as a bit of metadata for where it came from
   * *** Core Image ***
   * @C00210: 0040000 0000100 0000001 0000100 0000000  null    null    null  ; memory load
   * @S00202: Yi                                                             ; symbol for location 202
   * %Switch: chkalarm 0o5
   */
  this.read_core = function(data) {
    this.symTab = {}
    this.coreMem = new CorememClass(this);
    var lines = data.split('\n')
    for (var lineNumber = 1; lineNumber <= lines.length; lineNumber++) {
      var line = lines[lineNumber - 1];
      if (line.length == 0) {  // skip blank lines
        continue;
      }
      var input_minus_comment = line.replace(/;.*/, '').replace(/\s+$/, '')  // strip comments and spaces
      if (input_minus_comment.length == 0) {  // skip blank lines
        continue;
      }
      if (!input_minus_comment.match(/^@C|^@S|^%[a-zA-Z]/)) {  // ignore anything that doesn't start with @C, @S, %<something>
        continue;
      }
      if (input_minus_comment.match(/^@C/)) {  // read a line of core memory contents
        var tokens = input_minus_comment.split(/[: \t]+/);
        // print "tokens:", tokens
        if (tokens.length == 0) {
          alert("parse error, read_core @C: tokens=" + tokens);
          continue;
        }
        var address = parseInt(tokens[0].substr(2), 8);
        for (var i = 1; i < tokens.length; i++) {
          var token = tokens[i];
          if (token != "None") {
            this.coreMem.wr(address, parseInt(token, 8));
          }
          address += 1;
        }
      } else if (input_minus_comment.match(/^@S/)) { // read a line with a single symbol
        var tokens = input_minus_comment.split(/[: \t]+/);
        // print "tokens:", tokens
        if (tokens.length != 2) {
          alert("parse error, read_core @S: tokens=" + tokens);
          continue;
        }
        var address = parseInt(tokens[0].substr(2), 8);
        this.symTab[address] = tokens[1];
      } else if (input_minus_comment.match(/^%Switch/)) {
        tokens = input_minus_comment.split(/ +/);
        this.sw.parse_switch_directive(tokens.slice(1));
      } else if (input_minus_comment.match(/^%JumpTo/)) {
        tokens = input_minus_comment.split(/ +/);
        this.jumpto_addr = parseInt(tokens[1], 8)
        console.log("corefile JumpTo address = 0" + toOctal(this.jumpto_addr) + "o");
      } else if (input_minus_comment.match(/^%File/)) {
        tokens = input_minus_comment.split(/ +/);
        this.ww_file = tokens[1]
        console.log("Whirlwind tape file name: " + this.ww_file);
      } else if (input_minus_comment.match(/^%TapeID/)) {
        tokens = input_minus_comment.split(/ +/);
        this.ww_tapeid = tokens[1];
        console.log("Whirlwind tape identifier: " + this.ww_tapeid);
      } else {
        alert("unexpected line '" + line + " ' in " + filename + ", Line " + lineNumber);
      }
    }
  };

};

/**
 * Represents core memory.
 * WW ended up with 6K words of memory, but a 2K word address space.  Overlays and bank
 * switching were the order of the day
 * The address space was divided into a high group and low group, each of which could
 * be mapped to one of six 1K word pages.
 * Mapping is controlled by the CF instruction
 *
 * @constructor
 */
CorememClass = function(sim) {
  // Create NBANKS arrays, each one holding CORE_SIZE words
  this.sim = sim;
  this.core = Array(this.sim.cb.NBANKS).fill(0).map(x => Array(this.sim.cb.CORE_SIZE / 2).fill(null))
  this.MemGroupA = 0;
  this.MemGroupB = 1;

  /**
   * Writes to memory
   * @param {int} addr - The address to write.
   * @param {int} val - The value to store.
   * @function
   */
  this.wr = function(addr, val) {
    if (addr & this.sim.cb.WWBIT5) { // High half of the address space, Group B
      this.core[this.MemGroupB][addr & this.sim.cb.WWBIT6_15] = val;
    } else {
      this.core[this.MemGroupA][addr & this.sim.cb.WWBIT6_15] = val;
    }
  };

  /**
   * Reads from memory
   * @param {int} addr - The address to read.
   * @function
   * @returns {int};
   */
  this.rd = function(addr) {
    if (addr & this.sim.cb.WWBIT5) { // High half of the address space, Group B
      return this.core[this.MemGroupB][addr & this.sim.cb.WWBIT6_15];
    } else {
      return this.core[this.MemGroupA][addr & this.sim.cb.WWBIT6_15];
    }
  }

};

/**
 * Represents CPU.
 * @constructor
 */
this.CPUClass = function(sim) {
  this._AReg = 0;
  this._BReg = 0;
  this._AC = 0;   // Accumulator
  this._SAM = 0;    // two-bit carry-out register; it's only legal values appear to be 1, 0, -1
           // SAM is set by many instructions, but used only by ca cs and cm
  this.PC = null;   // Program Counter
  this.ioDevice = -1;   // Device last selected by SI instruction
  this.ioDeviceClass = null;
  this.drumClass = new DrumClass(sim); // Expose this for GUI
  this.ioDeviceList = [
    new FlexoClass(sim),
    new tty_class(sim),
    this.drumClass,
    new DisplayScopeClass(sim),
    new CoreClearIoClass(sim),
  ];
  this.MemGroupA = 0;   // I think Reset sets logical Group A to point to Physical Bank 0
  this.MemGroupB = 1;  // I *think* Reset sets logical Group B to point to Physical Bank 1
  this.sim = sim;
  this.tracePC = false;
  this.debug = false;
  this.breakpoints = {
    //  address   action
    0032: this.breakp_dump_sim_state,
    //    0565: breakp_start_trace,
    };

  /**
   * Starts tracing at a breakpoint.
   */
  this.breakp_start_trace = function(cpu) {
    console.log("breakpoint @PC=" + cpu.wwaddr_to_str(cpu.PC));
    this.tracePC = true;
  };


  /**
   * convert an integer to a string, including negative number notation
   */
  this.wwint_to_str = function(num) {
    if (num == null) {
        return " null ";
    }
    var neg = "";
    if (num & this.sim.cb.WWBIT0) {
      neg = "(" + "-" + toOctal(~num & this.sim.cb.WWBIT1_15, 4) + "o)";
    }
    return toOctal(num, 6) + "o" + neg;
  }

  /**
   * convert an address to a string, adding a label from the symbol table if there is one
   */
  this.wwaddr_to_str = function(num) {
    var label = "";
    if (num in this.sim.symTab) {
      label = "(" + this.sim.symTab[num] + ")";
    }
    return "0o" + toOctal(num, 6) + label;
  }

  /**
   */
  this.print_cpu_state = function(pc, short_opcode, op_description, address) {
    if (this.tracePC) {
      var s1 = " pc:" + this.wwaddr_to_str(pc) + ":  " + short_opcode + " " + this.wwaddr_to_str(address);
      var s2 = "   AC=" + this.wwint_to_str(this._AC) + ", AR=" + this.wwint_to_str(this._AReg) + ", BR=" + this.wwint_to_str(this._BReg) + ", SAM=" + toOctal(this._SAM, 2);
      var s3 = "   nextPC=" + this.wwaddr_to_str(this.PC) + ", Core@" + this.wwaddr_to_str(address) + "=" + this.wwint_to_str(this.sim.coreMem.rd(address));
      console.log(s1 + " " + s2 + " " + s3 + " ; " + op_description);
    }
  }

  /**
   */
  this.run_cycle = function() {
    var instruction = this.sim.coreMem.rd(this.PC)
    if (instruction == null) {
      alert("\nrun_cycle: instruction is 'null' at " + toOctal(this.PC, 5) + "o" % this.PC);
      return this.sim.cb.UNIMPLEMENTED_ALARM;
    }
    var opcode = (instruction >> 11) & 037
    var address = instruction & this.sim.cb.WW_ADDR_MASK
    if (this.debug) {
      console.log("run_cycle: ProgramCounter " + toOctal(this.PC, 5) + "o, opcode " + toOctal(opcode, 3) + "o, address " + toOctal(address, 6) + "o");
    }
    var current_pc = this.PC
    this.PC += 1  // default is just the next instruction -- if it's a branch, we'll reset the PC later
    var oplist = this.op_decode[opcode]
    if (typeof oplist == "undefined" || typeof oplist[0] == "undefined") {
      alert("Bad opcode " + toOctal(opcode, 4) + " at address " + toOctal(current_pc, 4));
      console.log("opcode " + opcode);
      console.log("oplist " + oplist);
      console.log("op_decode " + this.op_decode);
      console.log("op_decode[0] " + this.op_decode[0]);
      return this.sim.cb.UNIMPLEMENTED_ALARM;
    }
    var ret = oplist[0].call(this, current_pc, address, oplist[1], oplist[2])
    if (current_pc in this.breakpoints) {
      this.breakpoints[current_pc](cpu)
    }
    return ret;
  }

  /*
   * # here's the world's clumsiest ones-complement add
   * parameters a & b are in WW Ones Complement format
   * parameter sam_in is +1, 0, or -1 in python native format
   * All this assumes that the python native int is bigger than 16 bits!

   * from M-1624-1 WW-page 3:
   * Zero. All sums and differences resulting in zero are represented as
   * negative zero (1.111 111 111 111 111) except in the two cases: (+0) + (+0)
   * and (+0) - (-0). The sign of a zero resulting from multiplication, division,
   * or shifting is in accordance with the usual sign convention.
   */

  this.ones_comp_add16 = function(num1, num2, carry_in) {
    // add ones-complement, return the sum and an overflow bit 
    var MOD = this.sim.cb.WWBIT0 << 1
    if (carry_in) {
      alert("ones_comp_add: check carry_in handling");
    }
    var result = num1 + num2 + carry_in
    if (result < MOD) {
      return[result, 0];
    } else {
      return [(result + 1) % MOD, 1];
    }
  };

  this.ww_add = function(a, b, sam_in) {
    if ((a == null) | (b == null)) {
        console.log("'null' Operand, a=", a, " b=", b);
        return [0, 0, this.sim.cb.READ_BEFORE_WRITE_ALARM];
    }

    var r = this.ones_comp_add16(a, b, sam_in);
    var sum = r[0];
    var carry_out = r[1];

    var sam_out = 0;   // the default is "no overflow" and "no alarm"
    var alarm = this.sim.cb.NO_ALARM;
    // check for positive or negative overflow.  Since we're adding two 15-bit numbers, it can't overflow
    // by more than one bit (even with the carry-in, I think:-))
    if ((a & this.sim.cb.WWBIT0) && (b & this.sim.cb.WWBIT0) && !(sum & this.sim.cb.WWBIT0)) {
        sam_out = -1;
        alarm = this.sim.cb.OVERFLOW_ALARM;
    }
    if (!(a & this.sim.cb.WWBIT0) && !(b & this.sim.cb.WWBIT0) && (sum & this.sim.cb.WWBIT0)) {
        sam_out = -1
        alarm = this.sim.cb.OVERFLOW_ALARM
    }

    if (this.traceALU) {
      console.log("ww_add: WWVals: a=" + this.wwint_to_str(a) + 
          ", b=" + this.wwint_to_str(b) +
          ", sam_in=" + toOctal(sam_in) +
          ", sum=" + this.wwint_to_str(sum) +
          ", sam_out=" + toOctal(sam_out) +
          ", alarm=" + toOctal(alarm));
    }

    return [sum, sam_out, alarm];
  }

  /**
   * basic negation function for ones-complement
   */
  this.ww_negate = function(a) {
    // ones complement negation 
    var neg_a = a ^ this.sim.cb.WWBIT0_15;
    if (this.traceALU) {
      console.log("ww_negate: a=" + toOctal(a) + "  neg_a=" + toOctal(neg_a));
    }
    return neg_a;
  }

  /**
   * basic multiplication for ones-complement
   * Assuming the Python Int is 30 bits or longer...
   */
  this.ww_multiply = function(a, b) {

    var tc_a = null;
    var tc_b = null;
    // convert ones-complement numbers to native twos complement
    if (a & this.sim.cb.WWBIT0) {   // sign bit == 1 means Negative
      tc_a = -(this.sim.cb.WWBIT0_15 ^ a);
    } else {
      tc_a = a & this.sim.cb.WWBIT1_15;
    }

    if (b & this.sim.cb.WWBIT0) {   // sign bit == 1 means Negative
      tc_b = -(this.sim.cb.WWBIT0_15 ^ b);
    } else {
      tc_b = b & this.sim.cb.WWBIT1_15;
    }

    var tc_product = tc_a * tc_b;
    var product = null;

    // convert to ones-complement
    if (tc_product < 0) {
      product = -tc_product; // convert to positive
      product = product ^ this.sim.cb.pyBIT31_0;  // convert to ones-complement
    } else {
      product = tc_product
    }

    // The 30-bit result is taken apart into two 16-bit registers.  Reg_A is the most-significant
    // part with the sign bit, Reg_B is the least significant, with WW Bit 15 unused and zero (I think)
    var reg_b = (product & this.sim.cb.WWBIT0_15)
    var reg_a = (product >> 16) & this.sim.cb.WWBIT0_15

    if (this.traceALU) {
      console.log("ww_multiply: tc_a=" + toOctal(tc_a) + ", tc_b=" + toOctal(tc_b) + 
          ", tc_product=" + toOctal(tc_product));
      console.log("ww_multiply: a=" + this.wwint_to_str(a) +
          ", b=" + this.wwint_to_str(b) +
          ", reg_a=" + this.wwint_to_str(reg_a) +
          ", reg_b=" + this.wwint_to_str(reg_b));
    }

    return [reg_a, reg_b, this.sim.cb.NO_ALARM];
  }

  /**
   * from "Whirlwind_Training_Program_Material.pdf", M_1624-1, dated November 28, 1952
   * File Whirlwind_Training_Program_Material.pdf  page ~8
   */
  this.si_inst = function(pc, address, opcode, op_description) {  // select I/O Device
    if ((address == 0) | (address == 1)) {
      console.log("Halt!  (Code=" + toOctal(address) + ")");
      return this.sim.cb.HALT_ALARM;
    }

    // SI 010(o) seems to clear the Flip Flop Register Set
    // not sure yet how to tell where they are all the time...
    if (address == 010) {
      console.log("Clear FF Registers (currently unimplemented)");
      return this.sim.cb.NO_ALARM;
    }

    this.ioDeviceClass = null;  // forget whatever the last device was
    // scan the table of devices to see if any device matches; if so, run the si initialization for the device
    for (var i = 0; i < this.ioDeviceList.length; i++) {
      var dev = this.ioDeviceList[i];
      var cl = dev.is_this_for_me(address);
      if (cl != null) {
        if (this.ioDeviceClass != null) {  // we shouldn't match more than one IO address;
                          // if we get a match here , there must be an in correct table entry
          console.log("overlapping IO address 0o", toOctal(address));
          return this.sim.cb.UNKNOWN_IO_DEVICE_ALARM;
        }
        this.ioDevice = address;
        this.ioDeviceClass = cl;
      }
    }
    if (this.ioDeviceClass == null) {
      console.log("unknown IO address 0o", toOctal(address));
      return this.sim.cb.UNKNOWN_IO_DEVICE_ALARM;
    }
    var ret = this.ioDeviceClass.si(address, this._AC);
    this.print_cpu_state(pc, opcode, op_description, address);
    return ret;
  }

  /**
   * caution!  the RC instruction increments the PC by one for a normal operation,
   * but increments it by two if (the device is busy
   * expect to see something like this:
   *     si  0402;  select  input;
   * L057: rc  0000;  record;  ## print #171740o
   *     sp  L062;  sub - program;  branch  L0062, continue program
   *     sp  L057;  sub - program;  failure - branch back to retry the RC
   * In simulation, I haven't found a case to trigger a retry, so that code
   * will remain unexercised.
   */
  this.rc_inst = function(pc, address, opcode, op_description) {
    if (this.ioDeviceClass == null) {
      console.log("unknown I/O device " + toOctal(this.ioDevice) + "o");
      return this.sim.cb.UNKNOWN_IO_DEVICE_ALARM;
    }

    var r = this.ioDeviceClass.rc(this.sim.coreMem.rd(address), this._AC);
    var alarm = r[0];
    var symbol = r[1];

    this.print_cpu_state(pc, opcode, op_description, address);
    if (!this.sim.traceQuiet) {
      console.log("RC: Record to Device" + this.ioDeviceClass.name +
          ": Dev=" + toOctal(this.ioDevice) +
          "o, Output Word=" + toOctal(this._AC, 6) +
          "o, (" + symbol +
          ") Sub-Address " + toOctal(address, 5) + " o");
    }
    return alarm;
  }

  /**
   * I/O Read instruction - fetch the contents of IOR into the accumulator
   * The function of this instruction depends entirely on the previous SI.  Read the book.  = function(2M-0277)
   */
  this.rd_inst = function(pc, address, opcode, op_description) {
    if (this.ioDeviceClass == null) {
      console.log("unknown I/O device " + toOctal(this.ioDevice) + "o");
      return this.sim.cb.UNKNOWN_IO_DEVICE_ALARM;
    }
    var r = this.ioDeviceClass.rd(this.sim.coreMem.rd(address), this._AC);
    var ret = r[0];
    var acc = r[1];
    this._AC = acc;
    this.print_cpu_state(pc, opcode, op_description, address);
    return ret;
  }

  this.bi_inst = function(pc, address, opcode, op_description) {
    if (this.ioDeviceClass == null) {
      console.log("unknown I/O device " + toOctal(this.ioDevice) + "o");
      return this.sim.cb.UNKNOWN_IO_DEVICE_ALARM;
    }

    var ret = this.ioDeviceClass.bi(address, this._AC, this.sim.coreMem);
    this._AC = this._AC + address;
    this._AR = address;
    this.print_cpu_state(pc, opcode, op_description, address);
    return ret;
  }

  this.bo_inst = function(pc, address, opcode, op_description) {
    if (this.ioDeviceClass == null) {
      console.log("unknown I/O device " + toOctal(this.ioDevice) + "o");
      return this.sim.cb.UNKNOWN_IO_DEVICE_ALARM;
    }

    var ret = this.ioDeviceClass.bo(address, this._AC, this.sim.coreMem);
    this._AC = this._AC + address;
    this._AR = address;
    this.print_cpu_state(pc, opcode, op_description, address);
    return ret;
  }


  /**
   * ts x transfer to storage  #8 01000 86 microsec
   * Transfer contents of AO to register %0 The original contents
   * of x is destroyed.
   */
  this.ts_inst = function(pc, address, opcode, op_description) {
    this.sim.coreMem.wr(address, this._AC);
    this.print_cpu_state(pc, opcode, op_description, address);
    return this.sim.cb.NO_ALARM;
  }

  /**
   * td x transfer digits #9 01001 86 microseo
   * transfer last 11 digits of AC to last 11 digit positions of
   * register x. The original contents of the last 11 digit positions
   * of register x is destroyed.
   */
  this.td_inst = function(pc, address, opcode, op_description) {
    var mask = this.sim.cb.WW_ADDR_MASK;
    var m = this.sim.coreMem.rd(address);
    this.sim.coreMem.wr(address, m & (this.sim.cb.WWBIT0_15 & ~mask) | (mask & this._AC));
    if (this.traceALU) {
      console.log("td_inst  AC=" +
          toOctal(this._AC) +
          "o, oldCore=" + toOctal(m) +
          "o, newCore=" + toOctal(this.sim.coreMem.rd(address)) + "o");
    }
    this.print_cpu_state(pc, opcode, op_description, address);
    return this.sim.cb.NO_ALARM;
  }

  /**
   * ta X transfer address  01010  29 microsec
   * Transfer last 11 digits of AR to last 11 digit positions of register x.
   * The original contents of the last 11 digit positions of register x are
   * destroyed. The ta operation is normally executed after an sp or cp
   * instruction in connection with sub-programing; less frequently after ao,
   * sf or other operations.
   */
  this.ta_inst = function(pc, address, opcode, op_description) {
    var mask = this.sim.cb.WW_ADDR_MASK;
    var m = this.sim.coreMem.rd(address);
    this.sim.coreMem.wr(address, m & (this.sim.cb.WWBIT0_15 & ~mask) | (mask & this._AReg));
    if (this.traceALU) {
      console.log("ta_inst  AR=" + toOctal(this._AReg) +
          "o, oldCore=" + toOctal(m) +
          "o, newCore=" + toOctal(this.sim.coreMem.rd(address)) + "o");
    }
    this.print_cpu_state(pc, opcode, op_description, address);
    return this.sim.cb.NO_ALARM;
  }


  this.ww_branch = function(pc, address, opcode, op_description) {
    if (address >= this.sim.cb.CORE_SIZE) {
      alert("branch: looks like a bad branch pointer to " + toOctal(address) +
          "o at pc=" + toOctal(pc) + "o");
      exit();
    }
    if (!this.sim.traceQuiet) {
      var dir = null;
      if (address < this.PC) {
        dir = "backwards";
      } else {
        dir = "forwards";
      }
      if (this.traceBranch) {
        console.log("branch " + dir +
            " from pc=" + this.wwaddr_to_str(this.PC - 1) +
            " to " + this.wwaddr_to_str(address));
      }
    }

    // save the current PC+1 as a return address;
    this._AReg = this.PC;  // the PC was incremented in the calling routine (I think)
    this.PC = address;
    return this.sim.cb.NO_ALARM;
  }


  /**
   * cp x  conditiona1 program #14d  16o   01110   30 miorosec
   * If the number in AC is negative, proceed as in sp. If number in AC
   * 1s positive proceed to next instruction, but clear the AR

   * 2M-0277 says something different about AR in case of a positive result
   * "...and place in the last 11
   * digit positions of the AR the address of this next instruction."
   */

  this.cp_inst = function(pc, address, opcode, op_description) {
    if (this._AC & this.sim.cb.WWBIT0) {  // if AC is negative
      this.ww_branch(pc, address, opcode, op_description);
    } else {
      this._AReg = this.PC;
    }

    this.print_cpu_state(pc, opcode, op_description, address);
    return this.sim.cb.NO_ALARM;
  }

  /**
   * sp x  subprogram   #15d  17o    01111   30 microseo
   * Take next instruction from register x. If the sp instruction was
   * at address y, store y + 1 in last 11 digit positions of AR.  All of the
   * original contents 0 AR is lost.
   */
  this.sp_inst = function(pc, address, opcode, op_description) {
    this.ww_branch(pc, address, opcode, op_description);
    this.print_cpu_state(pc, opcode, op_description, address);
    return this.sim.cb.NO_ALARM;
  }

  /**
   * ck X   check   01011  o13  #11d   22 microsec
   * Operates in two modes depending upon position of console switch (at T.C.)
   * labelled "Pogram Check Alarm on Special Mode." The NORMAL MODE is
   * selected if (the switch is off or down. Normal Mode compares contents of
   * AC with contents of register x. If contents of AC are identical to cotents
   * of register x, proceed to next instruction; othrwise, stop the computer
   * and give a "check register alarm" (Note: +0 is not identical to
   * -0.) SPECIAL MODE ia chosen if (switch is on. Special Mode operates in
   * same way as above if (the numbers being checked agree. If there is disagreement,
   * no check alarm will occur but the program counter (PC) will be
   * indexed by one, causing the next instruction to be skipped.
   */
  this.ck_inst = function(pc, address, opcode, op_description) {
    var ret = this.sim.cb.NO_ALARM;
    var m = this.sim.coreMem.rd(address);
    if (m == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    if ((m != this._AC)) {
      if (this.sim.sw.read_switch("CheckAlarmSpecial") == 0) {
        ret = this.sim.cb.CHECK_ALARM;
      } else {
        // if there's a check and the Program Check Alarm switch is set to Special, then
        // instead of doing a trap, we double-increment the PC, i.e., skip the non-trap
        // instruction and do the one following.
        this.PC += 1;
        if (!this.sim.traceQuiet) {
          console.log("Check Special Instruction going to addr=0o" + toOctal(this.PC) + "o");
        }
      }
    }
    this.print_cpu_state(pc, opcode, op_description, address)
    return ret;
  }

  /**
   * ca x clear and add #16  10000 48 microsec ca
   * Clear AC and BR, then obtain oontents of SAM (+1, 0, or -1) times 2-15
   * and add contents of register x, storing result in AC. The oontents of
   * register x appears in AR. SAM is c1eared. Overflow may occur, giving an
   * arithmetic check a1arm
   */
  this.ca_inst = function(pc, address, opcode, op_description) {

    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    var r = this.ww_add (0, operand, this._SAM);
    var sum = r[0];
    var alarm = r[2];

    this._AC = sum;
    this._AReg = operand;
    this._BReg = 0;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }

  /**
   * Operation  Function     Number  Binary   Time
   * cs x     Clear and subtract #021o  #17d   10001  48 microsec
   * Clear AC and BR, then obtain contents of SAM (+1, 0, or -l)
   * times 2**-15 and subtract contents of register x, atortng result in
   * AC.  The contents of register x appears in AR.  SAM is oleared.
   * Overflow may occur, giving an arithmetic check alarm.
   */
  this.cs_inst = function(pc, address, opcode, op_description) {

    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    var r = this.ww_add (0, this.ww_negate(operand), this._SAM);
    var sum = r[0];
    var alarm = r[2];

    this._AC = sum;
    this._AReg = operand;
    this._BReg = 0;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }

  /**
   * ad x   add    #022o  #18d   10010  48 micros8c
   * Add the oontents of register x to oontents of AC, storing result
   * in AC. The contents of register x appears in AR. SAM is oleared
   * Overflow may oocur, giving an arithmetic cheok alarm.
   */
  this.ad_inst = function(pc, address, opcode, op_description) {
    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    var r = this.ww_add (this._AC, operand, 0);
    var sum = r[0];
    var alarm = r[2];

    this._AC = sum;
    this._AReg = operand;
    this._BReg = 0;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }

  /**
   * ao x add one #22 (26o)  10110 86 microsec
   *  Add the number 1 t1mes 2**-15 to contents of register x, storing
   * the result in AC and in register x. The original oontents of register x
   * appears in AR. SAM is cleared. Overflow may ooour, giving an arithmetic
   * oheck alarm.
   */
  this.ao_inst = function(pc, address, opcode, op_description) {
    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    var r = this.ww_add (1, operand, 0);
    var sum = r[0];
    this._SAM = r[1];
    var alarm = r[2];

    this._AC = sum;
    this.sim.coreMem.wr(address, sum);  // this seems to be the only case where the answer is written back to core?
    this._AReg = operand;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }

  /**
   * su x   subtract  #023o   #19d   10011   48   mioroseo
   * Subtract contents of register x from contents of AC, storing
   * result in AC.  The contents of register x appears in AR.  SAM is
   * cleared. Overflow may ooour, giving an arithmetic oheck alarm.
   */
  this.su_inst = function(pc, address, opcode, op_description) {
    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    var r = this.ww_add (this._AC, this.ww_negate(operand), 0);
    var sum = r[0];
    this._SAM = r[1];
    var alarm = r[2];

    this._AC = sum;
    this._AReg = operand;
    this._BReg = 0;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }

  /**
   * ex x exchange  01101  o15  d13  29 usec
   * Exchange contents of AC with contents of register x. (Original contents
   * of AC in register x; original contents of register x in AC and AR.) Ex 0
   * will clear AC without clearing BR.
   */
  this.ex_inst = function(pc, address, opcode, op_description) {

    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    this.sim.coreMem.wr(address, this._AC);

    this._AC = operand;
    this._AReg = operand;
    this._BReg = 0;
    this._SAM = 0;

    var alarm = this.sim.cb.NO_ALARM;
    if ((address == 0) & (this._AC != 0)) {
      console.log("EX instruction @0o" + toOctal(pc- 1) + ": the instruction book implies @0 should be zero, but it's 0o" + toOctal(this._AC));
    }

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }


  /**
   * dm x  Difference of Magnitudes  #027o   #23d  22 microsec
   * Subtract the magnitude of contents of register x from the magnitude of
   * contents of AC, leaving the result in AC.  The magnitude of contents of
   * register x appears in AR.  SAM is cleared.  BR will contain the initial
   * contents of the AC.  If |C(AC) = |C(x)|, the result is -0
   *
   * guy says: I think this instruction was added later in the process...  it's
   * in the 1958 manual, and M-1624, but not R-196 1951.
   */
  this.dm_inst = function(pc, address, opcode, op_description) {
    // to find difference of magnitude, make the first arg positive, the second
    // negative, and add them
    // I'm assuming this ignores SAM carry-in
    var ac = this._AC;
    if (ac & this.sim.cb.WWBIT0) {
      ac = this.ww_negate(ac);
    }
    var x = this.sim.coreMem.rd(address);
    if (x == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    if ((x & this.sim.cb.WWBIT0) == 0) {  // if x is positive, make it negative
      x = this.ww_negate(x);
    }

    var r = this.ww_add (ac, x, 0);
    var sum = r[0];
    this._SAM = r[1];
    var alarm = r[2];

    this._BReg = this._AC;
    this._AC = sum;
    this._AReg = this.ww_negate(x);  // we just made 'x' Negative above, this makes it positive
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }

  /**
   * cm x  clear and add magnitude #20d   #024o  10100   48 microsec
   * Clear AC and BR, then obtain contents of SIM (+1, 0, -1)
   * times 2-15 and add magnitude of contents of register x, storing
   * result in AC. The magnitude of the contents of register x appears
   * in AR.   SAM is cleared. Overflow may occur, giving an arithmetic
   * check alarm.
   */
  this.cm_inst = function(pc, address, opcode, op_description) {

    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    operand &= this.sim.cb.WWBIT1_15;   // mask off the sign bit TODO: negation doesn't work that way
    var r = this.ww_add (0, operand, this._SAM);
    var sum = r[0];
    this._SAM = r[1];
    var alarm = r[2];

    this._AC = sum;
    this._AReg = operand;
    this._BReg = 0;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }


  /**
   * mr x multiply and roundoff #24d   030o 11000 65 microsec
   * Multiply contents of AC by contents of register x. Roundoff
   * result to 15 significant binary digits and store it in AC. Clear BR.
   * The magnitude of contents of register x appears in AR. SAM is cleared.
   */
  this.mr_inst = function(pc, address, opcode, op_description) {

    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    var r = this.ww_multiply (this._AC, operand);
    var a = r[0];
    var b = r[1];
    var alarm = r[2];

    this._AC = a;
    // the instructions say that AR should contain the Magnitude of Register X,
    //  so I assume if it's negative, we make it positive first.
    if ((operand & this.sim.cb.WWBIT0) == 0) {   // i.e., if positive
      this._AReg = operand;
    } else {
      this._AReg = this.ww_negate(operand);
    }
    this._BReg = 0;   // roundoff
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }

  /**
   * mh X multiply and hold 11001 o31  25d 34-41 microsec
   * Multiply contents of AC by contents of register x. Retain the full product
   * in AC and in the first 15 digit positions of BR, the last digit
   * position of BR being cleared. The magnitude of contents of register x
   * appears in AR. SAM is cleared. The sign of AC is determined by sign of
   * product. Result in (AC + BR) is a double register product. The time is
   * determined the same as for MR.
   */
  this.mh_inst = function(pc, address, opcode, op_description) {

    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    var r = this.ww_multiply (this._AC, operand);
    var a = r[0];
    var b = r[1];
    var alarm = r[2];

    this._AC = a;
    if ((operand & this.sim.cb.WWBIT0) == 0) {   // i.e., if positive
      this._AReg = operand;
    } else {
      this._AReg = this.ww_negate(operand);
    }
    this._BReg = b;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }



  /**
   * md x   multiply digits no roundoff  #31 o37  22 microsec  [bitwise AND for pete's sake!]
   *  The product of the ith digit of the AC multiplied by the ith digit of
   * register x becomes stored in the ith digit of the AC.  The final value
   * of the ith digit of the AC is 1 if (the initial value of the ith digits
   * of the AC and register x are both 1; otherwise, the final value of the
   * ith digit of the AC is 0.  AR contains the complement of the final con-
   * tents of the AC
   *
   *  This instruction seems to have been a late addition; 037 used to be 'qp'
   *  It seems to simply be a logical AND of @core & AC
   */
  this.md_inst = function(pc, address, opcode, op_description) {
    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    var a = this._AC & operand;

    this._AC = a;
    this._AReg = ~a & this.sim.cb.WWBIT0_15;

    this.print_cpu_state(pc, opcode, op_description, address);
    return this.sim.cb.NO_ALARM;
  }

  /**
   * sd X sum digits   00110   22 micrsec
   * The sum of the original contents of digit i of AC and original contents
   * of digit i of register x becomes stored in digit i of AC. The final
   * value of digit i of AC is O if (the values of digit i of AC and of register
   * x are alike; the final value of digit i of AC is 1 if (the values of
   * digit i of AC and of register x are different.
   */
  this.sd_inst = function(pc, address, opcode, op_description) {
    var operand = this.sim.coreMem.rd(address);
    if (operand == null) {
      return(this.sim.cb.READ_BEFORE_WRITE_ALARM);
    }
    var a = this._AC ^ operand;   // xor of Register X and AC

    this._AC = a;

    if (this.traceALU) {
      console.log("sd_inst  AC=" + toOctal(this._AC) + "o, Core=" + toOctal(this.sim.coreMem.rd(address)) + "o");
    }
    this.print_cpu_state(pc, opcode, op_description, address);
    return this.sim.cb.NO_ALARM;
  }

  /**
   * srr n  shift right and roundoff #28 #34o   41 microsec
   *  Shift contents of AC and BR (except sign digit) to the right n
   * places. The integer n is treated modulo 32; digits shifted right out
   * of BR 15 are lost. (Shifting right n places is equivalent to multiplying
   * by 2**-n .) Roundoff the result to 15 binary digits and, store it in AC. Clear
   * BR. Negative numbers are complemented before and after the shift, hence
   * ones appear in the digit places made vacant by the shift of a negative number.
   * Digit 6 (the 2**9=512 digit of the address) of the instruction srr n must be
   * a zero to distinguish srr n from srh n described below. The instruction
   * srr a simply causes roundoff and clears BR. SAM is cleared. Roundoff
   * (in a srr 0) may cause overflow, with a consequent arithemetic check alarm.

   * srh n   shift right and hold #28 #34o   41 microsec
   * Shift contents of AC and BR (except sign digit) to the right n
   * places. The integer n is treated modulo 32; digits shifted right out of
   * BR 15 are lost. (Shifting right n places is equivalent to multiplying by
   * 2**-n) Do not roundoff the result nor clear BR. Negative numbers are
   * complemented before and after the shift, hence ones appear in the digit places
   * made vacant by the shift of a negative number. Digit 6 (the 2**9=512 digit
   * of the address) of the instruction srh n must be a one to distinguish srh n
   * from srr n described above. SAM is cleared.
   */
  this.ww_shift = function(a, b, n, shift_dir, hold) {
    // shift on negative numbers works by complementing negative input, shifting and then re-complementing
    var negative = (a & this.sim.cb.WWBIT0);  // sign bit == 1 means Negative
    if (negative) {   // sign bit == 1 means Negative, so we turn it positive
      a = ~a & this.sim.cb.WWBIT0_15;
      b = ~b & this.sim.cb.WWBIT0_15;
      // 2M-0277 says that BR is not to be complemented; That surely means that they do a bit-by-bit
      // sign fixup on each shift cycle, crossing between a&b.  See pg 9.
    }
    // having eliminated the sign bit, combine a and b into a single 32-bit native python number, then shift
    var shift_val = a << 16 | b;
    n &= 0o37;    // shift value is "modulo 32".  I'm assuming "shift zero" is a no-op

    // do the actual shift, zero-filling the empty bits
    if (shift_dir == this.sim.cb.SHIFT_RIGHT) {
      shift_val >>= n;
    } else {
      shift_val <<= n;
    }

    // Not Hold is Roundoff; that means checking the most significant bit of the new B register, and adding
    //  one to a if it's set, then clearing B.
    if (!hold) {  // i.e., if Roundoff
      if (this.sim.cb.pyBIT15 & shift_val) {  // python bit 15 == most significant bit of B Register
        shift_val += 1 << 16;   // increase a by one
      }
      // TODO: backport this fix to Python
      shift_val &= ~this.sim.cb.WWBIT0_15;   // roundoff means "clear BR"
    }

    // The 30-bit result is taken apart into two 16-bit regsiters.  A Reg is the most-significant
    // part with the sign bit, B Reg is the least significant, with Bit 15 unused and zero (I think)
    var reg_b = shift_val & this.sim.cb.WWBIT0_15;
    var reg_a = (shift_val >> 16) & this.sim.cb.WWBIT0_15;

    // figure whether we have an alarm...  According th 2M-0277 pg 99, shift only produces an alarm in
    // the case of a "shift and roundoff"
    var alarm = this.sim.cb.NO_ALARM;
    if (reg_a & this.sim.cb.WWBIT0 & (!hold)) {   // the result is supposed to stay positive
      alarm = this.sim.cb.OVERFLOW_ALARM;       // -- if it switched signs, that's Overflow
    }

    if (negative) {
      reg_a = ~reg_a & this.sim.cb.WWBIT0_15;
      reg_b = ~reg_b & this.sim.cb.WWBIT0_15;
    }

    if (this.traceALU) {
      console.log("ww_shift: neg=" + negative +
          ", " + (hold ? "Hold" : "Round") +
          ", n=" + n +
          " " + (shift_dir == this.sim.cb.SHIFT_RIGHT ? "Right" : "Left") +
          ", a=" + toOctal(a) +
          ", b=" + toOctal(b) +
          ",  output: shift_val=" + toOctal(shift_val) +
          "o, reg_a=" + this.wwint_to_str(reg_a) +
          ", reg_b=" + this.wwint_to_str(reg_b));
    }

    return [reg_a, reg_b, alarm];
  }

  /**
   * this op code does the two Shift Right instructions
   */
  this.sr_inst = function(pc, address, opcode, op_description) {
    var operand = address & 0o37  // the rule book says Mod 32
    var hold = false;
    if (this.sim.cb.WWBIT6 & address) {
      hold = true;
    }
    var r = this.ww_shift(this._AC, this._BReg, operand, this.sim.cb.SHIFT_RIGHT, hold);
    var a = r[0];
    var b = r[1];
    var alarm = r[2];

    this._AC = a;
    this._BReg = b;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }

  /**
   * slr n shift left and roundoff  033-0   15+.8n microsec
   * (page 21, 2m-0277)
   * Shift frational contents of AC (except sign digit) and BR to the left n
   * places. The positive integer n is treated modulo 32; digits shifted
   * left out of AC 1 are lost. (Shifting left n places is equivalent to multiplying
   * by 2**n, with the result reduced modulo l.) Roundoff the result to
   * 15 binary digits and store it in AC. Clear BR. Negative numbers are complemented
   * before the shift and after the roundoff; hence, ones appear in
   * the digit places made vacant by the shift of a negative number. Digit 6
   * (the 2**9 = 512 digit of the address) of the instruction slr n must be a zero
   * to distinguish slr n from slh n described below. The instruction slr 0
   * simply causes roundoff and clears BR. SAM is cleared. Roundoff may
   * cause overflow with a consequent arithmetic check alarm if
   * <see doc>. The excution time varies according to the size of n.

   * slh n shift left and hold 033-1  15 + .8n microsec
   * Shift contents of AC (except sign digit) and BR to the left n places.
   * The positive integer n is treated modulo 32; digits shifted left out of
   * AC 1 are lost. (Shifting left n places is equivlent to multiplying by
   * 2**n, with the result reduced modulo 1.) Leave final product in AC and BR.
   * Do not roundoff or clear BR. Negative numbers are complemented in AC before
   * and after the shift; hence, ones appear in the digit places made
   * vacant by the shift of a negative number. Digit 6 (the 2**9 = 512 digit of
   * the address) of the instruction slh n must be a one to distinguish slh n
   * from slr n described above. SAM is cleared. The excution time depends
   * upon the size of the n.
   */

  this.sl_inst = function(pc, address, opcode, op_description) {
    var operand = address & 037;  // the rule book says Mod 32
    var hold = false;
    if (this.sim.cb.WWBIT6 & address) {
      hold = true;
    }
    var r = this.ww_shift(this._AC, this._BReg, operand, this.sim.cb.SHIFT_LEFT, hold);
    var a = r[0];
    var b = r[1];
    var alarm = r[2];

    this._AC = a;
    this._BReg = b;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return alarm;
  }

  /**
   *  [description from 2M-0277]
   * clc n     cycle left and clear (BR)  llll0-0 1.70 15 + .8n usec
   * Shift the full contents of A (including sign digit) and BR to the left n
   * places. The psitive integer n is treated modulo 32; digits shifted left
   * out of AC O are carried around into BR 15 so that no digits are lost. No
   * roundoff. Clear BR.  With the c1c operation there is no complementing of
   * AC either before or after the shift; the actual nurical digits in AC and
   * BR are cycled to the left. The digit finally shifted into the sign digit
   * position determines whether the result is to be considered a positive or
   * negtive quantity. Digit 6 (the 2**9 = 512 digit of the address) of the
   * instruction clc n must be a zero to distinguish clh n from clc n described
   * below. The instruction clc 0 simply clears BR without affecting AC. The
   * excution time depends on the size of the integer n.

   * this op code does the two Cycle Left instructions, Cycle Left Hold and Cycle Left Clear
   */
  this.cy_inst = function(pc, address, opcode, op_description) {
    var n = address & 037;  // the rule book says Mod 32
    var hold = false;
    if (this.sim.cb.WWBIT6 & address) {
      hold = true;
    }

    // ignore the sign bit, combine a and b into a single 32-bit native python number, then shift
    var a = this._AC;
    var b = this._BReg;
    var shift_val = (a & this.sim.cb.WWBIT0_15) << 16 | (b & this.sim.cb.WWBIT0_15);

    var cycle = shift_val >> (32-n);
    shift_val <<= n;
    shift_val |= cycle;   // combine in the overflow from the top part of the word

    var reg_b = shift_val & this.sim.cb.WWBIT0_15;
    var reg_a = (shift_val >> 16) & this.sim.cb.WWBIT0_15;

    // Not Hold means Clear, i.e., clear the B Reg
    if (!hold) {
      reg_b = 0;

      if (this.traceALU) {
        console.log("ww_left_cycle: in: n=" + toOctal(n) + "o, a=" + toOctal(a) + "o, b=" + toOctal(b) + "o, out: a=" + toOctal(reg_a) + "o, b=" + toOctal(reg_b) + "o");
      }
    }

    this._AC = reg_a;
    this._BReg = reg_b;
    this._SAM = 0;

    this.print_cpu_state(pc, opcode, op_description, address);
    return this.sim.cb.NO_ALARM;
  }


  /**
   * [From M-0277]
   * 1.1.3 (3) Primary Storage
   * Primary storage consists of 6144 registers of magnetic core memory (MCM) with
   * an access time of 7 mcroseconds, and 32 registers of test storage. There are three
   * shower-stalls a magnetic core memory, two of which contain 17 planes of 1024 cores
   * each. The third and latest shower-stall contains 17 planes of 4096 cores each.
   * The 17 planes represent the 16 binary digits of WWI register and the one digit used
   * for checking purposes.  Since WWI is designed to operate with a full complement of
   * 2048 registers of storage, only 2048 of the 6144 availble registers may be used
   * at a time.  Thus, for manipulation ease core storage is divided into six equal parts
   * of 1024 registers, with the proviso that two parts (fields) be used at one time to
   * provide the reqired 2048 registers necessary for full computer operation. The
   * six equal parts or "fields" are numbered 0 through 5 and may be chosen by the use
   * of the WWI instruction "change f1elds," cf. A control system exists which considers
   * register adresses to be in one of two groups: Group A includes register adresses
   * 0 - 1023 inclusive, and Group B includes register adresses 1024 - 2047 inclusive.
   * Any combination of fields can be used with the exception that the same field cannot
   * occupy Group A and Group B locations simultaneously. The 32 toggle-switch registers
   * of test storage and the five flip-flop storage registers, which may be interchanged
   * with any of the 32 toggle-switch storage registers, occupy registers 0 - 31 inclusive,
   * of the field memory used in Group A, which are thus normally unavailable for programming.

   * Change Field - Memory Bank Manager - from 2M-0277
   * cf pqr change fields   O0l11    0d7    15 microsec
   * The address section does not refer to a register of storage in this
   * instruction, but supplies information to the computer requesting a change
   * in fields Group A and/or B. When the field to be changed contains the
   * program, it is necessary for the cf instruction to perform like an sp
   * instruction. Digit 7 of the cf address section causes the contents of
   * the accumulator to be read to the program counter (PC) prior to the field
   * change; thus, program continuity can be preserved during field changes.
   * The A-Register will contain the original PC address plus one upon
   * completion of the cf instruction. The digit allocation for the cf word is
   * as follows:
   *
   * digits O - 4: O0ll1 cf order.
   * digit 5: spare
   * digit 6: examine feature - causes contents of core
   *   memory Group A control and Group B control
   *   registers to be read into the Accumulator.
   * digit 7: sp enable - reads content of AC to PC to
   *   establish starting point of program in the
   *   new field.
   * digit 8: change Group A field enable. (If, when the
   *   examine feature of digit 6 is requested,
   *   there is a "1" in digit 8, the content of
   *   Group A control will be changed before read-out
   *   to the A-Register takes place.)
   * digit 9: change Group B field enable. (If, when the
   *   examine feature of digit 6 is requested,
   *   there is a "1" in digit 9, the content of
   *   Group B control will be changed before read-out
   *   to the A-Register takes place.)

   * digits 10 - 12: contain field designation for Group A
   *   (registers l - 1777).
   * digits 13 - 15: contain field designation for Group B
   *   (registers 2000 - 3777).
   */
  this.cf_inst = function(pc, address, opcode, op_description) {

    var pqr = address;
    var ret1 = this.sim.cb.NO_ALARM;

    if ((pqr & this.sim.cb.WWBIT7) & (pqr & this.sim.cb.WWBIT6)) {
      console.log("cf_inst reads PC and MemGroup both into AC??");
      ret1 = this.sim.cb.UNIMPLEMENTED_ALARM;
    }

    var old;
    if (pqr & this.sim.cb.WWBIT9) {
      old = this.sim.coreMem.MemGroupB;
      this.sim.coreMem.MemGroupB = pqr & 007;
      console.log("CF @" + toOctal(pc) + ": Change MemGroup B from " + toOctal(old) + " to " + toOctal(this.sim.coreMem.MemGroupB));
    }

    if (pqr & this.sim.cb.WWBIT8) {
      old = this.sim.coreMem.MemGroupA;
      this.sim.coreMem.MemGroupA = (pqr >> 3) & 007;
      console.log("CF @" + toOctal(pc) + ": Change MemGroup B from " + toOctal(old) + " to " + toOctal(this.sim.coreMem.MemGroupA));
    }

    if (pqr & this.sim.cb.WWBIT7) {   // this seems to swap PC+1 and AC;   I think PC is already incremented at this point...
      var tmp = this.PC;
      this.PC = this._AC;   // implicit branch on bank change
      this._AC = tmp;
      console.log("CF @" + toOctal(pc) + ": Branch on bank change to " + toOctal(this.PC));
    }
    

    if (pqr & this.sim.cb.WWBIT6) {   // read back bank selects to AC
      var readout = this.sim.coreMem.MemGroupB | (this.sim.coreMem.MemGroupA << 3);
      this._AC = readout;
      console.log("CF @" + toOctal(pc) + ": Read Back Group Registers A|B = " + toOctal(this._AC, 2));
    }
    this.print_cpu_state(pc, opcode, op_description, address);
    return ret1;
  }


  /**
   * placeholders
   */
  this.unused1_inst = function(pc, address, opcode, op_description) {
    this.print_cpu_state(pc, opcode, op_description, address);
    return this.sim.cb.UNIMPLEMENTED_ALARM;
  }


  this.xx_inst = function(pc, address, opcode, op_description) {
    console.log("unimplemented Instruction near PC=" + toOctal(pc) + ", opcode=" + opcode);
    this.print_cpu_state(pc, opcode, op_description, address)
    return this.sim.cb.UNIMPLEMENTED_ALARM;
  }
  /**
   * Aug 27, 2018 - started to add r/w designator to each op to see if it should trap on an uninitialized var
   *  And the microsecond count to estimate performance
   */
  // indexed by five-bit op code
  this.op_decode = [
    // function  op-name  description     r/w, usec
    [this.si_inst, "SI", "Select Input",    '',  30],
    [this.unused1_inst, "unused", "unused",   '',  0],
    [this.bi_inst, "BI", "Block Transfer In", 'w', 8000],
    [this.rd_inst, "RD", "Read",        '',  15],
    [this.bo_inst, "BO", "Block Transfer Out",'r', 8000],  // 04
    [this.rc_inst, "RC", "Record",      '',  22],  // 05
    [this.sd_inst, "SD", "Sum Digits",    'r', 22],  // 06
    [this.cf_inst, "CF", "Change Fields"],  // 07
    [this.ts_inst, "TS", "Transfer to Storage"],  // 010,
    [this.td_inst, "TD", "Transfer Digits"],    // 011
    [this.ta_inst, "TA", "Transfer Address"],
    [this.ck_inst, "CK", "check"],
    [this.xx_inst, "AB", "Add B Reg"],
    [this.ex_inst, "EX", "Exchange"],        // 015
    [this.cp_inst, "CP", "conditional program"],   // 016
    [this.sp_inst, "SP", "subprogram"],      // 017
    [this.ca_inst, "CA", "clear and add"],     // 020
    [this.cs_inst, "CS", "clear and subtract"],  // 021
    [this.ad_inst, "AD", "add to AC"],       // 022
    [this.su_inst, "SU", "subtract" ],       // 23o, 19d
    [this.cm_inst, "CM",  "clear and add magnitude"],  // 24o, 20d
    [this.xx_inst, "XX", "unimplemented 0251o"],
    [this.ao_inst, "AO", "Add One"],         // 26o, 22d
    [this.dm_inst, "DM", "Difference of Magnitudes"],    // 027o
    [this.mr_inst, "MR", "Multiply & Round"],    // 030
    [this.mh_inst, "MH", "Multiply & Hold"],     //031
    [this.xx_inst, "XX", "unimplemented 032o"],
    [this.sl_inst, "SL", "Shift Left Hold/Roundoff"],
    [this.sr_inst, "SR", "Shift Right Hold/Roundoff"],  // 034
    [this.xx_inst, "XX", "unimplemented 035o"],
    [this.cy_inst, "CL", "Cycle Left"],   // 036
    [this.md_inst, "MD",  "Multiply Digits no Roundoff (AND)"],  // 037
  ];
}

