// Whirlwind I/O Device Simulation
// This file contains classes to simulate various Whirlwind I/O devices
// Function is mostly based on Manual 2M-0277
// Based on Python simulator by Guy Fedorkow, July 2018

/**
 * This class is a template for new I/O devices
 * @constructor
 */
DummyIoClass = function(sim) {
  this.sim = sim;

  this.DUMMY_BASE_ADDRESS = 010000;  // starting address of Drum device(s)
  this.DUMMY_ADDR_MASK = ~(0001);  // mask out the sub-addresses

  this.name = "Dummy";

  // each device needs to identify its own unit number.
  this.is_this_for_me = function(io_address) {
    if ((io_address & this.DUMMY_ADDR_MASK) == this.DUMMY_BASE_ADDRESS) {
      return this;
    } else {
      return null;
    }
  };

  this.si = function(device, acc) {
    return this.sim.cb.NO_ALARM;
  };

  this.rc = function(acc) {  // "record", i.e. output instruction to device
    console.log("unimplemented " + this.name + " Record");
    return [this.sim.cb.UNIMPLEMENTED_ALARM, 0];
  };
};

/**
 * this class is a "device" that can be used to clear a block of memory
 * From 2M-0277

 * 4.7 ROGRAMMED CORE CLEARING
 * A special mde of clearing magnetic core memory can be selected by the
 * use of this instruction si 17 (o) which initiates the process of readng +O
 * into a block of registers, thereby clearing them. A bi y, where y is the
 * initial addess of the block, then clears the block length determined by
 * +/-n in the accumulator, where n is the number of registers to be cleared.
 * For example, the following octal program) {
 *  si 17
 *  ca RCn  (n is numebr of words in AC)
 *  bi RCy  (y is initial register adress of the block)
 */
CoreClearIoClass = function(sim) {
  this.sim = sim;

  this.CLEAR_BASE_ADDRESS = 0o17;  // starting address of Drum device(s)
  this.CLEAR_ADDR_MASK = ~0000;  // there aren't any sub-addresses

  this.name = "CoreClear";

  // each device needs to identify its own unit number.
  this.is_this_for_me = function(io_address) {
    if ((io_address & this.CLEAR_ADDR_MASK) == this.CLEAR_BASE_ADDRESS) {
      return this;
    } else {
      return null;
    }
  };

  this.si = function(device, acc) {  // the drum's mode of operation comes hidden in the Device field
    console.log("SI: 'Clear Memory' device initialized ");
    return this.sim.cb.NO_ALARM;
  };

  this.rc = function(acc) {  // "record"
    console.log("unimplemented rc: Core Clear Record");
    return [this.sim.cb.NO_ALARM, 0];
  };

  this.rd = function(acc) {  // "read"
    console.log("unimplemented rc: Core Clear Read");
    return 0
  };

  this.bi = function(address, acc, cm) {  // "block transfer in"
    /**
     * perform a block transfer input instruction
     * address: starting address for block
     * acc: contents of accumulator (the word count)
     * cm: core memory instance
     */
    console.log("block transfer Clear Memory: start address " + toOctal(address) + "o, length 0o" + toOctal(acc));
    if (address + acc > this.sim.cb.WW_ADDR_MASK) {
      console.log("block transfer in Clear Mem out of range");
      return this.sim.cb.QUIT_ALARM;
    }
    for (var m = address; m < address + acc; m++) {
      cm.wr(m, 0);  // write zero
    }
    return this.sim.cb.NO_ALARM;
  };
};


 /**
  * Whirlwind had a teletype system in addition to the Flexowriter
  */
tty_class = function(sim) {
  this.sim= sim;
  this.figureShift = false;
  /**
   *  2M-0277_Whirlwind_Programming_Manual_Oct58.pdf
   *  pg 94 for the teletype code chart
   *  Watch for the typo in Fig 18 -- Two 'Z', no 'A'
   *  Page 95 also says that the Record instruction increments the PC
   *  by two if the printer is busy, the instruction is always followed
   * by two branch instructions, (often one forward, one back)
   *  ls = Letter Shift
   *  fs = Figure Shift
   */
  var tty_letters = ["<null>", "T", "\n", 'O', " ", "H", "N", "M",
           "lf", "L", "R", "G", "I", "P", 'C', 'V',
           'E', 'Z', 'D', 'B', 'S', 'Y', 'F', 'X',
           'A', 'W', 'J', 'fs', 'U', 'Q', 'K', 'ls']

  var tty_figures = ['00', '5', '\n', '9', ' ', '//', '7/8', '.',
           'lf', '3/4', '4', '&', '8', '0', '1/8', '3/8',
           '3', '"', '$', '5/8', '^g', '6', '1/4', '/',
           '-', '2', "'", 'fs', '7', '1', '1/2', 'ls']

  this.tty_charset = [tty_letters, tty_figures]
  this.IO_ADDRESS_TTY = 0402
  this.ttyOutput = []
  this.name = "Teletype"
  this.figureShift = 0;

  this.code_to_letter = function(w) {  // input is a 16-bit word with three packed 5-bit characters
    var str = "";
    for (var shift = 2; shift > -1; shift--) {
      var code = (w >> shift * 5) & 037;
      if (code == 033) {
        this.figureShift = 1;
      } else if (code == 037) {
        this.figureShift = 0;
      } else {
        str += this.tty_charset[this.figureShift][code];
      }
    }
    return str;
  };

  // each device needs to identify its own unit number.
  this.is_this_for_me = function(io_address) {
    if (io_address == this.IO_ADDRESS_TTY) {
      return this;
    } else {
      return null;
    }
  };

  this.si = function(device, acc) {   // the tty device needs no further initialization
    return this.sim.cb.NO_ALARM;
  };

  this.rc = function(acc) {  // "record", i.e. output instruction to tty
    var code = acc;
    var symbol = this.code_to_letter(code)  // this actually returns three symbols
    this.ttyOutput.push(symbol);
    return [this.sim.cb.NO_ALARM, symbol];
  };

  this.get_saved_output = function() {
    return this.ttyOutput;
  };
};



DrumClass = function(sim) {
  this.sim = sim;
  this.word_address = 0;    // this would have to be angular offset from the index point of the drum
  this.group_address = 0;   // this would have to be track number
  this.drum_unit = 0;     // 1 for "Buffer Drum", 0 for "Aux Drum"
  this.wrap_address = 0;
  this.record_mode = 0;


  this.DRUM_BASE_ADDRESS = 0700;  // starting address of Drum device(s)
  this.DRUM_ADDR_MASK = ~(01017);  // mask out the sub-addresses

  // drum capacity
  this.DRUM_NUM_GROUPS = 12;     // 12 tracks
  this.DRUM_NUM_WORDS = 2048;    // words per track
  // Array of 2 drums, each an Array of size DRUM_NUM_GROUPS, each element an array of DRUM_NUM_WORDS
  this._drum_content = Array(2).fill(0).map(x => Array(this.DRUM_NUM_GROUPS).fill(0).map(x => Array(this.DRUM_NUM_WORDS).fill(null)))

  // drum address decode
  this.DRUM_SI_WORD_ADDRESS = 0001;
  this.DRUM_SI_GROUP_ADDRESS = 0002;
  this.DRUM_SI_RECORD_MODE = 0004;
  this.DRUM_SI_BUFFERDRUM = 0010;
  this.DRUM_SI_WRAP_ADDRESS = 001000;

  this.name = "Drum";

  /**
   * each device needs to identify its own unit number.
   */
  this.is_this_for_me = function(io_address) {
    if ((io_address & this.DRUM_ADDR_MASK) == this.DRUM_BASE_ADDRESS) {
      return this;
    } else {
      return null;
    }
  };

  /**
   * 2M-0277, pg 25/26
   * The drun address to be selected is determined by the si instruction and
   * by any necessary portions of the contents of AC at the time the si is executed.
   * The si instruction may call for a new goup number, a new initial storage
   * address, neither, or both. When a new goup number is needed, it is taken
   * from digits l - 4 of AC. When a new 1nitial storage address is needed, it is
   * taken from digts 5 - 15 of AC. Either the group selectd on the drum can remain
   * selected until an si instruction specifically calls for a change of goup
   * or, by adding 1000 to the original auxiliary drum si, a block transfer is permitted
   * to run off the end of one drum goup to the beginning of the next drum
   * group. The next storage address selected wlll be one greater than the atorge
   * address most recently referred to, unless an si instruction specifically calls
   * for a new initial storage address
   *
   * we must note that the doc seems ambiguous about how to tell if it's Buffer or Aux drum
   */
  this.si = function(device, acc) {  // the drum's mode of operation is in the Device field, with address in the acc
    this.wrap_address = (device & this.DRUM_SI_WRAP_ADDRESS);
    this.record_mode = (device & this.DRUM_SI_RECORD_MODE);
    if (device & this.DRUM_SI_GROUP_ADDRESS) {
      this.group_address = (acc >> 11) & 017;
    }
    if (device & this.DRUM_SI_WORD_ADDRESS) {
      this.word_address = acc & 03777; // TODO: fix this in Python from 3777
    }
    if (device & this.DRUM_SI_BUFFERDRUM) { // TODO: fix this in Python: shouldn't check acc
      this.drum_unit = 1;
    } else {
      this.drum_unit = 0; // TODO: fix in Python
    }
    console.log("SI: configured drum; Group (track)=0o" + toOctal(this.group_address) + ", WordAddress=0o" + toOctal(this.word_address) + ", Unit=" + toOctal(this.drum_unit));
    return this.sim.cb.NO_ALARM;
  };

  this.rc = function(unused, acc) {  // "record", i.e. output instruction to drum
    console.log("RC: write-to-drum; Word=0o" + toOctal(acc) + ", Group (track)=0o" + toOctal(this.group_address) + ", WordAddress=0o" + toOctal(this.word_address) + ", Unit=" + toOctal(this.drum_unit));
    this._drum_content[this.drum_unit][this.group_address][this.word_address] = acc;
    this.word_address += 1;
    if (this.word_address > this.DRUM_NUM_WORDS) {
      console.log("Haven't implemented Drum Address Wrap");
      return [this.sim.cb.UNIMPLEMENTED_ALARM, 0];
    }
    return [this.sim.cb.NO_ALARM, 0];
  };

  this.bi = function(address, acc, cm) {  // "block transfer in"
    /**
     * perform a block transfer input instruction
     * address: starting address for block
     * acc: contents of accumulator (the word count)
     * cm: core memory instance
     */
    console.log("block transfer Read from Drum: start address 0o" + toOctal(address) + ", length 0o" + toOctal(acc));
    if (address + acc > this.sim.cb.WW_ADDR_MASK) {
      console.log("block-transfer-in Drum address out of range");
      return this.sim.cb.QUIT_ALARM;
    }
    for (var m = address; m < address + acc; m++) {
      wrd = this._drum_content[this.drum_unit][this.group_address][this.word_address] // read the word from drum
      this.word_address += 1
      if (this.word_address > this.DRUM_NUM_WORDS) {
        console.log("Haven't implemented Drum Address Wrap");
        return this.sim.cb.UNIMPLEMENTED_ALARM;
      }
      cm.wr(m, wrd);  // write the word to mem
    }
    return this.sim.cb.NO_ALARM;
  };
};

/**
 * The Oscilloscope Display contains a character generator that can draw arbitrary seven segment
 * characters at a screen location based on a bit map supplied by the programmer
 * This routine renders the bit map in ASCII Art, returning an array of lines with dashes, spaces
 * and bars to render the glyph.
 * See page 60 of 2M-277 for a picture of the segment layout
 */
SevenSegClass = function() {
  this.TOP_RIGHT = 0001;
  this.TOP = 0002;
  this.TOP_LEFT = 0004;
  this.MIDDLE = 00010;
  this.BOTTOM_RIGHT = 00020;
  this.BOTTOM = 00040;
  this.BOTTOM_LEFT = 00100;

  this.HORIZ_STROKE = "-----";
  this.HORIZ_BLANK = "   ";
  this.HORIZ_SPACER = "   ";

  this.render_char = function(bits) {
    // do the three horizontal strokes
    this.lines = [];
    this.line = ['', '', '', '', ''];
    i = 0;
    for (var j = 0; j < 3; j++) {
      var mask = [this.TOP, this.MIDDLE, this.BOTTOM][j];
      if (bits & mask) {
        this.line[i] = this.HORIZ_STROKE;
      } else {
        this.line[i] = this.HORIZ_BLANK;
      }
      i += 2;
    }

    // do the two vertical strokes
    i = 1
    for (var j = 0; j < 2; j++) {
      var mask = [[this.TOP_LEFT, this.TOP_RIGHT], [this.BOTTOM_LEFT, this.BOTTOM_RIGHT]][j];
      if (bits & mask[0]) {
        this.line[i] = "|";
      } else {
        this.line[i] = " ";
      }
      this.line[i] += this.HORIZ_SPACER;
      if (bits & mask[1]) {
        this.line[i] += "|";
      } else {
        this.line[i] += " ";
      }
      i += 2;
    }

    // glue the result together into an array of strings
    this.lines.push(this.line[0]);
    this.lines.push(this.line[1]);
    this.lines.push(this.line[1]);
    this.lines.push(this.line[2]);
    this.lines.push(this.line[3]);
    this.lines.push(this.line[3]);
    this.lines.push(this.line[4]);
    return this.lines;
  };
};


/**
 * Whirlwind had a number of display scopes attached, but it appears that the same basic picture was
 * displayed on all of them, with some variation.  So each command selects a subset of the scopes for
 * display of the next point, line or character.
 * Similarly, there were a number of light guns, each of which can detect the same point when/if it's
 * drawn on the holder's display.
 * See 2M-0277
 */
DisplayScopeClass = function(sim) {
  // instantiate the class full of constants; what's the Right way to do this??
  if (typeof sim == "undefined" || sim == null) {
    alert("Bad!");
  }
  this.sim = sim;

  // I'm not sure yet if the Display Scope driver should be three independent drivers, or if the interact.
  // Obviously, it's One for now.
  this.DISPLAY_POINTS_BASE_ADDRESS = 00600;   // starting address of point display
  this.DISPLAY_POINTS_ADDR_MASK = ~(0077);  // mask out the sub-addresses
  this.DISPLAY_VECTORS_BASE_ADDRESS = 01600;  // starting address of vector display
  this.DISPLAY_VECTORS_ADDR_MASK = ~(0077);   // mask out the sub-addresses
  this.DISPLAY_CHARACTERS_BASE_ADDRESS = 02600;  // starting address of vector display
  this.DISPLAY_CHARACTERS_ADDR_MASK = ~(01077);  // mask out the sub-addresses
  this.DISPLAY_EXPAND_BASE_ADDRESS = 00014;   // starting address of vector display
  this.DISPLAY_EXPAND_ADDR_MASK = ~(0001);  // mask out the sub-addresses

  this.DISPLAY_MODE_POINTS = 1;
  this.DISPLAY_MODE_VECTORS = 2;
  this.DISPLAY_MODE_CHARACTERS = 3;
  this.ModeNames = ["No Mode", "Points", "Vectors", "Characters"];

  this.name = "DisplayScope";
  this.scope_select = null;  // identifies which of the zillion oscilloscopes to brighten for the next op
  this.scope_mode = null;  // Points, Vectors or Characters
  this.scope_expand = false;
  this.scope_vertical = null;   // scope coords are stored here as Pythonic numbers from -1023 to +1023
  this.scope_horizontal = null;
  this.characters_ready_to_draw = [];

  this.sevenseg = new SevenSegClass();
  this.saved_character_output = [];

  /**
   * the vertical axis for stuff coming up is stored in the left-most 11 bits of the accumulator.
   * convert the coords from ones complement into Pythonic Numbers
   * see pg 59 of 2M-0277

   * 3.8.2 Scope Deflection
   * The left-hand 11 digits of AC (including the sign digit), at the time a
   * display instruction is given, determine the direction and amount of deflection.
   * The positive direction of horizontal deflection is to the right ad positive
   * vertical deflection is upward. The value 1 - 2**10 or its negative will produce
   * the maximum deflection. The center of the scope represents the origin with zero
   * horizontal and vertical deflection.
   */
  this.convert_scope_coord = function(ac) {
    if (ac & this.sim.cb.WWBIT0) {  // test the sign bit for Negative
      extra_deflection_bits = ~ac & 037;
      ret = -(ac ^ this.sim.cb.WWBIT0_15) >> 5;
    } else {
      extra_deflection_bits = ac & 037;
      ret = ac >> 5;
    }

    if (extra_deflection_bits != 0) {
      console.log("Warning:  bits lost in scope deflection; AC=0o" + toOctal(ac));
    }
    return ret;
  };

  // each device needs to identify its own unit number.
  this.is_this_for_me = function(io_address) {
    if (((io_address & this.DISPLAY_POINTS_ADDR_MASK) == this.DISPLAY_POINTS_BASE_ADDRESS) ||
        ((io_address & this.DISPLAY_VECTORS_ADDR_MASK) == this.DISPLAY_VECTORS_BASE_ADDRESS) ||
        ((io_address & this.DISPLAY_CHARACTERS_ADDR_MASK) == this.DISPLAY_CHARACTERS_BASE_ADDRESS) ||
        ((io_address & this.DISPLAY_EXPAND_ADDR_MASK) == this.DISPLAY_EXPAND_BASE_ADDRESS)) {
      return this;
    } else {
      return null;
    }
  };

  this.si = function(io_address, acc) {
    if ((io_address & this.DISPLAY_EXPAND_ADDR_MASK) == this.DISPLAY_EXPAND_BASE_ADDRESS) {
      this.scope_expand = ((io_address & ~this.DISPLAY_EXPAND_ADDR_MASK) == 0)  // o14 is Expand, o15 is Unexpand
      if (!this.sim.traceQuiet) {
        console.log("DisplayScope SI: Display Expand set to " + toOctal(this.scope_expand));
      }
      return this.sim.cb.NO_ALARM;
    }

    if ((io_address & this.DISPLAY_POINTS_ADDR_MASK) == this.DISPLAY_POINTS_BASE_ADDRESS) {
      this.scope_mode = this.DISPLAY_MODE_POINTS;
    } else if ((io_address & this.DISPLAY_VECTORS_ADDR_MASK) == this.DISPLAY_VECTORS_BASE_ADDRESS) {
      this.scope_mode = this.DISPLAY_MODE_VECTORS;
    } else if ((io_address & this.DISPLAY_CHARACTERS_ADDR_MASK) == this.DISPLAY_CHARACTERS_BASE_ADDRESS) {
      if (!this.sim.traceQuiet && io_address & 001000) {
        console.log("DisplayScope SI: set scope selection to 0o" + toOctal(io_address) + "; ignoring ioaddr bit 001000...");
      }
      this.scope_mode = this.DISPLAY_MODE_CHARACTERS;
    }

    this.scope_select = ~this.DISPLAY_POINTS_ADDR_MASK & io_address;
    this.scope_vertical = this.convert_scope_coord(acc);

    if (!this.sim.traceQuiet) {
      console.log("DisplayScope SI: configured display mode " + this.ModeNames[this.scope_mode] + ", scope 0o" + toOctal(this.scope_select) + ", vertical=0o" + toOctal(this.scope_vertical));
    }
    return this.sim.cb.NO_ALARM;
  };

  this.rc = function(char_code, acc) {  // "record", i.e. output instruction to scope.
    /**
     * Plot something on the Oscilloscope
     * if it's a point, the vertical was previously specified in an SI instruction
     *   and the ACC contains the horizontal
     * if it's a character, the operand address part of the instruction gives the address of a
     *   memory location containing the bit mask of the seven-segment character to be rendered,
     *   in bits 1-7.
     */
    this.scope_horizontal = this.convert_scope_coord(acc);
    var char_str = '';
    if (this.scope_mode == this.DISPLAY_MODE_CHARACTERS) {
      var bits = char_code >> 8;
      char_str = ", char_code=0o" + toOctal(bits);
    }
    if (!this.sim.traceQuiet) {
      console.log("DisplayScope RC: record to display-scope mode=" + this.ModeNames[this.scope_mode] + ", horiz=0o" + toOctal(this.scope_horizontal) + ", vert=0o" + toOctal(this.scope_vertical) + " " + char_str);
    }
    if (this.scope_mode == this.DISPLAY_MODE_CHARACTERS) {
      // add each new character to a Pending list; draw them when the program asks for light gun input
      this.characters_ready_to_draw.push([this.scope_horizontal, this.scope_vertical, bits, this.display_time]);
      if (!this.sim.traceQuiet) {
        var art = this.sevenseg.render_char(bits);
        for (var i = 0; i < art.length; i++) {
          console.log(art[i]);
        }  
      }
      this.saved_character_output.push([this.scope_horizontal, this.scope_vertical, bits]);
    }
    return [this.sim.cb.NO_ALARM, 0];
  };

  /**
   * doing a "read" operation on the CRT display returns the status of the Light Gun, indicating if the
   *   trigger was pulled when the last spot was displayed
   * The sign bit will be off if the trigger was not pulled and the gun didn't see anything.
   * If the sign bit is on, the return code can be analyzed to figure out which gun had been triggered.
   * See 2M-0277 pg 72 for grubby details
   */
  this.rd = function(code, acc) {
    s = this.get_saved_output(true);
    if (s.length  && !this.sim.traceQuiet) {
      console.log("\nDisplayScope Said:");
      for (var i = 0; i < s.length; i++) {
        console.log(s[i]);
      }
    }

    for (var i = 0; i < this.characters_ready_to_draw.length; i++) {
      var crd = this.characters_ready_to_draw[i];;
      this.sim.crt.ww_draw_char(crd[0], crd[1], crd[2], true);
    }
    this.characters_ready_to_draw = [];

    this.sim.crt.ww_draw_point(this.scope_horizontal, this.scope_vertical, true);
    this.sim.crt.ww_scope_update();  // flush pending display commands
    var r = this.sim.crt.ww_check_light_gun();
    var alarm = r[0];
    var pt = r[1];

    if (alarm == this.sim.cb.QUIT_ALARM) {
      return [alarm, 0];
    }

    if (pt != null) {
      this.sim.crt.last_mouse = pt;
    }

    /**
     * check to see if the most recent mouse click was near the last dot to be drawn on the screen; if so,
     * count it as a hit, otherwise its a miss.  One it hits, "forget" the last mouse click
     */
    var val = 0;
    if (this.sim.crt.last_mouse != null && (
        Math.abs(this.sim.crt.last_pen_point[0] - this.sim.crt.last_mouse[0]) < this.sim.crt.WIN_MOUSE_BOX) &
        (Math.abs(this.sim.crt.last_pen_point[1] - this.sim.crt.last_mouse[1]) < this.sim.crt.WIN_MOUSE_BOX)) {
      this.sim.crt.ww_highlight_point();
      this.sim.crt.last_mouse = null;
      val = 0177777;
      return [this.sim.cb.NO_ALARM, val];
    }

    return [this.sim.cb.NO_ALARM, val];
  };

  /**
   * We're saving the sequence of characters generated by the oscilloscope character generator so we
   * can print the sequence later.  Since it's almost impossible to know what any given bit pattern means,
   * we need to save the bit pattern and location, and then render the display again.
   * This routine will return the output only once, and then delete it.  The screen is refreshed by
   * the ongoing WW software process, so I want to show what was put on the screen up until the time when
   * it stopped for Light Gun input.  Next time there's a Light Gun input request, we should display just
   * the new characters.
   */
  this.get_saved_output = function(text_render=true) {

    if (this.saved_character_output.length == 0) {
      return '';
    }

    var char_num = 0;
    var lines = ['', ''];
    /*
     * there must be a better way to do this...  I want to size the array to hold the number of rows returned
     * by the character generator, plus two for x & y
     * So I call the character generator on a random character, and count what comes back
     */
    var art = this.sevenseg.render_char(0);
    for (var idx = 0; idx < art.length; idx++) {
      lines.push('');
    }

    for (var idx = 0; idx < this.saved_character_output.length; idx++) {
      var c = this.saved_character_output[idx];
      var bits = c[2];
      var x = c[0];
      var y = c[1];

      if (text_render) {
//        this.sim.crt.ww_draw_char(x, y, bits)
        var i = 0;
        art = this.sevenseg.render_char(bits);
        for (var j = 0; j < art.length; j++) {
          lines[i] += (art[j] + '   ');
          i += 1;
        }
        lines[i] += "x=" + toOctal(x) + "o";
        lines[i+1] += "y=" + toOctal(y, 4) + "  ";
        char_num += 1
      } else {
        lines[0] += "(" + c[0] + ", " + c[1] + ", " + c[2] + ")";
      }
    }

    if (text_render == false) {
      lines[0] = '(' + lines[0] + ')';
    }
    this.saved_character_output = []  // Gosh I hope the Garbage Collectors aren't on strike
    return lines
  };
};
