/*****************************************************************************

       Copyright  1993, 1994 Digital Equipment Corporation,
                       Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, provided  
that the copyright notice and this permission notice appear in all copies  
of software and supporting documentation, and that the name of Digital not  
be used in advertising or publicity pertaining to distribution of the software 
without specific, written prior permission. Digital grants this permission 
provided that you prominently mark, as not part of the original, any 
modifications made to this software or documentation.

Digital Equipment Corporation disclaims all warranties and/or guarantees  
with regard to this software, including all implied warranties of fitness for 
a particular purpose and merchantability, and makes no representations 
regarding the use of, or the results of the use of, the software and 
documentation in terms of correctness, accuracy, reliability, currentness or
otherwise; and you rely on the software, documentation and results solely at 
your own risk. 

******************************************************************************/
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/

#undef PCIDEBUG
#undef TRACE_ENABLE

/*
 * Include files.
 */

#include "lib.h"
#include "uilib.h"
#include "northbridge.h"
#include "platform.h"

#include "pci.h"
#include "pcidefs.h"


/*
 * Odd definitions.
 */

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
#ifndef NULL
#define NULL 0
#endif




/*--------------------------------------------------------------------*/
/* Private definitions, data and functions */

static ub pci_init_done = FALSE;

/*
 * The root of pci devices found and the unallocated PCI devices/buses.
 */

#define MAX_PCI_HOSES 2
PCIBus_t *pci_hose[MAX_PCI_HOSES];
PCIDevice_t *devices = NULL;	        /* linked list of all devices */

static unsigned PCIIOBase;
static unsigned long PCIMemBase;	/* long for 64-bit address spaces */

int VgaDeviceDetected;

/*
 * Any macros that are local to this module.
 * PCI_MASK_UNIT is defined in the system specific module
 * as it varies between PCI chipsets.
 */
#define _PCIMaskToSize(m) \
  ( ((m >> 20) + 1) * PCI_MASK_UNIT )

#define MAX(val1, val2) ( ((val1) > (val2)) ? val1 : val2)
#define align(value,alignment) \
	( (value & (alignment - 1)) == 0 ? value : value + (alignment - (value & (alignment - 1))) )
#define _Align(value,alignment) \
	((((value) & ~(alignment)) == 0) ? (value) : ((value) & (alignment)) + ~(alignment) + 1)

/* Base addresses including ROM */
#define BAR_OFFSET(idx) (( (idx) < MAX_BASE_ADDRESS_REGISTERS ) ? \
			 (( (idx) * 4 )+0x10) : 0x30)	




/* PCI_allocate_bus - malloc a new bus, return the pointer or NULL */

static PCIBus_t *PCI_allocate_bus( void )
{
    PCIBus_t *bus;
/*
 *  Allocate one.
 */
    bus = malloc( sizeof( PCIBus_t ) );
    if ( bus == NULL )				return NULL;
    memset((char *)bus, 0, sizeof(PCIBus_t));
    return bus;
}


/* PCI_allocate_device - malloc a new PCI device, return the pointer or NULL */

static PCIDevice_t *PCI_allocate_device( void )
{
    PCIDevice_t *device;
/*
 *  Allocate one.
 */
    device = malloc( sizeof( PCIDevice_t ) );
    if ( device == NULL )			return NULL;
    memset((char *)device, 0, sizeof(PCIDevice_t) );
    return device;
}



/* Check to see if a PCI slot has a device, returns 0 if no device,
 * otherwise the device's Vendor and Device ID's 
 */

static ui PCI_slot_filled(int bus, int slot, int function)
{
    ui l;

    IOPCIClearNODEV();		/* Clear any outstanding NODEV errors. */
    MachineCheckExpected = TRUE;

/*  
 *  Read the vendor and device ids (first longword).
 */
    l = pcicfgrl(bus, slot, function, PCI_CFG_REG_VENDOR_DEVICE);
    usleep(1000);	/* Give the expected machine check a chance to happen */

/*
 *  Clear any pending PCI NODEV error and
 *  return if slot is unoccupied.
 */
    if (IOPCIClearNODEV()) return 0;

    MachineCheckExpected = FALSE;
#if 0		/* STIG DEBUG - makes things a little too verbose */
    TRACE("ExpectedMachineCheckTaken = 0x%lx\n", ExpectedMachineCheckTaken);
#endif
    if (ExpectedMachineCheckTaken) {			/* due to NODEV */
      TRACE("Expected Machine Check Taken\n");
      ExpectedMachineCheckTaken = FALSE;
      return 0;
    }

/*
 *  Check for all 1s as the vendor id (this also means
 *  that the slot doesn't contain a device, vendor ID 0xFFFF is invalid).
 */
    if ((l & 0xffff) == 0xffff) return 0;	/* Master Aborts return FFFF */
    return l;					/* Else the slot is occupied. */
}


/**************************************************************************
 * PCISetDeviceConfig()                                                   *
 **************************************************************************/
static void PCISetDeviceConfig(PCIDevice_t * device)
{
    PCIBus_t *parent;
    ui command, latency;
    ui busno = (ui) device->parent->number;
    ui alignment;
    int idx;
    ui bar1, bar2;

    command =
	pcicfgrl(busno, device->slot, device->function,
		 PCI_CFG_REG_STATUS_COMMAND);

/*
 *  Tell the device where the bases are (and enable use of them).
 */
#ifdef TRACE_ENABLE
    mobo_logf(LOG_DBG "  ConfigDevice, Bus:%d Slot:%d Func:%d\n",
	      busno, device->slot, device->function);
#endif
    for (idx = 0; idx < MAX_BASE_ADDRESS_REGISTERS + 1; ++idx) {
	if (device->PCI_Bar[idx].Size == 0)
	    continue;

	if (device->PCI_Bar[idx].Type & TYPE_MEM_64BIT) {

	    /* handle 64 bit address space case - write to 2 BARs */
	    bar1 = device->PCI_Bar[idx].Base & 0xFFFFFFFFU;
	    bar2 = (device->PCI_Bar[idx].Base >> 32) & 0xFFFFFFFFU;

#ifdef TRACE_ENABLE
	    mobo_logf(LOG_DBG "PCI: writing 64 bit address register:\n"
		      LOG_DBG "\t 0x%08x:%08x to BARs at 0x%0X:0x%0X\n",
		      bar2, bar1,
		      device->PCI_Bar[idx + 1].Reg,
		      device->PCI_Bar[idx].Reg);
#endif

	    pcicfgwl(busno, device->slot, device->function,
		     device->PCI_Bar[idx].Reg, bar1);
	    pcicfgwl(busno, device->slot, device->function,
		     device->PCI_Bar[idx + 1].Reg, bar2);

	} else {

	    /* 32 bit address space case */
	    pcicfgwl(busno, device->slot, device->function,
		     device->PCI_Bar[idx].Reg,
		     device->PCI_Bar[idx].Base | 0x1);

#ifdef TRACE_ENABLE
	    mobo_logf(LOG_DBG "\t%s Bar:%x, %x\n",
		      (device->PCI_Bar[idx].Type & 0x1) ? "I/O" : "Mem",
		      BAR_OFFSET(idx), device->PCI_Bar[idx].Base);
#endif
	}

	if (device->PCI_Bar[idx].Type & TYPE_IO) {
	    command |= PCI_COMMAND_IO;	/* Enable in I/O Space */
	} else {
	    if (idx < MAX_BASE_ADDRESS_REGISTERS)
		command |= PCI_COMMAND_MEMORY;	/* Enable in Memory Space */
	    else
		command |= PCI_COMMAND_IO;	/* Enable ROM Base */
	}
    }


/*
 *  If this is a VGA device (the first found), enable its IO regs and ROM
 */
    if ( IS_VGA_DEV( device ) ) {

	/* Enable I/O registers for only the first VGA device detected. */

	if (!VgaDeviceDetected) {
	    TRACE
		("Detected and enabling VGA device on bus %d slot %d\n",
		 device->parent->number, device->slot);

	    VgaDeviceDetected = TRUE;
	    command |= PCI_COMMAND_IO;	/* enable option ROM */

/*
 *  Query for and enable the BIOS option ROM
 *  Relocate it to 0xc0000 if it is on the primary PCI bus.
 */
	    if ( busno == 0 ) {
		pcicfgwl(busno, device->slot, device->function,
			 PCI_CFG_REG_EROM_BASE, 0xfffff000);
		alignment = pcicfgrl(busno, device->slot, device->function,
				     PCI_CFG_REG_EROM_BASE);
		pcicfgwl(busno, device->slot, device->function,
			 PCI_CFG_REG_EROM_BASE, _Align(0xc0000, alignment) | 1);

		device->PCI_Bar[MAX_BASE_ADDRESS_REGISTERS].Base = 0xc0000;
	    }

/*
 *  Walk back up the tree, enabling VGA compatible address
 *  support for each of the bridges along the way.
 */
	    parent = device->parent;
	    while (parent->bridge != NULL) {
		ui bridgeval;
		bridgeval =
		    pcicfgrl(parent->parent->number, parent->bridge->slot,
			     parent->bridge->function, 0x3c);
		pcicfgwl(parent->parent->number, parent->bridge->slot,
			 parent->bridge->function, 0x3c,
			 bridgeval | 0x80000);
		parent = parent->parent;
	    }
	} else {
	    TRACE("Detected and disabled secondary VGA on bus %d slot %d\n",
		 device->parent->number, device->slot);
	}
    }

/*
 * Enable bus mastering if supported
 */

    command |= PCI_COMMAND_MASTER;
    pcicfgwl(busno, device->slot, device->function,
	     PCI_CFG_REG_STATUS_COMMAND, command);
    command = pcicfgrl(busno, device->slot, device->function,
		       PCI_CFG_REG_STATUS_COMMAND);
    TRACE("\tCommand:%x\n", command);

/*
 *  If this is a bus-mastering device, set its latency timer to something sane
 */

#define DEFAULT_PCI_MASTER_LATENCY 32

    if (command & PCI_COMMAND_MASTER) {
	latency = pcicfgrb(busno, device->slot, device->function,
			   PCI_LATENCY_TIMER);
	if (latency != DEFAULT_PCI_MASTER_LATENCY) {
#ifdef PCIDEBUG
	    mobo_logf(LOG_DBG
		      "PCI: fixing latency from %d to %d for %x:%x\n",
		      latency, DEFAULT_PCI_MASTER_LATENCY,
		      device->vendor_id, device->device_id);
#endif
	    pcicfgwb(busno, device->slot, device->function,
		     PCI_LATENCY_TIMER, DEFAULT_PCI_MASTER_LATENCY);
	}
    }
}



/**************************************************************************
 * PCIQueryDevice() - Interrogate a device for BAR resource requirements  *
 **************************************************************************/

static void PCIQueryDevice(PCIDevice_t * device)
{
    ui bar32;
    ul bar64;
    PCIBus_t *bus;
    int idx;

/*
 *  Figure out how much memory (and what sort) the given device needs.
 *  Note: this routine also looks for any BIOS option ROM.
 */
    bus = device->parent;

#ifdef TRACE_ENABLE
    mobo_logf(LOG_DBG "  QueryDevice, Bus:%d Slot:%d Func:%d\n",
	      bus->number, device->slot, device->function);
#endif	/* TRACE_ENABLE */

    for (idx = 0; idx < MAX_BASE_ADDRESS_REGISTERS + 1; ++idx) {
	pcicfgwl(bus->number, device->slot, device->function,
		 BAR_OFFSET(idx), 0xffffffff);
	bar32 =
	    pcicfgrl(bus->number, device->slot, device->function,
		     BAR_OFFSET(idx));

#ifdef TRACE_ENABLE
	mobo_logf( LOG_DBG "  Bus:%d Slot:%d Func:%d BAR:0x%02X = 0x%08X\n",
		bus->number, device->slot, device->function, BAR_OFFSET(idx), bar32 );
#endif	/* TRACE_ENABLE */
	if (bar32 == 0)
	    continue;
	if ((int) bar32 > 0)		/* ie, MSB not set, makes bar32 +ve */
	    continue;
	TRACE("\tBar:%x, %x\n", BAR_OFFSET(idx), bar32);

	device->PCI_Bar[idx].Type = bar32 & 0xF;	/* BAR config options */
/*
 *  We've read the base address register back after writing all ones
 *  and so now we must decode it.
 */
	if ( (bar32 & TYPE_IO) && (idx < MAX_BASE_ADDRESS_REGISTERS) ) {

 	    /* I/O space address register. */
	    bar32 = bar32 & 0xFFFFFFFE;
	    device->PCI_Bar[idx].Size = ~bar32 + 1;
	    device->PCI_Bar[idx].Reg = BAR_OFFSET(idx);

	    TRACE("\tI/O Reg:%x, Size:%x\n",
		     device->PCI_Bar[idx].Reg, device->PCI_Bar[idx].Size);

	} else if( bar32 & TYPE_MEM_64BIT )  {		/* 64-bit memory */

/*
 *  A 64 bit wide memory address occupies this BAR and the next one.
 */
#ifdef TRACE_ENABLE
	    mobo_logf( LOG_DBG "PCI: 64-bit address request from %d:%d:%d\n",
		device->parent->number, device->slot, device->function );
#endif

	    /* query the next bar to get the upper 32 bits */
	    pcicfgwl(bus->number, device->slot, device->function,
		     BAR_OFFSET(idx+1), 0xffffffffU);

	    bar64 = (ul) pcicfgrl(bus->number, device->slot,
			device->function, BAR_OFFSET(idx+1) );

	    bar64 = (bar64 << 32) | (bar32 & ~0xFU);

	    device->PCI_Bar[idx].Size = ~bar64 + 1;
	    device->PCI_Bar[idx].Reg = BAR_OFFSET(idx);

	    TRACE("\tMem Reg (64-bit address):%lx, Size:%lx\n",
		     device->PCI_Bar[idx].Reg,
		     device->PCI_Bar[idx].Size);

	    idx++;
	    device->PCI_Bar[idx].Reg = BAR_OFFSET(idx);
	    device->PCI_Bar[idx].Size = 0;	/* makes it get ignored */

	} else {		/* 32-bit memory by process of elimination */
				/* note, option ROMs are also laid out here */

	    bar32 = bar32 & 0xfffffff0;
	    device->PCI_Bar[idx].Size = ~bar32 + 1;
	    device->PCI_Bar[idx].Reg = BAR_OFFSET(idx);
	    /* note - mask out IO bit, ROMs are memory mapped by default */
	    device->PCI_Bar[idx].Type = bar32 & 0xE;
	    TRACE("\tMem Reg:%x, Size:%x\n",
		     device->PCI_Bar[idx].Reg,
		     device->PCI_Bar[idx].Size);
	}
    }
}


/**************************************************************************
 * PCIInsertDevice()                                                      *
 **************************************************************************/
/*
 *  Insert the PCI device into the queue of devices.
 */
static void PCIInsertDevice(PCIBus_t *bus, PCIDevice_t *device)
{
#ifdef TRACE_ENABLE
  mobo_logf( LOG_DBG "  Inserting... bus:%d, device:%d, func:%d\n",
	   bus->number, device->slot, device->function);
#endif

  if (bus->devices != NULL) {
    PCIDevice_t *devp = bus->devices;
    while (devp->next != NULL)
      devp = devp->next;

    devp->next = device;
  }
  else {
/*
 *  The list is empty, so make this the first entry.
 */
    bus->devices = device;
  }

  device->next = NULL;
}


/**************************************************************************
 * PCIAllocateSpace()                                                     *
 **************************************************************************/
static void PCIAllocateSpace(PCIBus_t * bus)
{
    ui tio, bio, tmem, bmem, l, alignto;
    PCIBus_t *child;
    PCIDevice_t *device;
    int idx;

/*
 *  Don't do anything unless this bus has any devices/bridges.
 */
    if ((bus->devices != NULL) || (bus->children != NULL)) {
/*
 *  Align the current bases on appropriate boundaries (4K for IO and
 *  1Mb for memory).
 */
	bio = PCIIOBase = align(PCIIOBase, 0x4000);
	bmem = PCIMemBase = align(PCIMemBase, 0x100000);
/*
 *  Allocate space to each device.
 */
	device = bus->devices;
	while (device != NULL) {
	    if (device->bus == NULL) {	/* If this is not a bridge? */
#ifdef TRACE_ENABLE
		mobo_logf( LOG_DBG "  AllocateDevice, Hose:%d Bus:%d Slot:%d Func:%d\n",
		     bus->hose, bus->number, device->slot, device->function);
#endif
/*
 *  This is a normal device, allocate it some space.
 */
		for (idx = 0; idx < MAX_BASE_ADDRESS_REGISTERS + 1; ++idx) {

		    /* Skip BARs with no size to the allocation */
		    if (device->PCI_Bar[idx].Size == 0) 	continue;

		    if ( device->PCI_Bar[idx].Type & TYPE_IO ) {
			/* Align to multiple of size or minimum base */
			alignto = MAX(0x400, device->PCI_Bar[idx].Size);
			PCIIOBase = device->PCI_Bar[idx].Base =
			    align(PCIIOBase, alignto);
			PCIIOBase += alignto;
#ifdef TRACE_ENABLE
			mobo_logf( LOG_DBG
				"  Allocating %d:%d:%d BAR %d (IO) base 0x%X\n",
		    		device->parent->number, device->slot,
				device->function, idx,
				device->PCI_Bar[idx].Base );
#endif
		    } else {
			/* Align to multiple of size or minimum base */
			alignto = MAX(0x1000, device->PCI_Bar[idx].Size);
			PCIMemBase = device->PCI_Bar[idx].Base =
			    align(PCIMemBase, alignto);
			PCIMemBase += alignto;
#ifdef TRACE_ENABLE
			mobo_logf( LOG_DBG
				"  Allocating %d:%d:%d BAR %d (%s) base 0x%X\n",
		    		device->parent->number, device->slot,
				device->function, idx,
				(device->PCI_Bar[idx].Type & TYPE_MEM_64BIT) ?
				"64-bit Mem" : "32-bit Mem",
				device->PCI_Bar[idx].Base );
#endif
		    }
		}
/*
 *  Tell the device where the bases are.
 */
		PCISetDeviceConfig(device);
	    }
	    device = device->next;
	}
/*
 *  Allocate space for all of the sub-buses.
 */
	child = bus->children;
	child->hose = bus->hose;
	while (child != NULL) {
	    PCIAllocateSpace(child);
	    child = child->next;
	}
/*
 *  Align the current bases on 4K and 1MByte boundaries.
 */
	tio = PCIIOBase = align(PCIIOBase, 0x1000);
	tmem = PCIMemBase = align(PCIMemBase, 0x100000);
/*
 *  set up this bus's windows.
 */
	bus->PCI_IO_Size = tio - bio;
	if (bus->PCI_IO_Size != 0)
	    bus->PCI_IO_Base = bio;

	bus->PCI_Mem_Size = tmem - bmem;
	if (bus->PCI_Mem_Size != 0)
	    bus->PCI_Mem_Base = bmem;

	if (bus->bridge != NULL) {
	    PCIDevice_t *bridge = bus->bridge;
/*
 *  Set up the top and bottom of the I/O memory range for this bus.
 */
	    if (bus->PCI_IO_Size != 0) {
		l =
		    pcicfgrl(bridge->parent->number, bridge->slot,
			     bridge->function, 0x1c);
		l = (l & 0xffff0000) | (bio >> 8) | ((tio - 1) & 0xf000);
		pcicfgwl(bridge->parent->number, bridge->slot,
			 bridge->function, 0x1c, l);
	    }

	    if (bus->PCI_Mem_Size != 0) {
		l =
		    ((bmem & 0xfff00000) >> 16) | ((tmem - 1) &
						   0xfff00000);
		pcicfgwl(bridge->parent->number, bridge->slot,
			 bridge->function, 0x20, l);
	    }
/*
 *  clear status bits, enable I/O (for downstream I/O), turn on master
 *  enable (for upstream I/O), turn on memory enable (for downstream 
 *  memory), turn on master enable (for upstream memory and I/O),
 *  enable PCI system error reporting.
 */
	    pcicfgwl(bridge->parent->number, bridge->slot,
		     bridge->function, 0x4, 0xffff0107);
	}
    }
}




/**************************************************************************
 * PCIScanBus()                                                           *
 **************************************************************************/
static ui PCIScanBus(PCIBus_t * bus, ui min_dev, ui max_dev)
{
    ui i;
    ui l;
    ui max;
    int function;
    int bridge_flag;
    PCIDevice_t *device;
    PCIBus_t *child;

    max = bus->secondary;
    for (i = min_dev; i <= max_dev; i++) {

	for (function = 0; function < 8; function++) {

	    l = PCI_slot_filled(bus->number, i, function);

	    if (l == 0) {		/* empty slot */

#ifdef TRACE_ENABLE
		mobo_logf( LOG_DBG "Empty slot, Bus:%d, Slot:%d Func:%d!\n",
		       bus->number, i, function);
#endif
		break;			/* move on to next slot */
	    }

#ifdef TRACE_ENABLE
	    mobo_logf(LOG_DBG
		  "PCI: Device detected:0x%x, Bus:%d, Slot:%d Func:%d\n",
		  l, bus->number, i, function);
#endif

/*
 *  This slot (i) is filled.
 */
	    device = PCI_allocate_device();
	    device->parent = bus;
/*
 *  Put it into the simple chain of devices that is used to find 
 *  devices once everything is set up.
 */
	    device->sibling = devices;
	    devices = device;

	    device->slot = i;
	    device->function = function;
	    device->vendor_id = (uw) (l & 0xFFFF);
	    device->device_id = (uw) (l >> 16);
	    bridge_flag = FALSE;

/*
 *  Check to see if this device is a PCI-PCI bridge.
 */
	    l = pcicfgrl(bus->number, i, function, PCI_CFG_REG_REVISION_ID);
	    device->revision = (ub) (l & 0xff);
	    l = l >> 8;	/* upper 3 bytes */
	    device->class = l;
	    bridge_flag = (l == PCI_BRIDGE_CLASS);
/*
 *  Figure out just how much I/O and memory space this device needs.
 *  Note that we must have set its parent up by this time.  We don't
 *  do this for bridges.
 */
	    if (!bridge_flag)
		PCIQueryDevice(device);
/*
 *  Now insert it into the list of devices held by the parent bus.
 *  Note that we can only do this when we know how much memory and
 *  I/O it requires as the device list is in ascending memory order.
 */
	    PCIInsertDevice(bus, device);
/*
 *  Check to see if this device is a PCI-PCI bridge.
 */
#ifndef CONFIG_TINOSA	/* TINOSA debug - can't handle LDT to no-devs */
	    if (bridge_flag) {
		ui buses;
/*
 *  Insert it into the tree of buses.
 */
		child = PCI_allocate_bus();
		child->hose = bus->hose;
		bus->next = bus->children;
		bus->children = child;
		child->bridge = device;
		child->parent = bus;
		device->bus = child;
/*
 *  Set up the primary, secondary and subordinate bus numbers.
 */
		max = max + 1;
		child->number = (ub) max;
		child->primary = bus->secondary;
		child->secondary = (ub) max;
		child->subordinate = 0xFF;
/*
 *  Init address ranges putting things out of the way temporarily.
 */
		pcicfgwl(bus->number, i, function, 0x1c, 0x000000f0);
		pcicfgwl(bus->number, i, function, 0x20, 0x0000fff0);
/*
 *  Turn off downstream PF memory address range.
 */
		pcicfgwl(bus->number, i, function, 0x24, 0x0000fff0);
/*
 *  Enable Target Abort and PCI system error reporting.
 */
		pcicfgwl(bus->number, i, function, 0x3c, 0x00220000);
/*
 *  clear all status bits and turn off memory, I/O and master enables.
 */
		pcicfgwl(bus->number, i, function,
			    PCI_CFG_REG_STATUS_COMMAND, 0xffff0000 );
/*
 *  configure the bus numbers for this bridge.
 */
		buses = pcicfgrl(bus->number, i, function,
				    PCI_CFG_REG_BRIDGE_PRIMARY );
		buses = buses & 0xff000000;
		buses = buses | (ui) (child->primary) |
		    ((ui) (child->secondary) << 8) |
		    ((ui) (child->subordinate) << 16);
		pcicfgwl(bus->number, i, function, 
				    PCI_CFG_REG_BRIDGE_PRIMARY, buses);
/*
 *  scan all subordinate buses.
 */
		max = PCIScanBus(child, PCI_T1_FIRST_DEVICE,
			       PCI_T1_LAST_DEVICE);
/*
 *  Set the subordinate bus number to it's real value.
 */
		child->subordinate = (ub) max;
		buses =
		    (buses & 0xff00ffff) | ((ui) (child->subordinate)
					    << 16);
		pcicfgwl(bus->number, i, function, 
				    PCI_CFG_REG_BRIDGE_PRIMARY, buses);
	    }
#endif
/*
 *  If this is not a multi-function device move on.
 */
	    if ( !(pcicfgrb(bus->number, i, 0, PCI_HEADER_TYPE) 
		 & PCI_HEADER_TYPE_MULTI))
		break;
	}
    }
/*
 *  We've scanned the bus and so we know all about what's on the other
 *  side of any bridges that may be on this bus plus any devices.  
 *
 *  Return how far we've got finding sub-buses.
 */
    return max;
}


/*--------------------------------------------------------------------*/
/* Public interface routines */



/**************************************************************************
 * PCIInit()                                                              *
 **************************************************************************/

DBM_STATUS PCIInit(void)
{
    unsigned i;
    unsigned max;

    TRACE( "Running...\n" );
#if 0
    /* identity operation if bus is already setup */
    if ( pci_init_done )		return STATUS_SUCCESS;
#endif

/*
 *  Set initial values of variables.
 */
    PCIIOBase = 4UL << 10;
    PCIMemBase = 1UL << 30;

    VgaDeviceDetected = FALSE;


    for ( i=0, max=0; i < nb_pci_hosecount; i++, max++ ) 
    {
	mobo_logf( LOG_INFO "PCI: probing pci_hose %d...\n", i );

        /* allocate as many PCI buses as we have top-level pci_hoses */
	pci_hose[i] = PCI_allocate_bus( );
	if ( pci_hose[i] == NULL )		return STATUS_FAILURE;

	/* link the pci_hoses together */
	if ( i > 0 )
	    pci_hose[i-1]->next = pci_hose[i];

	pci_hose[i]->hose = i;
	pci_hose[i]->number = max;
	pci_hose[i]->primary = 0;
	pci_hose[i]->secondary = max;
        pci_hose[i]->subordinate = 0xFF;

	/*
	 *  And then for this PCI pci_hose, scan it and all it's sub-buses.
	 */
	max = (ub) PCIScanBus( pci_hose[i], PCI_T0_FIRST_IDSEL, PCI_T0_LAST_IDSEL);
	pci_hose[i]->subordinate = max;

	/*
	 *  Now scan this pci_hose allocating PCI memory and I/O space.
	 */
	PCIAllocateSpace( pci_hose[i] );
    }

    pci_init_done = TRUE;
    return STATUS_SUCCESS;
}


const String PCIVariant( const int hoseno, const char bridgedflag )
{
    const PCIStructure_t *P = PCIStructure + hoseno;

    /* matches the enum in include/pci.h */
    static const String PCIVariantOpts[2] = { "PCI", "AGP" };

    if ( bridgedflag == 1 )
    {
	return "Bridge device";
    }

    /* Sanity */
    if ( P->variant >= ARRAYLEN( PCIVariantOpts ) )
    {
	return "Unknown type";
    }

    return PCIVariantOpts[ P->variant ];
}
