/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* 
 * $Id:
 * Author:
 * 	Chris Gearing 14 June 1999
 * Modified for diags environment by Stig Telfer, 30 August 2000
 */

#undef TRACE_ENABLE		/* define this to enable compile-time debug */

#include "lib.h"
#include "uilib.h"
#include "i2c.h"
#include "lm75.h"
#include "platform.h"
#include "nvram.h"



/* LM75 registers: */

#define LM75_TEMP       0x00    /* Read-only */
#define LM75_CFG        0x01
#define LM75_HYST       0x02    /* deassertion threshold of overtemp shutdown */
#define LM75_OS         0x03    /* overtemperature shutdown threshold */


/* LM75 default settings: */

#define LM75_DFLT_OS    65      /* default overheating value, degrees C */
#define LM75_DFLT_CFG   0



/* Private data structure containing all the innards of an LM75 device */

typedef struct _LM75 {
    uint16 temp;        /* 2s complement, 9 bits start at bit 7, lsb=.5C */
    uint8 config;
    uint16 hyst;        /* Lower (abs) temp at which shutdown is deasserted */
    uint16 os;          /* Overtemperature Shutdown threshold temp */
} LM75_t;


/* Handy macros for getting temperature (with lsb=0.5C) */
/* Ie, in these macros we're working in units of half a degree */ 
#define LM2TEMP( x )       ((byteflip2( x )>>7) * 5 )
#define TEMP2LM( x )       byteflip2( ((x)/5) << 7)


/*------------------------------------------------------------------------*/
/* Private routines */

static uint16 byteflip2( const uint16 d )
{
    return (d & 0xFFU) << 8 | (d & 0xFF00U) >> 8;
}


static DBM_STATUS lm75_put( const uint8 a, LM75_t *L )
{
    if (
	i2cdev_write( a,
		LM75_CFG,
		sizeof( L->config ),
		&L->config ) != STATUS_SUCCESS	||
	i2cdev_write( a,
		LM75_HYST,
		sizeof( L->hyst ),
		(uint8 *)&L->hyst ) != STATUS_SUCCESS	||
	i2cdev_write( a,
		LM75_OS,
		sizeof( L->os ),
		(uint8 *)&L->os ) != STATUS_SUCCESS
	)
    {
	mobo_logf( LOG_CRIT "I2C: write error writing to LM75 at 0x%02X\n", a );
	return STATUS_FAILURE;
    }
    return STATUS_SUCCESS;
}


static DBM_STATUS lm75_get( const uint8 a, LM75_t *L )
{
    if (
	i2cdev_read( a,
		  LM75_TEMP,
		  sizeof(L->temp),
		  (uint8 *)&L->temp ) != STATUS_SUCCESS ||
	i2cdev_read( a,
		  LM75_CFG,
		  sizeof(L->config),
		  &L->config ) != STATUS_SUCCESS ||
	i2cdev_read( a,
		  LM75_HYST,
		  sizeof(L->hyst),
		  (uint8 *)&L->hyst) != STATUS_SUCCESS ||
	i2cdev_read( a,
		  LM75_OS,
		  sizeof(L->os),
		  (uint8 *)&L->os) != STATUS_SUCCESS
	)
    {
	mobo_logf(LOG_CRIT "I2C: read error reading from LM75 at 0x%02X\n", a);
	return STATUS_FAILURE;
    }
    return STATUS_SUCCESS;
}


/*------------------------------------------------------------------------*/
/* Public Interface functions */

DBM_STATUS lm75_init( uint8 a )
{
    uint16 os_temp, hyst_temp;
    String env_temp;
    LM75_t L;
    DBM_STATUS sval;

    /* Here we check NVRAM for a setting for the overheating thresholds */
    env_temp = nvenv_lookup( KEY_THERM_OS );
    if ( env_temp != NULL )
        os_temp = atoi( env_temp );
    else
        os_temp = LM75_DFLT_OS;

    env_temp = nvenv_lookup( KEY_THERM_HYST );
    if ( env_temp != NULL )
        hyst_temp = atoi( env_temp );
    else
        hyst_temp = os_temp - 5;        /* default hysteresis of 5 C less */

    hyst_temp *= 10, os_temp *= 10;       /* Work in units of 0.1 degrees */
    L.config = LM75_DFLT_CFG;
    L.os = TEMP2LM( os_temp );
    L.hyst = TEMP2LM( hyst_temp );
    sval = lm75_put( a, &L );
    if ( sval != STATUS_SUCCESS )
    {
        mobo_logf( LOG_CRIT "LM75: Initialisation failure of LM75 at 0x%02X\n",
                a );
        return STATUS_FAILURE;
    }

    return STATUS_SUCCESS;
}


/* Read the temp, OS & hyst CSRs */

DBM_STATUS lm75_read( uint8 a, i2c_therm_t *T )
{
    LM75_t L;
    DBM_STATUS sval;

    sval = lm75_get( a, &L );
    if( sval != STATUS_SUCCESS )
	return STATUS_FAILURE;

    TRACE( "got raw data 0x%04X 0x%04X 0x%04X\n", L.temp, L.os, L.hyst );

    T->temp = LM2TEMP( L.temp );
    T->max = LM2TEMP( L.os );
    T->hyst = LM2TEMP( L.hyst );

    return STATUS_SUCCESS;
}


/* Write the hyst, OS CSRs (temp obviously is read-only and cfg not touched) */

DBM_STATUS lm75_write( uint8 a, i2c_therm_t *T )
{
    return STATUS_FAILURE;		/* not yet implemented */
}


