/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Rudimentary ANSI escape-sequence grammar parser for the local VGA
 * enabling us to use whizz-bang colours and other magic on serial links
 * and the local screen to the same effect */
/* Begun by Stig Telfer, Alpha Processor Inc, 3 June 1999 */
/*--------------------------------------------------------------------*/

#include "lib.h"			/* external interface for VGA */
#include "uilib.h"
#include "vga.h"
#include "vga_console.h"		/* internal interface for VGA */
#include "ctype.h"

#define MAX_PARAMS	4
#define MAX_ANSI_LENGTH	16


/*--------------------------------------------------------------------*/
/* Global variables */

/* When parsing an ANSI code, divert from putchar on this predicate */
unsigned parser_active = 0;


/*--------------------------------------------------------------------*/
/* private functions */


/* Returns whether the current character matches a termination char */
static unsigned is_term( int c )	 { return isalpha( c ); }


/* ANSI colour management commands:
 * 1	- add bold to previous attribute
 * 5	- add blink to previous attribute
 * Foreground:				    Background:
 * 30		-	black		-	40
 * 31		- 	red		-	41
 * 32		- 	green		-	42
 * 33		- 	brown		-	43
 * 34		-	blue		-	44
 * 35		- 	magenta		-	45
 * 36		-	cyan		-	46
 * 37		- 	grey		-	47
 * 
 * VGA 16-colour palette (typical settings):
 *  0 Black 		 8 Dark gray
 *  1 Blue 		 9 Light blue
 *  2 Green 		10 Light green
 *  3 Cyan 		11 Light cyan
 *  4 Red 		12 Light red
 *  5 Magenta 		13 Light magenta
 *  6 Brown 		14 Yellow
 *  7 Light gray 	15 White
 */

void vga_attrib( unsigned *P, unsigned n )
{
    unsigned i, val;

    for ( i = 0; i < n; i++ )
    {
	val = *P++;

	if ( val >= 30 && val <= 37 ) {		/* foreground colour */
	    vgafg = val - 30;
	    if ( *P == 1 )		vgafg+=8, P++, i++;
	}

	if ( val >= 40 && val <= 47 ) {		/* background colour */
	    vgabg = val - 40;
	    if ( *P == 1 )		vgabg+=8, P++, i++;
	}
    }

    vgasetcol();
}


/* perform tokenization of an ANSI escape sequence.  The required semantics
 * are a code letter and a list of parameters.  Actually, we're very sloppy
 * here since we don't check the square bracket, or the semicolons */

static int perform_tokenization( char *B, unsigned *P )
{
    unsigned nparams = 0;

    while ( !is_term( *B ) ) 		/* still input to consume */
    {
	if ( !isdigit( *B ) ) {		/* some delimiter character */
	    B++;
	    continue;			/* ignored... */
	}

	sscanf( B, "%d", P );		/* swallow a new parameter */
	P++, nparams++;
	while ( isdigit( *B ) )  B++;	/* index on to next token */
    }
    return nparams;
}


/* perform the action identified */
static void perform_ansi_action( int a, unsigned *P, unsigned n )
{
    switch ( a ) { 
	case 'A': vgarow -= P[0]; break;		/* cursor upwards */
	case 'B': vgarow += P[0]; break;		/* cursor downwards */
	case 'D': vgacol -= P[0]; break;		/* cursor backwards */
	case 'H': vgarow  = P[0]-1, vgacol=P[1]-1;	/* to 0-based coords */
						break;	/* cursor position */
	case 'J': vgaerase(); break;			/* clear screen */
	case 'm': vga_attrib( P, n ); break;		/* attribute colours */
    }

    /* since most actions involve cursor movement, reset cursor here */
    if ( vgarow >= NROW )       vgarow = NROW - 1;
    if ( vgarow < 0 )           vgarow = 0;
    if ( vgacol >= NCOL )       vgacol = NCOL - 1;
    if ( vgacol < 0 )           vgacol = 0;
    vgasetloc();
}


/*--------------------------------------------------------------------*/
/* Module entry point */

void ansi_parser( int c )
{
    static int nchars = 0;
    static char buf[ MAX_ANSI_LENGTH ];
    unsigned params[ MAX_PARAMS ], nparams;

    /* if parser isn't already active, we're at the start of a sequence */
    if ( !parser_active ) {
	parser_active = 1;
	nchars = 0;
	return;				/* don't bother keeping the esc char */
    }

    /* swallow input until we reach a termination character */
    if ( nchars == MAX_ANSI_LENGTH )	/* too big!! */
    {
	parser_active = 0;
	return;
    }
    buf[nchars++] = c;

    if (is_term( c )) {
	parser_active = 0;
	nparams = perform_tokenization( buf, params );
	perform_ansi_action( c, params, nparams );
    }
}
