/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Alpha Diagnostics NVRAM environment (in SRM-console compatible format) */

#undef TRACE_ENABLE		/* define this for verbose debug */

#include "lib.h"
#include "uilib.h"
#include "platform.h"
#include "cmos_rtc.h"
#include "rom.h"
#include "nvram.h"

/*------------------------------------------------------------------------*/
/* Definitions for the nvram environment */


/* Standard Console-format (SRM) NVRAM environment implementation */

#define CNS_EV_LEN	2048		/* console standard NVRAM size */
#define CNS_VERSION	6
#define CNS_MAXVARLEN	256
#define CNS_MAXVARS	256


typedef struct _CNS_NVRAM {
    uint16 checksum;
    uint16 version;
    unsigned char text[CNS_EV_LEN - sizeof(uint16) - sizeof(uint16)];
} CNS_NVRAM, *PCNS_NVRAM;

typedef struct _SRMENV_T {
        unsigned short length;
        char name[ CNS_MAXVARLEN ];
} CNS_ENTRY, *PCNS_ENTRY;


/* Conversion between console-format NVRAM and diags-format envrionment */
static nvenv_t *cns2kvp( PCNS_NVRAM );
static PCNS_NVRAM kvp2cns( nvenv_t *);

static unsigned short cns_checksum( PCNS_NVRAM );
static PCNS_ENTRY cns_first( PCNS_NVRAM P );
static PCNS_ENTRY cns_next( PCNS_ENTRY E );
static String cns_key( PCNS_ENTRY E );
static String cns_val( PCNS_ENTRY E );
static int cns_valid( PCNS_ENTRY E );
static PCNS_NVRAM cns_new( void );
static nvenv_t *kvp_new( String key, String val, nvenv_t *next );

static PCNS_NVRAM cns_nvenv = NULL;
static nvenv_t *kvp_nvenv = NULL;


/*----------------------------------------------------------------------------*/
/* Conversions between console-format NVRAM (but memory-resident) and nvenv */

static nvenv_t *cns2kvp( PCNS_NVRAM P )
{
    nvenv_t *K = NULL, *root = NULL; 
    PCNS_ENTRY E;

    for ( E = cns_first( P ); cns_valid( E ); E = cns_next(E) )
    {
	/* Another valid variable found */
        TRACE("Digesting variable %s=%s\n", cns_key(E), cns_val(E));

	K = kvp_new( cns_key( E ), cns_val( E ), root );
	if ( K == NULL )
	{
	    mobo_logf( LOG_CRIT "Couldn't find resource for new variable!\n" );
	    return root;
	}
	root = K;		/* new entry goes to the head of the list */
    }
    return root;
}

static PCNS_NVRAM kvp2cns( nvenv_t *K )
{
    PCNS_NVRAM P;
    PCNS_ENTRY E;
    CNS_ENTRY tmp;
    unsigned nb=0, keylen, vallen, len;

    P = cns_new();
    if ( P == NULL )
	return NULL;

    /* Write out the console-format NVRAM using the macros to keep us at the
     * end of what is currently written */
    for( E = cns_first( P ); K != NULL; E = cns_next(E), K=K->next )
    {
	TRACE("Converting KVP %s=%s\n", K->key, K->val );
	keylen = strlen(K->key) + 1;
	vallen = strlen(K->val) + 1;
	len = sizeof( E->length ) + keylen + vallen;
	nb += len;
	if ( nb > sizeof( P->text ) )
	{
	    TRACE( "Environment has overflowed the allocated region!\n" );
	    P->checksum = cns_checksum( P );		/* tidy up & quit */
	    return P;
	}

	/* Note: we must be careful to avoid misaligned accesses, which are 
	 * unrecoverable in the diags.  To do this we fill out a local copy
	 * and then do a byte copy of that into the CNS record */

	sprintf_dbm( tmp.name, K->key );
	sprintf_dbm( tmp.name + keylen, K->val );
	tmp.length = len;
	memcpy( E, &tmp, len );

	TRACE( "Writing %s=%s as %d bytes\n", K->key, K->val, len );
    }

    P->checksum = cns_checksum( P );
    return P;
}


/*----------------------------------------------------------------------------*/
/* Cleanup and release of heap data */

static void kvpfree( nvenv_t *K )
{
    nvenv_t *K_next;

    TRACE( "Running\n" );

    while( K != NULL )
    {
	TRACE( "Freeing %s=%s\n", K->key, K->val );

	K_next = K->next;
	free( K->key );
	free( K->val );
	free( K );

	K = K_next;
    }

    TRACE( "Done\n" );
}

static void cnsfree( PCNS_NVRAM P )
{
    if ( P == NULL )
	return;
    free( P );
}


/*----------------------------------------------------------------------------*/
/* Console-format utilities */

/* Return a pointer to the first entry */
static PCNS_ENTRY cns_first( PCNS_NVRAM P )
{
    return (PCNS_ENTRY)P->text;
}

/* Return a pointer to the next entry, making an aligned copy of it also */
static PCNS_ENTRY cns_next( PCNS_ENTRY E )
{
    CNS_ENTRY A;
    memcpy( &A, E, sizeof( CNS_ENTRY ) );		/* aligns it */
    E = (PCNS_ENTRY)( (char *)E + A.length );
    return E;
}

/* Return a pointer to the key */
static String cns_key( PCNS_ENTRY E )
{
    return E->name;
}

/* Return a pointer to the value */
static String cns_val( PCNS_ENTRY E )
{
    return E->name + strlen( E->name ) + 1;
}

/* Return whether the record is valid */
static int cns_valid( PCNS_ENTRY E )
{
    CNS_ENTRY A;
    memcpy( &A, E, sizeof( CNS_ENTRY ) );		/* aligns it */
    return (A.length != 0);
}


static PCNS_NVRAM cns_new( void )
{
    PCNS_NVRAM P;

    P = malloc( CNS_EV_LEN );
    if ( P == NULL )
    {
	mobo_logf( LOG_CRIT "malloc failed on creating of environment!\n" );
	return NULL;
    }
    memset( P, 0, CNS_EV_LEN );		/* clean the heap allocation */
    P->version = CNS_VERSION;
    P->checksum = cns_checksum(P);
    return P;
}

static unsigned short cns_checksum( PCNS_NVRAM P )
{
    unsigned short checksum;
    int i;
    unsigned short *p = (unsigned short *)P;

    checksum = 0;
    p++;

    for (i = 1; i < (CNS_EV_LEN / sizeof(*p)); i++)
	checksum += *p++;

    TRACE("nvenv Checksum %x\n", checksum);
    return (checksum);
}


static DBM_STATUS cns_get( PCNS_NVRAM P )
{
    int i;
    rom_t *rom_info = rom_search( FW_NVENV, plat_rom_rsvd );

    if (rom_info == NULL) {
	TRACE("No rom_info\n");
	return (STATUS_FAILURE);
    }
    TRACE("%dKB of environment allocated at 0x%X\n",
	  rom_info->rsize >> 10, rom_info->rstart);

    for (i = 0; i < sizeof(CNS_NVRAM); i++)
	((char *) P)[i] = plat_inromb(rom_info->rstart + i);

    /* Validation: compare checksums */
    if ( P->version != CNS_VERSION || P->checksum != cns_checksum(P) )
    {
	mobo_logf(
	    LOG_WARN
		"Console-format checksum & version are %x,%x should be %x,%x\n"
	    LOG_WARN
		"The NVRAM contents are being discarded\n",
		P->checksum, P->version, cns_checksum( P ), CNS_VERSION);
	memset( P, 0, CNS_EV_LEN );
	P->version = CNS_VERSION;
	P->checksum = cns_checksum( P );
	return STATUS_WARNING;
    }
    return STATUS_SUCCESS;
}


static DBM_STATUS cns_put( PCNS_NVRAM P )
{
    int i;
    rom_t *rom_info = rom_search( FW_NVENV, plat_rom_rsvd );

    if (rom_info == NULL) {
	TRACE("No rom_info\n");
	return (STATUS_FAILURE);
    }

    TRACE("Writing %dKB of environment allocated at 0x%X\n",
	  rom_info->rsize >> 10, rom_info->rstart);

    plat_romerase(rom_info->astart, rom_info->astart + rom_info->asize - 1, NULL, 0);

    for (i = 0; i < sizeof(CNS_NVRAM); i++)
	plat_outromb(rom_info->rstart + i, ((char *) P)[i]);

    return (STATUS_SUCCESS);
}


/* This function takes care to create private heap copies of anything that gets
 * put onto the kvp linked list.  Not doing so would cause internal
 * inconsistency...
 */
static nvenv_t *kvp_new( String key, String val, nvenv_t *next )
{
    nvenv_t *K = malloc( sizeof( nvenv_t ) );

    if ( K == NULL )
    {
	mobo_logf( LOG_CRIT "malloc failed on reading of environment!\n" );
	return NULL;
    }

    /* Copy over the fields for the new key-value pair */
    K->key = malloc( strlen( key ) + 1 );	/* + null char */
    K->val = malloc( strlen( val ) + 1 );	/* + null char */
    if ( K->key == NULL || K->val == NULL )
    {
	mobo_logf( LOG_CRIT "couldn't parse environment - no more heap\n" );
	K->next = NULL;
	kvpfree( K );
	return NULL;
    }
    strcpy( K->key, key );
    strcpy( K->val, val );
    K->next = next;
    return K;
}


/*----------------------------------------------------------------------------*/
/* nvenv environment public functions */

static uint8 init_done = FALSE;

DBM_STATUS nvenv_init(void)
{
    DBM_STATUS sval;

    TRACE( "Running\n" );

    TRACE( "Freeing old cns_nvenv = 0x%lx\n", cns_nvenv );
    cnsfree( cns_nvenv );
    TRACE( "Making new cns_nvenv\n" );
    cns_nvenv = cns_new();
    if ( cns_nvenv == NULL )
	    return STATUS_FAILURE;

    TRACE("Getting environment in console format\n");
    sval = cns_get( cns_nvenv );

    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "Couldn't read NVRAM from firmware\n" );
	cnsfree( cns_nvenv );
	return STATUS_FAILURE;
    }

    /* Build a diags format (key/value pair) data structure by parsing */
    TRACE("Digesting freshly read console-format NVRAM\n" );
    kvpfree( kvp_nvenv );
    kvp_nvenv = cns2kvp( cns_nvenv );

    TRACE("nvenv Success\n");
    init_done = TRUE;
    return STATUS_SUCCESS;
}


String nvenv_lookup(const String key)
{
    nvenv_t *K;
    DBM_STATUS sval;

    TRACE( "running... looking up %s\n", key );
    if ( init_done != TRUE )
    {
	TRACE( "Calling init\n" );
	sval = nvenv_init();
	if ( sval == STATUS_FAILURE )
	    return NULL;
    }

    for ( K=kvp_nvenv; K != NULL; K=K->next )
    {
	TRACE( "Comparing %s with %s\n", key, K->key );
	if ( strcmp( K->key, key ) == 0 )
		return K->val;
    }
    return NULL;
}


/* FIXME: Some more careful thought might be needed here about who allocates
 * what; for example, should we take the pointers the user passed in and put
 * them into our own data structures, or should we copy the value across to
 * a buffer allocated by us?
 * 
 * Policy: we're going to make a private copy of whatever the user passes,
 * so that they can use stack or initialised data regions and not worry.
 */

DBM_STATUS nvenv_write(const String key, const String val )
{
    nvenv_t *K;
    DBM_STATUS sval;

    if ( init_done != TRUE )
    {
	sval = nvenv_init();
	if ( sval == STATUS_FAILURE )
	    return sval;
    }

    /* Does it exist already?  If so, remove it for replacement with new. */
    nvenv_remove( key );


    /* Create a new key-value pair, putting it at the head of the list */
    /* kvp_new creates a private heap-space copy of all strings so we're OK */
    TRACE( "Creating variable %s=%s\n", key, val );
    K = kvp_new( key, val, kvp_nvenv );
    if ( K == NULL )
    {
	mobo_logf( LOG_CRIT "no heap space for new NVRAM variable!\n" );
	return STATUS_FAILURE;
    }
    kvp_nvenv = K;


    /* Convert to console-format NVRAM */
    cnsfree( cns_nvenv );
    cns_nvenv = kvp2cns( kvp_nvenv );

    /* Write to ROM */
    cns_put( cns_nvenv );
 
    return (STATUS_SUCCESS);
}


/* Remove a variable from NVRAM */ 
DBM_STATUS nvenv_remove( const String key )
{
    nvenv_t *K, *Kprev;

    for ( K=kvp_nvenv, Kprev=NULL; K != NULL; K=K->next )
    {
	/* Is this the key-value pair we're looking for? */
        if ( strcmp( K->key, key ) == 0 )
        {
	    /* Unlink from the list */

	    if ( Kprev == NULL )	/* Are we looking at the first entry? */
		kvp_nvenv = K->next;
	    else			/* Otherwise patch up previous link */
		Kprev->next = K->next;

		
	    /* Deallocate */
	    free( K->key );
	    free( K->val );
	    free( K );

	    /* Done */
            break;
        }

	Kprev = K;
    }

    /* Convert to console-format NVRAM */
    cnsfree( cns_nvenv );
    cns_nvenv = kvp2cns( kvp_nvenv );

    /* Write to ROM */
    cns_put( cns_nvenv );
 
    return STATUS_SUCCESS;
}




nvenv_t *nvenv_nextentry(nvenv_t *p)
{
    DBM_STATUS sval;

    if ( init_done != TRUE )
    {
	sval = nvenv_init();
	if ( sval == STATUS_FAILURE )
	    return NULL;
    }

    if (p == NULL)		// Start at the start
	p = kvp_nvenv;
    else			// Step to the next name.
	p = p->next;

    TRACE("At entry %x\n", p);

    return p;
}

