#define TRACE_ENABLE

#include "lib.h"
#include "uilib.h"
#include "specifics.h"
#include "platform.h"
#include "southbridge.h"
#include "northbridge.h"
#include "i2c/adm9240.h"

#define TIMEOUTVALUE    25000


/* Fudging of prototypes to get local scoping */

static DBM_STATUS i2cdev_init( void );
static DBM_STATUS i2cdev_read( const uint8 node, const int cmd,
			const size_t bytes, uint8 *data);

static DBM_STATUS i2cdev_write( const uint8 node, const int cmd,
			 const size_t bytes, const uint8 *data);
static DBM_STATUS plat_i2cinit( void );

static DBM_STATUS plat_i2c_select( const I2Cbus_t *I );

static void smb_dump( void );

/* Test applet */
int main( int argc, char *argv[] )
{
    DBM_STATUS sval;
    uint8 rdata;

    mobo_cls();
    TRACE( "Applet for I2C testing\n" ); 

    /* Try to init */
    TRACE( "Attempting init...\n" );
    sval = plat_i2cinit( );
    TRACE( "Init %s\n", sval == STATUS_SUCCESS ? "Passed :-)" : "FAILED" );

#if 0
    /* Try to read the mux */
    TRACE( "Attempting read of reg...\n" );
    rdata = 0xFF;
    sval = i2cdev_read( SHARK_I2C_REG0, -1, 1, &rdata );
    TRACE( "Read %s, got 0x%02X\n", 
	sval == STATUS_SUCCESS ? "passed :-)" : "FAILED", rdata );
	
#endif
    
    TRACE( "All done, press any key to return\n" );
    mobo_key( 0 );
    return STATUS_SUCCESS;
}


/*----------------------------------------------------------------------*/
/* Southbridge I2C and System Management Bus support */

#define ALI_BUGGY_DELAY		100		/* breather period in usec */

/* Operations relating to the IO-space-mapped SMBus registers */
static uint32 smb_base_reg = 0;
#define SMB_READ( reg )		inportb( smb_base_reg + (reg) )
#define SMB_WRITE( reg, val )	outportb( smb_base_reg + (reg), val )


static void i2c_error_analyse( const uint8 status )
{
    int i;
    static const String interp[] = {
	"Host Slave Busy (slave is going to receive a command)",
	"Host Slave Status (SMI raised by slave interface)",
	"SMBus is at idle status",
	"Host Busy (SMBus host controller is completing cmd)",
	"SMI asserted by SMBus controller (after command)",
	"SMI asserted due to the generation of an error on I2C bus",
	"I2C Bus collision occurred (or no acknowledge from target)",
	"Terminated bus transaction in response to ABORT command"
    };

    mobo_logf( LOG_CRIT "ALI M1543C SMBus error analysis (0x%02X):\n",
		status );

    for ( i=0; i<8; i++ )
    {
	if ( status & (1<<i) )
	    mobo_logf( LOG_CRIT "  %s\n", interp[ i ] );
    }

    /* Attempt an i2c bus reset to clear any error status */
    mobo_logf( LOG_WARN "I2C: Attempting bus reset...\n" );

    SMB_WRITE( SMBUS_SMBCMD, SMBUS_SMBCMD_TERMINATE ); 
    msleep(200);				/* let things calm down again */
    SMB_WRITE( SMBUS_SMBCMD, SMBUS_SMBCMD_T_OUT_CMD ); 
    msleep(200);				/* let things calm down again */

    /* Clear any outstanding errors */
    SMB_WRITE( SMBUS_SMBSTS, 0xFF );
    mobo_logf( LOG_CRIT "SMBus status register after clearing was 0x%02X\n", 
	SMB_READ( SMBUS_SMBSTS ) );
}


/* Wait until the I2C bus is idle */

#define TIMEOUTVAL 100000			/* in usec */

static DBM_STATUS i2c_wait_ready( void )
{
    int n;
    int status, lastStatus;

    /* First, clear old status bits */
    SMB_WRITE( SMBUS_SMBSTS, 0xFF );

    /* Wait for Bus Idle. */
    for ( n=0, status=0, lastStatus=-1; n < TIMEOUTVAL; n+= ALI_BUGGY_DELAY )
    {
	/* Read status. */
	status = SMB_READ( SMBUS_SMBSTS );

	/* Debug */
	if (status != lastStatus)
	{
	    lastStatus = status;
	}

	if ( status & SMBUS_SMBSTS_IDL_STS )
		return STATUS_SUCCESS;

	/* DEBUG: ALI SMBus controller needs a breather? */
	usleep( ALI_BUGGY_DELAY );
    }

    /* Timeout condition */
    mobo_logf( LOG_CRIT "ALI M1543C SMBus is locked up busy.\n");
    i2c_error_analyse( status );
    return STATUS_FAILURE;
}


/* Wait until a transaction has been completed */

static DBM_STATUS i2c_wait_complete( void ) 
{
    int n;
    uint8 status, lastStatus;

    for ( n=0, status=0, lastStatus=0xFF; n < TIMEOUTVAL; n += ALI_BUGGY_DELAY )
    {
        /* Read status. */
        status = SMB_READ( SMBUS_SMBSTS );

	/* DEBUG - ALI controller may need short breather before ready? */
	usleep( ALI_BUGGY_DELAY );

#if 0
        /* Debug */
        if (status != lastStatus)
        {
            lastStatus = status;
            TRACE("status=%x\n", status);
        }
#endif

	/* Check for presence of any errors */
	if ( status & SMBUS_SMBSTS_ERROR )
	{
	    i2c_error_analyse( status );
	    return STATUS_FAILURE;
	}

        if ( status & SMBUS_SMBSTS_SMI_I_STS )	/* completed successfully */
	{
	    SMB_WRITE( SMBUS_SMBSTS, 0xFF );
	    return STATUS_SUCCESS;
	}
    }

    /* Timeout condition */
    mobo_logf( LOG_CRIT "ALI M1543C SMBus timed out on transaction.\n");
    i2c_error_analyse( status );
    return STATUS_FAILURE;
}


/* The ALI M1543C southbridge supports one i2c bus interface.  No knowledge
 * of device target addresses, wiring, multiplexers etc is in here.  Just the
 * reading and writing of a sequence of bytes...
 */

static DBM_STATUS i2cdev_init( void )
{
    uint16 val16;
    uint8 rdata;
    int oldrdata;
    int i, j;

    TRACE( "Looking at ALI SMB config...\n");


    /* Enable the PMU to respond to IO space transactions */
    /* NOTE: should I do this and what if the PMU doesn't respond to IO? */
    val16 = pcicfgrw( 0, PLAT_PMUID, 0, PMU_CMD );
    if ( (val16 & 1) == 0 )
    {
	mobo_logf( LOG_CRIT "PMU was not enabled for IO transactions!\n" );
	val16 |= 1;
	pcicfgww( 0, PLAT_PMUID, 0, PMU_CMD, val16 );
    }

    smb_base_reg = pcicfgrl( 0, PLAT_PMUID, 0, PMU_SBA ) & ~0x3U;
    TRACE("smb_base_reg=%x\n", smb_base_reg);

    pcicfgwb( 0, PLAT_ISAID, 0, 0x77, 0x09 );
    TRACE( "SMBus interrupt routing: 0x%02X\n", pcicfgrb( 0, PLAT_ISAID, 0, 0x77 ) );

    swpipl( 0 );
    plat_intclr();
    plat_nmiclr();
    plat_intsetup( 1 );

    /* Enable SMB Bus Controller by setting host and clearing slave bits */
    rdata = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHSI );
    rdata = (rdata & ~PMU_SMBHSI_SLAVE_EN) | PMU_SMBHSI_HOST_EN;
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHSI, rdata );

#if 0
    /* SRM magic: write 0x02 to SMB HSC (slave address) FIXME - need this? */
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHSC, 0x02 );
#endif

    /* Set speed to 74K with IDLE detect as BaseClk*64. */
    pcicfgwb( 0, PLAT_PMUID, 0, PMU_SMBHCBC, 0x30 /* SOOHOON MAGIC */
		/* PMU_SMBHCBC_74K | PMU_SMBHCBC_DL64 */ );

    smb_dump();

    TRACE( "Soohoon magic: 0x04 = 0x%X, 0x5B = 0x%X\n",
	pcicfgrb( 0, PLAT_PMUID, 0, 0x04 ),
	pcicfgrb( 0, PLAT_PMUID, 0, 0x5B ) );

#if 0
    /* Reset host controller (in case it was in the middle of something) */
    SMB_WRITE( SMBUS_SMBCMD, SMBUS_SMBCMD_TERMINATE );
    usleep( 500 );
#endif


    for( j=0; j<4; j++ )
    {
	if ( j % 2 != 0 )
	{
	    TRACE( "Reseting host controller\n" );
	    SMB_WRITE( SMBUS_SMBCMD, SMBUS_SMBCMD_TERMINATE );
	    usleep( 50 );
	}

	if ( j > 2 )
	{
	    TRACE( "Reseting SMBus\n" );
	    SMB_WRITE( SMBUS_SMBCMD, SMBUS_SMBCMD_T_OUT_CMD );
	    usleep( 50 );
	}


	/* Clear status and wait for idle */
	TRACE( "Attempting to clear status\n" );
	SMB_WRITE( SMBUS_SMBSTS, 0xFF );
	usleep( 20 );

	for( i=0, oldrdata = -1; i<TIMEOUTVALUE; i++ )
	{
	    usleep( 10 );
	    rdata = SMB_READ( SMBUS_SMBSTS );

	    if ( oldrdata != (int)rdata )
	    {
		TRACE( "Status (%03d ms): 0x%02X\n", i, rdata );
		oldrdata = rdata;
	    }

#if 1
	    /* Ack any errors in the status reg */
	    if ( rdata & ~SMBUS_SMBSTS_IDL_STS )
	    {
		SMB_WRITE( SMBUS_SMBSTS, 0xFF );
	    }
#endif

	    if ( rdata & SMBUS_SMBSTS_IDL_STS )
		break;
	}

	if ( rdata & SMBUS_SMBSTS_IDL_STS )
	    break;
    }

    TRACE( "After PMU setup:\n" );
    smb_dump();

    if ( i==TIMEOUTVALUE )
    {
	TRACE( "Error: I2C timed out in init, final status of 0x%02X\n", rdata);
	return STATUS_FAILURE;
    }

    TRACE( "After PMU setup:\n" );
    smb_dump();
    return STATUS_SUCCESS;

#if 0
    /* Try sending a timeout. */
    SMB_WRITE( SMBUS_SMBCMD, SMBUS_SMBCMD_T_OUT_CMD );
    msleep( 500 );
    TRACE( "After Timeout Command:\n" );
    smb_dump();

    SMB_WRITE( SMBUS_SMBCMD, SMBUS_SMBCMD_TERMINATE );
    msleep( 500 );
    TRACE( "After Reset Command:\n" );
    smb_dump();

    SMB_WRITE( SMBUS_SMBSTS, 0x80 );
    msleep( 500 );
    TRACE( "After Clear Command:\n" );
    smb_dump();

    SMB_WRITE( SMBUS_SMBSTS, 0x20 );
    msleep( 500 );
    TRACE( "After Second Clear Command:\n" );
    smb_dump();

    return STATUS_SUCCESS;
#endif
}

static void smb_dump( void )
{
    uint8 rdata;
    /* Paranoia: can we get sense out of the SMBus registers now? */
    rdata = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHSI );
    TRACE( "HSI (0x%02X) reads:  0x%02X\n", PMU_SMBHSI, rdata );

    rdata = pcicfgrb( 0, PLAT_PMUID, 0, PMU_SMBHCBC );
    TRACE( "HCBC (0x%02X) reads: 0x%02X\n", PMU_SMBHCBC, rdata );


    /* Take a look at the SMB controller IO space registers */
    rdata = SMB_READ( SMBUS_SMBSTS );
    TRACE( "  SMBSTS reads:    0x%02X\n", rdata );

    rdata = SMB_READ( SMBUS_ADDR );
    TRACE( "  ADDR reads:      0x%02X\n", rdata );
} 




/* I2C bus read.  The bus numberings used here start at zero.
 * Read a byte from the i2c bus target specified */

/* node = I2C bus address of target
 * cmd = goes into SMBus command byte, often used by targets as a reg index
 * 	(set to -1 if not required)
 * bytes = number of bytes to be read from target
 * data = storage of read data
 */


static DBM_STATUS i2cdev_read( const uint8 node, const int cmd,
			const size_t bytes, uint8 *data)
{
    static unsigned last_byte_read = ~0;

    TRACE( "I2C bus read of %d bytes from node %x cmd %02x\n",
		bytes, node, cmd  );

    if ( i2c_wait_ready() != STATUS_SUCCESS )
	return STATUS_FAILURE;

    /* Set address.  Adding LSB=1 signifies read request */
    SMB_WRITE( SMBUS_ADDR, node | 0x01 );

    /* Set command. */
    if (cmd == -1 )
    {
	/* NOTE: here we only support 1-byte transfer requests */

	/* Set Send/Receive byte. */
	SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_SNDRCV );

    } else {

	/* NOTE: here we only support 1-byte or 2-byte transfer requests */

	if (bytes == 1)				/* Set Read/Write byte. */
	    SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_WRRDB );
	else					/* Set Read/Write word. */
	    SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_WORD );

	/* Set command register, interpreted by targets in various ways. */
	SMB_WRITE( SMBUS_CMD, cmd );

    }

    /* Start I2C transaction. */
    SMB_WRITE( SMBUS_START, 0xFFU );


    /* Wait for command completion. */
    if ( i2c_wait_complete() != STATUS_SUCCESS )
	return STATUS_FAILURE;


    /* Read back the requested byte(s) */
    data[0] = SMB_READ( SMBUS_DATA_A );
    if ( bytes == 2 ) 
	data[1] = SMB_READ( SMBUS_DATA_B );	/* high byte in here */

    /* ALI problem - a byte can get duplicated and read the next transaction */
    if ( last_byte_read != data[0] )
    {
	last_byte_read = (unsigned)data[0];
    } else {
	mobo_logf( LOG_WARN 
		"I2C: duplicated byte on read from device at 0x%02X\n",
		node );
	
	data[0] = SMB_READ( SMBUS_DATA_A );
    }

    /* DEBUG - ALI controller may require a short breather before ready */
    usleep( ALI_BUGGY_DELAY );

    TRACE( "Read data 0x%02X\n", data[0] );
    return STATUS_SUCCESS;
}



/* node = I2C bus address of target
 * cmd = goes into SMBus command byte, often used by targets as a reg index 
 *      (set to -1 if not required)
 * bytes = number of bytes to be written to target
 * data = location of data to write
 */

static DBM_STATUS i2cdev_write( const uint8 node, const int cmd,
			 const size_t bytes, const uint8 *data)
{
    TRACE( "I2C bus write of %d bytes to node %x cmd %02x\n",
		bytes, node, cmd  );

    /* Wait for Bus Idle. */
    if ( i2c_wait_ready() != STATUS_SUCCESS )
	return STATUS_FAILURE;


    /* Set address. */
    SMB_WRITE( SMBUS_ADDR, node );


    if ( cmd == -1 )		/* No command byte to be specified */
    {
	/* Set Send/Receive byte. */
	SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_SNDRCV );

	/* Queue first byte */
	SMB_WRITE( SMBUS_CMD, data[0] );

    } else {

	if (bytes == 1)				/* Set Read/Write byte. */
	    SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_WRRDB );
	else					/* Set Read/Write word. */
	    SMB_WRITE( SMBUS_SMBCMD, SMBUS_COMMAND_WORD );

	/* Set command register, interpreted by targets in various ways. */
	SMB_WRITE( SMBUS_CMD, cmd );

	/* Queue first byte(s) */
	SMB_WRITE( SMBUS_DATA_A, data[0] );

	if ( bytes == 2 )
	    SMB_WRITE( SMBUS_DATA_B, data[1] );
    }

    /* Start I2C transaction. */
    SMB_WRITE( SMBUS_START, 0xFF );


    /* Wait for command completion. */
    if ( i2c_wait_complete() != STATUS_SUCCESS )
	return STATUS_FAILURE;

    return STATUS_SUCCESS;
}

/*----------------------------------------------------------------------*/
/* Platform I2C access functions */

/* Platform-specific I2C initialisations */
static DBM_STATUS plat_i2cinit( void )
{
    DBM_STATUS sval;
    uint8 rdata;
    static BOOLEAN i2c_init_done = FALSE;

    if ( i2c_init_done == TRUE )
	return STATUS_SUCCESS;

    diags_subsys_stat( SYS_I2C, DEV_PROBING );

    /* Bring up the controller */
    sval = i2cdev_init();
    if ( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "platform I2C initialisation failed\n" );
	diags_subsys_stat( SYS_I2C, DEV_FAILED );
	return sval;
    }


    /* DEBUG: take a look at the value of the SROM/COM selector bit */
    sval = i2cdev_read( SHARK_I2C_REG0, -1, 1, &rdata );
    if ( sval == STATUS_FAILURE )
    {
	TRACE( "I2C didn't read serial control register correctly!\n" );
    }
    else 
    {
	mobo_logf( LOG_DBG "I2C: control reg is 0x%02X, meaning %s selected\n",
		SHARK_I2C_REG0, rdata & SH_R0_SERIAL ? "SERIAL" : "SROM" );
    }


    /* Bring up the individual devices */
    sval = adm_init( SHARK_I2C_SMD0 );
    if( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "I2C: Primary processor monitor failed init!\n" );
	TRACE( "Primary processor monitor failed initialisation\n");
    }
    sval = adm_init( SHARK_I2C_SMD1 );
    if( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "I2C: Secondary processor monitor failed init!\n" );
	TRACE( "Secondary processor monitor failed initialisation\n");
    }
    sval = adm_init( SHARK_I2C_SMD2 );
    if( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "I2C: Motherboard monitor failed init!\n" );
	TRACE( "Motherboard monitor failed initialisation\n");
    }
    sval = adm_init( SHARK_I2C_SMD3 );
    if( sval == STATUS_FAILURE )
    {
	mobo_logf( LOG_CRIT "I2C: PCI riser monitor failed init!\n" );
	TRACE( "PCI riser monitor failed initialisation\n");
    }


    /* configure the temperature sensors */
    adm_add_dev( SHARK_I2C_SMD0, I2CSENS_TEMP0,
		 70 * I2C_TEMP_SCALE, 65 * I2C_TEMP_SCALE );
    adm_add_dev( SHARK_I2C_SMD1, I2CSENS_TEMP0,
		 70 * I2C_TEMP_SCALE, 65 * I2C_TEMP_SCALE );
    adm_add_dev( SHARK_I2C_SMD2, I2CSENS_TEMP0,
		 50 * I2C_TEMP_SCALE, 45 * I2C_TEMP_SCALE );
    adm_add_dev( SHARK_I2C_SMD3, I2CSENS_TEMP0,
		 50 * I2C_TEMP_SCALE, 45 * I2C_TEMP_SCALE );

    /* Bring all the fan monitors online */
    adm_add_dev( SHARK_I2C_SMD0, I2CSENS_FAN0, 7800, 7000 );
    adm_add_dev( SHARK_I2C_SMD0, I2CSENS_FAN1, 7800, 7000 );
    adm_add_dev( SHARK_I2C_SMD1, I2CSENS_FAN0, 7800, 7000 );
    adm_add_dev( SHARK_I2C_SMD1, I2CSENS_FAN1, 7800, 7000 );
    adm_add_dev( SHARK_I2C_SMD2, I2CSENS_FAN0, 7800, 7000 );
    adm_add_dev( SHARK_I2C_SMD2, I2CSENS_FAN1, 7800, 7000 );
    adm_add_dev( SHARK_I2C_SMD3, I2CSENS_FAN0, 7800, 7000 );
    adm_add_dev( SHARK_I2C_SMD3, I2CSENS_FAN1, 7800, 7000 );

    i2c_init_done = TRUE;
    diags_subsys_stat( SYS_I2C, DEV_SETUP );
    return STATUS_SUCCESS;
}



/* Wire up the I2C bus (eg, by setting multiplexers) for this device */
/* For Shark, this is very straightforward, there's nothing to be done.
 * Processors are hard wired down onto the motherboard so all the devices
 * should be always present.
 */

static DBM_STATUS plat_i2c_select( const I2Cbus_t *I )
{ 
    /* nothing */
    return STATUS_SUCCESS;
}


