
%{
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/*
 * $Author: slee $
 * $Revision: 1.6 $
 * $Id: hwmon_cfg.y,v 1.6 2001/01/12 04:34:39 slee Exp $
 */

#include <stdlib.h>
#include <stdio.h>

#include "listmgr.h"
#include "hwmon_cfg.h"



extern FILE *yyin;

static param_t *volt_param_data( const char info, volt_data_t data );
static param_t *temp_param_data( const char info, temp_data_t data );
static param_t *fan_param_data( const char info, fan_data_t data );


static sensor_t *sensor_data( const char type, const char *desc, 
				const address_t addr, list_t *P );

static device_t *device_data( const char *desc, address_t addr,
				list_t *M, list_t *S );

static bus_t *bus_data( const char *desc, list_t *M, list_t *D );

list_t *i2c_buses = NULL;
list_t *i2c_toplevel_modules = NULL;
char *i2c_shutdown_command = NULL;

int bus_count = 0;

%}

%union {
	int num;
	float flt;
	char *str;
	address_t addr;
	list_t *list;
	param_t *param_info;
	sensor_t *sensor_info;
	device_t *device_info;
	bus_t *bus_info;
}

/* Keyword Tokens */
%token	TOK_BUS
%token	TOK_DEVICE
%token	TOK_SENSOR
%token	TOK_SHUTDOWN

%token	<num> TOK_ADDRESS TOK_NUMBER
%token	<flt> TOK_FLOAT
%token	<str> TOK_NAME TOK_OPTLINE

%token	TOK_MOD_KEY
%token	TOK_ADDR_KEY

%token	TOK_SENSOR_FAN
%token 	TOK_FAN_NOM
%token	TOK_FAN_MIN

%token	TOK_SENSOR_THERMAL
%token	TOK_THERM_MAX
%token	TOK_THERM_HYST

%token	TOK_SENSOR_VOLT
%token	TOK_VOLT_NOM
%token	TOK_VOLT_MIN
%token	TOK_VOLT_MAX


%type <addr> address
%type <list> volt_list fan_list temp_list sns_list dev_list mod_list bus_list
%type <param_info> volt_param fan_param temp_param
%type <sensor_info> sns_def
%type <device_info> dev_decl
%type <bus_info> bus_decl
%type <str> mod_decl shutdown_string

%%		/* Grammar rules and actions follow */

/*------------------------------------------------------------------------*/
/* Buses: A complete I2C network with associated bus controller */

/* Start symbol */
top_level:	mod_list shutdown_string bus_list { 
					    i2c_toplevel_modules = list_rev($1);
					    i2c_shutdown_command = $2;
					    i2c_buses = $3;
					}

shutdown_string:			{ $$ = NULL; }
	|	TOK_SHUTDOWN TOK_OPTLINE { $$ = $2; }
	;

/* OK (if _really_ boring) for no buses to be present? */
bus_list:				{ $$ = NULL; }
	|	bus_list bus_decl	{
					  $$ = list_add( $1, $2 );
					  bus_count++;
					}
	;

bus_decl:	TOK_BUS TOK_NAME '{' mod_list dev_list '}'
				{ $$ = bus_data( $2, list_rev($4), $5 ); }
	;

mod_decl:	TOK_MOD_KEY TOK_NAME	{ $$ = $2; }
	;

/* Having no module dependencies (or multiple dependencies) is OK */
mod_list:				{ $$ = NULL; }
	|	mod_list mod_decl	{ $$ = list_add( $1, $2 ); }
	;

/*------------------------------------------------------------------------*/
/* Devices: distinct chips, an addressible location on the I2C bus */

/* Should be OK (if boring) for a bus to contain no devices */
dev_list:				{ $$ = NULL; }
	|	dev_list dev_decl	{ $$ = list_add( $1, $2 ); }
	;

dev_decl:	TOK_DEVICE TOK_NAME '{' address mod_list sns_list '}'
			{ $$ = device_data( $2, $4, list_rev($5), $6 ); }
	;


/*------------------------------------------------------------------------*/
/* Sensors: parts of devices that measure distinct quantities */

/* A device must have at least one sensor active to be configured */
sns_list:	sns_def			{ $$ = makelist( $1 ); }
	|	sns_list sns_def	{ $$ = list_add( $1, $2 ); }
	;

sns_def:	TOK_SENSOR_VOLT TOK_NAME '{' address volt_list '}'
			{ $$ = sensor_data( SENSOR_VOLT, $2, $4, $5 ); }
	|	TOK_SENSOR_FAN TOK_NAME '{' address fan_list '}'
			{ $$ = sensor_data( SENSOR_FAN, $2, $4, $5 ); }
	|	TOK_SENSOR_THERMAL TOK_NAME '{' address temp_list '}'
			{ $$ = sensor_data( SENSOR_TEMP, $2, $4, $5 ); }
	;

/* A sensor must have at least one parameter associated? */
volt_list:	volt_param		{ $$ = makelist( $1 ); }
	|	volt_list volt_param	{ $$ = list_add( $1, $2 ); }
	;

volt_param:	TOK_VOLT_NOM TOK_FLOAT
			{ $$ = volt_param_data( VOLT_NOM, $2 ); }
	|	TOK_VOLT_MIN TOK_FLOAT
			{ $$ = volt_param_data( VOLT_MIN, $2 ); }
	|	TOK_VOLT_MAX TOK_FLOAT
			{ $$ = volt_param_data( VOLT_MAX, $2 ); }
	;

/* A sensor must have at least one parameter associated? */
fan_list:	fan_param		{ $$ = makelist( $1 ); }
	|	fan_list fan_param	{ $$ = list_add( $1, $2 ); }
	;

fan_param:	TOK_FAN_NOM TOK_NUMBER	{ $$ = fan_param_data( FAN_NOM, $2 ); }
	|	TOK_FAN_MIN TOK_NUMBER	{ $$ = fan_param_data( FAN_MIN, $2 ); }
	;

/* A sensor must have at least one parameter associated? */
temp_list:	temp_param		{ $$ = makelist( $1 ); }
	|	temp_list temp_param	{ $$ = list_add( $1, $2 ); }
	;

temp_param:	TOK_THERM_MAX TOK_FLOAT	{ $$ = temp_param_data( TEMP_MAX, $2 ); }
	|	TOK_THERM_HYST TOK_FLOAT {$$ = temp_param_data( TEMP_HYST, $2 ); }
	;
	

address:	TOK_ADDR_KEY TOK_ADDRESS	{ $$ = $2; }
	;

%%

/*------------------------------------------------------------------------*/
/* Handy heap, list management and other widgets */

static void *malloc_check( size_t nob, const char *msg )
{
    void *ptr = malloc( nob );
    if ( ptr == NULL )
    {
	fprintf( stderr, "MALLOC FAILED: on allocation of %s\n", msg );
	exit( -1 );
    }
    return ptr;
}


/*------------------------------------------------------------------------*/
/* PARAMETERS - Attributes relating to sensors; the smallest pieces of data */

static param_t *volt_param_data( const char info, volt_data_t data )
{
    volt_param_t *V=malloc_check( sizeof( volt_param_t ), "new volt info" );
    V->info = info;
    V->data = data;
    return (param_t *)V;
}

static param_t *fan_param_data( const char info, fan_data_t data )
{
    fan_param_t *V=malloc_check( sizeof( fan_param_t ), "new fan info" );
    V->info = info;
    V->data = data;
    return (param_t *)V;
}

static param_t *temp_param_data( const char info, temp_data_t data )
{
    temp_param_t *V=malloc_check( sizeof( temp_param_t ), "new temp info" );
    V->info = info;
    V->data = data;
    return (param_t *)V;
}



/* SENSORS - a measured quantity. */
static sensor_t *sensor_data( const char type, const char *desc, 
				const address_t addr, list_t *P )
{
    sensor_t *S=malloc_check( sizeof(sensor_t), "new sensor info" );
    S->type = type;
    S->addr = addr;
    S->desc = desc;
    S->P = P;
    return S;
}



/* DEVICES - a chip that may contain any (non-zero) number of sensors */
static device_t *device_data( const char *desc, address_t addr,
				list_t *M, list_t *S )
{
    device_t *D=malloc_check( sizeof(device_t), "new device info" );
    D->desc = desc;
    D->modname = NULL;		/* Should be filled in later */
    D->addr = addr;
    D->M = M;
    D->S = S;
    return D;
}


/* BUSES - a network linking any number of devices */
static bus_t *bus_data( const char *desc, list_t *M, list_t *D )
{
    bus_t *B=malloc_check( sizeof( bus_t ), "new bus info" );
    B->desc = desc;
    B->M = M;
    B->D = D;
    return B;
}



/*------------------------------------------------------------------------*/
/* Main routine and parser support code */

int parse_cfg( void )
{
    yyin = fopen( CONFIG_FILE_NAME, "r" );
    if ( yyin == NULL )
    {
	perror( "Opening of config file " CONFIG_FILE_NAME );
	return -1;
    }

    if ( yyparse() != 0 )
    {
	fprintf( stderr, "Parsing of config file failed, exiting.\n");
	return -2;
    }

    return bus_count;
}


yyerror (char *s)  /* Called by yyparse on error */
{
    printf ("%s\n", s);
}

