/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Hardware monitor daemon.
 * Written by Soohoon Lee and Stig Telfer, API NetWorks Inc.
 * $Revision: 1.12 $
 * $Author: slee $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <syslog.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "listmgr.h"
#include "hwmon_cfg.h"

#define MAX_CMD_LEN 64		/* longest command or file name length */
#define MAX_MSG_LEN 200		/* longest message length when logging */

#define DEFAULT_POLL_INTERVAL 60
#define DEFAULT_DEVNAME	"dev"	/* If we don't have a name for a device (bad) */

#define DAEMON_NAME	"hwmon"
#define DAEMON_BANNER	"API NetWorks Hardware Monitor, build " __DATE__ "\n"


void hwmon_vlog( const int priority, const char *msg_fmt, va_list ap );
void hwmon_log( const int priority, const char *msg_fmt, ... );
void hwmon_exit( const char *msg_fmt, ... );


/*------------------------------------------------------------------------*/
/* Private data */

/* Default switches */
static unsigned char hwmon_daemon = 0; 
static unsigned char hwmon_verbose = 0; 

/* Default values */
static unsigned hwmon_poll_interval = DEFAULT_POLL_INTERVAL;

/* Global data that comes down from above in the list traversals */
static unsigned short ck_busno;
static device_t *ck_dev;
static sensor_t *ck_sens;


/*------------------------------------------------------------------------*/
/* Interrogating of I2C devices via the lm_sensors interface */

static void read_data( const char *modname,
			 const unsigned short bus,
			 const address_t addr,
			 const char *sname,
			 const address_t saddr,
			 char *sdata )
{
    char fname_buf[ MAX_CMD_LEN ];
    int len;
    FILE *fp;

    /* Here we compensate for a little quirk - if a sensor is unique in the 
     * device, lm_sensors doesn't append a number to its ref filename.
     * Sensors of the same type enumerate from 1, so we can use 0 to signify
     * a unique sensor.
     */

    if ( saddr == 0 )
	len = snprintf( fname_buf, MAX_CMD_LEN,
		"/proc/sys/dev/sensors/%s-i2c-%d-%x/%s",
		modname, bus, addr, sname );
    else
	len = snprintf( fname_buf, MAX_CMD_LEN,
		"/proc/sys/dev/sensors/%s-i2c-%d-%x/%s%d",
		modname, bus, addr, sname, saddr );

    if ( len > MAX_CMD_LEN )
	hwmon_exit( "ERROR - Sensor filename is too long - %s\n", fname_buf );

    fp = fopen( fname_buf, "r" );
    if ( fp == NULL )
	hwmon_exit( "ERROR - Couldn't open sensor file - %s\n", fname_buf );

    /* Defensive: zero the read buffer to avoid confusion if fgets fails */
    memset( sdata, 0, MAX_CMD_LEN );
    fgets( sdata, MAX_CMD_LEN, fp );
    fclose( fp );
}


void read_temp( const unsigned short bus, device_t *D, sensor_t *S )
{
    /* ADM9240 and LM75 drivers (known temperature sensing devices) 
     * return three floating point numbers, being high setting, hyst setting 
     * and value read, in that order.
     * 
     * Hopefully any other temp-sensing devices in lm_sensors follow that lead!
     */
    float max=0.0, hyst=0.0, temp=0.0;
    char data_buf[ MAX_CMD_LEN ];
    int scans;

    read_data( D->modname, bus, D->addr, "temp", S->addr, data_buf );
    scans = sscanf( data_buf, "%f %f %f", &max, &hyst, &temp );
    if ( scans != 3 )			/* expect three floats on read */
	hwmon_exit( "Bizarre! Read confusing data from temperature sensor\n" );

    /* Is the temperature OK? */
    if ( temp > max )
    {
	hwmon_log( LOG_ALERT, "Thermal sensor %s operating out of range: "
		"reads %.1fC (max %f.1C)\n",
		S->desc, temp, max );
    }
    else
    {
	hwmon_log( LOG_INFO, "Thermal sensor %s reads %.1fC\n", S->desc,temp );
    }
}


void read_fan( const unsigned short bus, device_t *D, sensor_t *S )
{
    /* The ADM9240 driver returns two integers from a fan read, being
     * the low threshold and the measured rpm.
     * 
     * Hopefully any other fan-sensing devices in lm_sensors follow that lead!
     */
    int min=0, fan=0;
    char data_buf[ MAX_CMD_LEN ];
    int scans;

    read_data( D->modname, bus, D->addr, "fan", S->addr, data_buf );
    scans = sscanf( data_buf, "%d %d", &min, &fan );
    if ( scans != 2 )			/* expect two ints on read */
	hwmon_exit( "Bizarre! Read confusing data from fan sensor\n" );

    /* Is the fan RPM OK? */
    if ( fan < min )
    {
	hwmon_log( LOG_ALERT, "Fan sensor %s operating out of range: "
		"reads %d RPM (minimum %d RPM)\n",
		S->desc, fan, min );
    }
    else
    {
	hwmon_log( LOG_INFO, "Fan sensor %s reads %d RPM\n", S->desc, fan );
    }
}




void read_volt( const unsigned short bus, device_t *D, sensor_t *S )
{
    /* With reading voltages we have an extra layer of indirection which
     * also seems to require either some magic numbers or some magic strings
     * in the config file.  Magic numbers are easier for us to handle and 
     * in this case the user's going to lose out some simplicity :-(
     */
    static const char *magic_volttab[] = {
		"2.5V", "3.3V", "5V", "12V", "Vccp1", "Vccp2" };
    float min=0.0, max=0.0, volt=0.0;
    char data_buf[ MAX_CMD_LEN ];
    int scans;
    
    read_data( D->modname, bus, D->addr, magic_volttab[S->addr], 0, data_buf );
    scans = sscanf( data_buf, "%f %f %f", &min, &max, &volt );

    if ( scans != 3 )			/* expect three floats on read */
	hwmon_exit( "Bizarre! Read confusing data from volt sensor\n" );

    /* is the voltage level OK? */
    if ( ( volt < min ) || ( volt > max ) )
    {
	hwmon_log( LOG_ALERT, "Volt sensor %s operating out of range: "
		"reads %.1fV (min %.1fV, max %.1fV)\n",
                S->desc, volt, min, max );
    } 
    else 
    {
	hwmon_log( LOG_INFO, "Volt sensor %s reads %.1fV\n", S->desc, volt );
    }
}


/*------------------------------------------------------------------------*/
/* Setup of sensors, buses and devices */

static void write_data( const char *modname,
			 const unsigned short bus,
			 const address_t addr,
			 const char *sname,
			 const address_t saddr,
			 char *sdata )
{
    char fname_buf[ MAX_CMD_LEN ];
    int len;
    FILE *fp;

    /* Here we compensate for a little quirk - if a sensor is unique in the 
     * device, lm_sensors doesn't append a number to its ref filename.
     * Sensors of the same type enumerate from 1, so we can use 0 to signify
     * a unique sensor.
     */

    if ( saddr == 0 )
	len = snprintf( fname_buf, MAX_CMD_LEN,
		"/proc/sys/dev/sensors/%s-i2c-%d-%x/%s",
		modname, bus, addr, sname );
    else
	len = snprintf( fname_buf, MAX_CMD_LEN,
		"/proc/sys/dev/sensors/%s-i2c-%d-%x/%s%d",
		modname, bus, addr, sname, saddr );

    if ( len > MAX_CMD_LEN ) {
	hwmon_log( LOG_WARNING, "Sensor filename is too long\n" );
	return;
    }

    fp = fopen( fname_buf, "w" );
    if ( fp == NULL ) {
	hwmon_log( LOG_WARNING, "Couldn't open sensor file %s\n", fname_buf );
	return;
    }

    fprintf(fp, sdata );
    fflush(fp);
    fclose(fp);

    fp = fopen( fname_buf, "r" );
    if ( fp == NULL ) {
	hwmon_log( LOG_WARNING, "Couldn't open sensor file %s\n", fname_buf );
	return;
    }

    /* Defensive: zero the read buffer to avoid confusion if fgets fails */
    memset( sdata, 0, MAX_CMD_LEN );
    fgets( sdata, MAX_CMD_LEN, fp );
    fclose(fp);
}


static void setup_temp_param( const unsigned short bus, device_t *D, sensor_t *S )
{
    /* ADM9240 and LM75 drivers (known temperature sensing devices) 
     * return three floating point numbers, being high setting, hyst setting 
     * and value read, in that order.
     * 
     * Hopefully any other temp-sensing devices in lm_sensors follow that lead!
     */
    float max=60.0, hyst=50.0, temp=0.0;
    char data_buf[ MAX_CMD_LEN ];
    int scans;
    list_t *param;

    S->stat = 0;
    for (param = S->P; param; param = list_getnext(param)) {
	    if (((temp_param_t*)list_getatom(param))->info == TEMP_MAX)
		max = ((temp_param_t*)list_getatom(param))->data;
	    if (((temp_param_t*)list_getatom(param))->info == TEMP_HYST)
		hyst = ((temp_param_t*)list_getatom(param))->data;
    }

    if (max < hyst) 
	hyst = max - 10;

    sprintf(data_buf, "%2.2f %2.2f", max, hyst);

    write_data( D->modname, bus, D->addr, "temp", S->addr, data_buf );

    scans = sscanf( data_buf, "%f %f %f", &max, &hyst, &temp );

    if ( scans != 3 ) {			/* expect three floats on read */
	hwmon_log( LOG_WARNING, "Read confusing data from temperature sensor\n" );
	hwmon_log( LOG_WARNING, "     sensor, %s, will be disabled\n", S->desc );
	S->stat = SENSOR_BROKEN;
	return;
    }

    S->stat = SENSOR_NORMAL;

    /* Is the temperature OK? */
    if ( temp > max )
    {
	hwmon_log( LOG_ALERT, "Thermal monitor %s reads %.1fC (max %f.1C)\n",
		S->desc, temp, max );
    	S->stat = SENSOR_ALERT;
    }
    else
    {
	hwmon_log( LOG_INFO, "Thermal monitor %s reads %.1fC\n", S->desc,temp );
    }
}

static void setup_fan_param( const unsigned short bus, device_t *D, sensor_t *S )
{
    /* The ADM9240 driver returns two integers from a fan read, being
     * the low threshold and the measured rpm.
     * 
     * Hopefully any other fan-sensing devices in lm_sensors follow that lead!
     */
    int min=3000, fan=0;
    char data_buf[ MAX_CMD_LEN ];
    int scans;
    list_t *param;

    S->stat = 0;
    for (param = S->P; param; param = list_getnext(param)) 
	    if (((fan_param_t*)list_getatom(param))->info == FAN_NOM)
		min = ((fan_param_t*)list_getatom(param))->data;

    sprintf(data_buf, "%d", min);
    write_data( D->modname, bus, D->addr, "fan", S->addr, data_buf );

    scans = sscanf( data_buf, "%d %d", &min, &fan );

    if ( scans != 2 ) {			/* expect two ints on read */
	hwmon_log( LOG_WARNING, "Read confusing data from fan sensor\n" );
	hwmon_log( LOG_WARNING, "     sensor, %s, will be disabled\n", S->desc );
	S->stat = SENSOR_BROKEN;
	return;
    }

    S->stat = SENSOR_NORMAL;

    /* Is the fan RPM OK? */
    if ( fan < min )
    {
	hwmon_log( LOG_WARNING, "Fan monitor %s reads %d RPM (minimum %d RPM)\n",
		S->desc, fan, min );
    	S->stat = SENSOR_BROKEN; //For FAN case
    }
    else
    {
	hwmon_log( LOG_INFO, "Fan monitor %s reads %d RPM\n", S->desc, fan );
    }
}

static void setup_volt_param( const unsigned short bus, device_t *D, sensor_t *S )
{
    /* With reading voltages we have an extra layer of indirection which
     * also seems to require either some magic numbers or some magic strings
     * in the config file.  Magic numbers are easier for us to handle and 
     * in this case the user's going to lose out some simplicity :-(
     */
    static const char *magic_volttab[] = {
		"2.5V", "3.3V", "5V", "12V", "Vccp1", "Vccp2" };
    float min=0.0, max=0.0, volt=0.0;
    char data_buf[ MAX_CMD_LEN ];
    int scans;
    list_t *param;
    
    S->stat = 0;
    for (param = S->P; param; param = list_getnext(param)) {
	    if (((volt_param_t*)list_getatom(param))->info == VOLT_MIN)
		min = ((volt_param_t*)list_getatom(param))->data;
	    if (((volt_param_t*)list_getatom(param))->info == VOLT_MAX)
		max = ((volt_param_t*)list_getatom(param))->data;
    }

    if (min > max)
	min = max - 1.0;

    sprintf(data_buf, "%2.2f %2.2f", min, max);

    write_data( D->modname, bus, D->addr, magic_volttab[S->addr], 0, data_buf );
    scans = sscanf( data_buf, "%f %f %f", &min, &max, &volt );

    if ( scans != 3 ) {			/* expect three floats on read */
	hwmon_log( LOG_WARNING, "Read confusing data from voltage sensor\n" );
	hwmon_log( LOG_WARNING, "     sensor, %s, will be disabled\n", S->desc );
	S->stat = SENSOR_BROKEN;
	return;
    }

    S->stat = SENSOR_NORMAL;
    /* is the voltage level OK? */
    if ( ( volt < min ) || ( volt > max ) )
    {
	hwmon_log( LOG_ALERT, "Volt sensor %s operating out of range: "
		"reads %.1fV (min %.1fV, max %.1fV)\n",
                S->desc, volt, min, max );
	S->stat = SENSOR_ALERT;
    } 
    else 
    {
	hwmon_log( LOG_INFO, "Volt sensor %s reads %.1fV\n", S->desc, volt );
    }
}


static void setup_sensor( void *L )
{
    list_operator_t *action;
    sensor_t *S = (sensor_t *)L;		/* saves typecasting */
    char *stype = NULL;

    ck_sens = S;		/* Global copy of sensor under scrutiny */

    switch( S->type )
    {
	case SENSOR_TEMP:
	    stype = "temp";
	    setup_temp_param( ck_busno, ck_dev, ck_sens );
	    break;

	case SENSOR_FAN:
	    stype = "fan";
	    setup_fan_param( ck_busno, ck_dev, ck_sens );
	    break;

	case SENSOR_VOLT:
	    stype = "volt";
	    setup_volt_param( ck_busno, ck_dev, ck_sens );
	    break;
	
	default:
	    hwmon_exit( "Internal error: sensor type %d found\n", S->type );
    }
    hwmon_log( LOG_INFO, "\t\tSensor: %s (%s) at internal address %d\n",
		stype, S->desc, S->addr );

}

static void setup_device( void *L )
{
    device_t *D = (device_t *)L;		/* saves typecasting */
    int len;

    ck_dev = D;			/* Global copy of device under scrutiny */

    hwmon_log( LOG_INFO, "\tDevice %s at address 0x%02X\n", D->desc, D->addr );

    /* Bit of fixup here: assign a name for the device.
     * We choose the name of the first module.  If there isn't exactly one
     * module declaration, there could be problems...
     */
    len = list_len( D->M );
    if ( len < 1 )
    {
	hwmon_log( LOG_WARNING,
	    "No module name given for device %s, using " DEFAULT_DEVNAME "\n",
	    D->desc );
	D->modname = DEFAULT_DEVNAME;

    } else {

	/* Choose first module name in the list */
	D->modname = (char *)D->M->L;

	if ( len > 1 )
	{
	    hwmon_log( LOG_NOTICE,
		"More than one module associated with %s, using name %s\n",
		 D->desc, D->modname );
	}
    }



    list_act( D->S, setup_sensor );		/* Dump attached sensors */
}

static void setup_bus( void *L )
{
    bus_t *B = (bus_t *)L;

    hwmon_log( LOG_INFO, "Configuring I2C Bus %s:\n", B->desc );

    list_act( B->D, setup_device );		/* Setup devices found */
    ck_busno++;
}


/*------------------------------------------------------------------------*/
/* Polling code */

/* Global data that comes down from above in the list traversals */
static unsigned short ck_busno;
static device_t *ck_dev;
static sensor_t *ck_sens;


static void check_sensor( void *L )
{
    sensor_t *S = (sensor_t *)L;
    char *sname=NULL;
    ck_sens = S;		/* Global copy of sensor under scrutiny */

    if (!(S->stat & SENSOR_BROKEN)) {
	    switch( S->type )
	    {
		case SENSOR_TEMP:
		    read_temp( ck_busno, ck_dev, ck_sens );
		    break;

		case SENSOR_FAN:
		    read_fan( ck_busno, ck_dev, ck_sens );
		    break;

		case SENSOR_VOLT:
		    read_volt( ck_busno, ck_dev, ck_sens );
		    break;

		default:		/* Should never happen */
		    hwmon_exit( "ERROR Confused sensor types\n" );
	    }
    }
}


static void check_device( void *L )
{
    device_t *D = (device_t *)L;
    ck_dev = D;			/* Global copy of device under scrutiny */
    list_act( D->S, check_sensor );
}


/* To use this properly, ck_busno must have been set to zero first */
static void check_bus( void *L )
{
    bus_t *B = (bus_t *)L;
    list_act( B->D, check_device );
    ck_busno++;
}


/*------------------------------------------------------------------------*/
/* Module management code */


static list_t *all_modules = NULL;


static void unload_module( void *L )
{
    char *name = (char *)L;
    int len, rval;
    char cmd_buf[ MAX_CMD_LEN ];

    len = snprintf( cmd_buf, MAX_CMD_LEN, "/sbin/rmmod %s %s",
		hwmon_daemon ? "-s" : "",
		name );

    if ( len > MAX_CMD_LEN )
	hwmon_exit( "Module remove command would have overflowed buffer\n" );
    
    rval = system( cmd_buf );
#if 0			/* FIXME */
    if ( rval != 0 )
	hwmon_exit( "Unload of module %s returned error %d\n", name, rval );
#endif
}

static void load_module( void *L )
{
    char *name = (char *)L;
    int len, rval;
    char cmd_buf[ MAX_CMD_LEN ];

    hwmon_log( LOG_INFO, "Requires module: %s\n", name );

    len = snprintf( cmd_buf, MAX_CMD_LEN, "/sbin/insmod %s %s %s",
		hwmon_daemon ? "--syslog" : "",
		hwmon_verbose ? "--verbose" : "",
		name );

    if ( len > MAX_CMD_LEN )
	hwmon_exit( "Module insert command would have overflowed buffer\n" );
    
    rval = system( cmd_buf );
#if 0			/* FIXME */
    if ( rval != 0 )
	hwmon_exit( "Load of module %s returned error %d\n", name, rval );
#endif
}


static void collect_dev_modules( void *L )
{
    device_t *D = (device_t *)L;
    all_modules = list_join( D->M, all_modules );
}

static void collect_modules( void *L )
{
    bus_t *B = (bus_t *)L;
    all_modules = list_join( B->M, all_modules );
    list_act( B->D, collect_dev_modules );
}




/*------------------------------------------------------------------------*/
/* All Logging output to syslog or stderr comes through this function, where
 * finally a differentiation is made
 */

void hwmon_vlog( const int priority, const char *msg_fmt, va_list ap )
{
    int len;
    char msg_buf[ MAX_MSG_LEN ];

    /* Silently drop anything that is not important enough */
    if ( !hwmon_verbose && ( priority == LOG_INFO || priority == LOG_DEBUG ) )
	    return;


    /* vsnprintf returns the number of bytes that would have been written, but
     * doesn't exceed actually writing MAX_MSG_LEN bytes */
    len = vsnprintf( msg_buf, MAX_MSG_LEN, msg_fmt, ap );
    if ( len < 0 || len > MAX_MSG_LEN )
	sprintf( msg_buf, "log error: buffer would have overflowed\n" );

    if ( hwmon_daemon )
    {
	/* In daemon mode we write to syslog */
	syslog( priority, msg_buf );
    } else {
	/* If we're not in daemon mode, we're writing to stderr */
	fprintf( stderr, DAEMON_NAME ": %s", msg_buf );
    }

    if (priority == LOG_ALERT) {
	//system(i2c_shutdown_command);
    }
}


void hwmon_log( const int priority, const char *msg_fmt, ... )
{
    va_list ap;
    va_start(ap, msg_fmt);

    hwmon_vlog( priority, msg_fmt, ap );

    va_end( ap );
}



void hwmon_exit( const char *msg_fmt, ... )
{
    va_list ap;
    va_start(ap, msg_fmt);

    hwmon_vlog( LOG_CRIT, msg_fmt, ap );

    va_end( ap );
    exit( -1 );
}


/*------------------------------------------------------------------------*/
/* Main routine */

int main( int argc, char *argv[] )
{
    int rval;
    int c;

    /* Look at command line options */
     while (1)
     {
	 c = getopt (argc, argv, "df:l:u:i:vh");
	 if (c == -1)
		break;

	switch (c)
	{
	    case 'd':			/* Open connection to syslog */
		hwmon_daemon = 1;
		break;

	    case 'i':
		hwmon_poll_interval = atoi(optarg);
		break;

	    case 'v':
		hwmon_verbose = 1;
		break;

	    default:
	    case 'h':
		fprintf( stderr,
			 "\n" DAEMON_BANNER
			 "\nUsage: %s [-d] [-i poll_interval] [-v]\n\n"
			 "\t-d\tRun as a daemon using syslog\n"
			 "\t-i\tPolling interval in seconds (default %d)\n"
			 "\t-v\tVerbose output\n\n",
			 argv[0], DEFAULT_POLL_INTERVAL );
		exit( -1 );
	}
    }

    if ( hwmon_daemon )
    {
	long maxfd;
	long fd;
#if 0 // Do this after full initialization
	/* Become a daemon by detaching from the console, etc */
	pid_t pid = fork();

	if ( pid < 0 )
	    return -1;

	/* Parent process exits */
	if ( pid != 0 )
	    exit( 0 );

	/* Child continues */
	setsid();		/* become session leader */

	chdir( "/" );		/* ...so previous WD FS may be unmounted */

	umask( 0 );		/* Clear our file creation mask */
#endif

	/* Close all file descriptors */
	maxfd = sysconf( _SC_OPEN_MAX );
	for ( fd=0; fd<maxfd; fd++ )
	    close( fd );

	/* Attach to Syslog */
	openlog( DAEMON_NAME, LOG_CONS, LOG_DAEMON );
    }

    hwmon_log( LOG_NOTICE, DAEMON_BANNER );

    /* Do we have anything parseable in the config file? */
    hwmon_log( LOG_DEBUG, "Opening configuration file " CONFIG_FILE_NAME "\n" );

    /* Default shutdown command, It could be changed in parse_cfg() */
    i2c_shutdown_command = "/sbin/shutdown -f -h +2 \"Failure detected; System Shutting Down\"";

    rval = parse_cfg( );

    if ( rval <= 0 )
    {
	hwmon_exit( "Config file parsing was unsuccessful\n" );
    }
    hwmon_log( LOG_DEBUG, "Configuration file parsed %d buses\n", rval );


    /*
     * Initialise/setup the I2C Modules
     */

    /* Here we collect a list of all modules, in order of dependency going
     * from head to end.  This assumes that device modules depend on bus 
     * modules which depend on core modules (and not vice versa)
     *
     * We also make use of some knowledge about how list joins order the 
     * joined modules.
     */
    all_modules = list_rev( i2c_toplevel_modules );
    list_act( i2c_buses, collect_modules );
    list_act( all_modules, unload_module );


    /* Reverse the module list from bottom-up to top-down in dependency */
    /* Then bring the modules into the kernel meeting dep. requirements */
    all_modules = list_rev( all_modules );
    list_act( all_modules, load_module );
    

    /*
     * Initialise/setup the I2C Buses
     */
    ck_busno = 0;
    list_act( i2c_buses, setup_bus );

    if ( hwmon_daemon )
    {
	/* Become a daemon by detaching from the console, etc */
	pid_t pid = fork();
	long maxfd;
	long fd;

	if ( pid < 0 )
	    return -1;

	/* Parent process exits */
	if ( pid != 0 )
	    exit( 0 );

	/* Child continues */
	setsid();		/* become session leader */

	chdir( "/" );		/* ...so previous WD FS may be unmounted */

	umask( 0 );		/* Clear our file creation mask */
    }


    /*
     * Polling loop
     */
    while( 1 )
    {
	ck_busno = 0;
	list_act( i2c_buses, check_bus );
	sleep( hwmon_poll_interval );
    }

    /* Should never be reached */
    return -1;
}
