/*****************************************************************************

       Copyright  1993, 1994 Digital Equipment Corporation,
                       Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, provided  
that the copyright notice and this permission notice appear in all copies  
of software and supporting documentation, and that the name of Digital not  
be used in advertising or publicity pertaining to distribution of the software 
without specific, written prior permission. Digital grants this permission 
provided that you prominently mark, as not part of the original, any 
modifications made to this software or documentation.

Digital Equipment Corporation disclaims all warranties and/or guarantees  
with regard to this software, including all implied warranties of fitness for 
a particular purpose and merchantability, and makes no representations 
regarding the use of, or the results of the use of, the software and 
documentation in terms of correctness, accuracy, reliability, currentness or
otherwise; and you rely on the software, documentation and results solely at 
your own risk. 

******************************************************************************/
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/


#undef TRACE_ENABLE			/* define to enable debugging output */

#include "lib.h"
#include "uilib.h"
#include "northbridge.h"
#include "platform.h"

#include "pcidefs.h"
#include "ledcodes.h"

#include "pci.h"
#include "mcheck.h"

/* private functions */
static void PCIShowDevice( const PCIBus_t *bus, const PCIDevice_t *device );
static void PCIShowBusLog( const PCIBus_t *bus );
static char *DeviceClassString( ui class );


#ifdef TRACE_ENABLE
static void PCIDumpDevice(ui bus, ui dev, ui function);
#endif


/**************************************************************************
 * PCIDeviceFind()                                                        *
 **************************************************************************/
/* STIG: added the option of specifying 0xFFFF as a wildcard option */

PCIDevice_t *PCIDeviceFind(uw vendor, uw device)
{
    PCIDevice_t *PCIDevice = devices;

    TRACE("Looking for 0x%04X:0x%04X\n", vendor, device );

    while (PCIDevice != NULL) {

        TRACE("  Comparing with 0x%04X:0x%04X\n",
		PCIDevice->vendor_id, PCIDevice->device_id );

	if (((PCIDevice->vendor_id == vendor) || (vendor == 0xFFFF))
	    && ((PCIDevice->device_id == device) || (device == 0xFFFF))) {
	    return PCIDevice;
	}

	PCIDevice = PCIDevice->sibling;
    }
    return NULL;
}

/**************************************************************************
 * PCIDeviceFindNext()                                                    *
 **************************************************************************/
PCIDevice_t *PCIDeviceFindNext(PCIDevice_t * PCIDevice)
{
    if (PCIDevice != NULL) {
	uw vendor, device;

	vendor = PCIDevice->vendor_id;
	device = PCIDevice->device_id;

        TRACE("Seeking next 0x%04X:0x%04X\n", vendor, device );

	PCIDevice = PCIDevice->sibling;
	while (PCIDevice != NULL) {
	    if (   (PCIDevice->vendor_id == vendor)
		&& (PCIDevice->device_id == device) )
			return PCIDevice;
	    else PCIDevice = PCIDevice->sibling;
	}
    }
    TRACE("Found nothing\n");
    return NULL;
}

/**************************************************************************
 * PCIShowDevice()                                                        *
 **************************************************************************/
static void PCIShowDevice( const PCIBus_t * bus, const PCIDevice_t * device )
{
    char buffer[20];
    int IoX = 0;
    int MemX = 0;
    int LinePrinted = FALSE;

#ifdef PCIDEBUG
    mobo_logf( LOG_INFO "  at address %X\n", device );
    mobo_logf( LOG_INFO "  parent = %X, sibling = %X, next = %X\n",
	      device->parent, device->sibling, device->next );
#endif

    while (TRUE)
    {
	for (IoX = IoX; IoX < MAX_BASE_ADDRESS_REGISTERS + 1; ++IoX) {
	    if ((device->PCI_Bar[IoX].Type & TYPE_IO) &&
		(device->PCI_Bar[IoX].Size != 0))
		break;
	}

	for (MemX = MemX; MemX < MAX_BASE_ADDRESS_REGISTERS + 1; ++MemX) {
	    if (!(device->PCI_Bar[MemX].Type & TYPE_IO) &&
		(device->PCI_Bar[MemX].Size != 0))
		break;
	}

	if (LinePrinted && ((IoX > MAX_BASE_ADDRESS_REGISTERS)
			    && (MemX > MAX_BASE_ADDRESS_REGISTERS)))
	    break;

	if (!LinePrinted) {
	    if (device->function == 0) {
		sprintf_dbm(buffer, "%3d  ", device->slot);
	    } else {
		sprintf_dbm(buffer, "%2d/%1d ", device->slot,
			    device->function);
	    }
	    mobo_logf(LOG_INFO "%s %04X   %04X   %02X %04X ",
		      buffer, device->vendor_id, device->device_id,
		      device->revision, 
		      pcicfgrw(bus->number, device->slot,
			       device->function, PCI_COMMAND));
	} else
	    mobo_logf(LOG_INFO "%3s   %4s   %4s   %2s %4s ", "", "", "",
		      "", "");

	if (device->bus == NULL) {
	    /* Non-Bridge Device */
	    if (IoX < MAX_BASE_ADDRESS_REGISTERS + 1)
		mobo_logf("%8X ", device->PCI_Bar[IoX].Base);
	    else
		mobo_logf("%8s ", "");

	    if (MemX < MAX_BASE_ADDRESS_REGISTERS + 1)
		mobo_logf("%8X%c", device->PCI_Bar[MemX].Base,
			  (MemX == MAX_BASE_ADDRESS_REGISTERS) ? '<' : ' ');
	    else
		mobo_logf("%8s ", "");
	} else {
	    /* Bridge Device */
	    if (device->bus->PCI_IO_Base != 0)
		mobo_logf("%8X ", device->bus->PCI_IO_Base);
	    else

	    if (device->bus->PCI_Mem_Base != 0)
		mobo_logf("%8X ", device->bus->PCI_Mem_Base);
	    else
		mobo_logf("%8s ", "");
	}

	if (!LinePrinted) {
	    mobo_logf("%6X %s\n",
		      device->class, DeviceClassString(device->class));
	    LinePrinted = TRUE;
	} else	 mobo_logf("\n");

#ifdef TRACE_ENABLE
	if ((IoX < MAX_BASE_ADDRESS_REGISTERS + 1)
	    || (MemX < MAX_BASE_ADDRESS_REGISTERS + 1)) {
	    mobo_logf(LOG_INFO "%3s   %4s   %4s   %2s %4s ", "", "", "",
		      "", "");

	    if (IoX < MAX_BASE_ADDRESS_REGISTERS + 1)
		mobo_logf("%8X ", device->PCI_Bar[IoX].Size);
	    else
		mobo_logf("%8s ", "");

	    if (MemX < MAX_BASE_ADDRESS_REGISTERS + 1)
		mobo_logf("%8X ", device->PCI_Bar[MemX].Size);
	    else
		mobo_logf("%8s ", "");
	    mobo_logf("\n");
	}
#endif				/* TRACE_ENABLE */

	++IoX;
	++MemX;
    }

#ifdef PCIDEBUG
    PCIDumpDevice((ui) bus->number, (ui) device->slot, 0);
#endif

}



static char *DeviceClassString(ui class)
{
    switch (class >> 16) {

    case 0x00:			/* Built before code definitions */
	switch (class) {
	case 0x000100:
	    return ("VGA compatible device");
	default:
	    return ("Unknown device class");
	}

    case 0x01:			/* Mass Storage Controller */
	switch (class >> 8) {
	case 0x0100:
	    return ("SCSI bus controller");
	case 0x0101:
	    return ("IDE controller");
	case 0x0102:
	    return ("Floppy disk controller");
	case 0x0103:
	    return ("IPI bus controller");
	default:
	    return ("Mass storage controller");
	}

    case 0x02:			/* Network Controller */
	switch (class >> 8) {
	case 0x0200:
	    return ("Ethernet controller");
	case 0x0201:
	    return ("Token Ring controller");
	case 0x0202:
	    return ("FDDI controller");
	default:
	    return ("Network controller");
	}

    case 0x03:			/* Display Controller */
	switch (class) {
	case 0x030000:
	    return ("VGA compatible controller");
	case 0x030001:
	    return ("8514-compatible controller");
	case 0x030100:
	    return ("XGA controller");
	default:
	    return ("Display controller");
	}

    case 0x04:			/* Multimedia Device */
	return ("Multimedia Device");

    case 0x05:			/* Memory Controller */
	return ("Memory Controller");

    case 0x06:			/* Bridge Device */
	switch (class) {
	case 0x060100:
	    return ("ISA bridge");
	case 0x060400:
	    return ("PCI-PCI bridge");
	default:
	    return ("Bridge Device");
	}

    case 0x07:			/* Simple Communication Controller */
	return ("Communication Controller");

    case 0x08:			/* Base System Peripheral */
	return ("Base System Peripheral");

    case 0x09:			/* Input Device */
	return ("Input Device");

    case 0x0A:			/* Docking Station */
	return ("docking Station");

    case 0x0B:			/* Processor */
	return ("processor");

    case 0x0C:			/* Serial Bus Controller */
	return ("Serial Bus controller");

    default:
	return ("Unknown");
    }
}


#ifdef PCIDEBUG
/**************************************************************************
 * PCIDumpDevice()                                                        *
 **************************************************************************/
static void PCIDumpDevice(ui bus, ui dev, ui function)
{
    int i;
    printf_dbm("\n\tPCI Configuration Space: bus %d, slot %d, function %d\n",
		bus, dev, function );
    for (i = 0; i < 0x10; i += 0x4) {
	printf_dbm("\t%02x:%08x  %02x:%08x  %02x:%08x  %02x:%08x\n",
		   i, pcicfgrl(bus, dev, function, i),
		   i + 0x10, pcicfgrl(bus, dev, function, i + 0x10),
		   i + 0x20, pcicfgrl(bus, dev, function, i + 0x20),
		   i + 0x30, pcicfgrl(bus, dev, function, i + 0x30));
    }
    printf_dbm("\n");
}
#endif


/*
 * PCIDev2Str - convert a bus, device pairing to a physical location
 * This is described by a platform-specific PCIStructure_t data structure
 *
 */

const char *PCIDev2Str(unsigned hose, unsigned dev, char bridgedflag)
{
    const PCIStructure_t *P;
    const PCISlot_t *S;

    /* Are we talking about an unknown bus behind a PCI card bridge? */
    if ( bridgedflag == 1 )
    {
	return "Bridged device";
    }

    /* Otherwise we're referring to a device on a motherboard bus */
    P = PCIStructure + hose;	/* find the bus */

    /* loop over its device-to-physical location mappings */
    for (S = P->slots; S->desc != 0; S++) {
	if (S->dev == dev)
	    return S->desc;
    }

    return "Unknown";
}


/*--------------------------------------------------------------------*/
/* Hose level display routines */

static void PCIShowBusUI( PCIBus_t *B )
{
    PCIDevice_t *D;
    PCIBus_t *child;
    int n, h, i;
    Point posn, kposn;
    char buf[80];
    char bridgedbus = 0;

    /* find this PCI bus number and hose number */
    n = B->number;
    h = B->hose;
    if ( B->parent != NULL )
	bridgedbus = 1;

    TRACE( "Devices on hose %d bus %d (%s)", h, n, PCIVariant( h, bridgedbus ) );

    /* Setup screen */
    sprintf_dbm(buf, "Devices on hose %d bus %d (%s)", h, n, PCIVariant( h, bridgedbus ) );
    kposn.x = r_app.x + (r_app.w >> 1) - (strlen(s_ekey) >> 1);
    kposn.y = r_app.y + r_app.h - 1;

    mobo_box(r_app, buf);

    /*
     * If there's no devices detected on this bus,
     * display a message to that effect and don't
     * bother going through the whole for loop.
     */
    if (B->devices == NULL)
        printf_dbm( "No devices detected on this bus\n");
    else
    {
        for (D = B->devices, i = 0; D != NULL; i++, D = D->next)
        {
	    TRACE( "%d: ID %d (%s) -> %s\n", i, D->slot, 
		   PCIDev2Str( h, D->slot, bridgedbus ), DeviceClassString( D->class ) );

	    if (i > 10)
	        continue;		/* too many! */

	    if (i <= 5)
	        posn.x = 3;
	    else
	        posn.x = 42;
	    posn.y = r_app.y + 1 + i % (r_app.h - 2);

	    mobo_goto(posn);

	    printf_dbm("%14s: %-20s",
		       PCIDev2Str(h, D->slot, bridgedbus), DeviceClassString(D->class));
        }
    }
    mobo_goto(kposn);
    printf_dbm(s_ekey);
    mobo_key(0);

/*
 *  Show all of the sub-buses.
 */
    child = B->children;
    while (child != NULL)
    {
	TRACE( "Bus 0x%lX displaying child at 0x%lX\n", B, child );

	PCIShowBusUI( child );	/* reuse same buffer */
	child = child->next;
    }
}


/**************************************************************************
 * PCIShowBusLog()                                                           *
 **************************************************************************/
static void PCIShowBusLog( const PCIBus_t * bus)
{
    const PCIDevice_t *device = bus->devices;
    const PCIBus_t *child = bus->children;
    char bridgedflag = 0;

    if ( bus->parent != NULL )
	bridgedflag = 1;

    mobo_logf( LOG_INFO "Devices on hose %d bus %d (%s)\n",
			bus->hose, bus->number,
			PCIVariant( bus->hose, bridgedflag ) );

#ifdef PCIDEBUG
    mobo_logf(LOG_INFO "\tprimary:%d, secondary:%d, subordinate:%d\n",
	      bus->primary, bus->secondary, bus->subordinate);
    mobo_logf(LOG_INFO "\tI/O base:%X, Mem base:%X\n", bus->PCI_IO_Base,
	      bus->PCI_Mem_Base);
    mobo_logf(LOG_INFO "\tI/O size:%X, Mem size:%X\n", bus->PCI_IO_Size,
	      bus->PCI_Mem_Size);
    mobo_logf(LOG_INFO "Bus = %d (address = %X)\n", bus->number, bus);
    mobo_logf(LOG_INFO "\tbridge = %X\n", bus->bridge);
    mobo_logf(LOG_INFO "\tchildren = %X, next = %X\n", bus->children,
	      bus->next);
    mobo_logf(LOG_INFO "\those = %d\n", bus->hose);
#endif


/*
 *  Show all the devices on this bus.
 */
    mobo_logf(
	LOG_INFO
	  "Slot Vendor Device Rev Cmnd I/O Base Mem Base  Class Description\n"
	LOG_INFO
	  "==== ====== ====== === ==== ======== ========  ===== ===========\n");

    while (device != NULL)
    {
	PCIShowDevice( bus, device );
	device = device->next;
    }


/*
 *  Show all of the sub-buses.
 */
    while (child != NULL)
    {
	PCIShowBusLog( child );
	child = child->next;
    }
}


/*--------------------------------------------------------------------*/
/* Diagnostic Routine Entry Points */

/* Describe the PCI bus configuration using platform-specific higher level
 * information.
 */

DBM_STATUS pcishow( int argc, char *argv[] )
{
    unsigned i;

    TRACE( "Displaying %d hoses (%d static buses), pci_hose is 0x%lX "
	   "[0x%lX, 0x%lX, ...]\n", 
		nb_pci_hosecount, plat_pci_hose_count, pci_hose,
		pci_hose[0], pci_hose[1] );

    for (i=0; i < nb_pci_hosecount; i++)
    {
	/* Display this pci_hose, ie tree of PCI buses and bridged buses */

	TRACE( "Displaying hose %d (of %d), pci_hose is 0x%lX\n",
		i, nb_pci_hosecount, pci_hose[i] );

	PCIShowBusUI( pci_hose[i] );	/* kick off recursive traversal */

	TRACE( "Logging hose %d (of %d), pci_hose is 0x%lX\n",
		i, nb_pci_hosecount, pci_hose[i] );

	PCIShowBusLog( pci_hose[i] );
    }


    /* clean-up afterwards */

    mobo_zap(r_app);
    return STATUS_SUCCESS;
}


/*--------------------------------------------------------------------*/
/* Display the PCI configuration space header of a given device */

DBM_STATUS pcihdr( int argc, char *argv[] )
{
    unsigned bus, dev, fun;
    String slotname;
    String devclass;
    PCIDevice_t *D;
    unsigned i, j, uval;
    int bridgedflag = 0;

    /*----------------------------------------------------------------*/
    /* First find the device the user is talking about */

    if ( argc != 4 )
    {
	mobo_alertf( "Incorrect usage",
			"Usage: pcihdr <pci bus> <device> <function>" );
	return STATUS_FAILURE;
    }
    bus = atoi( argv[ 1 ] );
    dev = atoi( argv[ 2 ] );
    fun = atoi( argv[ 3 ] );

    slotname = PCIDev2Str( bus, dev, bridgedflag );

    /* Find the corresponding PCI device in the device tree */
    for ( D=devices; D != NULL; D=D->sibling )
    {
	if ( (D->parent->number == bus) && (D->slot == dev) )
		break;
    }

    if ( D==NULL ) {
	mobo_alertf( "Can't find device",
			"There is no PCI device %d (is it really in %s?)",
			dev, slotname );
	return STATUS_FAILURE;
    }

    devclass = DeviceClassString( D->class );


    /*----------------------------------------------------------------*/
    /* Now dump the PCI configuration space header of that device */

    mobo_cls();
    printf_dbm( "Bus %d Device %d Function %d: %s in %s\r\r",
		bus, dev, fun, devclass, slotname );
#define COLSIZE 0x40

    for ( i=0; i< COLSIZE; i+=4 ) {
	for ( j = 0; j < 4; j++ ) {
	    uval = pcicfgrl( bus, dev, fun, j*COLSIZE + i );
	    printf_dbm( "0x%02X: 0x%08X   ", j*COLSIZE + i, uval );
	}
	printf_dbm( "\r" );
    }


    printf_dbm( "\r\rPress any key to return\r" );
    mobo_key( 0 );
    return STATUS_SUCCESS;
}
