/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* 'go' test - all tests in one */
/* Begun by Stig Telfer, API, May 11 1999 */


#include "lib.h"
#include "uilib.h"
#include "info.h"
#include "platform.h"
#include "pci.h"
#include "northbridge.h"
#include "cmos_rtc.h"
#include "kbd.h"
#include "floppy.h"
#include "rom.h"
#include "mem.h"		/* local memory test module interface */





/*----------------------------------------------------------------------*/
/* Data, functions and definitions that are private to this module */


/* Record a new failure in the list */

typedef struct _failure { String S; struct _failure *next; } list_t;
static list_t *fail = NULL;

static list_t *new_failure( const String S )
{
    list_t *tmp;

    tmp = malloc( sizeof( list_t ) );
    if ( tmp != NULL )
    {
	tmp->S = S, tmp->next = NULL;
    }
    return tmp;					/* NULL  if failure */
}

static void add_to_failures( const String S )
{
    list_t *tmp = fail;

    if ( fail == NULL )		/* first failure */
    {
	fail = new_failure( S );
	return;
    }

    while( tmp->next != NULL )		tmp = tmp->next;
    tmp->next = new_failure( S );
    return;
}



static DBM_STATUS global_error = STATUS_SUCCESS;

static DBM_STATUS try_command( const String cmd, const String desc )
{
    DBM_STATUS sval;

    sval = diags_usercmd( cmd );
    if ( sval != STATUS_SUCCESS )
    {
	global_error = STATUS_FAILURE;
	add_to_failures( desc );
    }

    mobo_cls();			/* clean up any chaff left behind */
    return sval;
}


static void free_fail_list( void )
{
    list_t *tmp = fail;

    while ( fail != NULL )
    {
	tmp = fail;
	fail = fail->next;
	free( tmp );
    }
}



/*----------------------------------------------------------------------*/
/* Main driver routine */



DBM_STATUS go( int argc, char *argv[] )
{
    DBM_STATUS sval = STATUS_SUCCESS;
    int ipl = swpipl( 7 );
    int rval;
    int bcache;
    char cmdstr[16];
    list_t *tmp;
    Point P;

    global_error = STATUS_SUCCESS;		/* reset in case second run */

    /* user may have set a new log stream (this is the only arg we accept) */
    if ( argc > 1 )
    {
	sprintf_dbm( cmdstr, "log %s", argv[1] );
	diags_usercmd( cmdstr );
    }


    mobo_uistate( UI_BATCH );
    system_datestamp( "Starting full diagnostic" );
    plat_nmiclr();		/* Clear the slate before tests */


    /* List all known configuration data */
    info_log();

    /* PCI bus initialisation */
    try_command( "pci", "PCI bus initialisation" );


    /* Platform-specific POST routines */
    mobo_logf( LOG_INFO "\n++++++++Starting POST routines+++++++++\n");
    sval = plat_post();
    if ( sval != STATUS_SUCCESS )
	add_to_failures( "Platform POST routine" );
    mobo_logf( LOG_INFO "++++++++Finished POST routines+++++++++\n\n");


#ifdef CONFIG_SMP
    try_command( "smp start", "Secondary CPU startup" );
    try_command( "smp ipi", "Interprocessor ping-pong" );
#endif
 
    /* Run the memory test for the bcache size */
    if ( primary_impure->BEHAV_DATA == 0 )
		bcache = 2;
    else 	
		bcache = (primary_impure->BEHAV_DATA >> 8 ) & 0xFF;

#define BCACHESECS	60
#define MEMSECS		180		/* expect length of tests in seconds */

    if ( bcache == 0 ) {
	mobo_logf( LOG_INFO "---------Bcache test skipped, zero size----------\n");
    } else {
	mobo_cls();
	printf_dbm( "Cache-resident small memory test (%d mins)", 
		BCACHESECS / 60 );
        memtest_secs = BCACHESECS;		/* first a test of bcache */
	swpipl( 0 );		/* enable ints for memory test */
	sprintf_dbm( cmdstr, "mem %d %d", STACK_MEGS, STACK_MEGS+bcache );
	try_command( cmdstr, "Cache-resident memory test" );
    }


    /* full memory test - here we make some kind of approximation to the
     * test duration. */

    mobo_cls();
    printf_dbm( "Full-scale memory test (%d mins)", MEMSECS / 60 );

    memtest_secs = MEMSECS;
    swpipl( 0 );			/* enable ints for memory test */
    try_command( "mem", "Full memory test" );

    try_command( "i2c", "I2C bus test" );

    try_command( "eeprom", "Asset Information EEPROM integrity" );

    try_command( "irq", "Interrupt subsystem test" );


    swpipl( 7 );		/* debug: this test runs with ints disabled */
    try_command( "isa", "Integrated devices test" );


    /* VGA card test */
    try_command( "vga", "VGA adapter test" );
    plat_nmiclr();		/* BIOSem sometimes screws up with NXMs */


    /* return to interactive state and report */
    system_datestamp( "Full diagnostic complete" );
    mobo_uistate( UI_INTERACT );


    if ( global_error == STATUS_SUCCESS )
    {
	mobo_alertf( "The full test suite passed",
		     "All tests completed successfully" );
	mobo_logf( LOG_INFO "--------Final Result: Passed--------\n" );

    } else {

	mobo_logf( LOG_CRIT "--------Final Result: FAILED--------\n"
		   LOG_CRIT "Failing test modules:\n" );

	mobo_box( r_lrgapp, "The full test suite failed!" );
	mobo_goto( p_app );
	printf_dbm( "The failing test modules were:\r" );

	for ( tmp = fail; tmp != NULL; tmp = tmp->next ) 
	{
	    mobo_logf( LOG_CRIT "%s\n", tmp->S );
	    printf_dbm( "%s\r", tmp->S );
	}

        /* add a 'press a key' message */
	P.x = r_lrgapp.x + ( (r_lrgapp.w-strlen( s_ekey )) >> 1 );
	P.y = r_lrgapp.y + r_lrgapp.h - 1;
        mobo_goto( P );
        printf_dbm( s_ekey );
        rval = mobo_key( 0 );
	mobo_zap( r_lrgapp );

	free_fail_list( );
    }


    /* user may want to start the AlphaBIOS for next level of tests */

    rval = mobo_alertf(	"OS-level diagnostics",
			"Transfer to next-level firmware? (y/n)");

    if ( tolower( rval ) == 'y' )
	diags_usercmd( "bios" );

    swpipl( ipl );

    plat_reset( 0, NULL );	/* clean the slate */
    return sval;		/* reaches here only if reset doesn't work */
}


