/*
 *  linux/drivers/block/triton.c	Version 1.13  Aug 12, 1996
 *
 *  Copyright (c) 1995-1996  Mark Lord
 *  May be copied or modified under the terms of the GNU General Public License
 */

/*
 * This module provides support for the Bus Master IDE DMA function
 * of the Intel PCI Triton I/II chipsets (i82371FB or i82371SB).
 *
 * Pretty much the same code will work for the OPTi "Viper" chipset.
 * Look for DMA support for this in linux kernel 2.1.xx, when it appears.
 *
 * DMA is currently supported only for hard disk drives (not cdroms).
 *
 * Support for cdroms will likely be added at a later date,
 * after broader experience has been obtained with hard disks.
 *
 * Up to four drives may be enabled for DMA, and the Triton chipset will
 * (hopefully) arbitrate the PCI bus among them.  Note that the i82371 chip
 * provides a single "line buffer" for the BM IDE function, so performance of
 * multiple (two) drives doing DMA simultaneously will suffer somewhat,
 * as they contest for that resource bottleneck.  This is handled transparently
 * inside the i82371 chip.
 *
 * By default, DMA support is prepared for use, but is currently enabled only
 * for drives which support multi-word DMA mode2 (mword2), or which are
 * recognized as "good" (see table below).  Drives with only mode0 or mode1
 * (single or multi) DMA should also work with this chipset/driver (eg. MC2112A)
 * but are not enabled by default.  Use "hdparm -i" to view modes supported
 * by a given drive.
 *
 * The hdparm-2.4 (or later) utility can be used for manually enabling/disabling
 * DMA support, but must be (re-)compiled against this kernel version or later.
 *
 * To enable DMA, use "hdparm -d1 /dev/hd?" on a per-drive basis after booting.
 * If problems arise, ide.c will disable DMA operation after a few retries.
 * This error recovery mechanism works and has been extremely well exercised.
 *
 * IDE drives, depending on their vintage, may support several different modes
 * of DMA operation.  The boot-time modes are indicated with a "*" in
 * the "hdparm -i" listing, and can be changed with *knowledgeable* use of
 * the "hdparm -X" feature.  There is seldom a need to do this, as drives
 * normally power-up with their "best" PIO/DMA modes enabled.
 *
 * Testing was done with an ASUS P55TP4XE/100 system and the following drives:
 *
 *   Quantum Fireball 1080A (1Gig w/83kB buffer), DMA mode2, PIO mode4.
 *	- DMA mode2 works well (7.4MB/sec), despite the tiny on-drive buffer.
 *	- This drive also does PIO mode4, at about the same speed as DMA mode2.
 *	  An awesome drive for the price!
 *
 *   Fujitsu M1606TA (1Gig w/256kB buffer), DMA mode2, PIO mode4.
 *	- DMA mode2 gives horrible performance (1.6MB/sec), despite the good
 *	  size of the on-drive buffer and a boasted 10ms average access time.
 *	- PIO mode4 was better, but peaked at a mere 4.5MB/sec.
 *
 *   Micropolis MC2112A (1Gig w/508kB buffer), drive pre-dates EIDE and ATA2.
 *	- DMA works fine (2.2MB/sec), probably due to the large on-drive buffer.
 *	- This older drive can also be tweaked for fastPIO (3.7MB/sec) by using
 *	  maximum clock settings (5,4) and setting all flags except prefetch.
 *
 *   Western Digital AC31000H (1Gig w/128kB buffer), DMA mode1, PIO mode3.
 *	- DMA does not work reliably.  The drive appears to be somewhat tardy
 *	  in deasserting DMARQ at the end of a sector.  This is evident in
 *	  the observation that WRITEs work most of the time, depending on
 *	  cache-buffer occupancy, but multi-sector reads seldom work.
 *
 * Testing was done with a Gigabyte GA-586 ATE system and the following drive:
 * (Uwe Bonnes - bon@elektron.ikp.physik.th-darmstadt.de)
 *
 *   Western Digital AC31600H (1.6Gig w/128kB buffer), DMA mode2, PIO mode4.
 *	- much better than its 1Gig cousin, this drive is reported to work
 *	  very well with DMA (7.3MB/sec).
 *
 * Other drives:
 *
 *   Maxtor 7540AV (515Meg w/32kB buffer), DMA modes mword0/sword2, PIO mode3.
 *	- a budget drive, with budget performance, around 3MB/sec.
 *
 *   Western Digital AC2850F (814Meg w/64kB buffer), DMA mode1, PIO mode3.
 *	- another "caviar" drive, similar to the AC31000, except that this one
 *	  worked with DMA in at least one system.  Throughput is about 3.8MB/sec
 *	  for both DMA and PIO.
 *
 *   Conner CFS850A (812Meg w/64kB buffer), DMA mode2, PIO mode4.
 *	- like most Conner models, this drive proves that even a fast interface
 *	  cannot improve slow media.  Both DMA and PIO peak around 3.5MB/sec.
 *
 *   Maxtor 71260AT (1204Meg w/256kB buffer), DMA mword0/sword2, PIO mode3.
 *	- works with DMA, on some systems (but not always on others, eg. Dell),
 *	giving 3-4MB/sec performance, about the same as mode3.
 *
 * If you have any drive models to add, email your results to:  mlord@pobox.com
 * Keep an eye on /var/adm/messages for "DMA disabled" messages.
 *
 * Some people have reported trouble with Intel Zappa motherboards.
 * This can be fixed by upgrading the AMI BIOS to version 1.00.04.BS0,
 * available from ftp://ftp.intel.com/pub/bios/10004bs0.exe
 * (thanks to Glen Morrell <glen@spin.Stanford.edu> for researching this).
 *
 * And, yes, Intel Zappa boards really *do* use the Triton IDE ports.
 */


/* PAGE_SHIFT determines the page size */
#define PAGE_SHIFT      13
#define PAGE_SIZE       (1UL << PAGE_SHIFT)
#define PAGE_MASK       (~(PAGE_SIZE-1))

#undef DISPLAY_TRITON_TIMINGS	/* define this to display timings */

/*
 * good_dma_drives() lists the model names (from "hdparm -i")
 * of drives which do not support mword2 DMA but which are
 * known to work fine with this interface under Linux.
 */
const char *good_dma_drives[] = {"Micropolis 2112A",
				 "CONNER CTMA 4000",
				 "CONNER CTT8000-A",
				 NULL};

/*
 * Our Physical Region Descriptor (PRD) table should be large enough
 * to handle the biggest I/O request we are likely to see.  Since requests
 * can have no more than 256 sectors, and since the typical blocksize is
 * two sectors, we could get by with a limit of 128 entries here for the
 * usual worst case.  Most requests seem to include some contiguous blocks,
 * further reducing the number of table entries required.
 *
 * The driver reverts to PIO mode for individual requests that exceed
 * this limit (possible with 512 byte blocksizes, eg. MSDOS f/s), so handling
 * 100% of all crazy scenarios here is not necessary.
 *
 * As it turns out though, we must allocate a full 4KB page for this,
 * so the two PRD tables (ide0 & ide1) will each get half of that,
 * allowing each to have about 256 entries (8 bytes each) from this.
 */
#define PRD_BYTES	8
#define PRD_ENTRIES	(PAGE_SIZE / (2 * PRD_BYTES))
#define DEFAULT_BMIBA	0xe800	/* in case BIOS did not init it */

/*
 * dma_intr() is the handler for disk read/write DMA interrupts
 */
static void dma_intr (ide_drive_t *drive)
{
	byte stat, dma_stat;
	int i;
	struct request *rq = drive->rq;
	unsigned int dma_base = HWIF(drive)->dma_base;

/*
	if (!ide_wait_stat(drive, DRIVE_READY,drive->bad_wstat|DRQ_STAT, WAIT_CMD)) {
*/
	udelay(1);
	while (!(inb(dma_base+2) & 0x6) ) 
		udelay(1);
	stat = GET_STAT();
	{
printk("dma OK %x\n", inb(dma_base+2));
		dma_stat = inb(dma_base+2);		/* get DMA status */
		outb(inb(dma_base)&~1, dma_base);	/* stop DMA operation */
		if ((dma_stat & 6) == 4) {	/* verify good DMA status */
			rq = drive->rq;
			for (i = rq->nr_sectors; i > 0;) {
				i -= rq->current_nr_sectors;
				drive->rq = drive->rq->next;
			}
			return;
		}
		printk("%s: bad DMA status: 0x%02x\n", drive->name, dma_stat);
	}
printk("dma bad\n");
	ide_error(drive, "dma_intr", stat);
}

/*
 * build_dmatable() prepares a dma request.
 * Returns 0 if all went okay, returns 1 otherwise.
 */
static int build_dmatable (ide_drive_t *drive)
{
	struct request *rq = drive->rq;
	unsigned int size, addr, *table = HWIF(drive)->dmatable;
	unsigned int count = 0;

	addr = rq->buffer;
if (addr > 0x7fffffff)
	printk("too big address\n");
	size = rq->nr_sectors << 9;

printk("build table = %x\n", table);
	/*
	 * Fill in the dma table, without crossing any 64kB boundaries.
	 * We assume 16-bit alignment of all blocks.
	 */
	while (size) {
		if (++count >= PRD_ENTRIES) {
			printk("%s: DMA table too small\n", drive->name);
			return 1; /* revert to PIO for this request */
		} else {
			unsigned int bcount = 0x10000 - (addr & 0xffff);
			if (bcount > size)
				bcount = size;
			*table++ = addr;
printk("table %x = %x\n", table - 1, *(table -1));
			*table++ = bcount & 0xffff;
printk("table %x = %x\n", table - 1, *(table -1));
			mb();
			addr += bcount;
			size -= bcount;
		}
	}
	if (count) {
		*--table |= 0x80000000;	/* set End-Of-Table (EOT) bit */
		return 0;
	}
	printk("%s: empty DMA table?\n", drive->name);
	return 1;	/* let the PIO routines handle this weirdness */
}

static int config_drive_for_dma (ide_drive_t *drive)
{
	const char **list;
	struct hd_driveid *id = drive->id;

printk("capa %x field %x ultra %x\n", id->capability, id->field_valid, id->dma_ultra);

	if (id && (id->capability & 1)) {
		/* Enable DMA on any drive that has UltraDMA (mode 0/1/2) enabled */
		if (id->field_valid & 4)	/* UltraDMA */
			if  ((id->dma_ultra & (id->dma_ultra >> 8) & 7)) {
				drive->using_dma = 1;
				return 0;	/* dma enabled */
			}
		/* Enable DMA on any drive that has mode2 DMA (multi or single) enabled */
		if (id->field_valid & 2)	/* regular DMA */
			if  ((id->dma_mword & 0x404) == 0x404 || (id->dma_1word & 0x404) == 0x404) {
				drive->using_dma = 1;
				return 0;	/* dma enabled */
			}
		/* Consult the list of known "good" drives */
		list = good_dma_drives;
		while (*list) {
			if (!strcmp(*list++,id->model)) {
				drive->using_dma = 1;
				return 0;	/* DMA enabled */
			}
		}
	}
	return 1;	/* DMA not enabled */
}

/*
 * triton_dmaproc() initiates/aborts DMA read/write operations on a drive.
 *
 * The caller is assumed to have selected the drive and programmed the drive's
 * sector address using CHS or LBA.  All that remains is to prepare for DMA
 * and then issue the actual read/write DMA/PIO command to the drive.
 *
 * For ATAPI devices, we just prepare for DMA and return. The caller should
 * then issue the packet command to the drive and call us again with
 * ide_dma_begin afterwards.
 *
 * Returns 0 if all went well.
 * Returns 1 if DMA read/write could not be started, in which case
 * the caller should revert to PIO for the current request.
 */
static int triton_dmaproc (ide_dma_action_t func, ide_drive_t *drive)
{
	unsigned int dma_base = HWIF(drive)->dma_base;
	unsigned int reading = (1 << 3);

DBG("dmaproc\n");
	switch (func) {
		case ide_dma_abort:
DBG("ide_dma_abort\n");
			outb(inb(dma_base)&~1, dma_base);	/* stop DMA */
			return 0;
		case ide_dma_check:
DBG("ide_dma_check\n");
			return config_drive_for_dma (drive);
		case ide_dma_write:
DBG("ide_dma_write\n");
			reading = 0;
		case ide_dma_read:
DBG("ide_dma_read\n");
			break;
		case ide_dma_status_bad:
			return ((inb(dma_base+2) & 7) != 4);	/* verify good DMA status */
		case ide_dma_transferred:
#if 0
			return (number of bytes actually transferred);
#else
			return (0);
#endif
		case ide_dma_begin:
			outb(inb(dma_base)|1, dma_base);	/* begin DMA */
			return 0;
		default:
			printk("triton_dmaproc: unsupported func: %d\n", func);
			return 1;
	}
	if (build_dmatable (drive))
		return 1;
	outl(HWIF(drive)->dmatable, dma_base + 4); /* PRD table */
	outb(reading, dma_base);			/* specify r/w */
	outb(inb(dma_base+2)|0x06, dma_base+2);		/* clear status bits */
#ifdef CONFIG_BLK_DEV_IDEATAPI
	if (drive->media != ide_disk)
		return 0;
#endif /* CONFIG_BLK_DEV_IDEATAPI */	
	OUT_BYTE(reading ? WIN_READDMA : WIN_WRITEDMA, IDE_COMMAND_REG);
	outb(inb(dma_base)|1, dma_base);		/* begin DMA */
	dma_intr(drive);	/* issue cmd to drive */
	return 0;
}

#ifdef DISPLAY_TRITON_TIMINGS
/*
 * print_triton_drive_flags() displays the currently programmed options
 * in the i82371 (Triton) for a given drive.
 *
 *	If fastDMA  is "no", then slow ISA timings are used for DMA data xfers.
 *	If fastPIO  is "no", then slow ISA timings are used for PIO data xfers.
 *	If IORDY    is "no", then IORDY is assumed to always be asserted.
 *	If PreFetch is "no", then data pre-fetch/post are not used.
 *
 * When "fastPIO" and/or "fastDMA" are "yes", then faster PCI timings and
 * back-to-back 16-bit data transfers are enabled, using the sample_CLKs
 * and recovery_CLKs (PCI clock cycles) timing parameters for that interface.
 */
static void print_triton_drive_flags (unsigned int unit, byte flags)
{
	printk("         %s ", unit ? "slave :" : "master:");
	printk( "fastDMA=%s",	(flags&9)	? "on " : "off");
	printk(" PreFetch=%s",	(flags&4)	? "on " : "off");
	printk(" IORDY=%s",	(flags&2)	? "on " : "off");
	printk(" fastPIO=%s\n",	((flags&9)==1)	? "on " : "off");
}
#endif /* DISPLAY_TRITON_TIMINGS */

static unsigned int ali_base;

static void init_triton_dma (ide_hwif_t *hwif)
{
	static unsigned int dmatable = 0;
	static unsigned int  base;

	base = ali_base;
	printk("    %s: BM-DMA at 0x%04x-0x%04x\n", hwif->name, base, base+7);
	hwif->dma_base = base;
	if (!dmatable) {
		/*
		 * The BM-DMA uses a full 32-bits, so we can
		 * safely use __get_free_page() here instead
		 * of __get_dma_pages() -- no ISA limitations.
		dmatable = __get_free_pages(GFP_KERNEL, 1, 0);
		 */
		dmatable = malloc(PAGE_SIZE);
printk("dmatable %x\n", dmatable);
printk("dmatable %lx\n", dmatable);
	}
	if (dmatable) {
		hwif->dmatable = (unsigned int *) dmatable;
		dmatable += (PRD_ENTRIES * PRD_BYTES);
		outl(hwif->dmatable, base + 4);
		hwif->dmaproc  = &triton_dmaproc;
	}
}

#define ALI_DEVFN (16<<3 | 0)

#define pcibios_read_config_byte(A,B,C,D) *(D) = pcicfgrb((A), (B)>>3, (B)&0x3, (C))
#define pcibios_write_config_byte(A,B,C,D) pcicfgwb((A), (B)>>3, (B)&0x3, (C), (D))
void ide_init_alim15x3 ()
{
        byte confreg0 = 0, confreg1 =0, progif = 0;
        int errors = 0;

        pcicfgwb(0, 7, 0, 0x58, 0x4d);
        pcicfgwb(0, 7, 0, 0x48, 0x31);
        pcicfgwb(0, 7, 0, 0x49, 0xb9);
        pcicfgwb(0, 7, 0, 0x4c, 0x0f);
        pcicfgwb(0, 16, 0, 4, 5);
        pcicfgwl(0, 16, 0, 0x20, 0xf000);
        pcicfgwb(0, 0, 0, 4, 6);
	confreg0 = pcicfgrb(0,0,0,0x5a);
        pcicfgwb(0, 0, 0, 0x5a, confreg0 | 0x4);
        pcicfgwb(0, 0, 0, 0x56, 0xcc);

	confreg0 = pcicfgrb(0,0,0,0x84);
        pcicfgwb(0, 0, 0, 0x84, confreg0 | 0x10);
        pcicfgwb(0, 0, 0, 0x71, 0x06);

        ali_base = pcicfgrw(0, 16, 0, 0x20) & 0xfffe;
	if (ali_base != 0xf000) printk("base if not 0xf000\n");

        pcibios_read_config_byte(0, ALI_DEVFN, 0x50, &confreg1);
        if (!(confreg1 & 0x02))
                pcibios_write_config_byte(0, ALI_DEVFN, 0x50, confreg1 | 0x02);

        pcibios_read_config_byte(0, ALI_DEVFN, 0x09, &progif);
        if (!(progif & 0x40)) {
                /*
                 * The way to enable them is to set progif
                 * writable at 0x4Dh register, and set bit 6
                 * of progif to 1:
                 */
                pcibios_read_config_byte(0, ALI_DEVFN, 0x4d, &confreg0);
                if (confreg0 & 0x80)
                        pcibios_write_config_byte(0, ALI_DEVFN, 0x4d, confreg0 & ~0x80);
                pcibios_write_config_byte(0, ALI_DEVFN, 0x09, progif | 0x40);
                if (confreg0 & 0x80)
                        pcibios_write_config_byte(0, ALI_DEVFN, 0x4d, confreg0);
        }

        pcibios_read_config_byte(0, ALI_DEVFN, 0x09, &progif);
	if (!(progif & 0x40))
                goto veryspecialsettingserror;

        printk("ALI enabled read of IDE channels state (en/dis-abled) %s.\n",
                errors ? "with Error(s)" : "Succeeded" );

        return 0;

veryspecialsettingserror:
        printk("ALI impossible to enable read of IDE channels state (en/dis-abled)!\n");
        return 0;
}

