/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/
/* Alpha Diagnostics User Interface Framework */

#include <stdarg.h>

#undef TRACE_ENABLE		/* define this to enable verbose debug */

#include "lib.h"
#include "uilib.h"
#include "platform.h"
#include "northbridge.h"
#include "southbridge.h"
#include "smp.h"			/* for concurrency control */
#include "include/vga.h"

#include "osf.h"			/* for startup codes */



/* ANSI escape sequences */
#define CLS ESC "[2J"

/* ANSI sequences - Note: 1-based coordinate system */
#define CUP(x,y) ESC "[" #y ";" #x "H"	/* Cursor position */
#define CUB(n)   ESC "[" #n "D"        	/* Cursor backwards */
#define CUD(n)   ESC "[" #n "B"        	/* cursor downwards */
#define CUU(n)   ESC "[" #n "A"        	/* cursor upwards */


/*----------------------------------------------------------------------*/
/* Global Data */

char cbuf[CBUFSIZ];			/* command text goes in here */
Point soft_cr;
Point cursor = { 1, 1 };
Point beginning = {1, 1};
const String s_ekey = "Press Any Key...";	/* has a thousand uses... */
const Point p_prompt = {3, 23}, p_help = { 3, 22 };


smp_mutex disp_lock = M_FREE;


/*----------------------------------------------------------------------*/
/* User Interface Operations */



int mobo_key( int to )
{
    /* in batch mode we don't interact,
     * but we still accept keypress checks with timeouts */
    if ( state == UI_BATCH && to == 0 )			return -1;

    /* if timeout == 0, we have an infinite wait */
    if (to == 0)	return GetChar();
    else		return GetChar_t( to );
}


/* mobo_input() returns the string length, not including CR or \0 */

int mobo_input( String buf, int bufsz )
{
    int i;
    char c;

    for (i = 0; i < bufsz-1; ) {

	c = mobo_key( 0 );

	TRACE( "Got input '%c' (0x%02X)\n", c, c );

	if (c == '\r' || c == '\n') {		/* carriage return */
	    buf[i] = '\x00';			/* null-terminate the cmd */
	    break;
	}

	if (c == '\b' ) {			/* backspace */
	    if ( i > 0 ) {			/* no action at line start */
		i--;
		printf_dbm("\b \b");		/* bksp, spc, bksp to clear */
	    }
	    continue;
	}

	/* Usual case (another keypress) */
	buf[i] = c;
	printf_dbm( "%c", c );
	i++;
    }

    return i;					/* number of chars entered */
}


void mobo_printf(const String s,...)
{
    va_list ap;
    va_start(ap, s);

    vfprintf_dbm( stdout_dbm, s, ap);		/* for now, just pass it on */

    va_end(ap);
}


void mobo_cls(void)
{
    printf_dbm(CLS);
    mobo_goto( beginning );
}


void mobo_goto(Point p)
{
    printf_dbm("\x1B[%d;%dH", p.y, p.x);

    soft_cr.x = p.x;
    soft_cr.y = p.y + 1;
    cursor = p;				/* keep track of current location */
}


void mobo_zap(Rect r)
{
    Point p;
    unsigned i, j;

    p.x = r.x, p.y = r.y;
    mobo_goto(p);

    for (i = 0; i < r.h; i++) {
	for (j = 0; j < r.w; j++)
	    printf_dbm(" ");

	mobo_goto( soft_cr );
    }
}


void mobo_box(Rect r, String s)
{
    unsigned i, xiters, yiters;
    char *l;
    unsigned len;
    Point p, ptmp;

    p.x = r.x, p.y = r.y;

    printf_dbm( CU_NORMAL );			/* set colour up */
    mobo_zap(r);

    /* Draw the Box */
    xiters = r.w - 2, yiters = r.h - 2;

    mobo_goto(p);			/* top left corner */
    printf_dbm(BOX_TL);
    for (i = 0; i < xiters; i++)	/* top edge */
	printf_dbm(BOX_H);
    printf_dbm(BOX_TR);			/* top right corner */

    /* right edge - can mess up if edge is against screen edge, so be careful */
    for (ptmp.x = r.x+r.w-1, ptmp.y = r.y + 1, i = 0; i < yiters; i++, ptmp.y++)
    {
	mobo_goto( ptmp );
	printf_dbm( BOX_V );
    }

    mobo_goto(p);				/* back to top left corner */
    for (i = 0; i < yiters; i++)
	printf_dbm(CUD(1) BOX_V CUB(1));	/* left edge */
    printf_dbm(CUD(1) BOX_BL);			/* bottom left corner */

    for (i = 0; i < xiters; i++)		/* bottom edge */
	printf_dbm(BOX_H);

    printf_dbm(BOX_BR);			/* bottom right corner */


    /* print the box title string */
    for (l = s, len = 0; *l; l++)
	len++;			/* calc string length */
    len >>= 1;
    p.x += (r.w >> 1) - len;
    mobo_goto(p);
    printf_dbm( FG_WHITE "%s" CU_NORMAL, s);


    /* move to a point inside the box */
    p.x = r.x + 1, p.y = r.y + 1;
    mobo_goto( p );
}




/*----------------------------------------------------------------------*/
/* Menu management */

/* routine to perform lookup in the command table, given command string */
const Cmd_t *mobo_cmdlookup( const char *S, const Cmd_t *C, const unsigned nC)
{
    const Cmd_t *p;
    unsigned cmd, len;

    /* attempt a match with each command in the table */

    for (p=C, cmd=0; cmd < nC; p++, cmd++)
    {
	len = strlen( p->name );
	if ( strncmp( S, p->name, len ) != 0 )	continue;
	
	/* matched so far... is cmd all used? */
	if ( isalpha( S[len] ) )		continue;

	/* otherwise, we have found our command */
	return p;
    }

    return NULL;	/* if we are here, there was no match, return NULL */
}


/* Routine to execute a function (if the environment is set up OK) */

extern unsigned diags_environment;

#define MAX_ARGS 8

DBM_STATUS mobo_runcmd( const Cmd_t *cmd, String cmdline )
{
    String argv[ MAX_ARGS ];
    int argc;
    int i;
    unsigned func_req;

    /* FIXME: does the command meet the current environment mask? */
    
    func_req = cmd->env_mask & ~diags_environment;

    if( func_req != SYS_NONE )
    {
	mobo_logf( LOG_INFO "some machine initialisation is required "
			    "to run command '%s'\n", cmd->name );

	/* Here we attempt to set the machine up to meet the requirements of
	 * the command.  We're trying to be robust, so don't pay too much 
	 * attention to failures. */

	if ( func_req & (SYS_PCI | SYS_ISA) )
	{
	    nb_setup();
	    sb_setup();
	    plat_setup();

	    nb_fixup();
	    sb_fixup();
	    plat_fixup();
	}

	if ( func_req & SYS_VGA )
	    InitVGA();

	if ( func_req & SYS_IRQ )
	    plat_intsetup( 1 );

	if ( func_req & SYS_SMP )
	    plat_smpstart( primary_impure->PAL_BASE );

	if ( func_req & SYS_I2C )
	    plat_i2cinit();

 	if ( func_req & SYS_CONS )
	    bring_up_console();
    }


    for ( i=0; i<MAX_ARGS; i++ )		/* for tidiness */
    {
	argv[ i ] = NULL;
    }


    if ( cmdline == NULL )			/* no arguments passed */
    {
	argv[0] = cmd->name;
	argc = 1;
    } else {

	/* Tokenize command line to produce arguments */
	for ( i=0, argc=0; cmdline[i] != 0; i++ )
	{
	    if( isgraph( cmdline[i] ) )		/* found our next token */
	    {
		argv[argc] = cmdline+i;
		argc++;
		
		while( isgraph( cmdline[i] ) )
		    i++;

		if ( cmdline[i] == 0 )		/* end of command string? */
		    break;

		cmdline[i] = 0;			/* Add null-char termination */
	    }
	}
    }

    /* run it and return */
    return cmd->action( argc, argv );
}



/* Routine to parse a command, and pass on to execute */
int mobo_menu(  const Cmd_t *M, const unsigned nM, 
		const Cmd_t *C, const unsigned nC )
{
    char c;
    int i;
    int opt;
    const Cmd_t *p;
    BOOLEAN escflag = FALSE;
    static const String HelpMsg = "Type 'help' for command summary";

    /* setup for user interaction */

    mobo_goto(p_help);
    printf_dbm( HelpMsg );			/* 'type help' string */

    mobo_goto(p_prompt);
    printf_dbm( Prompt );			/* platform prompt string */


    /* phase 1: accept keyboard input until carriage return  */

    for (i = 0; i < CBUFSIZ - 1;) {

	c = mobo_key(0);

	TRACE( "Got input '%c' (0x%02X)\n", c, c );

	/* Special case characters */

	/* Control codes (eg DEL) are preceded by ESC */
	if ( c == '\x1B' )
	{
	    escflag = TRUE;
	    continue;
	}

	/* If previous character was ESC, is the next one something we know? */
	if ( escflag == TRUE )
	{
	    escflag = FALSE;

	    /* Note: on VT102, DEL is ESC-0x7E (as is other ctrl codes??) */
	    /* On ANSI terminals, DEL is ESC-0x7F */
	    if ( c == '\x7E' || c == '\x7F' )
	    {
		c = '\b';		/* DEL converts to backspace */
	    }
	}

	if (i == 0 && isdigit(c)) {	/* menu action */
	    opt = c - '1';	/* if invalid, just return */

	    /* find the menu it refers to, perform the action */
	    if ( opt >= 0 && opt < nM )
	    {
		if ( M[opt].action != NULL )
		{
		    mobo_runcmd( M + opt, NULL );
	    	    return 0;
		} else return -1;		/* signifies a quit option */
	    }
	}
	if (c == '\r' || c == '\n') {	/* carriage return */
	    cbuf[i] = '\x00';	/* null-terminate the cmd */
	    break;
	}
	if (c == '\b') {	/* backspace */
	    if (i > 0) {	/* no action at line start */
		i--;
		printf_dbm("\b \b");	/* bksp, spc, bksp to clear */
	    }
	    continue;
	}
	/* Usual case (another keypress) */
	cbuf[i] = c;
	printf_dbm("%c", c);
	i++;
    }

    /* At this point, we have a command string in cbuf, what is it? */
    p = mobo_cmdlookup( cbuf, C, nC );
    if ( p==NULL )			/* no match */
    {
	mobo_alertf( "Unrecognised command",
			"Try 'help' for a full list of commands" );
	return 0;
    }

    if ( p->action != NULL)		/* a quit command? */
    {
	printf_dbm( "..." );		/* indicate that somethings happening */
	mobo_runcmd( p, cbuf );
	mobo_goto( p_prompt );
	printf_dbm( "%-64s", Prompt );	/* Command completed, blank prompt */
	return 0;
    } else return -1;
}

/*----------------------------------------------------------------------*/
/* User interface internal workings */




/* Rectangular regions of the user interface.  We have no save-under, so 
 * we need to allocate error boxes, etc. to a disjoint area that is 
 * otherwise clear */

static const Point p_ekey  = {33, 22};

const Rect r_error  = {2, 19, 78, 4};			/* error region */
const Rect r_app  = {2, 11, 78, 8};			/* app-specific */
const Rect r_lrgapp  = {2, 11, 78, 11};			/* app-specific */
const Point p_app = {3, 12};

const Rect r_scrn = {1, 1, 80, 25};

static const Rect r_opts = {2, 3, 48, 8};			/* menu opts */
static const Rect r_status = {50, 3, 28, 8};			/* status */

const String err_std = "An Error has occurred";




/*----------------------------------------------------------------------*/
/* Higher-level functionality */


/* mobo_uistate decides whether or not keyboard input should be waited for */
/* It returns the previous state in the manner of swpipl */

static UIstate state = UI_BATCH;

UIstate mobo_uistate( UIstate u )
{
    UIstate oldstate = state;
    state = u;

    return oldstate;
}


/* Error-logging mechanism. */

char mobo_alertf(const String title, const String fmt,...)
{
    int ipl;
    char rval = ' ';			/* dummy value in case of batch mode */
    Point last;
    va_list ap;

    va_start(ap, fmt);

    /* run this code without interrupts - don't want disturbance */
    ipl = swpipl(7);

    /* save current cursor location */
    last = cursor;

    mobo_box(r_error, title);	/* draw the box */
    vfprintf_dbm( stdout_dbm, fmt, ap);


    /* in non-interactive mode, errors just flash up for a short period */
    if ( state == UI_BATCH )	
    {
	msleep( 1000 );
    } else {
	/* add a 'press a key' message */
	mobo_goto(p_ekey);
	printf_dbm( s_ekey );
	rval = mobo_key( 0 );
    }

    mobo_zap( r_error );

    mobo_goto( last );			/* restore cursor position */
    swpipl(ipl);
    va_end(ap);

    return rval;
}

void mobo_logf(const String fmt,...)
{
    va_list ap;
    va_start(ap, fmt);

    vfprintf_dbm( stderr_dbm, fmt, ap );

    va_end( ap );
}


