/*****************************************************************************

       Copyright 1993, 1994, 1995 Digital Equipment Corporation,
                       Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, provided  
that the copyright notice and this permission notice appear in all copies  
of software and supporting documentation, and that the name of Digital not  
be used in advertising or publicity pertaining to distribution of the software 
without specific, written prior permission. Digital grants this permission 
provided that you prominently mark, as not part of the original, any 
modifications made to this software or documentation.

Digital Equipment Corporation disclaims all warranties and/or guarantees  
with regard to this software, including all implied warranties of fitness for 
a particular purpose and merchantability, and makes no representations 
regarding the use of, or the results of the use of, the software and 
documentation in terms of correctness, accuracy, reliability, currentness or
otherwise; and you rely on the software, documentation and results solely at 
your own risk. 

******************************************************************************/
/*---------------------------------------------------------------------
 *        [ Copyright (c) 1999 Alpha Processor Inc.] - Unpublished Work
 *          All rights reserved
 * 
 *    This file contains source code written by Alpha Processor, Inc.
 *    It may not be used without express written permission. The
 *    expression of the information contained herein is protected under
 *    federal copyright laws as an unpublished work and all copying
 *    without permission is prohibited and may be subject to criminal
 *    and civil penalties. Alpha Processor, Inc.  assumes no
 *    responsibility for errors, omissions, or damages caused by the use
 *    of these programs or from use of the information contained herein.
 *  
 *-------------------------------------------------------------------*/


#undef TRACE_ENABLE	/* be careful when enabling this, we're early... */
#undef WALLACE_GROMIT
#undef LOG_TO_ROM		/* still breaks now and then on some kit? */

#include "lib.h"
#include "uilib.h"
#include "info.h"

#include "ledcodes.h"
#include "beepcodes.h"
#include "osf.h"
#include "hwrpb.h"
#include "impure.h"
#include "impure_struct.h"

#include "specifics.h"		/* platform-specific definitions & overrides */
#include "platform.h"
#include "northbridge.h"
#include "southbridge.h"
#include "cpu.h"
#include "smp.h"
#include "rom.h"
#include "nvram.h"
#include "cmos_rtc.h"
#include "diagnostics/diagnostics.h"		/* diag module prototypes */


static void mobo( void );


unsigned char StartupMode = PASSTHROUGH; 	/* default, changed in main */
unsigned char halt_code=0, behav_code=0, env_code=0, os_code=0;


/* This variable points into the decompressor code.  It can be set to TRUE to
 * invoke decompression of the Diags main kernel (ie, this) by the next CPU
 * to pass through. */
unsigned char *diags_do_decompress;

/* This variable is set to TRUE on first decompression of the diags main
 * kernel and after the first passthrough of the main kernel entry routine
 * it becomes FALSE.
 */
unsigned char diags_do_inits = TRUE;

/* Pointers (passed in by decompressor routine) for the compressed Diags img */
unsigned char *compressed_img_start, *compressed_img_end;

ImpureRgn_t *impure[MAX_CPUS];
ImpureRgn_t *primary_impure;

extern int storage_initialized;
extern ul storage_pool_start;
extern ul storage_pool_end;


/* This routine defines the rules for what action should be taken by the 
 * diags, which has multiple personalities depending on the user's
 * requirements and the state of the machine hardware
 */

static int post_failed = 0;

static unsigned char action( void )
{
    /* If there were any ECC errors during startup, run diags */
    /* If diags was already selected, run the most paranoid option */

    if ( post_failed + sys_crd + sys_mchk + proc_crd + proc_mchk != 0 )
    {
	if ( behav_code == MFRG0 || behav_code == MFRG1 ||
	     behav_code == MFRG2 || behav_code == PARANOIA )
	{ 
	    return behav_code;
	}
	return DIAGSLEVEL1;
    }


    switch( halt_code )
    {
	/* If the machine has come from reset then follow the behavioural
	 * code or the code set by the environment */

	case HLT_K_RESET:
	    if ( behav_code == UNSPECIFIED )
	    {
		if ( env_code != UNSPECIFIED )
		    /* Do what the NVRAM configuration tells us */
		    return env_code;
		else
		    /* go to friendly diags if we have no clue what to do */
		    return DIAGSLEVEL0;
	    }
	    else
	    {
		return behav_code;
	    }
	    break;			/* not reached, defensive coding */


	/* If we've just re-entered via software halt (eg, software crash 
	 * or OS halt), check to see if there was a requested action. */
	/* Note: OS-level reboot requests must get intercepted at an earlier
	 * stage and intercepted there (cold, warm reboots both intercepted) */
	
	case HLT_K_SW_HALT:
	    switch( os_code )
	    {
		case REASON_HALT:
		    TRACE( "Request to stay halted\n" );
		    return DIAGSLEVEL0;

		case REASON_DEFAULT:
		    TRACE("Coming up with default action\n" );
		    return DIAGSLEVEL0;

		default:
		    TRACE( "Don't know how to handle OS request code 0x%X\n",
				os_code );
		    break;
	    }
	    break;


	/* Other halt codes we aren't interested in */
	default:
	    break;
    }


    /* Something bad, eg OS crash */
    return DIAGSLEVEL0;
}



static const unsigned char console[] =
{
	GRP_LOCAL, GRP_SROM, GRP_COM1, GRP_COM2
};

#ifdef WALLACE_GROMIT
static const struct { char note, duration; } wallace[] = 
{
	{ 6, 2 }, { 4, 1 }, { 3, 1 }, { 6, 2 }, { 4, 1 },
	{ 3, 1 }, { 6, 1 }, { 1, 7 }, { 8, 1 }, { 6, 1 },
	{ 8, 1 }, { 9, 2 }, { 8, 1 }, { 6, 1 }, { 4, 1 }, { 3, 4 },
	{ 0, 0 }
};
#endif

static void play_happy_chime( void )
{
#ifdef WALLACE_GROMIT
    unsigned pitch;
    for( i=0; wallace[i].note != 0; i++ )
    {
	switch ( wallace[i].note ) {
	    case 1:	pitch = 466; break;
	    case 3:	pitch = 523; break;
	    case 4:	pitch = 554; break;
	    case 6:	pitch = 622; break;
	    case 8:	pitch = 698; break;
	    case 9:	pitch = 783; break;
	    default:	pitch = 1000;
	}
	pitch = pitch * 7 / 5;
	Beep( wallace[i].duration * 200, pitch );
    }
#else

    Beep( 150, 512 );				/* play the happy chime */
    Beep( 150, 2048 );
    Beep( 150, 8192 );
#endif 
}


/* Arguments are unused and only present for clean compilation... */
static DBM_STATUS do_basic_setup( int argc, char *argv[] )
{
    static BOOLEAN setup_done = FALSE;

    if ( setup_done == TRUE )
		return STATUS_SUCCESS;

    TRACE("Running...\n");

#ifdef LOG_TO_ROM
    setlog( 
#ifdef CONFIG_NAUTILUS
	GRP_SROM | GRP_ROM 
#else
	GRP_ROM
#endif
	 );
#else
    setlog( GRP_SROM );
#endif
    setty( GRP_SROM );
    mobo_logf( "\n" LOG_INFO "Welcome to the diagnostics!\n");
    mobo_logf( LOG_INFO "Reason for diagnostics entry: %s\n", 
		halt_code_string( halt_code ) );

    nb_setup();			/* init stage 1 */
    sb_setup();
    plat_setup();
    mobo_logf( LOG_INFO "All setups done\n");

    nb_fixup();			/* init stage 2 */
    sb_fixup();
    plat_fixup();
    mobo_logf( LOG_INFO "All fixups done\n" );
    
    setup_done = TRUE;
    return STATUS_SUCCESS;
}


/*
 * A generic routine to decide what console to open and attempt interaction
 * with the user.
 *
 * This routine examines user preferences and defaults to attempting a list
 * of likely candidates.
 */
void bring_up_console( void )
{
    String ttypref, logpref;
    char cmdbuf[16];
    DBM_STATUS sval=STATUS_FAILURE;
    int i;

    /* This command will return happy if setup has already been done */
    do_basic_setup( 1, NULL );

    ttypref = nvenv_lookup( KEY_TTY_DEV );
    logpref = nvenv_lookup( KEY_LOG_DEV );

    mobo_logf( LOG_INFO "Preferred console choices: TTY %s, LOG %s\n",
		ttypref == NULL ? "[not set]" : ttypref,
		logpref == NULL ? "[not set]" : logpref );

    /* Firstly redirect the log stream if requested */
    if ( logpref != NULL )
    {
	sprintf_dbm( cmdbuf, "log %s", logpref );
	diags_usercmd( cmdbuf );
    }
    else
    {
	/* By default, unless the user explicitly wants it, log is dropped */
	diags_usercmd( "log none" );
    }


    /* Take a look at the environment to see if the user has a preference */
    if ( ttypref != NULL )
    {
	sprintf_dbm( cmdbuf, "tty %s", ttypref );
	sval = diags_usercmd( cmdbuf );

	/* if it worked, we're done, else tell user the 1st choice is off */
	if ( sval == STATUS_FAILURE )
	    BeepCode( beep_k_noconsole );
    }


    /* In the absence of user preference... */
    /* bring up the first console from our list of favourites */
    if ( sval == STATUS_FAILURE )
    {
	for ( i=0; i<ARRAYLEN( console ); i++ )
	{
	    sval = setty( console[i] );
	    if ( sval == STATUS_SUCCESS )
		break;
	    mobo_logf( LOG_WARN
		"Console choice %d unavailable, trying next\n", i );
	}
    }

    /* Not sure what to do in case of total failure - just beep a lot? */
    if ( sval != STATUS_SUCCESS )
    {
	while( TRUE ) 
	{
	    BeepCode( beep_k_noconsole );
	    sleep( 1 );
	}
    } else {
	mobo_cls();
	mobo_uistate( UI_INTERACT );
	diags_subsys_stat( SYS_CONS, DEV_SETUP );
    }
}




/*--------------------------------------------------------------------*/
/* Main routine - the first C function called in the compressed main kernel */

int dbm_main( 	ImpureRgn_t *sysdata,
		unsigned char *p_decompress_needed,
		unsigned char *p_start, unsigned char *p_end,
		unsigned phys_id )
{
    static smp_mutex online_reg = M_FREE;
    uint64 flags=REASON_DEFAULT;
    DBM_STATUS sval;
    String val;
    int i;
    rom_t *R;
    rom_t *NVRAM;
    int choice;
    unsigned *palbase, pal_major, pal_minor;
    BOOLEAN first_time_through;


    outled( LED_INITMAIN );			/* we made it this far */


#ifdef TRACE_ENABLE
    /* Be careful enabling TRACE messages, it will almost certainly break when
     * SMP is started.  However it can be invaluable for getting early output
     */
    primary_impure = impure[0] = sysdata;
    compressed_img_start = p_start, compressed_img_end = p_end;
    simple_putstr( "Diags main routine running\n\r", DEV_SROM );

    hwrpb_init();
    hwrpb_add( phys_id );
    mem_init();
    setty( GRP_COM1 );
    setlog( GRP_SROM );
    TRACE( "Logging on-line early!\n" );

    hwrpb_dump();
#endif

    TRACE( "Running...\n" );


    /*--------------------------------------------------------------------*/
    /* Essential code, data and state initialisations */
    /* CAREFUL: at this point, we're early, a printf may cause problems */



    /* If the very first pass through, initialise the SMP data structures */
    /* We can guarrantee serialisation here on this variable */
    if ( diags_do_inits )
    {
	/* Keep record of the extent of our compressed image */
	compressed_img_start = p_start, compressed_img_end = p_end;

	primary_impure = sysdata;		/* Data shared with PALcode */

	diags_do_inits = FALSE;		/* Only done by first CPU through */
    }


    TRACE( "CPU %d waiting to register\n", phys_id );

    /* ---- Begin critical region ---- */
    smp_acquire( &online_reg );

    TRACE( "CPU %d registering\n", phys_id );

    impure[ phys_id ] = sysdata;		/* My CPU's impure region */

    /* Setup basic CPU information such as logical ID, register my presence
     * to the SMP interface. */
    first_time_through = smp_register();



    /* As early as possible we make sure there's an HWRPB present.
     * the init calls check to see if the data structures exist before 
     * overwriting any prior state */

    hwrpb_init();
    hwrpb_add( phys_id );
    mem_init();			/* force mem re-init even if HWRPB is intact */

    /* We need to know if we're already present (ie re-entry from OS) */
    /* We can tell this by having a valid perCPU entry already in place 
     * with flags indicating the requested course of action */

    flags = percpu_getflags( phys_id );

    /* Linux only sets reboot actions on the primary processor */
    if ( smp_primary() )
	os_code = PERCPU_REASON( flags );

    TRACE( "ID %d: My flags are 0x%X, OS code is %d\n",
		phys_id, flags, os_code );


    /* Report my circumstances */
    /* Am I entering for the first time or re-entering on a crash (etc)? */
    if ( first_time_through )		/* First time? */
    {
	if( os_code != REASON_DEFAULT )
	{
	    /* If we're here, we just re-entered diags from a booted OS */

	    mobo_logf( LOG_INFO
		"Processor %d re-starting from OS with flags 0x%lx\n",
		phys_id, flags );

	} else {

	    /* If we're here, we just entered diags for the first time */

	    mobo_logf( LOG_INFO
		"CPU %d executing main routine on first pass\n", phys_id );
	}

    } else {

	/* If we're here, we just re-entered diags from within diags (crash) */

	mobo_logf( LOG_CRIT "CPU %d: re-entering diags\n", phys_id );
    }

    smp_release( &online_reg );
    /* ---- End critical region ---- */

    if ( !smp_primary() )		smp_corral();


    /*--------------------------------------------------------------------*/
    /* Once the heap has been setup, the primary processor can log useful
     * information about the hardware configuration, as passed up from the 
     * SROM and PALcode. */
   
    info_submit( SRC_SROM, SUBJ_MEMORY, "Memory size",
		"%d MBytes", primary_impure->MEM_SIZE >> 20 );

    info_submit( SRC_SROM, SUBJ_FIRMWARE, "SROM Firmware Revision",
		"%d.%d (%lX)",
		primary_impure->SROM_REV & 0xFFU,
		(primary_impure->SROM_REV >> 8) & 0xFFU,
		primary_impure->SROM_REV );

    info_submit( SRC_DIAGS, SUBJ_FIRMWARE, "Alpha Diagnostics Revision",
		"%s (%s)", MOBO_VERSION, COMPILE_DATE );

    palbase = (unsigned *)(DBM_SUPER | primary_impure->PAL_BASE);
    pal_major = ( palbase[2] >> 8 ) & 0xFFU;
    pal_minor = palbase[2] & 0xFFU;

    info_submit( SRC_DIAGS, SUBJ_FIRMWARE, "Alpha Diagnostics PALcode Rev",
		"%d.%d (at 0x%X)", pal_major, pal_minor, palbase );

    /* Register processor information */
    cpu_info_submit( phys_id );


    /*--------------------------------------------------------------------*/
    /* This variable is a pointer into the data segment of the decompressor
     * routine.  Setting this variable to TRUE will cause the next CPU to 
     * pass through the compressed Diags entry point to do a Diags kernel
     * unpack.  This variable should be set to TRUE at all points where the
     * main Diags kernel is scrapped, eg on loading Linux.
     */
    diags_do_decompress = p_decompress_needed;

    /* Secondaries re-entering from kernel block on this value
     * very early on, here we release them to enter main diags */
    *p_decompress_needed = FALSE;

    play_happy_chime();		/* We're up & running - let the world know */



    /*--------------------------------------------------------------------*/
    /* We may have just re-entered from an operating system.
     * Are we doing a bootmem thing?  If so, take action here. */

    if ( hwrpb_bootmem_addr != NULL )
    {
	TRACE( "Bootmem found at 0x%lX\n", hwrpb_bootmem_addr );

	rom_ram_scan( hwrpb_bootmem_addr, hwrpb_bootmem_size );
	R = rom_ram_map;
	if ( R != NULL )			/* Do we have something? */
	{
	    TRACE( "Valid ROM-in-RAM image(s) found\n" );

	    do {

		/* Here is a divide in functionality between things that are
		 * Linux related and things that are firmware related.
		 * Linux-related stuff gets kept on one side and firmware merits
		 * immediate loading. */

		TRACE( "Processing BootMem image of ID %d\n", R->fwid );

		switch( R->fwid )
		{
		    case FW_LINUX:		/* Skip these... */
		    case FW_OSFPAL:
		    case FW_INITRD:
			break;

		    default:
			rom_exec( R );		/* Execute anything else */
			break;
		}
		R = R->next;

	    } while( R != NULL );
	}
	else
	{
	    /* If we found nothing valid in the bootmem region, we free it,
	     * which may be more than necessary but is a defensive practise.
	     */
	    page_mark_range( ADDR2PFN( hwrpb_bootmem_addr, DBM_SUPER ),
				PFNCOUNT( hwrpb_bootmem_size ), FREE_PAGE );

	    hwrpb_bootmem_addr = NULL, hwrpb_bootmem_size = 0;
	}
    }


    /*--------------------------------------------------------------------*/
    /* Have we just re-entered from an operating system? */

    /* Note, whatever the halt code was, we're only paying attention to the 
     * primary CPU.  Is this acceptable? */

#define POLL_TIMEOUT 4

    if ( os_code == REASON_COLDBOOT || os_code == REASON_WARMBOOT )
    {
	/* Primary processor waits here until all processors registered in the 
	 * HWRPB have also passed through the init above... */

	/* Re-init of southbridge devices.  This is useful if the kernel has
	 * noodled with such things as system timers etc to get them back into
	 * a known state
	 */
	sb_setup();
	sb_fixup();

	mobo_logf( LOG_INFO "Primary CPU awaiting secondary CPUs\n" );
	for ( i=0; i<POLL_TIMEOUT; i++ )
	{
	    if ( hwrpb->nr_processors == smp_online_count )
		 break;

	    msleep( 1000 );
	    BeepCode( beep_k_reboot_timeout );
	    mobo_logf( LOG_WARN "Wanting %d CPUs, got %d\n",
		hwrpb->nr_processors, smp_online_count );
	}
	
	if ( hwrpb->nr_processors != smp_online_count )
	    mobo_logf( LOG_CRIT "Rebooting with fewer processors!\n" );

	mobo_logf( LOG_INFO
		"%s reboot requested, going back up using previous settings\n",
		 os_code == REASON_COLDBOOT ? "Cold" : "Warm" );


	sval = reboot_linux( );
	if ( sval != STATUS_SUCCESS )
	{
	    mobo_logf( LOG_CRIT "Reboot attempt failed!\n" );
	    BeepCode( beep_k_reboot_failed );
	}
    }


    /*--------------------------------------------------------------------*/
    /* We're not fast-rebooting.  Determine the level of interaction. 
     *
     * 0) Pass-through.  Load an image from ROM and set machine up for it 
     * 1) Manufacturing mode,  paranoia.  Use the SROM UART and 
     *    step on as little of the hardware as possible
     * 2) Friendly mode.  Screen and keyboard, assume things mostly work
     * 3) FSB mode.  AlphaBIOS image corrupt, reflash/analyse firmware
     * 4) Boot manager - set default actions that affect startup mode, etc.
     *
     * The selection process is done according to rules based on the
     * status of platform specific power-on self-test routines (POST),
     * halt code, the behavioural code and NVRAM environment settings
     */

    halt_code = primary_impure->HALT;

    TRACE( "Going to initialisations console\n" );

#ifdef TRACE_ENABLE
    setty( GRP_SROM );
    setlog( GRP_SROM );
#endif

    /* Perform Power-On Self-Test - quick tests that are run every time */
    TRACE( "Running POST\n" );
    if ( plat_post() != STATUS_SUCCESS )	post_failed = 1;
    else					post_failed = 0;


#ifdef CONFIG_SWORDFISH
    /* Look for Swordfish jumper block settings for requested action */
    behav_code = up2k_get_behav_code( );
#else
    if ( primary_impure->BEHAV_DATA != 0 )
	behav_code = primary_impure->BEHAV_DATA & 0x0F;
    else
	behav_code = UNSPECIFIED;
#endif


    TRACE( "Looking for default action\n" );
    StartupMode = action( );


#ifdef CONFIG_TINOSA
    /* Bypass all that stuff for Tinosa which doesn't need anything else yet */
    StartupMode = DIAGSLEVEL0;	/* now go to diags main menu */
#endif

    TRACE( "Decided action was %d\n", StartupMode );
    for ( ;; )				/* infinite event loop */
    {
        switch ( StartupMode )
	{

	default:
	case PASSTHROUGH:		/* straight to BIOS/SRM */

	    /* If we're in a PASSTHROUGH condition, we have no overriding 
	     * reason for doing anything else.  At this point, we can run 
	     * a default command string, as stored in NVRAM variable 
	     * diags_action, if present.  This has the effect of a kind of
	     * software jumper */

	    sval = nvenv_init( );
	    if ( sval != STATUS_FAILURE )
	    {
		TRACE( "Looking up " KEY_DIAGS_ACTION "\n" );
		val = nvenv_lookup( KEY_DIAGS_ACTION );
		if ( val != NULL )
		{
		    /* Check for magic diags instruction */
		    if ( strcmp( val, DA_DIAGS ) == 0 )
		    {
			StartupMode = DIAGSLEVEL0;
			break;
		    }
		
		    TRACE( "Executing requested command '%s'\n", val );
		    diags_usercmd( val );

		    /* What should happen now that the requested command has
		     * completed? */
		}
	    }

	    TRACE( "No requested command found, going to bios exec\n" );
	    diags_usercmd( "bios" );

	    /* if it didn't work we're back here; try bios recover */
	    StartupMode = BIOSRECOVER;
	    break;


        case FACTORY:			/* clear NVRAM settings */

	    bring_up_console();	/* bring screen, keyboard online */

	    mobo_box( r_lrgapp, "Erasure of NVRAM and CMOS" );
	    printf_dbm(
	"You have requested, for example by changing jumper settings,\r"
	"to restore NVRAM and CMOS settings to factory defaults.  This\r"
	"will erase all configuration data stored in this machine.  If\r"
	"you intend to do this, press 'y' now.  Otherwise, press any other\r"
	"key to abort this operation.  Individual settings may be changed by\r"
	"using the 'nvram' command from within Alpha Diagnostics.\r\r"
	"[Erasure will begin automatically in 30 seconds.]" );

	    choice = mobo_key( 30000 );
	    if( choice != -1 && tolower( choice ) != 'y' )
	    {
		/* Abort!  Do something else. */
		StartupMode = DIAGSLEVEL0;
		break;
	    }
 
	    mobo_cls();

	    /* Lookup the the NVRAM region address in the reserved regions */
	    NVRAM = rom_search( FW_NVENV, plat_rom_rsvd );
	    if ( NVRAM == NULL )
	    {
		mobo_box( r_lrgapp, "System does not support NVRAM" );
		printf_dbm( "This system does not appear to support NVRAM!\r"
			"CMOS settings will still be cleared.\r"
			"Press any key to continue." );

		mobo_key( 4000 );
		mobo_cls();
	    }
	    else
	    {
		/* Erase the NVRAM sector. */
		/* FIXME: we're erasing entire sector */
		/* ... Ultimately this might not be what we want */
		printf_dbm( "Erasing system NVRAM variables...\n" );
		plat_romerase( NVRAM->astart, NVRAM->astart + NVRAM->asize - 1,
			NULL, 0 );
	    }

	    /* Zero the CMOS as an encore */
	    printf_dbm( "Erasing system CMOS variables...\n" );
	    for ( i=CM_BYTE_BASE; i<64; i++ )
		cmoswb( i, 0x00 );

	    /* Invite the user to regenerate their NVRAM */
	    diags_usercmd( "nvram" );
	    StartupMode = DIAGSLEVEL0;
	    break;


        case BIOSRECOVER:		/* Fail-safe Boot options */
        case ADERECOVER:

	    bring_up_console();	/* bring screen, keyboard online */

	    diags_usercmd( "flash" );

	    StartupMode = DIAGSLEVEL0;	/* now go to diags main menu */
	    break;


        case DIAGSLEVEL0:		/* Diags level 0 (batch) */
        case DIAGSLEVEL1:		/* Diags level 1 (batch) */
        case DIAGSLEVEL2:		/* Diags level 2 (batch) */
        case DIAGSLEVEL3:		/* Diags level 3 (batch) */

        case DIAGSINT0:			/* Interactive diags level 0 */
        case DIAGSINT1:			/* Interactive diags level 1 */
        case DIAGSINT2:			/* Interactive diags level 2 */
        case DIAGSINT3:			/* Interactive diags level 3 */

        case MFRG0:			/* Manufacturing diags test level 0 */
        case MFRG1:			/* Manufacturing diags test level 1 */
        case MFRG2:			/* Manufacturing diags test level 2 */
     
	    bring_up_console();		/* bring screen, keyboard online */

	    mobo();			/* now lets jump to a higher level */
	    break;			/* should never be reached */


	case BOOTMANAGER:		/* boot time configurator */

	case EMBEDLINUX:
	    bring_up_console();		/* bring screen, keyboard online */


#ifdef CONFIG_SMP
	    diags_usercmd( "smp start" );
#endif
	    diags_usercmd( "linux" );
	    StartupMode = DIAGSLEVEL0;	 /* if it fails, drop back to diags */
	    break;


        case PARANOIA:			/* sick board, assume nothing works */

	    setty( GRP_SROM );
	    setlog( GRP_SROM );
	    mobo_uistate( UI_INTERACT );
	    mobo_logf( LOG_INFO "Diagnostics: main routine started\n" );
	    mobo();			/* go direct to interaction */
	    break;

	}
    }
    return 0;				/* should never reach here */
}



/*--------------------------------------------------------------------*/
/* Main user interface is defined and controlled from here down */


/*----------------------------------------------------------------------*/
/* Private Data */

/* tells functions if the first screen has been put up */
static BOOLEAN mobo_uidrawn = FALSE;

static const Point p_opts = {3, 3}, p_system = {50, 3};


/* Code for commands to setup and re-direct console and log IO */

typedef struct { io_grp G; String S; } gdesc;

static const gdesc gtab[] = {
        { GRP_SROM, "srom" },
        { GRP_COM1, "com1" },
        { GRP_COM2, "com2" },
        { GRP_COM3, "com3" },
        { GRP_COM4, "com4" },
        { GRP_LPT1, "lpt1" },
        { GRP_LPT2, "lpt2" },
        { GRP_LOCAL,"local" },
	{ GRP_ROM, "rom" },
        { GRP_NONE,  "none" }
};
#define NGRPS ( sizeof( gtab ) / sizeof( gdesc ) )


static io_grp str2grp( String S )
{
    unsigned i;
    const gdesc *G;

    for ( i=0, G=gtab; i < NGRPS; i++, G++ )
        if ( strcmp( G->S, S ) == 0 )           return G->G;

    return GRP_ERRGRP;
}

enum streams { CONSOLE, LOG };


static DBM_STATUS ioctl( int stream, int argc, char *argv[] )
{
    int i;
    DBM_STATUS sval = STATUS_FAILURE;
    io_grp g;
    unsigned gmask;

    if ( argc <= 1 )
    {
        mobo_alertf( "Argument needed",
                     "Please specify the new routing" );
        return STATUS_FAILURE;
    }

    /* construct a bitmask of the new routing */
    for ( i=1, gmask=GRP_NONE; i<argc; i++ )
    {
	g = str2grp( argv[i] );
	if ( g == GRP_ERRGRP ) {
	    mobo_alertf( "Argument Error",
			 "Sorry, no IO group option matches %s", argv[i] );
	    return STATUS_FAILURE;
	}

	gmask |= g;
    }

    if ( stream == CONSOLE )    sval = setty( gmask );
    if ( stream == LOG )        sval = setlog( gmask );

    if ( sval != STATUS_SUCCESS )
    {
        mobo_alertf( "Transfer failed",
	     "Diags attempted an IO routing that could not be initialised:\r"
	     "%s choice %s is not available",
	     stream == CONSOLE ? "Console" : "Log",
	     argc == 2 ? argv[1] : "(one of the devices listed)" );
    }

    return sval;
}

static DBM_STATUS ttyctl( int argc, char *argv[] )
{
    return ioctl( CONSOLE, argc, argv );
}

static DBM_STATUS logctl( int argc, char *argv[] )
{
    return ioctl( LOG, argc, argv );
}


/*----------------------------------------------------------------------*/
/* Data structures and definitions regarding the diagnostics command set */

static DBM_STATUS commands( int argc, char *argv[] );

static void usage( const String fmt, ... )
{
    char buf[ 160 ];
    va_list ap;
    va_start(ap, fmt);

    vsprintf_dbm( buf, fmt, ap );
    mobo_alertf( "Usage", buf );

    va_end( ap );
}


/* IO port peek and poke routines */

static DBM_STATUS iord( int argc, char *argv[] )
{
    unsigned portno, rval;

    if ( argc != 2 )
    {
	usage( "%s <port-hex>", argv[0] );
	return STATUS_FAILURE;
    }

    /* in: read a byte from an io port */
    sscanf(argv[1], "%X", &portno);
    rval = inportb( portno );

    mobo_logf( LOG_INFO "IO: port 0x%02X = 0x%02X\n", portno, rval );
    mobo_alertf( "IO port read", "port 0x%02X = 0x%02X\n", portno, rval );

    return STATUS_SUCCESS;
}

static DBM_STATUS iowr( int argc, char *argv[] )
{
    unsigned portno, wval;

    if ( argc != 3 )
    {
	usage( "%s <port-hex> <val-hex>", argv[0] );
	return STATUS_FAILURE;
    }

    /* out: write a byte to an io port */
    sscanf(argv[1], "%X", &portno);
    sscanf(argv[2], "%X", &wval);
    outportb( portno, wval );

    mobo_logf( LOG_INFO "IO: port 0x%02X -> 0x%02X\n", portno, wval );
    mobo_alertf( "IO port write", "port 0x%02X -> 0x%02X\n", portno, wval );

    return STATUS_SUCCESS;
}


/* pci port peek and poke routines */
/* this routine acts as wrapper for 'pciin' and 'pciout' */
static DBM_STATUS pciiord( int argc, char *argv[] )
{
    unsigned bus, dev, fun, reg, rval;

    if ( argc != 5 )
    {
	usage( "%s <bus> <dev> <fun> <reg>\r"
	       "Where <bus>, <dev>, <func> are decimal and <reg> is hex",
		argv[0] );
	return STATUS_FAILURE;
    }

    /* in: read a byte from a PCI CSR */
    bus = atoi( argv[1] );
    dev = atoi( argv[2] );
    fun = atoi( argv[3] );
    sscanf( argv[4], "%x", &reg );

    rval = pcicfgrb(bus,dev,fun,reg);

    mobo_logf( LOG_INFO "PCI: bus %d dev %d fn %d reg 0x%02X = 0x%02X\n",
	    bus, dev, fun, reg, rval );
    mobo_alertf( "PCI CSR read", "bus %d dev %d fn %d reg 0x%02X = 0x%02X\n",
	    bus, dev, fun, reg, rval );

    return STATUS_SUCCESS;
}


static DBM_STATUS pciiowr( int argc, char *argv[] )
{
    unsigned bus, dev, fun, reg, wval;

    if ( argc != 6 )
    {
	usage( "%s <bus> <dev> <fun> <reg> <val>\r"
	       "Where <bus>, <dev>, <func> are decimal and <reg>, <val> hex",
		argv[0] );
	return STATUS_FAILURE;
    }

    /* out: write a byte to PCI CSR */
    bus = atoi( argv[1] );
    dev = atoi( argv[2] );
    fun = atoi( argv[3] );
    sscanf( argv[4], "%x", &reg );
    sscanf( argv[5], "%x", &wval );

    pcicfgwb(bus,dev,fun,reg,wval);

    mobo_logf( LOG_INFO "PCI: bus %d dev %d fn %d reg 0x%02X -> 0x%02X\n",
	    bus, dev, fun, reg, wval );
    mobo_alertf( "PCI CSR write", "bus %d dev %d fn %d reg 0x%02X -> 0x%02X\n",
	    bus, dev, fun, reg, wval );
    pcicfgwb( bus,dev,fun,reg, wval );

    return STATUS_SUCCESS;
}


/* Execute a command in an infinite loop - or until failure */

static DBM_STATUS loop( int argc, char *argv[] )
{
    int iters=0;
    char cmdarg[CBUFSIZ]; 
    DBM_STATUS sval = STATUS_SUCCESS;

    if ( argc < 2 )
    {
	usage( "%s <opt-loopcount> cmd <opt-cmdargs>", argv[0] );
	return STATUS_FAILURE;
    }

    /* there is a slightly tricky parsing and copying process here.  
     * first we must look out for a loop count argument (optional)
     * and then we must copy the body of the argument command (plus
     * any arguments applied to it) into the start of cbuf */
                
    iters = atoi( argv[1] );
    if ( iters == 0 )		/* ie, non-numeric first argument */
	iters = -1;		/* [-1 signifies eternal loop] */

    /* FIXME: THIS IS INEFFICIENT and needs a better approach */
    sprintf_dbm( cmdarg, "%s %s %s %s %s %s %s",
	iters == -1 ? argv[1] : "",
	argc > 2 ? argv[2] : "",
	argc > 3 ? argv[3] : "",
	argc > 4 ? argv[4] : "",
	argc > 5 ? argv[5] : "",
	argc > 6 ? argv[6] : "",
	argc > 7 ? argv[7] : "" );

    mobo_uistate( UI_BATCH );
    do {
	TRACE("Loop-executing '%s'\n", cmdarg );
	mobo_logf( LOG_DBG "Loop-executing '%s'\n", cmdarg );	/* DEBUG */
        sval = diags_usercmd( cmdarg );
        if ( sval != STATUS_SUCCESS )   break;          /* return on fail */
        if ( iters > 0 )                iters--;
    } while ( iters != 0 );

    mobo_uistate( UI_INTERACT );
    return sval;
}


/* Commands accessible by menu choice */

static Cmd_t mobo_tests[] =
{
    {"Run full test set",               "",     go,             SYS_NONE },
    {"Scan PCI buses",                  "",     pcishow,	SYS_NONE },
    {"Test integrated peripherals",     "",     isadevs,        SYS_PCI },
    {"Test interrupt mechanisms",       "",     irq,            SYS_PCI },
#if 0
    {"Stress I/O bus devices",          "",     vgastr, SYS_PCI | SYS_VGA },
#endif
    {"Stress memory",                   "",     mem,            SYS_NONE },
    {"Reset system",                    "",     plat_reset,     SYS_NONE },
    {"Exit Alpha Diagnostics",          "",     go_bios,        SYS_NONE },
};


#if 0			/* FIXME - DON'T USE IDE AFTER ALL */
#ifdef USE_IDE
void ide_test(void);
#endif  /* USE_IDE */
#endif

DBM_STATUS fsb_main( void );			/* DEBUG!*/

/* Commands accessible by typing in a command word */

static Cmd_t mobo_cmds[] =
{    /* long name */            /* command */   /* function */  /* requires */
    {"Full diagnostic test run",				"go",           
			go,		SYS_CONS | SYS_PCI | SYS_ISA },
    {"View hardware configuration details",			"info",
			info_dump,	SYS_CONS },
    {"Change console device",					"tty",          
			ttyctl,		SYS_NONE },
    {"Enable or direct log stream output",			"log",          
			logctl,		SYS_NONE },
    {"Read diagnostic log record saved in ROM",			"logrd",
			logread,	SYS_CONS },
    {"IO port read <hex-port>",					"in",           
			iord,		SYS_NONE },
    {"IO port write <hex-port> <hex-val>",			"out",          
			iowr,		SYS_NONE },
    {"Read/set/test the CMOS/Real-time clock",			"cmos",
			cmos,		SYS_CONS | SYS_ISA },
    {"PCI CSR get <dec-bus> <dec-dev> <dec-fun> <hex-port>",	"pciin",
			pciiord,	SYS_NONE },
    {"PCI CSR put <dec-bus> <dec-dev> <dec-fun> <hex-port> <hex-val>", 
								"pciout",       
			pciiowr,	SYS_NONE },
    {"PCI bus configuration",    				"pci",
			pcishow,	SYS_CONS | SYS_PCI },
    {"PCI device configuration space <dec-bus> <dec-dev> <dec-fun>",
								"pcihdr",
			pcihdr,		SYS_CONS | SYS_PCI },
    {"Integrated system devices test",  			"isa",
			isadevs,	SYS_CONS | SYS_ISA },
    {"Set/Get Interrupt Priority Level (IPL)",			"ipl",
			ipl,            SYS_ISA },
    {"Interrupt test",						"irq",
			irq,            SYS_CONS | SYS_ISA },
    {"Memory stress test [start-mb] [end-mb]",			"mem",
		 	mem,            SYS_CONS },
    {"Initialise VGA",          				"vga",
			vga,            SYS_PCI },
    {"Probe I2C bus", 		 				"i2c",
			i2c,            SYS_CONS | SYS_I2C },
#ifdef NOT_YET_IMPLEMENTED
    {"VGA Stress test",						"vgastr", 
			vgastr,		SYS_PCI | SYS_VGA },
    {"VGA & Memory stress",					"memiostr",
			memiostr,	SYS_PCI | SYS_VGA },
#endif

    {"Floppy read/write test",					"floppy",
			floppy,		SYS_CONS | SYS_ISA | SYS_PCI },
    {"Load Linux via ROM, floppy or serial cable", 		"linux",
			go_linux,       SYS_PCI },
    {"SMP tests",               				"smp",
			smp,            SYS_NONE },
    {"Flash firmware test and upgrade",				"flash",
			reflash,        SYS_CONS },
    {"Read/Set system NVRAM settings",				"nvram",
			nvram,		SYS_CONS },
    {"Read NVRAM Environment",					"env",
			envread,	SYS_CONS },
    {"Write NVRAM Environment [key] [value]",			"set",
			envwrite,	SYS_NONE },
    {"Loop a test",             				"loop",
			loop,           SYS_NONE },
    {"Start next level firmware (AlphaBIOS or SRM console)",	"bios",
			go_bios,        SYS_NONE },
#ifdef USE_DLD
    {"Download test applet",					"dld",
			dld,            SYS_NONE },
#endif
#if 0 		/* FIXME BODGE - DON'T USE IDE AFTER ALL */
#ifdef USE_IDE
    {"IDE read/write test",					"ide",
			ide_test, 	SYS_PCI | SYS_ISA },
#endif /* USE_IDE */
#endif		/* 0 */
#ifdef USE_TEST
    {"Development/Test vehicle",				"test",
			test,		SYS_NONE },
#endif
    {"EEPROM / Asset manager",					"asset",
			asset,          SYS_CONS | SYS_I2C  },
    {"Asset EEPROM integrity test",				"eeprom",
			eeprom_integrity,          SYS_CONS | SYS_I2C  },
    {"",							"setup",
			do_basic_setup, SYS_NONE },
    {"This help screen",					"help",
			commands,       SYS_NONE },
};


/* produces a list of commands, this is the response to 'help' */
static DBM_STATUS commands( int argc, char *argv[] )
{
    int i;
    Cmd_t *C;

    mobo_cls();
    printf_dbm("Available commands:\r\r");
    for ( i=0, C=mobo_cmds; i < ARRAYLEN(mobo_cmds); i++, C++ )
    {
        printf_dbm("  %-10s %s\r", C->name, C->desc);

        if ( i % (NROW - 4) == 0 && i != 0 ) {
            printf_dbm("\rPress any key to continue...");
            mobo_key( 0 );
            mobo_cls();                                 /* also homes cursor */
        }
    }

    printf_dbm("\nAny key to return");
    mobo_key( 0 );
    return STATUS_SUCCESS;
}


/* Execute a command as though typed by the user: a good implementation
 * independent way of running high-level operations without needing to know
 * the name of the function that implements it this week...
 */

DBM_STATUS diags_usercmd( const String C )
{
    const Cmd_t *p;

    strcpy( cbuf, C );                  /* as though the user typed it */
    p = mobo_cmdlookup( cbuf, mobo_cmds, ARRAYLEN(mobo_cmds) );
    if ( p==NULL )      return STATUS_FAILURE;          /* no match */

    if ( p->action == NULL)             /* a quit command? */
	return STATUS_FAILURE;

    /* Otherwise pass our command data and arguments on for execution */
    return mobo_runcmd( p, C );
}



/*--------------------------------------------------------------------*/

/* Subsystem status record keeping - controls what is printed in the
 * status column on the right of the main screen */

unsigned diags_environment = SYS_NONE;

typedef struct {
    enum systems sys;
    String name;
    enum dev_state state;
} Subsystem_t;

static Subsystem_t mobo_subs[] =
{
    { SYS_MEM,	"Memory......", DEV_SETUP   }, 	/* diags depends on this */
    { SYS_PCI,	"PCI I/O.....", DEV_UNKNOWN },
    { SYS_ISA,	"ISA I/O.....", DEV_UNKNOWN },
    { SYS_VGA,	"Video.......", DEV_UNKNOWN },
    { SYS_IRQ,	"Interrupts..", DEV_UNKNOWN },
    { SYS_SMP,	"SMP.........",
#ifdef CONFIG_SMP
	DEV_UNKNOWN 
#else
	DEV_NOTCONFIG
#endif
	},
    { SYS_I2C,	"I2C.........",	DEV_UNKNOWN },
    { SYS_CONS,	NULL,		DEV_UNKNOWN },
};


typedef struct {
        enum dev_state S;
        const String str;
} dev_st_t;

static const String state2str( enum dev_state S )
{
    static const dev_st_t dev_state_string[] = {
	{ DEV_UNKNOWN,	"Unknown" },
	{ DEV_SETUP,	"Set up" },
	{ DEV_PROBING,	"Probing" },
	{ DEV_PROBED,	"Probed OK" },
	{ DEV_STRESSING,"Stressing" },
	{ DEV_PASSED,	"Passed" },
	{ DEV_NOTCONFIG,"Not Applicable" },
	{ DEV_FAILED,	"FAILED" }
    };
    int i;

    for ( i=0; i<ARRAYLEN( dev_state_string ); i++ )
	if ( dev_state_string[i].S == S )
	    return( dev_state_string[i].str );

    /* Something very odd has happened if we got here... */
    BUGCHECK( TRUE );
    return NULL;
}


void diags_subsys_stat(enum systems sy, enum dev_state st)
{
    Point last = cursor;        /* last cursor position */
    Point p = p_system;
    int i;
    String str = NULL;

    /* setup the internal environment mask.  Here we guess that a device must
     * must be setup or stressing to be considered a part of the environment.
     * anything that is failed is removed from the environment */

    if ( (st == DEV_SETUP) || (st == DEV_PASSED) )	/* pass case - add */
        diags_environment |= sy;
    if ( st == DEV_FAILED )				/* fail case - remove */
        diags_environment &= ~sy;



    /* Look for the device in question and update its status */
    for ( i=0; i<ARRAYLEN(mobo_subs); i++ )
    {
	if( mobo_subs[i].sys == sy )
	{
	    mobo_subs[i].state = st;
	    str = state2str( st );
	    break;
	}
    }

    /* If we fail here, something odd has happened: an internal error */
    BUGCHECK( i==ARRAYLEN( mobo_subs ) );


    /* Display update: */
    /* We may be called before the interface, in which case keep quiet */
    if ( !mobo_uidrawn )                return;

    /* Find the right point in the state area for updating and do the update */
    p.y += i+1;			/* +1 to allow for the title */
    mobo_goto(p);
    printf_dbm("%s%-10s", mobo_subs[i].name, str );
    mobo_logf( LOG_INFO "--------%s: %s--------\n",
	  mobo_subs[i].name, str );

    /* Return the cursor to where it started */
    mobo_goto(last);
}


/*--------------------------------------------------------------------*/
/* Internal routine to redraw the main screen from first principles */
static void draw_main(void)
{
    Cmd_t *tst;
    int i;

    /* Our version banner */
    mobo_box(r_scrn, "Alpha Diagnostics " MOBO_VERSION " (" COMPILE_DATE ")" ); 

    /* print the options available */
    mobo_goto(p_opts);
    printf_dbm("TEST OPTIONS\r");

    for (tst = mobo_tests, i = 1; i <= ARRAYLEN( mobo_tests ); tst++, i++)
        printf_dbm("%d) %s\r", i, tst->desc);

    /* print the subsystem status */
    mobo_goto(p_system);
    printf_dbm("SUBSYSTEM STATUS\r");
    for ( i=0; i<ARRAYLEN(mobo_subs); i++)
    {
	if ( mobo_subs[i].name != NULL )
	{
	    printf_dbm("%s%s\r",
		mobo_subs[i].name, state2str( mobo_subs[i].state ) );
	}
    }
}



/* Main event loop for the Manufacturing diagnostic code */
static void mobo(void)
{
    uint64 flags;

    draw_main();
    mobo_uidrawn = TRUE;


    /* check the halt code before we go interactive */
    /* check we're not arriving via OS-directed halt or unexpected event */
    flags = percpu_getflags( smp_phys_id() );
    if ( PERCPU_REASON( flags ) != REASON_HALT && halt_code != HLT_K_RESET )
    {
        mobo_alertf( "An Unexpected Event Occurred",
                     "Diags re-entered because: %s (PC=0x%lX)\r"
                     "If you did not expect this, try a full diagnostic test",
                     halt_code_string( halt_code ), primary_impure->EXC_ADDR ); 
    }
    
    while ( mobo_menu( mobo_tests, ARRAYLEN( mobo_tests ),
			mobo_cmds, ARRAYLEN(mobo_cmds) ) == 0)
    {
        draw_main();
    }
    
    mobo_cls();
}

