	cpu Z8601
	page 0
	include stddefZ8.inc

;********************************************************************
; 
;              Apple 10MB ProFile - Z8 controller firmware
;                         Version 4.04
;            original code (c) by Apple Computer Inc., 1981
;
;           disassembly and comments by Patrick Schfer, 2013
;                      last changes 2014-02-04 PS
;
;********************************************************************


ROMsize		SET 4096
DontListIncls	SET 1				; set to skip listing for .inc
	
VersionHi	EQU 004h
VersionLo	EQU 004h



	include DefsProFile10RW404.inc

	org 00000h
; step motor: 2 bipolar phases connected to P0.4..P0.7
; it takes two steps from one track to the next one
StepVal		DB 050h,040h,060h,020h		; sw, w, nw, n directions
		DB 0A0h,080h,090h,010h		; ne, e, se, s directions

		DW HostCmdRead
		DW HostCmdRead
;
Start		jp ColdStart
;
; Step motor speed profiles
AccelTab	DB 006h, 08Bh, 016h, 012h	; prescaler values
		DB 00Eh, 00Ah, 00Eh, 00Ah
		DB 00Eh, 00Ah, 04Eh, 04Eh
		DB 016h, 00Eh, 00Ah, 016h
		DB 012h, 012h, 01Ah, 00Bh
		DB 086h, 00Bh, 086h, 001h
		DB 00Eh, 08Fh, 066h, 066h
 		DB 01Ah, 00Bh, 086h, 001h

 		DB 090h, 060h, 060h, 0A0h	; delay after two steps
		DB 090h, 0C0h, 070h, 080h
		DB 090h, 080h, 070h, 050h
		DB 060h, 060h, 0C0h, 060h
		DB 0C0h, 0d0h, 0b0h, 060h
		DB 050h, 080h, 050h, 000h
		DB 050h, 050h, 070h, 070h
		DB 0C0h, 0b0h

		DB 0C0h, 012h, 01Fh, 038h	; delay between two steps 
		DB 014h, 02Ch, 019h, 00Ch
		DB 01Dh, 02Ch, 01Dh, 00Ch
		DB 00Ch, 038h, 01Eh, 019h
		DB 038h, 013h, 00Dh, 025h
		DB 01Fh, 015h, 01Dh, 015h
		DB 000h, 00Ch, 019h, 01Dh
		DB 01Dh, 045h, 01Ah, 007h

_006d  		DB 001h, 002h, 004h, 008h
		DB 010h, 020h, 040h, 080h
;
;********************************************************************
;  This is the Reset entry point
;********************************************************************
;
ColdStart	clr IMR				; disable all interrupts
		ei     
		di     
		ld P01M,#016h			; P1,P0L = A8..11, P0H = output
		ld SPL,#Stack_Top		; stack starts at 080h  
		srp #WorkingRegs		; select register bank 5
	assume RP:WorkingRegs
		ld R0,#0FFh
 		ld P2M,#007h			; P2.7..P2.3 output, P2.2..P2.0 input
		ld P2,R0			; set all bits of P2 high
		ld R1,#10
ClrLoop1	ld ActCylH-1(R1),R0		; set 01A..023h to 0FFh
		djnz R1,ClrLoop1
		ld R2,#18
ClrLoop2	ld SkipSpared-1(R2),R1		; set 008..019h to 000h
		djnz R2,ClrLoop2
		ld ActCylH,#309/256		; far away from cylinder 0
		ld ActCylL,#309#256
 		ld P3M,#001h			; port 2 push-pull, P3H output, P3L input
		ld P3,#WRTSM			; set WRTSM, clear HS0, HS1, RDHDR
		clr P0				; turn off stepper
; initialize i8253 counter (one shot mode: a L->H transition at gate starts the pulse)
		ld R1,#003h			; control register at 0FF03h
		ld R3,#032h			; counter 0 mode 1 (one shot 16 bit)
		ld R4,#072h			; counter 1 mode 1 (one shot 16 bit)
		ld R5,#092h			; counter 2 mode 1 (one shot 8 bit)
		ld P01M,#036h			; set slow memory timing
		and P2,#0FFh-Msel1  		; select Z8 --> Timer 
		lde @RR0,R3
		lde @RR0,R4
		lde @RR0,R5
		call RAMtoZ8a			; Z8 --> Mem
		dec PwrFlg1			; check for cold start
		jr nz,SpinUpDly			; PwrFlg1 <> 1   -> power on!
		inc PwrFlg0			; PwrFlg0 <> 0FFh -> power on!
		jr nz,SpinUpDly
		or Status3,#Power_Reset
		call CheckFence
		jp z,WarmStart			; ***
		call GetSpareTable
		jp WarmStart
; wait 18 s for the spindle motor
SpinUpDly	ld R0,#105
SpinUpLoop	call WaitR12ms
		djnz R0,SpinUpLoop
		or Status2,#020h		; ***
; seek to spare table track and read spare table
		call GetSpareTable
		tm Status2,#Stat_SprTbl_Err	; fault?
		jp nz,WarmStart1a		;  yes --> abort selftest
;
; read all pending bad blocks and spare them if required
Scrub		ld SpareTblDirty,#0FFh		; set spare table invalid
		ld SparingTh,#30		; sparing threshold 30
		clr ReadWithHdr
		clr LBAhi
		ld LBAmi,#04Bh
ScrubLp2	ld R1,LBAmi
		rl R1
		add R1,#SpareMap#256
		ld R0,#SpareMap/256		; point to sublist address
		lde R2,@RR0			; get MSB
		cp R2,#0FFh			; NIL?
		jr z,ScrubNxt1			;  yes --> next sublist
		incw RR0			
		lde R3,@RR0			; get LSB, now RR2 points to sublist
		lde R4,@RR2			; get sublist length
		ld ScrubCtr1,R4			;  into ScrubCtr1
ScrubLp1	incw RR2
		lde R4,@RR2			; get block number from sublist		
		incw RR2
		ld LBAlo,R4			; into LBAlo
		lde R4,@RR2			; get type byte
		and R4,#080h			; spare?
		jr z,ScrubNxt2			;  yes -->
; so this spare table entry is a bad block
		push R2				; save sublist pointer
		push R3
		call LBAtoCHS
		ld RetryCnt,#100
		call ReadBlockCHS		; scrub bad block
		pop R3
		pop R2
ScrubNxt2	dec ScrubCtr1
		jp nz,ScrubLp1			; and process next sublist
ScrubNxt1	dec LBAmi
		jp pl,ScrubLp2
;
; read each sector (including CRC check and sparing if required)
; reading is done in physical order, i.e. deinterleavind is done.
Scan		clr LBAmi
		clr LBAlo			; start at block 0
		ld ScanCtr1,#8			; 8 + 8 sectors per track
		ld ScanCtr2,#64			; 4x 8 sectors per cylinder
		ld SkipSpared,#0FFh		; do not scan spared blocks
ScanLp		call LBAtoCHS			; calculate C/H/S
		push ActSector
		ld RetryCnt,#100
		call ReadBlockCHS		; check this block
		pop ActSector
		and LBAlo,#0F0h
		add ActSector,#10		; 0, A, 4, E, 8, 2, C, 6
		dec ScanCtr1			; next sector
		jr nz,Scan1
		clr ReadWithHdr			; now let state machine check headers
		sub ActSector,#3		; 3, D, 7, 1, B, 5, F, 9
		ld ScanCtr1,#8
Scan1		and ActSector,#15
		or LBAlo,ActSector		; merge in new sector number
		dec ScanCtr2			; all done?
		jr nz,Scan2			;  no -->
		and LBAlo,#0C0h			; next cylinder
		add LBAlo,#64
		adc LBAmi,#0
		ld ScanCtr2,#64
		cp LBAmi,#04Ch			; all done?
		jp z,WarmStart			;  yes -->
		jr Scan3
Scan2		tm ScanCtr2,#15			; track finished?
		jr nz,Scan3			;  no -->
		clr ReadWithHdr			
		add LBAlo,#16			; next track
		and LBAlo,#0F0h
		add ActSector,#13
		and ActSector,#15
		or LBAlo,ActSector
Scan3		jp ScanLp
;
;********************************************************************
; and here we go...
;********************************************************************
;
WarmStart	ld PwrFlg0,#0FFh		; set warm start flags
		ld PwrFlg1,#001h
WarmStart1a	and P2,#0FFh-BSY   		; set BSY
		clr ReadWithHdr			; state machine checks headers
		clr SkipSpared
WarmStart1	ld SPL,#Stack_Top		; dump stack
WarmStart2	clr BlockIsBad
		ld R14,#1			; 01 means waiting for command
		call DoHandshake		; wait for host to assert /CMD
; 
; *** we end up here after successful 01 - 55 handshake ***
		tm Status2,#020h		; ***
		jr z,WarmStart3			; ***
WarmStart2a	clr SpareTblDirty		; ***
		or Status1,#Stat_SpTblUpd	; host data lost
		call GetSpareTable
WarmStart3	rl SpareTblDirty		; does our spare table need to be updated?
		jr z,GetHostCmd			;  no --> skip
; update spare table
		or Status1,#Stat_SpTblUpd	; host data lost
		or P2,#DRW_Read
		or P2,#Msel0+Msel1		; Z8 --> Mem
		ld P01M,#016h			; reconnect bus
		call ReadSpareTrk
		ld SpareTblDirty,#0FFh		; now table *really* needs to be updated...
		jp nz,GetHostCmd		; ***
		call MoveBlock			; move data from ReadBuffer into WriteBuffer
		ld R2,#(WriteBuffer+ST_FwRev)/256	; update firmware revision
		ld R3,#(WriteBuffer+ST_FwRev)#256	
		ld R5,#VersionHi
		lde @RR2,R5			; (1038h):= FW version MSB
		inc R3
		ld R5,#VersionLo
		lde @RR2,R5			; (1039h):= FW version LSB
		ld R3,#(WriteBuffer+ST_SparesAlloc)#256	; update bad block & spare count
		ld R5,#SpareBlocks
		ldei @RR2,@R5			; (1040h):=(3Ah) number of spares 
		ldei @RR2,@R5			; (1041h):=(3Bh) number of bad blocks
		ld R0,#SpareMap/256
		ld R1,#SpareMap#256
		ld R4,#1
		ld R5,#0F0h			;  copy 01F0h bytes
CopySpareMap	lde R6,@RR0
		lde @RR2,R6			; copy local spare map into spare table
		incw RR0
		incw RR2
		decw RR4
		jp nz,CopySpareMap
		call CheckFence
		jp z,_020d			; ***
		or Status3,#010h		; ***
		jp WarmStart2a			; ***
_020d		call WriteSpareTrk		; ***
		tm Status1,#002h		; ***
		jr nz,GetHostCmd		;   --> yes, failed!
		clr SpareTblDirty		;  otherwise set spare table valid
;
; get command bytes from host
GetHostCmd	ld R0,#HostCmdBuffer/256
		ld R1,#HostCmdBuffer#256
		and P2,#0FFh-Msel0-Msel1	; Apple --> Mem
		lde @RR0,R2			; dummy write to set RAM address to R0.R1
		ld P01M,#01Ch			; release data & address bus
		and P2,#0FFh-DRW_Read-Msel0-Msel1-BSY	; Apple --> Mem & set BSY
		call WaitCmdLine		; wait for host to dma in command bytes and assert /CMD
		or P2,#DRW_Read
		or P2,#Msel0+Msel1		; Z8 --> Mem
		ld P01M,#016h			; reconnect bus
		ld R0,#HostCmdBuffer/256
		ld R1,#HostCmdBuffer#256
		ld R15,#HostCommand		; copy them into 00Eh..013h
		ld R14,#6
CopyHostCmdLp	ldei @R15,@RR0		
		djnz R14,CopyHostCmdLp
		cp HostCommand,#3		; invalid command (>03h)?
		jp ugt,WarmStart2		;  yes --> abort
; process host command
		ld R14,HostCommand
		inc R14
		inc R14				; generate response byte
		call SetResponse		; and send it to host
		and Status1,#Stat_SpTblUpd+Stat_Seek_Err+002h
		ld R5,#070h			; ***
		tm Status2,#020h		; ***
		jr z,_0257			; ***
		ld R5,#078h			; ***
_0257		and Status2,R5
		clr Status4			; zero number of errors
		rl HostCommand
		jp z,HostCmdRead		; 00 --> Read Block
; 
;********************************************************************
; *** Host Command 01/02/03: Write / WriteVerify / Force Sparing
;
; Parameter:	LBAhi/LBAmi/LBAlo/RetryCnt/SparingTh
;		RAM: user data
; Returns:	Status
;********************************************************************
;	
HostCmdWrite	or Status1,#Op_Failed		; ***
		ld 04Dh,#0FFh			; ***
		call LBAtoCHS			; ***
		jp nz,HostCmdWrite1		; ***
		call Seek			; ***
		clr 04Dh			; ***
		and Status1,#0FFh-Op_Failed	; ***
HostCmdWrite1	ld R0,#WriteBuffer/256
		ld R1,#WriteBuffer#256
		and P2,#0FFh-Msel0-Msel1	; Apple --> Mem
		lde @RR0,R2			; dummy write to set RAM address to R0.R1
		ld P01M,#01Ch			; release data & address bus
		and P2,#0FFh-DRW_Read-BSY	; clear P2.7, set BSY
		ld R14,#6			; 06 means write confirmation
		call DoHandshake		; wait for host to assert /CMD
; 
; *** now the host DMAs his data into the write buffer... ***
; we end up here after successful 01 - 55 handshake
		ld StatusBufLSB,#(WriteBuffer-4)#256
		and Status1,#Stat_Seek_Err	; clear everything but seek error
		rl 04Dh				; ***
		jr nz,_0296			; ***
		call WriteBlockCHS
_0296		jp SendState
; 
;********************************************************************
; *** Host Command 00: Read
;
; Parameter:	LBAhi/LBAmi/LBAlo/RetryCnt/SparingTh
; Returns:	Status
;		RAM: user data
;********************************************************************
;
HostCmdRead	call LBAtoCHS
		jr nz,_02a6			; spare table? -->
		jp c,SendState			; abort if illegal block
		call ReadBlockCHS
		jr SendState
_02a6		jp nc,_02d0
;
; copy status bytes into RAM buffer at 01000h+(02Ch) and prepare for DMA 
;
SendState 	ld R0,#RAM/256
		ld R1,StatusBufLSB
		ld R2,#Status1
		ld R3,#4			; lenght 4 bytes
		tm Status2,#020h		; ***
		jp z,SendSt_Lp			; ***
_02b7		or Status1,#Op_Failed		; ***
SendSt_Lp	ldei @RR0,@R2			; copy status bytes into buffer
		djnz R3,SendSt_Lp
		ld R1,StatusBufLSB
		and P2,#0FFh-Msel0-Msel1	; Apple --> Mem
		lde @RR0,R2			; dummy write to set RAM address to R0.R1
		ld P01M,#01Ch			; release data & address bus
		and P2,#0FFh-BSY		; set BSY
; *** now the host may pick up his status bytes ***
		clr Status3
		jp WarmStart1
;
_02d0		tm Status2,#020h		; ***
		jr z,ReadSpareTbl		; ***
		call GetSpareTable		; ***
		jp nz,_02b7			; ***
		jr _02e4			; ***
;
;********************************************************************
; read spare table for host
;
; The data you see after reading block FFFFFF is not what is actually
; stored on disk. Both spares and bad block lists are generated on
; the fly before the ReadBuffer is passed over to the host.
;
; The spare map is a list with 026h two-byte entries, one for each 
; LBAmi value. Each entry either contains FFFF (NIL) or points to
; a sublist in memory. The first byte of this sublist holds the
; number of two-byte entries in this list. Each entry consists of
; the LBAlo value of the questionable block, followed by a type
; byte. Bit 7 =1 indicates a spare, bit 6 =1 a bad block.
;********************************************************************
;
ReadSpareTbl	push SpareTblDirty		; ***
		call ReadSpareTrk		; ***
		pop SpareTblDirty		; ***
_02e4		and Status1,#0FFh-Stat_SpTblUpd	; clear host data lost flag
		ld R1,#2			; we have to build two lists
		ld R8,#080h			; let's start with the spares
		ld R6,#(ReadBuffer+ST_SpareMap)/256	; here we store the lists
		ld R7,#(ReadBuffer+ST_SpareMap)#256
		ld R2,#SpareMap/256
		ld R12,#0			; LBAhi is always zero
		ld R11,#0FFh			; 0FFh means NIL
; build one list for the spared and one for the bad blocks
RdST_List	ld R3,#SpareMap#256		; and this is where the data comes from
		ld R13,#0			; LBAmi starts at zero
		ld R0,#04Dh			;  and goes up to 04Ch
; for each LBAmi process one sublist
RdST_Lp2	lde R4,@RR2			; get sublist pointer MSB from table
		inc R3
		lde R5,@RR2			; get sublist pointer LSB from table
		inc R3
		cp R4,R11			; first byte = NIL?
		jr z,RdST_Lp2sk			;  yes --> no sublist present
		lde R9,@RR4			;  else get number of entries
; check each sublist entry for the wanted type
RdST_Lp1	incw RR4
		lde R14,@RR4			; get LBAlo value
		incw RR4
		lde R15,@RR4			; get type byte
		and R15,R8			; is it what we want?
		jr z,RdST_Lp1sk			;  no --> skip it
		ld R10,#WorkingRegs+12		; = R12 address
		ldei @RR6,@R10			; store LBAhi = 0
		ldei @RR6,@R10			; store LBAmi = sublist number
		ldei @RR6,@R10			; store LBAlo = value from sublist
RdST_Lp1sk	djnz R9,RdST_Lp1
RdST_Lp2sk	inc R13				; now process next sublist
		djnz R0,RdST_Lp2
; write 3x 0FFh as delimiter
		ld R5,#3
RdST_Lp3	lde @RR6,R11			; R11 holds 0FFh
		incw RR6
		djnz R5,RdST_Lp3
; and repeat the above for the bad block list
		ld R8,#040h			; bad block marker
		djnz R1,RdST_List		; repeat for bad blocks
		jp SendState
;
;********************************************************************
; translate LBA into CHS data
;
; LBA has the format 00000000.00cccccc.cchhssss
; input:  LBAhi/LBAmi/LBAlo
; output: CylinderH/CylinderL/Head/ActSector , R4=Cylinder+1
;         Zero Flag is cleared when LBA is FFFFFFh
;	  Carry Flag is set when illegal LBA
;********************************************************************
;
LBAtoCHS	ld BlockStat,#040h
		inc LBAhi
		jp z,LtoC_FF			; LBA starts with FFh -> this RAM or spare table
		dec LBAhi
		jp nz,LtoC_failed		; LBA > FFFFh -> out of range
		cp LBAmi,#04Dh
		jp nc,LtoC_failed		; LBA >= 4Dxxh -> out of range
		cp LBAmi,#04Ch
		jr nz,LBAtoCHS1
		cp LBAlo,#040h
		jp nc,LtoC_failed		; LBA >= 4C3Fh -> out of range
LBAtoCHS1	ld R4,LBAmi
		ld R5,LBAlo
		ld ActSector,R5
		rl R5
		rlc R4
		rl R5
		rlc R4				; shift LBA 2 steps left to grab all cylinder bits
		ld CylinderL,R4
		clr CylinderH
		jr nc,LBAtoCHS2			; <4000h -->
		inc CylinderH
		inc CylinderL
		jp LtoC_done
LBAtoCHS2	cp CylinderL,#153
		jp c,LtoC_done
		incw CylinderH			; +1 for cylinders above 153 to skip the spares
		jp LtoC_done
; get CHS on spares track 153
LtoC_SpareTrk	ld CylinderL,#153
		clr CylinderH
LtoC_done	ld Head,ActSector
		swap Head
		and Head,#003h			; -> head number 0..3
		and ActSector,#00Fh		; -> sector number 0..15
		ld FLAGS,#040h			; set zero flag
		ret    
;
; get CHS for (R15)th spare block
;
LtoC_Spared	ld ActSector,R15
		jp LtoC_SpareTrk
;
; handle requests for RAM buffer and spare table
LtoC_FF		incw LBA			; spare table (0FFFFh)?
		jr nz,_039a			;  no -->
		cp HostCommand,#0		; anything but read?
		jp nz,LtoC_failed			;  yes -->
		ld FLAGS,#000h			; clear Z and C
		ret    
;
_039a		incw LBA			; last block (0FFFEh)?
		jr nz,LtoC_failed			;  no -->
		ld FLAGS,#0C0h			; set Z and C
		jr DoHandshake
;
LtoC_failed	or Status3,#Illegal_Block
		or Status1,#Op_Failed 
		ld FLAGS,#080h			; set C
		ret 
;
; ***** do handshake with host *****
; profile waits for CMD to go active, then puts (R14) onto data bus
; after CMD went inactive again, 55h is expected as ACK
;	
DoHandshake	call WaitCmdLine		; wait for CMD
SetResponse	or P2,#Msel0+Msel1    		; Z8 --> Mem
		ld P01M,#006h	 		; set P1 as output
		ld P1,R14			; and send byte to host
		or P2,#BSY	   		; clear BSY
DoHsk_Wait	tm P2,#CMD
		jp nz,DoHsk_Wait		; wait until CMD line goes low
		ld P01M,#00Eh			; set P1 as input
		and P2,#0FFh-Msel0-Msel1  	; Apple --> Mem
		and P2,#0FFh-Msel0-Msel1-DRW_Read
		ld R0,P1			; get answer from host
		or P2,#DRW_Read
		or P2,#Msel0+Msel1		; Z8 --> Mem
		ld P01M,#016h			; reconnect bus
		cp R0,#055h			; ACK?
		jp z,Ret_2			; ok, finished
		or Status1,#Bad_55		; record NAK in status byte
		jp SendState
;
; ***** wait for host /CMD *****
; this routine waits until CMD (P2.2) is pulled high by the host
; if nothing happens, the heads are moved off the data area of the disk
;
WaitCmdLine	ld P01M,#01Ch			; tristate bus lines
		ld T0,#0FFh
		clr PRE0			; count down from 16384
		ld R5,#0FFh-128			; for 128 times
		jp WaitCmdLine1
WaitCmdLineLp	cp R14,#6			; no timeout between 2nd and 3rd handshake
		jp z,WaitCmdLine2
		tm T0,#0FFh			; poll CMD until timer finished
		jp nz,WaitCmdLine2
WaitCmdLine1	ld TMR,#03h			; load and start T0
		inc R5				; did timer run off 128 times?
		jr mi,WaitCmdLine2		;  no, check for CMD
		cp CylinderH,#308/256
		jp nz,WaitCmdLine3
		cp CylinderL,#308#256
		jr z,WaitCmdLine2		; skip positioning if already off data area
WaitCmdLine3	ld CylinderH,#308/256
		ld CylinderL,#308#256
		push 03Bh			; ***
		ld 03Bh,#0FFh			; ***
		call Seek			; ***
		pop 03Bh			; ***
		clr P0				;  turn off stepper
		dec StpMotOff			;  and set flag
WaitCmdLine2	tm P2,#CMD			; did host pull CMD high?
		jp z,WaitCmdLineLp
Ret_2		ret    
;
;********************************************************************
; read current Cylinder/Head/Sector into ReadBuffer
;********************************************************************
;
ReadBlockCHS	and Status1,#Stat_Seek_Err+002h	; clear all but 'Seek Error'
		call Seek			; seek to the desired track
		and BlockIsSpared,SkipSpared	; ignore spared blocks?
		jp nz,Ret_2			;  yes --> abort
		tm Status2,#020h+Stat_SprTbl_Err	; spare table bad?
		jp nz,Ret_2			;  yes --> abort
RdBkCHS_HdrErr	call DoRead
		jr c,ReadBkCHS1			; read fault -->
		rl ReadWithHdr			; check headers manually?
		jr nz,_044a			; ***
		rl SkipSpared			; ***
		jr z,_0447			; ***
		dec ReadWithHdr			; ***
_0447		jp ReadBkCHS1			; ***
; check sector header
_044a		ld R0,#(ReadBuffer-19)/256
		ld R1,#(ReadBuffer-19)#256
		ld R15,#WorkingRegs+2		; =R2
		ld R14,#9
RdBkCHS_Hdr1	ldei @R15,@RR0			; move header into WorkingRegs
		djnz R14,RdBkCHS_Hdr1
		clr ReadWithHdr			; do retry with automatic header compare
		cp R2,#0			; header starts with 0?
		jp nz,RdBkCHS_HdrErr		;  no --> retry
		cp R6,ActSector			; correct sector?
		jp nz,RdBkCHS_HdrErr		;  no --> retry
		scf    
		adc R8,R4			; complement matches cylinderLo?
		jp nz,RdBkCHS_HdrErr		;  no --> retry
		adc R7,R3			; complement matches cylinderHi?
		jp nz,RdBkCHS_HdrErr		;  no --> retry
		adc R9,R5			; complement matches head?
		jp nz,RdBkCHS_HdrErr		;  no --> retry
		adc R10,R6			; complement matches sector?
		jp nz,RdBkCHS_HdrErr		;  no --> retry
		ld R7,#6
RdBkCHS_Hdr2	lde R8,@RR0
		incw RR0
		cp R8,#0			; last 6 header bytes zero?
		jp nz,RdBkCHS_HdrErr		;  no --> retry
		djnz R7,RdBkCHS_Hdr2
		dec ReadWithHdr
		ret    				; passed, exit
; 
ReadBkCHS1	or Status1,RWstat		; merge result into Status1
		rl DiskSM_Timeout
		jp nc,Ret_4
;
; DoRead produced errors, so retry as requested by the host
Retry		clr 043h			; ***
		ld R14,#0FFh			; count timeouts as single fault
		inc Status1			; operation failed		
		cp RetryCnt,#0
		jp z,Ret_4
		call ReRead1			; re-read block (RetryCnt) times
		ld Status4,RetriesLeft		; number of retries needed
		push FLAGS			; save ReRead result (UF1+2)
		jr z,RetryGood			;  block was readable -->
; block was not readable, try another up to 90 times
		clr R14
		ld RetryCnt,#90			; re-read another 90 times
		dec ReReadMode			;  but stop when good block found
		call ReRead
		jp z,RetryGood			;  block was readable -->
; block was unreadable, enter block into bad block table
		pop FLAGS
		call WriteDone
		jp c,Ret_2			; ***
		cp BadBlocks,#59		; max. 100 bad blocks allowed
		jr c,BadBlock
		or Status2,#BadBkTbl_Full	; Bad block table overflow
		ret    
BadBlock	inc BadBlocks			; increment bad block number
		jp SpMap_Lookup2		; and enter block into spare table
; we had at least one good read
RetryGood	pop FLAGS			; retrieve C and Z from ReRead
		jp nc,WriteDone			; threshold not exceeded, leave
; re-write block
		tm Status1,#Stat_Seek_Err	; seek error?
		jp nz,Ret_4			;  yes --> abort
		rl 03Bh				; spared block?
		jp nz,Ret_4			;  yes --> abort
		call MoveBlock			; shift data from read into write buffer
		call DoWrite			; and re-write questionable block
		jp c,ForceSparing		; failed --> go sparing!
; check re-written block
GoDiag		call Diagnostic			; check disk surface
		jp c,ForceSparing		; bad --> go sparing!
		jp WriteDone			; leave
;
;********************************************************************
; store WriteBuffer into current Cylinder/Head/Sector
;********************************************************************
;
WriteBlockCHS	cp HostCommand,#6		; force sparing command?
		jp z,ForceSparing		;  yes -->
;
		call DoWrite			; perform actual write operation
		or Status1,RWstat		; merge result into Status1
		tm Status1,#Stat_Seek_Err+Op_Failed	; seek error or unsuccessful op?
		jp nz,Ret_4			;  yes --> abort!
		jp c,ForceSparing		; write fault --> spare this block!
		rl BlockIsBad			; is the current block in the bad block table?
		jp c,GoDiag			;  yes --> check this block!
		cp HostCommand,#2		; simple write
		jp z,WriteDone			;  yes --> done.
; verify the block we just have written
		ld R12,#10
		call WaitR12ms			; wait 10ms
		call SetSeekHdrBuf		; set header buffer to 13E0h
		call DoRead1			; read back user data
		or Status1,RWstat		; merge result into Status1
		jp nc,WriteDone			;  ok -->
; verify failed. Check disk surface and spare block if needed
		call Diagnostic			; check disk surface
		jp nc,WriteDone			;  good -->
ForceSparing	cp SpareBlocks,#32		; some spares left?
		jr c,SpareBlock			;  yes --> go sparing
		or Status2,#SprTbl_Full		;  otherwise set spare table overflow
		ret    
;
; store the contents of WriteBuffer into the next free spare block
;
SpareBlock	ld R15,SpareBlocks		; number of next free spare block
		call LtoC_Spared		; setup CHS
		push 03Bh
		call GoCHS			; seek there
		pop 03Bh
		call DoWrite			; write data into spare block
		tm Status1,#Stat_Seek_Err+Op_Failed
		jp nz,Ret_4			; exit on fault
		inc SpareBlocks			; increment number of spares used
		jp c,ForceSparing
		call Diagnostic			; check disk surface of spare block
		jp c,ForceSparing		; bad --> spare again
		or Status2,#Stat_Spare		; sparing occurred
		rl BlockStat			; next entry is spared block
		call SpMap_Lookup2		; enter block into spare table
; see if the current block was a bad block in the spare map. If it has been spared,
; change its type from bad into spared. If it is good now, remove it from list.
		jp c,Ret_4			; ***
WriteDone	dec SpareMap_RW
		ld SpMap_Read,#0FFh		; read access
		call SpMap_Lookup2
		clr SpareMap_RW
		clr SpMap_Read
Ret_4		ret    
;
;********************************************************************
; process the spare table image in RAM
;
; This code is used
; a) to look up a block in the spare table (SpMap_Read = 0FFh)
; b) to enter a block into the spare table (SpMap_Read = 000h)
;********************************************************************
; 
; We enter here after bad block found or sparing occurred (SpMap_Read=0)
; The current block number is entered into the spare table. BlockStat tells 
; us whether it is a bad or spared block.

SpMap_Lookup2	call CheckFence			; spare map ok?
		jr c,_056e			;  no --> abort
		call _056f			; ***
		rcf    				; ***
_056e		ret
;
_056f		ld R14,#0FFh			; (NIL)
		rl SpMap_Read			; spare table update planned?
		jp nc,SpMap_GetSList		;  yes -->
; for read operation, see if it is necessary to process a new sublist
		cp LBAmi,LastSublist		; did the sublist change?
		ld LastSublist,LBAmi		; update sublist number
		jp nz,SpMap_GetSList		;  changed --> 
		cp LBAlo,SpMap_Lowest		; is our block above the lowest in list?
		jp ule,SpMap_GetSList		;  yes -->
		cp LBAlo,SpMap_Highest		; is our block below the highest in list?
		jp c,Ret_4			;  no --> exit, nothing to do
; process new sublist
SpMap_GetSList	ld R0,#SpareMapEnd/256
		ld R1,#SpareMapEnd#256
		lde R2,@RR0			; RR2 points to first free RAM address
		inc R1				;  after the spare table
		lde R3,@RR0
		ld R4,#SpareMap/256
		ld R5,#SpareMap#256
		ld R6,LBAmi		
		rl R6				; RR4 points to the SpareMap entry
		add R5,R6			;  for LBAmi
		lde R6,@RR4			; get sublist address MSB
		cp R6,R14			; = NIL?
		jr nz,SpMap_GetSLst1		;  no --> 
; no sublist exists for this LBAmi, so create one. Our new sublist will
; start at the first free RAM location, given in RR2
		clr SpMap_Lowest		; SpMap_Lowest = 0
		ld SpMap_Highest,R6		; SpMap_Highest = 255
		rl SpMap_Read			; are we just reading?
SpMap_Ret1	jp nz,Ret_4			;  yes --> exit, nothing to do
		ld R6,R2			; set sublist address MSB
		lde @RR4,R2			; enter start address of newly 
		inc R5				;  created sublist into SpareMap
		lde @RR4,R3
		dec R5				; move SpareMap pointer RR4 back to MSB
		ld R8,#0
		lde @RR2,R8			; set sublist length to zero
		incw RR2			; increment first free RAM address
SpMap_GetSLst1	ld SpMap_Lowest,LBAlo		; set our LBAlo to be the lowest in list
		inc R5
		lde R7,@RR4			; get sublist address LSB from SpareMap
		ld R8,R6			; now RR6 contains the sublist address
		ld R9,R7			;  copy it into RR8
		lde R12,@RR6			; get number of entries in list
; now search the sublist for our block's LBA
SpMap_GetSLst3	dec R12				;  decrement number of entries to process
		ld SpMap_Highest,R14		; SpMap_Highest = 0FFh
		jp mi,SpMap_GetSLst2		;  all entries done -->
		incw RR6			; move sublist pointer to next entry's LBAlo
		ld R15,#SpMap_Highest
		ldei @R15,@RR6			;  and assume it to be the highest
		lde R11,@RR6			; move type byte into R11
		cp SpMap_Highest,LBAlo		; is this our block?
		jp z,SpMap_Found		;  yes --> we found it!
		jp c,SpMap_GetSLst3		;  no, we are below it --> do next one
		decw RR6			; move back sublist pointer
		decw RR6
; now RR6 points to the byte before our new entry's target position
; in case we are just looking for an entry, we can leave because it isn't there
SpMap_GetSLst2	rl SpMap_Read			; are we just reading?
		jp nz,SpMap_Ret1		;  yes --> exit, nothing to do
;
; *** enter current block into its sublist
; this code is reached only for write access, i.e. after sparing
SpMap_Write	ld SpareTblDirty,R14		; after that spare table tracks need an update
		lde R15,@RR8			; RR8 points to sublist start
		inc R15				; increment number of entries
		lde @RR8,R15
		ld R8,R2			; RR2 points to first free RAM address
		ld R9,R3			; save this into RR8
		incw RR2			; and increment by two
		incw RR2			;  (which is the room our new entry needs)
		decw RR0			; RR0 points to SpareMapEnd LSB, move to MSB
		lde @RR0,R2			; enter new value
		inc R1
		lde @RR0,R3			; and update LSB as well
; move all bytes between RR6 (the place where our entry should be) and RR8 (the
; first unused RAM location, should contain 0FFh) two bytes forward to make room
SpMap_UpdtSLst	lde R15,@RR8
		lde @RR2,R15
		decw RR8
		decw RR2
		cp R8,R6
		jp nz,SpMap_UpdtSLst
		cp R9,R7
		jp nz,SpMap_UpdtSLst
		incw RR6
		ld R15,#LBAlo
		ldei @RR6,@R15			; store our LBAlo into sublist
		clr R15
		call BuildBlockType		; build block type byte and store into sublist 
; adjust the pointers in SpareMap
; i.e. increment all pointers that point to an address above RR8 by two
		ld R0,#SpareMap/256
		ld R1,#SpareMap#256
		ld R2,#04Dh			; 04Dh sublists to process
SpMap_UpdtMap3	lde R4,@RR0			; MSB of pointer to sublist
		inc R1
		lde R5,@RR0			; LSB of pointer to sublist
		cp R4,R14			; MSB = NIL?
		jr z,SpMap_UpdtMap1		;  yes --> next one
		cp R4,R8			; pointer MSB < R8?
		jr c,SpMap_UpdtMap1		;  yes --> next one
		jp nz,SpMap_UpdtMap2		; don't check LSB if MSBs are different
		cp R5,R9			; pointer LSB < R9?
		jr ule,SpMap_UpdtMap1		;  yes --> next one
SpMap_UpdtMap2	incw RR4			; increment sublist address by 2
		incw RR4
		dec R1
		lde @RR0,R4			; and write back sublist pointer	
		inc R1
		lde @RR0,R5
SpMap_UpdtMap1	inc R1				; go to next SpareMap entry
		djnz R2,SpMap_UpdtMap3		; and process next sublist
		ret
;
; *** we found our block in the sublist
SpMap_Found	dec R12				; are we the first entry?
		jr mi,SpMap_Fnd1		;  yes -->
		incw RR6
		lde R10,@RR6			;  else set SpMap_Highest to 
		ld SpMap_Highest,R10		;  the next entry's LBAlo
		decw RR6	
SpMap_Fnd1	rl SpMap_Read			; are we just reading?
		jr c,SpMap_Fnd_Rd		;  yes -->
;
; update sublist entry at RR6 with BlockStat (080h = spared, 
; 040h = bad) and spare number
		lde R15,@RR6			; get block type byte from sublist
BuildBlockType	tm BlockStat,#080h		; did we just spare?
		jr pl,BuildBlkTp1		;  no -->
		and R15,#0E0h			; clear spare number and
		or R15,SpareBlocks		; merge in number of last used spare
		dec R15
BuildBlkTp1	ld SpareTblDirty,#0FFh		; spare table track needs update
		or R15,BlockStat		; merge in spare and bad block flags
		lde @RR6,R15			;  and write back block type byte
		ret 
;   
; our block is in the spare table, RR6 points at its sublist entry 
SpMap_Fnd_Rd	rl SpareMap_RW			; called after Read/WriteCHS?
		jr c,SpMap_Fnd_Sp		;  yes -->
;
; this spare table lookup has been requested by a Seek
SpMap_Fnd_Sk	ld R8,#0			; Z=0
		rl R11				; is this a spared block?
		jr nc,SpMap_Fnd_Sk1		;  no -->
; replace user block cylinder/head/sector with spare block's CHS
		ld ActSector,R11		; get sector number from type byte
		rr ActSector			;   (correct rl)
		ld CylinderL,#153		;  and update CHS
		clr CylinderH
		ld Head,ActSector
		swap Head
		and Head,#3
		and ActSector,#15
		ld R8,#040h			; Z=1
SpMap_Fnd_Sk1	rl R11				; is this a bad block?
		jr nc,SpMap_Fnd_Sk2		;  no -->
		ld BlockIsBad,#0FFh		; the current block is a bad block
SpMap_Fnd_Sk2	ld FLAGS,R8			; return Z=1 if spared block
SpMap_Fnd_Sp1	ret    
;
; this spare table lookup has been requested after sparing occured during a
; ReadCHS or WriteCHS operation
SpMap_Fnd_Sp	tm R11,#040h			; is it a bad block?
		jp z,SpMap_Fnd_Sp1		;  no --> abort
		ld SpareTblDirty,R14		; set spare table invalid
		dec BadBlocks			;  this block is not a bad block anymore
		and Status2,#0FFh-BadBkTbl_Full
		and R11,#0BFh			; clear bad block bit
		lde @RR6,R11			; and store type byte into sublist
		jp mi,SpMap_Fnd_Sp1		; is it a spared block? --> exit
; if the bad block has not been spared, its entry at RR6 has to be removed
		dec R5				; move SpareMap pointer to sublist MSB
		ld R0,R6			; save sublist pointer in RR0
		ld R1,R7
		lde R10,@RR8			; RR8 points to sublist start
		dec R10				; decrement number of entries
		lde @RR8,R10
		jr nz,SpMap_Remove1		; list not empty? -->
		decw RR6			; compensate for length byte
		lde @RR4,R14			; set sublist MSB in SpareMap to 0FFh
		inc R14				;  (R14 =0 if two, =0FFh if three bytes to delete)
; first, adjust SpareMap
SpMap_Remove1	ld R4,#SpareMap/256
		ld R5,#SpareMap#256		; RR4 points to first SpareMap entry
		ld R10,#04Eh			; process 04Dh entries + SpareMapEnd
SpMap_RmvLp	lde R8,@RR4			; get entry into RR8
		inc R5
		lde R9,@RR4
		cp R8,#0FFh			; is a sublist present?
		jr z,SpMap_RmvNxt		;  no --> next one
		cp R8,R6			; is its MSB below our entry's?
		jr c,SpMap_RmvNxt		;  no --> next one
		jp nz,SpMap_Remove2		; MSB not identical? --> must be bigger
		cp R9,R7			; check LSBs
		jr ule,SpMap_RmvNxt		;  less --> next one
; sublist starts after the deleted bytes
SpMap_Remove2	decw RR8			; adjust sublist address			
		decw RR8	
		rl R14				; two bytes removed?
		jr nz,SpMap_Remove3		;  yes -->
		decw RR8			; otherwise decrement once more
SpMap_Remove3	dec R5
		lde @RR4,R8			; store new sublist address into SpareMap
		inc R5
		lde @RR4,R9
SpMap_RmvNxt	inc R5				; advance SpareMap pointer
		djnz R10,SpMap_RmvLp			; until all done
; now, move together the sublists
		decw RR6			; RR6 points to first byte to be overwritten
SpMap_Remove4	incw RR0			; RR0 points to first byte after deleted entry
		lde R15,@RR0
		lde @RR6,R15
		incw RR6
		cp R0,R2			; RR2 points to first free RAM address
		jp nz,SpMap_Remove4
		cp R1,R3
		jp nz,SpMap_Remove4
		ret    
;
;********************************************************************
; diagnostics routine: check media quality of current block
;
; re-read block 100 times. RetriesLeft holds the number of bad reads. 
; C=0 when 30% threshold exceeded, Z=1 if at least one good read.
;********************************************************************
;
Diagnostic	ld RetryCnt,#100
		ld SparingTh,#30
		clr R14				; count timeouts as eight faults
		call SetSeekHdrBuf
; re-read block (RetryCnt) times
; count timeouts as single fault if R14=0FFh or as eight faults if R14=0
ReRead1		clr ReReadMode			; don't stop when good read occurred
; re-read block until good read or (RetryCnt) expired
ReRead		clr ReReadGood
		ld R13,RetryCnt
		ld RetriesLeft,R13		; assume all reads bad
; re-read the block (R13) times
DiagRdLp	call DoRead1			; read block again
		jr c,DiagRdBad			; failed -->
		and Status1,#0FFh-Op_Failed	;  else clear fault bit
		dec RetriesLeft			;  decrement number of bad reads
		call SetSeekHdrBuf1		; set header and flag successful read
		rl ReReadMode			; stop after first good read?
		jp c,DiagRdDone			;  yes -->
DiagRdBad	or Status1,RWstat
		rl R14				; count timeouts x8?
		jr nz,DiagRdNext		;  no --> next try
		tm RWstat,#Stat_No_Hdr		; timeout occurred?
		jr z,DiagRdNext			;  no --> next try
		ld R15,#8
DiagRd1		dec R13				; one timeout counts as 8 CRCs
		jp z,DiagRdDone			; all done?
		djnz R15,DiagRd1
DiagRdNext	djnz R13,DiagRdLp
DiagRdDone	rl ReReadGood
		rrc R0				; Z=1 if good read occurred
		cp RetriesLeft,SparingTh	; compare bad reads with threshold
		ccf    				; C=0 if more than allowed
		rrc R0
		ld FLAGS,R0
		ret    
;
_074c		jp SpMap_GetSLst1		; ***
_074f		jp WarmStart1a			; ***
;
; wait for a sector pulse. timeout if none has arrived within 26 ms.
;
WaitSctrPulse	call SetT0_26ms			; set timeout for 1.625 disk revolutions
		clr IRQ				;  and clear P3.1 interrupt flag
WaitSP_Loop	tm T0,#0FFh			; did timer run off?
		jp z,WaitSP_Done
		tm IRQ,#4			; did a sector pulse (P3.1=hi) occur?
		jp z,WaitSP_Loop
WaitSP_Done	ret    				; return if timer run off or sector pulse arrived
;
; wait for the state machine to finish. DiskSM_Timeout = 0FFh if timeout, Z=0 if ok.
;
WaitDiskSM	clr RWstat
		ld R5,#6			; timeout = 6* 26ms = 156ms
		clr DiskSM_Timeout
WaitDSM4	call SetT0_26ms
WaitDSM2	tm T0,#0FFh			; timer expired?
		jr nz,WaitDSM1			;  no --> check state machine
		rl DSM_FastTo			;  slow timeout?
		jp nz,WaitDSM5			;   no --> fault!
		rl ReadWithHdr			; manual header checking?
		jp nz,WaitDSM5			;  yes --> abort!
		dec R5				;   else wait again
		jp nz,WaitDSM4
WaitDSM5	or RWstat,#Stat_No_Hdr		; report timeout error 
		dec DiskSM_Timeout
		ret    
;
WaitDSM1	ld R4,#2
WaitDSM3	tm P2,#SECTDN			; state machine finished?
		jp nz,WaitDSM2			;  no --> continue waiting
		djnz R4,WaitDSM3		;  yes, check again to be sure
		ret    
;
; setup T0 for a single 26.1 ms delay
;
SetT0_26ms	ld T0,#0FFh
		clr PRE0
		ld TMR,#00000011b		; load & start T0
		ret  
;  
; build a header for current CHS and put it into (R0.02Eh)
;
BuildHeader	ld R1,HeaderBufLSB 		; LSB of header address
		ld R10,#1
		ld R3,#4			; complement 4 bytes
BldHdr_Lp1	ld R4,ActCylH-1(R3)		; copy 017..019h (cylinder, head, sector)
		com R4
		ld ComCylH-1(R3),R4		;  complemented into 01A..01Ch
		djnz R3,BldHdr_Lp1
		ld R2,#9			; copy 9 bytes
		ld R15,#HeaderImage
BldHdr_Lp2	ldei @RR0,@R15			; copy 016..01C into RAM (R0.02Eh)
		djnz R2,BldHdr_Lp2
		ld R9,#7
BldHdr_Lp3	lde @RR0,R2			; clear the following 7 bytes
		inc R1
		djnz R9,BldHdr_Lp3
		ret   
;
; prepare i8253 CTC
;
LoadCTCforRd	ld R2,#551#256
		ld R3,#553#256
		ld R4,#17
LoadCTC		and P2,#0FFh-Msel1		; Z8 --> Timer
		ld R0,RAM_MSB
		ld R1,#0			; counter 0 data register
		ld R5,#551/256			; MSB is the same for all ctrs
		ld P01M,#036h			; enable extended memory timing
		lde @RR0,R2			; set counter 0 = 551 (16 bit, LSB first)
		lde @RR0,R5
		inc R1
		lde @RR0,R3			; set counter 1 = 553 (16 bit, LSB first)
		lde @RR0,R5
		inc R1
		lde @RR0,R4			; set counter 2 = 17 (8 bit)
		ld P3,P3Mask2			; WRTSM + Heads
		jp RAMtoZ8a
;
;
; *** do actual read operation ***
; returns carry set when error occurred
;
DoRead		ld StatusBufLSB,#(ReadBuffer-4)#256
		ld RAM_MSB,#RAM/256
		clr HeaderBufLSB 		; read header will be built at 01000h
		rl ReadWithHdr			; manual header checking?
		jr nz,LoadCTCforRd1
;
_07eb		call DoRead1			; ***
		push FLAGS			; ***
		jr c,_081c			; ***
		cp MaskChkHdr,#3		; ***
		jr ge,_0832			; ***
		ld R13,#03Dh			; ***
		ld R12,#000h			; ***
		ld R6,ActSector			; ***
		cp R6,#8			; ***
		jr lt,_0806			; ***
		inc R13				; ***
		and R6,#7			; ***
_0806		ld R11,@R13			; ***
		ld R0,#_006d/256		; ***
		ld R1,#_006d#256		; ***
		add R1,R6			; ***
		ldc R12,@RR0			; ***
		tm R11,R12			; ***
		jr nz,_081a			; ***
		inc MaskChkHdr			; ***
		or R11,R12			; ***
		ld @R13,R11			; ***
_081a		jr _0832			; ***
;
_081c		cp MaskChkHdr,#3		; ***
		jr ge,_0832			; ***
		push RAM_MSB			; ***
		push HeaderBufLSB		; ***
		call _0a3e			; ***
		pop HeaderBufLSB		; ***
		pop RAM_MSB			; ***
		jr c,_0832			; ***
		pop FLAGS			; ***
		jr _07eb			; ***
_0832		pop FLAGS			; ***
		ret    				; ***
;
; prepare i8253 CTC
;
LoadCTCforRd1	ld R2,#551#256
		ld R3,#553#256
		ld R4,#17
		and P2,#0FFh-Msel1		; Z8 --> Timer
		ld R0,RAM_MSB
		ld R1,#0			; counter 0 data register
		ld R5,#551/256			; MSB is the same for all ctrs
		ld P01M,#036h			; enable extended memory timing
		lde @RR0,R2			; set counter 0 = 551 (16 bit, LSB first)
		lde @RR0,R5
		inc R1
		lde @RR0,R3			; set counter 1 = 553 (16 bit, LSB first)
		lde @RR0,R5
		inc R1
		lde @RR0,R4			; set counter 2 = 17 (8 bit)
		ld P3,P3Mask2			; WRTSM + Heads
		or P2,#Msel0+Msel1		; Z8 --> Mem
		ld P01M,#016h			; reconnect bus
		ld P3,P3Mask1			; read including header
		jr ReadHeader
;
DoRead1		call WaitSctrPulse		; wait for sector mark
		call LoadCTCforRd		; start timers
		call BuildHeader		; build header image in RAM
ReadHeader	ld R1,HeaderBufLSB
		and P2,#0FFh-DSTART-Msel0	; State Machine --> Mem, reset state machine
		lde @RR0,R2			; dummy write to set RAM address to R0.R1
		ld P01M,#01Ch			; release data & address bus
		call WaitDiskSM			; and wait for its job to be done
		jr nz,ReadHdr1			; timeout --> 
		tm P3,#CRCERR			;  no, CRC fault?
		jr nz,ReadHdr1			;   no -->
		dec DiskSM_Timeout		;   else set fault flag,
		or RWstat,#Stat_Rd_Err		;   and set CRC error flag
ReadHdr1	and P3,#0FFh-RDHDR
DoWrite3	or P2,#DSTART			; stop state machine
; give RAM back to our microcontroller
RAMtoZ8		or P2,#DRW_Read
RAMtoZ8a	or P2,#Msel0+Msel1		; Z8 --> Mem
		ld P01M,#016h			; reconnect bus
		rl DiskSM_Timeout		; and set carry if fault occurred
		ret   
; 
; *** write block to disk
; returns carry set when error occurred
;
DoWrite		ld StatusBufLSB,#(WriteBuffer-4)#256
		tm Status1,#Stat_Seek_Err	; did the seek fail?
		jp nz,DoWrite_Abort		;  yes --> abort
		tm Status2,#020h    		; ***
		jp nz,DoWrite_Abort		; ***
		cp MaskChkHdr,#3    		; ***
		jr ge,_08ae    			; ***
		call _0a3e    			; ***
		jr c,DoWrite_Abort		; ***
_08ae		rl SettlingReqd			; did we just move the heads?
		jr nc,DoWrite1			;  no --> continue
		clr SettlingReqd
		ld R12,#40			; else wait some time to settle
		call WaitR12ms
DoWrite1	call WaitSctrPulse		; wait for sector mark
		ld RAM_MSB,#RAM/256
		clr HeaderBufLSB 		; build write header at 01000h
		ld R2,#570#256
		ld R3,#574#256
		ld R4,#38
		call LoadCTC			; prepare i8253 for write sequence
		call BuildHeader		; build header image in RAM
		ld R9,#22
		call BldHdr_Lp3			; zero another 22 bytes
		lde @RR0,R10			; followed by a 001h
		inc R1
		lde @RR0,R2			; followed by another 000h
		ld R1,HeaderBufLSB
		and P2,#0FFh-DSTART-Msel0	; State Machine --> Mem, reset state machine
		lde @RR0,R2			; dummy write to set RAM address to R0.R1
		ld P01M,#01Ch			; release data & address bus
		and P2,#0FFh-DRW_Read
		and P0,#0FFh-PreComp		; (moved here with V3.98)
		ld R4,ActCylH
		ld R5,ActCylL
		sub R5,#128
		sbc R4,#0
		jr mi,DoWrite2
		or P0,#PreComp			; enable precomp for cylinders >128
DoWrite2	call WaitDiskSM			; wait for state machine to be finished
		jp DoWrite3			; and leave via ReadHeader
;
DoWrite_Abort	or Status1,#Op_Failed
		ld DiskSM_Timeout,#0FFh		; set fault flag
		jp RAMtoZ8a			; and leave
SetSeekHdrBuf1	ld ReReadGood,#0FFh		; UF2=1
SetSeekHdrBuf	ld RAM_MSB,#SeekHeaderBuf/256
		ld HeaderBufLSB,#SeekHeaderBuf#256
		ret 
;********************************************************************
; *** GoCHS routine ***
;
; Seek to the given CHS target. In case of fault, seek to track 0
; and try again. If this also fails, seek to track 155 and try again.
; Abort with Stat_Seek_Err if 3 retries failed.
;********************************************************************
;
GoCHS		ld SpareTblInvalid,#0FFh	; set spare table invalid
; enter here for R/W seek
Seek		clr BlockIsSpared
		clr DelayMotorStep		; don't wait before next motor step
		clr OnTrackFlg
		rl StpMotOff
		jr z,Seek2			; stepper is already powered
		ld R10,#2
		call MoveStepMot1		; turn on step motor
		clr DelayMotorStep		; don't wait before next motor step
		ld R12,#37
		call WaitR12ms
		clr StpMotOff
Seek2		rl SpareTblInvalid		; ***
		jp nz,Seek5			; ***
		call CheckFence			; ***
		jp nz,Ret_6			;  yes --> skip lookup
; see if our seek target has been spared
		ld R14,#0FFh			; (NIL)
		cp LBAmi,LastSublist		; current spare table sublist in memory?
		ld LastSublist,LBAmi		; update sublist number
		jp nz,Seek3			;  no -->
		cp LBAlo,SpMap_Lowest		; is our block above the lowest in list?
		jp ule,Seek3			;  yes -->
		cp LBAlo,SpMap_Highest		; is our block below the highest in list?
		jp c,Seek5			; ***
Seek3		ld R4,#SpareMap/256
		ld R5,#SpareMap#256
		ld R6,LBAmi		
		rl R6				; RR4 points to the SpareMap entry
		add R5,R6			;  for LBAmi
		lde R6,@RR4			; get sublist address MSB
		cp R6,R14			; = NIL?
		jr nz,Seek4			;  no --> 
; no sublist exists for this LBAmi. Set boundary values and continue
		clr SpMap_Lowest		; SpMap_Lowest = 0
		ld SpMap_Highest,R6		; SpMap_Highest = 255
		jr Seek5			; ***
; our block may be in the sublist. Look it up and update C/H/S if required
Seek4		ld SpMap_Read,#0FFh
		call _074c			; lookup block in sublist
		clr SpMap_Read			; ***
		jr nz,Seek5			; nothing found -->
		ld BlockIsSpared,#0FFh
		rl SkipSpared			; shall we skip spared blocks?
		jp nz,Ret_6			; ***
; calculate seek offset
; returns Z=0 if on target track, 04Bh = delay, 046.47h = difference
Seek5		clr R6
		clr 046h			; ***
		clr 047h			; ***
		ld 048h,#1
		ld R4,CylinderH 
		ld R5,CylinderL
		sub R5,ActCylL			; RR4 = target - current
		sbc R4,ActCylH
		ld R7,FLAGS			; remember sign
		jr pl,_0994			; go forward? -->
		ld 048h,#-1	
		com R4				; make difference positive
		com R5
		incw RR4
_0994		decw RR4
; fast seek algorithm
; RR4: distance, S: direction
AccelSeek	incw RR4			; set zero flag if difference = 0
		push FLAGS			;  and save it
		sub R5,#9			; difference >9?
		sbc R4,R6
		jr pl,AccelSk1			; yes -->
; come here if less then 9 tracks to go
		dec 046h
		ld 049h,#26			; delay = 26
		add 047h,#24			; 047h +24
		add R5,#8			; now RR4 = difference -1
		jr AccelSk2
; come here if more than 9 tracks to go 
AccelSk1	ld 049h,#28			; delay = 28
		sub R5,#10			; subtract another 10 
		sbc R4,R6
		jr pl,AccelSk3			; still positive -->
		ld 049h,#35			; else delay = 35
AccelSk3	add R5,#11			; now RR4 = difference -8
;
AccelSk2	adc R4,R6			; adjust MSB
		ld 044h,R4
		ld 045h,R5
		pop FLAGS			; restore zero flag
		jp z,Seek_OnTrack
		tm P2,#Trk0			; are we on Track 0?
		jr nz,Seek_NxtStp1		;  no -->
		cp 048h,#0Ffh			; 048h set?
		jr nz,Seek_NxtStp1		;  yes -->
		cp CurStep,#0			; are we at Phase 0?
		jp z,Seek_OnTrack		;  yes -->
;
Seek_NxtStp1	call _0b34
		call _0b34
		jr z,Seek_OnTrack
		cp CurStep,#0			; are we at Phase 0?
		jr nz,Seek_NxtStp3		;  no -->
		tm P2,#Trk0			; are we on Track 0?
Seek_NxtStp3	jp nz,Seek_NxtStp1		;  no -->
		rr R8
		jp nc,Seek_NxtStp1
		cp 048h,#0FFh			; 048h set?
		jp nz,Seek_NxtStp1		;  yes -->
Seek_OnTrack	clr R1				; ***
		clr R12				; ***
		cp Head,ActHead			; ***
		jr z,_0a15			; ***
		inc R1				; ***
		inc R12				; ***
		ld R0,Head
		swap R0
		or R0,#WRTSM
		ld P3Mask2,R0			; generate P3 masks: 005 = 01hh0000
		ld P3,R0			; select head
		or R0,#WRTSM+RDHDR
		ld P3Mask1,R0			; 004 = 11hh0000
		ld ActHead,Head			;  and update ActHead
_0a15		inc OnTrackFlg			; have the heads been moved?
		jr nz,Seek_OnTrk2		;  no -->
		ld ActCylH,CylinderH 
		ld ActCylL,CylinderL
		ld R12,049h			; ***
		inc R1				; ***
		clr 043h			; ***
Seek_OnTrk2	dec  R1				; ***
		jr mi,_0a37			; ***
		clr 03Dh			; ***
		clr 03Eh			; ***
		clr MaskChkHdr			; ***
		ld 02Bh,#2
		and Status1,#0FFh-Stat_Seek_Err	; clear seek error
		and Status2,#0FFh-Stat_Seek_Flt
_0a37		tm  R12,#0Ffh			; ***
		jp nz,WaitR12ms			; ***
Ret_6		ret   
; 
; now let's read three consecutive headers within four disk revolutions
_0a3e		ld OnTrackCtr,#4*16		; four disk revolutions = 64 sectors
		call SetSeekHdrBuf
		call WaitSctrPulse
_0a47		ld T1,#15			; ***
		ld R0,OnTrackCtr		; ***
		and R0,#7			; ***
		jr nz,_0a54			; ***
		ld T1,#178			; ***
_0a54		call Wait1ms_1			; ***
		dec DSM_FastTo			; set fast state machine timeout
		call LoadCTCforRd
		ld P3,P3Mask1			; WRTSM + RDHDR + Heads
		call ReadHeader			; read next header
		clr DSM_FastTo			; restore slow timeout
		jp nz,Seek_BadHdr		; abort on fault
		inc R1
		inc R1				; RR0 = 13E2h
		ld R15,#WorkingRegs+3
		ld R14,#8			; copy 8 bytes
Seek_ChkHdr4	ldei @R15,@RR0			; copy header into R3..R10
		djnz R14,Seek_ChkHdr4
		cp R3,#2
		jp nc,Seek_BadHdr		; cylinder out of range
		cp R6,#16
		jp nc,Seek_BadHdr		; sector out of range
		adc R7,R3
		jp nz,Seek_BadHdr		; cylinder & complement mismatch
		adc R8,R4
		jp nz,Seek_BadHdr		; cylinder & complement mismatch
		adc R9,R5
		jp nz,Seek_BadHdr		; head & complement mismatch
		adc R10,R6
		jp nz,Seek_BadHdr		; sector & complement mismatch
		ld ActCylH,R3
		ld ActCylL,R4
		cp CylinderH,ActCylH
		jr nz,Seek_ChkHdr1		; wrong cylinder
		cp CylinderL,ActCylL
Seek_ChkHdr1	jp nz,Seek_ChkHdr2
		cp ActHead,R5
		jr z,_0ab7
Seek_ChkHdr2	tm Status2,#Stat_Seek		; ****
		jr z,Seek_ChkHdr3		; ****
		or Status2,#001h		; ****
		scf    				; ****
		ret    

Seek_ChkHdr3	or Status2,#2    		; ****
		call Seek5    			; ****
		jr _0a3e    			; ****
_0ab7		ld R13,#03Dh			; ****
		ld R12,#0    			; ****
		cp R6,#8    			; ****
		jr lt,_0ac4    			; ****
		inc R13    			; ****
		and R6,#7    			; ****
_0ac4		ld R11,@R13    			; ****
		ld R0,#_006d/256		; ****
		ld R1,#_006d#256		; ****
		add R1,R6    			; ****
		ldc R12,@RR0    		; ****
		tm R11,R12    			; ****
		jr nz,Seek_BadHdr    		; ****
		inc MaskChkHdr    		; ****
		cp MaskChkHdr,#3    		; ****
		jr lt,_0add    			; ****
		and Status1,#0FFh-Stat_Seek_Err	; ****
		ret        			; ****
;
_0add		or R11,R12        		; ****
		ld @R13,R11        		; ****
Seek_BadHdr	dec OnTrackCtr        		; ****
		jp nz,_0a47        		; ****
		or Status2,#Stat_Seek_Flt
		dec 02Bh        		; ****
		jr pl,_0af2        		; ****
		scf       	     		; ****
		or Status1,#Stat_Seek_Err	; set seek error if counter expired
		ret    
;
_0af2		push Status1           		; ****
		push Status2           		; ****
		push SpareTblInvalid        	; ****
		push 02Bh       	     	; ****
		push CylinderH      	     	; ****
		push CylinderL      	     	; ****
		clr CylinderH       	     	; **** seek to track 0
		clr CylinderL       	     	; ****
		jr nz,_0b09       	     	; ****
		inc CylinderH         		; **** seek to track 308
		ld CylinderL,#308#256  		; ****
_0b09		call GoCHS       	     	; ****
		pop CylinderL       	    	; ****
		pop CylinderH      	     	; ****
		call GoCHS       	     	; ****
		pop 02Bh       	     		; ****
		pop SpareTblInvalid       	; ****
		pop Status2       	     	; ****
		pop Status1     	     	; ****
		jp _0a3e       	     		; ****
;
; ***** pause for 0.64 ms *****
;
Wait1ms		ld T1,#100
Wait1ms_1	ld PRE1,#012h			; use internal clock/4 (156.25kHz)
		ld TMR,#00Ch			; load & start timer
Wait1msLoop	tm T1,#0FFh
		jp nz,Wait1msLoop
		ret    
;
; pause for R12 * .064 ms
WaitR12ms	call Wait1ms			; wait for 0.64 ms
		djnz R12,WaitR12ms
		ret    
;
; called after _099a did not return zero
_0b34		ld R6,#AccelTab/256
		ld R7,#AccelTab#256
		add R7,047h
		ldc R8,@RR6			; get prescaler value
		ld R10,R8			;  into R10
		and R10,#03Eh			; clear bit 7+6+0
		or R10,#002h			; set bit 1
		add R7,#32			; next table starts 32 bytes later
		ldc R9,@RR6			; get timercnt from second table into R9
		add R7,#30			; third table starts 30(!) bytes later
		ldc R11,@RR6			; get interstep delay from 3rd table into R11
		tm R8,#080h			; has prescaler bit7 set? 
		jr z,_0b62			;  no -->
		com 048h
		inc 048h			; swap sign
		add CurStep,048h		; and add to current phase
		add CurStep,048h		;  (i.e. move phase one step back)
		com 048h			; restore sign 
		inc 048h
_0b62		tm R8,#040h			; has prescaler bit6 set? 
		jr z,MoveStepMot		;  no -->
		tm 047h,#001h			; table offset odd?
		jr nz,_0b77
		decw 044h
		jp pl,MoveStepMot
		add 047h,#2
		jp _0b34
_0b77		sub 047h,#2
;
; ***** move step motor ****
;
MoveStepMot	add CurStep,048h
		and CurStep,#7

MoveStepMot1	ld OnTrackFlg,#0FFh
		ld SettlingReqd,#0FFh
		ld R2,#StepVal/256
		ld R3,#StepVal#256
		add R3,CurStep
		ldc R6,@RR2			; get step value from table
		rl DelayMotorStep		; skip delay?
		ld DelayMotorStep,#0FFh
		jr z,MoveSM1			;  yes -->
MoveSMdly1	tm T1,#0FFh
		jp nz,MoveSMdly1		; wait until T1 expired
		ld PRE1,#6
		ld T1,R11			; and restart with delay from table 3
		ld TMR,#00Ch
MoveSMdly2	tm T1,#0FFh
		jp nz,MoveSMdly2		; wait until T1 expired
MoveSM1		ld PRE1,R10
		ld T1,R9			; and restart with delay from table 2
		ld P0,R6			; set motor phases
		ld TMR,#00Ch
		inc 047h
		dec R8
		ret    
;
; move 532 bytes from ReadBuffer to WriteBuffer
;
MoveBlock	ld R0,#(ReadBuffer+532-1)/256
		ld R1,#(ReadBuffer+532-1)#256
		ld R2,#(WriteBuffer+532-1)/256
		ld R3,#(WriteBuffer+532-1)#256
		ld R4,#532/256			; block length
		ld R5,#532#256
CopyBlk_Loop	lde R6,@RR0
		lde @RR2,R6
		decw RR0
		decw RR2
		decw RR4
		jp nz,CopyBlk_Loop
_0bd1		ret    
;
;********************************************************************
; read spare table and set up spare map in RAM at 01240h
;********************************************************************
;
GetSpareTable	or Status1,#Stat_SpTblUpd	; RAM buffer purged
		call ReadSpareTrk
		jp nz,_0bd1			; exit on failure
; build spare map fence
		call SetupFence			; preload registers
BuildFenceLp	ldei @RR2,@R15			; rebuild fence
		djnz R5,BuildFenceLp
		ld R0,#(ReadBuffer+ST_SparesAlloc)/256
		ld R1,#(ReadBuffer+ST_SparesAlloc)#256
		ld R4,#SpareBlocks
		ldei @R4,@RR0			; extract spare and bad block counts
		ldei @R4,@RR0			; (37h)=(102Ch), (38h)=(102Dh)
		ld R4,#416/256
		ld R5,#416#256
; copy RR4 bytes from (RR0) to (RR2)
CopySpareMap1	lde R6,@RR0			; copy spare map
		lde @RR2,R6
		incw RR0
		incw RR2
		decw RR4
		jp nz,CopySpareMap1
; check spare map fence
CheckFence	clr 04Eh			; clear fault flag
		call SetupFence			; preload registers
CheckFence_Lp	lde R4,@RR2			; get fence byte from RAM
		incw RR2
		cp R4,@R15			; compare with register
		jr z,CheckFence1		;  match -->
		ld 04Eh,#0FFh			;  else set fault flag
CheckFence1	inc R15
		djnz R5,CheckFence_Lp		; check next one
		rl 04Eh				; check fault flag
		jr z,_0c20			;  ok -->
_0c14		or Status2,#020h
		rl SkipSpared
		jp nz,_074f
		ld FLAGS,#080h			; set carry
		ret    
;
_0c20		lde R4,@RR2
		inc R4
		jp z,_0c32
		cp R4,#19
		jp lt,_0c14
		cp R4,#21
		jp ge,_0c14
_0c32		ld R3,#0DAh
		lde R4,@RR2
		cp R4,#18
		jp lt,_0c14
		cp R4,#19
		jp gt,_0c14
		and Status2,#0FFh-020h
		ld FLAGS,#040h			; set zero flag
		ret    
;
SetupFence	ld R2,#SpareMapFence/256
		ld R3,#SpareMapFence#256
		ld R5,#3			; check 3 bytes
		ld R15,#WorkingRegs+7		; address of R7
		ld R7,#055h
		ld R8,#099h
		ld R9,#0AAh
		ret   
;
; ***** read/write spare track *****
; if SpareTableDirty =0, read all spare table sectors until one good one found.
; if SpareTableDirty <>0, write all spare table sectors
;
ReadSpareTrk	clr SpareTblDirty		; set spare table valid
WriteSpareTrk	ld 041h,#1
		clr 029h
		clr 042h
		ld Head,#2
RWST_Lp1	ld ActSector,#15		;  and highest sector
		ld CylinderL,#153
		clr CylinderH
		call GoCHS			; seek to spare table sector
		clr SpareTblInvalid		; set spare table valid (=0)
		rl SpareTblDirty		; does the spare table need an update
		jr c,WriteST			;  yes --> write it
RWST_Lp		call SetSeekHdrBuf
		call _07eb			; otherwise read it from disk
		jr c,RWST_Next			; bad block? -->
; this was a good read
		ld R0,#013h
		ld R1,#0FFh
		lde R2,@RR0			; R2 = (13FF)
		ld R3,R2
		rl 042h				; 044h set?
		jp nz,ReadST1			;  yes -->
		dec 042h			;  else set it
		ld 029h,R2			; 029h = (13FF)
		jp ReadST2
ReadST1		dec R2
		sub R2,029h			; R2 = R2 - 029h -1
		cp R2,#080h
		jr nc,RWST_Next
ReadST2		ld 041h,R3			; 043 = (13FF)
		call ReadST_Retry
; next sector
RWST_Next	dec ActSector			; decrement sector
		jp pl,RWST_Lp			; and retry
		inc Head			; continue with head 3
		and Head,#3
		jp nz,RWST_Lp1
; all sectors done
		dec 041h
		sub 041h,029h
		cp 041h,#080h
		jr nc,_0CBA
; if we came here, we have read all sectors without finding a good one
		or Status2,#Stat_SprTbl_Err	; unable to read status sector
_0CBA		tm Status2,#Stat_SprTbl_Err
		ret    
;
WriteST		rl 042h				; 044h set?
		jp nz,WriteST1			;  yes -->
		dec 042h			;  else set 044h
		or Status1,#002h		; set fault flag
		ld R0,#010h
		ld R1,#033h
		lde R2,@RR0
		inc R2				; increment (1033)
		lde @RR0,R2
WriteST1	ld 04Bh,#3
WriteST_Next1	call DoWrite			; write block
		jr c,_0CDF			;  fault -->
		call SetSeekHdrBuf
		call DoRead1			; read back
_0CDF		jr nc,WriteST2			;  ok -->
		dec 04Bh
		jp z,WriteST3
		jr WriteST_Next
WriteST2	ld 04Bh,#3
		and Status1,#0FFh-002h		; clear fault flag
WriteST_Next 	dec ActSector			; next sector
		jp pl,WriteST_Next1
WriteST3	ld 04Bh,#3
		inc Head			; next head
		and Head,#3
		jp nz,RWST_Lp1
		tm Status1,#002h
WriteST4	ret    
;
; re-read current block up to 10 times
ReadST_Retry	ld 04Bh,#10
		push R3
ReadST_RtrLp	call DoRead			; read block
		jr nc,ReadST_RtrOk		;  ok -->
		dec 04Bh			;  else decrement retry counter
		jp nz,ReadST_RtrLp		;  and try again
		or Status2,#Stat_SprTbl_Err
		pop R0
		jr Ret_9
ReadST_RtrOk	and Status2,#0FFh-Stat_SprTbl_Err
		pop 029h
Ret_9		ret    
  
