;/***************************************************************************\
;* driver.s                                                                  *
;* Version 1.00                                                              *
;* SCC2691 UART Driver                                                       *
;* Copyright (C) 1991 Advanced RISC Machines Limited. All rights reserved.   *
;* Written by Dave Jaggar.                                                   *
;* Project started on 2st January 1992.                                      *
;* Last change was 2 minutes ago.                                            *
;\***************************************************************************/

                KEEP

SerialChipBase  EQU     &80000000
MR1             EQU     &0
MR2             EQU     &0
SR              EQU     &4
CSR             EQU     &4
CR              EQU     &8
RHR             EQU     &c
THR             EQU     &c
ACR             EQU     &10
ISR             EQU     &14
IMR             EQU     &14
CTUR            EQU     &18
CTLR            EQU     &1c

; Values for the fields in MR1

BitsPerChar     EQU     3 ; 0 for 5, 1 for 6, 2 for 7, 3 for 8
ParityType      EQU     4 ; 0 for Even, 4 for Odd
ParityMode      EQU     16 ; 0 for on, 8 for force, 16 for off
ErrorMode       EQU     0 ; 0 for char, 32 for block
RxInterrupt     EQU     0 ; 0 for char ready, 64 for FIFO full
RxAutoRTS       EQU     0 ; 0 for off, 128 for on

MR1Value        EQU     BitsPerChar + ParityType + ParityMode + ErrorMode + RxInterrupt + RxAutoRTS

; Values for the fields in MR2

StopBits        EQU     7 ; 7 for 1, 15 for 2 (others in the data sheet)
AutoCTS         EQU     0 ; 0 for off, 16 for on
TxAutoRTS       EQU     0 ; 0 for off, 32 for on

MR2Value        EQU     StopBits + AutoCTS + TxAutoRTS

; Values for the fields in the CSR

; The baud rates are defined both here, and in the top bit of the ACR
; called (the BRG bit). Most useful values are
; BRG = 0, 9 for 4K8, 11 for 9K6, 12 for 38K4
; BRG = 1, 12 for 19K2

Baud9600        EQU     &0000000b
Baud19200       EQU     &8000000c
Baud38400       EQU     &0000000c

; Values for the CR

ResetMR1        EQU     &10 ; Reset MR1
ResetRx         EQU     &20 ; Reset Receiver
ResetTx         EQU     &30 ; Reset Transmitter
ResetError      EQU     &40 ; Reset Error Status
ResetBreak      EQU     &50 ; Reset Break Status
StartTimer      EQU     &80 ; Start the timer
ResetTimer      EQU     &90 ; Reset the timer interrupt
AssertMPO       EQU     &a0 ; MPO goes low
NegateMPO       EQU     &b0 ; MPO goes high
ResetMPI        EQU     &c0 ; Reset MPI
DisableRxTx     EQU     &0A ; Disable the receiver and the transmitter
EnableRxTx      EQU     &05 ; Enable the receiver and the transmitter

; Values in the SR

SRRxReady       EQU     1 ; a character has been received
SRRxFull        EQU     2 ; Rx FIFO is full
SRTxReady       EQU     4 ; space in the Tx FIFO
SRTxEmpty       EQU     8 ; Tx FIFO is empty
SROverrunError  EQU     16
SRParityError   EQU     32
SRFrameError    EQU     64
SRReceivedBreak EQU     128

SRNastyError    EQU     SROverrunError + SRParityError + SRFrameError + SRReceivedBreak

; Values for ACR, the BaudRateGroup (BRG) bit is set at run time

MPOOutput       EQU     0 ; see data sheet
PowerDown       EQU     8 ; 0 for on, 8 for off
TimerSource     EQU     &70 ; see data sheet

ACRValue        EQU     MPOOutput + PowerDown + TimerSource

; Values for ISR and IMR

ISRTxReady      EQU     1 ; space in the Tx FIFO
ISRTxEmpty      EQU     2 ; Tx FIFO is empty
ISRRxReady      EQU     4 ; a character has been received
ISRRxFull       EQU     4 ; Rx FIFO is full
ISRBreak        EQU     8 ; change in break
ISRTimerTicked  EQU     16 ; timer has reached zero twice
MPIState        EQU     64 ; latched state of MPI
MPIChanged      EQU     128 ; MPI has changed state

IMRValue        EQU     ISRRxReady + ISRTimerTicked + ISRBreak

; Values for CTUR and CTLR

ClockFreq       EQU     3686400 ; 3.6864 MHz external clock
TimerFreq       EQU     ClockFreq / 16 / 2 ; the frequency of the timer
TimerTick       EQU     TimerFreq / 100 ; 100 times per sec


; This is a general purpose Reset for the serial chip, restoring
; everything to the defaults, and cancelling the current transmission
; and reception.  This routine doesn't touch r0, and other code
; depends on this.

ResetDriver     MOV     r3, #SerialChipBase

                MOV     r2, #DisableRxTx
                STR     r2, [r3, #CR]

                MOV     r2, #ResetMR1 ; make sure we write MR1
                STR     r2, [r3, #CR]
                MOV     r2, #MR1Value
                STR     r2, [r3, #MR1]
                MOV     r2, #MR2Value
                STR     r2, [r3, #MR2]

                MOV     r2, #Baud9600 ; default baud rate
                ORR     r2, r2, r2, LSL #4
                STR     r2, [r3, #CSR]

                TST     r2, #&80000000
                MOVEQ   r2, #ACRValue
                MOVNE   r2, #ACRValue + &80
                STR     r2, [r3, #ACR]

                MOV     r2, #TimerTick / 256
                STR     r2, [r3, #CTUR]
                MOV     r2, #TimerTick :MOD: 256
                STR     r2, [r3, #CTLR]

                MOV     r2, #ResetError
                STR     r2, [r3, #CR]
                MOV     r2, #ResetBreak
                STR     r2, [r3, #CR]
                MOV     r2, #ResetMPI
                STR     r2, [r3, #CR]
                MOV     r2, #ResetRx
                STR     r2, [r3, #CR]
                MOV     r2, #ResetTx
                STR     r2, [r3, #CR]

                MOV     r2, #StartTimer
                STR     r2, [r3, #CR]

                MOV     r2, #IMRValue
                STR     r2, [r3, #IMR]

                MOV     r2, #EnableRxTx
                STR     r2, [r3, #CR]

                MOV     pc, lr


; This routine sets the serial chip speed to a value corresponding to
; the value in r0: 1 = 9K6, 2 = 19K2, 3 = 38K4

DriverSpeed     MOV     r3, #SerialChipBase

01              LDR     r1, [r3, #SR] ; get the status register
                TST     r1, #SRTxEmpty ; finished sending ?
                BEQ     %B01

                CMP     r0, #0 ; really change it ?
                BLE     %F02

                MOV     r1, #DisableRxTx
                STR     r1, [r3, #CR]

                MOV     r1, #Baud9600 ; default baud rate
                CMP     r0, #2
                MOVEQ   r1, #Baud19200
                CMP     r0, #3
                MOVEQ   r1, #Baud38400

                ORR     r1, r1, r1, LSL #4
                STR     r1, [r3, #CSR]

                TST     r1, #&80000000
                MOVEQ   r1, #ACRValue
                MOVNE   r1, #ACRValue + &80
                STR     r1, [r3, #ACR]

                MOV     r1, #ResetRx
                STR     r1, [r3, #CR]
                MOV     r1, #ResetTx
                STR     r1, [r3, #CR]

                MOV     r1, #EnableRxTx
                STR     r1, [r3, #CR]

02              MOV     pc, lr

; this routine is the Serial chip interrupt handler. When an interrupt
; occurs, one of three things has happened
;       1) the timer has started another period
;       2) an error happened (character or break)
;       3) a character has been received

SerialInt       STR     r14, [r14,-r14] ; store lr at zero
                LDR     r14, =SavedRegs
                STMIA   r14!, {r0-r13}
                LDR     r0, [r0,-r0] ; get old r14 into r0
                SUB     r0, r0, #4 ; adjust ready for the return
                MRS     r1, SPSR ; get the old CPSR
                STMIA   r14!, {r0,r1} ; store them too
                LDR     r0, =ResetVectorCopy ; restore location zero
                LDR     r0, [r0]
                STR     r0, [r0,-r0] ; store it

                MOV     r3, #SerialChipBase
                LDR     r0, [r3, #ISR] ; get the interrupt status register
                TST     r0, #ISRTimerTicked ; Timer Interrupt ?
                BEQ     IntNoTick

                LDR     r2, =TimerVal
                LDR     r1, [r2] ; load the current value
                ADD     r1, r1, #1 ; increment it
                STR     r1, [r2] ; store it back
                MOV     r1, #ResetTimer ; clear the interrupt
                STR     r1, [r3, #CR]

IntNoTick       LDR     r0, [r3, #SR] ; get the status register
                TST     r0, #SRNastyError ; any errors ?
                BLNE    ROMReset ; Hard Reset
                
                MOV     r3, #SerialChipBase
                LDR     r0, [r3, #SR] ; get the status register
                TST     r0, #SRRxReady ; character arrived
                BLNE    NewMessage

                LDR     r14, =SavedRegs
                LDR     r0, =NextIntHandler ; pass the interrupt on ?
                LDR     r0, [r0] ; pass the interrupt on ?
                CMP     r0, #0
                BEQ     SerialResume

                STR     r0, [r14, #64] ; fake the PC
                LDMIA   r14!, {r0-r13,pc} ; pass it on

SerialResume    LDR     r0, [r14, #60] ; pick up the CPSR
                MSR     SPSR, r0 ; ready to restore
                LDMIA   r14, {r0-lr} ; restore registers
                MOVS    pc, lr ; and resume

; I've been interrupted because a character has been received on the
; serial line, so read it and pass it to the RDI handler, which should
; return to SerialInt. On entry r3 contains the serial chip base address

NewMessage      LDR     r0, [r3, #RHR] ; get the character (and clear int)
                AND     r0, r0, #&ff
                LDR     r1, =RDPHandlerV ; load the base address
                LDR     pc, [r1] ; call the handler


; Get the ticker's value

ReadTimer       LDR     r0, =TimerVal
                LDR     r0, [r0] ; load the current value
                MOV     pc, lr

; Return the next byte from the serial line

GetByte         MOV     r3, #SerialChipBase
GetByteLoop     LDR     r1, [r3, #ISR] ; get the interrupt status register
                TST     r1, #ISRTimerTicked ; timer interrupt ?
                BEQ     GetNoTick

                LDR     r2, =TimerVal
                LDR     r1, [r2] ; load the current value
                ADD     r1, r1, #1 ; increment it
                STR     r1, [r2] ; store it back
                MOV     r1, #ResetTimer ; clear the interrupt
                STR     r1, [r3, #CR]

GetNoTick       LDR     r1, [r3, #SR] ; get the status register
                TST     r1, #SRNastyError ; any errors seen ?
                BEQ     GetNoError
                BNE     ROMReset ; Hard Reset

GetNoError      TST     r1, #SRRxReady ; character arrived
                BEQ     GetByteLoop ; spin until it does
                LDR     r0, [r3, #RHR] ; get the character (and clear int)
                AND     r0, r0, #&ff
                ; notice that the TST cleared the N flag too
                MOV     pc, lr

; Send a byte to the serial line

PutByte         MOV     r3, #SerialChipBase
PutByteLoop     LDR     r1, [r3, #ISR] ; get the interrupt status register
                TST     r1, #ISRTimerTicked ; timer interrupt ?
                BEQ     PutNoTick

                LDR     r2, =TimerVal
                LDR     r1, [r2] ; load the current value
                ADD     r1, r1, #1 ; increment it
                STR     r1, [r2] ; store it back
                MOV     r1, #ResetTimer ; clear the interrupt
                STR     r1, [r3, #CR]

PutNoTick       LDR     r1, [r3, #SR] ; get the status register
                TST     r1, #SRNastyError ; any errors seen ?
                BEQ     PutNoError
                BNE     ROMReset ; Hard Reset

PutNoError      TST     r1, #SRTxReady ; space for the next character
                BEQ     PutByteLoop ; spin until there is
                STR     r0, [r3, #THR] ; put the character
                ; notice that the TST cleared the N flag too
                MOV     pc, lr


; Switch the LED off and on

SetLEDs         MOV     r3, #SerialChipBase
                TST     r0, #1
                MOVEQ   r0, #NegateMPO
                MOVNE   r0, #AssertMPO
                STR     r0, [r3, #CR]
                MOV     pc, lr

                END
