/***************************************************************************
 * armvirt.c
 * ARMulator II virtual memory interface.
 * Copyright (C) 1991 Advanced RISC Machines Limited.  All rights reserved.
 ***************************************************************************/

/*
 * RCS $Revision: 1.16 $
 * Checkin $Date: 93/11/25 15:59:22 $
 * Revising $Author: irickard $
 */

/* This file contains a complete ARMulator memory model, modelling a
"virtual memory" system.  A much simpler model can be found in
armfast.c, and that model goes faster too, but has a fixed amount of
memory.  This model's memory has 64K pages, allocated on demand from a
64K entry page table, when the page is first written.  Reads to an
unwritten (un-allocated) page are serviced by an access to a dummy page. 
Pages are never freed as they might be needed again.  A single area of
memory can be defined to generate aborts.  */

#include "armdefs.h"

#ifdef VALIDATE /* for running the validate suite */
#define TUBE 48 * 1024 * 1024 /* write a char on the screen */
#define ABORTS 1
#endif

#ifdef ABORTS /* the memory system will abort */
/* For the old test suite Abort between 32 Kbytes and 32 Mbytes
   For the new test suite Abort between 8 Mbytes and 26 Mbytes */
#define LOWABORT 32 * 1024
#define HIGHABORT 32 * 1024 * 1024
/* #define LOWABORT 8 * 1024 * 1024
#define HIGHABORT 26 * 1024 * 1024 */
#endif

#define TOP_OF_MEM 0x80000000   /* 2Gb to avoid stack-checking probs */
#define NUMPAGES 64 * 1024
#define PAGESIZE 64 * 1024
#define PAGEBITS 16
#define WORDOFFSETBITS 0xfffc
#define BYTEOFFSETBITS 0xffff
#define ENDSWAP(addr) (addr ^ 3)
static unsigned HostEndian ;

#define PAGETABLE ((unsigned char **)state->MemDataPtr)
#define DUMMYPAGE ((unsigned char *)state->MemSparePtr)

/***************************************************************************\
*                   Get a Word/Byte from Virtual Memory                     *
\***************************************************************************/

#define GetWord(state, address)                                         \
   *( (ARMword *)                                                       \
      ( *(PAGETABLE + (address >> PAGEBITS)) + (address & WORDOFFSETBITS) ) \
      )   

#define GetByte(state, address)                                         \
   *(                                                                   \
      ( *(PAGETABLE + (address >> PAGEBITS)) + (address & BYTEOFFSETBITS) ) \
      )   

/***************************************************************************\
*      Put a Word/Byte into Virtual Memory, maybe allocating the page       *
\***************************************************************************/

#define PutWord(state, address, data)                                   \
  {unsigned char *pageptr ;                                             \
   if ((pageptr = *(PAGETABLE + (address >> PAGEBITS))) == DUMMYPAGE)   \
      pageptr = AllocatePage(state,address) ;                           \
   *(ARMword *)(pageptr + (address & WORDOFFSETBITS)) = data ;          \
   }
   
#define PutByte(state, address, data)                                   \
  {unsigned char *pageptr ;                                             \
   if ((pageptr = *(PAGETABLE + (address >> PAGEBITS))) == DUMMYPAGE)   \
      pageptr = AllocatePage(state,address) ;                           \
   *(pageptr + (address & BYTEOFFSETBITS)) = data ;                             \
   }

/***************************************************************************\
*                    Allocate and return a memory page                      *
\***************************************************************************/

static unsigned char * AllocatePage(ARMul_State *state, ARMword address)
{unsigned char *pageptr ;

 pageptr = (unsigned char *)malloc(PAGESIZE) ;
 if (pageptr == NULL) {
    perror("ARMulator can't allocate VM page") ;
    exit(13) ;
    }
 *(PAGETABLE + (address >> PAGEBITS)) = pageptr ;
 return(pageptr) ;
 }

/***************************************************************************\
*                      Initialise the memory interface                      *
\***************************************************************************/

unsigned ARMul_MemoryInit(ARMul_State *state, unsigned long initmemsize)
{unsigned char **pagetable ;
 unsigned page ;
 unsigned char *dummypage ;

 if (initmemsize == 0)
   state->MemSize = TOP_OF_MEM; /* initialise to 4Gb if no size specified */
 else
   state->MemSize = initmemsize;
 if ((pagetable = (unsigned char **)malloc(sizeof(ARMword)*NUMPAGES))==NULL)
    return(FALSE) ;
 if ((dummypage = (unsigned char *)malloc(PAGESIZE))==NULL)
    return(FALSE) ;
 for (page = 0 ; page < NUMPAGES ; page++)
    *(pagetable + page) = dummypage ;
 state->MemDataPtr = (unsigned char *)pagetable ;
 state->MemSparePtr = (unsigned char *)dummypage ;
 *(ARMword *)dummypage = 1 ;
 HostEndian = (*dummypage != 1) ; /* 1 for big endian, 0 for little */
  *(ARMword *)dummypage = 0 ;
#ifdef BIGEND
 state->bigendSig = HIGH ;
#endif
#ifdef LITTLEEND
 state->bigendSig = LOW ;
#endif
 if (state->MemSize >= 10 * 1024 * 1024)
   ARMul_ConsolePrint(state, ", %dMbyte",state->MemSize/1024/1024);
 else
   ARMul_ConsolePrint(state, ", %dKbyte",state->MemSize/1024);
 return(TRUE) ;
}

/***************************************************************************\
*                         Remove the memory interface                       *
\***************************************************************************/

void ARMul_MemoryExit(ARMul_State *state)
{ARMword page ;
 unsigned char *pageptr ;

 for (page = 0 ; page < NUMPAGES ; page++) {
    pageptr = *(PAGETABLE + page) ;
    if (pageptr != DUMMYPAGE)
       free((unsigned char *)pageptr) ;
    }
 free((unsigned char *)DUMMYPAGE) ;
 free((unsigned char *)PAGETABLE) ;
 return ;
 }

/***************************************************************************\
*                   Load Instruction, Sequential Cycle                      *
\***************************************************************************/

ARMword ARMul_LoadInstrS(ARMul_State *state,ARMword address)
{state->NumScycles++ ;

#ifdef HOURGLASS_RATE
 if( ( state->NumScycles & HOURGLASS_RATE ) == 0 ) {
    armsd_hourglass();
    }
#endif

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_PREFETCHABORT(address) ;
    return(ARMul_ABORTWORD) ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 return(GetWord(state,address)) ;
}

/***************************************************************************\
*                 Load Instruction, Non Sequential Cycle                    *
\***************************************************************************/

ARMword ARMul_LoadInstrN(ARMul_State *state,ARMword address)
{state->NumNcycles++ ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_PREFETCHABORT(address) ;
    return(ARMul_ABORTWORD) ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 return(GetWord(state,address)) ;
 }

/***************************************************************************\
*                        Load Word, Sequential Cycle                        *
\***************************************************************************/

ARMword ARMul_LoadWordS(ARMul_State *state,ARMword address)
{state->NumScycles++ ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return(0) ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 return(GetWord(state,address)) ;
 }

/***************************************************************************\
*                      Load Word, Non Sequential Cycle                      *
\***************************************************************************/

ARMword ARMul_LoadWordN(ARMul_State *state,ARMword address)
{state->NumNcycles++ ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return(0) ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 return(GetWord(state,address)) ;
 }

/***************************************************************************\
*                     Load Byte, (Non Sequential Cycle)                     *
\***************************************************************************/

ARMword ARMul_LoadByte(ARMul_State *state,ARMword address)
{ARMword temp ;

 state->NumNcycles++ ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return(0) ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 temp = (HostEndian == state->bigendSig)?address:ENDSWAP(address) ;
 return((ARMword)GetByte(state,temp)) ;
 }

/***************************************************************************\
*                       Store Word, Sequential Cycle                        *
\***************************************************************************/

void ARMul_StoreWordS(ARMul_State *state,ARMword address, ARMword data)
{state->NumScycles++ ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 PutWord(state,address,data) ;
 }

/***************************************************************************\
*                       Store Word, Non Sequential Cycle                        *
\***************************************************************************/

void ARMul_StoreWordN(ARMul_State *state, ARMword address, ARMword data)
{state->NumNcycles++ ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 PutWord(state,address,data) ;
 }

/***************************************************************************\
*                    Store Byte, (Non Sequential Cycle)                     *
\***************************************************************************/

void ARMul_StoreByte(ARMul_State *state, ARMword address, ARMword data)
{ARMword temp ;

 state->NumNcycles++ ;

#ifdef VALIDATE
 if (address == TUBE) {
    if (data == 4)
       state->Emulate = FALSE ;
    else
       (void)putc((char)data,stderr) ; /* Write Char */
    return ;
    }
#endif

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 temp = (HostEndian == state->bigendSig)?address:ENDSWAP(address) ;
 PutByte(state,temp,(unsigned char)data) ;
 }

/***************************************************************************\
*                   Swap Word, (Two Non Sequential Cycles)                  *
\***************************************************************************/

ARMword ARMul_SwapWord(ARMul_State *state, ARMword address, ARMword data)
{ARMword temp ;

 state->NumNcycles++ ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return(ARMul_ABORTWORD) ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 temp = GetWord(state,address) ;
 state->NumNcycles++ ;
 PutWord(state,address,data) ;
 return(temp) ;
 }

/***************************************************************************\
*                   Swap Byte, (Two Non Sequential Cycles)                  *
\***************************************************************************/

ARMword ARMul_SwapByte(ARMul_State *state, ARMword address, ARMword data)
{ARMword temp ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return(ARMul_ABORTWORD) ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif
 temp = ARMul_LoadByte(state,address) ;
 ARMul_StoreByte(state,address,data) ;
 return(temp) ;
 }

/***************************************************************************\
*                             Count I Cycles                                *
\***************************************************************************/

void ARMul_Icycles(ARMul_State *state, unsigned number, ARMword address)
{state->NumIcycles += number ;
 ARMul_CLEARABORT ;
 }

/***************************************************************************\
*                             Count C Cycles                                *
\***************************************************************************/

void ARMul_Ccycles(ARMul_State *state, unsigned number, ARMword address)
{state->NumCcycles += number ;
 ARMul_CLEARABORT ;
 }

/***************************************************************************\
*                      Read Word (but don't tell anyone!)                   *
\***************************************************************************/

ARMword ARMul_ReadWord(ARMul_State *state, ARMword address)
{
#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return(ARMul_ABORTWORD) ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 return(GetWord(state,address)) ;
 }

/***************************************************************************\
*                      Read Byte (but don't tell anyone!)                   *
\***************************************************************************/

ARMword ARMul_ReadByte(ARMul_State *state, ARMword address)
{ARMword temp ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return(ARMul_ABORTWORD) ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 temp = (HostEndian == state->bigendSig)?address:ENDSWAP(address) ;
 return((ARMword)GetByte(state,temp)) ;
 }

/***************************************************************************\
*                     Write Word (but don't tell anyone!)                   *
\***************************************************************************/

void ARMul_WriteWord(ARMul_State *state, ARMword address, ARMword data)
{
#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 PutWord(state,address,data) ;
 }

/***************************************************************************\
*                     Write Byte (but don't tell anyone!)                   *
\***************************************************************************/

void ARMul_WriteByte(ARMul_State *state, ARMword address, ARMword data)
{ARMword temp ;

#ifdef ABORTS
 if (address >= LOWABORT && address < HIGHABORT) {
    ARMul_DATAABORT(address) ;
    return ;
    }
 else {
    ARMul_CLEARABORT ;
    }
#endif

 temp = (HostEndian == state->bigendSig)?address:ENDSWAP(address) ;
 PutByte(state,temp,(unsigned char)data) ;
 }
