//---------------------------------------------------------------------------
//
// %FILE     pcmtest.c
// %VSS-REV  $Revision: 12 $
// %CREATED  1995.04.30
// %REVISED  $Date: 4/18/97 4:16p $
// %AUTHOR   Noreen Bell
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF (B0+)
// %SUMMARY  PCMCIA test module
//     
// %VSS      $Author: Miked $ $Date: 4/18/97 4:16p $ $Revision: 12 $
//
// DESCRIPTION
//
//   Uses the PCMCIA module to perform a few basic tests of the PCMCIA
//   Controller, including interrupts.
//
// HISTORY
//
/*
 *
 * $History: pcmtest.c $
 * 
 * *****************  Version 12  *****************
 * User: Miked        Date: 4/18/97    Time: 4:16p
 * Updated in $/nsdemo
 * Improvements in interrupt handling.   New header (comment) changes.
 * 
 * *****************  Version 10  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 9  *****************
 * User: Miked        Date: 8/05/96    Time: 6:05p
 * Updated in $/nsdemo
 * Added RTTarget support (hooking of IRQ, #ifdef ONTIME).  Also
 * eliminated Watcom differences by using __cdecl for ISR functions.
 * 
 * *****************  Version 8  *****************
 * User: Miked        Date: 7/23/96    Time: 2:25p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 7/23/96    Time: 11:53a
 * Updated in $/nsdemo
 * Modified to work with ISR THUNK routines in CPU.ASM.  Interrupts now
 * vector to ISR Thunk routines, which in turn call the C ISRs.  This is a
 * more stable and consistent way to handle interrupts (vs. trying to
 * force C functions to properly
 * save and restore registers and do an IRETD).
 * 
 * *****************  Version 6  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 5  *****************
 * User: Miked        Date: 7/12/96    Time: 3:47p
 * Updated in $/nsdemo
 * Changed printed output slightly to fit in with rest of NSDEMO.
 * 
 * *****************  Version 4  *****************
 * User: Noreen       Date: 7/11/96    Time: 9:57a
 * Updated in $/nsdemo
 * Added definitions for setting and clearing of interrupts in general
 * control register.
 * 
 * *****************  Version 3  *****************
 * User: Miked        Date: 6/28/96    Time: 1:44p
 * Updated in $/nsdemo
 * Modified to call external assembly function for IRETD.  Also SSI and
 * Pharlap
 * now use same code.
 * 
 * *****************  Version 2  *****************
 * User: Noreen       Date: 5/07/96    Time: 4:00p
 * Updated in $/nsdemo
 * Reduced delay for removing PCMCIA card
 * 
 * *****************  Version 1  *****************
 * User: Noreen       Date: 5/07/96    Time: 3:31p
 * Created in $/nstest
 * Module created to call PCMCIA functions. 
 *
 */
//
// COPYRIGHT
//
//      (c) 1995, 1996, 1997 National Semiconductor Corporation
//
//---------------------------------------------------------------------------

#include "pcmtest.h"

//---------------------------------------------------------------------------

PRIVATE volatile BYTE pcmflag;  // to see if PCMCIA Interrupt happens

//---------------------------------------------------------------------------

void __cdecl ISR_PCM_THUNK(void);  // assembly ISR hooked to interrupt

//---------------------------------------------------------------------------

PRIVATE void ControlINT(USHORT value)
{
	IOW_BYTE(PCM_INDEX, PCM_GENERAL_CTRL);
	IOW_BYTE(PCM_DATA, value);
}

//---------------------------------------------------------------------------
//
// FUNCTION    PCM_Test
//
// INPUT       none
// OUTPUT      none
// RETURN      USHORT
//               SUCCESS - if card is detected and removal of card is detected.
//               FAIL - if the PCMCIA card is not found or if removal of card
//                      is not detected.
//
// DESCRIPTION
//  This function demonstrates use of the PCMCIA controller. Two simple tests
//  are included which test the basic functionality of the device. The first
//  test is based on accessing a PCMCIA Memory Card, verifying that the PCMCIA
//  controller is enabled and the mapping of the SRAM card into system memory
//  is correct. The second test is based on detecting a change in the status
//  of the PCMCIA card. An interrupt is generated in software based on the
//  removal of the PCMCIA card. This test verifies that the PCMCIA Controller
//  can generate an interrupt when a PCMCIA status change takes place.
//  Interrupts can be configured in a similar manner to detect status events
//  such as low battery warnings and write-protect switch position change.
//
//---------------------------------------------------------------------------

USHORT PCM_Test(void)
{
	PCM_STATUS Status;
	int i;
    
	//Pointer to PCMCIA memory window
	ULONG * pPCM_memory = (ULONG *) WINDOW_START;

	//Pointer to Status structure
	PCM_STATUS * pStatus = &Status;

	#ifdef PHARLAP
	  USHORT irqvec;
	  FARPTR old_handler;
	  FARPTR new_handler;
	#endif

	// Hook the PCMCIA interrupt
	#ifdef PHARLAP
	  if (IRQ_PCM < 8)
	      irqvec = IRQIV_Controller1 + IRQ_PCM;
	  else
	    irqvec = IRQIV_Controller2 + IRQ_PCM - 8;

	  FP_SET(new_handler, ISR_PCM_THUNK, 0x18); // set up the "far pointer"
	  _dx_pmiv_get(irqvec, &old_handler);       // get the old handler
	  _dx_pmiv_set(irqvec, new_handler);        // set the new handler
	#endif
	
	#ifdef ONTIME
	   RTSetVector(RTIRQ0Vector+IRQ_PCM, (void (*)(void)) ISR_PCM_THUNK);
	#endif

	// Enable PCMCIA
	dprintf("PCMCIA Controller:\r\n");
	if(PCMCIA_Enable() == SUCCESS)
	{
	  dprintf("  Card Detected\r\n");
	  if((*pPCM_memory & 0x0000ffff) == PCM_Signature)
	      dprintf("  NS486SXF Signature found\r\n");
	}
	else
	{
	  dprintf("  Card NOT Detected\r\n");
	  dprintf("\r\n");
	  return FAIL;
	}

	// PCMCIA Card Detect Interrupt Test
	PIC_Enable(IRQ_PCM);
    PIT_Delay(1,50);                   //50ms delay
	pcmflag = FALSE;
	dprintf("  Card Detect Interrupt Test:\r\n");
	dprintf("    Remove PCMCIA Card");

	//Implement delay dropping out of
	//loop when pcmflag is set
	for(i=0; i<50; i++)
	{
	    if(pcmflag == FALSE)
	    {
	       PIT_Delay(1,70);                   //70ns delay
	       dprintf(".");
	       PCMCIA_Status(pStatus);
	       if(pStatus->CardChange == TRUE)
		     ControlINT(PCM_SETINT);          //Generate interrupt
	    }
	}

    // Disable interrupt
    PIC_Disable( IRQ_PCM );

    // Restore the old handler
  
    #ifdef PHARLAP
      _dx_pmiv_set(irqvec, old_handler);
    #endif

	if(pcmflag == TRUE)
	{
	   dprintf("\r\n    Card Change Detected\r\n");
	   dprintf("\r\n");
	   return SUCCESS;
	}
	else
	{
	   dprintf("\r\n    Card Change NOT Detected\r\n");
	   dprintf("\r\n");
	   return FAIL;
	}

}

//---------------------------------------------------------------------------

// This is the ISR for the PCMCIA interrupt.  The assembly function 
// ISR_PCM_THUNK is the actual function pointed to by the IDT.  That
// function calls this function.

void __cdecl ISR_PCM(void)
{

	pcmflag = TRUE;

	// Clear Interrupt
	ControlINT(PCM_CLEARINT);

	// Issue EOI
	// Note in Pharlap ETS lite, using Borland, we might not have a stack
	// here. Not yet tested. If so, issue EOI directly instead of
	// calling PIC_EOI.

	PIC_EOI(IRQ_PCM);

}


//--------------------------------------------------------------------------
// END       pcmtest.c
//---------------------------------------------------------------------------
