//---------------------------------------------------------------------------
//
// %FILE     lcdtest.c
// %VSS-REV  $Revision: 7 $
// %CREATED  1996.06.26
// %REVISED  $Date: 4/18/97 4:07p $ 
// %AUTHOR   Michael C. Draeger 
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF
// %SUMMARY  LCD Test Code
//     
// %VSS      $Author: Miked $ $Date: 4/18/97 4:07p $ $Revision: 7 $
//
// DESCRIPTION
//
//   Put an image on the LCD screen.  Requires external hardware, but will
//   not damage anything if the external hardware is not present.
//
//   This code is fairly straightforward.  Please see the README.TXT file
//   for information on the LCD hardware required to see the output of this
//   code.
//
//   There is now a detailed LCD application note titled "Using an LCD with the
//   NS486SXF Evaluation Board".  Please contact ns486@arador.nsc.com for
//   this document.
//
// HISTORY
//
/*
 *
 * $History: lcdtest.c $
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 4/18/97    Time: 4:07p
 * Updated in $/nsdemo
 * Moved #defines to LCDTEST.H file.   New header (comment) changes.
 * 
 * *****************  Version 6  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 5  *****************
 * User: Miked        Date: 7/23/96    Time: 2:25p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 4  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 3  *****************
 * User: Miked        Date: 7/15/96    Time: 4:46p
 * Updated in $/nsdemo
 * Updated LCD code for Digi-Key LCD panel (and external hardware) to be
 * described to customers.
 * 
 * *****************  Version 2  *****************
 * User: Miked        Date: 7/12/96    Time: 3:49p
 * Updated in $/nsdemo
 * Changed LCD delay 5 second.
 * 
 * *****************  Version 1  *****************
 * User: Miked        Date: 6/27/96    Time: 11:42a
 * Created in $/nsdemo
 * Initial LCD test code.  Works, but needs to be cleaned up.
 *
 */
//
// COPYRIGHT
//
//      (c) 1996, 1997 National Semiconductor Corporation
//
// NOTES
//
// OUT1/OUT2 signal use to enable LCD hardware
//
//   The use of the OUT1 and OUT2 signals to sequence the LCD signals
//   is somewhat generic.  The LCD panel described in the README.TXT
//   file only requires one signal, a display enable.  This signal goes
//   active when OUT1 and OUT2 are both active.
//
//   Since the UART OUT2 signal is used to enable the UART interrupt in
//   PCs and Super I/O devices, it could be set prior to getting here.
//   OUT1 is not typically used, so is likely to be inactive.  The
//   use of both signals if a stab at providing some protection.  When
//   both of these signals go active, the display will be enabled.
//   The use of the UART OUT signals eliminated the need to put a register
//   on the external hardware to enable the LCD.  In actual applications,
//   more protection logic should be used, as specified by the LCD panel
//   makers, to prevent damage to the LCD.  For most LCD's, having the
//   display enabled but not having LCD clock signals can damage the
//   panel.
//
//   In this code, the LCD enable sequence is as follows:
//
//     (1) set OUT1 active
//     (2) start LCD signals
//     (3) set OUT2 active
//
//   The LCD disable sequence is the reverse of this.  For the panel
//   this code is intended for, step (1) does do anything except allow
//   step (3) to enable the display.  The reason that this step was left in
//   is to support other LCD panels with fewer software changes.  An LCD
//   panel without a display enable may require sequencing of actual
//   power signals to the LCD.  One example would be that the VDD voltage
//   needs to be applied, and soon thereafter the LCD signals started and
//   then a VEE voltage applied.  In this case step (1) could be used
//   to apply VDD to the panel, and step (3) to apply VEE.  Another
//   variation would have the LCD signals starting, followed by the
//   VEE signal, followed by a Display enable.  This case would require
//   some modification to the code.  For any panel other than the one
//   described in README.TXT, consult the panel data sheet and application
//   notes for power sequencing requirements.
//
// LCD frequency calculation
//
//   A useful equation to calculate the refresh frequence of the LCD
//   controller:
//
//      Frame Time (s) = (Div/OSC_CLK) * ( (Pixels/4) + (Rows * Offset) )
//
//      Refresh Frequency (Hz) = 1/Frame Time
//
//   Where:   
//
//     Div = clock dividor programmed into the LCD controller
//           in the LCD COnfiguration Register 2.  A Binary and
//           Integer divisor is programmed.  Div is the total
//           divisor.  So for a Binary Divisor of 4 and an Integer Divisor
//           of 9, Div is 36.
//
//     OSC_CLK = Oscillator speed of NS486SXF.  (I.e., 50MHz when the CPU
//               is operating at 25MHz)
//
//     Pixels = number of pixels per row * number of pixels per column
//
//     Rows = number of pixels per row
//
//     Offset = number of offsets programmed in the LCD Configuration
//              Register 3.  This value is from 1 to 16.  The offset
//              occurs at the end of each row, so this is used to
//              account for the time when the CL2 clock is not not
//              running at the end of each row.  (See the Datasheet)
//
//   For example, when the NS486SXF is running at 25MHz with a divisor
//   of 36 and 1 offset:
//
//     Frame Time = (36/50,000,000) * ( (320x240/4) + (240 * 1) )
//
//     Frame Time = 0.0139968 seconds
//
//     Refresh Frequency = 71.4 Hz.
//
// Frequency used in this demo code
//
//   A refresh frequency of 71.4 Hz is used in this code (when at 25 MHz
//   CPU operation).  This frequency provides a good display on the LCD
//   panel described in the README.TXT file.  There is a little bit of
//   flicker in the grey colors, which could be reduced by better tuning
//   the GLUT to the panel.  It is fairly minimal however.  Also note,
//   the panel in the README.TXT can operate at a much higher frequency.
//   With a divisor of 20, the LCD will run at 128Hz.  This is so fast that
//   no grey scale flicker will be seen.  This will consume much more of
//   the NS486SXF memory bandwidth, so this demo uses 71Hz which is more
//   reasonable if the NS486SXF were used for other purposes than a
//   standalone LCD controller demo.
//
//---------------------------------------------------------------------------

#include "lcdtest.h"

//---------------------------------------------------------------------------

USHORT LCD_Test( void )
{

  // Locals
  int i;                        // used for a delay while the LCD is running
  BYTE tByte;                   // temporary variable
  BYTE * lcd_address;           // set to address of LCD buffer
  DMA_TRANSFER Transfer;        // structure to program DMA
  
  // Start of test message
  dprintf("LCD Controller Test:\r\n");

  // get address of lcd buffer
  lcd_address = (BYTE *) &lcd_buffer;
  dprintf("  LCD buffer at address 0x%lX\r\n", lcd_address);
    
  // Set up DMA Controller for LCD

  Transfer.Mode = TransferMode_Demand;          // demand mode
  Transfer.TC_EOP = DMA_EOP;                    // LCD uses EOP pin
  Transfer.IO_MEM = DMA_IO;                     // requestor is IO
  Transfer.Direction = DMA_Write;               // write (to IO)
  Transfer.AutoInit = DMA_Enable;               // enable autoinitialization
  Transfer.Channel = 1;                         // on channel 1
  Transfer.RequestorDirection = DMA_Increment;  // increment (N/A)
  Transfer.TargetDirection = DMA_Increment;     // increment
  Transfer.Clocks = Clocks2;                    // 2 clocks per transfer
  Transfer.BusSize = DMA_bit16;                 // 16-bit data transfer
  Transfer.CPUDMARequest = DMA_NoAction;        // CPU DMA request not req.
  Transfer.TargetAddress = (ULONG) lcd_address; // address of buffer
  Transfer.TransferByteCount = LCD_BUFSIZE;     // size of buffer
  DMA_Transfer(Transfer);

  // Configure LCD

  IOW_BYTE ( LCD_CFG1,        0x46 ); // interrupts disabled, normal video
                                        // blink disabled
  
  // Set up 320x240x2
  // Divisor is 28 @20MHz (73.4Hz LCD refresh rate)
  //            36 @25MHz (71.4Hz LCD refresh rate)

  if ( gTargetInfo.valid == TRUE )
  {  
    switch( gTargetInfo.cpu_speed )
    {
    case 20:
      IOW_BYTE ( LCD_CFG2,        0x99 ); // 320x240x2, div 28
      break;
    case 25:
    default:
      IOW_BYTE ( LCD_CFG2,        0xA1 ); // 320x240x2, div 36
      break;
    }
  }
  else
    IOW_BYTE ( LCD_CFG2,        0xA1 ); // 320x240x2, div 36
  
  
  IOW_BYTE ( LCD_CFG3,        0x10 ); // 1/4 fifo, no offset
  IOW_BYTE ( LCD_CFG4,        0x0E ); // disable LCD, glut size 16 (words)
  
  dprintf("  LCD image on for 5 seconds\r\n");

// Make sure OUT1 and OUT2 signals are inactive (high)  
// This makes sure that the signals are sequenced properly, as chances are
// that OUT2 is active do to other software programming the UART
// (OUT2 is the interrupt enable bit on PC compatibles and many UARTs
// in multifunction chips.)

  tByte = IOR_BYTE ( DEBUG_UART_MCR );
  tByte &= ~OUT1;
  tByte &= ~OUT2;
  IOW_BYTE ( DEBUG_UART_MCR, tByte );

// LCD Enable step 1 - enable OUT1
//  
// on the LCD panel desribed in README.TXT, this step does not start
// or enable any signals.  See the Notes section above.  The OUT1 signal
// must be set active, however, for the second step, which is below.

  tByte = IOR_BYTE ( DEBUG_UART_MCR );
  tByte |= OUT1;
  IOW_BYTE ( DEBUG_UART_MCR, tByte );

// 1 ms delay

  PIT_Delay( 1, 1 );                   

// Start LCD signals

  tByte = IOR_BYTE ( LCD_CFG4 );
  tByte |= 0x01;
  IOW_BYTE ( LCD_CFG4, tByte );  // enable LCD       

// 1 ms delay

  PIT_Delay( 1, 1 );                   

// LCD Enable step 2 - enable OUT2
//  
// on the LCD panel described in README.TXT, this sets the Display Enable
// signal active.

  tByte = IOR_BYTE ( DEBUG_UART_MCR );
  tByte |= OUT2;
  IOW_BYTE ( DEBUG_UART_MCR, tByte );
  
// *** LCD IS RUNNING *** - do five second delay
  
for ( i=0; i < 100; i++ )
  PIT_Delay( 1, 50 );                   
  
// LCD Disable step 1 - disable OUT2
  
  tByte = IOR_BYTE ( DEBUG_UART_MCR );
  tByte &= ~OUT2;
  IOW_BYTE ( DEBUG_UART_MCR, tByte );

// 1 ms delay

  PIT_Delay( 1, 1 );                   

// Disable LCD controller

  IOW_BYTE ( LCD_CFG4,        0x0C ); // disable LCD, glut size 12

// 1 ms delay

  PIT_Delay( 1, 1 );                   

// LCD Disable step 2 - disable OUT1
  
  tByte = IOR_BYTE ( DEBUG_UART_MCR );
  tByte &= ~OUT1;
  IOW_BYTE ( DEBUG_UART_MCR, tByte );
  
  dprintf("  LCD image off\r\n");

  // End of test message
  dprintf("\r\n");

  // All Done
  return SUCCESS;

}

//---------------------------------------------------------------------------
// END       lcdtest.c
//---------------------------------------------------------------------------
