//---------------------------------------------------------------------------
//
// %FILE     ica.c
// %VSS-REV  $Revision: 7 $
// %CREATED  1996.04.11
// %REVISED  $Date: 4/18/97 4:05p $
// %AUTHOR   Michael C. Draeger 
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF, NS486SXL
// %SUMMARY  Reads the Target Loader ICA area
//     
// %VSS      $Author: Miked $ $Date: 4/18/97 4:05p $ $Revision: 7 $
//
// DESCRIPTION
//
//   This module reads the ICA area set up by the Target Loader.  If this
//   area points to a valid Target Info structure, this module will
//   read this structure as well.
//
// INPUTS
//
// HISTORY
//
/*
 *
 * $History: ica.c $ 
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 4/18/97    Time: 4:05p
 * Updated in $/nsdemo
 *  New header (comment) changes.
 * 
 * *****************  Version 6  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 5  *****************
 * User: Miked        Date: 7/23/96    Time: 2:25p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 4  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 3  *****************
 * User: Miked        Date: 5/03/96    Time: 2:06p
 * Updated in $/nsdemo
 * Added ICA sanity checks for overwritten values.
 * 
 * *****************  Version 2  *****************
 * User: Miked        Date: 4/18/96    Time: 11:09a
 * Updated in $/nsdemo
 * Clean up for release.
 * 
 * *****************  Version 1  *****************
 * User: Miked        Date: 4/11/96    Time: 3:26p
 * Created in $/board test
 * code to read ICA Target Info structure
 *
 */
//
// COPYRIGHT
//
//      (c) 1996, 1997 National Semiconductor Corporation
//
//---------------------------------------------------------------------------

#include "ica.h"

//---------------------------------------------------------------------------

// INPUT        ICA_TARGET_INFO * - points to structure for output
// OUTPUT       ICA_TARGET_INFO * - structure is filled in if return
//                                  equals SUCCESS, otherwise untouched
//                                  the return value is not SUCCESS
// RETURN       USHORT - FAIL if ICA is not valid, is not OK, does not
//                            point to Target Info structure, or the size
//                            pointed to is too small
//                       INVALID_PARAMETER is target_info is NULL
//                       SUCCESS if Target Info read
//
// DESCRIPTION  This routine scans the ICA area.  If this area is valid
//              and points to a Target Info structure, it will fill in a
//              structure with the target info

USHORT ICA_Target_Info_Read( ICA_TARGET_INFO * target_info )
{

  // locals

  ICA_TABLE * table;
  ICA_TARGET_INFO * temp_target_info;

  // check parameters

  if ( target_info == NULL )
    return INVALID_PARAMETER;

  // point to ICA
  
  table = (ICA_TABLE *) ICA_TBL_ADDR;
  
  // see if ICA is valid

  if ( table->valid != ICA_VALID )
    return FAIL;

  // see if ICA status is OK

  if ( table->status != ICA_STATUS_OK )
    return FAIL;

  // see if ICA command is Target Info

  if ( table->command != ICA_CMD_TARGET_INFO )
    return FAIL;

  // make sure Target Info size is at least as big as we expect

  if ( table->data_size < sizeof ( temp_target_info ) )
    return FAIL;

  // point temp_target_info

  temp_target_info = (ICA_TARGET_INFO *) table->data_addr;

  // copy information to correct structure

  target_info->boot_status  = temp_target_info->boot_status;
  target_info->original_eax = temp_target_info->original_eax; 
  target_info->original_edx = temp_target_info->original_edx;
  target_info->cpu_speed    = temp_target_info->cpu_speed;    
  target_info->ram_size     = temp_target_info->ram_size;     

  // do more sanity checks to see if data is invalid
  // we were getting a valid ICA structure but the target info
  // structure had been mucked up.  these checks will invalidate the
  // structure if it looks like they may have been stepped on

  // boot status should be zero
  if ( target_info->boot_status != 0 )
    return FAIL;
  
  // original EAX should be zero
  if ( target_info->original_eax != 0 )
    return FAIL;

  // CPU speed should be 20MHz or 25MHz.  but allow a fair about on
  // slack on this one (if the others have not been stepped on, and this
  // one is in this range, chances are it is OK)
  if ( ( target_info->cpu_speed < (15728640L) ) ||   // less than 15MHz 
       ( target_info->cpu_speed > (34603008L) ) )    // greater than 33MHz
    return FAIL;
  
  // DRAM size on eval board should be 1, 4, or 16MB
  if ( ! ( ( target_info->ram_size == ( 1048576L) ) ||  // 1MB
           ( target_info->ram_size == ( 4194403L) ) ||  // 4MB
           ( target_info->ram_size == (16777216L) ) ) ) // 16MB
    return FAIL;

  // thats it

  return SUCCESS;

}

//---------------------------------------------------------------------------
// END       ica.c
//---------------------------------------------------------------------------
