//---------------------------------------------------------------------------
//
// %FILE     i2c.c
// %VSS-REV  $Revision: 5 $
// %CREATED  1997.03.04
// %REVISED  $Date: 4/16/97 3:04p $
// %AUTHOR   Rhinda Bal-Firpo 
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF, NS486SXL 
// %SUMMARY  i2c (Access.bus) module 
//     
// %VSS      $Author: Rhinda $ $Date: 4/16/97 3:04p $ $Revision: 5 $
//
// DESCRIPTION
//
//   This file contains code that will initialize the i2c and perform read and 
//   writes from the i2c.
//
//   I2C and Access.bus are synonymous.
//
// HISTORY
//
/*
 *
 * $History: I2C.C $ 
 * 
 * *****************  Version 5  *****************
 * User: Rhinda       Date: 4/16/97    Time: 3:04p
 * Updated in $/nsdemo
 * needed it to do some testing
 * 
 * *****************  Version 4  *****************
 * User: Miked        Date: 4/11/97    Time: 6:27p
 * Updated in $/nsdemo
 * Formatting changes.  Also fixed usage of I2C_SET_BUSY and I2C_BUSY.
 * 
 * *****************  Version 3  *****************
 * User: Rhinda       Date: 4/01/97    Time: 6:45p
 * Updated in $/nstest
 * maintenance changes
 * 
 * *****************  Version 2  *****************
 * User: Rhinda       Date: 3/21/97    Time: 4:03p
 * Updated in $/nsdemo
 * Made many changes to initial I2C code.
 *
 */
//
// COPYRIGHT
//
//      (c) 1997 National Semiconductor Corporation
//
// NOTES
//
//   In this file the i2c (Access.bus) is in the master transmit and master 
//   receive mode.  The slave used to test this file is the LM75 Digital 
//   Temperature Sensor.  Data is transmitted to and received from the LM75.
//   Please see the application note for more information.
// 
//---------------------------------------------------------------------------

#include "i2c.h"

//---------------------------------------------------------------------
//
// FUNCTION     I2c_Initialize()
//
// INPUT        none
// OUTPUT       none
// RETURN       USHORT
//                FAIL - If bus is unable to initialize
//                SUCCESS - if successfull
// DESCRIPTION
//
//      This function is called to initialize the i2c interface.
//
//--------------------------------------------------------------------

USHORT  I2C_Initialize()
{
  
// initialize the MicroWire
  if ((IOR_BYTE(BIU_CONTROL1) & 0x80) != 0x80)
    return FAIL;

// configure the Access.Bus; I2C
  IOW_BYTE( TWI_CONTROL, 0x9f );  // selects access.bus and does software reset 
  IOW_BYTE( TWI_CONTROL, 0x9e );  // selects access.bus and enables 
  IOW_BYTE( TWI_A_CTRL2, 0x03 );  // set the OSCX1 Clock periods
    
// all done 
  return SUCCESS;
      
}

//---------------------------------------------------------------------
//
// FUNCTION     I2c_Data_Read(int reg_read, int numberbytes, BYTE *data_read)
//
// INPUT        BYTE SlaveAddress:  I2C address of device to read
//              int reg_read:  address of register that will be read.
//              int numberbytes: number of bytes that will read; this is not 
//                               being used currently; currently we are reading
//                               2 bytes.
//
// OUTPUT       BYTE *data_read:  pointer to the array that will hold the data 
//                                read
//
// RETURN       USHORT
//                FAIL - If unable to read data
//                SUCCESS - if successful
//                
// DESCRIPTION
//
//      This function is called to read data from I2C device.
//
//--------------------------------------------------------------------

USHORT I2C_Data_Read(BYTE SlaveAddress, int reg_read, int numberbytes, 
                     BYTE *data_read)
{

  BYTE busy;        //reading of the register is stored in this
  int i = 0;        //counter for the while loops
    
  //set the access.bus to read the slave
  //write the address byte, and set the access.bus for writing to the pointer 
  //address

  IOW_BYTE (TWI_SDA,  (SlaveAddress | I2C_SLAVE_WRITE));
   
  //If the access.bus is a multiple master or slave, it might be busy with other 
  //check if access.bus is busy with other  transmits or receives. 
  
  busy = IOR_BYTE(TWI_A_STATUS);
  while (((busy & I2C_BB) == I2C_BB) && (i < 10000))
  {
    i++;
	busy = IOR_BYTE(TWI_A_STATUS);
  }

  //This should only happen if the i2c is a multiple master or slave bus
  if ((busy & I2C_BB) == I2C_BB)
	return FAIL;

  //setup access.bus control register for initial byte transfer
  // set master mode, set BUSY and ACK
  IOW_BYTE (TWI_A_CONTROL,  ((I2C_MASTRQ | I2C_SET_BUSY) | I2C_ACK) );
  
  //wait for transfer to finish; check the control register 
  i=0;
  busy = IOR_BYTE(TWI_A_CONTROL);
  while(((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
         ( I2C_BUSY | I2C_ABINT | I2C_ACK)) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_CONTROL);
  }
    
  //The transfer did not complete
  if ((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
      ( I2C_BUSY | I2C_ABINT | I2C_ACK))
	return FAIL;

  //Set the pointer register to point to the register to read in the slave
  IOW_BYTE(TWI_SDA, reg_read);
  IOW_BYTE(TWI_A_CONTROL, I2C_SET_BUSY);   //set BUSY

  //wait for transfer to finish; check the control register 
  i=0;
  busy = IOR_BYTE(TWI_A_CONTROL);
  while(((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
         ( I2C_BUSY | I2C_ABINT | I2C_ACK)) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_CONTROL);
  }
    
  //The transfer did not complete
  if ((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
      ( I2C_BUSY | I2C_ABINT | I2C_ACK))
    return FAIL;

  //setup the i2c to read the data
    //repeat start; set BUSY and MASTRQ, 0X12
  IOW_BYTE(TWI_A_CONTROL, (I2C_MASTRQ | I2C_SET_BUSY));  
    //set the address of the slave and set for read
  IOW_BYTE(TWI_SDA, (SlaveAddress | I2C_SLAVE_READ));

  //wait for transfer to finish; check the control register 
  i=0;
  busy = IOR_BYTE(TWI_A_CONTROL);
  while(((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
         ( I2C_BUSY | I2C_ABINT | I2C_ACK)) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_CONTROL);
  }
   
  //The transfer did not complete
  if ((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
      ( I2C_BUSY | I2C_ABINT | I2C_ACK))
    return FAIL;

  //program to receive data
  IOW_BYTE(TWI_A_CONTROL, (I2C_SET_BUSY | I2C_ACK));  //set BUSY and ACK

  i = 0;
  busy = IOR_BYTE(TWI_A_CONTROL);
  while(((busy & I2C_ABINT) != I2C_ABINT) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_CONTROL);
  }

  //The data was not received by i2c
  if ((busy & I2C_ABINT) != I2C_ABINT)
	return FAIL;
    
  //Read the slave temperature, currently pointing to the temperature register
  data_read[0] = IOR_BYTE(TWI_SDA); 
   
  //Generate STOP condition before reading the last byte, and set for NO ACK
  IOW_BYTE (TWI_A_CONTROL, (I2C_SET_BUSY | I2C_STOP));

  //wait for receive to finish
  i = 0;
  busy = IOR_BYTE(TWI_A_CONTROL);
  while(((busy & I2C_SET_BUSY) == I2C_SET_BUSY) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_CONTROL);
  }

  //Read the last byte of temperature data from slave
  data_read[1] = IOR_BYTE(TWI_SDA);

  //Transfer is finished
  return SUCCESS;
  
}

//---------------------------------------------------------------------
//
// FUNCTION     I2c_Data_Write(int reg_write, int bytes_write, BYTE *data_write)
//
// INPUT        BYTE SlaveAddress:  I2C address of device to read
//              int reg_write:  register to write to        
//              int bytes_write: number of bytes to write
//              BYTE *data_write: pointer to the array holding the data to send 
//                                to the slave
//
// OUTPUT       none
//
// RETURN       USHORT
//                FAIL - If unable to read data from slave
//                SUCCESS - successful
// DESCRIPTION
//
//      This function is called to send data to the slave
//
//--------------------------------------------------------------------

USHORT I2C_Data_Write(BYTE SlaveAddress, int reg_write, int bytes_write, 
                      BYTE *data_write)
{

  BYTE busy;        //reading of the register is stored in this
  int i = 0;        //counter for the while loops

  //set the access.bus to write the slave
    //write the address byte, and set the R/W bit for write
  IOW_BYTE (TWI_SDA,  (SlaveAddress | I2C_SLAVE_WRITE));
  

  //If the access.bus is a multiple master or slave, it might be busy with other 
  //check if access.bus is busy with other transmits or receives. 
  busy = IOR_BYTE(TWI_A_STATUS);
  while (((busy & I2C_BB) == I2C_BB) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_STATUS);
  }

  //This should only happen if the i2c is a multiple master or slave bus
  if ((busy &I2C_BB) == I2C_BB)
	return FAIL;

  //setup access.bus control register for initial byte
    // set master mode, set BUSY and ACK
  IOW_BYTE (TWI_A_CONTROL, ((I2C_MASTRQ | I2C_SET_BUSY) | I2C_ACK)); 
  
  //wait for transfer to finish; check the control register 
  i=0;
  busy = IOR_BYTE(TWI_A_CONTROL);
  while(((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
         ( I2C_BUSY | I2C_ABINT | I2C_ACK)) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_CONTROL);
  }

  //The transfer did not complete
  if ((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
      ( I2C_BUSY | I2C_ABINT | I2C_ACK))
	return FAIL;

  //program to write data
    //set the pointer register to point to the needed register
  IOW_BYTE(TWI_SDA, reg_write);
    //  set BUSY and ACK
  IOW_BYTE (TWI_A_CONTROL, I2C_SET_BUSY);

  //check the ABINT bit and ACK bit to see if transfer completed
  i = 0;
  busy = IOR_BYTE(TWI_A_CONTROL);
  while(((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
         ( I2C_BUSY | I2C_ABINT | I2C_ACK)) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_CONTROL);
  }

  //Access.bus is not ready to write data
  if ((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
      ( I2C_BUSY | I2C_ABINT | I2C_ACK))
	return FAIL;
    
  //write the data to the appropriate register
  IOW_BYTE( TWI_SDA, data_write[0] ); 
  IOW_BYTE (TWI_A_CONTROL, I2C_SET_BUSY); //BUSY and ACK

  //check the ABINT bit and ACK bit to see if transfer completed
  i = 0;
  busy = IOR_BYTE(TWI_A_CONTROL);
  while(((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
         ( I2C_BUSY | I2C_ABINT | I2C_ACK)) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_CONTROL);
  }

  //Access.bus is not ready to write data
  if ((busy & ( I2C_SET_BUSY | I2C_ABINT | I2C_ACK)) != 
      ( I2C_BUSY | I2C_ABINT | I2C_ACK))
	return FAIL;
     
  //Generate STOP condition before sending the last byte
  IOW_BYTE( TWI_SDA, data_write[1] ); 
  IOW_BYTE (TWI_A_CONTROL, (I2C_SET_BUSY | I2C_STOP));

  //wait for the write to finish
  i = 0;
  busy = IOR_BYTE(TWI_A_CONTROL);
  while(((busy & I2C_SET_BUSY) == I2C_SET_BUSY) && (i < 10000))
  {
	i++;
	busy = IOR_BYTE(TWI_A_CONTROL);
  }

  //Transfer is finished
   return SUCCESS;

}

//---------------------------------------------------------------------------
// END       i2c.c
//---------------------------------------------------------------------------
