;---------------------------------------------------------------------------
;
; %FILE     boiler.asm
; %VSS-REV  $Revision: 3 $
; %VERSION  1.6
; %CREATED  1995.07.31
; %REVISED  $Date: 11/26/96 3:08p $
; %AUTHOR   Michael C. Draeger 
; %PROJECT  NS486SXF evaluation board software
; %PART     NS486SXF 
; %SUMMARY  sample assembly program for NS486SXF eval board
;     
; %VSS      $Author: Miked $ $Date: 11/26/96 3:08p $ $Revision: 3 $
;
; DESCRIPTION
;
;   This is an assembly language program that is loaded into flash
;   on the ns486sxf evaluation board at address 0xffff0000.  It performs
;   basic ns486sxf and board setup, sets up protected mode descriptors,
;   and jumps to the label "user_start".  This label is in the sample 
;   module "INT.ASM" which shows software interrupt handling.
;
;   This should be loaded with the NS486SXF eval board flash loader
;   utility with the "BOILER.FLC" script, which downloads the file 
;   "BOILER.BIN", which contains this code and the sample "INT" code.
;
;   This is intended as an example of configuring the evaluation board
;   after being jumped to from the target loader.  See the "README.TXT"
;   file for more information.
;
;   There is only one segment in this file.  It is used for both code
;   and data.  Most data is copied into RAM for use.  This file is setup
;   like this so that it can generate a ".COM" file, which is a pure
;   binary with no relocation info.  This negates the need for a locater
;   to locate the code in memory.  This also means that it would be
;   difficult to use C (or other high level code) with this code.
;
;   Please note: there are some hardcoded addresses in the file.  For
;   examples, the GDT and IDT entries contain physical addresses that
;   are used for this demo.  Some addresses can be changed by simply
;   changing defines in "BOILER.INC", but some addresses would need
;   to be changed in this file if you wanted to relocate this test or
;   change the memory layout (GDT, IDT, stack, ISR, etc.)
;
; RELEASE HISTORY
;
;   1.0   1995.08.01  mcd  genesis
;   1.1   1996.01.10  mcd  evaluation board version
;   1.2   1996.01.19  mcd  changed DRAM cycle miss to 3 and
;                          turned off extended CAS.  This improves
;                          performance (a lot).  Requires 60ns
;                          DRAMs, which are being shipped with the
;                          evaluation board.
;   1.3   1996.02.06  mcd  modified to use new ns486reg.inc file
;   1.4   1996.03.07  mcd  decreased FLASH wait states to 2
;   1.5   1996.10.31  mcd  combined boiler into one .BIN file
;                          changed boot rom to: 0xFFF00000 to 0xFFFFFFFF
;                          changed BIU timing to use 2 RDY (Ready)
;                          extensions, which will better support PC104 cards.
;                          changed logical CS1 timing be consistent with
;                          boot rom and UART requirements
;                          changed headers for VSS SCCS
;                          now uses 32 bit IO macros for some setup
;                          added lots of comments
;   1.6   1996.11.26  mcd  changed BIU_RBATR to 0x5A for clarity
;                          changed UART to 4 wait states - latest rev. Altera
;                          for evb requires this as it no longer uses RDY for
;                          UART accesses
;
; COPYRIGHT
;
;      (c) 1995, 1996 National Semiconductor Corporation
;
;---------------------------------------------------------------------------

.486p  ; allow all instructions in assembler

;---------------------------------------------------------------------------
; include files
;---------------------------------------------------------------------------

include ns486reg.inc       ; NS486SXF registers
include io.inc             ; macros for port IO
include x86.inc            ; a few x86 instructions MASM doesn't deal with
include boiler.inc         ; boiler plate defines

;---------------------------------------------------------------------------
; defines
;---------------------------------------------------------------------------

DRAM_COUNT      equ     04E2h   ; used by DRAM setup to estimate a 100us
                                ; delay to meet DRAM setup spec
                                ; 4E2 should give about 5000 clocks
                                ; delay, or 200us at 25MHz.
                                ; (and more for slower clock speeds)

;---------------------------------------------------------------------------
; external symbols
;---------------------------------------------------------------------------

EXTRN user_start:near  ; code to jump to

;---------------------------------------------------------------------------
; ************ Code ********************************************************
;---------------------------------------------------------------------------

CODESG  SEGMENT PARA USE32 PUBLIC 'code'
BOILER  PROC    NEAR
        ASSUME  CS:CODESG

;---------------------------------------------------------------------------
; entry point
;---------------------------------------------------------------------------

entry:        
        cli  ; disable maskable interrupts
        cld  ; clear direction flag

;---------------------------------------------------------------------------
; voltage/oscillator setup
;---------------------------------------------------------------------------
; The power management registers that determine 3.3/5V operation should
; be setup early to avoid problems.  They should definitely be setup before
; DRAM access.

        outpb PM_PMR3, 000h  ; TTL level output voltage/drive
                             ; this is the reset default, so this
                             ; is redundant, but the DRVCON bit (bit 6)
                             ; will be meaningful once we set up PM_PMR4,
                             ; so may as well do this
        outpb PM_PMR4, 020h  ; set 5V operation

;---------------------------------------------------------------------------
; chip select setup
;---------------------------------------------------------------------------
; Use of chip selects is board specific.  The following setup is for the        
; NS486SXF evaluation board.  Note: there are 8 logical chip selects
; that can be setup.  These can be mapped to any of 8 physical chip select
; pins.  The use of the pins is board specific.  However, you can use
; any logical chip selects to define the ranges and timing of your
; chip selects.  These can then be mapped as needed.

  ; --- disable chip selects ---
        
        outpb BIU_CS_EN, 0h  ; disable all chip selects
                             ; OK since we are running above 0FFFF0000h
        
  ; --- CS ranges ---
                                       
                                        ; CS1 - system flash (boot)
        outpd BIU_CSBAR1_0, 0FFF00000h  ; base 0FFF00000h
        outpd BIU_CSARR1_0, 0000FFFFFh  ; top  0FFFFFFFFh
                                        ; AMD 29F200 device in evaluation
                                        ; board goes from 0FFFC0000h to
                                        ; 0FFFFFFFFh.  It is OK to program
                                        ; this range to the larger range
                                        ; defined in the evb spec.  The
                                        ; flash may be shadowed in the
                                        ; lower addresses.

                                        ; CS2 - external (DEBUG) UART
        outpd BIU_CSBAR2_0, 0000002F8h  ; base 02F8h (COM 2)
        outpd BIU_CSARR2_0, 000000007h  ; top  02FFh
                                        ; external UART may be mapped
                                        ; at any COM address by changing
                                        ; this code

                                        ; CS5 - PC/104 memory
        outpd BIU_CSBAR5_0, 004000000h  ; base 4000000h 
        outpd BIU_CSARR5_0, 000FFFFFFh  ; top  4FFFFFFh
                                        ; this range is defined in the
                                        ; evaluation board spec.  It could
                                        ; be changed as desired to move
                                        ; the PC/104 memory window
        
                                        ; CS6 - PC/104 IO
        outpd BIU_CSBAR6_0, 000000000h  ; base 00000h
        outpd BIU_CSARR6_0, 00000FFFFh  ; top  0FFFFh
                                        ; PC/104 IO can be at any IO
                                        ; address.  Note: although this
                                        ; chip select will go active for
                                        ; internal IO (to NS486SXF
                                        ; registers), no command strobes
                                        ; (IOW*, IOR*) will be seen

                                        ; CS7 - PC/104 memory, 1st MB
        outpd BIU_CSBAR7_0, 004000000h  ; base 4000000h 
        outpd BIU_CSARR7_0, 0000FFFFFh  ; top  40FFFFFh
                                        ; used on eval board to qualify
                                        ; some PC/104 signals.  This
                                        ; should be the first 1MB of the
                                        ; range set up in CS5

  ; --- CS types ---

        outpb BIU_CS_TYPE, 051h  ; select which are memory (1,5,7)
        outpb BIU_P16LCSR, 001h  ; select which are forced 16 bit devices (1)
                                 ; flash is 16bit on eval board
                                 ; PC/104 is 8-bit/16-bit, so uses CS16*
                                 ; UART is 8-bit
  
  ; --- CS timing ---

        outpb BIU_RBATR, 05Ah  ; * 2 periods of Ready (RDY) extension  
                               ;   affects all transfers that use the RDY
                               ;   signal (global setting)
                               ;
                               ; boot rom (CS0):
                               ; * cacheable
                               ; * 1 command delay
                               ; * 2 wait states
                               ;   sufficient for 120ns flash at 25MHz

        outpb BIU_CSATR1, 0CAh ; flash (CS1):
                               ; * 1 command delay
                               ; * 2 wait states
                               ;
                               ; UART (CS2):
                               ; * 1 command delay
                               ; * 4 wait states

        outpb BIU_CSATR3, 0BAh ; PC/104 memory (CS5):  
                               ; * 1 command delay
                               ; * 2 wait states
                               ; 
                               ; PC/104 IO (CS6):
                               ; * 1 command delay
                               ; * 3 wait states

        outpb BIU_CSATR4, 00Ah ; PC/104 first meg (CS7):  
                               ; * 1 command delay
                               ; * 2 wait states

  ; --- logical CS to physical CS pin mappings ---
        
        outpb BIU_RBSR,   001h  ; add CS1 to physical CS0-pin
        outpb BIU_ECSSR3, 002h  ; physical CS3-pin is UART (CS2)
        outpb BIU_ECSSR4, 030h  ; physical CS4-pin is PC/104 memory and IO
                                ; (CS5, CS6)
        outpb BIU_ECSSR5, 040h  ; physical CS5-pin is PC/104 1st meg (CS7)

  ; --- set cache settings ---

        outpb BIU_CCSR, 0FFh  ; all chip selects are for cacheable ranges
                              ; note: rev A, B2 requires this to avoid
                              ; bugs in the chip
        
  ; --- enable chip selects ---
        
        outpb BIU_CS_EN, 073h  ; enable the used chip selects (1,2,5,6,7)
        
;---------------------------------------------------------------------------
; enable instruction cache
;---------------------------------------------------------------------------
; Enabling the cache is recommended on revision B2 and newer.  The cache
; is essential to the performance of the NS486SXF.

        mov     eax, cr0         ; get CR0
        and     eax, 0BFFFFFFFh  ; clear cache disable (CD) bit
        mov     cr0, eax         ; write CR0

;---------------------------------------------------------------------------
; initialize DRAM
;---------------------------------------------------------------------------
; DRAM setup is board specific.  This setup is for the evaluation board        
; with a 1MB SIMM.  This will work fine with a 4MB or 16MB SIMM, but only
; 1MB will be available.  See the "DRAMSIZE" example for sample code to
; automatically size the DRAM on the evaluation board.  The timing is
; set up to allow 20MHz or 25MHz operation.
;
; This code does not enable parity.  If parity checking is desired, the
; following steps are suggested:
;   (1) enable parity in DRAM_CONTROL below (bit 2), but disable NMI (bit 14)
;   (2) after the following DRAM code, initialize all DRAM to a known value
;       (probably 0x00)
;   (3) if the DRAM_STATUS register indicates a parity error detected,
;       clear the parity error (bit 2)
; After the above setup, parity checking will occur.  The DRAM_STATUS
; register can be read to see if a parity error has occurred.  Alternately,
; an NMI may be generated on parity error detection (bit 14 of DRAM_CONTROL).
; To use the NMI, you must set up a valid NMI interrupt handler before
; enabling the NMI on parity error detection.
;
; It is possible to write code to automatically size the DRAM.  However, this
; code is generally specific on the system design and types of memory supported.
; The example "DRAMSIZE" included with the evaluation board software has sample
; code that automagically sizes the DRAM on the evaluation board (1, 4, or 16MB).
; If automatic sizing is used, the following code could be replaced with the
; code to size and configure the DRAM.

        outpb DRAM_BANK,  000h   ; both banks are 0.5MB (1MB total)
                         
        outpb DRAM_MASK0, 0FFh   ; mask for 1/2 megabytes
        outpb DRAM_MASK1, 0FFh   ; mask for 1/2 megabytes
        outpb DRAM_ADDR1, 001h   ; bank 1 starts at 080000h

        outpw DRAM_REFRESH, 625  ; 15.625us between each refresh
                                 ; cycle, or 16ms to refresh
                                 ; the total DRAM (1024 refreshes)
                                 ; This is set for 20MHz operation.
                                 ; Could be bumped to about 780
                                 ; for 25MHz operation, but
                                 ; performance difference will be
                                 ; trivial.

        outpw DRAM_RAS, 4000     ; 100us maximum RAS low time.
                                 ; Again this is at 20MHz.
                                 ; Since this number is greater
                                 ; than the refresh cycle time,
                                 ; it is irrelevant, as refresh
                                 ; will force RAS low.  This would
                                 ; be 5000 at 25MHz, but as long as
                                 ; this is greater than the
                                 ; refresh time, it is irrelevant.

        outpw DRAM_CONTROL, 0903h ; DRAM control
                                  ; parity error NMI disabled
                                  ; bank 1, 9 bit page size
                                  ; bank 0, 9 bit page size
                                  ; three cycle page miss
                                  ; normal CAS low
                                  ; normal refresh
                                  ; parity disabled
                                  ; bank 1 present
                                  ; bank 0 present

  ; --- wait minimum 100us specified in Spec ---

        mov     cx, DRAM_COUNT  ; defined at top of file
centuryus:
        dec     cx
        jnz     centuryus

  ; --- make sure the DRAM ready bit is set ---

        mov     dx, 0ef86h
eskimo:
        in      ax, dx
        test    ax, 01h
        jz      eskimo

;---------------------------------------------------------------------------
; GDT
;---------------------------------------------------------------------------
; GDT setup is application specific.  This sets up a GDT with many
; entires so that this code could be used as boot code for many different
; applications.  The GDT area is cleared so that when dumping the GDT in
; a debugger, there are no garbage entries.

        ; Clear GDT area

        xor     eax, eax     ; fill with 0
        mov     ecx, 04000h  ; 4000h double words is 64Kbytes
        mov     edi, GDTB    ; RAM destination
rep     stosd   es:[edi]     ; clear it!

        ; Copy GDT entries

        mov     esi, offset gdt00 + boiler_base    ; start of GDT in ROM
        mov     edi, GDTB                          ; RAM destination
        mov     ecx, ( offset gdtxx - gdt00 ) / 4  ; number of dwords
rep     movsd   es:[edi], ds:[esi]                 ; copy entries!

        ; load GDT

        lgdt FWORD PTR pgdt + boiler_base  ; load CPU GDT registers

;---------------------------------------------------------------------------
; IDT 
;---------------------------------------------------------------------------
; IDT setup is also application specific.  This setup points all interrupts
; at an arbitrary ISR, which we but in DRAM.  The ISR by default just
; does an IRET.  This setup allows macros in "BOILER.INC" to be used to
; direct a specific interrupt to a specific routine.

        ; fill IDT area with IDT entry

        mov     edi, IDTB                          ; RAM destination
        
        mov     ecx, ( IDTL + 1 ) / 8              ; number of IDT entries
idt_loop:
        mov     esi, offset idtlo + boiler_base    ; low dword in ROM
        movsd   es:[edi], ds:[esi]                 ; copy to RAM
        mov     esi, offset idthi + boiler_base    ; high dword in ROM
        movsd   es:[edi], ds:[esi]                 ; copy to RAM
        dec     ecx
        jnz     idt_loop

        ; load IDT

        lidt FWORD PTR pidt + boiler_base  ; load CPU IDT registers
       
;---------------------------------------------------------------------------
; copy default ISR to RAM
;---------------------------------------------------------------------------

        mov     esi, offset start_isr + boiler_base  ; start of ISR in ROM
        mov     edi, ISRB                            ; RAM destination
        mov     ecx, ( offset end_isr - start_isr )  ; size of ISR
rep     movsb   es:[edi], ds:[esi]                   ; copy it!

;---------------------------------------------------------------------------
; setup segment registers
;---------------------------------------------------------------------------
; now that the GDT is setup, we can setup the segment registers with
; valid entries.  Note: on reset the segment register values are not
; valid, but the hidden "shadow registers" make all segments 32 bit
; flat (4GB range) segments.  The shadow registers are reloaded when
; the segments are setup.

        ; set up datasegments

        mov     ax, DS32F
        mov     ds, ax
        mov     es, ax
        mov     fs, ax
        mov     gs, ax

        ; set up stack

        mov     ax, SSR0
        mov     ss, ax
        mov     esp, 0fffch

        ; reload code segment
        ; need to do far jump or return to reload CS

        jmpfp CS32F, offset postal + boiler_base
postal:

;---------------------------------------------------------------------------
; User Code Starts Here
;---------------------------------------------------------------------------
; You can put code here, or jump to somewhere else.  Right now it jumps
; the assembly procedure user_start, which is in the "INT.ASM" example
; file.

        jmp user_start;

;---------------------------------------------------------------------------
; Thats All Folks!
;---------------------------------------------------------------------------
        
        HLT

;---------------------------------------------------------------------------
; ************ Data ********************************************************
;---------------------------------------------------------------------------
        
;---------------------------------------------------------------------------
; Default Interrupt Service Routine
;---------------------------------------------------------------------------

        ; This routine is copied to RAM where all IDT entries
        ; initially point.  It is in RAM so that a test may easily
        ; change the default interrupt handler.

        ; Note, this is code, and must be location independent!

start_isr:

        iretd

end_isr db 0h;

;---------------------------------------------------------------------------
; GDT
;---------------------------------------------------------------------------
        
        ; pointer to GDT

pgdt    LABEL   FWORD    ; pointer to GDT
glim    DW GDTL          ; limit
gbas    DD GDTB          ; base

        ; Global Descriptor Table Entries
        
        ; legend:
        ;       c  = code
        ;       d  = data
        ;       16 = 16 bit segment    
        ;       32 = 32 bit segment
        ;       rw = read/write
        ;       re = read/execute
        ;       ro = read only
        ;       f  = flat (4GB from 0x0 to 0xFFFFFFFF)
        ;       r0 = ring 0 protection level
        ;       r3 = ring 3 protection level
        
        ;  lim0  lim1  base0 base1 base2 type  lim2  base3
        ;                                dpl   misc

gdt00   db 000h, 000h, 000h, 000h, 000h, 000h, 000h, 000h ; null

gdt08   db 0FFh, 0FFh, 000h, 000h, 000h, 09Bh, 0CFh, 000h ; c32rw f r0
gdt10   db 0FFh, 0FFh, 000h, 000h, 000h, 093h, 0CFh, 000h ; d32rw f r0
gdt18   db 0FFh, 0FFh, 000h, 000h, 000h, 09Bh, 08Fh, 000h ; c16rw f r0
gdt20   db 0FFh, 0FFh, 000h, 000h, 000h, 093h, 08Fh, 000h ; d16rw f r0

gdt28   db 0FFh, 0FFh, 000h, 000h, 000h, 0FBh, 0CFh, 000h ; c32rw f r3
gdt30   db 0FFh, 0FFh, 000h, 000h, 000h, 0F3h, 0CFh, 000h ; d32rw f r3
gdt38   db 0FFh, 0FFh, 000h, 000h, 000h, 0FBh, 08Fh, 000h ; c16rw f r3
gdt40   db 0FFh, 0FFh, 000h, 000h, 000h, 0F3h, 08Fh, 000h ; d16rw f r3

gdt48   db 0FFh, 0FFh, 000h, 000h, 00Ch, 093h, 040h, 000h ; d32rw   r0
gdt50   db 0FFh, 0FFh, 000h, 000h, 00Dh, 0F3h, 040h, 000h ; d32rw   r3

gdt58   db 0FFh, 0FFh, 000h, 000h, 00Ah, 093h, 040h, 000h ; d32rw   r0
gdt60   db 0FFh, 0FFh, 000h, 000h, 00Ah, 0F3h, 040h, 000h ; d32rw   r3
gdt68   db 0FFh, 007h, 000h, 000h, 00Bh, 093h, 040h, 000h ; d32rw   r0
gdt70   db 0FFh, 007h, 000h, 000h, 00Bh, 0F3h, 040h, 000h ; d32rw   r3

gdtxx   db 000h, 000h, 000h, 000h, 000h, 000h, 000h, 000h ; null

        ; segments

    ;  0  00  null
    ;  1  08  32 bit code ring 0 flat  - normal code segment
    ;  2  10  32 bit data ring 0 flat  - normal data segment
    ;  3  18  16 bit code ring 0 flat
    ;  4  20  16 bit data ring 0 flat
    ;  5  28  32 bit code ring 3 flat
    ;  6  30  32 bit data ring 3 flat
    ;  7  38  16 bit code ring 3 flat
    ;  8  40  16 bit data ring 3 flat
    ;  9  48  32 bit data ring 0 0000C0000 to 0000CFFFF - ring 0 stack
    ; 10  50  32 bit data ring 3 0000D0000 to 0000DFFFF - ring 3 stack
    ; 11  58  32 bit data ring 0 0000A0000 to 0000AFFFF - gdt area
    ; 12  60  32 bit data ring 3 0000A0000 to 0000AFFFF - gdt area
    ; 13  68  32 bit data ring 0 0000B0000 to 0000B07FF - idt area
    ; 14  70  32 bit data ring 3 0000B0000 to 0000B07FF - idt area

;---------------------------------------------------------------------------
; IDT
;---------------------------------------------------------------------------

        ; pointer to IDT

pidt    LABEL   FWORD    ; pointer to GDT
ilim    DW IDTL          ; limit
ibas    DD IDTB          ; base

        ; IDT default entry

idtlo    db 000h, 000h, 008h, 000h
idthi    db 000h, 08Eh, 00Eh, 000h

        ; this IDT entry goes to 08:000E0000 which is 32 bit ring 0 code

;---------------------------------------------------------------------------
; end of segment
;---------------------------------------------------------------------------

BOILER  ENDP
CODESG  ENDS
        
; end of program - list entry proc (FAR)

        END     BOILER

;---------------------------------------------------------------------------
; END       boiler.asm 
;---------------------------------------------------------------------------

